/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/portal/portaltour', [
    'io.ox/office/tk/config',
    'io.ox/office/baseframework/app/guidedtour',
    'io.ox/office/baseframework/app/baseapplication',
    'io.ox/office/portal/portalutils',
    'gettext!io.ox/office/portal/main'
], function (Config, GuidedTour, BaseApplication, PortalUtils, gt) {

    'use strict';

    var // the 'shown' configuration key for any of the portal tours
        ANY_PORTAL_TOUR_SHOWN_KEY = 'io.ox/office/portal/alltours/shown';

    // class PortalTour =======================================================

    /**
     * Base class for all guided tours for the Portal applications.
     *
     * @constructor
     *
     * @extends GuidedTour
     *
     * @param {String} appBaseName
     *  The base name of the application represented by this tour.
     */
    function PortalTour(appBaseName) {

        var // self reference
            self = this,

            // type identifier of the Portal application
            PORTAL_APP_TYPE = PortalUtils.getPortalModuleName(appBaseName),

            // type identifier of the Portal application
            EDITOR_APP_TYPE = PortalUtils.getEditorModuleName(appBaseName),

            // CSS selector for the 'My Files/Documents' folder in Drive
            DOCUMENTRS_FOLDER_SELECTOR = '.folder-tree .folder[data-id="' + PortalUtils.getNewLaunchOptions().folderId + '"] .folder-node';

        // base constructor ---------------------------------------------------

        GuidedTour.call(this, appBaseName);

        // constants ----------------------------------------------------------

        // selector for the root window node of the editor application
        this.EDITOR_WINDOW_SELECTOR = '.window-container[data-app-name="' + EDITOR_APP_TYPE + '"]:visible';

        // selector for the top pane of the editor application
        this.EDITOR_TOP_PANE_SELECTOR = this.EDITOR_WINDOW_SELECTOR + ' .tool-pane.top-pane';

        // selector for the main tool pane of the editor application
        this.EDITOR_TOOL_PANE_SELECTOR = this.EDITOR_WINDOW_SELECTOR + ' .tool-pane.edit-pane';

        // selector for the application pane of the editor application
        this.EDITOR_APP_PANE_SELECTOR = this.EDITOR_WINDOW_SELECTOR + ' .app-pane';

        // private methods ----------------------------------------------------

        /**
         * Adds all settings for a tour step that shows the settings drop-down
         * menu, and refers to the 'Restart This Tour' menu entry.
         */
        function addRestartTourSettings(step) {
            return step
                .title(gt.pgettext('tour', 'Restart Guided Tour'))
                .content(gt.pgettext('tour', 'Hint: you can start guided tours, any time you need them, from the system menu.'))
                // open the global drop-down menu
                .on('navigate', function () { $('#io-ox-topbar-dropdown-icon>a').dropdown('toggle'); })
                // wait for and highlight the 'Restart tour' menu entry
                .waitForAndReferTo('#topbar-settings-dropdown [data-action="guided-tour"]', { position: 'left', hotspot: true });
        }

        // public methods -----------------------------------------------------

        /**
         * Defines a generic tour step that creates a new blank document,
         * activates the editor application, and waits for launching.
         *
         * @attention
         *  The created tour step will neither contain a title nor any content.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.genericLaunchDocumentStep = function () {
            return this.step()
                // register the application launcher, create a new blank document
                .navigateTo(EDITOR_APP_TYPE + '/main', PortalUtils.getNewLaunchOptions())
                // wait for the application window node, and the 'data-doc-loaded' marker that will be set after import
                .waitFor(this.EDITOR_WINDOW_SELECTOR + '[data-doc-loaded="true"]', 10)
                // TODO: step back and activate launcher, return without launching
                .enableBackButton(false);
        };

        /**
         * Defines a generic tour step that closes the current editor document,
         * and launches another application.
         *
         * @attention
         *  The created tour step will neither contain a title nor any content.
         *
         * @param {String} launchAppName
         *  The name of the application to be launched after the document has
         *  been closed.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.genericCloseDocumentStep = function (launchAppName) {

            // cache for the active application, for exchange between different event handlers
            var app = null;

            // close document, and launch the other application
            return this.step()
                // grab the active application before launching the new application
                .on('before:show', function () { app = this.getApp(); })
                // launch the Drive application
                .navigateTo(launchAppName)
                // quit in a 'show' handler, will be triggered after launching (first launch,
                // then quit, otherwise another application may be launched automatically
                // inbetween (and may start another tour)
                .on('show', function () { if (BaseApplication.isInstance(app)) { app.quit(); } })
                // TODO: step back and reopen the document?
                .enableBackButton(false);
        };

       // specific portal tour steps -----------------------------------------

        /**
         * Defines the leading welcome step of the portal tour, referring to
         * the launcher tab of the portal application.
         *
         * @attention
         *  The created tour step will neither contain a title nor any content.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.welcomeStep = function () {
            return this.step()
                .referTo('#io-ox-topbar .launcher[data-app-name="' + PORTAL_APP_TYPE + '"]', { position: 'bottom', hotspot: true });
        };

        /**
         * Defines the tour step referring to the templates area in the portal
         * application window.
         *
         * @attention
         *  The created tour step will neither contain a title nor any content.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.templatesAreaStep = function () {
            return this.step()
                .hotspot('.office-portal-templates h2')
                .referTo('.office-portal-templates .template-list', { position: 'bottom' });
        };

        /**
         * Defines the tour step referring to the 'Blank document' button of
         * the portal application.
         *
         * @attention
         *  The created tour step will neither contain a title nor any content.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.blankDocumentStep = function () {
            return this.step()
                .referTo('.office-portal-templates .template-item.template-blank', { position: 'bottom', hotspot: '.template-preview' });
        };

        /**
         * Defines the tour step that creates a new blank document, activates
         * the editor application, waits for launching, and refers to the tool
         * pane of the application window. The launch step will already contain
         * a title and a description text.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.launchDocumentStep = function () {
            return this.genericLaunchDocumentStep()
                .title(gt.pgettext('tour', 'Toolbar'))
                .content(gt.pgettext('tour', 'At the top of the screen you will find the toolbar which shows all the common functions.'))
                // highlight the entire edit tool pane
                .referTo(this.EDITOR_TOOL_PANE_SELECTOR, { position: 'bottom', hotspot: true })
                .on('align', function () { this.$el.addClass('center').css({ left: '', right: '' }); });
        };

        /**
         * Defines the tour step that inserts some contents in the new
         * document created by the launch step, and refers to the application
         * status label shown in the top pane. The auto-save step will already
         * contain a title and a description text.
         *
         * @param {Function} callback
         *  A callback function that must insert some contents into the new
         *  document, in order to show the 'Saving changes' label.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.autoSaveStep = function (callback) {
            this.step()
                .title(gt.pgettext('tour', 'Autosaving changes'))
                .content(gt.pgettext('tour', 'Changes are saved automatically whenever you work on your document.'))
                // invoke the callback function that will insert some contents into the document
                .once('before:show', callback)
                // highlight the 'All changes saved' status label
                .referTo(this.EDITOR_TOP_PANE_SELECTOR + ' .app-status-label', { position: 'bottom', hotspot: 'i' })
                // TODO: step back without launching again
                .enableBackButton(false);
        };

        /**
         * Defines the tour step that refers to the tab button of the 'File'
         * toolbar. The step will already contain a title and a description
         * text.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.fileTabStep = function () {
            return this.step()
                .title(gt.pgettext('tour', 'File tab'))
                .content(gt.pgettext('tour', 'The file tab offers you various ways to share or save your document.'))
                .on('before:show', function () { this.triggerOptionButton(self.EDITOR_TOP_PANE_SELECTOR, 'view/toolbars/tab', 'file'); })
                .hotspot(GuidedTour.getOptionButtonSelector(this.EDITOR_TOP_PANE_SELECTOR, 'view/toolbars/tab', 'file'))
                // do not cover the edit tool pane with the pop-up box
                .referTo(this.EDITOR_TOOL_PANE_SELECTOR, { position: 'bottom' });
        };

        /**
         * Defines the tour step that opens and refers to the 'Save In Drive'
         * drop-down menu. The save-as step will already contain a title and a
         * description text.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.saveAsStep = function () {
            return this.step()
                .title(gt.pgettext('tour', 'Saving options'))
                .content(gt.pgettext('tour', 'If you would like to store your document in a specific folder, just click on "Save as".'))
                .content($('<br>'))
                .content(gt.pgettext('tour', 'You can also click on "Save as template" to create your own templates.'))
                // open and highlight the 'Save As' drop-down menu
                .showDropDownMenu(this.EDITOR_TOOL_PANE_SELECTOR, 'view/saveas/menu')
                .waitForAndReferToPopup({ position: 'right', hotspot: true });
        };

        /**
         * Defines the tour step that refers to the 'Send As Mail' button. The
         * mail step will already contain a title and a description text.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.sendAsMailStep = function () {
            return this.step()
                .title(gt.pgettext('tour', 'Sending your document'))
                .content(gt.pgettext('tour', 'Sending your document by email is easy. Just click on the mail icon and you are ready to go!'))
                .referToGroup(this.EDITOR_TOOL_PANE_SELECTOR, 'document/sendmail', { position: 'bottom', hotspot: 'i' });
        };

        /**
         * Defines the tour step that refers to the 'Close' button in the top
         * bar, without actually closing the document. The step will already
         * contain a title and a description text.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.closeButtonStep = function () {
            return this.step()
                .title(gt.pgettext('tour', 'Closing document'))
                .content(gt.pgettext('tour', 'The document can be closed now. As all changes are stored automatically, you do not need to save documents manually.'))
                .referToGroup(this.EDITOR_TOP_PANE_SELECTOR, 'app/quit', { position: 'bottom', hotspot: 'i' });
        };

        /**
         * Defines the tour step that closes the document, launches the 'Drive'
         * application, and refers to the 'Documents' folder. The close step
         * will already contain a title and a description text.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.closeDocumentStep = function () {
            return this.genericCloseDocumentStep('io.ox/files/main')
                .title(gt.pgettext('tour', 'Documents folder'))
                .content(gt.pgettext('tour', 'You can find your document in your "Documents" folder.'))
                // wait for and highlight the 'Documents' folder (TODO: ensure visible folder side-pane)
                .waitForAndReferTo('.window-container[data-app-name="io.ox/files"] ' + DOCUMENTRS_FOLDER_SELECTOR, { position: 'bottom', hotspot: '.folder-label>div' });
        };

        /**
         * Defines the tour step that opens the settings drop-down menu, and
         * refers to the 'Restart This Tour' menu entry. The step will already
         * contain a title and a description text.
         *
         * @returns {Step}
         *  The new tour step. The instance has been extended with the mix-in
         *  class StepMixin defined in the GuidedTour module.
         */
        this.restartTourStep = function () {
            return addRestartTourSettings(this.step())
                // show the Portal application again
                .navigateTo(PORTAL_APP_TYPE + '/main')
                // TODO: step back to Drive without closing editor application
                .enableBackButton(false);
        };

        this.closeDocumentAndRestartTourStep = function () {
            return addRestartTourSettings(this.genericCloseDocumentStep(PORTAL_APP_TYPE + '/main'));
        };

        // initialization -----------------------------------------------------

        // set the 'shown' configuration flag for any of the portal welcome tours,
        // needed to distinguish between full tours and shortened tours
        this.on('start', function () {
            Config.set(ANY_PORTAL_TOUR_SHOWN_KEY, true);
        });

    } // class PortalTour

    // static methods ---------------------------------------------------------

    /**
     * Returns whether any of the welcome tours for a Documents portal
     * application has already been shown.
     *
     * @returns {Boolean}
     *  Whether any of the welcome tours for a Documents portal application has
     *  already been shown.
     */
    PortalTour.isAnyPortalTourShown = function () {
        return Config.getFlag(ANY_PORTAL_TOUR_SHOWN_KEY);
    };

    // exports ================================================================

    return GuidedTour.extend({ constructor: PortalTour });

});
