/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/fileactions', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/baseframework/app/extensionregistry',
    'io.ox/office/editframework/utils/editconfig',
    'gettext!io.ox/office/editframework/actions'
], function (ext, links, DriveUtils, Utils, IO, ExtensionRegistry, Config, gt) {

    'use strict';

    // static private functions ===============================================

    /**
     * Returns whether the files in the folder described by the passed baton
     * object are considered to be editable. The trash folder for example will
     * be treated as read-only folder.
     *
     * @param {Object} baton
     *  The baton object providing access to a folder.
     *
     * @returns {Boolean|jQuery.Promise}
     *  A Boolean value specifying whether the folder is considered editable,
     *  or a Promise that will resolve with a Boolean value, if the folder data
     *  in the baton object is not available yet.
     */
    function isEditableFolder(baton) {
        // no baton.data, no trash folder (e.g. main portal)
        if (!baton.data.id) { return $.when(true); }

        return DriveUtils.getTrashState(baton.data.folder_id).then(function (state) {
            return !state;
        });
    }

    /**
     * Disables the passed extension point for invalid files (all files names
     * with a specific error suffix in their extension).
     *
     * @param {String} actionId
     *  The identifier of the action to be disabled.
     */
    function disableActionForInvalidFiles(actionId) {
        new links.Action(actionId, {
            id: 'disable_action',
            index: 'first',
            requires: function (data) {
                if (ExtensionRegistry.isError(data.baton.data.filename)) {
                    data.stopPropagation();
                    return false;
                }
            }
        });
    }

    /**
     *  e.collection.has('one', 'read', 'modify') is the normal appsuite code
     *  e.baton.models[0].hasWritePermissions() is the new code for sharing
     *  Bug 40846
     */
    function hasWritePermissions(e) {
        if (e.collection.has('one', 'read')) {
            if (e.collection.has('modify')) {
                return true;
            }
            if (e.baton.models.length) {
                return e.baton.models[0].hasWritePermissions();
            }
        }
        return false;
    }

    // static class FileActions ===============================================

    var FileActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the OX Drive application for a
     * specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {Object} options
     *  Optional parameters:
     *  @param {String} options.newDocumentLabel
     *      The string shown for the 'Create new document' action link inserted
     *      into the tool bar of OX Drive.
     *  @param {Number} options.newDocumentIndex
     *      The index of the 'Create new document' action link that will be
     *      inserted into the tool bar of OX Drive. The 'Create new document'
     *      action links of the various OX Document applications will be
     *      ordered increasing by this index.
     */
    FileActions.create = function (appBaseName, options) {

        var // root for extension points in OX Drive application
            ACTION_POINT = appBaseName,
            // the module name of the edit application
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(launchOptions) {
            ox.launch(MODULE_NAME + '/main', launchOptions);
        }

        // initialization -----------------------------------------------------

        // Creates a new empty document in the current folder of the OX Drive application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '-newblank', {
                requires: function (e) {
                    //Bug 40479
                    //it is important to call getData before of "can" or "is" calls
                    //DriveUtils does this already, but in this situation we dont have the id of the folder yet
                    return e.baton.app.folder.getData().then(function (data) {
                        return DriveUtils.getCreateState(data.id).then(function (create) {
                            if (create) {
                                return DriveUtils.getTrashState(data.id).then(function (state) {
                                    return !state;
                                });
                            } else {
                                return false;
                            }
                        });
                    });
                },
                action: function (baton) {
                    launchApplication({ action: 'new', folderId: baton.app.folder.get() });
                }
            });
        }

        var editAction = function (baton) {
            DriveUtils.getFile(baton.data).done(function (data) {
                if (ExtensionRegistry.isNative(data.filename)) {
                    launchApplication({ action: 'load', file: data });
                } else {
                    launchApplication({ action: 'convert', folderId: data.folder_id, templateFile: data, preserveFileName: true });
                }
            });
        };
        var templAction = function (baton) {
            launchApplication({ action: 'convert', template: true, folderId: baton.data.folder_id, initial_folderId: DriveUtils.getStandardDocumentsFolderId(), templateFile: baton.data, initial_filename: gt('unnamed') });
        };

        // Loads an existing document from the current file in the OX Drive application.
        // If the document requires conversion to a supported file format, creates the
        // converted file and loads it.
        new links.Action(ACTION_POINT + '-edit', {
            requires: function (e) {
                var filename = e.context.filename;
                if (hasWritePermissions(e) && ExtensionRegistry.isEditable(filename, MODULE_NAME) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME) && ExtensionRegistry.isNative(filename, MODULE_NAME)) {
                    // prefetch application source code
                    IO.prefetchModuleSource(MODULE_NAME);
                    return isEditableFolder(e.baton);
                } else {
                    return $.when(false);
                }
            },
            filter: function (obj) {
                return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
            },
            action: editAction
        });

        new links.Action(ACTION_POINT + '-new-fromtemplate', {
            requires: function (e) {

                // workaround for Guest/anonymous
                // guest -> no new-from-template button!
                // TODO: make param newfolderid, this could happen in current folder, if there are create rights for the guest
                if (DriveUtils.isGuest()) {
                    return $.when(false);
                }

                var filename = e.context.filename;
                if (e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                    return isEditableFolder(e.baton);
                } else {
                    return $.when(false);
                }
            },
            filter: function (obj) {
                return ExtensionRegistry.isTemplate(obj.filename, MODULE_NAME);
            },
            action: templAction
        });

        //double click or enter in Drive
        new links.Action('io.ox/files/actions/default', {
            id: MODULE_NAME,
            //requires is not called for 'io.ox/files/actions/default' so we must use filter function!
            filter: function (obj) {
                if (ExtensionRegistry.isTemplate(obj.filename, MODULE_NAME)) {
                    return true;
                } else if (DriveUtils.canWrite(obj.folder_id) && ExtensionRegistry.isNative(obj.filename) && ExtensionRegistry.isEditable(obj.filename, MODULE_NAME)) {
                    return true;
                } else {
                    return false;
                }
            },
            action: function (baton) {
                if (ExtensionRegistry.isTemplate(baton.data.filename, MODULE_NAME)) {
                    templAction(baton);
                } else {
                    editAction(baton);
                }
            }
        });

        // Creates a new document as copy of the current file in the OX Drive application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {

            ////////////////////////////////////////////////////////////////////////

            var editAsNew = function (ref, requires) {
                new links.Action(ACTION_POINT + ref, {
                    requires: requires,
                    filter: function (obj) {
                        return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                    },
                    action: function (baton) {
                        var convert = ExtensionRegistry.isConvertible(baton.data.filename);
                        launchApplication({ action: convert ? 'convert' : 'new', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: convert ? true : false });
                    }
                });
            };

            var isExtraEditAsNew = function (e) {
                var filename = e.context.filename;
                return (e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME) && ExtensionRegistry.isEditable(filename, MODULE_NAME));
            };

            var isNormalEditAsNew = function (e) {
                var filename = e.context.filename;
                return (hasWritePermissions(e) && ExtensionRegistry.isNative(filename, MODULE_NAME));
            };

            editAsNew('-edit-asnew', function (e) {
                if (isExtraEditAsNew(e) && isNormalEditAsNew(e)) {
                    return isEditableFolder(e.baton);
                }
            });

            editAsNew('-edit-asnew-hi', function (e) {

                // workaround for Guest/anonymous
                // guest -> no my files -> no edit-as-new button!
                if (DriveUtils.isGuest()) {
                    return $.when(false);
                }

                if (isExtraEditAsNew(e) && !isNormalEditAsNew(e)) {
                    return isEditableFolder(e.baton);
                }
            });

            ////////////////////////////////////////////////////////////////////////

            new links.Action(ACTION_POINT + '-edit-template', {
                requires: function (e) {
                    var filename = e.context.filename;
                    if (hasWritePermissions(e) && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isTemplate(filename, MODULE_NAME) && !ExtensionRegistry.isTemplateLocked(filename, MODULE_NAME) && ExtensionRegistry.isEditable(filename, MODULE_NAME)) {
                        return isEditableFolder(e.baton);
                    } else {
                        return $.when(false);
                    }
                },
                filter: function (obj) {
                    return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                },
                action: function (baton) {
                    if (ExtensionRegistry.isNative(baton.data.filename)) {
                        launchApplication({ action: 'load', file: baton.data });
                    } else {
                        launchApplication({ action: 'convert', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: true });
                    }
                }
            });
        }

        // An action link in the main tool bar of the OX Drive application used
        // to create a new empty document.
        new links.ActionLink('io.ox/files/links/toolbar/default', {
            index: 200 + Utils.getIntegerOption(options, 'newDocumentIndex', 0),
            id: appBaseName + '-newblank',
            label: Utils.getStringOption(options, 'newDocumentLabel', ''),
            ref: ACTION_POINT + '-newblank'
        });

        function extend(id, label, prio, prioMobile, index, ref, section, customize) {
            var link = {
                    id: appBaseName + id,
                    index: index,
                    prio: prio,
                    mobile: prioMobile,
                    label: label,
                    title: label,
                    ref: ACTION_POINT + ref
                };
            if (section) {
                link.section = section;
            }
            if (customize) {
                link.customize = customize;
            }

            //@Deprecated old halo view
            ext.point('io.ox/files/links/inline').extend(new links.Link(link));

            // define links for document actions in the OX Viewer toolbar.
            ext.point('io.ox/core/viewer/toolbar/links/drive').extend(new links.Link(link));

            ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link(link));

        }

        //      id                  label                       prio    prioMobile    index     ref                 section
        extend('-edit',            gt('Edit'),                 'hi',   'lo',         150,      '-edit',            'edit');
        extend('-newfromtemplate', gt('New from template'),    'hi',   'lo',         150,      '-new-fromtemplate');
        extend('-edittemplate',    gt('Edit template'),        'lo',   'lo',         'first',  '-edit-template',   'edit');

        extend('-editasnew',       gt('Edit as new'),          'lo',   'lo',         'first',  '-edit-asnew',     'edit');
        extend('-editasnewhi',     gt('Edit as new'),          'hi',   'lo',         150,      '-edit-asnew-hi',  'edit');

    };

    // static initialization ==================================================

    if (!DriveUtils.isGuest()) {
        //isguest check is a workaround for Bug 40480
        disableActionForInvalidFiles('io.ox/files/actions/add-to-portal');
        disableActionForInvalidFiles('io.ox/files/actions/publish');
    }

    // exports ================================================================

    return FileActions;

});
