/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/pane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/view/viewobjectmixin'
], function (Utils, KeyCodes, Forms, Config, Tracking, TriggerObject, TimerMixin, ViewObjectMixin) {

    'use strict';

    // class Pane =============================================================

    /**
     * Represents a container element attached to a specific border of the
     * application window.
     *
     * Instances of this class trigger the following events:
     * - 'pane:beforeshow'
     *      Before the view pane will be shown.
     * - 'pane:show'
     *      After the view pane has been shown.
     * - 'pane:beforehide'
     *      Before the view pane will be hidden.
     * - 'pane:hide'
     *      After the view pane has been hidden.
     * - 'pane:resize'
     *      After the size of the root node has been changed. The event handler
     *      receives the new width and height of the root node, in pixels.
     * - 'pane:layout'
     *      After any view component in this view pane has changed its
     *      visibility state or size, even if the size of the root node did not
     *      change.
     * - 'pane:focus'
     *      After a view component has been focused, by initially focusing any
     *      of groups it contains.
     * - 'pane:blur'
     *      After a view component in this pane has lost the browser focus,
     *      after focusing any other DOM node outside the component.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     * @extends ViewObjectMixin
     *
     * @param {BaseView} docView
     *  The document view instance containing this pane element.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the root DOM node of the
     *      view pane.
     *  @param {String} [initOptions.position='top']
     *      The border of the application window to attach the view pane to.
     *      Supported values are 'top', 'bottom', 'left', and 'right'.
     *  @param {String} [initOptions.size]
     *      The size of the pane, between window border and application pane.
     *      If omitted, the size will be determined by the DOM contents of the
     *      pane root node.
     *  @param {String} [initOptions.resizable=false]
     *      If set to true, the pane will be resizable at its inner border. Has
     *      no effect for transparent overlay panes.
     *  @param {String} [initOptions.minSize=1]
     *      The minimum size of resizable panes (when the option 'resizable' is
     *      set to true).
     *  @param {String} [initOptions.maxSize=0x7FFFFFFF]
     *      The maximum size of resizable panes (when the option 'resizable' is
     *      set to true).
     *  @param {Boolean} [initOptions.cursorNavigate=false]
     *      If set to true, the cursor keys can be used to move the browser
     *      focus to the next/previous focusable form control in this pane.
     */
    function Pane(docView, initOptions) {

        var // self reference
            self = this,

            // the container element representing the pane
            rootNode = $('<div class="view-pane">').addClass(Utils.getStringOption(initOptions, 'classes', '')),

            // position of the pane in the application window
            position = Utils.getStringOption(initOptions, 'position', 'top'),

            // minimum size of the view pane (for resizable panes)
            minSize = Utils.getIntegerOption(initOptions, 'minSize', 1, 1),

            // maximum size of the view pane (for resizable panes)
            maxSize = Utils.getIntegerOption(initOptions, 'maxSize', 0x7FFFFFFF, minSize),

            // the last cached size of the root node, used to detect layout changes
            nodeSize = null,

            // view components contained in this pane
            components = [],

            // callback handlers for inserting components into this view pane
            componentInserters = [],

            // whether the view pane is set to visible (regardless of effective visibility)
            visible = true,

            // whether a 'pane:resize' event is currently triggering (prevent recursion)
            isResizeTriggering = false,

            // whether a 'pane:layout' event is currently triggering (prevent recursion)
            isLayoutTriggering = false;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);
        ViewObjectMixin.call(this, docView);

        // private methods ----------------------------------------------------

        /**
         * Updates the effective visibility of the root node, according to the
         * own visible state.
         *
         * @returns {Boolean}
         *  Whether the view pane has changed from hidden to visible state.
         */
        function updateNodeVisibility() {

            // update node visibility and notify listeners, if really changed
            if (self.isVisible() !== visible) {
                self.trigger(visible ? 'pane:beforeshow' : 'pane:beforehide');
                Forms.showNodes(rootNode, visible);
                self.trigger(visible ? 'pane:show' : 'pane:hide');
                return visible;
            }
            return false;
        }

        /**
         * Checks the current outer size of the root node of this view pane.
         * Triggers a 'pane:resize' event if the pane is visible and its size
         * has changed. Does nothing if the pane is currently hidden.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.updateLayout=false]
         *      If set to true, updates the layout of all view components.
         *  @param {Boolean} [options.triggerLayout=false]
         *      If set to true, forces to trigger a 'pane:layout' event. This
         *      event will be triggered anyway if the size of the root node has
         *      changed, or if the option 'updateLayout' has been set.
         */
        function updateNodeSize(options) {

            // do nothing if the pane is hidden, or is inside a hidden container node
            if (!self.isVisible() || !Forms.isDeeplyVisibleNode(rootNode)) { return; }

            var // whether to update the view components
                updateLayout = Utils.getBooleanOption(options, 'updateLayout', false),
                // whether to trigger a 'pane:layout' event
                triggerLayout = updateLayout || Utils.getBooleanOption(options, 'triggerLayout', false);

            // update layout of all components if specified
            if (updateLayout) { _.invoke(components, 'layout'); }

            var // current size of the DOM root node
                newSize = { width: rootNode.outerWidth(), height: rootNode.outerHeight() };

            // notify listeners if size has changed
            if (!_.isEqual(nodeSize, newSize)) {
                nodeSize = newSize;
                triggerLayout = true; // always trigger 'pane:layout' event after resize
                if (!isResizeTriggering) {
                    isResizeTriggering = true;
                    self.trigger('pane:resize', nodeSize.width, nodeSize.height);
                    isResizeTriggering = false;
                }
            }

            // notify listeners that a component has changed
            if (triggerLayout) {
                isLayoutTriggering = true;
                self.trigger('pane:layout');
                isLayoutTriggering = false;
            }
        }

        /**
         * Updates visibility and size of the root node of this view pane.
         *
         * @param {Boolean} [updateLayout=false]
         *  If set to true, updates the layout of all view components, and
         *  forces to trigger a 'pane:layout' event (this event will be
         *  triggered anyway if the pane became visible, or if the size of the
         *  root node has changed).
         */
        function updateNode(updateLayout) {
            var becameVisible = updateNodeVisibility();
            updateNodeSize({ updateLayout: updateLayout || becameVisible });
        }

        /**
         * Handles all view components in this view pane whose node size has
         * changed (shown, hidden, resized). Triggers a 'pane:resize' event to
         * all listeners, if the size of the pane has been changed due to the
         * changed component. Afterwards, a 'pane:layout' event will always be
         * triggered.
         */
        var componentResizeHandler = (function () {
            var triggerLayout = false;
            return self.createDebouncedMethod(function () {
                if (!isLayoutTriggering) { triggerLayout = true; }
            }, function () {
                var options = { triggerLayout: triggerLayout };
                triggerLayout = false;
                updateNodeSize(options);
            });
        }());

        /**
         * Handles focus events from all registered view components. Updates
         * the CSS marker class at the root node of this view pane, and
         * forwards the event to all listeners.
         */
        function componentFocusHandler(event) {

            var focused = event.type === 'component:focus',
                type = focused ? 'pane:focus' : 'pane:blur';

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
            self.trigger(type);
        }

        /**
         * Handles all tracking events to resize this view pane.
         */
        var trackingHandler = (function () {

            var // the original size of the view pane when tracking has been started
                origSize = 0;

            function trackingHandler(event) {
                switch (event.type) {
                case 'tracking:start':
                    origSize = self.isVerticalPosition() ? nodeSize.height : nodeSize.width;
                    break;
                case 'tracking:move':
                    var size = origSize + (self.isLeadingPosition() ? 1 : -1) * (self.isVerticalPosition() ? event.offsetY : event.offsetX);
                    self.setSize(Utils.minMax(size, minSize, maxSize));
                    break;
                case 'tracking:end':
                    docView.grabFocus();
                    break;
                case 'tracking:cancel':
                    self.setSize(origSize);
                    docView.grabFocus();
                    break;
                }
            }

            return trackingHandler;
        }()); // end of local scope of method trackingHandler()

        /**
         * Handles keyboard events.
         */
        function keyDownHandler(event) {

            // ESCAPE key
            if (event.keyCode === KeyCodes.ESCAPE) {
                docView.grabFocus();
                return false;
            }

            // shortcut in panes to open drop-down menus with simple UP/DOWN cursor key
            var menuGroupNode = $(event.target).closest('.group.dropdown-group', rootNode[0]),
                menuButtonNode = menuGroupNode.find(Forms.BUTTON_SELECTOR + '.caret-button');
            if ((menuButtonNode.length === 1) && Forms.isVisibleNode(menuButtonNode) && !KeyCodes.hasModifierKeys(event)) {
                if ((event.keyCode === KeyCodes.UP_ARROW) || (event.keyCode === KeyCodes.DOWN_ARROW)) {
                    event.target = menuButtonNode[0];
                    Forms.triggerClickForKey(event);
                    return false;
                }
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Registers a callback function that implements inserting the root DOM
         * node of a new view component into this view pane. The last registered
         * callback function will be invoked first when adding a new view
         * component. If it returns the Boolean value false, the preceding
         * inserter callback functions will be invoked as long as they return
         * false too. If all inserter callback functions have returned false,
         * or if no callback function has been registered at all, the root
         * node of the view component will be appended to the root node of this
         * pane.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub class
         *  constructors. DO NOT CALL from external code!
         *
         * @param {Function} inserter
         *  The callback function called for every view component inserted into
         *  this view pane by calling the method Pane.addViewComponent().
         *  Receives the reference to the new view component instance as first
         *  parameter, and the options passed to Pane.addViewComponent() as
         *  second parameter. Will be called in the context of this instance.
         *  May return false to indicate that the view component has not been
         *  inserted yet.
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.registerComponentInserter = function (inserter) {
            componentInserters.unshift(inserter);
            return this;
        };

        // methods ------------------------------------------------------------

        /**
         * Returns the root element representing this pane as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the options map that has been passed to the constructor.
         */
        this.getOptions = function () {
            return initOptions;
        };

        /**
         * Returns the position of this view pane.
         *
         * @returns {String}
         *  The border of the application window this view pane is attached to.
         *  Possible values are 'top', 'bottom', 'left', and 'right'.
         */
        this.getPosition = function () {
            return position;
        };

        /**
         * Returns whether this pane is in a vertical position (on the top or
         * bottom border of the application window).
         *
         * @returns {Boolean}
         *  Whether this pane is in a vertical position.
         */
        this.isVerticalPosition = function () {
            return Utils.isVerticalPosition(position);
        };

        /**
         * Returns whether this pane is in a leading position (on the top or
         * left border of the application window).
         *
         * @returns {Boolean}
         *  Whether this pane is in a leading position.
         */
        this.isLeadingPosition = function () {
            return Utils.isLeadingPosition(position);
        };

        /**
         * Checks the current outer size of the root node of this view pane.
         * Triggers a 'pane:resize' event if the pane is visible and its size
         * has changed. Does nothing if the pane is currently hidden.
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.layout = function () {
            updateNode(true);
            return this;
        };

        /**
         * Returns whether this view pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the view pane is currently visible.
         */
        this.isVisible = function () {
            return Forms.isVisibleNode(rootNode);
        };

        /**
         * Makes this view pane visible.
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.show = function () {
            return this.toggle(true);
        };

        /**
         * Hides this view pane.
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.hide = function () {
            return this.toggle(false);
        };

        /**
         * Changes the visibility of this view pane.
         *
         * @param {Boolean} [state]
         *  If specified, shows or hides the view pane independently from its
         *  current visibility state. If omitted, toggles the visibility of the
         *  view pane.
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.toggle = function (state) {

            // resolve missing parameter for toggling visibility
            visible = (state === true) || ((state !== false) && !visible);

            // update layout, this sets the effective visibility
            updateNode();
            return this;
        };

        /**
         * Changes the size of this view pane in the variable direction.
         *
         * @param {Number|Null} size
         *  The new size of the view pane, or null to return to automatic size.
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.setSize = function (size) {

            // set fixed size, or remove size attribute for automatic size
            rootNode.css(this.isVerticalPosition() ? 'height' : 'width', _.isNumber(size) ? (size + 'px') : '');

            // notify listeners, if pane is visible, and its size has changed
            updateNode();
            return this;
        };

        /**
         * Adds the passed view component into this view pane.
         *
         * @param {Component} component
         *  The view component to be added to this pane.
         *
         * @param {Object} [options]
         *  Optional parameters that will be passed to the registered component
         *  inserter callback functions.
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.addViewComponent = function (component, options) {

            // store component
            components.push(component);

            // insert the root node of the component into this view pane
            if (_.all(componentInserters, function (inserter) { return inserter.call(self, component, options) === false; })) {
                rootNode.append(component.getNode());
            }

            // handle and forward component events
            this.listenTo(component, 'component:show component:hide component:resize', componentResizeHandler);
            this.listenTo(component, 'component:focus component:blur', componentFocusHandler);

            // update internal layout (pane visibility and size)
            updateNode();

            return this;
        };

        /**
         * Removes the passed view component from this view pane.
         *
         * @param {Component} component
         *  The view component to be removed from this pane. Nothing will
         *  happen, if the component has not been added to this view pane
         *  before using the method Pane.addViewComponent().
         *
         * @returns {Pane}
         *  A reference to this instance.
         */
        this.removeViewComponent = function (component) {

            var // the number of view components
                count = components.length;

            // remove the component from the internal array
            components = _.without(components, component);

            // unregister listeners and remove DOM node if the component was part of this pane
            if (components.length !== count) {

                // stop listening to the view component
                this.stopListeningTo(component);

                // remove the root node of the component from this pane
                component.getNode().detach();

                // update internal layout (pane visibility and size)
                updateNode();
            }
            return this;
        };

        /**
         * Checks whether the component already exists
         *
         * @param {Component} component
         *  The view component to be removed from this pane. Nothing will
         *  happen, if the component has not been added to this view pane
         *  before using the method Pane.addViewComponent().
         *
         * @return {Boolean}
         *  Whether the component exists.
         */
        this.hasViewComponent = function (component) {
            return _.contains(components, component);
        };

        /**
         * Returns whether this view pane contains the control group that is
         * currently focused. Searches in all control groups of all registered
         * view components. Returns also true, if a DOM element is focused that
         * is related to a control group in this view pane, but is not a child
         * of the group's root node, e.g. a pop-up menu.
         *
         * @returns {Boolean}
         *  Whether a control group of this view pane currently owns the
         *  browser focus.
         */
        this.hasFocus = function () {
            return _.any(components, function (component) { return component.hasFocus(); });
        };

        /**
         * Sets the focus to the first enabled control group object in this
         * view pane, unless the pane already contains the browser focus.
         *
         * @returns {Pane}
         *  A reference to this view pane.
         */
        this.grabFocus = function () {
            if (!this.hasFocus()) { Forms.grabFocus(rootNode); }
            return this;
        };

        // initialization -----------------------------------------------------

        // set position at element for CSS selectors
        rootNode.attr('data-pos', position);

        // marker for touch devices and browser types
        Forms.addDeviceMarkers(rootNode);

        // fixed size if specified (zero will be ignored)
        this.setSize(Utils.getIntegerOption(initOptions, 'size', null, 1));

        // no size tracking for transparent view panes
        if (Utils.getBooleanOption(initOptions, 'resizable', false)) {
            Tracking.enableTracking($('<div class="resizer" tabindex="-1">'))
                .on('tracking:start tracking:move tracking:end tracking:cancel', trackingHandler)
                .appendTo(rootNode);
        }

        // whether the pane will be a land mark for F6 focus traveling
        rootNode.toggleClass('f6-target', Utils.getBooleanOption(initOptions, 'landmark', true));

        // keyboard handler
        rootNode.on('keydown', keyDownHandler);
        if (Utils.getBooleanOption(initOptions, 'cursorNavigate', false)) {
            Forms.enableCursorFocusNavigation(rootNode);
        }

        // disable dragging of controls or dropping (otherwise, it is possible to drag buttons and other controls
        // around, or images can be dropped which will be loaded by the browser without any notification)
        rootNode.on('drop dragstart dragenter dragexit dragover dragleave', false);

        // log focus and layout events
        Utils.FocusLogger.logEvent(this, 'pane:focus pane:blur');
        Utils.LayoutLogger.logEvent(this, 'pane:beforeshow pane:show pane:beforehide pane:hide pane:layout');
        Utils.LayoutLogger.logEvent(this, 'pane:resize', 'width', 'height');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(components, 'destroy');
            rootNode.remove();
            self = docView = initOptions = rootNode = components = componentInserters = null;
        });

    } // class Pane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: Pane });

});
