/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/interval'
], function (Utils, PaneUtils, Interval) {

    'use strict';

    // static class PaneUtils =================================================

    describe('Spreadsheet module PaneUtils', function () {

        it('should exist', function () {
            expect(PaneUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var BORDER_WIDTH_MAP = {
                hair: Utils.convertLengthToHmm(0.5, 'px'),
                thin: Utils.convertLengthToHmm(1, 'px'),
                medium: Utils.convertLengthToHmm(2, 'px'),
                thick: Utils.convertLengthToHmm(3, 'px')
            };

        // constants ----------------------------------------------------------

        describe('constant "ALL_PANE_SIDES"', function () {
            it('should exist', function () {
                expect(PaneUtils).to.have.a.property('ALL_PANE_SIDES').that.is.an('array');
                expect(PaneUtils.ALL_PANE_SIDES).to.have.length.above(0);
            });
        });

        describe('constant "ALL_PANE_POSITIONS"', function () {
            it('should exist', function () {
                expect(PaneUtils).to.have.a.property('ALL_PANE_POSITIONS').that.is.an('array');
                expect(PaneUtils.ALL_PANE_POSITIONS).to.have.length.above(0);
            });
        });

        describe('constant "MIN_PANE_SIZE"', function () {
            it('should exist', function () {
                expect(PaneUtils).to.have.a.property('MIN_PANE_SIZE').that.is.a('number');
                expect(PaneUtils.MIN_PANE_SIZE).to.be.above(0);
            });
        });

        describe('constant "ADDITIONAL_SIZE_RATIO"', function () {
            it('should exist', function () {
                expect(PaneUtils).to.have.a.property('ADDITIONAL_SIZE_RATIO').that.is.a('number');
                expect(PaneUtils.ADDITIONAL_SIZE_RATIO).to.be.at.least(0);
            });
        });

        describe('constant "ZOOM_FACTORS"', function () {
            it('should exist', function () {
                expect(PaneUtils).to.have.a.property('ZOOM_FACTORS').that.is.an('array');
                expect(PaneUtils.ZOOM_FACTORS).to.not.be.empty;
            });
            it('should be unique and sorted', function () {
                var sorted = PaneUtils.ZOOM_FACTORS.every(function (f, i, a) { return !i || (a[i - 1] < f); });
                expect(sorted).to.equal(true);
            });
            it('should be around the value 1', function () {
                expect(_.first(PaneUtils.ZOOM_FACTORS)).to.be.at.most(1);
                expect(_.last(PaneUtils.ZOOM_FACTORS)).to.be.at.least(1);
            });
        });

        describe('constant "DEFAULT_TRACKING_OPTIONS"', function () {
            it('should exist', function () {
                expect(PaneUtils).to.have.a.property('DEFAULT_TRACKING_OPTIONS').that.is.an('object');
                expect(PaneUtils.DEFAULT_TRACKING_OPTIONS.autoScroll).to.equal(true);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "isColumnSide"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('isColumnSide');
            });
            it('should return true for left/right', function () {
                expect(PaneUtils.isColumnSide('left')).to.equal(true);
                expect(PaneUtils.isColumnSide('right')).to.equal(true);
            });
            it('should return false for top/botttom', function () {
                expect(PaneUtils.isColumnSide('top')).to.equal(false);
                expect(PaneUtils.isColumnSide('bottom')).to.equal(false);
            });
        });

        describe('method "isLeadingSide"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('isLeadingSide');
            });
            it('should return true for left/top', function () {
                expect(PaneUtils.isLeadingSide('left')).to.equal(true);
                expect(PaneUtils.isLeadingSide('top')).to.equal(true);
            });
            it('should return false for right/bottom', function () {
                expect(PaneUtils.isLeadingSide('right')).to.equal(false);
                expect(PaneUtils.isLeadingSide('bottom')).to.equal(false);
            });
        });

        describe('method "getScrollAnchorName"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getScrollAnchorName');
            });
            it('should return the correct anchor name', function () {
                expect(PaneUtils.getScrollAnchorName('left')).to.equal('anchorLeft');
                expect(PaneUtils.getScrollAnchorName('right')).to.equal('anchorRight');
                expect(PaneUtils.getScrollAnchorName('top')).to.equal('anchorTop');
                expect(PaneUtils.getScrollAnchorName('bottom')).to.equal('anchorBottom');
            });
        });

        describe('method "getColPaneSide"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getColPaneSide');
            });
            it('should return the correct pane side', function () {
                expect(PaneUtils.getColPaneSide('topLeft')).to.equal('left');
                expect(PaneUtils.getColPaneSide('topRight')).to.equal('right');
                expect(PaneUtils.getColPaneSide('bottomLeft')).to.equal('left');
                expect(PaneUtils.getColPaneSide('bottomRight')).to.equal('right');
            });
        });

        describe('method "getRowPaneSide"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getRowPaneSide');
            });
            it('should return the correct pane side', function () {
                expect(PaneUtils.getRowPaneSide('topLeft')).to.equal('top');
                expect(PaneUtils.getRowPaneSide('topRight')).to.equal('top');
                expect(PaneUtils.getRowPaneSide('bottomLeft')).to.equal('bottom');
                expect(PaneUtils.getRowPaneSide('bottomRight')).to.equal('bottom');
            });
        });

        describe('method "getNextColPanePos"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getNextColPanePos');
            });
            it('should return the correct pane position', function () {
                expect(PaneUtils.getNextColPanePos('topLeft')).to.equal('topRight');
                expect(PaneUtils.getNextColPanePos('topRight')).to.equal('topLeft');
                expect(PaneUtils.getNextColPanePos('bottomLeft')).to.equal('bottomRight');
                expect(PaneUtils.getNextColPanePos('bottomRight')).to.equal('bottomLeft');
            });
        });

        describe('method "getNextRowPanePos"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getNextRowPanePos');
            });
            it('should return the correct pane position', function () {
                expect(PaneUtils.getNextRowPanePos('topLeft')).to.equal('bottomLeft');
                expect(PaneUtils.getNextRowPanePos('topRight')).to.equal('bottomRight');
                expect(PaneUtils.getNextRowPanePos('bottomLeft')).to.equal('topLeft');
                expect(PaneUtils.getNextRowPanePos('bottomRight')).to.equal('topRight');
            });
        });

        describe('method "getPanePos"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getPanePos');
            });
            it('should return the correct pane position', function () {
                expect(PaneUtils.getPanePos('left', 'top')).to.equal('topLeft');
                expect(PaneUtils.getPanePos('left', 'bottom')).to.equal('bottomLeft');
                expect(PaneUtils.getPanePos('right', 'top')).to.equal('topRight');
                expect(PaneUtils.getPanePos('right', 'bottom')).to.equal('bottomRight');
            });
        });

        describe('method "getNextPanePos"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getNextPanePos');
            });
            it('should return the correct pane position', function () {
                expect(PaneUtils.getNextPanePos('topLeft', 'top')).to.equal('topLeft');
                expect(PaneUtils.getNextPanePos('topLeft', 'bottom')).to.equal('bottomLeft');
                expect(PaneUtils.getNextPanePos('topLeft', 'left')).to.equal('topLeft');
                expect(PaneUtils.getNextPanePos('topLeft', 'right')).to.equal('topRight');
                expect(PaneUtils.getNextPanePos('topRight', 'top')).to.equal('topRight');
                expect(PaneUtils.getNextPanePos('topRight', 'bottom')).to.equal('bottomRight');
                expect(PaneUtils.getNextPanePos('topRight', 'left')).to.equal('topLeft');
                expect(PaneUtils.getNextPanePos('topRight', 'right')).to.equal('topRight');
                expect(PaneUtils.getNextPanePos('bottomLeft', 'top')).to.equal('topLeft');
                expect(PaneUtils.getNextPanePos('bottomLeft', 'bottom')).to.equal('bottomLeft');
                expect(PaneUtils.getNextPanePos('bottomLeft', 'left')).to.equal('bottomLeft');
                expect(PaneUtils.getNextPanePos('bottomLeft', 'right')).to.equal('bottomRight');
                expect(PaneUtils.getNextPanePos('bottomRight', 'top')).to.equal('topRight');
                expect(PaneUtils.getNextPanePos('bottomRight', 'bottom')).to.equal('bottomRight');
                expect(PaneUtils.getNextPanePos('bottomRight', 'left')).to.equal('bottomLeft');
                expect(PaneUtils.getNextPanePos('bottomRight', 'right')).to.equal('bottomRight');
            });
        });

        describe('method "getRectangleStyleMarkup"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getRectangleStyleMarkup');
            });
            it('should return the correct mark-up', function () {
                expect(PaneUtils.getRectangleStyleMarkup({ left: 1, top: 2, width: 3, height: 4 })).to.equal('left:1px;top:2px;width:3px;height:4px;');
            });
        });

        describe('method "getPresetStyleForBorder"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getPresetStyleForBorder');
            });
            it('should return "none" if the style attribute is "none"', function () {
                expect(PaneUtils.getPresetStyleForBorder({ style: 'none' })).to.equal('none');
                expect(PaneUtils.getPresetStyleForBorder({ style: 'none', width: BORDER_WIDTH_MAP.thick })).to.equal('none');
            });
            it('should return "null" if the given attributes are ambiguous', function () {
                expect(PaneUtils.getPresetStyleForBorder({ style: 'dashed', width: 0 })).to.equal(null);
                expect(PaneUtils.getPresetStyleForBorder({ width: 2 })).to.equal(null);
            });
            it('should return the correct preset style', function () {
                expect(PaneUtils.getPresetStyleForBorder({ style: 'dashed', width: BORDER_WIDTH_MAP.hair })).to.equal('dashed:hair');
                expect(PaneUtils.getPresetStyleForBorder({ style: 'solid', width: BORDER_WIDTH_MAP.thin })).to.equal('solid:thin');
                expect(PaneUtils.getPresetStyleForBorder({ style: 'dotted', width: BORDER_WIDTH_MAP.medium })).to.equal('dotted:medium');
                expect(PaneUtils.getPresetStyleForBorder({ style: 'dashed', width: BORDER_WIDTH_MAP.thick })).to.equal('dashed:thick');
            });
        });

        describe('method "getBorderForPresetStyle"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getBorderForPresetStyle');
            });
            it('should return object with correct border attributes', function () {
                expect(PaneUtils.getBorderForPresetStyle('dashed:hair')).to.be.deep.equal({ style: 'dashed', width: BORDER_WIDTH_MAP.hair });
                expect(PaneUtils.getBorderForPresetStyle('solid:thin')).to.be.deep.equal({ style: 'solid', width: BORDER_WIDTH_MAP.thin });
                expect(PaneUtils.getBorderForPresetStyle('dotted:medium')).to.be.deep.equal({ style: 'dotted', width: BORDER_WIDTH_MAP.medium });
                expect(PaneUtils.getBorderForPresetStyle('dashed:thick')).to.be.deep.equal({ style: 'dashed', width: BORDER_WIDTH_MAP.thick });
            });
            it('should return "empty" border object', function () {
                expect(PaneUtils.getBorderForPresetStyle('none')).to.be.deep.equal({ style: 'none', width: 0 });
            });
            it('should return "empty" border object (unknown preset border)', function () {
                expect(PaneUtils.getBorderForPresetStyle('notExist')).to.be.deep.equal({ style: 'none', width: 0 });
            });
        });

        describe('method "processTrackingCycle"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('processTrackingCycle');
            });
            // TODO
        });

        describe('method "getSelectionMode"', function () {
            it('should exist', function () {
                expect(PaneUtils).itself.to.respondTo('getSelectionMode');
            });
            it('should return the correct selection mode', function () {
                expect(PaneUtils.getSelectionMode({ shiftKey: false, ctrlKey: false, metaKey: false })).to.equal('select');
                expect(PaneUtils.getSelectionMode({ shiftKey: true, ctrlKey: false, metaKey: false })).to.equal('extend');
                expect(PaneUtils.getSelectionMode({ shiftKey: false, ctrlKey: true, metaKey: false })).to.equal('append');
                expect(PaneUtils.getSelectionMode({ shiftKey: true, ctrlKey: true, metaKey: false })).to.equal('select');
                expect(PaneUtils.getSelectionMode({ shiftKey: false, ctrlKey: false, metaKey: true })).to.equal('append');
                expect(PaneUtils.getSelectionMode({ shiftKey: true, ctrlKey: false, metaKey: true })).to.equal('select');
                expect(PaneUtils.getSelectionMode({ shiftKey: false, ctrlKey: true, metaKey: true })).to.equal('append');
                expect(PaneUtils.getSelectionMode({ shiftKey: true, ctrlKey: true, metaKey: true })).to.equal('select');
            });
        });
    });

    // class HeaderBoundary ===================================================

    describe('Spreadsheet class HeaderBoundary', function () {

        it('should exist', function () {
            expect(HeaderBoundary).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var HeaderBoundary = PaneUtils.HeaderBoundary,
            i = new Interval(1, 3),
            p = { offset: 100, size: 10 };

        // constructor --------------------------------------------------------

        it('should create empty instance', function () {
            var b = new HeaderBoundary();
            expect(b).to.have.a.property('interval').that.equals(null);
            expect(b).to.have.a.property('position').that.equals(null);
        });
        it('should create new instance with cloned parameters', function () {
            var b = new HeaderBoundary(i, p);
            expect(b.interval).to.deep.equal(i);
            expect(b.interval).to.not.equal(i);
            expect(b.position).to.deep.equal(p);
            expect(b.position).to.not.equal(p);
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(HeaderBoundary).to.respondTo('clone');
            });
            it('should create a deep clone', function () {
                var b = new HeaderBoundary(), c = b.clone();
                expect(c).to.be.an['instanceof'](HeaderBoundary);
                expect(c).to.not.equal(b);
                expect(c.interval).to.equal(null);
                expect(c.position).to.equal(null);
            });
            it('should create a deep clone', function () {
                var b = new HeaderBoundary(i, p), c = b.clone();
                expect(c).to.be.an['instanceof'](HeaderBoundary);
                expect(c).to.not.equal(b);
                expect(c.interval).to.deep.equal(b.interval);
                expect(c.interval).to.not.equal(b.interval);
                expect(c.position).to.deep.equal(b.position);
                expect(c.position).to.not.equal(b.position);
            });
        });

        describe('method "isValid"', function () {
            it('should exist', function () {
                expect(HeaderBoundary).to.respondTo('isValid');
            });
            it('should return false for empty instance', function () {
                expect(new HeaderBoundary().isValid()).to.equal(false);
            });
            it('should return true for valid instance', function () {
                expect(new HeaderBoundary(i, p).isValid()).to.equal(true);
            });
        });
    });

    // ========================================================================
});
