/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/fontmetrics', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/canvas/canvas'
], function (Utils, LocaleData, Canvas) {

    'use strict';

    var // global helper node for calculation of normal line height of a font
        LINEHEIGHT_NODE = (function () {
            // create the complete HTML mark-up for ten text lines (IE uses fractional line heights internally)
            var helperNode = $('<div class="font-metrics-helper">' + Utils.repeatString('<span>X</span>', 10, '<br>') + '</div>')[0];
            Utils.insertHiddenNodes(helperNode);
            return helperNode;
        }()),

        // global helper node for calculation of the base line position of a font
        BASELINE_NODE = (function () {
            var helperNode = $('<div class="font-metrics-helper"><span>X</span><span style="font-size:0;"></span></div>')[0];
            Utils.insertHiddenNodes(helperNode);
            return helperNode;
        }()),

        // global helper node for splitting text into multiple lines
        TEXTLINES_NODE = (function () {
            var helperNode = $('<div class="font-metrics-helper" style="white-space:pre-wrap;word-wrap:break-word;vertical-align:top;">')[0];
            Utils.insertHiddenNodes(helperNode);
            return helperNode;
        }()),

        // global helper node for detection of writing direction
        BIDI_NODE = (function () {
            var helperNode = $('<div class="font-metrics-helper" style="width:0;"><span style="position:relative;">X</span></div>')[0];
            Utils.insertHiddenNodes(helperNode);
            return helperNode;
        }()),

        // global canvas for font width calculation
        TEXTWIDTH_CANVAS = (function () {
            var canvas = new Canvas();
            // Chrome reports different/wrong text widths, if the canvas element remains detached... :-O
            Utils.insertHiddenNodes(canvas.getNode());
            return canvas;
        }()),

        // global helper node for custom font metrics calculation
        CUSTOM_NODE = (function () {
            var helperNode = $('<div class="font-metrics-helper">')[0];
            Utils.insertHiddenNodes(helperNode);
            return helperNode;
        }()),

        // cache for calculated font metrics, mapped by font descriptor (one cache per instance of this
        // class, to support different fall-back font settings per document imported from operations)
        fontMetricsCache = {};

    // private global functions ===============================================

    /**
     * Fetches, caches, and returns a specific font metrics property. Invokes
     * If a cache key has been specified, and the font metrics cache contains
     * an entry for the passed font descriptor and cache key, that cached value
     * will be returned. Otherwise, the callback function will be invoked, and
     * its result will be cached.
     *
     * @param {FontDescriptor} fontDesc
     *  The font descriptor containing the font settings.
     *
     * @param {Function} callback
     *  The callback function that will be invoked, unless the font metrics
     *  cache already contains a cached value for the font descriptor and cache
     *  key. The return value of this function will be cached if specified (see
     *  parameter 'cacheKey' below), and will be returned by this method.
     *
     * @param {String} [cacheKey]
     *  If specified, a map key for the font metrics cache that will be checked
     *  first. An existing cache entry will be returned without invoking the
     *  callback function. Otherwise, the new value returned by the callback
     *  function will be inserted into the cache.
     *
     * @return {Any}
     *  The return value of the callback function.
     */
    function fetchAndCacheResult(fontDesc, callback, cacheKey) {

        var // the unique key of the passed font descriptor
            fontKey = fontDesc.key(),
            // all known cached metrics settings for the font descriptor
            cacheEntry = _.isString(cacheKey) ? (fontMetricsCache[fontKey] || (fontMetricsCache[fontKey] = {})) : null;

        // look for an existing value in the cache
        if (cacheEntry && (cacheKey in cacheEntry)) {
            return cacheEntry[cacheKey];
        }

        // fetch result from the specified callback function
        var result = callback();

        // insert the result into the cache entry
        if (cacheEntry) { cacheEntry[cacheKey] = result; }
        return result;
    }

    /**
     * Prepares a helper DOM node used to calculate a specific font metrics
     * property, and invokes the passed callback function.
     *
     * @param {FontDescriptor} fontDesc
     *  The font descriptor containing the font settings for the helper node.
     *
     * @param {HTMLElement} node
     *  The DOM element to be prepared with the passed font settings.
     *
     * @param {Function} callback
     *  The callback function that will be invoked after preparing the passed
     *  helper node. The return value of this function will be cached if
     *  specified (see parameter 'cacheKey' below), and will be returned by
     *  this method.
     *
     * @param {String} [cacheKey]
     *  If specified, a map key for the font metrics cache that will be checked
     *  first. An existing cache entry will be returned without invoking the
     *  callback function. Otherwise, the new value returned by the callback
     *  function will be inserted into the cache.
     *
     * @return {Any}
     *  The return value of the callback function.
     */
    function calculateFontMetrics(fontDesc, node, callback, cacheKey) {
        return fetchAndCacheResult(fontDesc, function () {

            // initialize formatting of the helper node
            node.style.fontFamily = fontDesc.family;
            node.style.fontSize = fontDesc.size + 'pt';
            node.style.fontWeight = fontDesc.bold ? 'bold' : 'normal';
            node.style.fontStyle = fontDesc.italic ? 'italic' : 'normal';

            // fetch result from the specified callback function
            return callback(node);

        }, cacheKey);
    }

    /**
     * Prepares the helper canvas used to calculate specific text widths, and
     * invokes the passed callback function.
     *
     * @param {FontDescriptor} fontDesc
     *  The font descriptor containing the font settings for the canvas.
     *
     * @param {Function} callback
     *  The callback function that will be invoked after preparing the helper
     *  helper canvas. Receives the prepared rendering context of the canvas
     *  element as first parameter. Must return the resulting text width that
     *  will be returned to the caller of this method, and that will be stored
     *  in the cache if needed.
     *
     * @param {String} [cacheKey]
     *  If specified, a map key for the text width cache that will be checked
     *  first. An existing cache entry will be returned without invoking the
     *  callback function. Otherwise, the new value returned by the callback
     *  function will be inserted into the cache.
     *
     * @return {Any}
     *  The return value of the callback function.
     */
    function calculateTextWidth(fontDesc, callback, cacheKey) {
        return fetchAndCacheResult(fontDesc, function () {

            var // the result of the callback function
                result = null;

            // initialize font in the rendering context, fetch callback result
            TEXTWIDTH_CANVAS.render(function (context) {
                context.setFontStyle({ font: fontDesc.getCanvasFont() });
                result = callback(context);
            });

            // insert the result into the cache entry
            return result;

        }, cacheKey);
    }

    // static class FontMetrics ===============================================

    /**
     * Provides helper functions to receive specific font measurements, such as
     * the normal line height, the position of the font base line, or the exact
     * width of arbitrary text.
     */
    var FontMetrics = {};

    // constants --------------------------------------------------------------

    /**
     * A flag specifying whether the automatic detection of the default writing
     * direction of arbitrary texts is supported by the current browser.
     * Technically, this flag specifies whether the HTML element attribute
     * 'dir' supports the value 'auto'.
     *
     * @constant
     */
    FontMetrics.AUTO_BIDI_SUPPORT = (function () {
        try {
            // IE and PhantomJS throw when executing this assignment
            BIDI_NODE.dir = 'auto';
            return true;
        } catch (ex) {
            Utils.warn('FontMetrics: no support for automatic detection of writing direction');
        }
        return false;
    }());

    // static methods ---------------------------------------------------------

    /**
     * Calculates the normal line height for the specified font, as reported by
     * the current browser. Note that this value may vary for the same font in
     * different browsers.
     *
     * @param {FontDescriptor} fontDesc
     *  A descriptor with the font settings influencing the normal line height
     *  of the font.
     *
     * @returns {Number}
     *  The normal line height for the specified font, in pixels (with a
     *  precision of 1/10 pixel for browsers using fractional line heights
     *  internally).
     */
    FontMetrics.getNormalLineHeight = function (fontDesc) {
        return calculateFontMetrics(fontDesc, LINEHEIGHT_NODE, function (node) {
            return node.offsetHeight / 10;
        }, 'int.lineheight');
    };

    /**
     * Calculates the position of the font base line for the specified font and
     * line height, as reported by the current browser. Note that this value
     * may vary for the same font in different browsers.
     *
     * @param {FontDescriptor} fontDesc
     *  A descriptor with the font settings influencing the position of the
     *  font base line.
     *
     * @param {Number} [lineHeight]
     *  The line height to be used to determine the position of the font base
     *  line. If omitted, the normal line height, as reported by the method
     *  FontMetrics.getNormalLineHeight(), will be used.
     *
     * @returns {Number}
     *  The position of the font base line for the specified font and line
     *  height, in pixels, relative to the top border of a text line.
     */
    FontMetrics.getBaseLineOffset = function (fontDesc, lineHeight) {

        var // CSS line height with unit, or 'normal'; also used as cache key
            cssLineHeight = _.isNumber(lineHeight) ? (Math.round(lineHeight) + 'px') : 'normal';

        return calculateFontMetrics(fontDesc, BASELINE_NODE, function (node) {
            node.style.lineHeight = cssLineHeight;
            return node.lastChild.offsetTop;
        }, 'int.baseline.' + cssLineHeight);
    };

    /**
     * Returns the exact width of the passed text, in pixels.
     *
     * @param {FontDescriptor} fontDesc
     *  A descriptor with the font settings influencing the text width.
     *
     * @param {String} text
     *  The text whose width will be calculated. If the text consists of one
     *  character only, the calculated text width will be cached internally for
     *  subsequent invocations with the same font settings.
     *
     * @return {Number}
     *  The exact width of the passed text, in pixels.
     */
    FontMetrics.getTextWidth = function (fontDesc, text) {

        switch (text.length) {
        case 0:
            // empty string: always zero
            return 0;

        case 1:
            // single character: calculate width of 10 characters for higher precision, store in cache
            return calculateTextWidth(fontDesc, function (context) {
                return context.getCharacterWidth(text);
            }, 'int.textwidth.' + text);

        default:
            // multiple characters: calculate text width, do not store in cache
            return calculateTextWidth(fontDesc, function (context) {
                return context.getTextWidth(text);
            });
        }
    };

    /**
     * Returns the largest text width of the digits '0' to '9', in pixels.
     *
     * @param {FontDescriptor} fontDesc
     *  A descriptor with the font settings influencing the text width of the
     *  digits.
     *
     * @return {Number}
     *  The largest width of any of the digits '0' to '9', in pixels.
     */
    FontMetrics.getDigitWidth = function (fontDesc) {
        return calculateTextWidth(fontDesc, function (context) {
            return _.reduce('0123456789', function (width, digit) {
                return Math.max(width, context.getCharacterWidth(digit));
            }, 0);
        }, 'int.digitwidth');
    };

    /**
     * Splits the passed string to text lines that all fit into the passed
     * pixel width, using the specified font.
     *
     * @param {FontDescriptor} fontDesc
     *  A descriptor with the font settings influencing the text width.
     *
     * @param {String} text
     *  The text that will be split into single text lines.
     *
     * @param {Number} width
     *  The maximum width of the text lines, in pixels.
     *
     * @returns {Array<String>}
     *  An array of strings containing the single text lines, all fitting into
     *  the specified text line width.
     */
    FontMetrics.getTextLines = function (fontDesc, text, width) {

        // empty strings, or single-character strings cannot be split
        if (text.length <= 1) { return [text]; }

        // use a DOM helper node to take advantage of text auto-wrapping provided by the browser
        return calculateFontMetrics(fontDesc, TEXTLINES_NODE, function (node) {

            var // the resulting substrings
                textLines = [];

            // set the passed width at the helper node
            node.style.width = width + 'px';

            // create the HTML mark-up, wrap each character into its own span element
            // (for performance: with this solution, the DOM will be modified once,
            // and all node offsets can be read without altering the DOM again)
            node.innerHTML = Utils.escapeHTML(text).replace(/&(#\d+|\w+);|./g, '<span>$&</span>');

            var // all DOM span elements
                spanNodes = node.childNodes,
                // shortcut to the number of spans
                spanCount = spanNodes.length,
                // start index of the current text line
                lineIndex = 0,
                // vertical position of the spans in the current text line
                lineOffset = spanNodes[0].offsetTop,
                // vertical position of the spans in the last text line
                maxOffset = spanNodes[spanCount - 1].offsetTop;

            // returns the index of the first span element located in the next text line
            function findStartOfNextTextLine() {

                var // index increment size size while searching forwards for next text lines
                    increment = 8,
                    // start of the search interval (index in the current text line)
                    beginIndex = lineIndex,
                    // end of the search interval (search until this index is in one of the following text lines)
                    endIndex = Math.min(spanCount, beginIndex + increment);

                // search for a span element located in any of the next text lines,
                // by picking a few following spans with increasing increment value
                while ((endIndex < spanCount) && (spanNodes[endIndex].offsetTop === lineOffset)) {
                    beginIndex = endIndex + 1;
                    increment *= 2;
                    endIndex = Math.min(spanCount, beginIndex + increment);
                }

                // binary search for the first span element in the resulting search interval
                return Utils.findFirstIndex(spanNodes, function (span) {
                    return lineOffset < span.offsetTop;
                }, { begin: beginIndex, end: endIndex, sorted: true });
            }

            // process all text lines but the last text line
            while (lineOffset < maxOffset) {

                var // index of the first span element in the next text line
                    nextIndex = findStartOfNextTextLine();

                // append the text of the current text line to the result array
                textLines.push(text.substring(lineIndex, nextIndex));

                // continue with the next text line
                lineIndex = nextIndex;
                lineOffset = spanNodes[lineIndex].offsetTop;
            }

            // push the remaining last text line to the result array
            if (lineIndex < text.length) {
                textLines.push(text.substr(lineIndex));
            }

            return textLines;
        });
    };

    /**
     * Provides the ability to calculate and cache custom font metrics.
     *
     * @param {FontDescriptor} fontDesc
     *  A descriptor with the font settings influencing the font metrics.
     *
     * @param {Function} callback
     *  The callback function that will be invoked after preparing the helper
     *  node.  Receives the following parameters:
     *  (1) {HTMLDivElement} node
     *      The helper element that can be used to calculate the result.
     *  The return value of this function will be cached if specified (see
     *  parameter 'cacheKey' below), and will be returned by this method.
     *
     * @param {String} [cacheKey]
     *  If specified, a unique map key for the font metrics cache that will be
     *  checked first. An existing cache entry will be returned without
     *  invoking the callback function. Otherwise, the new value returned by
     *  the callback function will be inserted into the cache. If omitted, the
     *  callback function will always be invoked, and the result will not be
     *  inserted into the cache.
     *
     * @return {Any}
     *  The font metric property from the cache, or returned by the callback.
     */
    FontMetrics.getCustomFontMetrics = function (fontDesc, callback, cacheKey) {
        cacheKey = _.isString(cacheKey) ? ('ext.' + cacheKey) : null;
        return calculateFontMetrics(fontDesc, CUSTOM_NODE, function (node) {
            var result = callback(node);
            node.innerHTML = '';
            node.removeAttribute('style');
            return result;
        }, cacheKey);
    };

    /**
     * Determines the default writing direction of the passed text, based on
     * the first strong character in the text (that is significant for
     * bidirectional writing), and the current browser's implementation of the
     * Unicode Bidirectional Algorithm (UBA).
     *
     * @param {String} text
     *  The text to be examined.
     *
     * @return {String}
     *  The string 'ltr', if the text should be written from left to right; or
     *  'rtl', if the text should be written from right to left.
     */
    FontMetrics.getTextDirection = (function () {

        // fall-back if automatic detection is not supported
        if (!FontMetrics.AUTO_BIDI_SUPPORT) {
            return _.constant(LocaleData.DIR);
        }

        var // the span element containing the text to be examined
            spanNode = BIDI_NODE.firstChild,
            // the text node to be filled with the actual text
            textNode = spanNode.firstChild;

        // the actual method getTextDirection() to be returned from local scope
        function getTextDirection(text) {

            // do not touch the DOM for empty strings
            if (text.length === 0) { return LocaleData.DIR; }

            // insert passed text; if the span element floats out of its parent to the left, is is right-to-left
            textNode.nodeValue = text;
            return (spanNode.offsetLeft < 0) ? 'rtl' : 'ltr';
        }

        return getTextDirection;
    }());

    // exports ================================================================

    return FontMetrics;

});
