/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/fontdescriptor', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // class FontDescriptor ===================================================

    /**
     * Helper structure used to transport font information. Used by the static
     * class FontMetrics to calculate font metrics for a specific font, and by
     * the class FontCollection to convert character formatting properties to
     * font information.
     *
     * @property {String} family
     *  The CSS font family, including all known fall-back fonts.
     *
     * @property {Number} size
     *  The font size, in points, rounded to 1/10 of points. Note that the unit
     *  string 'pt' has to be added to this number when setting this font size
     *  as CSS property.
     *
     * @property {Boolean} bold
     *  Whether the font is set to bold characters.
     *
     * @property {Boolean} italic
     *  Whether the font is set to italic characters.
     */
    function FontDescriptor(family, size, bold, italic) {

        this.family = family.toLowerCase();
        this.size = Utils.round(size, 0.1);
        this.bold = bold === true;
        this.italic = italic === true;

    } // class FontDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns a unique string key for this font descriptor that can be used
     * for example as key in an associative map.
     *
     * @returns {String}
     *  A unique string key for this font descriptor.
     */
    FontDescriptor.prototype.key = function () {
        return this.family + '|' + this.size + '|' + ((this.bold ? 1 : 0) + (this.italic ? 2 : 0));
    };

    /**
     * Converts the font settings in this descriptor to a string suitable to be
     * set as font style at an HTML canvas element.
     *
     * @returns {String}
     *  The font settings in this descriptor, as HTML canvas font style.
     */
    FontDescriptor.prototype.getCanvasFont = function () {
        return (this.bold ? 'bold ' : '') + (this.italic ? 'italic ' : '') + this.size + 'pt ' + this.family;
    };

    // exports ================================================================

    return FontDescriptor;

});
