/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/locale/formatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/object/baseobject'
], function (Utils, DateUtils, LocaleData, Parser, BaseObject) {

    'use strict';

    var // the exponent character
        EXP = 'E',

        // zero characters, used for formatting numbers with leading zeros
        ZEROS = Utils.repeatString('0000000000000000', 16),

        // number of milliseconds per day
        MSEC_PER_DAY = 86400000;

    // private static functions ===============================================

    /**
     * Formats the passed integral number with leading zero characters.
     *
     * @param {Number} number
     *  The number to be formatted.
     *
     * @param {Number} length
     *  The target length of the formatted number. If the resulting number is
     *  shorter, leading zero characters will be inserted; if it is already
     *  longer, it will NOT be truncated.
     *
     * @returns {String}
     *  The formatted number with leading zero characters as needed.
     */
    function formatInteger(number, length) {
        var text = String(Math.floor(Math.abs(number)));
        return (text.length < length) ? (ZEROS + text).substr(-length) : text;
    }

    /**
     * Returns the text representation of the year in the passed date.
     *
     * @param {Date} date
     *  The UTC date to be formatted as year.
     *
     * @param {Number} length
     *  The length of the formatted year: 2 for short years, 4 for full
     *  years.
     *
     * @returns {String}
     *  The formatted year of the passed date.
     */
    function formatYear(date, length) {
        return (ZEROS + date.getUTCFullYear()).substr(-length);
    }

    /**
     * Returns the text representation of the month in the passed date.
     *
     * @param {Date} date
     *  The UTC date to be formatted as month.
     *
     * @param {Number} length
     *  The length of the formatted month: 1 for short month number, 2 for
     *  month number with leading zero, 3 for abbreviated month name, 4 for
     *  full month name, or 5 for the first character of the month name.
     *
     * @returns {String}
     *  The formatted month of the passed date.
     */
    function formatMonth(date, length) {
        var month = date.getUTCMonth();
        switch (length) {
        case 1:
            return String(month + 1);
        case 3:
            return DateUtils.getLocaleMonthShort(month);
        case 4:
            return DateUtils.getLocaleMonth(month);
        case 5:
            return DateUtils.getLocaleMonthShort(month)[0];
        default:
            return formatInteger(month + 1, length);
        }
    }

    /**
     * Returns the text representation of the day in the passed date.
     *
     * @param {Date} date
     *  The UTC date to be formatted as day.
     *
     * @param {Number} length
     *  The length of the formatted day: 1 for short day number, 2 for day
     *  number with leading zero, 3 for abbreviated name of week day, or 4
     *  for full name of week day.
     *
     * @returns {String}
     *  The formatted day of the passed date.
     */
    function formatDay(date, length) {
        switch (length) {
        case 1:
            return String(date.getUTCDate());
        case 3:
            return DateUtils.getLocaleDayShort(date.getUTCDay());
        case 4:
            return DateUtils.getLocaleDay(date.getUTCDay());
        default:
            return formatInteger(date.getUTCDate(), length);
        }
    }

    // class Formatter ========================================================

    /**
     * A formatter converts numbers and other values to strings, according to
     * specific format codes.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Date} [initOptions.nullDate]
     *      The null date used by this formatter (the number 0 formatted as
     *      date/time will result in this date). By default, the date
     *      1899-11-30 will be used as null date.
     *  @param {Boolean} [initOptions.negativeDates=false]
     *      Whether to support negative date values (e.g. number -1 is one day
     *      before the null date).
     */
    function Formatter(initOptions) {

        var // self reference
            self = this,

            // null date (corresponding to number zero)
            nullDate = Date.UTC(1899, 11, 30),

            // whether to support negative date values
            negativeDates = false;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the decimal string representation of the passed number.
         *
         * @param {Number} number
         *  The number whose decimal string representation will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed number
         *  will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete decimal string representation of the number.
         *  - {String} abs
         *      The decimal string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      number.
         */
        function formatAsDecimal(number, digits) {

            var // the absolute number, rounded to the specified digit count after decimal separator
                round = Utils.round(Math.abs(number), Math.pow(10, -digits)),
                // the integral part of the absolute number
                int = Math.floor(round),
                // the fractional part of the absolute number
                frac = Math.round((round - int) * Math.pow(10, digits)),
                // the text representation of the integral part
                intText = String(int),
                // the text representation of the significant fractional part (without leading zeros)
                fracText = String(frac),
                // the zeros between decimal separator and significant fractional part
                fillText = null,
                // the result returned by this method
                result = { abs: intText, digits: (int === 0) ? 0 : intText.length };

            // add the rounded number to the result
            result.number = (number < 0) ? -round : round;

            // no fraction available: do not add decimal separator
            if (frac > 0) {

                // leading zeros for the fractional part
                fillText = Utils.repeatString('0', digits - fracText.length);
                // remove trailing zeros
                fracText = fracText.replace(/0+$/, '');
                // concatenate integer and fractional part, remove trailing zeros
                result.abs += LocaleData.DEC + fillText + fracText;
                // update number of significant digits
                if (int > 0) { result.digits += fillText.length; }
                result.digits += fracText.length;
            }

            // add final string representation (with sign)
            result.text = (number < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Returns the scientific string representation of the passed number.
         *
         * @param {Number} mant
         *  The mantissa of the number whose scientific string representation
         *  will be returned. Must be in the half-open intervals +-[1, 10).
         *
         * @param {Number} exp
         *  The exponent of the number whose scientific string representation
         *  will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed
         *  mantissa will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete scientific string representation of the number.
         *  - {String} abs
         *      The scientific string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      mantissa.
         */
        function formatAsScientific(mant, exp, digits) {

            var // get the formatted mantissa
                result = formatAsDecimal(mant, digits);

            // if mantissa has been rounded up to 10, increase the exponent
            if (Math.abs(result.number) === 10) {
                result.number /= 10;
                result.abs = '1';
                exp += 1;
            }

            // update the rounded number in the result
            result.number *= Math.pow(10, exp);

            // add the exponent (at least two digits, but may be three digits)
            result.abs += EXP + ((exp < 0) ? '-' : '+');
            exp = Math.abs(exp);
            if (exp < 10) { result.abs += '0'; }
            result.abs += String(exp);

            // add final string representation (with sign)
            result.text = (mant < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Formats a number with increasing number of digits following the
         * decimal separator, as long as the resulting number fits according to
         * the provided result validator.
         *
         * @param {Number} minDigits
         *  The minimum number of digits after the decimal separator.
         *
         * @param {Number} maxDigits
         *  The maximum number of digits after the decimal separator.
         *
         * @param {Function} formatCallback
         *  The callback function that receives the current number of digits
         *  following the decimal separator, and must return a result object
         *  containing the properties 'number', 'text', 'abs', and 'digits'.
         *  See methods formatAsDecimal() and formatAsScientific() above for a
         *  detailed description about this result object.
         *
         * @param {Function} validateCallback
         *  The callback function that checks the current result object it
         *  receives. Must return a Boolean value specifying whether the result
         *  object is valid.
         *
         * @returns {Object}
         *  The last matching result object returned by the callback function.
         */
        function findLongestFormat(minDigits, maxDigits, formatCallback, validateCallback) {

            var currDigits = minDigits,
                currResult = null,
                lastResult = null;

            do {
                currResult = formatCallback(currDigits);
                if (!validateCallback(currResult)) { break; }
                lastResult = currResult;
                currDigits += 1;
            } while (currDigits <= maxDigits);

            return lastResult;
        }

        // public methods -----------------------------------------------------

        /**
         * Changes the configuration of this formatter.
         *
         * @param {Object} [options]
         *  All configuration properties to be changed. See constructor
         *  parameter 'initOptions' for details.
         *
         * @returns {Formatter}
         *  A reference to this instance.
         */
        this.configure = function (options) {

            // change the null date
            if (_.isObject(options) && (options.nullDate instanceof Date)) {
                nullDate = options.nullDate;
            }

            // change the negative date mode
            negativeDates = Utils.getBooleanOption(options, 'negativeDates', negativeDates);

            return this;
        };

        /**
         * Returns whether the passed date object is valid to be processed by
         * the formatter. The year must be in the range from 0 to 9999, and
         * must not be located before the null date (see constructor option
         * 'nullDate' for details), unless such dates are supported by this
         * formatter (see constructor option 'negativeDates' for details).
         *
         * @param {Date} date
         *  The UTC date to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed date object is valid.
         */
        this.isValidDate = function (date) {

            var // milliseconds from 1970-01-01
                time = date.getTime(),
                // the year
                year = date.getUTCFullYear();

            // check the year, check for 'negative' dates
            return isFinite(time) && (negativeDates || (nullDate <= time)) && (year >= 0) && (year <= 9999);
        };

        /**
         * Returns a date object representing the passed floating-point number,
         * using the current null date of this formatter.
         *
         * @param {Number} number
         *  A floating-point number. The integral part represents the number of
         *  days elapsed since the null date of this formatter, the fractional
         *  part represents the time.
         *
         * @returns {Date|Null}
         *  An UTC date object representing the passed number; or null, if the
         *  passed number cannot be converted to a valid date (e.g. too large).
         */
        this.convertNumberToDate = function (number) {

            var // get correct date/time for current null date
                date = new Date(nullDate + Math.round(number * MSEC_PER_DAY));

            // return the date object if it is valid
            return this.isValidDate(date) ? date : null;
        };

        /**
         * Returns the floating-point number representing the passed date,
         * using the current null date of this formatter.
         *
         * @param {Date} date
         *  An UTC date object.
         *
         * @returns {Number|Null}
         *  The floating-point number representing the passed date; or null, if
         *  the passed date cannot be converted to a number.
         */
        this.convertDateToNumber = function (date) {

            // error, if date is invalid, or the year is not in the range 0 to 9999
            if (!this.isValidDate(date)) { return null; }

            // return the floating-point number
            return (date.getTime() - nullDate) / MSEC_PER_DAY;
        };

        /**
         * Converts the passed text to a floating-point number, if possible.
         * The text may represent a floating-point number in decimal or
         * scientific notation, a percentage (a floating-point number with
         * trailing percent sign), a currency value, or a date, time, or
         * combined date/time value which will be converted to the serial
         * number according to the null date of this formatter.
         *
         * @param {String} text
         *  The string to be converted to a floating-point number.
         *
         * @returns {Object|Null}
         *  A result descriptor, if the passed string starts with a valid
         *  floating-point number (according to the passed options); otherwise
         *  null. The result descriptor contains the following properties:
         *  - {Number} number
         *      The resulting floating-point number.
         *  - {String} category
         *      The number format category the number in the passed text has
         *      been formatted with:
         *      - 'number': Floating-point number (decimal or scientific).
         *      - 'percent': Floating-point number with trailing percent sign.
         *      - 'currency': Floating-point number with leading or trailing
         *          currency symbol.
         *      - 'date': A date, time, or combined date/time.
         */
        this.parseFormattedNumber = function (text) {

            // try to parse a leading floating-point number (decimal or scientific), TODO: group separator
            var parseResult = Parser.parseLeadingNumber(text, { sign: true });
            if (_.isObject(parseResult)) {

                // text consists entirely of the floating-point number
                if (parseResult.remaining.length === 0) {
                    return { number: parseResult.number, category: 'number' };
                }

                // text consists of a floating-point number with trailing percent sign (white-space allowed)
                if (/^\s*%$/.test(parseResult.remaining)) {
                    return { number: parseResult.number / 100, category: 'percent' };
                }

                // reject any other trailing garbage text
                return null;
            }

            // TODO: parse currency, date/time, fractions, etc.

            return null;
        };

        /**
         * Returns the display string of the passed number formatted with the
         * 'General' number format, fitting into the specified text width.
         *
         * @param {Number} number
         *  The number whose string representation in 'General' number format
         *  will be returned.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation.
         *
         * @returns {String|Null}
         *  The resulting display text of the number, formatted with the
         *  'General' number format. If the passed maximum length is too small
         *  to take the passed number in either decimal or scientific notation,
         *  the null value will be returned instead.
         */
        this.formatStandardNumber = function (number, maxLength) {

            var // the mantissa and exponent (-INF for zero) of the passed number
                normalized = Utils.normalizeNumber(number),
                // available, minimum, and maximum number of digits for the passed width
                availDigits = 0, minDigits = 0, maxDigits = 0,
                // the formatter callback function (decimal or scientific)
                formatterFunc = null,
                // conversion result in decimal notation
                decResult = null,
                // conversion result in scientific notation
                scientResult = null,
                // the result to be used (either decimal or scientific)
                finalResult = null;

            // validates the passed result by character count only
            function validateLength(result) {
                return result.abs.length <= maxLength;
            }

            // find the best decimal representation
            if (normalized.exp < 2 - maxLength) {
                // start with zero representation for numbers equal to or very close to zero
                decResult = { number: number, abs: '0', text: (number < 0) ? '-0' : '0', digits: 0 };
            } else if (normalized.exp <= maxLength - 1) {
                // try to find the best explicit representation in a specific interval of exponents
                availDigits = maxLength;
                minDigits = Math.max(0, -(normalized.exp + 2));
                maxDigits = Math.min(15, Math.max(0, 14 - normalized.exp), maxLength - Math.max(2, normalized.exp), availDigits - Math.max(0, normalized.exp) - 1);
                formatterFunc = _.partial(formatAsDecimal, number);
                decResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateLength);
            }

            // find the best scientific representation (exclude zero)
            if (isFinite(normalized.exp) && ((normalized.exp <= -4) || (normalized.exp >= 4))) {
                availDigits = maxLength - 5;
                minDigits = Utils.minMax(availDigits - 2, 0, 13);
                maxDigits = Utils.minMax(availDigits - 1, 0, 14);
                formatterFunc = _.partial(formatAsScientific, normalized.mant, normalized.exp);
                scientResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateLength);
            }

            // prefer decimal notation, if it shows at least as much significant digits as the scientific notation
            finalResult = (decResult && (!scientResult || (decResult.digits >= scientResult.digits))) ? decResult : scientResult;
            return finalResult ? finalResult.text : null;
        };

        /**
         * Formats the passed value according to the format code.
         *
         * @param {Number|String} value
         *  The value to be formatted.
         *
         * @param {String} formatCode
         *  The format code used to format the passed value.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.generalLength=11]
         *      Maximum number of characters used to format a number for the
         *      'General' format code token, including the decimal separator
         *      and the complete exponent in scientific notation, but without
         *      the minus sign.
         *
         * @returns {String|Null}
         *  The formatted value; or null, if the passed format code is invalid,
         *  or if the value cannot be formatted with the format code (e.g., a
         *  number is too large to be formatted as date/time).
         */
        this.formatValue = function (value, formatCode, options) {

            /*
             * fills up the "text" number with whitespace to the length of the assigned code
             * iterate the code and the text and does:
             *      code: ? text: '0' -> ' '
             *      code: # text: '0' -> delete
             *      code: 0 text: ' ' -> '0'
             *      code: 0-9 -> break iteration
             *
             * @param {Boolean} [rightToLeft=false]
             *      if true the text is iterated from right to left (only used in significant part of the number)
             */
            function setPlaceHolders(text, code, rightToLeft) {

                if (_.isNumber(text)) {
                    text = String(text);
                }
                if (text.length > code.length) {
                    return text;
                }

                var i, counter, end;
                if (rightToLeft) {
                    text = text + Utils.repeatString(' ', code.length - text.length);
                    end = -1;
                    counter = -1;
                    i = text.length - 1;
                } else {
                    text = Utils.repeatString(' ', code.length - text.length) + text;
                    i = 0;
                    counter = 1;
                    end = text.length;
                }

                var arrT = [];
                for (var j = 0; j < text.length; j++) {
                    arrT.push(text[j]);
                }

                for (; i !== end; i += counter) {
                    var actT = arrT[i];
                    var actC = code[i];

                    if (actT !== '0' && actT !== ' ') {
                        break;
                    }
                    if (actC === '0') {
                        arrT[i] = '0';
                    } else {
                        if (actC === '?') {
                            arrT[i] = ' ';
                        } else if (actC === '#') {
                            arrT[i] = '';
                        }
                    }
                }
                return arrT.join('');
            }

            // Reduce a fraction by finding the Greatest Common Divisor and dividing by it.
            // see http://stackoverflow.com/questions/4652468/is-there-a-javascript-function-that-reduces-a-fraction
            function cancelFract(frac) {
                var num = frac.num;
                var deno = frac.deno;

                var gcd = function gcd(a, b) {
                    return b ? gcd(b, a % b) : a;
                };
                gcd = gcd(num, deno);

                frac.num = num / gcd;
                frac.deno = deno / gcd;
            }

            // format of the integer part of the number
            function formatNumber(token) {

                var text = String(section.fractional ? Math.floor(number) : Math.round(number));

                if (/,/.test(token.text)) {
                    var replacer = '$1' + LocaleData.GROUP + '$2';
                    var rgx = /(\d+)(\d{3})/;
                    while (rgx.test(text)) {
                        text = text.replace(rgx, replacer);
                    }
                }

                text = setPlaceHolders(text, token.text.replace(/\,/g, ''));

                return text;
            }

            //format of the fraction part numerator & denominator
            //at the moment we only support until 3 signs for the denominator
            function formatFraction(token) {
                var parts = token.text.split('/');

                var denomLen = parts[1].length;
                if (denomLen > 3) {
                    parts[1] = parts[1].substring(0, 3);
                    denomLen = 3;
                }
                var fractStart = Math.pow(10, denomLen - 1);
                var fractEnd = fractStart * 10 - 1;

                var frac = number - Math.floor(number);

                var closest = null;
                var near = Number.POSITIVE_INFINITY;

                if (frac === 0) {
                    closest = { num: 0, deno: fractEnd };
                } else {
                    for (var i = fractStart; i <= fractEnd; i++) {
                        var num = Math.round(frac * i);
                        var compare = num / i;
                        var dif = Math.abs(frac - compare);
                        if (dif < near) {
                            closest = { num: num, deno: i };
                            near = dif;
                            if (dif === 0) { break; }
                        }
                    }
                }

                if (closest) {
                    if (parts[1].indexOf('?') >= 0) {
                        if (frac === 0) {
                            //placeholder complete fraction only white spaces!!!
                            return setPlaceHolders('', parts[0]) + ' ' + setPlaceHolders('', parts[1]);
                        }
                        cancelFract(closest);
                    }
                    closest.num = setPlaceHolders(closest.num, parts[0]);
                    closest.deno = setPlaceHolders(closest.deno, parts[1]);

                    return closest.num + '/' + closest.deno;
                }
                return LocaleData.DEC + frac;
            }

            //format the significant part of the number incl. decimal-symbol
            function formatSign(token) {
                var format = token.text.substring(1, token.text.length);
                var digits = format.length;

                var frac = number - parseInt(number);
                var round = Utils.round(frac, Math.pow(10, -digits));

                var result = String(round).substring(2);
                var res = setPlaceHolders(result, format, true);
                if (res.length === 0) { return ''; }
                if (res[0] === ' ') { return ' ' + res; }
                return LocaleData.DEC + res;
            }

            function formatScience(token) {
                var res = setPlaceHolders(Math.abs(exp), token.text);
                return EXP + ((exp < 0) ? '-' : '+') + res;
            }

            var // the section of the format code to be used for the passed value
                section = Parser.parseFormatCode(formatCode).getSection(value),
                // the absolute value of the passed number, or zero for strings
                number = _.isNumber(value) ? Math.abs(value) : 0,
                // the passed number, converted to a date
                date = _.isNumber(value) ? this.convertNumberToDate(value) : null,
                // the passed string, or empty string for numbers
                string = _.isString(value) ? value : '',
                // number of characters for a 'General' format token
                generalLength = Utils.getIntegerOption(options, 'generalLength', 11, 5),
                // the formatted string representation
                formatted = '';

            // bail out if the passed format code is invalid, or cannot be used to
            // format the value, or if the passed number is not finite
            if (!section || !isFinite(number)) { return null; }

            // bail out if the passed format code contains date/time tokens,
            // but no valid date is available
            if (section.dateTime && !date) { return null; }

            // prepare percentage formats (number may become infinite)
            if (section.percent > 0) {
                number *= Math.pow(100, section.percent);
                if (!isFinite(number)) { return null; }
            }

            // prepare scientific formats
            var exp = 0;
            if (section.scientific && (number !== 0)) {
                var normalized = Utils.normalizeNumber(number);
                exp = normalized.exp;
                number = normalized.mant;
            }

            // initialize token formatters (called with token as context)
            var tokenFormatters = {
                    general: function () { return self.formatStandardNumber(number, generalLength); },
                    nr: formatNumber,
                    frac: formatFraction,
                    sign: formatSign,
                    scien: formatScience,
                    year: function (token) { return formatYear(date, token.length); },
                    month: function (token) { return formatMonth(date, token.length); },
                    day: function (token) { return formatDay(date, token.length); },
                    hour: function (token) { return formatInteger(section.hasAmPm ? (((date.getUTCHours() + 11) % 12) + 1) : date.getUTCHours(), token.length); },
                    minute: function (token) { return formatInteger(date.getUTCMinutes(), token.length); },
                    second: function (token) { return formatInteger(date.getUTCSeconds(), token.length); },
                    hours: function (token) { return formatInteger(Math.floor(number * 24), token.length); },
                    minutes: function (token) { return formatInteger(Math.floor(number * 1440), token.length); },
                    seconds: function (token) { return formatInteger(Math.round(number * 86400), token.length); },
                    ampm: function (token) { return ((date.getUTCHours() < 12) ? 'am' : 'pm').substr(0, token.length); },
                    lit: function (token) { return token.text; },
                    fill: function () { return ''; }, // bug 36439: ignore fill tokens silently for now
                    text: function () { return string; }
                };

            // process all format code tokens
            section.tokens.forEach(function (token) {
                formatted += tokenFormatters[token.type](token);
            });

            // automatically add a minus sign for code sections covering positive and negative numbers
            if (section.autoMinus && _.isNumber(value) && (value < 0) && !section.dateTime) {
                formatted = '-' + formatted;
            }

            return formatted;
        };

        // initialization -----------------------------------------------------

        // initialize with options apssed to constructor
        this.configure(initOptions);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = initOptions = null;
        });

    } // class Formatter

    // exports ================================================================

    return BaseObject.extend({ constructor: Formatter });

});
