/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/tk/control/colorslider', [
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/group',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/utils'
], function (Forms, Group, Tracking, Utils) {

    'use strict';

    // class colorSlider ===========================================================

    /**
     * Creates a color slider element with markup and event handling.
     *
     * @extends Group
     *
     * @param {Array} colorFieldsColorData
     * Scheme color values in the same order as the buttons
     * are shown in the mobile color picker.
     *
     * @param {Object} [initOptions]
     * Optional Parameters. Supports all options of the Group base class.
     *
     * @param {Number} [initOptions.defaultHandlePosition=2]
     * The default position from the handle.
    */

    function ColorSlider(colorFieldsColorData, initOptions) {

        var
            self = this,

            // default position on the given colorNumber for the handle
            defaultHandlePosition = Utils.getIntegerOption(initOptions, 'defaultHandlePosition', 2),

            // range definitions to detect to color under the handle. Same order as in colorpicker.js SCHEME_COLOR_DEFINITIONS.
            colorFieldsRangeData = [],

            // colorSlider handle
            handle = $(Forms.createElementMarkup('div', { attributes: { 'class': 'colorSliderHandle' }, content: '' })),

            // the whole colorSlider mark-up
            sliderUI = null,

            // container with all colorFields
            colorSlider = null,

            // the currently shown colorNumber
            currentColor = defaultHandlePosition,

            // used to show that the handle is currently dragged
            handleMovedFlag = false,

            // used to show that the handle is moved via tab on the slider
            tapOnSliderFlag = false,

            // offset in the dragged element to prevent a jumping handle
            xOffset = null,

            // ColorField width. Must have the same value as the width in the less definition (.colorSliderField.width).
            colorFieldWidth = 28,

            // Margin from the colorSlider. Must have the same value as the width in the less definition (.colorSlider.margin-left).
            marginLeft = 12,

            // Width from the handle border. Must have the same value as the width in the less definition (.colorSliderHandle.border)
            handleBorderWidth = 10,

            // needed to correct an offset when marinLeft and handleBorderWidth are not equal
            adjustHandleOffset = (marginLeft - handleBorderWidth),

            // limit the handle leftwards
            handleStartPos = 0 + adjustHandleOffset,

            // limit the handle rightwards
            handleEndPos = colorFieldWidth * 9 + adjustHandleOffset;

        // base constructor ---------------------------------------------------

        Group.call(this, initOptions);
        // -------

        /**
         * Create the data for createColorFieldsRangeData
         */
        function createColorFieldsRangeData () {
            colorFieldsRangeData = _.times(colorFieldsColorData.length, function (i) {
                return { range1: colorFieldWidth * i + marginLeft, range2: colorFieldWidth * (i + 1) + marginLeft };
            });
        }

        /**
         *  Create and returns the markup for all colorFields
         */
        function createColorFields () {
            var colorFieldsMarkUp = '';

            _.times(colorFieldsColorData.length, function (i) {
                colorFieldsMarkUp += Forms.createElementMarkup('div', { attributes: { 'class': 'colorSliderField' }, style: 'background: #' + colorFieldsColorData[i] + ';', content: i });
            });
            return colorFieldsMarkUp;
        }

        /**
         *  Creates the markup for the whole colorSlider
         */
        function createColorSlider() {
            var container = $(Forms.createElementMarkup('div', { attributes: { 'class': 'colorSliderContainer' }, content: '' })),
                colorFields = createColorFields();

            colorSlider = $(Forms.createElementMarkup('div', { attributes: { 'class': 'colorSlider' }, content: colorFields }));
            sliderUI = $(container).append(colorSlider, handle);
            moveHandleToColor(currentColor);
        }

        /**
         *  Creates the event listener for drag&drop and tapping interaction.
         */
        function setListener () {
            // enable tracking and register tracking event handlers
            Tracking.enableTracking(sliderUI);
            sliderUI.on('tracking:start tracking:move tracking:end', trackingHandler);
        }

        /**
         *  Event handling for drag&drop and tapping.
         */
        function trackingHandler(event) {
            switch (event.type) {

            case 'tracking:start':
                // to prevent a delayed click after touchstart - necessary due to prevent.default in tracking:move
                event.preventDefault();
                // case: drag&drop - when start target is the handle
                if (event.target === handle[0]) {
                    // offset in the handle: pointerposition in the handle remains the same while dragged
                    xOffset = (event.pageX - handle[0].offsetLeft);
                    // flag that the slider can be dragged
                    handleMovedFlag = true;
                    // remove the transition for a delay free drag&drop
                    handle.removeClass('transition');
                }
                // case: tap on the slider to select a color
                if (event.target === colorSlider[0]) {
                    // flag to indicate that we have a tap interaction
                    tapOnSliderFlag = true;
                }
                break;

            case 'tracking:move':
                // to prevent events on iOS which triggered under the menu
                event.preventDefault();
                // only move the handle in case of drag&drop
                if (handleMovedFlag) {
                    // set the handle to new position on the DOM
                    handle[0].style.left = event.pageX - xOffset + 'px';
                    // restrict the handle to the start position
                    if ((handle[0].offsetLeft) < handleStartPos) {
                        handle[0].style.left = handleStartPos + 'px';
                    }
                    // restrict the handle to the end position
                    if ((handle[0].offsetLeft) > handleEndPos) {
                        handle[0].style.left = handleEndPos + 'px';
                    }
                    // set the color which is under the center from the handle
                    setColorAtPosition(handle[0].offsetLeft + (handle[0].offsetWidth / 2));
                }
                break;

            case 'tracking:end':
                // add a transition to animate the handle movment on tapping and quantizise
                handle.addClass('transition');

                if (handleMovedFlag) {
                    // to quantizise the handle position to the current color
                    moveHandleToColor(currentColor);
                    handleMovedFlag = false;
                }

                if (tapOnSliderFlag) {
                    setColorAtPosition(event.pageX - sliderUI.offset().left);
                    moveHandleToColor(currentColor);
                    tapOnSliderFlag = false;
                }
                break;
            }
        }

        /**
         *  Moves the handle to a given color number. The handleBorderWidth is
         *  included in the calculation as a offset.
         */
        function moveHandleToColor(colorNumber) {
            handle[0].style.left = colorFieldsRangeData[colorNumber].range1 - handleBorderWidth + 'px';
        }

        /**
         *  Set the currentColor according to the given position. Tigger an event to unhide the
         *  corresponding color row in the mobile colorpicker after that.
         */
        function setColorAtPosition(pos) {
            var lastColor = currentColor;
            // check which color is at the position
            _.times(colorFieldsColorData.length, function (i) {
                if (pos > colorFieldsRangeData[i].range1 && pos < colorFieldsRangeData[i].range2) {
                    currentColor = i;
                }
            });

            // only update the color when it really changes
            if (lastColor !== currentColor) {
                // show the corresponding scheme color row in the mobile color picker
                self.trigger('colorSlider:showRow', { colorNumber: currentColor, lastColor: lastColor });
            }
        }

        ////////////////////////////////////////////////////////////////////////
        // public methods

        /**
         * Returns the whole colorSlider markup
         *
         * @returns {jQuery}
         *  The DOM element, as jQuery object.
         */
        this.getColorSliderMarkUp = function () {
            return sliderUI;
        };

        /**
         * Moves the handle to a given color number. Tigger an event to unhide the
         * corresponding color row in the mobile colorpicker after that.
         *
         * @param {Number} colorNumber
         *  The position from the color in the colorSlider
         */
        this.moveHandleToCorrespondingColor = function (colorNumber) {
            var lastColor = currentColor;
            // no animation is needed here
            handle.removeClass('transition');
            moveHandleToColor(colorNumber);
            // updates the currentColor
            currentColor = colorNumber;
            self.trigger('colorSlider:showRow', { colorNumber: currentColor, lastColor: lastColor });
        };

        // initialize -----
        createColorFieldsRangeData();
        createColorSlider();
        setListener();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sliderUI.remove();
            initOptions = self = colorFieldsColorData = defaultHandlePosition = colorFieldsRangeData = handle = sliderUI = colorSlider = currentColor = handleMovedFlag = tapOnSliderFlag = xOffset = colorFieldWidth = marginLeft = handleBorderWidth = adjustHandleOffset = handleStartPos = handleEndPos = null;
        });

    } // class ColorSlider

    // derive this class from class Group
    return Group.extend({ constructor: ColorSlider });
});
