/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/text/utils/textutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/text/dom'
], function (Utils, AttributeUtils, DOM) {

    'use strict';

    // static class TextUtils ================================================

    // copy tk/utils methods
    var TextUtils = _.clone(Utils);

    // methods ----------------------------------------------------------------

    /**
     * Searches inside a specified text string for any character, that is
     * included in a list of characters. So this function is an expanded
     * version of the indexOf() method. Additionally it is possible to define a
     * start index inside the text string and to define the search direction.
     * The search in the text string stops immediately, when the first
     * specified character from the character list is found.
     *
     * TODO: This function can probably be performance improved by using a
     * regular expression.
     *
     * @param {String} text
     *  The text string, in which the characters will be searched.
     *
     * @param {Number} pos
     *  The start position, at which the search will begin.
     *
     * @param {Array<String>} charList
     *  The list of characters, that will be searched inside the string.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.reverse=false]
     *      Whether the search shall happen in reverse order, meaning from the
     *      end of the string to the beginning. Default is, that the string is
     *      searched from beginning to end.
     *
     * @returns {Number}
     *  The index of the first found character specified in the character list
     *  inside the specified text string. If no character from the list was
     *  found, '-1' is returned.
     */
    TextUtils.indexOfValuesInsideTextSpan = function (text, pos, charList, options) {

        var // whether at least one character in the specified char list was found
            found = false,
            // the length of the specified text string
            length = 0,
            // whether the search shall be done in reverse direction
            reverse = Utils.getBooleanOption(options, 'reverse', false);

        if (reverse) {
            while (pos >= 0 && !_.contains(charList, text[pos])) { pos--; }
            found = (pos >= 0);
        } else {
            length = text.length;
            while (pos < length && !_.contains(charList, text[pos])) { pos++; }
            found = (pos < length);
        }

        return found ? pos : -1;
    };

    /**
     * Tries to merge the passed text span with its next or previous sibling
     * text span. To be able to merge two text spans, they must contain equal
     * formatting attributes. If merging was successful, the sibling span will
     * be removed from the DOM.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM node to be merged with its sibling text span. If this object is
     *  a jQuery object, uses the first DOM node it contains.
     *
     * @param {Boolean} next
     *  If set to true, will try to merge with the next span, otherwise with
     *  the previous text span.
     */
    TextUtils.mergeSiblingTextSpans = function (node, next) {

        var // the sibling text span, depending on the passed direction
            sibling = null,
            // text in the passed and in the sibling node
            text = null, siblingText = null;

        // passed node and sibling node, as DOM nodes
        node = Utils.getDomNode(node);
        sibling = node[next ? 'nextSibling' : 'previousSibling'];

        // both nodes must be text spans with the same attributes
        if (sibling && DOM.isTextSpan(node) && DOM.isTextSpan(sibling) && AttributeUtils.hasEqualElementAttributes(node, sibling)) {

            // add text of the sibling text node to the passed text node
            text = node.firstChild.nodeValue;
            siblingText = sibling.firstChild.nodeValue;
            node.firstChild.nodeValue = next ? (text + siblingText) : (siblingText + text);

            // remove the entire sibling span element
            $(sibling).remove();
        }
    };

    /**
     * Returns the absolute CSS position of the passed DOM.Point, relative to a
     * specific root node and zoom factor.
     *
     * Info: Please use 'Position.getPixelPositionFromDomPoint' from the OX Text
     * pixel API.
     *
     * @param {DOM.Point} point
     *  the point object, from which the CSS position should be calculated
     *
     * @param {jQuery| HTMLElement} rootNode
     *  the calculated CSS positions are relative to this root node.
     *
     * @param {Number} zoomFactor
     *  the current active zoom factor of the application view.
     *
     * @returns {Object}
     *  with the following CSS position properties:
     *  - {Number} top
     *  - {Number} left
     */
    TextUtils.getCSSPositionFromPoint = function (point, rootNode, zoomFactor) {

        if (!point) { return null; }

        var caretSpan = $('<span>').text('|'),
            cursorElement = $(point.node.parentNode),
            cursorElementLineHeight = parseFloat(cursorElement.css('line-height')),
            zoom = zoomFactor / 100;

        // break cursor element on the text offset
        if (point.offset === 0) {
            cursorElement.before(caretSpan);
        } else {
            if (DOM.isSpan(cursorElement)) {
                DOM.splitTextSpan(cursorElement, point.offset, { append: true });
            }
            cursorElement.after(caretSpan);
        }

        // create caret overlay and calculate its position
        var caretTop = (caretSpan.offset().top - rootNode.offset().top) / zoom  - cursorElementLineHeight  + caretSpan.outerHeight(),
            caretLeft = (caretSpan.offset().left - rootNode.offset().left) / zoom;

        // restore original state of document
        caretSpan.remove();

        if (point.offset > 0) { TextUtils.mergeSiblingTextSpans(cursorElement, true); }

        return { top: caretTop, left: caretLeft };
    };

    /**
     * Returns whether the passed intervals overlap with at least one index.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed intervals are overlapping.
     */
    TextUtils.intervalOverlapsInterval = function (interval1, interval2) {
        return (interval1.first <= interval2.last) && (interval2.first <= interval1.last);
    };

    /**
     * Returns an array without 'falsy' elements.
     *
     * @param {Array} array
     *  The array, in that 'falsy' elements will be removed.
     *
     * @returns {Array}
     *  An array without 'falsy' elements.
     */
    TextUtils.removeFalsyItemsInArray = function (array) {
        return _.filter(array, function (item) {
            return item;
        });
    };

    /**
     * Calculating an internal user id, that is saved at the node as attribute
     * and saved in the xml file. The base for this id is the app server id.
     *
     * @param {Number} id
     *  The user id used by the app suite server.
     *
     * @returns {Number}
     *  The user id, that is saved in xml file and saved as node attribute.
     */
    TextUtils.calculateUserId = function (id) {
        return _.isNumber(id) ? (id + 123) * 10 - 123 : null;
    };

    /**
     * Calculating the app suite user id from the saved and transported user id.
     *
     * @param {Number} id
     *  The user id, that is saved in xml file and saved as node attribute.
     *
     * @returns {Number}
     *  The user id used by the app suite server.
     */
    TextUtils.resolveUserId = function (id) {
        return _.isNumber(id) ? ((id + 123) / 10 - 123) : null;
    };

    // exports ================================================================

    return TextUtils;

});
