/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/components/field/simplefield', [
    'io.ox/office/text/components/field/basefield',
    'io.ox/office/text/utils/textutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/position'
], function (BaseField, Utils, LocaleData, Operations, DOM, Position) {

    'use strict';

    // class SimpleField =====================================================

    /**
     * An instance of this class represents the model for all simple fields in the
     * edited document.
     *
     * @constructor
     *
     * @extends Field
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function SimpleField(app) {

        var // self reference
            self = this,
            // collector for simple fields
            allSimpleFields = {},
            // counter for simple field Id
            simpleFieldID = 0,
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // page layout object
            pageLayout = null,
            // number formatter object
            numberFormatter = null;

        // base constructors --------------------------------------------------

        BaseField.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one simple field in the collection (the model).
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         * @param {String} [target]
         *  The target, where the simple field is located.
         */
        function addSimpleFieldToCollection(fieldNode, target) {
            var $fieldNode = $(fieldNode),
                id = $fieldNode.data('sFieldId') ? $fieldNode.data('sFieldId') : 'sf' + simpleFieldID,
                isMarginalId = target && pageLayout.isIdOfMarginalNode(target);

            $fieldNode.data('sFieldId', id);
            allSimpleFields[id] = isMarginalId ? target : $fieldNode;

            simpleFieldID += 1;
        }

        /**
         * Removing one simple field with specified id from the collection (the model).
         *
         * @param {String} id
         *  The unique id of the simple field node.
         *
         */
        function removeSimpleFieldFromCollection(id) {
            delete allSimpleFields[id];
        }

        /**
         * Finding in a header or footer specified by the target the simple field
         * with the specified id.
         *
         * @param {String} id
         *  The id string.
         *
         * @param {String} target
         *  The target string to identify the header or footer node
         *
         * @returns {jQuery|null}
         *  The simple field node with the specified id and type, or null, if no such
         *  field exists.
         *
         */
        function getMarginalSimpleField(id, target) {
            var // the specified header or footer node
                marginalNode = null,
                // the searched complex field with the specified id
                field = null;

            marginalNode = model.getRootNode(target);

            if (marginalNode) {
                field = _.find(marginalNode.find(DOM.FIELD_NODE_SELECTOR), function (field) { return DOM.getSimpleFieldId(field) === id; });
            }

            return field ? $(field) : null;
        }

        /**
         * Update of content for simple field: Date
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of date
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.time=false]
         *      If it's time format only.
         */
        function updateDateTimeField(node, format, options) {
            var rawDate = numberFormatter.makeDateTime(numberFormatter.getDateTimeComponents(new Date(), true)),
                date = numberFormatter.convertDateToNumber(rawDate),
                time = Utils.getBooleanOption(options, 'time', false),
                format = (format && format !== 'default') ? format : (time ? LocaleData.SHORT_TIME : LocaleData.SHORT_DATE),
                formattedDate = numberFormatter.formatValue(date, format),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                dateFormat = DOM.getFieldDateTimeFormat(node),
                isMarginal = DOM.isMarginalNode(node),
                rootNode = isMarginal ? DOM.getClosestMarginalTargetNode(node) : selection.getRootNode(),
                target = isMarginal ? DOM.getTargetContainerId(rootNode) : '',
                startPos = Position.getOxoPosition(rootNode, node),
                operation;

            if (startPos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    if (isMarginal) {
                        pageLayout.enterHeaderFooterEditMode(rootNode);
                        selection.setNewRootNode(rootNode);
                    }
                    model.deleteRange(startPos);
                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: formattedDate, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    if (dateFormat) {
                        operation = _.extend(operation, { attrs: { field: { dateFormat: dateFormat } } });
                    }

                    model.applyOperations(operation);
                });
            } else {
                Utils.error('complexField.updateDateTimeField(): Wrong start postition: ', startPos);
            }
        }

        /**
         * Update the content of simple field: FILENAME
         *
         * @param {jQuery|Node} node
         *  Simple field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateFileNameField(node, format) {
            var fileName = app.getFullFileName(),
                rootNode = selection.getRootNode(),
                startPos = Position.getOxoPosition(rootNode, node),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                target = DOM.isMarginalNode(node) ? DOM.getTargetContainerId(rootNode) : '',
                operation;

            if (format) {
                if ((/Lower/).test(format)) {
                    fileName = fileName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    fileName = fileName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    fileName = Utils.capitalize(fileName);
                } else if ((/Caps/).test(format)) {
                    fileName = Utils.capitalizeWords(fileName);
                }
            }

            if (startPos && fileName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);

                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: fileName, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    model.applyOperations(operation);
                });
            } else {
                Utils.error('complexField.updateFileNameField(): Wrong start postition or text: ', startPos, fileName);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Public method to add simple field to model.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         * @param {String} [target]
         *  The target, where the simple field is located.
         *
         */
        this.addSimpleFieldToCollection = function (fieldNode, target) {
            addSimpleFieldToCollection(fieldNode, target);
        };

        /**
         * Public method to remove simple field from model.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         */
        this.removeSimpleFieldFromCollection = function (node) {
            removeSimpleFieldFromCollection(DOM.getSimpleFieldId(node));
        };

        /**
         * Whether the document contains complex fields in the document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one complex field.
         */
        this.isEmpty = function () {
            return _.isEmpty(allSimpleFields);
        };

        /**
         * Public method to get collection of all simple fields from model.
         *
         * @returns {Object}
         */
        this.getAllFields = function () {
            return allSimpleFields;
        };

        /**
         * Gets simple field from collection by passed id.
         *
         * @param {String} id
         *  Id of queried field.
         *
         * @returns {jQuery|null}
         *  The simple field with the specified id, or null, if no such start marker exists.
         *
         */
        this.getSimpleField = function (id) {
            var // the value saved under the specified id
                field = allSimpleFields[id] || null;

            return _.isString(field) ? getMarginalSimpleField(id, field) : field;
        };

        /**
         * After deleting a complete paragraph, it is necessary, that all simple field nodes
         * in the deleted paragraph are also removed from the model collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.removeSimpleFieldsInNode = function (paragraph) {
            var // all simple field nodes inside the paragraph
                simpleFieldNodes = $(paragraph).find(DOM.FIELD_NODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(simpleFieldNodes, function (field) {

                var // the id of the marker node
                    fieldId = DOM.getSimpleFieldId(field);

                if (_.isString(fieldId)) {
                    removeSimpleFieldFromCollection(fieldId);
                }
            });
        };

        /**
         * Update simple field depending of type, by given instruction.
         *
         * @param {jQuery|Node} node
         *  Simple field node
         *
         * @param {String} instruction
         *  Simple field instruction, containing type,
         *  formating and optionally style, separated by \ .
         *  (In case of ODF, instruction is empty, and we fetch possible
         *  date/time format from data property of node).
         */
        this.updateByInstruction = function (node, instruction) {
            var parsedFormatInstruction,
                dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type,
                instruction = dividedData.instruction;

            if (type) {
                if (self.isCurrentDate(type)) {
                    parsedFormatInstruction = app.isODF() ? DOM.getFieldDateTimeFormat(node) : self.parseFormatInstruction(instruction, { dateTime: true });
                    updateDateTimeField(node, parsedFormatInstruction);
                } else if (self.isCurrentTime(type)) {
                    parsedFormatInstruction = app.isODF() ? DOM.getFieldDateTimeFormat(node) : self.parseFormatInstruction(instruction, { dateTime: true });
                    updateDateTimeField(node, parsedFormatInstruction, { time: true });
                } else if (self.isFileName(type)) {
                    parsedFormatInstruction = self.parseFormatInstruction(instruction);
                    updateFileNameField(node, parsedFormatInstruction);
                }
            }
        };

        /**
         * Update of content for simple field: Page number
         *
         * @param {jQuery|Node} node
         *  Simple field node.
         *
         * @param {String} format
         *  Format of the number.
         */
        this.updatePageNumberField = function (node, format) {
            var number = pageLayout.getPageNumber(node),
                rootNode = selection.getRootNode(),
                startPos = Position.getOxoPosition(rootNode, node),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                target = DOM.isMarginalNode(node) ? DOM.getTargetContainerId(rootNode) : '',
                operation;

            if (model.isHeaderFooterEditState()) { // no explicit update of pageNumber field in header/footer!
                return;
            }

            number = self.formatPageFieldInstruction(number, format);

            if (startPos && number) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);
                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: number, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    model.applyOperations(operation);
                });
            } else {
                Utils.error('complexField.updatePageNumberField(): Wrong start postition or text: ', startPos, number);
            }
        };

        /**
         * Update of content for simple field: Number of pages
         *
         * @param {jQuery|Node} node
         *   Simple field node
         *
         * @param {String} format
         *  Format of the number.
         */
        this.updateNumPagesField = function (node, format) {
            var numPages = pageLayout.getNumberOfDocumentPages(),
                rootNode = selection.getRootNode(),
                startPos = Position.getOxoPosition(rootNode, node),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                target = DOM.isMarginalNode(node) ? DOM.getTargetContainerId(rootNode) : '',
                operation;

            if (model.isHeaderFooterEditState()) { // no explicit update of numPages field in header/footer!
                return;
            }

            numPages = self.formatPageFieldInstruction(numPages, format, true);

            if (startPos && numPages) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);
                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: numPages, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    model.applyOperations(operation);
                });
            } else {
                Utils.error('complexField.updateNumPagesField(): Wrong start postition or text: ', startPos, numPages);
            }
        };

        /**
         * Update date and time fields on document load, or before running download or print action.
         *
         * @param {jQuery|Node} field
         *  Field to be updated.
         *
         * @param {String} instruction
         *  Instruction for updating.
         */
        this.updateDateTimeField = function (field, instruction) {
            var dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type,
                instruction = dividedData.instruction;

            if (type) {
                if ((self.isCurrentDate(type) || self.isCurrentTime(type))) {
                    instruction = app.isODF() ? DOM.getFieldDateTimeFormat(field) : self.parseFormatInstruction(instruction, { dateTime: true });
                    updateDateTimeField(field, instruction, { time: self.isCurrentTime(type) });
                }
            }
        };

        /**
         * After load from local storage or fast load the collectors for the simple fields need to be filled.
         */
        this.refreshSimpleFields = function () {
            var // the page content node of the document
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // all simple field nodes in the document
                allSimpleFieldNodes = pageContentNode.find(DOM.FIELD_NODE_SELECTOR),
                // a collector for all marginal template nodes
                allMargins = pageLayout.getHeaderFooterPlaceHolder().children(),
                // the target string node
                target = '';

            // reset model
            allSimpleFields = {};

            _.each(allSimpleFieldNodes, function (fieldNode) {
                if (!DOM.isMarginalNode($(fieldNode).parent())) {
                    addSimpleFieldToCollection(fieldNode);
                }
            });

            _.each(allMargins, function (margin) {
                var // a collector for all range marker nodes inside one margin
                    allMarginSimpleFieldNodes = $(margin).find(DOM.FIELD_NODE_SELECTOR);

                if (allMarginSimpleFieldNodes.length > 0) {
                    target = DOM.getTargetContainerId(margin);
                    _.each(allMarginSimpleFieldNodes, function (oneMarginalField) {
                        addSimpleFieldToCollection(oneMarginalField, target);
                    });
                }
            });

        };

        /**
         * After splitting a paragraph, it is necessary, that all simple field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateSimpleFieldCollector = function (para) {

            var // whether each single node needs to be checked
                checkMarginal = false,
                // all simple fields inside the paragraph
                allFields = null;

            // not necessary for paragraphs in header or footer -> only the target are stored, not the nodes
            if (DOM.isMarginalNode(para)) { return; }

            // if this is not a paragraph, each single node need to be checked
            checkMarginal = !DOM.isParagraphNode(para);

            allFields = $(para).find(DOM.FIELD_NODE_SELECTOR);

            // update  the complex field nodes in the collection objects
            _.each(allFields, function (fieldNode) {
                if (!checkMarginal || !DOM.isMarginalNode(fieldNode)) {
                    // simply overwriting the old simple fields with the new values
                    addSimpleFieldToCollection(fieldNode);
                }
            });
        };

        /**
         * On document load, this method loops all fields in collection
         * to find date/time fields and update them.
         */
        this.updateDateTimeFieldOnLoad = function () {
            _.each(allSimpleFields, function (field) {
                self.updateDateTimeField(field, DOM.getFieldInstruction(field));
            });
        };

        /**
         * Public method for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNode = function ($node, isHeader, pageCount) {
            if (isHeader) {
                if (DOM.isHeaderWrapper($node.parent())) {
                    $node.find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(1);
                } else {
                    _.each($node.find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                        $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num') + 1);
                    });
                }
            } else {
                if (DOM.isFooterWrapper($node.parent())) {
                    $node.find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(pageCount);
                } else {
                    _.each($node.find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                        $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num'));
                    });
                }
            }
        };

        /**
         * Public method for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNode = function ($node, pageCount) {
            $node.find(DOM.PAGECOUNT_FIELD_SELECTOR).children().empty().html(pageCount);
        };
        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            selection = model.getSelection();
            pageLayout = model.getPageLayout();
            numberFormatter = model.getNumberFormatter();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = selection = pageLayout = numberFormatter = null;
        });

    } // class SimpleField

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class BaseField
    return BaseField.extend({ constructor: SimpleField });

});
