/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/selectionmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/sheetselection'
], function (Utils, KeyCodes, SheetUtils, PaneUtils, SheetSelection) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Interval = SheetUtils.Interval,
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray;

    // mix-in class SelectionMixin ============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides methods to work
     * with the cell and drawing selection.
     *
     * @constructor
     */
    function SelectionMixin(rootNode) {

        var // self reference
            self = this,

            // the spreadsheet document model
            docModel = this.getDocModel(),

            // callback handlers to override cell selection behavior
            getCellSelectionHandler = null,
            setCellSelectionHandler = null,

            // user-defined identifier for current custom selection mode
            customSelectionId = null,

            // current pending deferred object for custom range selection
            customSelectionDef = null;

        // private methods ----------------------------------------------------

        /**
         * Returns a reference to the original selection from the active sheet.
         */
        function getSheetSelection() {
            return self.getSheetViewAttribute('selection');
        }

        /**
         * Changes the selection of the active sheet.
         */
        function setSheetSelection(selection) {
            self.setSheetViewAttribute('selection', selection);
        }

        /**
         * Leaves the custom selection mode, if it has been started with the
         * method SelectionMixin.enterCustomSelectionMode() before.
         *
         * @param {Boolean} apply
         *  If set to true, the promise representing the custom selection mode
         *  will be resolved with the current custom selection. Otherwise, the
         *  promise will be rejected (custom selection mode canceled).
         */
        function leaveCustomSelectionMode(apply) {

            // custom selection mode must be active
            if (!_.isString(customSelectionId)) { return; }

            var // the selection to be passed to the deferred object
                selection = (apply === true) ? self.getSheetViewAttribute('activeSelection') : null,
                // local copy of the deferred object, to prevent overwriting recursively from handlers
                selectionDef = customSelectionDef;

            // reset all settings for custom selection mode before resolving the deferred object
            customSelectionId = customSelectionDef = null;
            self.unregisterCellSelectionHandlers();
            self.setSheetViewAttribute('activeSelection', null);
            self.setStatusLabel(null);

            // finally, resolve or reject the deferred object
            if (selection) {
                selectionDef.resolve(selection);
            } else {
                selectionDef.reject();
            }
        }

        /**
         * Returns the address of the cell that will be selected with the HOME
         * key. In unfrozen views, this is always cell A1. In frozen views,
         * this is the first available cell in the unfrozen bottom-right grid
         * pane.
         *
         * @returns {Address}
         *  The address of the cell to be selected with the HOME key.
         */
        function getHomeCell() {
            var sheetModel = self.getSheetModel(),
                frozenSplit = sheetModel.hasFrozenSplit(),
                colInterval = frozenSplit ? sheetModel.getSplitColInterval() : null,
                rowInterval = frozenSplit ? sheetModel.getSplitRowInterval() : null;
            return new Address(colInterval ? (colInterval.last + 1) : 0, rowInterval ? (rowInterval.last + 1) : 0);
        }

        /**
         * Returns the address of the cell that will be selected with the END
         * key (the bottom-right cell in the used area of the sheet).
         *
         * @returns {Address}
         *  The address of the cell to be selected with the END key.
         */
        function getEndCell() {
            return self.getCellCollection().getLastUsedCell();
        }

        /**
         * Expands the passed range address so that it completely includes all
         * merged ranges it currently overlaps. In case the passed range is a
         * full column or full row range, it will be expanded to full-column or
         * full-row merged ranges only.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Range}
         *  The address of the expanded cell range.
         */
        function expandRangeToMergedRanges(range) {

            // full column/row ranges: expand to full-size merged ranges only (bug 33313)
            if (docModel.isColRange(range) || docModel.isRowRange(range)) {
                return self.getMergeCollection().getBoundingMergedRange(range) || range;
            }

            // expand other ranges to all the merged ranges it overlaps
            return self.getMergeCollection().expandRangeToMergedRanges(range);
        }

        /**
         * Returns a selection object that is suitable for keyboard navigation.
         * Prefers the registered cell selection handler.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.allowEmpty=false]
         *      If set to true, and the registered cell selection handler
         *      returns a selection without any ranges (property 'ranges' set
         *      to an empty array), this selection object will be returned
         *      unmodified. Otherwise, an empty selection returned by that
         *      handler will be replaced by a selection consisting of the
         *      current active cell.
         */
        function getActiveSelection(options) {

            var // the custom selection provided by the registered cell selection handler
                selection = _.isFunction(getCellSelectionHandler) ? getCellSelectionHandler.call(self) : null;

            if (!_.isObject(selection)) {
                // no custom selection: use real selection of this sheet
                selection = self.getSelection();
            } else if (selection.ranges.empty() && !Utils.getBooleanOption(options, 'allowEmpty', false)) {
                // replace empty custom selection with active cell
                selection = self.getActiveCellAsSelection();
            }

            return selection;
        }

        /**
         * Clears the current selection and selects the specified cell. If the
         * cell is hidden, tries to select the nearest visible cell.
         */
        function selectVisibleCell(address) {
            if ((address = self.getSheetModel().getVisibleCell(address, 'prevNext'))) {
                self.selectCell(address);
                self.scrollToCell(address);
            }
        }

        /**
         * Expands the active range of the current selection to the passed cell
         * address.
         */
        function expandToVisibleCell(address, shrinkEnd) {

            var // current selection in the active sheet
                selection = getActiveSelection(),
                // the active range in the selection
                activeRange = selection.activeRange();

            if ((address = self.getSheetModel().getVisibleCell(address, 'prevNext'))) {

                // start position of active range: extend to passed address,
                // if it is located before, but do not shrink the active range
                activeRange.start[0] = Math.min(address[0], activeRange.start[0]);
                activeRange.start[1] = Math.min(address[1], activeRange.start[1]);

                // end position of active range: extend to passed address, if it
                // is located after the range, but shrink to active cell if not
                activeRange.end[0] = Math.max(address[0], shrinkEnd ? selection.address[0] : activeRange.end[0]);
                activeRange.end[1] = Math.max(address[1], shrinkEnd ? selection.address[1] : activeRange.end[1]);

                // commit new selection, adjust scrolling position
                self.changeActiveRange(activeRange);
                self.scrollToCell(address);
            }
        }

        /**
         * Searches for the next/last filled cell in the current cell
         * collection.
         */
        function findNextUsedCell(address, direction) {

            var // the cell collection of the active sheet
                cellCollection = self.getCellCollection(),
                // whether to move the active cell forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to move the active cell horizontally
                columns = (direction === 'left') || (direction === 'right'),
                // the resulting address
                resultAddress = _.clone(address),

                // address of the last used cell
                lastUsedCell = cellCollection.getLastUsedCell(),
                // maximum column/row index in the sheet (in move direction)
                maxIndex = docModel.getMaxIndex(columns),
                // column/row index of the start cell
                startIndex = address.get(columns),
                // column/row index of last visited cell
                lastIndex = null;

            // forward mode: jump to end of sheet, if start cell is outside used area
            if (forward && ((address[0] >= lastUsedCell[0]) || (address[1] >= lastUsedCell[1]))) {
                resultAddress.set(maxIndex, columns);
                return resultAddress;
            }

            // backward mode: jump to beginning of sheet, if start cell is outside
            // used area, and the new address will also be outside the used area
            if (!forward && (address.get(!columns) > lastUsedCell.get(!columns))) {
                resultAddress.set(0, columns);
                return resultAddress;
            }

            // find next available content cell in the cell collection
            cellCollection.iterateCellsInLine(address, direction, function (cellData) {

                var // column/row index of the current cell
                    currIndex = cellData.address.get(columns);

                // started from an empty cell: stick to first content cell
                if (!_.isNumber(lastIndex) && (startIndex !== currIndex)) {
                    resultAddress.set(currIndex, columns);
                    return Utils.BREAK;
                }

                // found a content cell following an empty cell
                if (_.isNumber(lastIndex) && (Math.abs(currIndex - lastIndex) > 1)) {

                    // use the current cell, if iteration has been started at the cell
                    // before the gap; otherwise use the cell preceding the gap
                    resultAddress.set((startIndex === lastIndex) ? currIndex : lastIndex, columns);
                    return Utils.BREAK;
                }

                // remember address of current content cell for next iteration
                lastIndex = currIndex;

            }, { type: 'content' });

            // jump to last found content cell if it is not the start cell;
            // or to sheet boundaries, if no valid content cell could be found
            if (address.equals(resultAddress)) {
                resultAddress.set(_.isNumber(lastIndex) && (lastIndex !== startIndex) ? lastIndex : forward ? maxIndex : 0, columns);
            }

            return resultAddress;
        }

        /**
         * Clears the current selection and moves the active cell into the
         * specified direction. If the target cell is hidden, moves the cursor
         * to the next available visible cell.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         *
         * @param {String} mode
         *  The move mode. Supported values are:
         *  - 'cell':
         *      Moves the active cell to the next visible cell.
         *  - 'page':
         *      Moves the active cell by the width/height of the visible area,
         *      and adjusts the scroll position in a way that the new active
         *      cell remains on the same screen position if possible.
         *  - 'used':
         *      Moves the active cell to the next available non-empty cell; or,
         *      if none available, to the first or last cell in the sheet.
         */
        function selectNextCell(direction, mode) {

            var // current selection in the active sheet
                selection = getActiveSelection(),
                // the cell to start movement from
                activeCell = selection.origin || selection.address,
                // the merged range covering the active cell
                mergedRange = self.getMergeCollection().getMergedRange(activeCell),

                // whether to move the active cell forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to move the active cell horizontally
                columns = (direction === 'left') || (direction === 'right'),

                // the column/row collection in the move direction
                collection = columns ? self.getColCollection() : self.getRowCollection(),
                // the active (focused) grid pane
                gridPane = self.getActiveGridPane(),
                // the header pane in the move direction
                headerPane = self.getActiveHeaderPane(columns),

                // the new column/row index used to look for a visible column/row
                index = activeCell.get(columns),
                // position and size of the active cell (for page scroll)
                cellPosition = null, cellOffset = 0, cellSize = 0,
                // position and size of the visible area in this grid pane (for page scroll)
                visiblePosition = null, visibleOffset = 0, visibleSize = 0,

                // the new column/row entry to move to
                entryData = null,
                // the target cell to move to (for 'used' mode)
                targetCell = null;

            // returns the descriptor of a new entry when moving in 'page' mode
            function getPageTargetEntry(offset) {
                return collection.getEntryByOffset(offset + (forward ? visibleSize : -visibleSize) * 0.98, { pixel: true });
            }

            switch (mode) {
            case 'cell':
                // start from leading/trailing border of a merged range
                if (mergedRange) {
                    index = forward ? mergedRange.getEnd(columns) : mergedRange.getStart(columns);
                }
                index += (forward ? 1 : -1);

                // insert new column/row entry (old selection will collapse to active cell, if no other cell has been found)
                if ((entryData = collection.getVisibleEntry(index, forward ? 'next' : 'prev'))) {
                    activeCell.set(entryData.index, columns);
                }

                // set selection to the new active cell
                self.selectCell(activeCell, { storeOrigin: true });
                self.scrollToCell(activeCell);
                break;

            case 'page':
                // get position and size of the active cell
                cellPosition = mergedRange ? self.getRangeRectangle(mergedRange) : self.getCellRectangle(activeCell);
                cellOffset = cellPosition[columns ? 'left' : 'top'];
                cellSize = cellPosition[columns ? 'width' : 'height'];

                // get position and size of the visible area in this grid pane
                visiblePosition = gridPane.getVisibleRectangle();
                visibleOffset = visiblePosition[columns ? 'left' : 'top'];
                visibleSize = visiblePosition[columns ? 'width' : 'height'];

                // if the active cell is located inside the visible area, find the column/row one page away
                if ((visibleOffset <= cellOffset) && (cellOffset + cellSize <= visibleOffset + visibleSize)) {
                    entryData = getPageTargetEntry(cellOffset + cellSize / 2);
                    activeCell.set(entryData.index, columns);
                    self.selectCell(activeCell, { storeOrigin: true });
                    headerPane.scrollRelative(entryData.offset - cellOffset);
                // otherwise, scroll one page and select the top visible entry
                } else {
                    entryData = getPageTargetEntry(visibleOffset);
                    activeCell.set(entryData.index, columns);
                    self.selectCell(activeCell, { storeOrigin: true });
                    headerPane.scrollTo(entryData.offset);
                }
                break;

            case 'used':
                // search for a target cell
                targetCell = findNextUsedCell(activeCell, direction);

                // fail-safety: if cell address could not be determined or did not change, make a single cell step
                if (!targetCell || activeCell.equals(targetCell)) {
                    return selectNextCell(direction, 'cell');
                }

                // set selection to the new active cell
                self.selectCell(targetCell, { storeOrigin: true });
                self.scrollToCell(targetCell);
                break;
            }
        }

        /**
         * Extends the active range in the current selection into the specified
         * direction. If the target column/row is hidden, extends the active
         * range to the next available visible column/row.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         *
         * @param {String} mode
         *  The move mode. Supported values are:
         *  - 'cell':
         *      Expands the active range to the next visible column/row.
         *  - 'page':
         *      Expands the active range by the width/height of the visible
         *      area, and adjusts the scroll position in a way that the range
         *      border remains on the same screen position if possible.
         *  - 'used':
         *      Expands the active range to the next available non-empty cell;
         *      or, if none available, to the first or last column/row in the
         *      sheet.
         */
        function expandToNextCell(direction, mode) {

            var // collection of merged ranges in the active sheet
                mergeCollection = self.getMergeCollection(),
                // current selection in the active sheet
                selection = getActiveSelection(),
                // the cell to start movement from
                activeCell = (selection.origin || selection.address).clone(),
                // the current active range address
                activeRange = selection.activeRange(),
                // the merged range covering the active cell, or the active cell as range object
                mergedRange = mergeCollection.getMergedRange(activeCell) || Range.createFromAddresses(activeCell),
                // the new active range address
                newRange = null,

                // whether to expand the active range forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to expand the active range horizontally
                columns = (direction === 'left') || (direction === 'right'),

                // the column/row collection in the move direction
                collection = columns ? self.getColCollection() : self.getRowCollection(),
                // the active (focused) grid pane
                gridPane = self.getActiveGridPane(),
                // the header pane in the move direction
                headerPane = self.getActiveHeaderPane(columns),

                // the new column/row entry to move to
                entryData = null;

            // returns the descriptor of a new entry when moving in 'page' mode
            function getPageTargetEntry(index) {

                var // the descriptor for the start entry
                    entryData = collection.getEntry(index),
                    // the position in the middle of the entry
                    entryOffset = entryData.offset + entryData.size / 2,
                    // the size of the visible area, in pixels
                    visibleSize = gridPane.getVisibleRectangle()[columns ? 'width' : 'height'];

                // return the target entry
                return collection.getEntryByOffset(entryOffset + (forward ? visibleSize : -visibleSize) * 0.98, { pixel: true });
            }

            // returns the descriptor of a new entry when moving in 'used' mode
            function getUsedTargetEntry(address) {
                var targetCell = findNextUsedCell(address, direction);
                return targetCell ? collection.getVisibleEntry(targetCell.get(columns), forward ? 'next' : 'prev') : null;
            }

            // first, try to shrink the range at the correct border according to move direction
            activeCell.set(forward ? activeRange.getStart(columns) : activeRange.getEnd(columns), columns);
            switch (mode) {
            case 'cell':
                entryData = forward ?
                    collection.getVisibleEntry(activeCell.get(columns) + 1, 'next') :
                    collection.getVisibleEntry(activeCell.get(columns) - 1, 'prev');
                break;

            case 'page':
                entryData = getPageTargetEntry(activeCell.get(columns));
                break;

            case 'used':
                entryData = getUsedTargetEntry(activeCell);
                break;
            }

            // build the new range address according to the collection entry
            if (entryData && (forward ? (entryData.index <= mergedRange.getStart(columns)) : (mergedRange.getEnd(columns) <= entryData.index))) {
                newRange = activeRange.clone();
                (forward ? newRange.start : newRange.end).set(entryData.index, columns);
                // shrink the range until it does not contain any merged ranges partly
                newRange = mergeCollection.shrinkRangeFromMergedRanges(newRange, columns ? 'columns' : 'rows');
            }

            // if shrinking the range is not possible or has failed, expand the range away from active cell
            if (!newRange) {
                activeCell.set(forward ? activeRange.getEnd(columns) : activeRange.getStart(columns), columns);

                switch (mode) {
                case 'cell':
                    entryData = forward ?
                        collection.getVisibleEntry(activeCell.get(columns) + 1, 'next') :
                        collection.getVisibleEntry(activeCell.get(columns) - 1, 'prev');
                    break;

                case 'page':
                    entryData = getPageTargetEntry(activeCell.get(columns));
                    break;

                case 'used':
                    entryData = getUsedTargetEntry(activeCell);
                    break;
                }

                // build the new range address according to the collection entry
                if (entryData) {
                    newRange = activeRange.clone();
                    // expand in move direction, restrict to original active cell on the other border
                    (forward ? newRange.end : newRange.start).set(entryData.index, columns);
                    (forward ? newRange.start : newRange.end).set(selection.address.get(columns), columns);
                    // expand the range until it contains all merged ranges completely
                    newRange = mergeCollection.expandRangeToMergedRanges(newRange);
                }
            }

            // change the active range, if a new range has been found
            if (newRange && newRange.differs(activeRange)) {
                self.changeActiveRange(newRange);
            }

            // scroll to target entry (this allows e.g. to scroll inside a large merged range without changing the selection)
            if (entryData) {
                headerPane.scrollToEntry(entryData.index);
            }
        }

        /**
         * Moves the active cell in the current selection into the specified
         * direction. Cycles through all ranges contained in the selection. In
         * case the selection is a single cell, moves it in the entire sheet.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         */
        function moveActiveCell(direction) {

            var // collections of the active sheet
                colCollection = self.getColCollection(),
                rowCollection = self.getRowCollection(),
                cellCollection = self.getCellCollection(),
                mergeCollection = self.getMergeCollection(),
                // current selection in the active sheet
                selection = getActiveSelection(),
                // move in entire sheet, if the selection consists of a single cell only
                singleCell = self.isSingleCellSelection(),
                // whether to move the active cell forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to move the active cell horizontally
                columns = (direction === 'left') || (direction === 'right'),
                // opposite flag for convenience
                rows = !columns,
                // the column/row collection in the primary direction
                collection1 = columns ? colCollection : rowCollection,
                // the column/row collection in the secondary direction
                collection2 = columns ? rowCollection : colCollection,
                // whether to jump to unlocked cells only
                unlockedCells = columns && self.isSheetLocked();

            // returns the index of the preceding visible column/row
            function getPrevVisibleIndex(collection, index) {
                var entry = collection.getPrevVisibleEntry(index - 1);
                return entry ? entry.index : -1;
            }

            // returns the index of the following visible column/row
            function getNextVisibleIndex(collection, index) {
                var entry = collection.getNextVisibleEntry(index + 1);
                return entry ? entry.index : (collection.getMaxIndex() + 1);
            }

            // increases the passed range index, cycling in the available ranges
            function incRangeIndex(index) {
                return Utils.mod(index + 1, selection.ranges.length);
            }

            // decreases the passed range index, cycling in the available ranges
            function decRangeIndex(index) {
                return Utils.mod(index - 1, selection.ranges.length);
            }

            // moves the active cell to the preceding available position in the selection
            function moveToPrevCell() {

                var // reference to the current active range address
                    activeRange = selection.activeRange(),
                    // reference to the active cell address in the selection object
                    activeCell = selection.address,
                    // the merged range containing the active cell
                    mergedRange = mergeCollection.getMergedRange(activeCell);

                // TODO: skip protected cells (horizontal only)

                // jump to reference cell of a merged range, if located in hidden cell of first column/row
                if (mergedRange && (activeCell.get(columns) > mergedRange.getStart(columns)) && (activeCell.get(rows) === mergedRange.getStart(rows))) {
                    selection.address = mergedRange.start.clone();
                    return true;
                }

                // move one column/row backwards (this may leave the active range)
                activeCell.set(getPrevVisibleIndex(collection1, mergedRange ? mergedRange.getStart(columns) : activeCell.get(columns)), columns);
                if (activeCell.get(columns) < activeRange.getStart(columns)) {
                    // out of range: back to last column/row, move to the preceding row/column (this may leave the active range)
                    activeCell.set(activeRange.getEnd(columns), columns);
                    // move to start of merged range, if it covers range completely (no cells beside merged range available)
                    if (mergedRange && (mergedRange.getStart(columns) <= activeRange.getStart(columns)) && (activeRange.getEnd(columns) <= mergedRange.getEnd(columns))) {
                        activeCell.set(getPrevVisibleIndex(collection2, mergedRange.getStart(rows)), rows);
                    } else {
                        activeCell.set(getPrevVisibleIndex(collection2, activeCell.get(rows)), rows);
                    }
                    if (activeCell.get(rows) < activeRange.getStart(rows)) {
                        // out of range: go to last cell in previous range
                        selection.active = decRangeIndex(selection.active);
                        selection.address = selection.activeRange().end.clone();
                    }
                }
                return true;
            }

            // moves the active cell to the next available position in the selection
            function moveToNextCell() {

                var // reference to the current active range address
                    activeRange = selection.activeRange(),
                    // reference to the active cell address in the selection object
                    activeCell = selection.address,
                    // the merged range containing the active cell
                    mergedRange = mergeCollection.getMergedRange(activeCell);

                // TODO: skip protected cells (horizontal only)

                // move one column/row forwards (this may leave the active range)
                activeCell.set(getNextVisibleIndex(collection1, mergedRange ? mergedRange.getEnd(columns) : activeCell.get(columns)), columns);
                if (activeCell.get(columns) > activeRange.getEnd(columns)) {
                    // out of range: back to first column/row, move to the next row/column (this may leave the active range)
                    activeCell.set(activeRange.getStart(columns), columns);
                    // move to end of merged range, if it covers range completely (no cells beside merged range available)
                    if (mergedRange && (mergedRange.getStart(columns) <= activeRange.getStart(columns)) && (activeRange.getEnd(columns) <= mergedRange.getEnd(columns))) {
                        activeCell.set(getNextVisibleIndex(collection2, mergedRange.getEnd(rows)), rows);
                    } else {
                        activeCell.set(getNextVisibleIndex(collection2, activeCell.get(rows)), rows);
                    }
                    if (activeCell.get(rows) > activeRange.getEnd(rows)) {
                        // out of range: go to first cell in next range
                        selection.active = incRangeIndex(selection.active);
                        selection.address = selection.activeRange().start.clone();
                    }
                }
                return true;
            }

            // moving horizontally through a locked sheet without selection wraps inside used area of sheet
            if (singleCell && unlockedCells) {
                selection.ranges = new RangeArray(cellCollection.getUsedRange());
                selection.active = 0;
            }

            // simple movement if only a single cell is selected
            if (singleCell && !unlockedCells) {
                selectNextCell(direction, 'cell');
                return;
            }

            // start with the original active cell that was tried to select (it may be hidden by a merged range)
            if (selection.origin) {
                selection.address = selection.origin;
            }

            // try to find a visible cell (may fail, if the entire selection is hidden)
            var success = (function () {

                var // reference to the active cell address in the selection object
                    activeCell = selection.address,
                    // the position after moving the active cell the first time
                    firstPos = null,
                    // the merged range containing the initial active cell
                    mergedRange = null,
                    // the column/row collection entry to start in a visible column/row
                    entryData = null;

                // leave hidden column/row in move direction to prevent iterating all its (always hidden) cells
                if (!collection2.isEntryVisible(activeCell.get(rows))) {
                    // try to find a visible column/row inside the merged range
                    if ((mergedRange = mergeCollection.getMergedRange(activeCell))) {
                        entryData = collection2.getVisibleEntry(activeCell.get(rows), 'nextPrev', mergedRange.interval(rows));
                    }
                    // no merged range found, or completely hidden: try to find a visible entry in the entire sheet
                    if (!entryData) {
                        entryData = collection2.getVisibleEntry(activeCell.get(rows), 'nextPrev');
                    }
                    // adjust active cell position, if an entry has been found
                    if (entryData) {
                        activeCell.set(entryData.index, rows);
                    } else {
                        return false;
                    }
                }

                // move active cell until it is visible
                do {

                    // move once to the previous/next cell
                    if (!(forward ? moveToNextCell() : moveToPrevCell())) {
                        return false;
                    }

                    // prevent endless loop (the entire selection may be located in hidden columns or rows)
                    if (!firstPos) {
                        firstPos = { active: selection.active, address: activeCell.clone() };
                    } else if ((firstPos.active === selection.active) && firstPos.address.equals(activeCell)) {
                        return false;
                    }

                    // continue until a visible cell has been found (may either be hidden by columns/rows, or covered by a merged range)
                } while (!colCollection.isEntryVisible(activeCell[0]) || !rowCollection.isEntryVisible(activeCell[1]) || mergeCollection.isHiddenCell(activeCell));

                // search for unlocked cells in locked sheets (only in horizontal mode)
                if (unlockedCells && !cellCollection.getCellEntry(activeCell).attributes.cell.unlocked) {

                    var // the address of the active range in the selection
                        activeRange = selection.activeRange(),
                        // a range list that covers all selected cells but the active cell, starting right after the active cell
                        searchRanges = new RangeArray(),
                        // the index of the current range in the array of selected ranges
                        rangeIndex = selection.active;

                    var pushSearchRange = function (range) {
                        range.index = rangeIndex;
                        searchRanges.push(range);
                    };

                    // add the cells in the same row and following the active cell
                    if (activeCell[0] < activeRange.end[0]) {
                        pushSearchRange(Range.create(activeCell[0] + 1, activeCell[1], activeRange.end[0], activeCell[1]));
                    }

                    // add the remaining range below the active cell
                    if (activeCell[1] < activeRange.end[1]) {
                        pushSearchRange(Range.create(activeRange.start[0], activeCell[1] + 1, activeRange.end[0], activeRange.end[1]));
                    }

                    // add all other ranges in the selection, starting with the range following the active range
                    for (rangeIndex = incRangeIndex(rangeIndex); rangeIndex !== selection.active; rangeIndex = incRangeIndex(rangeIndex)) {
                        pushSearchRange(selection.ranges[rangeIndex].clone());
                    }

                    // back to active range: add the remaining range above the active cell
                    if (activeRange.start[1] < activeCell[1]) {
                        pushSearchRange(Range.create(activeRange.start[0], activeRange.start[1], activeRange.end[0], activeCell[1] - 1));
                    }

                    // add the cells in the same row and preceding the active cell
                    if (activeRange.start[0] < activeCell[0]) {
                        pushSearchRange(Range.create(activeRange.start[0], activeCell[1], activeCell[0] - 1, activeCell[1]));
                    }

                    // find the first or last unlocked cell in these ranges (TODO: consider unlocked columns)
                    cellCollection.iterateCellsInRanges(searchRanges, function (cellData, origRange) {
                        if (cellData.attributes.cell.unlocked) {
                            selection.address = cellData.address;
                            selection.active = origRange.index;
                            return Utils.BREAK;
                        }
                    }, { type: 'existing', ordered: true, reverse: !forward });
                }

                return true;
            }());

            // try to find a visible cell (may fail, if the entire selection is hidden)
            if (success) {

                // select the new cell (if jumping through unlocked cells), or set the selection with the new active cell
                if (singleCell) {
                    self.selectCell(selection.address, { storeOrigin: !unlockedCells });
                } else {
                    self.setCellSelection(selection);
                }
                // scroll to the effective active cell
                self.scrollToCell(selection.address);

            // if the entire selection is invisible (e.g. after hiding it completely when collapsing
            // columns or rows), remove selection and try to move the active cell in the sheet
            } else if (!singleCell && !unlockedCells) {
                selectNextCell(direction, 'cell');
            }
        }

        /**
         * Expands the selected ranges to entire columns or rows.
         *
         * @param {Boolean} columns
         *  If set to true, the current selection will be expanded to full
         *  columns. If set to false, the current selection will be expanded to
         *  full rows.
         */
        function expandSelectionToSheet(columns) {

            var // collection of merged ranges in the active sheet
                mergeCollection = self.getMergeCollection(),
                // current selection in the active sheet
                selection = getActiveSelection(),
                // the last available column/row index in the sheet
                lastIndex = docModel.getMaxIndex(!columns);

            // expand all ranges to the sheet borders
            selection.ranges.forEach(function (range, index) {
                range.setStart(0, columns);
                range.setEnd(lastIndex, columns);
                selection.ranges[index] = mergeCollection.expandRangeToMergedRanges(range);
            });

            self.setCellSelection(selection, { unique: true });
        }

        /**
         * Selects the content range (the entire range containing consecutive
         * content cells) surrounding the current active cell (ignoring the
         * current selected ranges).
         */
        function selectContentRange() {

            var // current active cell in the active sheet
                activeCell = getActiveSelection().address,
                // the content range to be selected
                contentRange = self.getCellCollection().findContentRange(new Range(activeCell));

            // select the resulting expanded range
            self.selectRange(contentRange, { active: activeCell });
        }

        /**
         * Selects the previous or next drawing frame.
         */
        function selectNextDrawing(backwards) {

            var // last drawing selected in the active sheet
                position = _.last(self.getSelectedDrawings()),
                // number of drawings in the sheet
                count = self.getDrawingCollection().getModelCount();

            // TODO: selection in embedded objects
            if ((count > 1) && _.isArray(position) && (position.length === 1) && (position[0] >= 0) && (position[0] < count)) {
                position[0] = (backwards ? (position[0] + count - 1) : (position[0] + 1)) % count;
                self.selectDrawing(position);
                self.scrollToDrawingFrame(position);
            }
        }

        /**
         * Handles 'keydown' events from all grid panes to adjust the current
         * selection.
         */
        function keyDownHandler(event) {

            var // whether cell in-place edit mode is active
                cellEditMode = self.isCellEditMode();

            // special handling for drawing selection
            if (self.hasDrawingSelection() && !cellEditMode) {

                // traverse through drawing frames with TAB key
                if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                    selectNextDrawing(event.shiftKey);
                    return false;
                }

                return;
            }

            // no modifier keys: delete selection, move active cell
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null })) {

                switch (event.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'cell');
                    return false;
                case KeyCodes.RIGHT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'cell');
                    return false;
                case KeyCodes.UP_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'cell');
                    return false;
                case KeyCodes.DOWN_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'cell');
                    return false;
                case KeyCodes.PAGE_UP:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'page');
                    return false;
                case KeyCodes.PAGE_DOWN:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'page');
                    return false;
                case KeyCodes.TAB:
                    moveActiveCell(event.shiftKey ? 'left' : 'right');
                    return false;
                case KeyCodes.ENTER:
                    moveActiveCell(event.shiftKey ? 'up' : 'down');
                    return false;
                case KeyCodes.HOME:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(new Address(getHomeCell()[0], getActiveSelection().address[1]), false);
                    return false;
                case KeyCodes.END:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(new Address(getEndCell()[0], getActiveSelection().address[1]), true);
                    return false;
                }
            }

            // additional selection with ALT modifier key
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null, alt: true })) {

                switch (event.keyCode) {
                case KeyCodes.PAGE_UP:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'page');
                    return false;
                case KeyCodes.PAGE_DOWN:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'page');
                    return false;
                }
            }

            // additional selection with CTRL or META modifier key
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null, ctrlOrMeta: true })) {

                switch (event.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'used');
                    return false;
                case KeyCodes.RIGHT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'used');
                    return false;
                case KeyCodes.UP_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'used');
                    return false;
                case KeyCodes.DOWN_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'used');
                    return false;
                case KeyCodes.HOME:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(getHomeCell(), false);
                    return false;
                case KeyCodes.END:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(getEndCell(), true);
                    return false;
                }
            }

            // CTRL+asterisk (numpad): select content range around active cell
            if (KeyCodes.matchKeyCode(event, 'NUM_MULTIPLY', { ctrlOrMeta: true })) {
                selectContentRange();
                return false;
            }

            // bug 31247: remaining shortcuts not in cell in-place edit mode
            if (cellEditMode) { return; }

            // convert selection to entire columns
            if (KeyCodes.matchKeyCode(event, 'SPACE', { ctrlOrMeta: true })) {
                expandSelectionToSheet(true);
                return false;
            }

            // convert selection to entire rows
            if (KeyCodes.matchKeyCode(event, 'SPACE', { shift: true })) {
                expandSelectionToSheet(false);
                return false;
            }

            // select entire sheet
            if (KeyCodes.matchKeyCode(event, 'SPACE', { shift: true, ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                self.selectRange(docModel.getSheetRange(), { active: self.getActiveCell() });
                return false;
            }
        }

        /**
         * Registers a listener at the specified object. Execution of the event
         * handlers will be deferred until the browser focus is located inside
         * the application pane.
         *
         * @param {Object} source
         *  The event source object. Can be any object that provides a method
         *  'on()' to register an event listener.
         *
         * @param {String} type
         *  The type of the event to listen to. Can be a space-separated list
         *  of event type names.
         *
         * @param {Function} listener
         *  The event listener that will be invoked for the events triggered by
         *  the event source object.
         */
        var listenToWithFocus = (function () {

            // This method is a fix for bug 30513: when entering something into a text field
            // (e.g. a new font size), and directly clicking on a new cell without pressing
            // ENTER key, the following events have been occurred before:
            // 1) 'mousedown' in the grid pane will start to select the clicked cell,
            // 2) 'blur' in the text field will trigger the action attached to the text field.
            // This results in setting the font size to the wrong (now selected) cell.
            //
            // With this method, all selection events triggered by the grid panes, header
            // panes, and corner pane will be deferred until the browser focus returns to the
            // application pane (i.e. any grid pane therein).

            // returns a Promise that will be resolved when the application pane has the focus
            function waitForViewFocus() {

                // do nothing if the application pane is currently focused
                if (self.getActiveGridPane().hasFocus()) { return $.when(); }

                var // the root node of the application pane
                    appPaneNode = self.getAppPaneNode(),
                    // the deferred object that will be resolved after the focus has returned into the application pane
                    def = $.Deferred();

                // let the focus event bubble up, then resolve the deferred object
                function focusHandler() {
                    _.defer(function () { def.resolve(); });
                }

                // listen to the next 'focusin' event at the application pane
                appPaneNode.one('focusin', focusHandler);

                // fail-safety (prevent deferring selection for too long): timeout after 200ms
                return self.createAbortablePromise(def, function () {
                    appPaneNode.off('focusin', focusHandler);
                }, 200);
            }

            // create a single synchronized method that checks the browser focus (and
            // waits for it if necessary), and invokes the passed event handler afterwards
            var synchronizedHandler = self.createSynchronizedMethod(function (handler, args) {

                var // store calling context for usage in always() handler
                    context = this;

                // wait for the browser focus to return to the application pane, then invoke the
                // event handler, return the promise to make the synchronized method working
                return waitForViewFocus().always(function () {
                    handler.apply(context, args);
                });
            });

            // the actual listenToWithFocus() method
            function listenToWithFocus(source, type, handler) {
                // listen to the event and invoke the synchronized method (pass the real event handler as argument)
                source.on(type, function () { synchronizedHandler.call(this, handler, arguments); });
            }

            return listenToWithFocus;
        }());

        /**
         * Handles changed cell or drawing selections.
         */
        function changeSelectionHandler() {
            // cancel custom selection mode when selecting drawings
            if (self.hasDrawingSelection()) {
                leaveCustomSelectionMode(false);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Registers a callback function that can change the behavior of the
         * application when selecting cell ranges with mouse or keyboard.
         *
         * @param {Function} getHandler
         *  A callback function that will be invoked to receive a different
         *  cell selection to be used for all selection code, especially
         *  changing the selection with keyboard shortcuts. Will be called in
         *  the context of this instance. Must return a selection object, or
         *  null. The latter indicates to use the actual cell selection of the
         *  active sheet.
         *
         * @param {Function} setHandler
         *  A callback function that will be invoked before actually changing
         *  the current cell selection in the active sheet. Will be called in
         *  the context of this instance. Receives the new selection intended
         *  to be set at the active sheet as first parameter. If this function
         *  has processed the selection by itself, it has to return true to
         *  prevent changing the cell selection in the active sheet.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.registerCellSelectionHandlers = function (getHandler, setHandler) {
            getCellSelectionHandler = getHandler;
            setCellSelectionHandler = setHandler;
            return this;
        };

        /**
         * Unregisters the callback functions that have been registered with
         * the method SelectionMixin.registerCellSelectionHandlers().
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.unregisterCellSelectionHandlers = function () {
            getCellSelectionHandler = setCellSelectionHandler = null;
            return this;
        };

        /**
         * Switches the view to a special mode where selecting the next cell
         * range in the document will be displayed as 'active selection' (with
         * a marching ants effect), and that can be handled afterwards via the
         * promise returned by this method.
         *
         * @param {String} id
         *  An arbitrary name that identifies the custom selection mode that is
         *  currently active.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {SheetSelection} [options.selection]
         *      The initial selection shown when starting the custom selection
         *      mode.
         *  @param {String|Object} [options.statusLabel]
         *      The status label to be shown in the status pane while custom
         *      selection mode is active. Can be a string, or a caption options
         *      object with label and icon settings.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the custom selection mode has
         *  finished successfully; or that will be rejected, if the custom
         *  selection mode has been canceled. If the custom selection mode has
         *  been completed successfully, the promise's done handlers receive
         *  the selection object as first parameter.
         */
        this.enterCustomSelectionMode = function (id, options) {

            // cancel previous custom selection if active
            leaveCustomSelectionMode(false);

            // create a new deferred object representing the custom selection mode
            customSelectionId = id;
            customSelectionDef = $.Deferred();

            // set the initial selection
            this.setSheetViewAttribute('activeSelection', Utils.getObjectOption(options, 'selection', null));

            // register custom selection handler to draw the active selection while user selects
            this.registerCellSelectionHandlers(function () {
                return self.getSheetViewAttribute('activeSelection');
            }, function (selection) {
                self.setSheetViewAttribute('activeSelection', selection);
                return true;
            });

            // set status label text
            this.setStatusLabel(Utils.getOption(options, 'statusLabel', null));

            return customSelectionDef.promise();
        };

        /**
         * Returns whether the specified custom selection mode is currently
         * active.
         *
         * @param {String} id
         *  An identifier of a custom selection mode, as passed to the method
         *  ViewFuncMixin.enterCustomSelectionMode().
         *
         * @returns {Boolean}
         *  Whether the specified custom selection mode is currently active.
         */
        this.isCustomSelectionMode = function (id) {
            return customSelectionId === id;
        };

        /**
         * Cancels the custom selection mode that has been started with the
         * method SelectionMixin.enterCustomSelectionMode() before.
         *
         * @returns {Boolean}
         *  Whether the custom selection mode was active and has been left.
         */
        this.cancelCustomSelectionMode = function () {
            if (_.isString(customSelectionId)) {
                leaveCustomSelectionMode(false);
                return true;
            }
            return false;
        };

        /**
         * Returns the current selection in the active sheet. This method
         * returns a clone of the selection that can be freely manipulated.
         *
         * @returns {SheetSelection}
         *  The current selection in the active sheet.
         */
        this.getSelection = function () {
            return getSheetSelection().clone();
        };

        /**
         * Returns the addresses of all selected cell ranges in the active
         * sheet. This method returns a clone of the selected ranges that can
         * be freely manipulated.
         *
         * @returns {RangeArray}
         *  The addresses of all selected cell ranges in the active sheet.
         */
        this.getSelectedRanges = function () {
            return getSheetSelection().ranges.clone(true);
        };

        /**
         * Returns the address of the active cell range in the selection of the
         * active sheet. This method returns a clone of the active range that
         * can be freely manipulated.
         *
         * @returns {Range}
         *  The address of the active cell range in the selection of the active
         *  sheet.
         */
        this.getActiveRange = function () {
            return getSheetSelection().activeRange().clone();
        };

        /**
         * Returns the address of the active cell in the selection of the
         * active sheet (the highlighted cell in the active range that will be
         * edited when switching to text edit mode). This method returns a
         * clone of the active cell address that can be freely manipulated.
         *
         * @returns {Address}
         *  The address of the active cell in the selection of the active
         *  sheet.
         */
        this.getActiveCell = function () {
            return getSheetSelection().address.clone();
        };

        /**
         * Returns the active cell of the selection in the active sheet (the
         * highlighted cell in the active range that will be edited when
         * switching to text edit mode), as a complete selection object
         * containing a single range that consists of the active cell.
         *
         * @returns {SheetSelection}
         *  The position of the active cell, as a new sheet selection object.
         */
        this.getActiveCellAsSelection = function () {
            return SheetSelection.createFromAddress(this.getActiveCell());
        };

        /**
         * Returns whether the current selection consists of the active cell
         * only, either as single cell, or exactly covering a merged range.
         *
         * @returns {Boolean}
         *  Whether the current selection consists of the active cell only.
         */
        this.isSingleCellSelection = function () {
            var selection = this.getSelection();
            return (selection.ranges.length === 1) && this.getSheetModel().isSingleCellInRange(selection.ranges[0]);
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleColumnsSelected = function () {
            if (this.isSingleCellSelection()) { return false; } // single cell checks also the merged range
            return this.getSelectedRanges().some(function (range) { return !range.singleCol(); });
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleRowsSelected = function () {
            if (this.isSingleCellSelection()) { return false; } // single cell checks also the merged range
            return this.getSelectedRanges().some(function (range) { return !range.singleRow(); });
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple cells.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple cells.
         */
        this.hasAnyRangeSelected = function () {
            return this.getSelectedRanges().some(function (range) { return !range.single(); });
        };

        /**
         * Returns whether the current selection contains or partly overlaps at
         * least one merged cell range.
         *
         * @returns {Boolean}
         *  Whether the current selection contains or partly overlaps at least
         *  one merged cell range.
         */
        this.hasMergedRangeSelected = function () {
            return this.getMergeCollection().rangesOverlapAnyMergedRange(this.getSelectedRanges());
        };

        /**
         * Returns, whether a drawing object is selected in the current active
         * sheet.
         *
         * @returns {Boolean}
         *  Whether a drawing object is selected in the active sheet.
         */
        this.hasDrawingSelection = function () {
            return getSheetSelection().drawings.length > 0;
        };

        /**
         * Returns the positions of all selected drawing frames in the active
         * sheet.
         *
         * @returns {Array<Array<Number>>}
         *  The selected drawing frames, as array of drawing positions.
         */
        this.getSelectedDrawings = function () {
            return _.copy(getSheetSelection().drawings, true);
        };

        /**
         * Changes the cell selection in the active sheet, and deselects all
         * drawing frames.
         *
         * @param {SheetSelection} selection
         *  The new sheet selection. The array of cell range addresses must not
         *  be empty. The array of selected drawing objects must be empty.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the active cell passed in the selection
         *      is hidden by a merged range, the address of the original active
         *      cell will be stored in the property 'origin' of the sheet
         *      selection.
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.setCellSelection = function (selection, options) {

            var // a merged range hiding the active cell
                mergedRange = this.getMergeCollection().getMergedRange(selection.address, 'hidden'),
                // the new active range
                activeRange = selection.activeRange();

            // deselect all drawing frames
            selection = selection.clone();
            selection.drawings = [];
            selection.origin = null;

            // adjust active cell if it is hidden by a merged range
            if (mergedRange) {
                if (Utils.getBooleanOption(options, 'storeOrigin', false)) {
                    selection.origin = selection.address;
                }
                selection.address = mergedRange.start.clone();
            }

            // restrict to single selection if multi-selection is not supported
            if (!SheetUtils.MULTI_SELECTION) {
                selection.ranges.clear().push(activeRange);
                selection.active = 0;
            }

            // remove duplicates
            if (Utils.getBooleanOption(options, 'unique', false) && (selection.ranges.length > 1)) {
                selection.ranges = selection.ranges.unify();
                selection.active = Utils.findFirstIndex(selection.ranges, function (range) { return range.equals(activeRange); });
            }

            // prefer the registered cell selection handler
            if (!_.isFunction(setCellSelectionHandler) || (setCellSelectionHandler.call(this, selection) !== true)) {

                // leave cell in-place edit mode before changing the selection
                if (this.leaveCellEditMode('auto', { validate: true })) {
                    setSheetSelection(selection);
                }
            }

            return this;
        };

        /**
         * Selects a single cell in the active sheet. The cell will become the
         * active cell of the sheet.
         *
         * @param {Address} address
         *  The address of the cell to be selected.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new cell will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  @param {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the specified cell is hidden by a merged
         *      range, the address of the passed cell will be stored in the
         *      property 'origin' of the sheet selection.
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection (only useful when appending a cell to the
         *      selection).
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectCell = function (address, options) {
            return this.selectRange(new Range(address), _.extend({}, options, { active: address }));
        };

        /**
         * Selects a single cell range in the active sheet.
         *
         * @param {Range} range
         *  The address of the cell range to be selected.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new range will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  @param {Address} [options.active]
         *      The address of the active cell in the selected range. If
         *      omitted, the top-left cell of the range will be activated. Must
         *      be located inside the passed range address.
         *  @param {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the new active cell is hidden by a
         *      merged range, the address of the active cell will be stored in
         *      the property 'origin' of the sheet selection.
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection (only useful when appending a range to the
         *      selection).
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectRange = function (range, options) {

            var // whether to append the range to the current selection
                append = Utils.getBooleanOption(options, 'append', false),
                // the initial selection to append the new range to
                selection = append ? getActiveSelection({ allowEmpty: true }) : new SheetSelection(),
                // the new active cell
                activeCell = Utils.getObjectOption(options, 'active'),
                // collection of merged ranges in the active sheet
                mergeCollection = this.getMergeCollection(),
                // merged ranges processed while validating the active cell
                mergedRange = null,
                // next cell tested while validating the active cell
                nextCell = activeCell ? activeCell.clone() : null;

            // returns a cell address object, if the specified cell position is inside the passed selection range
            function getValidCellAddress(col, row) {
                var address = new Address(col, row);
                return range.containsAddress(address) ? address : null;
            }

            // expand range to merged ranges
            range = expandRangeToMergedRanges(range);

            // bug 33313: move active cell if it is outside the active range due to a partly covered merged range
            while (nextCell && (mergedRange = mergeCollection.getMergedRange(nextCell)) && !range.contains(mergedRange)) {
                nextCell = getValidCellAddress(nextCell[0], mergedRange.end[1] + 1) || getValidCellAddress(mergedRange.end[0] + 1, nextCell[1]);
            }
            if (nextCell) { activeCell = nextCell; }

            // initialize and set the new selection
            selection.ranges.push(range);
            selection.active = selection.ranges.length - 1;

            // set active cell
            if (activeCell && range.containsAddress(activeCell)) {
                selection.address = activeCell.clone();
            } else {
                selection.address = range.start.clone();
            }

            // commit new selection
            return this.setCellSelection(selection, options);
        };

        /**
         * Modifies the position of the specified selection range.
         *
         * @param {Number} index
         *  The array index of the range to be changed.
         *
         * @param {Range} range
         *  The new address of the selected range. If the specified range index
         *  refers to the active range, but the new range does not contain the
         *  current active cell, the active cell will be moved inside the new
         *  range.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection.
         *  @param {Boolean} [options.tracking=false]
         *      If set to true, the view attribute 'trackingSelection' will be
         *      set or changed instead of the 'selection' view attribute.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.changeRange = function (index, range, options) {

            var // the selection to be modified
                selection = getActiveSelection();

            // expand range to merged ranges
            range = expandRangeToMergedRanges(range);

            // insert the new range into the selection object
            selection.ranges[index] = range;
            if ((index === selection.active) && !range.containsAddress(selection.address)) {
                selection.address[0] = Utils.minMax(selection.address[0], range.start[0], range.end[0]);
                selection.address[1] = Utils.minMax(selection.address[1], range.start[1], range.end[1]);
            }

            // commit new selection
            return this.setCellSelection(selection, options);
        };

        /**
         * Modifies the position of the current active selection range.
         *
         * @param {Range} range
         *  The new address of the active selection range. If the passed range
         *  does not contain the current active cell, the top-left cell of the
         *  range will be activated.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.changeActiveRange = function (range, options) {
            return this.changeRange(getActiveSelection().active, range, options);
        };

        /**
         * Modifies the position of the current active cell.
         *
         * @param {Number} index
         *  The array index of the range to be activated.
         *
         * @param {Address} address
         *  The new address of the active cell.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.changeActiveCell = function (index, address) {

            var // the selection to be modified
                selection = getActiveSelection();

            // update index of active range, and active cell
            selection.active = Utils.minMax(index, 0, selection.ranges.length - 1);
            selection.address = address.clone();

            // commit new selection
            return this.setCellSelection(selection);
        };

        /**
         * If a single cell is selected (also if it is a merged range), the
         * selection will be expanded to the surrounding content range.
         * Otherwise (multiple cells in selection, or multi-selection), the
         * selection remains unmodified.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.expandToContentRange = function () {

            var // the current selection
                selection = this.getSelection(),
                // the expanded content range for a single selected cell
                contentRange = (selection.ranges.length === 1) ? this.getSheetModel().getContentRangeForCell(selection.ranges.first()) : null;

            // select content range, if it is different from the current range
            if (contentRange && contentRange.differs(selection.ranges.first())) {
                this.changeRange(0, contentRange);
            }

            return this;
        };

        /**
         * Selects the specified drawing frames in the active sheet.
         *
         * @param {Array<Array<Number>>} positions
         *  The positions of all drawing frames to be selected.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.setDrawingSelection = function (positions) {

            // first, leave cell in-place edit mode
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return this;
            }

            // the current selection to be modified (keep cell selection as it is)
            var selection = this.getSelection();
            selection.drawings = positions;

            // commit new selection
            setSheetSelection(selection);
            return this;
        };

        /**
         * Selects a single drawing frame in the active sheet.
         *
         * @param {Array<Number>} position
         *  The position of the drawing frame to be selected.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  @param {Boolean} [options.toggle=false]
         *      If set to true, the selection state of the addressed drawing
         *      frame will be toggled. Otherwise, the old drawing selection
         *      will be removed, and the addressed drawing frame will be
         *      selected.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectDrawing = function (position, options) {

            // first, leave cell in-place edit mode
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return this;
            }

            var // the current selection to be modified (keep cell selection as it is)
                selection = this.getSelection();

            // construct the new drawing selection
            if (!Utils.getBooleanOption(options, 'toggle', false)) {
                selection.drawings = [position];
            } else if (SheetUtils.MULTI_SELECTION && (Utils.spliceValue(selection.drawings, position) === 0)) {
                selection.drawings.push(position);
            }

            // commit new selection
            setSheetSelection(selection);
            return this;
        };

        /**
         * Removes selection from all selected drawing objects and restores the
         * cell selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.removeDrawingSelection = function () {

            var // the current selection to be modified (keep cell selection as it is)
                selection = this.getSelection();

            // remove drawing selection
            selection.drawings.splice(0);

            // commit new selection
            setSheetSelection(selection);
            return this;
        };

        /**
         * Returns the drawing model instances of all selected drawing objects.
         *
         * @returns {Array<DrawingModel>}
         *  An array containing the models of all selected drawing objects, in
         *  order of the current selection.
         */
        this.getSelectedDrawingModels = function () {
            // TODO: move to a drawing layer mix-in that provides application abstraction

            var // the drawing collection of the active sheet
                drawingCollection = this.getDrawingCollection(),
                // all selected drawing models
                drawingModels = [];

            // pick all selected drawing models from the collection
            _.each(this.getSelection().drawings, function (position) {
                var drawingModel = drawingCollection.findModel(position, { deep: true });
                if (drawingModel) { drawingModels.push(drawingModel); }
            });

            return drawingModels;
        };

        // initialization -----------------------------------------------------

        // cancel the custom selection mode when the active sheet changes (TODO: allow on multiple sheets)
        this.on('change:activesheet', function () { leaveCustomSelectionMode(false); });

        // listen to events from corner pane to adjust the selection
        listenToWithFocus(this.getCornerPane(), 'select:all', function (event, mode) {

            var // select mode: move active cell the top-left cell in the first visible grid pane
                // extend mode: do not change the current active cell
                topLeftAddress = (mode === 'select') ? self.getVisibleGridPane('topLeft').getTopLeftAddress() : getActiveSelection().address;

            self.selectRange(docModel.getSheetRange(), { active: topLeftAddress });
            leaveCustomSelectionMode(true);
        });

        // listen to events from all header panes to adjust the selection
        this.iterateHeaderPanes(function (headerPane, paneSide) {

            var // whether the header pane shows column headers
                columns = PaneUtils.isColumnSide(paneSide),
                // the column/row index where selection tracking has started
                startIndex = 0;

            // makes a cell address according to the orientation of the header pane
            function makeCell(index1, index2) {
                return columns ? new Address(index1, index2) : new Address(index2, index1);
            }

            // makes an adjusted range between start index and passed index
            function makeRange(index) {
                return docModel.makeFullRange(Interval.create(startIndex, index), columns);
            }

            listenToWithFocus(headerPane, 'select:start', function (event, index, mode) {
                var selection = getActiveSelection({ allowEmpty: true }),
                    customMode = _.isString(customSelectionId);
                if (!customMode && (mode === 'extend') && !selection.ranges.empty()) {
                    // modify the active range (from active cell to tracking start index)
                    startIndex = selection.address.get(columns);
                    self.changeActiveRange(makeRange(index));
                } else {
                    // create/append a new selection range
                    startIndex = index;
                    // set active cell to current first visible cell in the column/row
                    self.selectRange(makeRange(index), {
                        append: !customMode && (mode === 'append'),
                        active: makeCell(startIndex, Math.max(0, self.getVisibleHeaderPane(!columns).getFirstVisibleIndex()))
                    });
                }
            });

            listenToWithFocus(headerPane, 'select:move', function (event, index) {
                self.changeActiveRange(makeRange(index));
            });

            listenToWithFocus(headerPane, 'select:end', function (event, index, success) {
                self.changeActiveRange(makeRange(index), { unique: true });
                leaveCustomSelectionMode(success);
            });
        });

        // listen to events from all grid panes to adjust the selection
        this.iterateGridPanes(function (gridPane) {

            var // the address of the cell where selection tracking has started
                startAddress = null;

            // makes an adjusted range from the start address to the passed address
            function makeRange(address) {
                return Range.createFromAddresses(startAddress, address);
            }

            listenToWithFocus(gridPane, 'select:start', function (event, address, mode) {
                var selection = getActiveSelection({ allowEmpty: true }),
                    customMode = _.isString(customSelectionId);
                if (!customMode && (mode === 'extend') && !selection.ranges.empty()) {
                    // modify the active range (from active cell to tracking start position)
                    startAddress = selection.address;
                    self.changeActiveRange(makeRange(address));
                } else {
                    // create/append a new selection range
                    startAddress = address;
                    self.selectCell(address, { append: !customMode && (mode === 'append') });
                }
            });

            listenToWithFocus(gridPane, 'select:move', function (event, address) {
                self.changeActiveRange(makeRange(address));
            });

            listenToWithFocus(gridPane, 'select:end', function (event, address, success) {
                self.changeActiveRange(makeRange(address), { unique: true });
                leaveCustomSelectionMode(success);
            });

            listenToWithFocus(gridPane, 'select:range', function (event, index, range) {
                leaveCustomSelectionMode(false);
                self.changeRange(index, range);
            });

            listenToWithFocus(gridPane, 'select:active', function (event, index, address) {
                leaveCustomSelectionMode(false);
                self.changeActiveCell(index, address);
            });

            listenToWithFocus(gridPane, 'select:dblclick', function () {
                self.enterCellEditMode();
            });

            // forward selection rendering events of the active grid pane
            self.listenTo(gridPane, 'render:cellselection render:drawingselection', function (event) {
                if (gridPane === self.getActiveGridPane()) {
                    self.trigger.apply(self, [event.type].concat(_.toArray(arguments).slice(1)));
                }
            });
        });

        // listen to key events from all panes to adjust the selection
        rootNode.on('keydown', keyDownHandler);

        // cancel custom selection mode when losing edit rights, or before applying any operations
        this.listenTo(docModel, 'change:editmode', function (event, editMode) { if (!editMode) { leaveCustomSelectionMode(false); } });
        this.listenTo(docModel, 'operations:before', function () { leaveCustomSelectionMode(false); });

        // cancel custom selection mode when entering cell edit mode
        this.on('celledit:enter', function () { leaveCustomSelectionMode(false); });

        // handle changed selection in sheet view attributes
        this.on('change:selection', changeSelectionHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = rootNode = null;
            getCellSelectionHandler = setCellSelectionHandler = customSelectionDef = null;
        });

    } // class SelectionMixin

    // exports ================================================================

    return SelectionMixin;

});
