/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/celleditmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/fontmetrics',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/popup/tooltip',
    'io.ox/office/tk/popup/listmenu',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/sheetselection',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/popup/signaturetooltip',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, KeyCodes, Forms, FontMetrics, Tracking, LocaleData, ToolTip, ListMenu, Color, HyperlinkUtils, SheetUtils, SheetSelection, CellCollection, TokenArray, SignatureToolTip, gt) {

    'use strict';

    var // convenience shortcuts
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray,

        // number of milliseconds per day
        MSEC_PER_DAY = 86400000,

        // key codes of all keys used to navigate in text area, or in the sheet (quick edit mode)
        NAVIGATION_KEYCODES = Utils.makeSet([
            KeyCodes.LEFT_ARROW,
            KeyCodes.RIGHT_ARROW,
            KeyCodes.UP_ARROW,
            KeyCodes.DOWN_ARROW,
            KeyCodes.PAGE_UP,
            KeyCodes.PAGE_DOWN,
            KeyCodes.HOME,
            KeyCodes.END
        ]);

    // mix-in class CellEditMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * the cell in-place edit mode.
     *
     * @constructor
     */
    function CellEditMixin() {

        var // self reference (spreadsheet view instance)
            self = this,

            // the application instance
            app = this.getApp(),

            // the spreadsheet model, and other model objects
            docModel = this.getDocModel(),
            undoManager = docModel.getUndoManager(),
            fontCollection = docModel.getFontCollection(),
            numberFormatter = docModel.getNumberFormatter(),

            // the underlay node containing highlighting for formulas
            textUnderlayNode = $('<div class="underlay">'),
            // the text area control
            textAreaNode = $(Forms.createTextAreaMarkup()),
            // the container element for the text underlay node and the text area control
            textAreaContainer = $('<div class="textarea-container" data-focus-role="textarea">').append(textUnderlayNode, textAreaNode),

            // tooltip attached to the text area for function/parameter help
            signatureToolTip = new SignatureToolTip(this, textAreaNode),
            // drop-down list attached to the text area (for function names)
            textAreaListMenu = new ListMenu({ anchor: textAreaNode, autoFocus: false, focusableNodes: textAreaNode }),
            // tooltip attached to the text area drop-down list (for the selected function)
            textAreaListItemToolTip = new ToolTip({ anchor: textAreaListMenu.getNode(), anchorBorder: 'right left', anchorAlign: 'center', anchorAlignBox: getTextAreaListButtonNode }),
            // localized function names of all visible functions
            functionNames = null,

            // whether the in-place cell edit mode is active
            editActive = false,
            // the index of the sheet containing the edited cell (active sheet may change during edit mode)
            editSheet = null,
            // the model of the sheet containing the edited cell
            editSheetModel = null,
            // the cell address of the edited cell
            editAddress = null,
            // the grid pane instance where in-place cell edit mode is active
            editGridPane = null,
            // the descriptor of the edited cell (result, attributes, etc.)
            editCellDesc = null,
            // the token descriptors received form the tokenizer for the current formula in the text area
            editTokenDescs = null,
            // the token array representing the current formula in the text area
            editTokenArray = null,
            // current character attributes for the text area
            editAttributes = null,

            // local undo/redo information while cell edit mode is active
            undoStack = null,
            // pointer into the local undo/redo stack
            undoIndex = 0,

            // initial text at the beginning of the edit mode
            originalText = null,
            // the text at the time the text area has been updated the last time
            prevEditText = null,
            // collected entries for type-ahead suggestions
            typeAheadEntries = null,
            // last committed cell range address, and additional data (e.g. circular reference detected)
            lastCommitData = null,

            // quick edit mode (true), or text cursor mode (false)
            quickEditMode = null,
            // the formula expression, if the input string starts with specific characters
            formulaString = null,
            // selection offset into the formula string (1, if formula starts with equality sign; or 0 for plus/minus)
            formulaOffset = 0,
            // unique identifier of the current range highlighting mode for formulas
            highlightUid = null,
            // whether a matrix formula is currently edited
            matrixFormula = false,
            // whether to show auto-completion list for a function or defined name
            funcAutoComplete = false,
            // token descriptor for function/name auto-completion
            funcAutoTokenDesc = null,
            // whether to keep auto-completion list open although focus is not in text area
            stickyAutoComplete = false,
            // how many closing parenthesis entered manually will be ignored
            funcAutoCloseData = [],
            // the text position of the current range address while in range selection mode
            rangeSelection = null;

        // private methods ----------------------------------------------------

        /**
         * Returns the initial text for the in-place cell edit mode. Returns
         * the formula expression for formula cells, otherwise the cell value
         * formatted with an appropriate default number format.
         *
         * @returns {String}
         *  The initial text for the in-place cell edit mode.
         */
        function getTextForCellEditMode() {

            // formulas: hide formula if the cell contains the hidden attribute
            if (_.isString(editCellDesc.formula)) {
                return (self.isSheetLocked() && editCellDesc.attributes.cell.hidden) ? '' : editCellDesc.formula;
            }

            // the result value of the cell to be formatted
            var value = editCellDesc.result;

            // strings: use plain unformatted string for editing (bug 34421: add an apostrophe if necessary)
            if (_.isString(value)) {
                return ((/^['=]/).test(value) || app.isLocalizedErrorCode(value) || numberFormatter.parseFormattedNumber(value)) ? ('\'' + value) : value;
            }

            // Booleans: use plain Boolean literal for editing
            if (_.isBoolean(value)) {
                return app.getBooleanLiteral(value);
            }

            // error codes: use plain error code literal for editing
            if (SheetUtils.isErrorCode(value)) {
                return app.convertErrorCodeToString(value);
            }

            // numbers: use appropriate number representation according to number format category
            if (_.isNumber(value) && isFinite(value)) {

                // the resulting formatted value
                var formatted = null;

                // process different format categories
                var category = editCellDesc.format.cat;
                switch (category) {

                // percent: multiply by 100, add percent sign without whitespace
                case 'percent':
                    value *= 100; // may become infinite
                    formatted = (isFinite(value) ? numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT) : '') + '%';
                    break;

                // automatically show date and/or time, according to the number
                case 'date':
                case 'time':
                case 'datetime':

                    var // number of milliseconds
                        milliSecs = Math.round(value * MSEC_PER_DAY),
                        // number of days
                        date = Math.floor(milliSecs / MSEC_PER_DAY),
                        // number of milliseconds in the day
                        time = Math.floor(milliSecs % MSEC_PER_DAY),
                        // whether to add the date and time part to the result
                        showDate = (category !== 'time') || (date !== 0),
                        showTime = (category !== 'date') || (time !== 0),
                        // the resulting format code
                        formatCode = (showDate ? LocaleData.SHORT_DATE : '') + ((showDate && showTime) ? ' ' : '') + (showTime ? LocaleData.LONG_TIME : '');

                    // the resulting formatted value (may be null for invalid dates)
                    formatted = numberFormatter.formatValue(value, formatCode);
                    break;
                }

                // use 'General' number format for all other format codes
                return _.isString(formatted) ? formatted : numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT);
            }

            // empty cells
            return '';
        }

        /**
         * Updates the CSS formatting of the text area control used for cell
         * in-place edit mode according to the current formatting of the active
         * cell.
         */
        function updateTextAreaStyle() {

            var // the current zoom factor in the sheet
                zoom = editSheetModel.getEffectiveZoom(),
                // the address of the active cell
                address = editAddress;

            // applies a transformation to the passed color, and returns the CSS color value
            function getModifiedCssColor(jsonColor, colorType) {

                var // the passed color, as instance of the class Color
                    color = Color.parseJSON(jsonColor),
                    // the resolved color, including the effective luma
                    colorDesc = docModel.resolveColor(color, colorType);

                if ((colorType === 'text') && (colorDesc.y > 0.25)) {
                    color.transform('shade', 25000 / colorDesc.y);
                } else if ((colorType === 'fill') && (colorDesc.y < 0.75)) {
                    color.transform('tint', 25000 / (1 - colorDesc.y));
                }
                return docModel.resolveColor(color, colorType).css;
            }

            // receive cell contents from cell collection, overwrite with current edit attributes
            editCellDesc = editSheetModel.getCellCollection().getCellEntry(address);
            editCellDesc.attributes = _.copy(editCellDesc.attributes, true);
            _.extend(editCellDesc.attributes.cell, editAttributes.cell);
            _.extend(editCellDesc.attributes.character, editAttributes.character);

            // receive font settings, text line height, and text orientation
            editCellDesc.fontDesc = fontCollection.getFontDescriptor(editCellDesc.attributes.character, zoom);
            editCellDesc.fontDesc.fontSize = Utils.minMax(editCellDesc.fontDesc.fontSize, 6, 48);
            editCellDesc.lineHeight = docModel.getRowHeight(editCellDesc.attributes.character, zoom);
            editCellDesc.orientation = CellCollection.getTextOrientation(editCellDesc);

            // set CSS attributes according to formatting of active cell
            textAreaContainer.children().css({
                padding: '0 ' + editSheetModel.getEffectiveTextPadding() + 'px',
                lineHeight: editCellDesc.lineHeight + 'px',
                fontFamily: editCellDesc.fontDesc.family,
                fontSize: editCellDesc.fontDesc.size + 'pt',
                fontWeight: editCellDesc.fontDesc.bold ? 'bold' : 'normal',
                fontStyle: editCellDesc.fontDesc.italic ? 'italic' : 'normal',
                textDecoration: docModel.getCssTextDecoration(editCellDesc.attributes.character),
                textAlign: (_.isString(editCellDesc.formula) && (editCellDesc.attributes.cell.alignHor === 'auto')) ? 'left' : editCellDesc.orientation.cssTextAlign
            }).attr('dir', editCellDesc.orientation.textDir);

            // set cell text color and fill color to the appropriate nodes
            textAreaNode.css('color', getModifiedCssColor(editCellDesc.attributes.character.color, 'text'));
            textUnderlayNode.css('background-color', getModifiedCssColor(editCellDesc.attributes.cell.fillColor, 'fill'));
        }

        /**
         * Updates the position and size of the text area control for the cell
         * in-place edit mode, according to the current length of the text in
         * the text area control, and the scroll position of the grid pane.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the position of the text area will be updated
         *      immediately. By default, updating the text area will be done
         *      debounced.
         *  @param {Boolean} [options.size=true]
         *      If set to false, the size of the text area will not be changed,
         *      only its position. Used in situations where the content of the
         *      text area does not change (e.g. while scrolling).
         */
        var updateTextAreaPosition = (function () {

            var // whether updating the position is pending
                updatePosition = false,
                // whether updating the size is pending
                updateSize = false;

            // direct callback, invoked immediately when calling updateEditTokenArray()
            function directCallback(options) {

                // set the pending update flags for the deferred callback
                updatePosition = true;
                updateSize = updateSize || Utils.getBooleanOption(options, 'size', true);

                // invoke deferred callback immediately if specified, or if the text has caused a new line break
                if (Utils.getBooleanOption(options, 'direct', false) || (textAreaNode.outerHeight() < textAreaNode[0].scrollHeight)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // do nothing if in-place edit mode is not active
                if (!editActive || (!updatePosition && !updateSize)) { return; }

                var // the position of the edited cell
                    position = null,
                    // the visible area of this grid pane
                    visiblePosition = null,
                    // whether the text in the cell wraps automatically
                    isWrapped = CellCollection.isWrappedText(editCellDesc),
                    // the effective horizontal CSS text alignment
                    textAlign = textAreaNode.css('text-align'),
                    // round up width to multiples of the line height
                    blockWidth = Math.max(20, editCellDesc.lineHeight),
                    // the new width and height of the text area
                    targetWidth = 0, targetHeight = 0;

                // returns required width needed to show all text lines in a single line
                function getMinimumWidth() {

                    var // required width for all text lines
                        requiredWidth = _.reduce(textAreaNode.val().split(/\n/), function (memo, textLine) {
                            return Math.max(memo, FontMetrics.getTextWidth(editCellDesc.fontDesc, textLine));
                        }, 0);

                    // round up to block width, enlarge to cell width
                    return Math.max(position.width, Utils.roundUp(requiredWidth, blockWidth) + blockWidth + Utils.SCROLLBAR_WIDTH);
                }

                // returns maximum width according to horizontal alignment and width of visible area
                function getMaximumWidth() {

                    var // the distance from left visible border to right cell border (for right alignment)
                        leftWidth = Utils.minMax(position.left + position.width - visiblePosition.left, 0, visiblePosition.width),
                        // the distance from left cell border to right visible border (for left alignment)
                        rightWidth = Utils.minMax(visiblePosition.left + visiblePosition.width - position.left, 0, visiblePosition.width),
                        // effective available width
                        availWidth = 0;

                    // get available width according to text alignment
                    switch (textAlign) {
                    case 'right':
                        availWidth = leftWidth;
                        break;
                    case 'center':
                        availWidth = Math.min(leftWidth, rightWidth) * 2 - position.width;
                        break;
                    default:
                        availWidth = rightWidth;
                    }

                    // enlarge to at least half of the width of the visible area
                    return Math.max(availWidth, Math.round(visiblePosition.width / 2));
                }

                // get horizontal and vertical cell position and size
                position = self.getCellRectangle(editAddress, { expandMerged: true });
                // reduce width by one pixel to exclude the grid line
                position.width = Math.max(blockWidth, position.width - 1);
                // reduce height by one pixel to exclude the grid line
                position.height -= 1;

                // get position and size of the visible area in the grid pane
                visiblePosition = editGridPane.getVisibleRectangle();

                // on small Android devices, soft keyboard can overlay complete grid pane,
                // then the automatic height calculation has zero height,
                // after closing the soft keyboard it still keeps this height and that looks really wrong
                if (Utils.CHROME_ON_ANDROID) {
                    visiblePosition.height = Math.max(visiblePosition.height, editCellDesc.lineHeight);
                }

                // calculate and set the new size of the text area
                if (updateSize) {

                    var // find best size, stick to cell width in auto-wrapping mode
                        minWidth = isWrapped ? position.width : getMinimumWidth(),
                        maxWidth = isWrapped ? visiblePosition.width : getMaximumWidth();

                    // calculate width and height of the text area
                    targetWidth = Math.min(minWidth, maxWidth);
                    FontMetrics.getCustomFontMetrics(editCellDesc.fontDesc, function (node) {

                        var // the text whose height will be determined
                            text = textAreaNode.val();

                        // use NBSP instead of empty text; bug 35143: add NBSP to empty last line
                        if (/(^|\n)$/.test(text)) { text = text + '\xa0'; }

                        // prepare the helper node for multi-line text
                        node.style.width = targetWidth + 'px';
                        node.style.padding = '0 ' + editSheetModel.getEffectiveTextPadding() + 'px';
                        node.style.lineHeight = editCellDesc.lineHeight + 'px';
                        node.style.whiteSpace = 'pre-wrap';
                        node.style.wordWrap = 'break-word';

                        // add a text node
                        node.appendChild(document.createTextNode(text));

                        targetHeight = Math.min(Math.max(node.offsetHeight, position.height), visiblePosition.height);
                    });

                    textAreaNode.css({ width: targetWidth, height: targetHeight });

                } else {
                    targetWidth = textAreaNode.outerWidth();
                    targetHeight = textAreaNode.outerHeight();
                }

                // calculate and set the new position of the text area
                if (updatePosition) {

                    var // left and top position of the text area
                        leftOffset = position.left,
                        topOffset = position.top;

                    // calculate horizontal position according to alignment
                    switch (textAlign) {
                    case 'right':
                        leftOffset += position.width - targetWidth;
                        break;
                    case 'center':
                        leftOffset += Math.round((position.width - targetWidth) / 2);
                        break;
                    }

                    // move text area into visible area, set effective position
                    leftOffset = Utils.minMax(leftOffset, visiblePosition.left, visiblePosition.left + visiblePosition.width - targetWidth);
                    topOffset = Utils.minMax(topOffset, visiblePosition.top, visiblePosition.top + visiblePosition.height - targetHeight);
                    textAreaContainer.css({ left: leftOffset - visiblePosition.left, top: topOffset - visiblePosition.top });
                }

                updatePosition = updateSize = false;
            }

            // return the debounced method updateTextAreaPosition()
            return self.createDebouncedMethod(directCallback, deferredCallback, { delay: 250, maxDelay: 500 });
        }());

        /**
         * Returns the selected list item node in the drop-down menu element
         * for the text area.
         *
         * @returns {jQuery}
         *  The selected menu item element in the text area drop-down menu.
         */
        function getTextAreaListButtonNode() {
            return textAreaListMenu.getSelectedItemNodes().first();
        }

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionDescriptionToolTip() {

            var // the selected button element in the auto-completion menu
                buttonNode = getTextAreaListButtonNode(),
                // the name and type of the selected list item
                itemValue = Forms.getButtonValue(buttonNode),
                // the type of the list entry
                itemType = Utils.getStringOption(itemValue, 'type'),
                // the model of the selected defined name (nothing for functions)
                nameModel = (itemType === 'name') ? docModel.getNameCollection().getNameModel(itemValue.name) : null,
                // the help descriptor for the selected function (nothing for defined names)
                functionHelp = (itemType === 'func') ? app.getFunctionHelp(app.getNativeFunctionName(itemValue.name)) : null,
                // the resulting tooltip text
                toolTipLabel = '';

            // only show tooltips for valid names and supported functions
            if (nameModel) {
                toolTipLabel = nameModel.getFormula('ui', editAddress);
                if (toolTipLabel) { toolTipLabel = '=' + toolTipLabel; }
            } else if (functionHelp) {
                toolTipLabel = functionHelp.description;
            }

            // initialize the tooltip
            if (toolTipLabel) {
                textAreaListItemToolTip.setText(_.noI18n(toolTipLabel)).show();
            } else {
                textAreaListItemToolTip.hide();
            }
        }

        /**
         * Creates entries in the text area pop-up list menu.
         *
         * @param {Array<String>} entries
         *  The unfiltered entries to be inserted into the list menu. Will be
         *  used as label text of the created list entries, and will be
         *  inserted as property 'name' into the object values of the list
         *  entries.
         *
         * @param {String} filter
         *  The filter string. Only entries starting with this string (ignoring
         *  character case) will be inserted into the list menu.
         *
         * @param {String} type
         *  The type of the list entries. Will be used as name for a section in
         *  the list menu, and will be inserted as property 'type' into the
         *  object values of the list entries.
         */
        function createListMenuEntries(entries, filter, type) {

            // create a section for the list entries
            textAreaListMenu.createSectionNode(type);

            // filter for entries starting with the passed filter string
            filter = filter.toUpperCase();
            entries = entries.filter(function (entry) {
                return entry.substr(0, filter.length).toUpperCase() === filter;
            });

            // create the list entries
            entries.forEach(function (entry) {
                textAreaListMenu.createItemNode({ name: entry, type: type }, { section: type, label: _.noI18n(entry) });
            });
        }

        /**
         * Returns information about the token containing or preceding the
         * passed text cursor position in the current formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the current formula string.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preferOperand=true]
         *      Specifies how to resolve the situation that the text cursor
         *      position is located between two tokens. If set to true, and one
         *      of the tokens is an operand token (constant value tokens, array
         *      literal tokens, reference tokens, name tokens, or function name
         *      tokens), this token will be returned. Otherwise, always returns
         *      the token preceding the text position.
         *
         * @returns {TokenDescriptor|Null}
         *  A descriptor for the token whose text representation contains the
         *  specified text cursor position. If the text cursor position is
         *  located outside of the formula text, or at its beginning (without
         *  the option 'preferOperand'), this method will return the value
         *  null.
         */
        function findTokenDescriptor(pos, options) {

            // returns whether the specified token is an operand token
            function isOperand(arrayIndex) {
                var tokenDesc = editTokenDescs[arrayIndex];
                return tokenDesc && (/^(value|array|ref|name|func)$/).test(tokenDesc.token.getType());
            }

            // whether to prefer operand tokens
            var preferOperand = Utils.getBooleanOption(options, 'preferOperand', false);

            // cursor is located at the beginning of the formula, and operands are preferred:
            // return the first token (following the text cursor position)
            if (preferOperand && (pos === 0) && isOperand(0)) {
                return editTokenDescs[0];
            }

            // invalid text position passed
            if (pos <= 0) { return null; }

            // find the token containing or preceding the passed text position
            var index = Utils.findFirstIndex(editTokenDescs, function (tokenDesc) { return pos <= tokenDesc.end; }, { sorted: true });
            var currTokenDesc = editTokenDescs[index];
            if (!currTokenDesc) { return null; }

            // check if the cursor is located between two tokens, and the trailing is the preferred operand
            if (preferOperand && (pos === currTokenDesc.end) && !isOperand(index) && isOperand(index + 1)) {
                return editTokenDescs[index + 1];
            }

            // return the descriptor of the found token
            return currTokenDesc;
        }

        /**
         * Returns information about the innermost function containing the
         * passed text cursor position in the current formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the current formula string.
         *
         * @returns {Object|Null}
         *  A descriptor for the innermost function containing the specified
         *  text cursor position, in the following properties:
         *  - {String} funcName
         *      The native function name.
         *  - {Number} paramIndex
         *      The zero-based index of the function parameter covering the
         *      specified text cursor position.
         *  - {Number} paramCount
         *      The total number of parameters found in the function parameter
         *      list. If the formula is incomplete or contains errors, this
         *      number may be wrong, but it will always be greater than the
         *      value returned in the property 'paramIndex'.
         *  If the text cursor position is outside of any function, this method
         *  will return the value null.
         */
        function findFunctionDescriptor(pos) {

            var // information about the token at the passed position
                tokenDesc = findTokenDescriptor(pos, { preferOperand: true }),
                // current token index, modified in seeking functions
                index = 0,
                // the parameter index and count
                param = 0, count = 0;

            // seek to the preceding opening parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToOpeningParenthesis() {
                var seps = 0;
                while (index >= 0) {
                    switch (editTokenDescs[index].token.getType()) {
                    case 'open':
                        return seps;
                    case 'close':
                        index -= 1;
                        seekToOpeningParenthesis();
                        break;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index -= 1;
                }
                return 0;
            }

            // seek to the following closing parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToClosingParenthesis() {
                var seps = 0;
                while (index < editTokenDescs.length) {
                    switch (editTokenDescs[index].token.getType()) {
                    case 'open':
                        index += 1;
                        seekToClosingParenthesis();
                        break;
                    case 'close':
                        return seps;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index += 1;
                }
                return seps;
            }

            // invalid text cursor position passed
            if (!tokenDesc) { return null; }

            // try as long as a function has been found, or the beginning of the formula is reached
            index = tokenDesc.index;
            while (index >= 0) {

                // seek to the preceding opening parenthesis
                param = seekToOpeningParenthesis();

                // check if a function name precedes the parenthesis
                var funcTokenDesc = editTokenDescs[index - 1];
                if (funcTokenDesc && funcTokenDesc.token.isType('func')) {
                    index += 1;
                    count = seekToClosingParenthesis() + 1;
                    return { funcName: funcTokenDesc.token.getValue(), paramIndex: param, paramCount: count };
                }

                // continue seeking to preceding opening parenthesis
                index -= 1;
            }

            // cursor is not located inside a function, but it may point to a top-level
            // function name, in that case return this function with invalid parameter index
            if (tokenDesc.token.isType('func') && editTokenDescs[tokenDesc.index + 1].token.isType('open')) {
                index = tokenDesc.index + 2;
                count = seekToClosingParenthesis() + 1;
                return { funcName: tokenDesc.token.getValue(), paramIndex: -1, paramCount: count };
            }

            return null;
        }

        /**
         * Hides the function signature tooltip and the function
         * auto-completion pop-up menu.
         */
        function hideTextAreaPopups() {
            signatureToolTip.hide();
            textAreaListMenu.hide();
        }

        /**
         * Shows the function signature tooltip or the function auto-completion
         * pop-up menu if possible.
         *
         * @param {Boolean} [suppressAutoComplete=false]
         *  If set to true, the pop-up menu for auto-completion of function
         *  names will be suppressed regardless of the current formula text and
         *  cursor position.
         */
        function updateTextAreaPopups(suppressAutoComplete) {

            var // the cursor position in the text area (exclude the equality sign)
                textPos = textAreaNode[0].selectionStart - formulaOffset,
                // the text to be used for function auto-completion
                autoText = null;

            // show pop-ups only in formula mode, do not show them while dragging
            // a highlighted range, or when selecting new ranges for the formula
            if (!_.isString(formulaString) ||
                self.hasTrackedHighlightedRange() ||
                _.isObject(self.getSheetViewAttribute('activeSelection'))
            ) {
                hideTextAreaPopups();
                return;
            }

            // prefer function auto-completion over function tooltip
            funcAutoTokenDesc = null;

            // auto completion of names: cursor must be located exactly at the end of a function or name token
            var tokenDesc = (funcAutoComplete && !suppressAutoComplete) ? findTokenDescriptor(textPos) : null;
            if (tokenDesc && (tokenDesc.end === textPos)) {

                // get the text entered before the text cursor (may be a name token or a function token)
                if (tokenDesc.token.matchesType(/^(name|func)$/) && !tokenDesc.token.hasSheetRef()) {
                    // Special case: do not show a pop-up menu when a literal number precedes the name.
                    // Otherwise, e.g. entering scientific numbers will show a pop-up for functions starting with 'E'.
                    if ((tokenDesc.index === 0) || !editTokenDescs[tokenDesc.index - 1].token.isType('lit')) {
                        autoText = tokenDesc.text;
                    }
                }

                // store token info (needed to replace the entered text with the full function name)
                funcAutoTokenDesc = tokenDesc;
            }

            // fill pop-up menu for auto-completion
            if (_.isString(autoText)) {

                // insert all matching defined names, and all matching function names
                textAreaListMenu.clearContents();
                createListMenuEntries(_.invoke(docModel.getNameCollection().getNameModels({ skipHidden: true, sort: true }), 'getLabel'), autoText, 'name');
                createListMenuEntries(functionNames, autoText, 'func');

                // show the menu, if it contains at least one entry
                if (textAreaListMenu.getItemNodes().length > 0) {
                    signatureToolTip.hide();
                    textAreaListMenu.show().selectItem(0, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return;
                }
            }

            // try to show a tooltip for the current function in a formula
            hideTextAreaPopups();
            var functionInfo = findFunctionDescriptor(textPos);
            if (functionInfo) {
                signatureToolTip.updateSignature(functionInfo.funcName, functionInfo.paramIndex);
            }
        }

        /**
         * Updates the function tooltip, and other settings according to the
         * text selection in the text area control.
         */
        function updateTextAreaAfterSelection() {

            // do not open the auto-completion menu even if the new cursor position is valid
            funcAutoComplete = false;

            // do not ignore closing parentheses entered manually anymore (function auto-complete)
            funcAutoCloseData = [];

            // defer execution to let the browser process pending key events
            self.executeDelayed(function () {
                // fail-safe check that cell edit mode is still active
                if (editActive) { updateTextAreaPopups(); }
            });
        }

        /**
         * Updates the highlighting of cell references in the current formula
         * while cell in-place edit mode is active.
         */
        function updateTextAreaHighlighting() {

            var // index of the active sheet (may be different to 'editSheet')
                activeSheet = self.getActiveSheet(),
                // info objects for all ranges in the formula
                rangeInfos = editTokenArray
                    .extractRanges({
                        refAddress: editAddress,
                        targetAddress: editAddress,
                        resolveNames: true
                    })
                    .filter(function (rangeInfo) {
                        return rangeInfo.range.containsSheet(activeSheet);
                    }),
                // unified token indexes of all ranges
                rangeIndexes = _.chain(rangeInfos).pluck('index').unique().value(),
                // the mark-up of the underlay node (span elements containing token texts)
                markup = '';

            // generate the new mark-up for the underlay node
            if (rangeIndexes.length > 0) {
                markup += '<span>' + ((formulaOffset === 0) ? '' : '=') + '</span>';
                editTokenDescs.forEach(function (tokenDesc, index) {
                    var styleIndex = _.indexOf(rangeIndexes, index, true);
                    markup += '<span data-token-type="' + tokenDesc.token.getType() + '"';
                    if (styleIndex >= 0) { markup += ' data-style="' + ((styleIndex % Utils.SCHEME_COLOR_COUNT) + 1) + '"'; }
                    markup += '>' + Utils.escapeHTML(tokenDesc.text) + '</span>';
                });
            }
            textUnderlayNode[0].innerHTML = markup;
        }

        /**
         * Updates the formula token array according to the current contents of
         * the text area control used for cell in-place edit mode. The token
         * array will trigger a 'change:tokens' event that will cause to redraw
         * the highlighted ranges in all grid panes.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the token array and all other dependent
         *      settings will be updated immediately. By default, updating the
         *      token array will be debounced.
         *  @param {Boolean} [options.suppressAutoComplete=false]
         *      If set to true, the pop-up menu for auto-completion of function
         *      names will be suppressed regardless of the current formula text
         *      and cursor position.
         */
        var updateEditTokenArray = (function () {

            // whether to suppress the auto-completion pop-up menu
            var suppressAutoComplete = false;

            // direct callback, invoked immediately when calling updateEditTokenArray()
            function directCallback(options) {

                var // the current input string in the text area
                    value = textAreaNode.val();

                // update the formula expression immediately
                formulaString = /^[-+=]/.test(value) ? value.replace(/^=/, '') : null;
                formulaOffset = (value === formulaString) ? 0 : 1;

                // update the 'suppress auto-completion' flag
                suppressAutoComplete = suppressAutoComplete || Utils.getBooleanOption(options, 'suppressAutoComplete', false);

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // the cell edit mode may have been finished already
                if (!editActive) { return; }

                // parse the formula string
                if (_.isString(formulaString)) {
                    editTokenDescs = editTokenArray.parseFormula('ui', formulaString);
                } else {
                    editTokenArray.clear();
                    editTokenDescs = [];
                }

                // insert highlighting mark-up for the underlay node
                updateTextAreaHighlighting();

                // update pop-up nodes and other settings dependent on the text selection
                updateTextAreaPopups(suppressAutoComplete);
                suppressAutoComplete = false;
            }

            // return the debounced method updateEditTokenArray()
            return self.createDebouncedMethod(directCallback, deferredCallback, { delay: 100, maxDelay: 300 });
        }());

        /**
         * Changes the contents and selection of the text area, and updates its
         * position and size.
         *
         * @param {String} text
         *  The new contents of the text area.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.start]
         *      The new start position of the selection. If omitted, the text
         *      cursor will be moved to the end of the new text.
         *  @param {Number} [options.end=options.start]
         *      The new end position of the selection. If omitted, and a start
         *      position has been specified with the option 'start', a simple
         *      text cursor will be shown at the start position.
         *  @param {Boolean} [options.suppressAutoComplete=false]
         *      If set to true, the pop-up menu for auto-completion of function
         *      names will be suppressed.
         */
        function setTextAreaContents(text, options) {

            var start = Utils.getIntegerOption(options, 'start', text.length, 0, text.length);
            var end = Utils.getIntegerOption(options, 'end', start, start, text.length);

            textAreaNode.val(text);
            prevEditText = text;
            Forms.setInputSelection(textAreaNode, start, end);
            updateTextAreaPosition({ direct: true });
            updateEditTokenArray(options);
        }

        /**
         * Replaces the token before the text cursor with the selected entry
         * from the auto-complete menu.
         */
        function applyAutoCompleteText(itemValue) {

            var // the text to be inserted into the formula, and the new cursor position
                newText = null,
                newStart = 0;

            // always hide the pop-up menu
            textAreaListMenu.hide();

            // fail-safety check
            if (!funcAutoTokenDesc || !_.isObject(itemValue)) { return; }

            // get new text and new cursor position
            newText = itemValue.name;
            newStart = funcAutoTokenDesc.start + newText.length + 1;

            // insert parentheses after function name (unless the existing token is
            // a function token which means it already has an opening parenthesis)
            if (itemValue.type === 'func') {
                if (funcAutoTokenDesc.token.isType('name')) {
                    newText += '()';
                    // ignore the next closing parenthesis entered manually
                    funcAutoCloseData.push(true);
                }
                // move cursor behind the opening parenthesis
                newStart += 1;
            }

            // insert the new text into the formula string, prevent showing the
            // auo-completion pop-up list immediately again (important for defined names)
            newText = Utils.replaceSubString(textAreaNode.val(), funcAutoTokenDesc.start + 1, funcAutoTokenDesc.end + 1, newText);
            setTextAreaContents(newText, { start: newStart, suppressAutoComplete: true });
            saveUndoState({ update: true });
        }

        /**
         * Changes the quick edit mode while in-place cell edit mode is active,
         * and shows a status label at the bottom of the application pane.
         *
         * @param {Boolean} newQuickEdit
         *  The new value of the quick edit mode.
         */
        function setQuickEditMode(newQuickEditMode) {

            // change the quick edit state
            if (quickEditMode === newQuickEditMode) { return; }
            quickEditMode = newQuickEditMode;

            // update the status label
            self.setStatusLabel(quickEditMode ?
                //#. Special edit mode in spreadsheet document: quickly type text into cell, use cursor keys to move to next cell
                gt('Insert mode') :
                //#. Standard edit mode in spreadsheet document: edit existing cell text, cursor keys move text cursor inside cell
                gt('Edit mode')
            );
        }

        /**
         * Saves the current state of the cell edit mode on top of the undo
         * stack. If the current undo action is not on top of the stack, all
         * following actions will be deleted first.
         */
        function saveUndoState(options) {

            // try to update the current (top-most, not fixed) stack entry with current text
            var update = Utils.getBooleanOption(options, 'update', false);
            if (update && (undoIndex + 1 === undoStack.length) && !undoStack[undoIndex].fixed) {
                undoStack[undoIndex].text = textAreaNode.val();
                undoStack[undoIndex].sel = Forms.getInputSelection(textAreaNode);
            } else {
                undoStack.splice(undoIndex + 1);
                undoStack.push({
                    text: textAreaNode.val(),
                    sel: Forms.getInputSelection(textAreaNode),
                    attrs: _.copy(editAttributes, true),
                    fixed: !update
                });
            }

            // update the stack pointer, and notify listeners
            undoIndex = undoStack.length - 1;
            self.trigger('celledit:change');
        }

        /**
         * Restores the current state in the undo stack.
         */
        function restoreUndoState() {

            // the undo action to be restored
            var action = undoStack[undoIndex];

            editAttributes = _.copy(action.attrs, true);
            updateTextAreaStyle();
            setTextAreaContents(action.text, action.sel);
            self.trigger('celledit:change');
        }

        /**
         * Removes the activated ranges from the view that have been inserted
         * into the formula while cell edit mode is active.
         */
        function resetActiveCellSelection() {
            self.setSheetViewAttribute('activeSelection', null);
        }

        /**
         * Updates the position of the text area control after the grid pane
         * has been scrolled (or modified in any other way, e.g. zoom).
         */
        function gridPaneScrollHandler() {
            updateTextAreaPosition({ direct: true, size: false });
        }

        /**
         * Handles 'keydown' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyDownHandler(event) {

            var // the list item currently selected in the auto-completion pop-up menu
                itemNode = textAreaListMenu.getSelectedItemNodes().first(),
                // the value of the item node
                itemValue = Forms.getButtonValue(itemNode);

            // first, try to cancel active tracking mode on ESCAPE key
            if ((event.keyCode === KeyCodes.ESCAPE) && Tracking.cancelTracking()) {
                return false;
            }

            // short-cuts in visible pop-up menu for auto-completion
            if (textAreaListMenu.isVisible()) {

                // close pop-up menu on ESCAPE key
                if (event.keyCode === KeyCodes.ESCAPE) {
                    textAreaListMenu.hide();
                    return false;
                }

                // insert selected list item into the formula
                if (KeyCodes.matchKeyCode(event, 'TAB') || KeyCodes.matchKeyCode(event, 'ENTER')) {

                    // Bug 31999: For performance reasons, the token array and the auto-completion pop-up
                    // menu will be updated debounced. To prevent that ENTER/TAB keys insert a function
                    // name unintentionally, the pop-up menu must be updated before applying a list item.
                    updateEditTokenArray({ direct: true });

                    // list menu may have been hidden by the updateTextAreaPopups() call,
                    // do not apply any list item in this case, but leave cell edit mode
                    if (textAreaListMenu.isVisible()) {
                        applyAutoCompleteText(itemValue);
                        return false;
                    }
                }

                // otherwise, try to move the selected list item
                itemNode = textAreaListMenu.getItemNodeForKeyEvent(event, itemNode);
                if (itemNode.length > 0) {
                    textAreaListMenu.selectItemNode(itemNode, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return false;
                }
            }

            // quit in-place edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                self.cancelCellEditMode();
                return false;
            }

            // toggle quick edit mode
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                setQuickEditMode(!quickEditMode);
                // forget last cell range selected in the sheet
                resetActiveCellSelection();
                return false;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                if (self.leaveCellEditMode('cell', { validate: true })) {
                    return; // let key event bubble up to move the cell cursor
                }
                return false;
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert matrix formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                self.leaveCellEditMode(event.shiftKey ? 'matrix' : 'fill', { validate: true });
                // do not let key event bubble up (no change of selection)
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Forms.replaceTextInInputSelection(textAreaNode, '\n');
                textAreaInputHandler();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to next/previous cell
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                if (self.leaveCellEditMode('cell', { validate: true })) {
                    return; // let key event bubble up to move the cell cursor
                }
                return false;
            }

            // any cursor navigation key
            if (event.keyCode in NAVIGATION_KEYCODES) {
                if (quickEditMode) {
                    // quick edit mode: do not let the text area process the key events
                    event.preventDefault();
                } else {
                    // standard edit mode: do not process the key events (no change of cell selection)
                    event.stopPropagation();
                    // update pop-up nodes and other settings dependent on the new text selection
                    updateTextAreaAfterSelection();
                    // forget last cell range selected in the sheet
                    resetActiveCellSelection();
                }
                // let key event bubble up to move the cell cursor, or text cursor in the text area
                return;
            }

            // all other events will be processed by the text field
        }

        /**
         * Handles 'keypress' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyPressHandler(event) {

            // the entered Unicode character
            switch (String.fromCharCode(event.charCode)) {

            // Register an opening parenthesis entered manually. The related closing parenthesis
            // will be inserted even if it occurs among other closing parentheses inserted via
            // function auto-completion (see below).
            case '(':
                funcAutoCloseData.push(false);
                break;

            // When a function name has been inserted from the auto-completion pop-up menu, skip
            // a closing parenthesis because it has been inserted already by the auto-completion).
            // But the text cursor needs to be moved one character forward.
            case ')':
                if (funcAutoCloseData.pop() === true) {
                    Forms.setInputSelection(textAreaNode, Forms.getInputSelection(textAreaNode).start + 1);
                    updateTextAreaPopups();
                    return false;
                }
                break;
            }
        }

        /**
         * Handles any changes in the text area.
         */
        function textAreaInputHandler() {

            var // the current value of the text area
                currEditText = textAreaNode.val(),
                // the text selection in the text area control
                textSelection = Forms.getInputSelection(textAreaNode),
                // whether the selection in the text area is a cursor only
                isCursor = textSelection.start === textSelection.end;

            // do nothing if the edit text did not change at all
            if (prevEditText === currEditText) { return; }

            // try to show drop-down list with function names and defined names
            funcAutoComplete = isCursor;

            // update text area settings (position, formula range highlighting)
            updateTextAreaPosition();
            updateEditTokenArray();
            saveUndoState({ update: true });

            // Mobile devices may add auto-correction text which modifies the selection after a text change event has
            // been processed. Recalculate the funcAutoComplete flag after the browser has set the final selection.
            self.executeDelayed(function () {
                textSelection = Forms.getInputSelection(textAreaNode);
                funcAutoComplete = textSelection.start === textSelection.end;
            });

            // forget last active cell range selected in the sheet (formulas with cell references)
            resetActiveCellSelection();

            // show type-ahead suggestion in the text area (not in formula edit mode)
            if (_.isString(prevEditText) && (prevEditText.length < currEditText.length) && isCursor && (textSelection.end === currEditText.length) && !_.isString(formulaString)) {

                // first cycle: collect the contents of all text cells and formula cells resulting in text above and below the edited cell
                if (!typeAheadEntries) {
                    typeAheadEntries = [];

                    // collect the values of all text cells, and the string results of formula cells
                    var boundRange = Range.create(editAddress[0], Math.max(0, editAddress[1] - 256), editAddress[0], Math.min(docModel.getMaxRow(), editAddress[1] + 256));
                    editSheetModel.getCellCollection().iterateCellsInLine(editAddress, 'up down', function (cellDesc) {
                        if (CellCollection.isText(cellDesc) && (cellDesc.result.length > 0)) {
                            typeAheadEntries.push({
                                text: cellDesc.result,
                                dist: Math.abs(editAddress[1] - cellDesc.address[1])
                            });
                        }
                    }, { type: 'content', hidden: 'all', boundRange: boundRange, skipStartCell: true });

                    // sort the entries in the array by distance to the edited cell
                    typeAheadEntries.sort(function (entry1, entry2) { return entry1.dist - entry2.dist; });

                    // reduce the entries to a simple string array, that is still sorted by distance
                    typeAheadEntries = _.pluck(typeAheadEntries, 'text');
                }

                // find the first matching type-ahead entry (the array is sorted by distance to edited cell, thus it will find the nearest matching entry)
                var suggestion = _.find(typeAheadEntries, function (entry) {
                    return entry.substr(0, currEditText.length).toUpperCase() === currEditText.toUpperCase();
                });

                // insert the remaining text of a matching entry, keep character case of the typed text
                if (suggestion) {
                    suggestion = currEditText + suggestion.substr(currEditText.length);
                    setTextAreaContents(suggestion, { start: currEditText.length, end: suggestion.length });
                    saveUndoState({ update: true });
                }
            }

            // store current text for next iteration
            prevEditText = currEditText;
        }

        /**
         * Handles mousedown or touchstart events in the text area.
         */
        function textAreaMouseTouchHandler() {
            // leave quick-edit mode (cursor keys will move the text cursor)
            setQuickEditMode(false);
            // update pop-up nodes and other settings dependent on the new text selection
            updateTextAreaAfterSelection();
            // forget last cell range selected in the sheet
            resetActiveCellSelection();
        }

        /**
         * Updates the scroll position of the underlay node, after the scroll
         * position of the text area node has been changed.
         */
        function textAreaScrollHandler() {
            textUnderlayNode.scrollTop(textAreaNode.scrollTop());
        }

        /**
         * Hides the pop-up nodes if the focus leaves the text area control,
         * unless the auto-completion menu is being clicked.
         */
        var textAreaFocusHandler = (function () {

            var // whether to ignore focus event in the text area (no update of pop-up nodes)
                ignoreFocusEvent = false;

            return function (event) {
                switch (event.type) {

                // text area focused: rebuild all active pop-up nodes
                case 'focus':
                    textAreaContainer.addClass(Forms.FOCUSED_CLASS);
                    if (!ignoreFocusEvent) {
                        updateTextAreaPopups();
                    }
                    break;

                // text area lost focus: hide pop-up nodes (unless the auto-complete menu
                // has been clicked, in this case, move focus back to text area immediately)
                case 'blur':
                    self.executeDelayed(function () {
                        if (editActive && textAreaListMenu.hasFocus()) {
                            ignoreFocusEvent = true;
                            textAreaNode.focus();
                            // IE sends focus event deferred
                            self.executeDelayed(function () { ignoreFocusEvent = false; });
                        } else {
                            textAreaContainer.removeClass(Forms.FOCUSED_CLASS);
                            if (!stickyAutoComplete) { hideTextAreaPopups(); }
                        }
                    });
                }
            };
        }());

        /**
         * Inserts the name clicked in the auto-completion pop-up menu into the
         * current formula text.
         */
        function textAreaListClickHandler(event) {
            var itemValue = Forms.getButtonValue(event.currentTarget);
            applyAutoCompleteText(itemValue);
        }

        /**
         * Returns whether the current text selection in the text area control
         * is valid to start selecting new ranges for the current formula. The
         * text area must contain a formula string (starting with an equality
         * sign), and the start position of the text selection must be located
         * behind the leading equality sign, an operator, a list separator, or
         * an opening parenthesis).
         */
        function isValidRangeSelectionPosition() {

            var // the cursor position in the token array (exclude leading equality sign)
                textPos = textAreaNode[0].selectionStart - formulaOffset;

            // invalid if not in cell in-place edit mode with a formula
            if (!editActive || !_.isString(formulaString)) { return false; }

            // existing view attribute, or cursor at the beginning of the formula is always valid
            if ((textPos === 0) || _.isObject(self.getSheetViewAttribute('activeSelection'))) { return true; }

            // find the formula token located before the text cursor
            var tokenDesc = findTokenDescriptor(textPos);

            // cursor must be located between tokens; leading token must be of correct
            // type (or cursor must be located at the beginning of the formula)
            return tokenDesc && (textPos === tokenDesc.end) && tokenDesc.token.matchesType(/^(op|sep|open)$/);
        }

        /**
         * When in cell edit mode with a formula, returns the active selection
         * which will be used for keyboard navigation and mouse/touch tracking.
         */
        function getCellSelectionHandler() {

            var // the current value of the 'activeSelection' view attribute
                activeSelection = self.getSheetViewAttribute('activeSelection');

            // always return existing view attribute
            if (_.isObject(activeSelection)) { return activeSelection; }

            // check the text selection in the text area control
            if (!isValidRangeSelectionPosition()) { return null; }

            // start with an empty active selection
            return new SheetSelection();
        }

        /**
         * When in cell edit mode with a formula, selects a single range and
         * inserts the range address into the formula string.
         */
        function setCellSelectionHandler(selection) {

            var // a helper token array to generate reference addresses for the active ranges
                tokenArray = null,
                // the string representation of the active ranges
                references = null,
                // the new formula text
                formulaText = null;

            // do nothing if ranges cannot be selected with the current text selection
            if (!isValidRangeSelectionPosition()) { return false; }

            // store current text selection, if no range are activated yet
            // (selected text will be replaced with initial range addresses)
            if (!_.isObject(self.getSheetViewAttribute('activeSelection'))) {
                rangeSelection = Forms.getInputSelection(textAreaNode);
            }

            // build the new formula term to be inserted into the text
            tokenArray = new TokenArray(editSheetModel, { temp: true });
            tokenArray.appendRangeList(selection.ranges);
            references = tokenArray.getFormula('ui');
            tokenArray.destroy();
            formulaText = Utils.replaceSubString(textAreaNode.val(), rangeSelection.start, rangeSelection.end, references);

            // insert the new formula text, set cursor behind the range (no real selection)
            rangeSelection.end = rangeSelection.start + references.length;
            setTextAreaContents(formulaText, { start: rangeSelection.end });
            saveUndoState({ update: true });

            // update the view attribute
            self.setSheetViewAttribute('activeSelection', selection);

            // do not perform standard sheet selection
            return true;
        }

        /**
         * Cancels cell edit mode after the application loses edit rights.
         */
        function editModeHandler(event, editMode) {
            // leave cell in-place edit mode, if document changes to read-only mode
            if (!editMode && app.isImportFinished()) {
                self.cancelCellEditMode();
            }
        }

        /**
         * Updates the text area for a changed zoom factor.
         */
        function changeSheetViewAttributesHandler(event, attributes) {
            if (editActive && ('zoom' in attributes)) {
                updateTextAreaStyle();
                updateTextAreaPosition({ direct: true });
            }
        }

        /**
         * Shows an alert box for circular references in new formulas.
         */
        function showCircularWarning() {
            self.yell({
                type: 'info',
                headline: /*#. title for a warning message: a formula addresses its own cell, e.g. =A1 in cell A1 */ gt('Circular Reference'),
                message: gt('A reference in the formula is dependent on the result of the formula cell.')
            });
        }

        /**
         * Processes 'docs:update:cells' events of the application, triggered
         * from a server notification. If the event contains a cell-related
         * error code, such as a circular reference error, and the error was
         * caused by a local change after in-place cell edit mode, an alert box
         * will be shown to the user.
         */
        function updateCellsHandler(changedData, allChangedRanges, errorCode) {

            // check whether all required data objects exist
            if (_.isString(errorCode) && _.isObject(lastCommitData) && (lastCommitData.sheet in changedData) && lastCommitData.ranges) {

                var // the notified changed ranges in the last edited sheet
                    changedRanges = changedData[lastCommitData.sheet].changedRanges;

                // the last committed cells must opverlap with the notified changed ranges
                if (changedRanges && changedRanges.overlaps(lastCommitData.ranges)) {

                    // show an appropriate alert for the error code
                    if (errorCode === 'circular') {
                        // do not show another circular reference warning, if it has been detected locally
                        if (!lastCommitData.circular) { showCircularWarning(); }
                    } else {
                        Utils.warn('CellEditMixin.updateCellsHandler(): unknown error code "' + errorCode + '"');
                    }
                }
            }

            lastCommitData = null;
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the in-place cell edit mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the in-place cell edit mode is currently active.
         */
        this.isCellEditMode = function () {
            return editActive;
        };

        /**
         * Starts in-place cell edit mode in the active grid pane.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {String} [options.text]
         *      The initial text to insert into the edit area. If omitted, the
         *      current value of the active cell will be inserted.
         *  @param {Number} [options.pos]
         *      The initial position of the text cursor. If omitted, the text
         *      cursor will be placed after the entire text. If negative, the
         *      text cursor will be placed from the end of the text.
         *  @param {Boolean} [options.quick=false]
         *      If set to true, the quick edit mode will be initiated. All
         *      cursor navigation keys will not move the text cursor in the
         *      text area control but will behave as in the normal sheet
         *      selection mode.
         *  @param {Boolean} [options.stickyList=false]
         *      If set to true, the auto-completion drop-down list will be kept
         *      open when the text edit control is not focused.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.enterCellEditMode = function (options) {

            var // the initial text for the text area
                initialText = Utils.getStringOption(options, 'text', null),
                // the initial cursor position for the text area
                initialPos = Utils.getIntegerOption(options, 'pos');

            // first, check document edit mode
            if (!this.requireEditMode()) { return this; }

            // check whether in-place edit mode is already active
            if (editActive) { return this; }

            // cancel current tracking
            Tracking.cancelTracking();

            // store location of edited cell (sheet may change during range selection mode for formulas)
            editAddress = this.getActiveCell();
            editSheet = this.getActiveSheet();
            editSheetModel = this.getSheetModel();
            editGridPane = this.getActiveGridPane();
            editAttributes = {};

            // initialize CSS formatting of the text area (this initializes the variable editCellDesc)
            updateTextAreaStyle();

            // scroll to the active cell (before checking edit mode); on touch devices,
            // scroll the cell to the top border of the grid pane (due to virtual keyboard)
            editGridPane.scrollToCell(editAddress, { forceTop: Utils.TOUCHDEVICE });

            // Bug 34366: Touch devices scroll the window out of visible area, so that neither the bars
            // (tool and tab) are visible, nor the editable cell. Root cause is the soft-keyboard.
            // However, scroll back to 0,0 and everything is fine. (Delay of 1000 necessary for iOS)
            if (Utils.TOUCHDEVICE) {
                this.executeDelayed(function () { window.scrollTo(0, 0); }, 1000);
            }

            // give user some info and quit if cell is locked
            if (!this.requireEditableActiveCell({ lockTables: 'header' })) {
                return this;
            }

            // initial state of quick edit mode and other settings
            editActive = true;
            setQuickEditMode(Utils.getBooleanOption(options, 'quick', false));

            // TODO: matrix formula support
            matrixFormula = false;

            // type-ahead suggestions will be collected on first access
            typeAheadEntries = null;

            // the token array stores the parsed formula for range highlighting
            editTokenDescs = [];
            editTokenArray = new TokenArray(editSheetModel);
            highlightUid = this.startRangeHighlighting(editTokenArray, {
                priority: true,
                draggable: true,
                refAddress: editAddress,
                targetAddress: editAddress,
                resolveNames: true
            });

            // update formula string in text area while dragging the highlighted ranges
            editTokenArray.on('change:token', function (event, token, index) {

                // override the text representation of the changed token
                var config = docModel.getFormulaTokenizer('ui').getConfig();
                editTokenDescs[index].text = token.getText(config);

                // update the text positions of the following tokens
                Utils.iterateArray(editTokenDescs, function (tokenDesc, i) {
                    tokenDesc.start = (i === 0) ? 0 : editTokenDescs[i - 1].end;
                    tokenDesc.end = tokenDesc.start + tokenDesc.text.length;
                }, { begin: index });

                // calculate the new formula string
                var formulaString = editTokenDescs.reduce(function (formula, tokenDesc) {
                    return formula + tokenDesc.text;
                }, '');

                // set the new formula, add an undo action
                setTextAreaContents(textAreaNode.val()[0] + formulaString, { start: editTokenDescs[index].end + 1 });
                saveUndoState({ update: true });

                // immediately update the highlighted cell references
                updateTextAreaHighlighting();
                resetActiveCellSelection();
            });

            // insert the text area control into the DOM of the active grid pane, and activate it
            editGridPane.getNode().append(textAreaContainer.show());
            editGridPane.registerFocusTargetNode(textAreaNode);
            editGridPane.on('change:scrollpos', gridPaneScrollHandler);

            // get the formatted edit string
            originalText = getTextForCellEditMode();
            textUnderlayNode.empty();

            // set initial contents, position, size, and selection of the text area
            var editText = _.isString(initialText) ? initialText : originalText;
            if (_.isNumber(initialPos) && (initialPos < 0)) { initialPos = Math.max(0, editText.length + initialPos); }
            setTextAreaContents(editText, { start: initialPos });

            // initialize local undo/redo stack
            undoStack = [{
                text: originalText,
                sel: _.isString(initialText) ? { start: 0, end: originalText.length } : Forms.getInputSelection(textAreaNode),
                attrs: {},
                fixed: true
            }];
            undoIndex = 0;

            // selection handler that changes cell ranges in formaulas in cell edit mode
            this.registerCellSelectionHandlers(getCellSelectionHandler, setCellSelectionHandler);

            // set last value and invoke input handler to handle custom initial text
            prevEditText = _.isString(initialText) ? '' : null;
            textAreaInputHandler();

            // special mode to keep auto-completion list menu open without focus in text area
            stickyAutoComplete = Utils.getBooleanOption(options, 'stickyList', false);
            textAreaListMenu.setAutoCloseMode(!stickyAutoComplete);

            // notify listeners
            this.trigger('celledit:enter');
            return this;
        };

        /**
         * Cancels in-place cell edit mode, without committing the current edit
         * text.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.cancelCellEditMode = function () {

            // nothing to do, if in-place edit mode is not active
            if (!editActive) { return this; }

            // deinitialize range highlighting
            editActive = false;
            this.endRangeHighlighting(highlightUid);
            this.unregisterCellSelectionHandlers();
            resetActiveCellSelection();
            editTokenArray.destroy();
            editSheetModel = editTokenArray = editTokenDescs = highlightUid = null;

            // back to cell selection mode
            quickEditMode = formulaString = typeAheadEntries = null;
            hideTextAreaPopups();

            // clean up grid pane
            editGridPane.off('change:scrollpos', gridPaneScrollHandler);
            editGridPane.unregisterFocusTargetNode();
            editGridPane = null;

            // hide the text area control (bug 40321: DO NOT detach from DOM, Egde may freeze)
            textAreaContainer.hide();

            // notify listeners
            this.setStatusLabel(null);
            this.trigger('celledit:leave');
            return this;
        };

        /**
         * Leaves in-place cell edit mode, and optionally commits the current
         * text. If the current text is a formula expression, it will be
         * validated before committing.
         *
         * @param {String} commitMode
         *  Specifies how to leave the cell edit mode. The following values are
         *  supported:
         *  - 'cell': The current value will be committed to the active cell of
         *      the selection.
         *  - 'fill': The current value will be filled to all cells of the
         *      selection.
         *  - 'matrix': The current value will be inserted as matrix formula
         *      into the selected cell range.
         *  - 'auto': Preserves the current formula type of the cell: either
         *      'cell' or 'matrix'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.validate=false]
         *      If set to true, a formula expression will be velidated. If it
         *      is invalid, a notification alert will be shown, and the cell
         *      edit mode will not be left.
         *
         * @returns {Boolean}
         *  Whether the cell edit mode has actually been left. Returns false,
         *  if the option 'validate' have been set to true, and the current
         *  formula expression is invalid.
         */
        this.leaveCellEditMode = function (commitMode, options) {

            var // the new cell text
                editText = textAreaNode.val(),
                // whether to validate the formula
                validate = Utils.getBooleanOption(options, 'validate', false),
                // the trimmed URL extracted from the value
                url = HyperlinkUtils.checkForHyperlink(editText),
                // additional options to be passed to the generator methods
                operationOptions = { parse: true };

            // adds a formatting attribute to the edit attributes
            function addAttribute(family, name, value) {
                editAttributes[family] = editAttributes[family] || {};
                editAttributes[family][name] = value;
            }

            // nothing to do, if in-place edit mode is not active
            if (!editActive) { return true; }

            // check validity of commit mode (e.g. disallow 'cell' in matrix formulas)
            switch (commitMode) {
            case 'cell':
            case 'fill':
                if (matrixFormula) {
                    // TODO: define behavior (update entire matrix automatically?)
                    self.yellOnResult('formula:matrix');
                    setQuickEditMode(false);
                    return false;
                }
                break;
            case 'matrix':
                // TODO: check that selection matches range of matrix formula
                break;
            case 'auto':
                commitMode = matrixFormula ? 'matrix' : 'cell';
                break;
            default:
                Utils.error('CellEditMixin.leaveCellEditMode(): unknown commit mode: ' + commitMode);
                commitMode = null;
            }

            // do nothing else, if the document does not have edit rights anymore
            if (!docModel.getEditMode() || !commitMode) {
                this.cancelCellEditMode();
                return true;
            }

            // prepare cached commit data (needed to evaluate server notifications with error codes)
            lastCommitData = { sheet: editSheet };

            // handle specific strings not to be recognized as formulas
            if (/^[-+=]+$/.test(editText)) {
                editText = '\'' + editText;
            } else {
                // validate formula
                updateEditTokenArray({ direct: true });
                if (_.isString(formulaString)) {

                    // TODO: get value/matrix result depending on commit mode
                    var result = editTokenArray.interpretFormula('val', { refAddress: editAddress, targetAddress: editAddress });

                    if (validate) {
                        switch (result.type) {
                        case 'error':
                            self.yellOnResult('formula:invalid');
                            setQuickEditMode(false);
                            return false;
                        case 'warn':
                            if (result.value === 'circular') {
                                showCircularWarning(); // TODO: yes/no dialog?
                                lastCommitData.circular = true;
                            }
                        }
                    }

                    // insert the formula result value into the operation
                    operationOptions.result = null;
                    if (result.type === 'result') {
                        // convert error code objects to string representation, add leading apostrophe to strings if necessary
                        if (SheetUtils.isErrorCode(result.value)) {
                            operationOptions.result = result.value.code;
                        } else if (_.isString(result.value)) {
                            operationOptions.result = result.value.replace(/^['#]/, '\'$&');
                        } else {
                            operationOptions.result = result.value;
                        }
                    }
                }
            }

            // deinitialize cell edit mode
            this.cancelCellEditMode();

            // add auto-wrapping attribute, if the text contains an explicit line break
            if (editText.indexOf('\n') >= 0) {
                addAttribute('cell', 'wrapText', true);
            }

            // add hyperlink specific attributes
            if (_.isString(url) && (url.length > 0)) {
                operationOptions.url = url;
                addAttribute('character', 'underline', true);
                addAttribute('character', 'color', Color.HYPERLINK);
            } else if (!editText && this.getCellCollection().getCellURL(editAddress)) {
                // delete the URL explicitly, if text is cleared
                operationOptions.url = '';
            }

            // do not pass empty attributes object
            if (_.isEmpty(editAttributes)) { editAttributes = null; }

            // send current value to document model after returning to selection mode
            switch (docModel.getEditMode() ? commitMode : 'discard') {
            case 'cell':
                // single cell mode: do not send operations if text and formatting did not change
                if ((originalText !== editText) || _.isObject(editAttributes)) {
                    this.setCellContents(editText, editAttributes, operationOptions);
                    lastCommitData.ranges = new RangeArray(new Range(editAddress));
                }
                break;
            case 'fill':
                // fill mode: always fill all cells in the selection
                this.fillCellRanges(editText, editAttributes, operationOptions);
                lastCommitData.ranges = this.getSelectedRanges();
                break;
            case 'matrix':
                // TODO: matrix formulas
                Utils.warn('CellEditMixin.leaveCellEditMode(): matrix formulas not implemented');
                lastCommitData.ranges = this.getSelectedRanges();
                break;
            }

            return true;
        };

        /**
         * Returns the additional formatting attributes used in in-place cell
         * edit mode.
         *
         * @return {Object|Null}
         *  The new attributes applied while the in-place cell edit mode is
         *  active.
         */
        this.getCellEditAttributes = function () {
            return editActive ? _.copy(editAttributes, true) : null;
        };

        /**
         * Modifies formatting attributes while the in-place cell edit mode is
         * active.
         *
         * @param {Object} [attributes]
         *  The new attributes to be applied to the text area. If omitted, the
         *  call of this method will be ignored.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.setCellEditAttributes = function (attributes) {

            function extendAttributeFamily(family) {
                if (_.isObject(attributes[family])) {
                    _.extend(editAttributes[family] || (editAttributes[family] = {}), attributes[family]);
                }
            }

            if (editActive && _.isObject(attributes)) {

                // extend the edit attribute set
                extendAttributeFamily('cell');
                extendAttributeFamily('character');

                // update the formatting of the text area, and the undo/redo stack
                updateTextAreaStyle();
                updateTextAreaPosition({ direct: true });
                saveUndoState();
            }

            return this;
        };

        // extended undo/redo -------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * are changed currently.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return editActive ? (undoIndex > 0) : (undoManager.getUndoCount() > 0);
        };

        /**
         * Applies the topmost undo action on the undo stack. In cell in-place
         * edit mode, restores the original contents of the cell.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            if (editActive) {
                if (undoIndex > 0) {
                    undoIndex -= 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.undo(1);
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * have been changed, but are not changed anymore (after undo).
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return editActive ? (undoIndex + 1 < undoStack.length) : (undoManager.getRedoCount() > 0);
        };

        /**
         * Applies the topmost redo action on the redo stack. In cell in-place
         * edit mode, restores the last changed contents of the cell, after
         * these changes have been undone.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            if (editActive) {
                if (undoIndex + 1 < undoStack.length) {
                    undoIndex += 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.redo(1);
        };

        // initialization -----------------------------------------------------

        // get localized names of all visible (non-hidden, non-obsolete) functions
        app.onInitLocale(function () {

            // descriptors of all supported fucntions according to current file format
            var descriptors = docModel.getFormulaInterpreter().getFunctionDescriptors();

            // native names of all visible functions
            functionNames = app.getNativeFunctionNames().filter(function (funcName) {
                return (funcName in descriptors) && !descriptors[funcName].hidden;
            });

            // convert to localized function names
            functionNames = functionNames.map(function (funcName) {
                return app.getLocalizedFunctionName(funcName);
            });
        });

        // cancel cell edit mode after the application loses edit rights
        this.listenTo(docModel, 'change:editmode', editModeHandler);

        // update zoom settings during edit mode
        this.on('change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // events for in-place edit mode
        textAreaNode.on({
            keydown: textAreaKeyDownHandler,
            keypress: textAreaKeyPressHandler,
            input: textAreaInputHandler,
            'mousedown mouseup touchstart touchend touchcancel': textAreaMouseTouchHandler,
            scroll: textAreaScrollHandler,
            'focus blur': textAreaFocusHandler
        });

        // handle click events in the auto-completion pop-up menu
        textAreaListMenu.getNode().on('click', Forms.BUTTON_SELECTOR, textAreaListClickHandler);

        // hide function description tooltip when closing the auto-completion menu
        textAreaListMenu.on('popup:hide', function () { textAreaListItemToolTip.hide(); });

        // post-process changed cells (e.g. for circular references alert box)
        this.listenTo(app, 'docs:update:cells', updateCellsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {

            textAreaContainer.remove();
            signatureToolTip.destroy();
            textAreaListMenu.destroy();
            textAreaListItemToolTip.destroy();

            self = app = docModel = undoManager = fontCollection = numberFormatter = null;
            textAreaContainer = textAreaNode = textUnderlayNode = null;
            signatureToolTip = textAreaListMenu = textAreaListItemToolTip = null;
            undoStack = null;
        });

    } // class CellEditMixin

    // exports ================================================================

    return CellEditMixin;

});
