/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/sheetorderdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialogs',
    'gettext!io.ox/office/spreadsheet/main',
    'less!io.ox/office/spreadsheet/view/dialog/sheetorderdialog'
], function (Utils, Forms, KeyCodes, Dialogs, gt) {

    'use strict';

    // class SheetOrderDialog =================================================

    /**
     * @constructor
     *
     * @extends ModalDialog
     */
    var SheetOrderDialog = Dialogs.ModalDialog.extend({ constructor: function (docView) {

        var docModel = docView.getDocModel(),

            currentSheet = docView.getActiveSheet(),

            currentChosen = null,

            collectOps = [],

            list = $('<div class="dialog-ordersheets-list" role=tablist">'),

            dHolder = $('<div class="dialog-ordersheets-holder">').append(list),

            upButton = $.button({ click: upButtonClickHandler })
                    .addClass('btn')
                    .attr('tabindex', 1)
                    .css({ marginTop: '-3px', marginBottom: '5px', display: 'block' })
                    .append(Forms.createIconMarkup('fa-arrow-up')),

            downButton = $.button({ data: { dir: 1 }, click: downButtonClickHandler })
                    .addClass('btn')
                    .attr('tabindex', 1)
                    .css('display', 'block')
                    .append(Forms.createIconMarkup('fa-arrow-down')),

            control = $('<div class="dialog-ordersheets-control">').append(upButton, downButton);

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, { title: gt('Reorder sheets') });

        // private functions --------------------------------------------------

        function upButtonClickHandler() {
            var index = currentChosen.attr('data-index') | 0;
            var other = currentChosen.prev('*');
            if (other.length > 0) {
                var otherIndex = other.attr('data-index') | 0;
                currentChosen.insertBefore(other);
                collectOps.push({
                    from: index,
                    to: otherIndex
                });
                currentChosen.attr('data-index', otherIndex);
                other.attr('data-index', index);
                checkButtonsActive();
            }
            checkScrollOutside();
        }

        function downButtonClickHandler() {
            var index = currentChosen.attr('data-index') | 0;
            var other = currentChosen.next('*');
            if (other.length > 0) {
                var otherIndex = other.attr('data-index') | 0;
                currentChosen.insertAfter(other);
                collectOps.push({
                    from: index,
                    to: otherIndex
                });
                currentChosen.attr('data-index', otherIndex);
                other.attr('data-index', index);
                checkButtonsActive();
            }
            checkScrollOutside();
        }

        function checkScrollOutside() {
            var pHeight = dHolder.height();
            var cHeight = currentChosen.height();
            var pos = currentChosen.position();
            if ((pos.top < 0) || (pos.top + cHeight > pHeight)) {
                Utils.scrollToChildNode(dHolder, currentChosen);
            }
        }

        function checkButtonsActive() {
            if (currentChosen.prev('*').length > 0) {
                upButton.removeClass('disabled');
                upButton.css({ pointerEvents: 'auto' });
                upButton.attr('tabindex', 1);
            } else {
                upButton.addClass('disabled');
                upButton.css({ pointerEvents: 'none' });
                upButton.removeAttr('tabindex');
            }

            if (currentChosen.next('*').length > 0) {
                downButton.removeClass('disabled');
                downButton.css({ pointerEvents: 'auto' });
                downButton.attr('tabindex', 1);
            } else {
                downButton.addClass('disabled');
                downButton.css({ pointerEvents: 'none' });
                downButton.removeAttr('tabindex');
            }
        }

        function selectButton(evt) {
            currentChosen.removeClass('btn-primary');
            currentChosen = $(evt.currentTarget);
            currentChosen.addClass('btn-primary');
            checkButtonsActive();
        }

        function handleUpDown(upDown, ctrlKey) {
            var button;
            if (ctrlKey) {
                button = upDown ? upButton : downButton;
            } else {
                button = upDown ? currentChosen.prev('*') : currentChosen.next('*');
            }
            button.click();
            currentChosen.focus();
        }

        /**
         * Handles 'keydown' events on the dialog node
         * up/down changes the selected sheet in the dialog
         * ctrl + up/down 'moves' the selected sheet inside the dialog
         * enter click on the ok button
         */
        function keyDownHandler(evt) {
            switch (evt.keyCode) {
            case KeyCodes.UP_ARROW:
                handleUpDown(true, evt.ctrlKey);
                return true;
            case KeyCodes.DOWN_ARROW:
                handleUpDown(false, evt.ctrlKey);
                return true;
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Shows this dialog.
         *
         * @attention
         *  The dialog can only be shown once. It will destroy itself after it
         *  has been closed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the new order of the sheets
         *  when the OK button has been pressed.
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).then(function () { return collectOps; });
        });

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        docModel.iterateSheetModels(function (sheetModel, sheet, sheetName) {

            // ignore hidden sheets, and sheet types that are not supported
            if (!sheetModel.isVisible() || !docView.isSheetTypeSupported(sheetModel.getType())) { return; }

            sheetName = _.noI18n(Utils.escapeHTML(sheetName));
            var button = $('<button class="btn" data-index="' + sheet + '" title="' + sheetName + '" tabindex="1" role="button">' + sheetName + '</button>');

            button.on('click', selectButton);
            if (sheet === currentSheet) {
                button.addClass('btn-primary');
                currentChosen = button;
            }
            list.append(button);
        });

        this.append(dHolder, control);
        this.getBody()
            .addClass('io-ox-office-spreadsheet-reorder-dialog')
            .css({ paddingRight: '2px', paddingTop: 0, paddingBottom: 0 });
        this.getBody().parent().on('keydown', keyDownHandler);

        this.on('show', function () {
            checkScrollOutside();
            checkButtonsActive();
        });

    }}); // class SheetOrderDialog

    // exports ================================================================

    return SheetOrderDialog;

});
