/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/range3d',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/range3darray',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Logger, Address, Interval, Range, Range3D, AddressArray, IntervalArray, RangeArray, Range3DArray, gt) {

    'use strict';

    var // maps native error codes to integers (used e.g. for comparison in filters)
        ERROR_CODE_NUMBERS = {
            '#NULL!': 1,
            '#DIV/0!': 2,
            '#VALUE!': 3,
            '#REF!': 4,
            '#NAME?': 5,
            '#NUM!': 6,
            '#N/A': 7
        };

    // class ErrorCode ========================================================

    /**
     * Instances of this class represent literal error codes, as used in cells,
     * in formula expressions, and as the result of formulas. Implemented as
     * named class to be able to use the 'instanceof' operator to distinguish
     * error codes from other cell values.
     *
     * @constructor
     *
     * @param {String} code
     *  The native string representation of the error code.
     *
     * @property {String} code
     *  The native string representation of the error code, as passed to the
     *  constructor.
     *
     * @property {Number} num
     *  The internal numeric index of built-in error codes, used for example
     *  when comparing error codes in filters and filtering sheet functions.
     *  Will be set to NaN, if the passed error code is not a built-in code.
     */
    function ErrorCode(code) {
        this.code = code;
        this.num = ERROR_CODE_NUMBERS[code] || Number.NaN;
    }

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed value is an error code object (an instance of
     * the class ErrorCode) of the same type as represented by this error code
     * object.
     *
     * @param {Any} value
     *  Any literal value that can be used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an error code object with the same type as
     *  this error code.
     */
    ErrorCode.prototype.equals = function (value) {
        return (value instanceof ErrorCode) && (value.code === this.code);
    };

    /**
     * Returns the JSON representation of this error code (a string with a
     * leading has character).
     */
    ErrorCode.prototype.toJSON = function () { return this.code; };

    /**
     * Returns the string representation of this error code for debug logging.
     */
    ErrorCode.prototype.toString = function () { return this.code; };

    // static class SheetUtils ================================================

    /**
     * The static class SheetUtils contains various helper methods for cell
     * addresses, cell range addresses, and column/row index intervals.
     * Additionally, the class is a console logger bound to the URL hash flag
     * 'spreadsheet:log-models', that logs detailed information about the
     * document model instances and collections.
     */
    var SheetUtils = {};

    // logger interface -------------------------------------------------------

    Logger.extend(SheetUtils, { enable: 'spreadsheet:log-models', prefix: 'MODEL' });

    // helper classes ---------------------------------------------------------

    // export address/interval/range helper classes for convenience
    SheetUtils.Address = Address;
    SheetUtils.Interval = Interval;
    SheetUtils.Range = Range;
    SheetUtils.Range3D = Range3D;
    SheetUtils.AddressArray = AddressArray;
    SheetUtils.IntervalArray = IntervalArray;
    SheetUtils.RangeArray = RangeArray;
    SheetUtils.Range3DArray = Range3DArray;

    // constants --------------------------------------------------------------

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum width of columns, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_COLUMN_WIDTH = 20000;

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum height of rows, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_ROW_HEIGHT = 20000;

    /**
     * Maximum length of a defined name.
     *
     * @constant
     */
    SheetUtils.MAX_NAME_LENGTH = 256;

    /**
     * The maximum number of cells whose content values can be queried from the
     * document with a single request.
     *
     * @constant
     */
    SheetUtils.MAX_QUERY_CELL_COUNT = 10000;

    /**
     * The maximum number of cells to be filled with one range operation, or
     * auto-fill operation.
     *
     * @constant
     */
    SheetUtils.MAX_FILL_CELL_COUNT = 2000;

    /**
     * The maximum number of entire columns/rows to be filled with one
     * auto-fill operation.
     *
     * @constant
     */
    SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT = 10;

    /**
     * Maximum number of merged ranges to be created with a single operation.
     *
     * @constant
     */
    SheetUtils.MAX_MERGED_RANGES_COUNT = 1000;

    /**
     * The maximum number of cells to be filled when unmerging a range.
     *
     * @constant
     */
    SheetUtils.MAX_UNMERGE_CELL_COUNT = 5000;

    /**
     * The maximum number of columns to be changed with one row operation.
     *
     * @constant
     */
    SheetUtils.MAX_CHANGE_COLS_COUNT = 2000;

    /**
     * The maximum number of rows to be changed with one row operation.
     *
     * @constant
     */
    SheetUtils.MAX_CHANGE_ROWS_COUNT = 2000;

    /**
     * Maximum number of cells contained in the selection for local update of
     * selection settings (prevent busy JS if selection is too large).
     *
     * @constant
     */
    SheetUtils.MAX_SELECTION_CELL_COUNT = 100000;

    /**
     * The maximum number of search results returned by a single request.
     *
     * @constant
     */
    SheetUtils.MAX_SEARCH_RESULT_COUNT = 100;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the 'General' number format (in cells).
     *
     * @constant
     */
    SheetUtils.MAX_LENGTH_STANDARD_CELL = 11;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the 'General' number format (in cell edit mode).
     *
     * @constant
     */
    SheetUtils.MAX_LENGTH_STANDARD_EDIT = 21;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the 'General' number format (in formula interpretation, e.g. when
     * converting numbers to strings for functions expecting strings).
     *
     * @constant
     */
    SheetUtils.MAX_LENGTH_STANDARD_FORMULA = 20;

    /**
     * Minimum effective size of a cell, in pixels. Also used as maximum width
     * of cell border lines, to prevent that the left/right borders or the top/
     * bottom borders of a cell overlay each other.
     *
     * @constant
     */
    SheetUtils.MIN_CELL_SIZE = 5;

    /**
     * Minimum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     */
    SheetUtils.MIN_ZOOM = 0.5;

    /**
     * Maximum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     */
    SheetUtils.MAX_ZOOM = 8;

    /**
     * Whether multi-selection (multiple ranges at the same time) is supported.
     *
     * @constant
     */
    SheetUtils.MULTI_SELECTION = !Utils.TOUCHDEVICE;

    /**
     * Predefined native error codes, as used in operations.
     *
     * @constant
     */
    SheetUtils.ErrorCodes = {
        DIV0:  new ErrorCode('#DIV/0!'),
        NA:    new ErrorCode('#N/A'),
        NAME:  new ErrorCode('#NAME?'),
        NULL:  new ErrorCode('#NULL!'),
        NUM:   new ErrorCode('#NUM!'),
        REF:   new ErrorCode('#REF!'),
        VALUE: new ErrorCode('#VALUE!')
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed value is an error code object (an instance of
     * the internal class ErrorCode).
     *
     * @param {Any} value
     *  Any literal value that can be used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an error code object.
     */
    SheetUtils.isErrorCode = function (value) {
        return value instanceof ErrorCode;
    };

    /**
     * Creates an error code object for the passed string representation of an
     * error code.
     *
     * @param {String} code
     *  The native string representation of an error code.
     *
     * @returns {ErrorCode}
     *  The error code object for the passed error code.
     */
    SheetUtils.makeErrorCode = function (code) {
        return new ErrorCode(code);
    };

    /**
     * Convenience shortcut that returns a rejected promise with a specific
     * error code as object with 'cause' property.
     *
     * @param {String} errorCode
     *  The error code to reject the returned promise with.
     *
     * @returns {jQuery.Promise}
     *  A promise that has already been rejected with an object containing
     *  a string property 'cause' set to the passed error code.
     */
    SheetUtils.makeRejected = function (errorCode) {
        return $.Deferred().reject({ cause: errorCode });
    };

    /**
     * Returns the localized sheet name with the specified index, ready to be
     * used in the document model (without any I18N marker characters used for
     * UI debugging).
     *
     * @param {Number} sheet
     *  The zero-based sheet index to be inserted into the sheet name. The
     *  number inserted into the sheet name will be increased by one.
     *
     * @returns {String}
     *  The generated sheet name (e.g. 'Sheet1', 'Sheet2', etc.).
     */
    SheetUtils.getSheetName = function (sheet) {
        return _.noI18n.fix(
            //#. Default sheet names in spreadsheet documents. Should be equal to the sheet
            //#. names used in existing spreadsheet applications (Excel, OpenOffice Calc, ...).
            //#. %1$d is the numeric index inside the sheet name
            //#. Resulting names e.g. "Sheet1", "Sheet2", etc.
            //#. No white-space between the "Sheet" label and sheet index!
            //#, c-format
            gt('Sheet%1$d', _.noI18n(sheet + 1)));
    };

    /**
     * Returns the translated word for 'Column', ready to be used in the
     * document model (without any I18N marker characters used for UI
     * debugging).
     *
     * @returns {String}
     *  The translated word for 'Column'.
     */
    SheetUtils.getTableColName = function () {
        //#. Default name for columns in filtered ranges in spreadsheet documents. Will be
        //#. used to create serially numbered columns, e.g. "Column1", "Column2", etc.
        return _.noI18n.fix(gt.pgettext('filter', 'Column'));
    };

    // exports ================================================================

    return SheetUtils;

});
