/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/address', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    var // maximum number of columns that will ever be allowed in a sheet (A to ZZZZ)
        MAX_COL_COUNT = 475254,

        // maximum number of rows that will ever be allowed in a sheet (1 to 99,999,999)
        MAX_ROW_COUNT = 99999999,

        // regular expression pattern matching a valid column name (A to ZZZZ)
        COL_PATTERN = '[A-Z]{1,4}',

        // regular expression pattern matching a row name (1 to 99,999,999)
        ROW_PATTERN = '0*[1-9][0-9]{0,7}',

        // regular expression object matching a column name only
        RE_COL_NAME = new RegExp('^' + COL_PATTERN + '$', 'i'),

        // regular expression object matching a row name only
        RE_ROW_NAME = new RegExp('^' + ROW_PATTERN + '$'),

        // regular expression object matching a cell name
        RE_CELL_NAME = new RegExp('^(' + COL_PATTERN + ')(' + ROW_PATTERN + ')$', 'i');

    // class Address ==========================================================

    /**
     * Represents the address of a single cell in a sheet of a spreadsheet
     * document.
     *
     * @constructor
     *
     * @property {Number} 0
     *  The zero-based index of the column containing the cell.
     *
     * @property {Number} 1
     *  The zero-based index of the row containing the cell.
     */
    function Address(col, row) {

        this[0] = col;
        this[1] = row;

    } // class Address

    // constants --------------------------------------------------------------

    /**
     * Regular expression pattern matching a valid column name (A to ZZZZ).
     *
     * @constant
     */
    Address.COL_PATTERN = COL_PATTERN;

    /**
     * Regular expression pattern matching a row name (1 to 99,999,999).
     *
     * @constant
     */
    Address.ROW_PATTERN = ROW_PATTERN;

    /**
     * Convenience shortcut for the address A1 (zero for column and row index).
     *
     * @constant
     */
    Address.A1 = new Address(0, 0);

    // static methods ---------------------------------------------------------

    /**
     * Returns a unique string key for the specified cell address components
     * that can be used for example as key in an associative map.
     *
     * @param {Number} col
     *  The zero-based column index.
     *
     * @param {Number} row
     *  The zero-based row index.
     *
     * @returns {String}
     *  A unique string key for this cell address.
     */
    Address.key = function (col, row) {
        return col + ',' + row;
    };

    /**
     * Returns the string representation of the passed column index.
     *
     * @param {Number} col
     *  The zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column index.
     */
    Address.stringifyCol = function (col) {
        var name = '';
        for (; col >= 0; col = Math.floor(col / 26) - 1) {
            name = String.fromCharCode(65 + col % 26) + name;
        }
        return name;
    };

    /**
     * Returns the string representation of the passed row index.
     *
     * @param {Number} row
     *  The zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row index.
     */
    Address.stringifyRow = function (row) {
        return (row + 1).toString();
    };

    /**
     * Returns the string representation of the passed column or row index.
     *
     * @param {Number} index
     *  The zero-based column or row index.
     *
     * @param {Boolean} columns
     *  Whether the passed index is a column index (true), or a row index
     *  (false).
     *
     * @returns {String}
     *  The string representation of the column or row index.
     */
    Address.stringifyIndex = function (index, columns) {
        return columns ? Address.stringifyCol(index) : Address.stringifyRow(index);
    };

    /**
     * Returns the column index of the passed column string representation.
     *
     * @param {String} str
     *  The string representation of a column (case-insensitive).
     *
     * @returns {Number}
     *  The zero-based column index, if the passed string is valid, otherwise
     *  the value -1.
     */
    Address.parseCol = function (str) {
        var col = -1;
        if (RE_COL_NAME.test(str)) {
            str = str.toUpperCase();
            for (var ichar = 0; ichar < str.length; ichar += 1) {
                col = (col + 1) * 26 + str.charCodeAt(ichar) - 65;
            }
        }
        return (col < MAX_COL_COUNT) ? col : -1;
    };

    /**
     * Returns the row index of the passed row string representation.
     *
     * @param {String} str
     *  The string representation of a row (one-based).
     *
     * @returns {Number}
     *  The zero-based row index, if the passed string is valid, otherwise the
     *  value -1.
     */
    Address.parseRow = function (str) {
        var row = RE_ROW_NAME.test(str) ? (parseInt(str, 10) - 1) : -1;
        return (row < MAX_ROW_COUNT) ? row : -1;
    };

    /**
     * Returns the cell address for the passed string representation.
     *
     * @param {String} str
     *  The string representation of a cell address, in A1 notation.
     *
     * @returns {Address|Null}
     *  The cell address, if the passed string is valid, otherwise null.
     */
    Address.parse = function (str) {
        var matches = RE_CELL_NAME.exec(str);
        if (!matches) { return null; }
        var col = Address.parseCol(matches[1]), row = Address.parseRow(matches[2]);
        return ((col >= 0) && (row >= 0)) ? new Address(col, row) : null;
    };

    /**
     * Compares the passed cell addresses. Defines a natural order for sorting
     * with the following behavior (let A and B be cell addresses): If the row
     * index of A is less than the row index of B; or if both row indexes are
     * equal, and the column index of A is less than the column index of B,
     * then A is considered less than B (horizontally ordered cell addresses).
     *
     * @param {Address} address1
     *  The first cell address to be compared to the second cell address.
     *
     * @param {Address} address2
     *  The second cell address to be compared to the first cell address.
     *
     * @returns {Number}
     *  - A negative number, if address1 is less than address2.
     *  - A positive number, if address1 is greater than address2.
     *  - Zero, if both cell addresses are equal.
     */
    Address.compare = function (address1, address2) {
        return Utils.comparePairs(address1[1], address1[0], address2[1], address2[0]);
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a clone of this cell address.
     *
     * @returns {Address}
     *  A clone of this address.
     */
    Address.prototype.clone = function () {
        return new Address(this[0], this[1]);
    };

    /**
     * Returns whether the passed cell address is equal to this cell address.
     *
     * @param {Address} address
     *  The other cell address to be compared to this cell address.
     *
     * @returns {Boolean}
     *  Whether both cell addresses contain the same column and row indexes.
     */
    Address.prototype.equals = function (address) {
        return (this[0] === address[0]) && (this[1] === address[1]);
    };

    /**
     * Returns whether the passed cell address differs from this cell address.
     *
     * @param {Address} address
     *  The other cell address to be compared to this cell address.
     *
     * @returns {Boolean}
     *  Whether the passed cell address differs from this cell address.
     */
    Address.prototype.differs = function (address) {
        return (this[0] !== address[0]) || (this[1] !== address[1]);
    };

    /**
     * Compares this cell address with the passed cell address. See static
     * method Address.compare() for details.
     *
     * @param {Address} address
     *  The other cell address to be compared to this cell address.
     *
     * @returns {Number}
     *  - A negative number, if this cell address is less than the passed cell
     *      address.
     *  - A positive number, if this cell address is greater than the passed
     *      cell address.
     *  - Zero, if both cell addresses are equal.
     */
    Address.prototype.compareTo = function (address) {
        return Address.compare(this, address);
    };

    /**
     * Returns the column or row index of this cell address.
     *
     * @param {Boolean} columns
     *  Whether to return the column index (true), or the row index (false).
     *
     * @returns {Number}
     *  The column or row index of this cell address.
     */
    Address.prototype.get = function (columns) {
        return this[columns ? 0 : 1];
    };

    /**
     * Sets a new column or row index for this cell address.
     *
     * @param {Number} index
     *  The new column or row index for this cell address.
     *
     * @param {Boolean} columns
     *  Whether to change the column index (true), or the row index (false).
     *
     * @returns {Address}
     *  A reference to this instance.
     */
    Address.prototype.set = function (index, columns) {
        this[columns ? 0 : 1] = index;
        return this;
    };

    /**
     * Modifies the column or row index of this cell address relatively.
     *
     * @param {Number} distance
     *  The distance to move the column or row index of this cell address.
     *  Negative values will decrease the index.
     *
     * @param {Boolean} columns
     *  Whether to change the column index (true), or the row index (false).
     *
     * @returns {Address}
     *  A reference to this instance.
     */
    Address.prototype.move = function (distance, columns) {
        this[columns ? 0 : 1] += distance;
        return this;
    };

    /**
     * Returns a unique string key for this cell address that can be used for
     * example as key in an associative map. This method is faster than the
     * method Address.toString().
     *
     * @returns {String}
     *  A unique string key for this cell address.
     */
    Address.prototype.key = function () {
        return Address.key(this[0], this[1]);
    };

    /**
     * Returns the string representation of this cell address, in upper-case A1
     * notation.
     *
     * @returns {String}
     *  The string representation of this cell address, in upper-case A1
     *  notation.
     */
    Address.prototype.toString = function () {
        return Address.stringifyCol(this[0]) + Address.stringifyRow(this[1]);
    };

    /**
     * Returns the JSON representation of this cell address.
     *
     * @returns {Array<Number>}
     *  The JSON representation of this cell address (a plain JavaScript array
     *  with two elements containing the column and row index).
     */
    Address.prototype.toJSON = function () {
        return [this[0], this[1]];
    };

    // exports ================================================================

    return _.makeExtendable(Address);

});
