/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/mergecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, ModelObject, SheetUtils) {

    'use strict';

    var // convenience shortcuts
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray;

    // class MergeCollection ==================================================

    /**
     * Collects information about all merged cell ranges of a single sheet in a
     * spreadsheet document.
     *
     * Triggers the following events:
     * - 'insert:merged'
     *      After new merged ranges have been inserted into the collection.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RangeArray} ranges
     *          An array with the addresses of all new merged ranges.
     *      (3) {Object} [options]
     *          Optional parameters:
     *          - {Boolean} [options.implicit=false]
     *              If set to true, the merged ranges have been inserted (or
     *              moved) implicitly, after rows, columns, or cell ranges have
     *              been inserted or deleted in the sheet.
     * - 'delete:merged'
     *      After merged ranges have been deleted from the collection. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RangeArray} ranges
     *          An array with the addresses of all deleted merged ranges.
     *      (3) {Object} [options]
     *          Optional parameters:
     *          - {Boolean} [options.implicit=false]
     *              If set to true, the merged ranges have been deleted (or
     *              moved) implicitly, after rows, columns, or cell ranges have
     *              been inserted or deleted in the sheet.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function MergeCollection(sheetModel) {

        var // self reference
            self = this,

            // the spreadsheet document model
            docModel = sheetModel.getDocModel(),

            // all merged ranges
            mergedRanges = new RangeArray();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Returns whether the single columns or rows of the passed range are
         * merged. The merged columns/rows must exactly fit into the range, and
         * all columns/rows in the range must be merged.
         *
         * @param {Range} range
         *  The address of a cell range.
         *
         * @returns {Boolean}
         *  Whether the columns/rows of the specified range are merged.
         */
        function isColRowMerged(range, columns) {

            var // all merged ranges that are contained in the passed range
                containedRanges = self.getMergedRanges(range, 'contain');

            // the number of merged ranges contained by the passed range must be equal to the
            // column/row count of the range, otherwise there is no need to iterate at all
            if (containedRanges.length !== range.size(columns)) {
                return false;
            }

            // check size and position of the merged ranges (if all merged ranges start and
            // end at the boundaries of the passed range, and are single-sized in the other
            // direction, the entire range is filled by single-sized merged ranges)
            return containedRanges.every(function (mergedRange) {
                return (mergedRange.size(columns) === 1) &&
                    (mergedRange.getStart(!columns) === range.getStart(!columns)) &&
                    (mergedRange.getEnd(!columns) === range.getEnd(!columns));
            });
        }

        /**
         * Triggers the specified event, if the passed array of range addresses
         * is not empty.
         */
        function triggerEvent(type, ranges, options) {
            if (!ranges.empty()) {
                self.trigger(type, ranges, options);
            }
        }

        /**
         * Recalculates the position of all merged ranges, after columns or
         * rows have been inserted into or deleted from the sheet.
         */
        function transformationHandler(interval, insert, columns) {

            var // all deleted merged ranges
                deletedRanges = new RangeArray(),
                // all inserted merged ranges
                insertedRanges = new RangeArray();

            // visit all merged ranges in reversed order, to be able to delete in-place
            Utils.iterateArray(mergedRanges, function (mergedRange, index) {

                var // get the transformed range (will return null for resulting 1x1 ranges)
                    newRange = self.transformRange(mergedRange, interval, insert, columns);

                // handle deleted and changed merged ranges
                if (!newRange) {
                    deletedRanges.push(mergedRange);
                    mergedRanges.splice(index, 1);
                } else if (mergedRange.differs(newRange)) {
                    insertedRanges.push(newRange.clone());
                    deletedRanges.push(mergedRange);
                    mergedRanges[index] = newRange;
                }
            }, { reverse: true });

            // notify listeners (with option 'implicit' due do implicit update while inserting columns/rows)
            triggerEvent('delete:merged', deletedRanges, { implicit: true });
            triggerEvent('insert:merged', insertedRanges, { implicit: true });
        }

        // protected methods --------------------------------------------------

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this.getCloneData = function () {
            return { mergedRanges: mergedRanges };
        };

        /**
         * Clones all contents from the passed collection into this collection.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {MergeCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @returns {MergeCollection}
         *  A reference to this instance.
         */
        this.cloneFrom = function (collection) {

            var // the internal contents of the source collection
                cloneData = collection.getCloneData();

            // clone the contents of the source collection
            mergedRanges = cloneData.mergedRanges.clone(true);

            return this;
        };

        /**
         * Callback handler for the document operation 'mergeCells'. Merges the
         * cell range, or removes any merged ranges from the range covered by
         * the operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'mergeCells' document operation.
         */
        this.applyMergeCellsOperation = function (context) {

            var // the range covered by the operation
                range = context.getRange(),
                // the method how to merge/unmerge the operation range
                type = context.getEnum('type', /^(merge|horizontal|vertical|unmerge)$/),
                // all existing merged ranges deleted from the collection
                deletedRanges = new RangeArray(),
                // all merged ranges added to the collection
                insertedRanges = new RangeArray();

            // inserts the passed range into the collection, and into the 'insertedRanges' array
            function insertMergedRange(mergedRange) {

                var // the array index of the passed range in the 'deletedRanges' array
                    deletedIndex = Utils.findFirstIndex(deletedRanges, mergedRange.equals.bind(mergedRange));

                // if the range has been deleted, remove it from the array, otherwise
                // insert it into the 'insertedRanges' array (this prevents to notify
                // ranges that have been unmerged *and* merged)
                if (deletedIndex >= 0) {
                    deletedRanges.splice(deletedIndex, 1);
                } else {
                    insertedRanges.push(mergedRange);
                }

                // push a copy of the range into the collection
                mergedRanges.push(mergedRange.clone());
            }

            // always remove all merged ranges that overlap with the passed range,
            // regardless of passed type (needed to merge over merged ranges)
            mergedRanges = RangeArray.reject(mergedRanges, function (mergedRange) {
                var overlaps = range.overlaps(mergedRange);
                if (overlaps) { deletedRanges.push(mergedRange); }
                return overlaps;
            });

            // handle the different merge types
            switch (type) {

            case 'merge':
                if (range.cells() > 1) {
                    insertMergedRange(range);
                }
                break;

            case 'horizontal':
                if (range.cols() > 1) {
                    Utils.iterateRange(range.start[1], range.end[1] + 1, function (row) {
                        insertMergedRange(Range.create(range.start[0], row, range.end[0], row));
                    });
                }
                break;

            case 'vertical':
                if (range.rows() > 1) {
                    Utils.iterateRange(range.start[0], range.end[0] + 1, function (col) {
                        insertMergedRange(Range.create(col, range.start[1], col, range.end[1]));
                    });
                }
                break;

            case 'unmerge':
                // all merged ranges have been removed already, see above
                break;
            }

            // trigger change events with array of changed merged ranges
            triggerEvent('delete:merged', deletedRanges);
            triggerEvent('insert:merged', insertedRanges);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns, whether the cell at the specified address is the upper left
         * corner of a merged cell range.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified address is the upper left corner
         *  of a merged cell range.
         */
        this.isReferenceCell = function (address) {
            return this.getMergedRange(address, 'reference') !== null;
        };

        /**
         * Returns, whether the cell at the specified address is a hidden cell
         * inside a merged cell range.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified address is a hidden cell inside a
         *  merged cell range.
         */
        this.isHiddenCell = function (address) {
            return this.getMergedRange(address, 'hidden') !== null;
        };

        /**
         * Returns, whether the cell at the specified address is part of a
         * merged cell. It can be the reference cell or a hidden cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell is part of a merged cell range.
         */
        this.isMergedCell = function (address) {
            return this.getMergedRange(address, 'all') !== null;
        };

        /**
         * Returns whether the passed range is contained in this collection.
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified range is contained in this collection.
         */
        this.isMergedRange = function (range) {
            return mergedRanges.some(function (mergedRange) { return range.equals(mergedRange); });
        };

        /**
         * Returns whether the single columns of the passed range are merged.
         * The merged columns must exactly fit into the range, and all columns
         * in the range must be merged.
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified range is entirely vertically merged.
         */
        this.isVerticallyMerged = function (range) {
            return isColRowMerged(range, true);
        };

        /**
         * Returns whether the single rows of the passed range are merged. The
         * merged rows must exactly fit into the range, and all rows in the
         * range must be merged.
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified range is entirely horizontally merged.
         */
        this.isHorizontallyMerged = function (range) {
            return isColRowMerged(range, false);
        };

        /**
         * Returns the range address of a merged cell range for a specified
         * cell address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @param {String} [type='all']
         *  The address type of the specified address. It can be an arbitrary
         *  cell inside a merged cell range (type === 'all'), or it can be
         *  restricted to the upper left reference cell (type === 'reference')
         *  or to a hidden cell (type === 'hidden'). A hidden cell is any cell
         *  inside the merged cell range, that is not the upper left reference
         *  cell. If this address type is not specified, or if it is not one of
         *  the three types 'all', 'reference' or 'hidden', it defaults to
         *  'all'.
         *
         * @returns {Range|Null}
         *  The range address of a merged range that covers the passed cell
         *  address; or null, if no merged range exists at the address.
         */
        this.getMergedRange = function (address, type) {

            var // the merged range containing the cell with the passed address
                mergedRange = mergedRanges.findByAddress(address);

            if (mergedRange) {
                switch (type) {
                case 'reference':
                    if (!mergedRange.startAt(address)) { mergedRange = null; }
                    break;
                case 'hidden':
                    if (mergedRange.startsAt(address)) { mergedRange = null; }
                    break;
                }
            }

            return mergedRange ? mergedRange.clone() : null;
        };

        /**
         * Returns the range addresses of all merged cell ranges that overlap
         * with one of the passed ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {String} [type='all']
         *  If set to 'all' or omitted, returns all merged ranges that overlap
         *  the passed ranges with any cell. If set to 'contain', returns only
         *  merged ranges that are completely contained in any of the passed
         *  ranges. If set to 'reference', returns only merged ranges whose
         *  reference cell is contained in any of the passed ranges.
         *
         * @returns {RangeArray}
         *  The range addresses of all merged ranges that overlap with one of
         *  the passed ranges.
         */
        this.getMergedRanges = function (ranges, type) {

            var // the merged ranges not matching any passed range yet
                pendingRanges = mergedRanges.clone(),
                // the resulting merged ranges
                resultRanges = new RangeArray();

            // process all passed ranges, until 'pendingRanges' is empty
            RangeArray.some(ranges, function (range) {

                // iterate backwards through pendingRanges to be able to remove array elements
                Utils.iterateArray(pendingRanges, function (mergedRange, index) {

                    // move matching merged range from pendingRanges to resultRanges
                    if (range.overlaps(mergedRange)) {
                        switch (type) {
                        case 'reference':
                            if (range.containsAddress(mergedRange.start)) {
                                resultRanges.push(mergedRange);
                            }
                            break;
                        case 'contain':
                            if (range.contains(mergedRange)) {
                                resultRanges.push(mergedRange);
                            }
                            break;
                        default:
                            resultRanges.push(mergedRange);
                        }
                        // remove ALL overlapping ranges from pendingRanges (regardless of passed type)
                        pendingRanges.splice(index, 1);
                    }
                }, { reverse: true });

                // early exit of the loop, if all merged ranges will be returned
                return pendingRanges.empty();
            });

            return resultRanges;
        };

        /**
         * Returns the merged range that completely contains or is equal to the
         * passed cell range. Note the different to the method
         * MergeCollection.getMergedRanges() which returns all merged ranges
         * that are contained IN the passed ranges.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Range|Null}
         *  The address of a merged range that contains the passed range if
         *  existing, otherwise null
         */
        this.getBoundingMergedRange = function (range) {
            var boundingRange = _.find(mergedRanges, function (mergedRange) { return mergedRange.contains(range); });
            return boundingRange ? boundingRange : null;
        };

        /**
         * Returns whether any of the passed cell ranges overlaps with at least
         * one merged cell range.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Boolean}
         *  Whether any of the passed cell ranges contains or overlaps with at
         *  least one merged cell range.
         */
        this.rangesOverlapAnyMergedRange = function (ranges) {
            return mergedRanges.overlaps(ranges);
        };

        /**
         * Expands the passed range address, so that it will include all merged
         * ranges it is currently covering partly.
         *
         * @param {Range} range
         *  The address of a cell range that will be expanded to the merged
         *  ranges contained in this collection.
         *
         * @returns {Range}
         *  The address of the expanded cell range.
         */
        this.expandRangeToMergedRanges = function (range) {

            var // flat copy of the merged ranges, to prevent processing them multiple times
                pendingRanges = mergedRanges.clone(),
                // the length of the pending ranges array
                pendingLength = pendingRanges.length;

            // loop iterator function must be defined outside the do-while loop
            function expandToMergedRange(mergedRange, index) {
                // process all merged ranges that overlap the current range
                if (range.overlaps(mergedRange)) {
                    range = range.boundary(mergedRange);
                    pendingRanges.splice(index, 1);
                }
            }

            // ensure to keep passed range intact
            range = range.clone();

            // loop as long as the range will still be expanded with other merged ranges
            if (pendingRanges.length > 0) {
                do {
                    // store the current length of the pending ranges array (checking length afterwards)
                    pendingLength = pendingRanges.length;
                    // iterate backwards to be able to remove an element from the array
                    Utils.iterateArray(pendingRanges, expandToMergedRange, { reverse: true });
                } while (pendingRanges.length < pendingLength);
            }

            // return the fully expanded range
            return range;
        };

        /**
         * Shrinks the passed range address, so that it will not include any
         * merged ranges it is currently covering partly.
         *
         * @param {Range} range
         *  The address of a cell range that will be shrunk until it does not
         *  partly contain any merged ranges.
         *
         * @param {String} direction
         *  The direction in which the passed range will be shrunk. Supported
         *  values are 'columns' to shrink the range at its left and right
         *  border, or 'rows' to shrink the range at its top and bottom border.
         *
         * @returns {Range|Null}
         *  The address of the shrunk range; or null, if no valid range was
         *  left after shrinking (the range has been shrunk to zero
         *  columns/rows).
         */
        this.shrinkRangeFromMergedRanges = function (range, direction) {

            var // flat copy of the merged ranges, to prevent processing them multiple times
                pendingRanges = mergedRanges.clone(),
                // the length of the pending ranges array
                pendingLength = pendingRanges.length,
                // the direction to shrink the range
                columns = direction === 'columns';

            // loop iterator function must be defined outside the do-while loop
            function shrinkFromMergedRange(mergedRange, index) {

                // do not process merged ranges that are fully contained in the current range
                if (range.contains(mergedRange)) { return; }
                // remove the merged range from the pending array
                pendingRanges.splice(index, 1);
                // nothing to do, if merged range is completely outside
                if (!range.overlaps(mergedRange)) { return; }

                // shrink leading border of the range
                if ((mergedRange.getStart(columns) < range.getStart(columns)) && (range.getStart(columns) <= mergedRange.getEnd(columns))) {
                    range.setStart(mergedRange.getEnd(columns) + 1, columns);
                }
                // shrink trailing border of the range
                if ((mergedRange.getStart(columns) <= range.getEnd(columns)) && (range.getEnd(columns) < mergedRange.getEnd(columns))) {
                    range.setEnd(mergedRange.getStart(columns) - 1, columns);
                }

                // check if the range has been shrunk to null size
                if (range.getStart(columns) > range.getEnd(columns)) {
                    range = null;
                    return Utils.BREAK;
                }
            }

            // loop as long as the range will still be shrunk to exclude other merged ranges
            if (pendingRanges.length > 0) {
                do {
                    // store the current length of the pending ranges array (checking length afterwards)
                    pendingLength = pendingRanges.length;
                    // iterate backwards to be able to remove an element from the array
                    Utils.iterateArray(pendingRanges, shrinkFromMergedRange, { reverse: true });
                } while (range && (pendingRanges.length < pendingLength));
            }

            // return the shrunk range
            return range;
        };

        /**
         * Transforms the passed merged range according to the specified column
         * or row operation. Does NOT modify any merged range in this merged
         * ranges collection.
         *
         * @param {Range} mergedRange
         *  The address of the merged range to be transformed.
         *
         * @param {Interval} operationInterval
         *  The column/row interval of the operation.
         *
         * @param {Boolean} insert
         *  Whether the specified column/row interval has been inserted into
         *  the sheet (true), or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the specified interval is a column interval (true), or a
         *  row interval (false).
         *
         * @returns {Range|Null}
         *  The address of the transformed merged range. If the range has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, or if
         *  the resulting merged range covers a single cell only, null will be
         *  returned instead.
         */
        this.transformRange = function (mergedRange, operationInterval, insert, columns) {
            // transform the passed range without expanding the end of the range while inserting
            mergedRange = docModel.transformRange(mergedRange, operationInterval, insert, columns);
            // a transformed merged range covering a single cell is considered deleted
            return (mergedRange && (mergedRange.cells() > 1)) ? mergedRange : null;
        };

        // initialization -----------------------------------------------------

        // update merged ranges after inserting/deleting columns or rows in the sheet
        sheetModel.registerTransformationHandler(transformationHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = sheetModel = null;
        });

    } // class MergeCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: MergeCollection });

});
