/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/spreadsheet/model/formula/interpreter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/complex',
    'io.ox/office/spreadsheet/model/formula/matrix'
], function (Utils, Parser, BaseObject, ModelObject, SheetUtils, FormulaUtils, Complex, Matrix) {

    'use strict';

    var // convenience shortcuts
        ErrorCodes = SheetUtils.ErrorCodes,
        Address = SheetUtils.Address,
        Range3D = SheetUtils.Range3D,
        Range3DArray = SheetUtils.Range3DArray,

        // descriptors/implementations of all operators and functions
        DESCRIPTORS = {},

        // maximum time available for evaluating a single formula, in milliseconds
        MAX_EVAL_TIME = 1000;

    // imports ================================================================

    var // import all external operator and function implementations
        operatorPromise = require([
            'io.ox/office/spreadsheet/model/formula/impl/operators'
        ], function (OPERATORS) {
            DESCRIPTORS.OPERATORS = OPERATORS;
        }),

        // import all external operator and function implementations
        functionPromise = require([
            'io.ox/office/spreadsheet/model/formula/impl/complexfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/conversionfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/databasefuncs',
            'io.ox/office/spreadsheet/model/formula/impl/datetimefuncs',
            'io.ox/office/spreadsheet/model/formula/impl/engineeringfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/financialfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/informationfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/logicalfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/mathfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/matrixfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/referencefuncs',
            'io.ox/office/spreadsheet/model/formula/impl/statisticalfuncs',
            'io.ox/office/spreadsheet/model/formula/impl/textfuncs'
        ], function () {
            // merge all imported function descriptors into a single map
            DESCRIPTORS.FUNCTIONS = _.extend.apply(_, [{}].concat(_.toArray(arguments)));
        }),

        // wait for all operator and function modules
        modulePromise = $.when(operatorPromise, functionPromise);

    // private global functions ===============================================

    /**
     * Throws a fatal exception, and prints the passed error message to the
     * browser console.
     */
    function fatalError(msg) {
        Utils.error(msg);
        throw 'fatal';
    }

    /**
     * Throws the passed value, if it is an error code literal. Otherwise,
     * the passed value will be returned.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  Any literal value used in formulas.
     *
     * @returns {Number|String|Boolean|Null}
     *  The passed value, if it is not an error code.
     *
     * @throws {ErrorCode}
     *  The passed value, if it is an error code.
     */
    function checkValueForErrorCode(value) {
        if (SheetUtils.isErrorCode(value)) { throw value; }
        return value;
    }

    /**
     * Throws the first error code literal found in the passed constant matrix.
     * If no error code has been found, the passed matrix will be returned.
     *
     * @param {Matrix} matrix
     *  An matrix literal containing any values used in formulas.
     *
     * @returns {Matrix}
     *  The passed matrix, if it does not contain an error code.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix.
     */
    function checkMatrixForErrorCode(matrix) {
        matrix.forEach(checkValueForErrorCode);
        return matrix;
    }

    /**
     * Checks that all elements in the passed matrix literal pass the specified
     * truth test. If any matrix element fails, the #VALUE! error code will be
     * thrown. If any matrix element is an error code literal, it will be
     * thrown instead. If all matrix elements are valid, the passed matrix will
     * be returned.
     *
     * @param {Matrix} matrix
     *  A matrix literal containing any values used in formulas.
     *
     * @param {Function} callback
     *  The predicate callback function invoked for every matrix element.
     *
     * @returns {Matrix}
     *  The passed matrix, if all its elements are valid.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix, or the #VALUE!
     *  error code, if a matrix element does not pass the truth test.
     */
    function checkMatrixForDataType(matrix, callback) {
        matrix.forEach(function (elem) {
            checkValueForErrorCode(elem);
            if (!callback(elem)) { throw ErrorCodes.VALUE; }
        });
        return matrix;
    }

    /**
     * Returns whether the passed matrix size is inside the limits supported by
     * the formula interpreter.
     */
    function isValidMatrixSize(rows, cols) {
        return (rows > 0) && (rows <= FormulaUtils.MAX_MATRIX_ROW_COUNT) && (cols > 0) && (cols <= FormulaUtils.MAX_MATRIX_COL_COUNT);
    }

    /**
     * Checks that the all cell range addresses in the passed array refer to
     * the same single sheet.
     *
     * @returns {Range3DArray}
     *  The range array passed to this function, if it is valid.
     *
     * @throws {ErrorCode}
     *  - The #NULL! error code, if the range array is empty.
     *  - The #VALUE! error code, if the ranges refer to different sheets.
     */
    function checkSingleSheetRanges(ranges) {
        if (ranges.empty()) { throw ErrorCodes.NULL; }
        if (!ranges.singleSheet()) { throw ErrorCodes.VALUE; }
        return ranges;
    }

    /**
     * Returns a map containing the resolved operator or function descriptors
     * supported by the specified file format.
     *
     * @param {Object} rawDescriptors
     *  The operator or function descriptors to be converted.
     *
     * @param {String} fileFormat
     *  The identifier of the file format.
     *
     * @returns {Object}
     *  A map containing the resolved descriptors.
     */
    function resolveDescriptors(rawDescriptors, fileFormat) {

        // the resulting map with resolved descriptors
        var descriptors = {};

        // process all raw descriptors depending on the file format
        _.each(rawDescriptors, function (rawDescriptor, descName) {

            // adds entries for alternative names to the descriptors map
            function processAlternateNames(propName, hidden) {
                Utils.getTokenListOption(descriptor, propName, []).forEach(function (altName) {
                    var cloned = descriptors[altName] = _.clone(descriptor);
                    cloned.name = altName;
                    cloned.origName = descName;
                    cloned.hidden = hidden;
                });
            }

            // check all property names in debug mode
            Utils.withLogging(function () {
                _.each(rawDescriptor, function (value, propName) {
                    if (!/^(supported|hidden|altNames(Hidden)?|category|(min|max|repeat)Params|type|signature|resolve)$/.test(propName)) {
                        Utils.error('resolveDescriptors(): unknown property "' + propName + '" in descriptor of ' + descName + '.');
                    }
                });
            });

            // skip descriptor, if not supported by the current file format
            var supportedFormats = Utils.getTokenListOption(rawDescriptor, 'supported');
            if (_.isArray(supportedFormats) && !_.contains(supportedFormats, fileFormat)) { return; }

            // create a new entry in the returned map, copy all properties of the raw descriptor (pick format specific values)
            var descriptor = descriptors[descName] = Utils.mapProperties(rawDescriptor, function (value) {
                if (!_.isObject(value) || _.isArray(value) || _.isFunction(value)) { return value; }
                if (_.isObject(value) && (fileFormat in value)) { return value[fileFormat]; }
            });

            // convert category list to a set
            descriptor.category = Utils.makeSet(Utils.getTokenListOption(descriptor, 'category', []));

            // add own name into the object for convenience
            descriptor.name = descName;

            // add the hidden property
            descriptor.hidden = Utils.getBooleanOption(descriptor, 'hidden', false);

            // convert signature strings to arrays
            descriptor.signature = Utils.getTokenListOption(descriptor, 'signature', []);

            // insert map entries for alternative names
            processAlternateNames('altNames', false);
            processAlternateNames('altNamesHidden', true);
        });

        // delete useless data from the new descriptors
        _.each(descriptors, function (descriptor) {
            delete descriptor.altNames;
            delete descriptor.altNamesHidden;
        });

        return descriptors;
    }

    /**
     * Returns a map containing the resolved operator descriptors supported by
     * the specified file format.
     *
     * @param {String} fileFormat
     *  The identifier of the file format.
     *
     * @returns {Object}
     *  A map containing the resolved operator descriptors.
     */
    var resolveOperatorDescriptors = _.memoize(function (fileFormat) {
        return resolveDescriptors(DESCRIPTORS.OPERATORS, fileFormat);
    });

    /**
     * Returns a map containing the resolved function descriptors supported by
     * the specified file format.
     *
     * @param {String} fileFormat
     *  The identifier of the file format.
     *
     * @returns {Object}
     *  A map containing the resolved function descriptors.
     */
    var resolveFunctionDescriptors = _.memoize(function (fileFormat) {
        return resolveDescriptors(DESCRIPTORS.FUNCTIONS, fileFormat);
    });

    // class FormulaContext ===================================================

    /**
     * An instance of this class serves as calling context object for the
     * implementations of operators and functions, so that the symbol 'this'
     * inside these implementations provides various useful helper methods.
     *
     * @constructor
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {Number|Null} refSheet
     *  The index of the sheet the interpreted formula is related to. Used to
     *  resolve reference tokens without sheet reference.
     *
     * @param {Address} targetAddress
     *  The address of the target reference cell in the specified reference
     *  sheet the formula is related to.
     */
    function FormulaContext(docModel, refSheet, targetAddress) {

        var // self reference
            self = this,

            // the application instance
            app = docModel.getApp(),

            // the number formatter
            numberFormatter = docModel.getNumberFormatter(),

            // the formula tokenizer for the UI language
            tokenizer = docModel.getFormulaTokenizer('ui'),

            // maximum length of a string, maximum valid string index (one-based)
            MAX_STRING_LEN = app.isODF() ? 65535 : 32767;

        // private methods ----------------------------------------------------

        /**
         * Converts the passed floating-point number to a string formatted with
         * the 'General' number format using the appropriate maximum number of
         * digits.
         *
         * @param {Number} number
         *  The floating-point number to be converted to a string.
         *
         * @returns {String}
         *  The converted floating-point number.
         */
        function convertNumberToString(number) {
            return numberFormatter.formatStandardNumber(number, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);
        }

        /**
         * Converts the passed complex number to a string. The real and
         * imaginary coefficients will be formatted with the 'General' number
         * format using the appropriate maximum number of digits.
         *
         * @param {Complex} complex
         *  The complex number to be converted to a string.
         *
         * @returns {String}
         *  The converted complex number.
         */
        function convertComplexToString(complex) {

            var // the real coefficient, as string
                real = convertNumberToString(complex.real),
                // the imaginary coefficient, as string
                imag = convertNumberToString(complex.imag),
                // fall-back to unit 'i'
                unit = complex.unit || 'i';

            // leave out imaginary part if zero
            if (complex.imag === 0) { return real; }

            // leave out real part if missing; do not add single '1' for imaginary part
            return ((complex.real === 0) ? '' : real) + (((complex.real !== 0) && (complex.imag > 0)) ? '+' : '') + (/^-?1$/.test(imag) ? '' : imag) + unit;
        }

        /**
         * Converts the passed text to a complex number.
         *
         * @param {String} text
         *  The string to be converted to a complex number.
         *
         * @returns {Complex|Null}
         *  The complex number represented by the passed string; or null, if
         *  the string cannot be parsed to a complex number.
         */
        function convertStringToComplex(text) {

            var // the matches of a regular expression
                matches = null,
                // the parse result for a floating-point number from formatter
                parseResult = null,
                // the parsed real coefficient
                real = 0;

            // do not accept empty strings
            if (text.length === 0) { return null; }

            // string may be a single imaginary unit without coefficients: i, +i, -i (same for j)
            if ((matches = /^([-+]?)([ij])$/.exec(text))) {
                return new Complex(0, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull leading floating-point number from the string
            if (!(parseResult = Parser.parseLeadingNumber(text, { sign: true }))) {
                return null;
            }

            // check for simple floating-point number without imaginary coefficient: a, +a, -a
            real = parseResult.number;
            text = parseResult.remaining;
            if (text === '') {
                return new Complex(real, 0);
            }

            // check for imaginary number without real coefficient: bi, +bi, -bi
            if ((text === 'i') || (text === 'j')) {
                return new Complex(0, real, text);
            }

            // check for following imaginary unit without coefficients, but with sign: a+i, a-i
            if ((matches = /^([-+])([ij])$/.exec(text))) {
                return new Complex(real, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull trailing floating-point number from the string: a+bi, a-bi (sign is required here, something like 'abi' is not valid)
            if (!(parseResult = Parser.parseLeadingNumber(text, { sign: true })) || (parseResult.sign.length === 0)) {
                return null;
            }

            // remaining text must be the imaginary unit
            text = parseResult.remaining;
            if ((text === 'i') || (text === 'j')) {
                return new Complex(real, parseResult.number, text);
            }

            return null;
        }

        /**
         * Throws a CIRCULAR_ERROR error code, if the passed cell range address
         * contains the reference cell of this formula context.
         *
         * @param {Range3D} range
         *  The cell range address to be checked.
         *
         * @returns {Range3D}
         *  The passed cell range address, if it does not contain the reference
         *  cell.
         *
         * @throws {ErrorCode}
         *  The CIRCULAR_ERROR error code, if the passed cell range address
         *  contains the reference cell of this formula context.
         */
        function checkCircularReference(range) {
            if (_.isNumber(refSheet) && range.containsSheet(refSheet) && range.containsAddress(targetAddress)) {
                throw FormulaUtils.CIRCULAR_ERROR;
            }
            return range;
        }

        /**
         * Returns the type of the passed operand.
         */
        function getOperandType(operand) {
            if (operand instanceof Operand) { return operand.getType(); }
            if (operand instanceof Matrix) { return 'mat'; }
            if ((operand instanceof Range3D) || (operand instanceof Range3DArray)) { return 'ref'; }
            return 'val';
        }

        /**
         * Visits all single values contained in the passed operand, matrix, or
         * reference (invokes the callback function with the result of the
         * value converter).
         */
        function iterateOperandValues(operand, converter, callback, options) {

            var // whether to visit the empty function parameter
                emptyParam = Utils.getBooleanOption(options, 'emptyParam', false),
                // whether to visit empty cells in references
                emptyCell = Utils.getBooleanOption(options, 'emptyCell', false),
                // whether to pass error codes to the iterator
                acceptErrors = Utils.getBooleanOption(options, 'acceptErrors', false),
                // whether to visit cells in references in order
                ordered = Utils.getBooleanOption(options, 'ordered', false),
                // performance: do not visit more than a specific number of cells in cell references
                count = 0;

            // invokes the callback function for a single value
            function invokeCallback(value, col, row) {
                // throw error codes, unless error codes will be accepted
                if (!acceptErrors) { checkValueForErrorCode(value); }
                // convert the passed value (may throw an error code)
                value = converter.call(self, value);
                // check maximum number of visited values (for performance)
                count += 1;
                if (count > FormulaUtils.MAX_CELL_ITERATION_COUNT) { throw FormulaUtils.UNSUPPORTED_ERROR; }
                // invoke the callback function
                callback.call(self, value, col, row);
            }

            // visits a single value operand (skip empty parameters unless specified otherwise)
            function iterateValue(value) {
                if (emptyParam || !_.isNull(value)) {
                    invokeCallback(value, 0, 0);
                }
            }

            // visits all elements of a matrix
            function iterateMatrix(matrix) {
                matrix.forEach(function (element, row, col) {
                    invokeCallback(element, col, row);
                });
            }

            // visits all filled cells in the passed cell range address
            function iterateRange(range) {

                var // start position of the iteration range
                    col = range.start[0], row = range.start[1];

                // loop through all sheets referred by the rage
                Utils.iterateRange(range.sheet1, range.sheet2 + 1, function (sheet) {

                    var // the sheet model containing the cell
                        sheetModel = docModel.getSheetModel(sheet);

                    // safety check: #REF! error if passed sheet index is invalid
                    if (!sheetModel) { throw ErrorCodes.REF; }

                    // iterate all content cells in the cell collection (skip empty cells)
                    sheetModel.getCellCollection().iterateCellsInRanges(range.toRange(), function (cellData) {
                        invokeCallback(cellData.result, cellData.address[0] - col, cellData.address[1] - row);
                    }, { type: emptyCell ? 'all' : 'content', hidden: 'all', ordered: ordered });
                });
            }

            // visit all filled cells in a cell reference
            function iterateRanges(ranges) {
                if (Utils.getBooleanOption(options, 'complexRef', false)) {
                    Range3DArray.forEach(ranges, function (range) {
                        checkCircularReference(range);
                        iterateRange(range);
                    });
                } else {
                    iterateRange(self.convertToValueRange(ranges));
                }
            }

            // passed operand can also be a matrix or a reference
            if (operand instanceof Matrix) {
                iterateMatrix(operand);
            } else if ((operand instanceof Range3D) || (operand instanceof Range3DArray)) {
                iterateRanges(operand);
            } else if (operand instanceof Operand) {
                if (operand.isMatrix()) {
                    iterateMatrix(operand.getMatrix());
                } else if (operand.isReference()) {
                    iterateRanges(operand.getReference());
                } else {
                    iterateValue(operand.getValue());
                }
            } else {
                fatalError('FormulaContext.iterateOperandValues(): invalid operand type');
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the application instance containing the interpreted formula.
         *
         * @returns {SpreadsheetApplication}
         *  The application instance containing the interpreted formula.
         */
        this.getApp = function () {
            return app;
        };

        /**
         * Returns whether the edited document is an Office-Open-XML file.
         *
         * @returns {Boolean}
         *  Whether the edited document is an Office-Open-XML file.
         */
        this.isOOXML = function () {
            return app.isOOXML();
        };

        /**
         * Returns whether the edited document is an OpenDocument file.
         *
         * @returns {Boolean}
         *  Whether the edited document is an OpenDocument file.
         */
        this.isODF = function () {
            return app.isODF();
        };

        /**
         * Returns the document model containing the interpreted formula.
         *
         * @returns {SpreadsheetModel}
         *  The document model containing the interpreted formula.
         */
        this.getDocModel = function () {
            return docModel;
        };

        /**
         * Returns the number formatter of the document model.
         *
         * @returns {NumberFormatter}
         *  The number formatter of the document model.
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Returns the formula tokenizer of the UI grammar from the parent
         * model, needed for specific conversion tasks.
         *
         * @returns {Tokenizer}
         *  The formula tokenizer needed for specific conversion tasks.
         */
        this.getTokenizer = function () {
            return tokenizer;
        };

        /**
         * Converts the passed value to be stored in an operand. A date will be
         * converted to a floating-point number, if possible. If the value is
         * INF or NaN, the #NUM! error code literal will be returned instead.
         * If the number is too close to zero (using a denormalized binary
         * representation internally), zero will be returned. If a string is
         * too long, a #VALUE! error code will be returned. Complex numbers
         * will be converted to their string representation, unless one of the
         * coefficients is INF or NaN which will result in the #NUM! error too.
         *
         * @param {Number|Date|String|Boolean|Complex|ErrorCode|Null} value
         *  The value to be validated.
         *
         * @returns {Number|String|Boolean|Null|ErrorCode}
         *  The validated value, ready to be inserted into a formula operand.
         */
        this.validateValue = function (value) {

            // convert date object to floating-point number
            if (value instanceof Date) {
                value = numberFormatter.convertDateToNumber(value);
                return _.isNumber(value) ? value : ErrorCodes.NUM;
            }

            // convert complex numbers to strings
            if (value instanceof Complex) {
                // convert INF or NaN to #NUM! error
                if (!_.isFinite(value.real) || !_.isFinite(value.imag)) { return ErrorCodes.NUM; }
                // convert very small (denormalized) numbers to zero
                if (Math.abs(value.real) < FormulaUtils.MIN_NUMBER) { value.real = 0; }
                if (Math.abs(value.imag) < FormulaUtils.MIN_NUMBER) { value.imag = 0; }
                // convert to string
                value = convertComplexToString(value);
            }

            // validate floating-point numbers
            if (_.isNumber(value)) {
                // convert INF or NaN to #NUM! error
                if (!_.isFinite(value)) { return ErrorCodes.NUM; }
                // convert very small (denormalized) numbers to zero
                return (Math.abs(value) < FormulaUtils.MIN_NUMBER) ? 0 : value;
            }

            // validate strings
            if (_.isString(value)) {
                // check maximum length of string (depends on file format)
                return (value.length <= MAX_STRING_LEN) ? value : ErrorCodes.VALUE;
            }

            // no conversion for other values
            return value;
        };

        /**
         * Checks the passed string length used in string functions. Throws the
         * #VALUE! error code, if the index is less than zero, or larger than
         * the maximum length of string results supported by the formula
         * engine.
         *
         * @param {Number} length
         *  The string length to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the passed string length is invalid.
         */
        this.checkStringLength = function (length) {
            if ((length < 0) || (length > MAX_STRING_LEN)) { throw ErrorCodes.VALUE; }
        };

        /**
         * Checks the passed character index used in string functions. Throws
         * the #VALUE! error code, if the index is less than one, or larger
         * than the maximum length of string results supported by the formula
         * engine.
         *
         * @param {Number} index
         *  The character index to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the passed character index is invalid.
         */
        this.checkStringIndex = function (index) {
            if ((index < 1) || (index > MAX_STRING_LEN)) { throw ErrorCodes.VALUE; }
        };

        /**
         * Checks the passed complex numbers for their imaginary unit. Throws
         * the #VALUE! error code, if the imaginary units of the numbers are
         * different.
         *
         * @param {Complex} complex1
         *  The first complex number to be checked.
         *
         * @param {Complex} complex2
         *  The second complex number to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the imaginary units of the passed
         *  complex numbers are different.
         */
        this.checkComplexUnits = function (complex1, complex2) {
            if (_.isString(complex1.unit) && _.isString(complex2.unit) && (complex1.unit !== complex2.unit)) {
                throw ErrorCodes.VALUE;
            }
        };

        /**
         * Returns whether the passed sheet index is equal to the reference
         * sheet the interpreted formula is located in.
         *
         * @param {Number} sheet
         *  A zero-based sheet index to be tested against the reference sheet.
         *
         * @returns {Boolean}
         *  Whether the passed sheet index is equal to the reference sheet.
         */
        this.isRefSheet = function (sheet) {
            return _.isNumber(refSheet) && (refSheet >= 0) && (refSheet === sheet);
        };

        /**
         * Returns the index of the reference sheet the interpreted formula is
         * located in.
         *
         * @returns {Number}
         *  The zero-based index of the reference sheet.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the formula is being interpreted without a
         *  reference sheet.
         */
        this.getRefSheet = function () {
            if (!_.isNumber(refSheet)) { throw ErrorCodes.REF; }
            return refSheet;
        };

        /**
         * Returns whether the passed cell address is equal to the address of
         * the target reference cell the interpreted formula is located in.
         *
         * @param {Address} address
         *  A cell address to be tested against the target reference address.
         *
         * @returns {Boolean}
         *  Whether the passed cell address is equal to the address of the
         *  target reference cell.
         */
        this.isRefAddress = function (address) {
            return targetAddress.equals(address);
        };

        /**
         * Returns the address of the target reference cell the interpreted
         * formula is located in.
         *
         * @returns {Address}
         *  The address of the target reference cell.
         */
        this.getRefAddress = function () {
            return targetAddress.clone();
        };

        /**
         * Returns the current result value of the specified cell.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Address} address
         *  The address of the cell in the specified sheet.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The current value of the specified cell.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the passed sheet index is invalid.
         */
        this.getCellValue = function (sheet, address) {

            var // the sheet model containing the cell
                sheetModel = docModel.getSheetModel(sheet);

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCodes.REF; }

            // return typed value from cell collection
            return sheetModel.getCellCollection().getCellResult(address);
        };

        /**
         * Returns the formula string of the specified cell.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Address} address
         *  The address of the cell in the specified sheet.
         *
         * @returns {String|Null}
         *  The current formula string of the specified cell; or null, if the
         *  cell does not contain a formula.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the passed sheet index is invalid.
         */
        this.getCellFormula = function (sheet, address) {

            var // the sheet model containing the cell
                sheetModel = docModel.getSheetModel(sheet);

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCodes.REF; }

            // get formula string from cell collection
            return sheetModel.getCellCollection().getCellFormula(address);
        };

        /**
         * Tries to convert the passed literal value to a floating-point
         * number.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a number. Strings that represent
         *  a valid number (containing the decimal separator of the current GUI
         *  language) will be converted to the number. The Boolean value FALSE
         *  will be converted to 0, the Boolean value TRUE will be converted to
         *  1. The special value null (representing an empty cell) will be
         *  converted to 0.
         *
         * @returns {Number}
         *  The floating-point number, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. If the value cannot be converted to a floating-point
         *  number, a #VALUE! error will be thrown.
         */
        this.convertToNumber = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // convert strings and Booleans to numbers
            if (_.isString(value)) {
                var parseResult = numberFormatter.parseFormattedNumber(value);
                value = parseResult ? parseResult.number : Number.NaN;
            } else if (_.isBoolean(value)) {
                value = value ? 1 : 0;
            } else if (value instanceof Date) {
                // value may become null which will cause the #VALUE! error below
                value = numberFormatter.convertDateToNumber(value);
            } else if (_.isNull(value)) {
                value = 0;
            }

            // the resulting value must be a finite floating-point number
            if (_.isNumber(value) && _.isFinite(value)) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed literal value to a Date object.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a number. Strings that represent
         *  a valid number (containing the decimal separator of the current GUI
         *  language) will be converted to a date representing that number. The
         *  Boolean value FALSE will be converted to the null date, the Boolean
         *  value TRUE will be converted to the day following the null date.
         *  The special value null (representing an empty cell) will be
         *  converted to 0.
         *
         * @returns {Date}
         *  The date, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. If the value cannot be converted to a date value, a #VALUE!
         *  error will be thrown.
         */
        this.convertToDate = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            if (!(value instanceof Date)) {
                // convert the value to a number (failing conversion will throw)
                value = this.convertToNumber(value);
                // convert the number to a date object
                value = numberFormatter.convertNumberToDate(value);
            }

            // the resulting value must be a Date object
            if (value instanceof Date) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed literal value to a string.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a string. Numbers will be
         *  converted to decimal or scientific notation (containing the decimal
         *  separator of the current GUI language), Boolean values will be
         *  converted to their translated text representation (formulas will
         *  always be calculated using the current GUI language). The special
         *  value null (representing an empty cell) will be converted to the
         *  empty string.
         *
         * @returns {String}
         *  The string representation of the value, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. All other values will be converted to strings.
         */
        this.convertToString = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and Booleans to strings
            if (_.isNumber(value) && _.isFinite(value)) {
                value = convertNumberToString(value);
            } else if (_.isBoolean(value)) {
                value = app.getBooleanLiteral(value); // always translated
            } else if (_.isNull(value)) {
                value = '';
            }

            // the resulting value must be a string
            if (_.isString(value)) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed value to a Boolean value.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a Boolean. Floating-point
         *  numbers will be converted to TRUE if not zero, otherwise FALSE.
         *  Strings containing the exact translated name of the TRUE or FALSE
         *  values (case-insensitive) will be converted to the respective
         *  Boolean value (formulas will always be calculated using the current
         *  GUI language). The special value null (representing an empty cell)
         *  will be converted to FALSE.
         *
         * @returns {Boolean}
         *  The Boolean value, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. If the value cannot be converted to a Boolean, a #VALUE!
         *  error will be thrown.
         */
        this.convertToBoolean = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and strings to Booleans
            if (_.isNumber(value) && _.isFinite(value)) {
                value = Math.abs(value) > FormulaUtils.MIN_NUMBER;
            } else if (_.isString(value)) {
                value = value.toUpperCase();
                if (value === app.getBooleanLiteral(true)) { // always translated
                    value = true;
                } else if (value === app.getBooleanLiteral(false)) {
                    value = false;
                }
            } else if (_.isNull(value)) {
                value = false;
            }

            // the resulting value must be a Boolean
            if (_.isBoolean(value)) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed literal value to a complex number.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a complex number. Floating-point
         *  numbers will be converted to complex numbers with the imaginary
         *  coefficient set to zero. Strings must represent a valid complex
         *  number (optional leading signed real coefficient, optional trailing
         *  signed imaginary coefficient, followed by the lower-case character
         *  'i' or 'j'). The coefficients must containing the decimal separator
         *  of the current GUI language. The special value null (representing
         *  an empty cell) will be converted to the complex number 0.
         *
         * @returns {Complex}
         *  The complex number, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. Boolean values will cause to throw a #VALUE! error. If the
         *  string cannot be converted to a complex number, a #NUM! error will
         *  be thrown (not the #VALUE! error code as thrown by the other
         *  conversion methods).
         */
        this.convertToComplex = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // Boolean values result in #VALUE! (but not invalid strings)
            if (_.isBoolean(value)) { throw ErrorCodes.VALUE; }

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and strings to complex numbers
            if (_.isNumber(value) && _.isFinite(value)) {
                value = new Complex(value, 0);
            } else if (_.isString(value)) {
                value = convertStringToComplex(value);
            } else if (_.isNull(value)) {
                // empty cells are treated as complex number 0 (in difference to empty strings)
                value = new Complex(0, 0);
            }

            // the resulting value must be a complex number
            if (value instanceof Complex) { return value; }
            // invalid strings will result in #NUM! instead of #VALUE!
            throw ErrorCodes.NUM;
        };

        /**
         * Extracts a single cell range from the passed array of cell range
         * addresses. The range array must contain exactly one cell range
         * address. Unless specified via an option, the cell range must refer
         * to a single sheet.
         *
         * @param {Range3DArray|Range3D} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.multiSheet=false]
         *      If set to true, the cell range address may refer to multiple
         *      sheets.
         *  @param {Boolean} [options.valueError=false]
         *      If set to true, a #VALUE! error code (instead of a #REF! error
         *      code) will be thrown, if the passed range array contains more
         *      than one cell range address.
         *
         * @returns {Range3D}
         *  The cell range address, if available.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the range array is empty.
         *  - The #REF! or #VALUE! error code (see option 'valueError'), if the
         *      range array contains more than one element.
         *  - The #VALUE! error code, if the option 'multiSheet' has not been
         *      set, and a single cell range address is available but refers to
         *      multiple sheets.
         */
        this.convertToRange = function (ranges, options) {

            // must not be empty
            if (ranges.empty()) { throw ErrorCodes.NULL; }

            // must be a single range in a single sheet
            if (ranges.length > 1) {
                throw Utils.getBooleanOption(options, 'valueError', false) ? ErrorCodes.VALUE : ErrorCodes.REF;
            }

            // must point to a single sheet, unless multiple sheets are allowed (always throw the #VALUE! error code)
            if (!Utils.getBooleanOption(options, 'multiSheet', false) && !ranges[0].singleSheet()) {
                throw ErrorCodes.VALUE;
            }

            return ranges[0];
        };

        /**
         * Extracts a single cell range from the passed array of cell range
         * addresses, intended to be resolved to the cell contents in that
         * range. The range array must contain exactly one cell range address
         * pointing to a single sheet, and the reference cell of this formula
         * context must not be part of that range.
         *
         * @param {Range3DArray|Range3D} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Range3D}
         *  The resulting cell range address, if available. The sheet indexes
         *  in that range will be equal.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the range array is empty.
         *  - The #VALUE! error code, if the range array contains more than one
         *      element, or if the only range refers to multiple sheets.
         *  - The CIRCULAR_ERROR error code, if a single range is available,
         *      but the reference cell is part of that range (a circular
         *      reference).
         */
        this.convertToValueRange = function (ranges) {
            // resolve to a single range (throw #VALUE! on multiple ranges), detect circular references
            return checkCircularReference(this.convertToRange(ranges, { valueError: true }));
        };

        /**
         * Visits all values contained in the passed operand. Provides a single
         * constant of a literal operand, all elements of a matrix operand, and
         * the cell contents referred by a reference operand.
         *
         * @param {Any} operand
         *  The operand value (single value of any supported data type,
         *  including null and error codes; or instances of the classes Matrix,
         *  Range3DArray, or Range3D) whose content values will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for each value contained in the
         *  operand. Will be called in the context of this instance. Receives
         *  the value as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the callback function will be invoked with the value
         *      null. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.acceptErrors=false]
         *      If set to true, the callback function will be invoked for error
         *      code literals too. By default, error codes will be thrown
         *      immediately.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         */
        this.iterateValues = function (operand, callback, options) {
            iterateOperandValues(operand, _.identity, callback, options);
            return this;
        };

        /**
         * Visits all floating-point numbers contained in the passed operand.
         * Provides a single constant of a literal operand, all elements of a
         * matrix operand, and the cell contents referred by a reference
         * operand. In difference to simple value parameters where strings and
         * Boolean values contained in matrixes and cell ranges will always be
         * converted to numbers if possible, this method implements different
         * conversion strategies as required by the respective functions.
         *
         * @param {Any} operand
         *  The operand value (single value of any supported data type,
         *  including null and error codes; or instances of the classes Matrix,
         *  Range3DArray, or Range3D) whose content values will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for each floating-point number
         *  contained in the operand. Will be called in the context of this
         *  instance. Receives the number as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.valMode='convert']
         *      Conversion strategy for operands of type value:
         *      - 'convert': Strings will be converted to numbers, if possible.
         *          Boolean values will be converted to numbers. Strings that
         *          represent a Boolean value result in a #VALUE! error code
         *          though.
         *      - 'rconvert' (restricted conversion): Strings will be converted
         *          to numbers, if possible. Boolean values and strings that
         *          represent a Boolean value result in a #VALUE! error code
         *          though.
         *      - 'exact': Accepts numbers only. All strings and Boolean values
         *          result in a #VALUE! error code.
         *      - 'skip': All strings and all Boolean values will be skipped
         *          silently, also strings that could be converted to numbers.
         *      - 'zero': All strings will be treated as the number zero, also
         *          strings that could be converted to numbers. Boolean values
         *          will be converted to numbers.
         *  @param {String} [options.matMode='convert']
         *      Conversion strategy for the elements in matrixes. See option
         *      'valMode' for details.
         *  @param {String} [options.refMode='convert']
         *      Conversion strategy for the contents of cells referenced by the
         *      operand. See option 'valMode' for details.
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the callback function will be invoked with the value
         *      zero. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         */
        this.iterateNumbers = function (operand, callback, options) {

            var // conversion strategy
                convertMode = Utils.getStringOption(options, getOperandType(operand) + 'Mode', 'convert');

            iterateOperandValues(operand, function (value) {
                switch (convertMode) {
                case 'convert':
                    // convert value to number (may still fail for strings)
                    return this.convertToNumber(value);
                case 'rconvert':
                    // throw #VALUE! error code on Booleans
                    if (_.isBoolean(value)) { throw ErrorCodes.VALUE; }
                    // convert value to number (may still fail for strings)
                    return this.convertToNumber(value);
                case 'exact':
                    // throw #VALUE! error code on Booleans and strings
                    if (_.isString(value) || _.isBoolean(value)) { throw ErrorCodes.VALUE; }
                    // convert dates to numbers
                    return this.convertToNumber(value);
                case 'skip':
                    // skip strings and Boolean values
                    return _.isNumber(value) ? value : null;
                case 'zero':
                    // replace all strings with zero
                    return _.isString(value) ? 0 : this.convertToNumber(value);
                default:
                    fatalError('FormulaContext.iterateNumbers(): unknown conversion mode: "' + convertMode + '"');
                }
            }, callback, Utils.extendOptions(options, { acceptErrors: false }));

            return this;
        };

        /**
         * Visits all Boolean values contained in the passed operand. Provides
         * a single constant of a literal operand, all elements of a matrix
         * operand, and the cell contents referred by a reference operand. In
         * difference to simple value parameters where numbers and strings
         * contained in matrixes and cell ranges will always be converted to
         * Boolean values if possible, this method implements different
         * conversion strategies as required by the respective functions.
         *
         * @param {Any} operand
         *  The operand value (single value of any supported data type,
         *  including null and error codes; or instances of the classes Matrix,
         *  Range3DArray, or Range3D) whose content values will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for each Boolean value contained in
         *  the operand. Will be called in the context of this instance.
         *  Receives the Boolean value as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.valMode='convert']
         *      Conversion strategy for operands of type value:
         *      - 'convert': Strings will be converted to Boolean values, if
         *          possible. Numbers will be converted to Boolean values.
         *          Strings that represent a number result in a #VALUE! error
         *          code though.
         *      - 'skip': All strings will be skipped silently, also strings
         *          that could be converted to Boolean values. Numbers will be
         *          converted though.
         *  @param {String} [options.matMode='convert']
         *      Conversion strategy for the elements in matrixes. See option
         *      'valMode' for details.
         *  @param {String} [options.refMode='convert']
         *      Conversion strategy for the contents of cells referenced by the
         *      operand. See option 'valMode' for details.
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the callback function will be invoked with the value
         *      FALSE. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         */
        this.iterateBooleans = function (operand, callback, options) {

            var // conversion strategy
                convertMode = Utils.getStringOption(options, getOperandType(operand) + 'Mode', 'convert');

            iterateOperandValues(operand, function (value) {
                switch (convertMode) {
                case 'convert':
                    // convert value to Boolean (may still fail for strings)
                    return this.convertToBoolean(value);
                case 'skip':
                    // skip strings (but not numbers)
                    return _.isString(value) ? null : this.convertToBoolean(value);
                default:
                    fatalError('FormulaContext.iterateBooleans(): unknown conversion mode: "' + convertMode + '"');
                }
            }, callback, Utils.extendOptions(options, { acceptErrors: false }));

            return this;
        };

        /**
         * Reduces all numbers (and other values convertible to numbers) of all
         * function parameters to the result of a spreadsheet function.
         *
         * @param {Array|Arguments} operands
         *  The operands containing numbers to be aggregated. The elements of
         *  the array can be instances of the classes Operand, Matrix,
         *  Range3DArray, Range3D, or may be any literal value used in
         *  formulas.
         *
         * @param {Number} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The current intermediate result. On first invocation, this is
         *      the number passed in the 'initial' parameter.
         *  (2) {Number} number
         *      The new number to be combined with the intermediate result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} count
         *      The count of all visited numbers.
         *  Must return the final numeric result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @param {Object} iteratorOptions
         *  Parameters passed to the number iterator used internally (see
         *  method FormulaContext.iterateNumbers() for details).
         *
         * @returns {Number}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateNumbers = function (operands, initial, aggregate, finalize, iteratorOptions) {

            var // the intermediate result
                result = initial,
                // the count of all visited numbers in all operands
                count = 0;

            // process all function parameters
            _.each(operands, function (operand) {
                this.iterateNumbers(operand, function (number) {
                    result = aggregate.call(this, result, number);
                    count += 1;
                }, iteratorOptions);
            }, this);

            // resolve the final result, throw error codes
            result = finalize.call(this, result, count);
            if (_.isNumber(result)) { return result; }
            if (SheetUtils.isErrorCode(result)) { throw result; }
            fatalError('FormulaContext.aggregateNumbers(): invalid aggregation result');
        };

        /**
         * Reduces all Boolean values (and other values convertible to Boolean
         * values) of all function parameters to the result of a spreadsheet
         * function.
         *
         * @param {Array|Arguments} operands
         *  The operands containing Boolean values to be aggregated. The
         *  elements of the array can be instances of the classes Operand,
         *  Matrix, Range3DArray, Range3D, or may be any literal value used in
         *  formulas.
         *
         * @param {Boolean} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Boolean} result
         *      The current intermediate result. On first invocation, this is
         *      the Boolean value passed in the 'initial' parameter.
         *  (2) {Boolean} bool
         *      The new Boolean value to be combined with the intermediate
         *      result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Boolean} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} count
         *      The count of all visited Boolean values.
         *  Must return the final Boolean result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @param {Object} iteratorOptions
         *  Parameters passed to the Boolean value iterator used internally
         *  (see method FormulaContext.iterateBooleans() for details).
         *
         * @returns {Boolean}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateBooleans = function (operands, initial, aggregate, finalize, iteratorOptions) {

            var // the intermediate result
                result = initial,
                // the count of all visited Boolean values in all operands
                count = 0;

            // process all function parameters
            _.each(operands, function (operand) {
                this.iterateBooleans(operand, function (bool) {
                    result = aggregate.call(this, result, bool);
                    count += 1;
                }, iteratorOptions);
            }, this);

            // resolve the final result, throw error codes
            result = finalize.call(this, result, count);
            if (_.isBoolean(result)) { return result; }
            if (SheetUtils.isErrorCode(result)) { throw result; }
            fatalError('FormulaContext.aggregateBooleans(): invalid aggregation result');
        };

        /**
         * Converts the passed filter pattern to a case-insensitive regular
         * expression object.
         *
         * @param {String} pattern
         *  The filter pattern. An asterisk character in the string matches any
         *  sequence of characters (also the empty sequence), and a question
         *  mark matches an arbitrary single character. If an asterisk or
         *  question mark is preceded by a tilde character, they lose their
         *  special meaning and will be matched literally.
         *
         * @param {Boolean} [complete=false]
         *  If set to true, the created regular expression will only match
         *  complete strings. By default, any substring of a string will be
         *  matched.
         *
         * @returns {RegExp}
         *  A case-insensitive regular expression that represents the passed
         *  filter pattern.
         */
        this.convertPatternToRegExp = function (pattern, complete) {

            // convert passed search text to regular expression pattern matching the text literally
            pattern = _.escapeRegExp(pattern)
                // replace all occurrences of asterisk (without leading tilde) to '.*' RE pattern (e.g. 'a\*b' => 'a.*b')
                .replace(/(^|[^~])\\\*/g, '$1.*')
                // replace all occurrences of question mark (without leading tilde) to '.' RE pattern (e.g. 'a\?b' => 'a.b')
                .replace(/(^|[^~])\\\?/g, '$1.')
                // remove tilde characters before (already escaped) asterisks and question marks (e.g. 'a~\*b' => 'a\*b')
                .replace(/~(\\[*?])/g, '$1');

            // optionally, let the regular expression only match complete strings
            if (complete) { pattern = '^' + pattern + '$'; }

            // create the case-insensitive regular expression
            return new RegExp(pattern, 'i');
        };

        /**
         * Creates a filter matcher predicate function for the specified filter
         * criterion.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} criterion
         *  The filter criterion. Numbers, Boolean values and error codes will
         *  match the corresponding values, and will match strings that can be
         *  converted to these values. The value null represents a reference to
         *  an empty cell, and will match the number zero only (but not the
         *  empty string, and not empty cells). Strings may contain leading
         *  comparison operators, or may specify filter patterns with asterisk
         *  characters or question marks. Otherwise, string criteria will be
         *  converted to numbers, Boolean values, or error codes, if possible,
         *  and will match the corresponding values.
         *
         * @returns {Function}
         *  A predicate function that takes a single parameter (a value of any
         *  type that can occur in a formula), and returns a Boolean whether
         *  the passed value matches the filter criterion.
         */
        this.createFilterMatcher = function (criterion) {

            var // the number formatter of the document
                numberFormatter = this.getNumberFormatter(),
                // the filter operator (empty string for criterion without operator)
                op = null,
                // the filter operand
                comp = null,
                // whether to test for equality (needs some special handling)
                equality = null;

            // parse the filter operator and the filter operand from a string criterion
            if (_.isString(criterion)) {
                var matches = /^(<=|>=|<>|<|>|=)?(.*)$/.exec(criterion);
                op = _.isString(matches[1]) ? matches[1] : '';
                // leading apostrophe in operand counts as regular character
                comp = numberFormatter.parseResultValue(matches[2], { keepApos: true });
            } else {
                // all other data types cause filtering by equality
                op = '';
                comp = criterion;
            }

            // testing for equality converts strings in source data to other data types
            equality = (op === '') || (op === '=');

            // empty filter operand (empty parameter, or reference to empty cell) matches zero
            // numbers (and strings that can be parsed to zero number) in source data, but not
            // empty strings, FALSE, or empty cells
            if (_.isNull(comp)) {
                return function zeroMatcher(value) {
                    return (_.isString(value) ? numberFormatter.parseResultValue(value) : value) === 0;
                };
            }

            // special handling for the empty string as filter operand
            if (comp === '') {
                switch (op) {

                // literal empty string without filter operator matches empty strings AND empty cells
                case '':
                    return function emptyMatcher(value) { return (value === '') || _.isNull(value); };

                // single equality operator without operand matches empty cells, but not empty strings
                case '=':
                    return _.isNull;

                // single inequality operator without operand matches anything but empty cells
                case '<>':
                    return function notEmptyMatcher(value) { return !_.isNull(value); };

                // all other comparison operators without operand do not match anything
                default:
                    return _.constant(false);
                }
            }

            // equality or inequality with pattern matching
            if (_.isString(comp) && /[*?]/.test(comp) && (equality || (op === '<>'))) {

                var // the regular expression for the pattern (match complete strings only)
                    regExp = this.convertPatternToRegExp(comp, true);

                // create the pattern matcher predicate function (pattern filters work on strings only)
                return function patternMatcher(value) {
                    return equality === (_.isString(value) && regExp.test(value));
                };
            }

            // strings are always matched case-insensitively
            if (_.isString(comp)) { comp = comp.toUpperCase(); }

            // create the comparison matcher predicate function (pattern filters work on strings only)
            return function comparisonMatcher(value) {

                var // the relation between the value and the filter operand (signed integer)
                    rel = Number.NaN;

                // equality operator (but not inequality operator) converts strings in source data to other data types
                if (equality && _.isString(value)) {
                    value = numberFormatter.parseResultValue(value);
                }

                // compare the passed value with the filter operand (NaN means incompatible types)
                if (_.isNumber(comp) && _.isNumber(value)) {
                    rel = FormulaUtils.compareNumbers(value, comp);
                } else if (_.isString(comp) && _.isString(value)) {
                    rel = FormulaUtils.compareStrings(value, comp);
                } else if (_.isBoolean(comp) && _.isBoolean(value)) {
                    rel = FormulaUtils.compareBooleans(value, comp);
                } else if (SheetUtils.isErrorCode(comp) && SheetUtils.isErrorCode(value)) {
                    rel = FormulaUtils.compareErrorCodes(value, comp);
                }

                // return whether the value matches, according to the filter operand
                switch (op) {
                case '':
                case '=':
                    return isFinite(rel) && (rel === 0);
                case '<>':
                    return !isFinite(rel) || (rel !== 0); // incompatible types match the inequality operator
                case '<':
                    return isFinite(rel) && (rel < 0);
                case '<=':
                    return isFinite(rel) && (rel <= 0);
                case '>':
                    return isFinite(rel) && (rel > 0);
                case '>=':
                    return isFinite(rel) && (rel >= 0);
                }

                fatalError('FormulaContext.comparisonMatcher(): invalid operator');
            };
        };

        /**
         * Reduces all values that match a specific filter criterion to the
         * result of a spreadsheet function.
         *
         * @param {Range3DArray|Range3D} sourceRanges
         *  The addresses of the cell ranges containing the source data to be
         *  filtered for.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} criterion
         *  The filter criterion. See FormulaContext.createFilterMatcher()
         *  method for more details.
         *
         * @param {Range3DArray|Range3D|Null} dataRanges
         *  The addresses of the data ranges that will be aggregated to the
         *  result of the spreadsheet function. If missing or null, the values
         *  in the filtered source ranges will be used.
         *
         * @param {Number} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The current intermediate result. On first invocation, this is
         *      the number passed in the 'initial' parameter.
         *  (2) {Number} number
         *      The new number to be combined with the intermediate result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} numberCount
         *      The count of all numbers the aggregation callback function has
         *      been invoked for.
         *  (3) {Number} matchingCount
         *      The count of all matching values in the source range (also
         *      non-numeric matching cells).
         *  Must return the final numeric result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @returns {Number}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateFiltered = function (sourceRanges, criterion, dataRanges, initial, aggregate, finalize) {

            var // value matcher predicate function for the source data
                matcher = this.createFilterMatcher(criterion),
                // extract the address of the data range (throws on error)
                dataRange = dataRanges ? this.convertToValueRange(dataRanges) : null,
                // the intermediate result
                result = initial,
                // the count of all numbers
                numberCount = 0,
                // the count of all matching values in the source range
                matchingCount = 0;

            // process all values in the passed operand (also empty cells in cell references)
            this.iterateValues(sourceRanges, function (value, col, row) {

                // skip non-matching cells completely
                if (!matcher(value)) { return; }

                var // the target address in the data range (invalid addresses will be ignored silently)
                    address = dataRange ? new Address(dataRange.start[0] + col, dataRange.start[1] + row) : null,
                    // pick the value from the data cell to be aggregated, or use the source value
                    dataValue = address ? this.getCellValue(dataRange.sheet1, address) : value;

                // aggregate numbers only, without automatic conversion of strings
                if (_.isNumber(dataValue)) {
                    result = aggregate.call(this, result, dataValue);
                    numberCount += 1;
                }

                // count all matching cells regardless of their type, or the type of the corresponding data cell
                matchingCount += 1;

            }, {
                emptyParam: true, // empty parameters count as zero
                emptyCell: true, // empty cells in references are matched by the empty string
                acceptErrors: true, // error codes can be matched by the filter criterion
                complexRef: false // do not accept multi-range and multi-sheet references
            });

            // resolve the final result
            return finalize.call(this, result, numberCount, matchingCount);
        };

    } // class FormulaContext

    // class ValueOperandMixin ================================================

    /**
     * Mix-in for the class Operand for constant values.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Number|Date|String|Boolean|Null|ErrorCode} value
     *  The value carried by the operand.
     */
    function ValueOperandMixin(context, value) {

        // public methods -----------------------------------------------------

        /**
         * Returns the operand type.
         *
         * @returns {String}
         *  The text 'val' representing single literal values.
         */
        this.getType = function () {
            return 'val';
        };

        /**
         * Returns the value contained in this operand.
         *
         * @returns {Number|String|Boolean|ErrorCode}
         *  The literal value contained in this operand.
         */
        this.getValue = function () {
            return value;
        };

        /**
         * Packs the value of this operand into a 1x1 matrix.
         *
         * @returns {Matrix}
         *  A constant 1x1 matrix with the value of this operand.
         */
        this.getMatrix = function () {
            return new Matrix([[value]]);
        };

        /**
         * Returns the unresolved cell range addresses contained in this
         * operand. Constant values and constant matrixes will immediately
         * result in throwing the #VALUE! error code.
         *
         * @throws {ErrorCode}
         *  The error code #VALUE!.
         */
        this.getReference = function () {
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns the string representation of the operand for debug logging.
         */
        this.toString = function () {
            return FormulaUtils.valueToString(value);
        };

        // initialization -----------------------------------------------------

        // validate the passed value
        value = context.validateValue(value);

    } // ValueOperandMixin

    // class MatrixOperandMixin ===============================================

    /**
     * Mix-in for the class Operand for matrix constants.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Matrix} matrix
     *  The matrix carried by the operand.
     */
    function MatrixOperandMixin(context, matrix) {

        // public methods -----------------------------------------------------

        /**
         * Returns the operand type.
         *
         * @returns {String}
         *  The text 'mat' representing a constant matrix literal.
         */
        this.getType = function () {
            return 'mat';
        };

        /**
         * Returns the top-left element of the matrix contained in this
         * operand.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The specified matrix element.
         */
        this.getValue = function (row, col) {
            return matrix.get(row || 0, col || 0);
        };

        /**
         * Returns the constant matrix contained in this operand.
         *
         * @returns {Matrix}
         *  The matrix contained in this operand.
         */
        this.getMatrix = function () {
            return matrix;
        };

        /**
         * Returns the unresolved cell range addresses contained in this
         * operand. Constant values and constant matrixes will immediately
         * result in throwing the #VALUE! error code.
         *
         * @throws {ErrorCode}
         *  The error code #VALUE!.
         */
        this.getReference = function () {
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns the string representation of the operand for debug logging.
         */
        this.toString = function () {
            return matrix.toString();
        };

        // initialization -----------------------------------------------------

        // validate all numbers in the matrix
        matrix.transform(context.validateValue, context);

    } // MatrixOperandMixin

    // class ReferenceOperandMixin ============================================

    /**
     * Mix-in for the class Operand for cell range references.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Range3DArray} ranges
     *  The cell range addresses carried by the operand. MUST NOT be an empty
     *  array.
     */
    function ReferenceOperandMixin(context, ranges) {

        var // the document model
            docModel = context.getDocModel();

        // public methods -----------------------------------------------------

        /**
         * Returns the operand type.
         *
         * @returns {String}
         *  The text 'ref' representing an unresolved cell range reference.
         */
        this.getType = function () {
            return 'ref';
        };

        /**
         * Returns the content value of the cell that is related to the passed
         * reference cell. The operand must consist of a single cell range
         * address that does not cover the reference cell.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  A single literal value, if available. The value null represents an
         *  empty cell, or a cell that has not been stored locally yet.
         *
         * @throws {ErrorCode}
         *  - CIRCULAR_ERROR, if the operand results in a circular reference.
         *  - The #VALUE! error code, if the reference contains more than one
         *      cell range, or if the cell range cannot be resolved to a single
         *      value according to the reference cell.
         */
        this.getValue = function () {

            // extract single cell range address (throws on error)
            var range = context.convertToValueRange(ranges);

            // pick value from a single cell
            if (range.single()) {
                return context.getCellValue(range.sheet1, range.start);
            }

            // the reference cell of the formula context
            var refAddress = context.getRefAddress();

            // pick matching cell from left or right (row interval must cover the reference cell)
            if ((range.end[0] < refAddress[0]) || (refAddress[0] < range.start[0])) {
                if (range.start[0] !== range.end[0]) { throw ErrorCodes.VALUE; }
                if (!range.containsRow(refAddress[1])) { throw ErrorCodes.VALUE; }
                return context.getCellValue(range.sheet1, new Address(range.start[0], refAddress[1]));
            }

            // pick matching cell from above or below (column interval must cover the reference cell)
            if ((range.end[1] < refAddress[1]) || (refAddress[1] < range.start[1])) {
                if (range.start[1] !== range.end[1]) { throw ErrorCodes.VALUE; }
                if (!range.containsCol(refAddress[0])) { throw ErrorCodes.VALUE; }
                return context.getCellValue(range.sheet1, new Address(refAddress[0], range.start[1]));
            }

            // range is not left of, right of, above, or below the reference cell
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns a two-dimensional matrix with the contents of the referenced
         * cells. The operand must consist of a single cell range address that
         * does not cover the reference cell, and the range must not exceed
         * specific limits in order to prevent performance problems.
         *
         * @returns {Matrix}
         *  A constant matrix, if available.
         *
         * @throws {ErrorCode}
         *  - UNSUPPORTED_ERROR, if the resulting matrix would be too large.
         *  - CIRCULAR_ERROR, if the operand results in a circular reference.
         *  - The #VALUE! error code, if the range array contains more than one
         *      cell range address.
         */
        this.getMatrix = function () {

            var // extract single cell range address (throws on error)
                range = context.convertToValueRange(ranges),
                // size of the matrix
                rows = range.rows(),
                cols = range.cols(),
                // the sheet model containing the cell
                sheetModel = docModel.getSheetModel(range.sheet1),
                // the resulting matrix
                matrix = null;

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCodes.REF; }

            // restrict matrix size
            if (!isValidMatrixSize(rows, cols)) {
                throw FormulaUtils.UNSUPPORTED_ERROR;
            }

            // build an empty matrix
            matrix = Matrix.create(rows, cols, null);

            // fill existing cell values
            sheetModel.getCellCollection().iterateCellsInRanges(range.toRange(), function (cellData) {
                matrix.set(cellData.address[1] - range.start[1], cellData.address[0] - range.start[0], cellData.result);
            }, { type: 'content', hidden: 'all' });

            return matrix;
        };

        /**
         * Returns the unresolved cell range addresses contained in this
         * operand.
         *
         * @returns {Range3DArray}
         *  The unresolved cell range addresses contained in this operand.
         */
        this.getReference = function () {
            return ranges;
        };

        /**
         * Returns the string representation of the operand for debug logging.
         */
        this.toString = function () {
            return ranges.toString();
        };

    } // ReferenceOperandMixin

    // class Operand ==========================================================

    /**
     * Represents a single operand stored in the operand stack of a formula
     * token interpreter.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Any} value
     *  The value to be stored as operand. Can be a constant value (numbers,
     *  strings, Boolean values, null), a Date object (will be converted to a
     *  floating-point number immediately), an error code literal (instance of
     *  the class ErrorCode), a matrix literal (instance of the class Matrix),
     *  a single cell range address (instance of the class Range3D), or an
     *  array of cell range addresses (instance of the class Range3DArray).
     */
    function Operand(context, value) {

        // public methods -----------------------------------------------------

        /**
         * Returns whether this operand contains a single value (type 'val').
         *
         * @returns {Boolean}
         *  Whether this operand contains a single value (type 'val').
         */
        this.isValue = function () {
            return this.getType() === 'val';
        };

        /**
         * Returns whether this operand contains a matrix (type 'mat').
         *
         * @returns {Boolean}
         *  Whether this operand contains a constant matrix (type 'mat').
         */
        this.isMatrix = function () {
            return this.getType() === 'mat';
        };

        /**
         * Returns whether this operand contains a cell reference (type 'ref').
         *
         * @returns {Boolean}
         *  Whether this operand contains a cell reference (type 'ref').
         */
        this.isReference = function () {
            return this.getType() === 'ref';
        };

        /**
         * Returns whether this operand is empty (e.g. the second operand in
         * the formula =SUM(1,,2) is empty).
         *
         * @returns {Boolean}
         *  Whether this operand is empty.
         */
        this.isEmpty = function () {
            return this.isValue() && _.isNull(this.getValue());
        };

        // initialization -----------------------------------------------------

        // cell range addresses passed: validate size
        if ((value instanceof Range3D) || (value instanceof Range3DArray)) {
            value = Range3DArray.get(value);
            if (value.empty()) {
                value = ErrorCodes.NULL;
            } else if (value.length > FormulaUtils.MAX_REF_LIST_SIZE) {
                value = FormulaUtils.UNSUPPORTED_ERROR;
            } else {
                ReferenceOperandMixin.call(this, context, value);
                return;
            }
        }

        // matrix passed: validate size
        if (value instanceof Matrix) {
            if (!isValidMatrixSize(value.rows(), value.cols())) {
                value = FormulaUtils.UNSUPPORTED_ERROR;
            } else {
                MatrixOperandMixin.call(this, context, value);
                return;
            }
        }

        // otherwise: simple value (as last check, preceding code may have
        // changed the passed value to an error code)
        ValueOperandMixin.call(this, context, value);

    } // class Operand

    // class OperandsMixin ====================================================

    /**
     * A mix-in class for an instance of FormulaContext. Adds more methods
     * providing access to the existing operands in a function call.
     *
     * @constructor
     *
     * @extends FormulaContext
     *
     * @param {Array} operands
     *  The original operands processed by the current operator or function.
     */
    function OperandsMixin(operands) {

        // public methods -----------------------------------------------------

        /**
         * Returns the number of existing operands passed to the processed
         * operator or function.
         *
         * @returns {Number}
         *  The number of existing operands.
         */
        this.getOperandCount = function () {
            return operands.length;
        };

        /**
         * Returns the specified operand passed to the processed operator or
         * function.
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Operand}
         *  The specified operand, if existing.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the operand does not exist.
         */
        this.getOperand = function (index) {
            if ((index >= 0) && (index < operands.length)) { return operands[index]; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns the specified operands passed to the processed operator or
         * function as array.
         *
         * @param {Number} index
         *  The zero-based index of the first operand to be returned.
         *
         * @returns {Array}
         *  The specified operand and all its successors, if existing.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the specified operand does not exist.
         */
        this.getOperands = function (index) {
            if ((index >= 0) && (index < operands.length)) { return operands.slice(index); }
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns whether the specified operand is missing (the optional
         * parameters at the end of a function; e.g. the third parameter in the
         * formula =SUM(1,2) is missing).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand is missing. Returns false for
         *  existing but empty operands, e.g. in the formula =SUM(1,,2).
         */
        this.isMissingOperand = function (index) {
            return index >= operands.length;
        };

        /**
         * Returns whether the specified operand exists, but is empty (e.g. the
         * second operand in the formula =SUM(1,,2) is empty).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand exists but is empty. Returns false
         *  for missing operands (optional parameters at the end of the
         *  function). Returns false for references to empty cells too.
         */
        this.isEmptyOperand = function (index) {
            var operand = operands[index];
            return _.isObject(operand) && operand.isEmpty();
        };

        /**
         * Returns whether the specified operand is missing (the optional
         * parameters at the end of a function; e.g. the third parameter in the
         * formula =SUM(1,2) is missing), or empty (e.g. the second operand in
         * the formula =SUM(1,,2) is empty).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand is missing or empty.
         */
        this.isMissingOrEmptyOperand = function (index) {
            return this.isMissingOperand(index) || this.isEmptyOperand(index);
        };

        /**
         * Visits the existing operands of the processed operator or function.
         *
         * @param {Number} start
         *  Specifies the zero-based index of the first operand to be visited.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each operand.
         *  Receives the following parameters:
         *  (1) {Operand} operand
         *      The original operand instance.
         *  (2) {Number} index
         *      The zero-based index of the operand.
         *  The callback function will be invoked in the context of this
         *  instance.
         *
         * @returns {OperandsMixin}
         *  A reference to this instance.
         */
        this.iterateOperands = function (start, callback) {
            operands.slice(start).forEach(function (operand, index) {
                callback.call(this, operand, start + index);
            }, this);
            return this;
        };

    } // class OperandsMixin

    // class InterpreterInstance ==============================================

    /**
     * Resolves a compiled formula token array to the result of the formula.
     * This implementation has been moved outside the Interpreter class to be
     * able to recursively resolve multiple formulas (e.g. defined names).
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {Array} compilerTokens
     *  An array of token descriptors, in prefix notation, as returned by the
     *  method Compiler.compileTokens().
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the method
     *  Interpreter.interpretTokens().
     */
    var InterpreterInstance = BaseObject.extend({ constructor: function (docModel, compilerTokens, initOptions) {

        var // the application instance
            app = docModel.getApp(),

            // the reference sheet
            refSheet = Utils.getIntegerOption(initOptions, 'refSheet', null),

            // the target reference address
            targetAddress = Utils.getOption(initOptions, 'targetAddress', Address.A1),

            // the calling context for the operator resolver callback function
            context = new FormulaContext(docModel, refSheet, targetAddress),

            // all operators supported by the current file format
            operatorDescriptors = resolveOperatorDescriptors(app.getFileFormat()),

            // all functions supported by the current file format
            functionDescriptors = resolveFunctionDescriptors(app.getFileFormat()),

            // current array index into the compiler token array
            tokenIndex = 0,

            // start time of formula evaluation
            t0 = 0;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Resolves the passed reference token to its result.
         *
         * @param {ReferenceToken} refToken
         *  The formula token representing a cell range reference. If the token
         *  cannot be resolved to a cell range address, this method returns the
         *  #REF! error code.
         *
         * @returns {Range3DArray|ErrorCode}
         *  The cell range addresses contained in the reference, or a #REF!
         *  error code.
         */
        function resolveReference(refToken) {
            var range = refToken.getRange3D(initOptions);
            return range ? new Range3DArray(range) : ErrorCodes.REF;
        }

        /**
         * Resolves the passed name token to its result.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {NameToken} nameToken
         *  The formula token representing a defined name. If the name does not
         *  exist in the document, this method returns the #NAME? error code.
         *
         * @returns {Any}
         *  The result value of the defined name (may be values, matrixes, or
         *  references).
         */
        function resolveDefinedName(contextType, nameToken) {

            // try to resolve the model of a defined name from the name token
            var nameModel = nameToken.resolveNameModel(refSheet);

            // unknown names result in the #NAME? error code
            if (!nameModel) { return ErrorCodes.NAME; }

            // resolve the formula in the defined name (defined names are always relative to cell A1)
            var result = nameModel.getTokenArray().interpretFormula(contextType, { refSheet: refSheet, refAddress: Address.A1, targetAddress: targetAddress });

            // return UNSUPPORTED_ERROR on any error (TODO: evaluate error/warning code?)
            return (result.type === 'result') ? result.value : FormulaUtils.UNSUPPORTED_ERROR;
        }

        /**
         * Checks that the passed return type, and the context type of the
         * formula or subexpression match. Throws a 'reference' error, if the
         * context type is 'ref', and the passed type is different from 'ref'
         * or 'any'. Excel rejects such formulas, and even announces a broken
         * file format if such formulas are written into the file. Examples of
         * such invalid formulas are:
         *  =A1:1 (combined number with range operator)
         *  =OFFSET(1,1,1) (passed number to a reference-only parameter)
         *
         * It is possible to use operands and functions with return type
         * 'any' in reference context, e.g.
         *  =A1:name (return type of defined names is 'any')
         *  =A1:IF(TRUE,A1,B1) (return type of IF is 'any').
         * If the defined name or function returns a non-reference value, the
         * formula simply results in the #VALUE! error.
         *
         * @param {String} type
         *  The return type of an operator or function, or the type of a single
         *  operand to be checked. Must be one of:
         *  - 'val': A single value (numbers, strings, Boolean values, or error
         *      codes),
         *  - 'mat': Constant matrixes (matrix literals, or functions returning
         *      matrixes),
         *  - 'ref': Cell references (literal references, reference operators,
         *      functions returning a reference, or the literal #REF! error
         *      code),
         *  - 'any': Any of the above (defined names, a few functions returning
         *      any data type).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens() for
         *  details. If the context type is 'ref', only the return types 'ref'
         *  and 'any' will be accepted. All other return types will cause to
         *  throw an internal 'reference' exception that marks the formula
         *  structure to be ill-formed (see description above).
         *
         * @throws {String}
         *  The special error code 'reference', if the passed return type is
         *  not supported in reference context.
         */
        function checkContextType(type, contextType) {
            // context type 'ref': accept 'ref' and 'any' only
            if ((contextType === 'ref') && (type !== 'ref') && (type !== 'any')) {
                throw 'reference';
            }
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'val', i.e. whether it is a single value literal.
         */
        function isValType(value) {
            return _.isNumber(value) || _.isString(value) || _.isBoolean(value) || (value instanceof Date) || (value instanceof Complex) || SheetUtils.isErrorCode(value);
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'mat', i.e. whether it is a matrix literal.
         */
        function isMatType(value) {
            return value instanceof Matrix;
        }

        /**
         * Returns whether the passed value is supported by the operand type
         * 'ref', i.e. whether it is a single cell range address, or an array
         * of cell range addresses.
         */
        function isRefType(value) {
            return (value instanceof Range3D) || (value instanceof Range3DArray);
        }

        /**
         * Returns whether the passed value has the specified base type.
         */
        function isValueOfType(value, type) {
            switch (type) {
            case 'val':
                return isValType(value);
            case 'mat':
                return isMatType(value);
            case 'ref':
                return isRefType(value);
            case 'any':
                return isValType(value) || isMatType(value) || isRefType(value) || (value instanceof Operand);
            }
            fatalError('isValueOfType(): unknown type identifier: "' + type + '"');
        }

        /**
         * Creates an operand object for the passed parser token.
         *
         * @param {Object} compilerToken
         *  A compiler token representing an operand (literal values, constant
         *  matrixes, cell references, defined names).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the passed token.
         *
         * @throws {ErrorCode}
         *  The special error code UNSUPPORTED_ERROR, if the type of the passed
         *  token is not supported.
         */
        function makeTokenOperand(compilerToken, contextType) {

            var // the parser token representing the operand value
                token = compilerToken.token,
                // the token type
                type = null,
                // the value to be inserted into the operand
                value = null;

            FormulaUtils.log('> resolving operand token: ' + token);

            switch (token.getType()) {
            case 'lit': // literal value
                value = token.getValue();
                type = ErrorCodes.REF.equals(value) ? 'ref' : 'val';
                break;
            case 'mat': // matrix literals
                value = token.getMatrix();
                type = 'mat';
                break;
            case 'ref': // cell range references
                value = resolveReference(token);
                type = 'ref';
                break;
            case 'name': // defined names
                value = resolveDefinedName(contextType, token);
                type = 'any';
                break;
            default:
                fatalError('InterpreterInstance.makeTokenOperand(): unknown token type: "' + token.getType() + '"');
            }

            // check context type (will throw 'reference' error)
            checkContextType(type, contextType);
            return new Operand(context, value);
        }

        /**
         * Processes the passed operator or function call. Calculates the
         * required number of operands from the following compiler tokens, and
         * calculates the result of the operator or function.
         *
         * @param {Object} compilerToken
         *  A compiler token representing an operator or function.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the result of the operator or
         *  function call.
         *
         * @throws {String}
         *  The error code 'missing', if there are not enough tokens available
         *  in the compiled token array.
         */
        function processOperator(compilerToken, contextType) {

            var // the parser token representing the operator or function
                token = compilerToken.token,
                // the operator/function specification
                descriptor = null,
                // minimum/maximum number of parameters supported by the operator
                minParams = 0,
                maxParams = 0,
                // length of repeated parameter sequences
                repeatParams = 0,
                // the expanded type signature
                signature = null,
                // the operands required by the operator or function
                operands = null,
                // matrix size for repeated processing of value operators
                repeatRows = 0,
                repeatCols = 0,
                // the resulting operand
                result = null;

            FormulaUtils.log('> processing operator ' + token + ' with ' + compilerToken.params + ' parameters');

            // returns the type of the specified parameter from the signature
            function getParamType(index) {
                if (descriptor.signature.length <= index) {
                    var repeat = _.isNumber(descriptor.repeatCount) ? descriptor.repeatCount : 1;
                    index = descriptor.signature.length - repeat + ((index - descriptor.signature.length) % repeat);
                }
                return descriptor.signature[index];
            }

            // convert operand according to the expected type from the signature
            function resolveParam(operand, paramType, row, col) {
                switch (paramType) {
                case 'any':
                    return operand;
                case 'val':
                    return checkValueForErrorCode(operand.getValue(row, col));
                case 'val:num':
                    return context.convertToNumber(operand.getValue(row, col));
                case 'val:int':
                    return Math.floor(context.convertToNumber(operand.getValue(row, col)));
                case 'val:date':
                    return context.convertToDate(operand.getValue(row, col));
                case 'val:str':
                    return context.convertToString(operand.getValue(row, col));
                case 'val:bool':
                    return context.convertToBoolean(operand.getValue(row, col));
                case 'val:comp':
                    return context.convertToComplex(operand.getValue(row, col));
                case 'val:any':
                    return operand.getValue(row, col);
                case 'mat':
                    return checkMatrixForErrorCode(operand.getMatrix());
                case 'mat:num':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isNumber);
                case 'mat:str':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isString);
                case 'mat:bool':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isBoolean);
                case 'mat:any':
                    return operand.getMatrix();
                case 'ref':
                    return operand.getReference();
                case 'ref:sheet':
                    return checkSingleSheetRanges(operand.getReference());
                case 'ref:single':
                    return context.convertToRange(operand.getReference());
                case 'ref:multi':
                    return context.convertToRange(operand.getReference(), { multiSheet: true });
                }
                fatalError('InterpreterInstance.resolveParam(): unknown parameter type: "' + paramType + '"');
            }

            // resolves the operands according to the type signature
            function resolveOperator(row, col) {

                var // the parameter values to be passed to the operator resolver callback
                    params = null,
                    // the resulting operand
                    result = null;

                try {
                    // resolves the operand values according to the signature
                    params = operands.map(function (operand, index) {
                        return resolveParam(operand, signature[index].paramType, row, col);
                    });

                    // invoke the resolver callback function of the operator
                    FormulaUtils.logTokens('\xa0 resolving with parameters', params);
                    result = descriptor.resolve.apply(context, params);

                    // compare result with the specified return type
                    if (!isValueOfType(result, descriptor.type)) {
                        fatalError('InterpreterInstance.resolveOperator(): ' + descriptor.name + ': operator result type does not match operator return type');
                    }

                } catch (error) {
                    if (SheetUtils.isErrorCode(error)) {
                        // operator may throw literal error codes
                        result = error;
                    } else {
                        // re-throw other error codes, or internal JavaScript exceptions
                        throw error;
                    }
                }

                // check elapsed time, return with UNSUPPORTED_ERROR if formula is too complex
                // (but not if the formula logger is active, to allow debugging the interpreter)
                if (!FormulaUtils.isLoggingActive() && (_.now() - t0 > MAX_EVAL_TIME)) {
                    result = FormulaUtils.UNSUPPORTED_ERROR;
                }

                return result;
            }

            // resolve token to operator or function descriptor
            if (token.isType('op')) {
                descriptor = operatorDescriptors[token.getValue()];
            } else if (token.isType('func') && !token.hasSheetRef() && app.isNativeFunctionName(token.getValue())) {
                descriptor = functionDescriptors[token.getValue().toUpperCase()];
            }

            // unknown function (external/macro function calls)
            if (!descriptor) {
                return new Operand(context, ErrorCodes.NAME);
            }

            // check return type of the operator or function
            if (!/^(val|mat|ref|any)$/.test(descriptor.type)) {
                fatalError('InterpreterInstance.processOperator(): invalid return type: "' + descriptor.type + '"');
            }

            // check context type (will throw 'reference' error)
            checkContextType(descriptor.type, contextType);

            // get minimum number of parameters
            if (_.isNumber(descriptor.minParams) && (descriptor.minParams >= 0)) {
                minParams = descriptor.minParams;
            } else {
                fatalError('InterpreterInstance.processOperator(): invalid value for minParams: ' + descriptor.minParams);
            }

            // get maximum number of parameters, and length of repeated sequences
            if (!('maxParams' in descriptor)) {
                repeatParams = _.isNumber(descriptor.repeatParams) ? descriptor.repeatParams : 1;
                if ((repeatParams < 1) || (minParams < repeatParams)) {
                    fatalError('InterpreterInstance.processOperator(): invalid value for repeatParams: ' + descriptor.repeatParams);
                }
                maxParams = minParams + Utils.roundDown(FormulaUtils.MAX_PARAM_COUNT - minParams, repeatParams);
            } else if (_.isNumber(descriptor.maxParams) && (minParams <= descriptor.maxParams)) {
                maxParams = descriptor.maxParams;
                repeatParams = 1;
            } else {
                fatalError('InterpreterInstance.processOperator(): invalid value for maxParams: ' + descriptor.maxParams);
            }

            // check operand count
            if (compilerToken.params < minParams) { throw 'missing'; }
            if (compilerToken.params > maxParams) { throw 'unexpected'; }

            // repeated parameter sequences must be complete
            if (((compilerToken.params - minParams) % repeatParams) !== 0) { throw 'missing'; }

            // check missing implementation
            if (!_.isFunction(descriptor.resolve)) {
                FormulaUtils.warn('InterpreterInstance.processOperator(): unsupported operator or function "' + token + '"');
                return new Operand(context, FormulaUtils.UNSUPPORTED_ERROR);
            }

            // check operator signature
            if ((compilerToken.params > 0) && (descriptor.signature.length === 0)) {
                fatalError('InterpreterInstance.processOperator(): missing operator signature');
            }

            // build the complete type signature
            signature = _.times(compilerToken.params, function (index) {

                var // get operand type from signature in the descriptor
                    paramType = getParamType(index),
                    // the base parameter type used as context type for the parameter
                    baseType = paramType.split(':')[0];

                // return type signature entry
                return { paramType: paramType, baseType: baseType };
            });

            // build the array of operands
            operands = signature.map(function (typeData) {

                var // the resulting context type for the operand
                    opContextType = typeData.baseType;

                // if context type for this operator is matrix, and the current parameter is of type value,
                // and this operator returns a value, pass the matrix context type through to the operand
                // (resolve nested cell references to matrixes instead of single values)
                if ((contextType === 'mat') && (typeData.baseType === 'val') && (descriptor.type === 'val')) {
                    opContextType = 'mat';
                }

                return getNextOperand(opContextType);
            });

            // special handling for all operands of operators returning values
            if (descriptor.type === 'val') {
                operands.forEach(function (operand, index) {

                    // convert or preprocess operands, if operator expects a value type
                    if (signature[index].baseType === 'val') {

                        // If outer operator context is matrix, convert cell references to matrixes.
                        // Example: In the formula =MMULT(A1:B2;ABS(A1:B2)), the function MMULT passes
                        // context type matrix to its operands which causes ABS to resolve the cell
                        // reference as matrix instead of value.
                        if ((contextType === 'mat') && operand.isReference()) {
                            operand = operands[index] = new Operand(context, operand.getMatrix());
                        }

                        // Check for matrixes in value operands. Resolve operator repeatedly to a matrix,
                        // if it returns values. Example: The plus operator in the formula =SUM({1;2|3;4}+1)
                        // will convert the number 1 to the matrix {1;1|1;1}.
                        if (operand.isMatrix()) {
                            var matrix = operand.getMatrix();
                            repeatRows = Math.max(repeatRows, matrix.rows());
                            repeatCols = Math.max(repeatCols, matrix.cols());
                        }
                    }
                });
            }

            // extend the calling context for the operator resolver with the operands
            OperandsMixin.call(context, operands);

            // resolve as matrix, if any parameter with value signature type is a matrix
            if (repeatRows * repeatCols > 0) {

                // restrict to supported matrix size
                if (!isValidMatrixSize(repeatRows, repeatCols)) {
                    return new Operand(context, FormulaUtils.UNSUPPORTED_ERROR);
                }

                // build a matrix as result value
                FormulaUtils.info('\xa0 process as ' + repeatRows + 'x' + repeatCols + ' matrix');
                result = Matrix.generate(repeatRows, repeatCols, resolveOperator);

            } else {
                result = resolveOperator();
            }

            // return result as operand instance
            return (result instanceof Operand) ? result : new Operand(context, result);
        }

        /**
         * Returns the result of the next compiler token. If it is an operand,
         * it will be returned; if it is an operator or function call, its
         * result will be calculated and returned. If the operands of the
         * operator or functions are also operators or functions, their results
         * will be calculated recursively.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the next compiler token.
         */
        function getNextOperand(contextType) {

            var // the compiler token top be processed
                compilerToken = compilerTokens[tokenIndex],
                // the resulting operand
                result = null;

            // double-check the token (compiler should not create invalid token arrays)
            if (!compilerToken) {
                fatalError('InterpreterInstance.getNextOperand(): missing compiler token');
            }

            // move to next unprocessed token
            tokenIndex += 1;

            // process token by type
            switch (compilerToken.type) {
            case 'val':
                // return operands directly
                result = makeTokenOperand(compilerToken, contextType);
                break;
            case 'op':
                // process operators recursively
                result = processOperator(compilerToken, contextType);
                break;
            default:
                fatalError('InterpreterInstance.getNextOperand(): unknown token type: "' + compilerToken.type + '"');
            }

            FormulaUtils.log('< result of ' + compilerToken + ': ' + result);
            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Calculates the result of the formula represented by the token array
         * passed in the constructor.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the formula, as unresolved operand object.
         */
        this.getResult = function (contextType) {
            t0 = _.now();
            return getNextOperand(contextType);
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            app = docModel = compilerTokens = initOptions = null;
            context = operatorDescriptors = functionDescriptors = null;
        });

    }}); // class InterpreterInstance

    // class Interpreter ======================================================

    /**
     * Calculates the result for an array of token descriptors that has been
     * compiled to prefix notation (a.k.a. Polish Notation). See class Compiler
     * for details about compiled token arrays.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    function Interpreter(docModel) {

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Returns the internal map with descriptors for all implemented
         * functions, regardless of the file format of the edited document.
         *
         * @returns {Object}
         *  A map containing descriptors with all supported functions.
         */
        this.getRawFunctionDescriptors = function () {
            return DESCRIPTORS.FUNCTIONS;
        };

        /**
         * Returns a map with descriptors for all supported functions. Specific
         * settings in the original descriptors will already be resolved for
         * the file format of the edited document.
         *
         * @returns {Object}
         *  A map containing descriptors with all supported functions.
         */
        this.getFunctionDescriptors = function () {
            return resolveFunctionDescriptors(docModel.getApp().getFileFormat());
        };

        /**
         * Calculates the result of the formula represented by the passed
         * compiled token array.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. The following context types are supported:
         *  - 'val': A single value is expected, e.g. in simple cell formulas,
         *      or operators and functions working with simple values (plus
         *      operator, ABS).
         *  - 'mat': A constant matrix of values is expected, e.g. in matrix
         *      formulas, or functions working on entire matrixes (MMULT).
         *  - 'ref': An unresolved cell reference is expected, e.g. reference
         *      operators (list, intersection, range), or functions calculating
         *      with range addresses instead of the cell contents (OFFSET).
         *  - 'any': Accepts any of the types mentioned above, with minimum
         *      conversion, e.g. the result of defined names, functions passing
         *      one of their original parameters through (IF, CHOOSE), or
         *      functions that iterate over available values in matrixes and
         *      cell references (SUM, PRODUCT).
         *
         * @param {Array} compilerTokens
         *  An array of token descriptors, in prefix notation, as returned by
         *  the method Compiler.compileTokens().
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the reference sheet that will be used to resolve
         *      reference tokens and defined names without explicit sheet
         *      reference. If omitted, reference tokens without sheet indexes
         *      will result in #REF! errors.
         *  @param {Address} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Address} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated ranges that are located outside the
         *      sheet will be wrapped at the sheet borders.
         *
         * @returns {Object}
         *  The result descriptor of the formula interpreter, with the
         *  following properties:
         *  - {String} type
         *      The result type:
         *      - 'result': A valid result has been calculated. The property
         *          'value' contains the formula result.
         *      - 'warn': The formula result has not been calculated, but the
         *          formula structure is valid. The property 'value' specifies
         *          the warning: 'circular' for circular references, or
         *          'unsupported', if an unsupported feature was found in the
         *          formula, or 'internal', if any internal error has occurred.
         *      - 'error': The formula structure is invalid. The property
         *          'value' contains an identifier for the reason: 'missing',
         *          if something is missing, e.g. a function was called with
         *          less arguments than required; or 'unexpected' if something
         *          was not expected, e.g. a function was called with too many
         *          arguments.
         *  - {Any} value
         *      The result of the formula, or an error code, if calculation has
         *      failed (see property 'type' above).
         */
        this.interpretTokens = function (contextType, compilerTokens, options) {

            var // the actual interpreter instance, needed for recursive interpretation of e.g. defined names
                instance = new InterpreterInstance(docModel, compilerTokens, options),
                // the resulting operand
                result = null;

            try {

                // calculate the formula, pull formula result operand
                FormulaUtils.takeTime('Interpreter.interpretTokens(): contextType=' + contextType, function () {
                    result = instance.getResult(contextType);
                });

                // 'any' context: resolve to actual type of the resulting operand
                contextType = (contextType === 'any') ? result.getType() : contextType;

                // resolve result operand to result value according to context type
                switch (contextType) {
                case 'val':
                    result = result.getValue();
                    // in value context, empty values (reference to empty cell) become zero
                    if (_.isNull(result)) { result = 0; }
                    break;
                case 'mat':
                    result = result.getMatrix();
                    break;
                case 'ref':
                    result = result.getReference();
                    break;
                default:
                    fatalError('FormulaInterpreter.interpretTokens(): unknown context type: "' + contextType + '"');
                }

                // throw error code literal (special handling for internal errors in catch clause)
                if (SheetUtils.isErrorCode(result)) { throw result; }

                // convert to valid result object
                result = { type: 'result', value: result };

            } catch (error) {

                // operator may throw literal error codes
                if (SheetUtils.isErrorCode(error)) {

                    // 'unsupported' state or circular references: return empty result with warning property
                    result = _.isString(error.internal) ?
                        { type: 'warn', value: error.internal } :
                        { type: 'result', value: error };

                // 'fatal' state for internal runtime errors: do not return error (this would state an
                // invalid formula structure), but return empty result
                } else if (error === 'fatal') {
                    result = { type: 'warn', value: 'internal' };

                // other internal errors (must be strings): invalid formula structure
                } else if (_.isString(error)) {
                    FormulaUtils.warn('\xa0 error: ' + error);
                    result = { type: 'error', value: error };

                // re-throw internal JS exceptions
                } else {
                    throw error;
                }

            } finally {
                instance.destroy();
            }

            FormulaUtils.info('result: type=' + result.type + ', value=' + FormulaUtils.valueToString(result.value));
            return result;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class Interpreter

    // exports ================================================================

    return modulePromise.then(function () {
        // derive this class from class ModelObject
        return ModelObject.extend({ constructor: Interpreter });
    });

});
