/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/statisticalfuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, SheetUtils, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all statistical spreadsheet functions.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes,

        // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
        SKIP_MAT_SKIP_REF = { // id: CSS5
            valMode: 'convert', // value operands: convert strings and Booleans to numbers
            matMode: 'skip', // matrix operands: skip strings and Booleans
            refMode: 'skip', // reference operands: skip strings and Booleans
            emptyParam: true, // empty parameters count as zero
            complexRef: true // accept multi-range and multi-sheet references
        },

        // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
        SKIP_MAT_ZERO_REF = { // id: CSZ5
            valMode: 'convert', // value operands: convert strings and Booleans to numbers
            matMode: 'skip', // matrix operands: skip strings and Booleans
            refMode: 'zero', // reference operands: replace all strings with zeros
            emptyParam: true, // empty parameters count as zero
            complexRef: true // accept multi-range and multi-sheet references
        },

        // standard options for numeric parameter iterators (method FormulaContext.iterateNumbers())
        ZERO_MAT_ZERO_REF = { // id: CZZ5
            valMode: 'convert', // value operands: convert strings and Booleans to numbers
            matMode: 'zero', // matrix operands: replace all strings with zeros
            refMode: 'zero', // reference operands: replace all strings with zeros
            emptyParam: true, // empty parameters count as zero
            complexRef: true // accept multi-range and multi-sheet references
        };

    // private global functions ===============================================

    // numeric aggregation ----------------------------------------------------

    /**
     * Creates and returns a resolver function that reduces all numbers of all
     * function parameters to their minimum or maximum.
     *
     * @param {String} method
     *  The aggregation method to be implemented, either 'min' or 'max'.
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all numbers of all its operands to the aggregation callback function.
     */
    function implementMinMaxAggregation(method, iteratorOptions) {

        var // start with signed infinity according to the function type
            initial = (method === 'max') ? Number.NEGATIVE_INFINITY : Number.POSITIVE_INFINITY,
            // the aggregation function (Math.min or Math.max)
            aggregate = _.bind(Math[method], Math);

        // default result (no numbers found in parameters) is zero (no #NUM! error code from infinity)
        function finalize(result) { return isFinite(result) ? result : 0; }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return FormulaUtils.implementNumericAggregation(initial, aggregate, finalize, iteratorOptions);
    }

    /**
     * Creates and returns a resolver function that reduces all numbers of all
     * function parameters to their standard deviation, or their variance.
     *
     * @param {String} method
     *  The aggregation method to be implemented, either 'dev' for the standard
     *  deviation, or 'var' for the variance.
     *
     * @param {Boolean} population
     *  If set to true, calculates the standard deviation or variance based on
     *  the entire population. If set to false, calculates the result based on
     *  a sample.
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally.
     *
     * @returns {Function}
     *  The resulting function implementation.
     */
    function implementStdDevAggregation(method, population, iteratorOptions) {

        // default result (no numbers found in parameters) is the #DIV/0! error code thrown by divide()
        function finalize(numbers, sum) {

            var mean = FormulaUtils.divide(sum, numbers.length),
                result = Utils.getSum(numbers, function (number) {
                    var diff = number - mean;
                    return diff * diff;
                }),
                size = population ? numbers.length : (numbers.length - 1);

            result = FormulaUtils.divide(result, size);
            return (method === 'dev') ? Math.sqrt(result) : result;
        }

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return FormulaUtils.implementNumericAggregationWithArray(finalize, iteratorOptions);
    }

    // exports ================================================================

    return {

        AVEDEV: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCodes.NUM; }
                    var mean = sum / numbers.length;
                    return Utils.getSum(numbers, function (number) { return Math.abs(number - mean); }) / numbers.length;
                }

                // empty parameters count as zero: AVEDEV(1,) = 0.5
                return FormulaUtils.implementNumericAggregationWithArray(finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        AVERAGE: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: AVERAGE(1,) = 0.5
            // default result (no numbers found in parameters) is the #DIV/0! error code thrown by FormulaUtils.divide()
            resolve: FormulaUtils.implementNumericAggregation(0, FormulaUtils.add, FormulaUtils.divide, SKIP_MAT_SKIP_REF)
        },

        AVERAGEA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: AVERAGEA(1,) = 0.5
            // default result (no numbers found in parameters) is the #DIV/0! error code thrown by FormulaUtils.divide()
            resolve: FormulaUtils.implementNumericAggregation(0, FormulaUtils.add, FormulaUtils.divide, ZERO_MAT_ZERO_REF)
        },

        AVERAGEIF: {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'ref val:any ref',
            resolve: FormulaUtils.implementFilterAggregation(0, FormulaUtils.add, FormulaUtils.divide)
        },

        AVERAGEIFS: {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 3,
            repeatParams: 2,
            type: 'val',
            signature: 'ref ref val:any'
        },

        B: {
            category: 'statistical',
            supported: 'odf', // OOXML name: BINOM.DIST.RANGE (new in XL2013)
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        BETADIST: {
            category: 'statistical',
            altNames: { ooxml: 'BETA.DIST' },
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        BETAINV: {
            category: 'statistical',
            altNames: { ooxml: 'BETA.INV' },
            minParams: 3,
            maxParams: 5,
            type: 'val'
        },

        BINOMDIST: {
            category: 'statistical',
            altNames: { ooxml: 'BINOM.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        'BINOM.DIST.RANGE': {
            category: 'statistical',
            supported: 'ooxml', // ODF name: B
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        CHIDIST: {
            category: 'statistical',
            altNames: { ooxml: 'CHISQ.DIST.RT' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHIINV: {
            category: 'statistical',
            altNames: { ooxml: 'CHISQ.INV.RT' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHISQDIST: {
            category: 'statistical',
            supported: 'odf', // OOXML name: CHISQ.DIST (new in XL2010)
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        'CHISQ.DIST': {
            category: 'statistical',
            supported: 'ooxml', // ODF name: CHISQDIST
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'CHISQ.DIST.RT': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHISQINV: {
            category: 'statistical',
            supported: 'odf', // OOXML name: CHISQ.INV (new in XL2010)
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'CHISQ.INV': {
            category: 'statistical',
            supported: 'ooxml', // ODF name: CHISQINV
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'CHISQ.INV.RT': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CHITEST: {
            category: 'statistical',
            altNames: { ooxml: 'CHISQ.TEST' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CONFIDENCE: {
            category: 'statistical',
            altNames: { ooxml: 'CONFIDENCE.NORM' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'CONFIDENCE.T': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        CORREL: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        COUNT: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    if (operand.isValue()) {
                        // count Booleans and strings that are convertible to numbers
                        // (simple values only, not counted in matrixes and references)
                        try {
                            this.convertToNumber(operand.getValue());
                            count += 1;
                        } catch (ex) {}
                    } else {
                        // matrixes and references: count real numbers only (no strings, no Booleans)
                        this.iterateValues(operand, function (value) {
                            if (_.isNumber(value)) { count += 1; }
                        }, {
                            acceptErrors: true, // do not fail on error codes: =COUNT({1,#VALUE!}) = 1
                            complexRef: true // accept multi-range and multi-sheet references
                        });
                    }
                });
                return count;
            }
        },

        COUNTA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function () {
                var count = 0;
                this.iterateOperands(0, function (operand) {
                    switch (operand.getType()) {
                    case 'val':
                        // all values are counted (also zeros, empty strings, FALSE,
                        // error codes, empty parameters): COUNTA(1,,#VALUE!) = 3
                        count += 1;
                        break;
                    case 'mat':
                        // all matrix elements are counted (also zeros, empty strings, FALSE, error codes)
                        count += operand.getMatrix().size();
                        break;
                    case 'ref':
                        // all filled cells are counted (also error codes)
                        this.iterateValues(operand, function () { count += 1; }, {
                            acceptErrors: true, // count all error codes
                            complexRef: true // accept multi-range and multi-sheet references
                        });
                    }
                });
                return count;
            }
        },

        COUNTBLANK: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ranges) {

                var // number of non-blank cells
                    count = 0,
                    // whether empty strings (formula results) are blank (Excel only)
                    emptyStr = this.isOOXML();

                // count all non-blank cells in the reference
                this.iterateValues(ranges, function (value) {
                    if (!emptyStr || (value !== '')) { count += 1; }
                }, {
                    acceptErrors: true, // count all error codes
                    complexRef: false // do not accept multi-range and multi-sheet references
                });

                // return number of remaining blank cells
                return ranges.cells() - count;
            }
        },

        COUNTIF: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'ref val:any',
            // COUNTIF returns the count of all matching cells in the source range (last parameter of the finalizer)
            resolve: FormulaUtils.implementFilterAggregation(0, _.identity, function (result, count1, count2) { return count2; })
        },

        COUNTIFS: {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            repeatParams: 2,
            type: 'val'
        },

        COVAR: {
            category: 'statistical',
            altNames: { ooxml: 'COVARIANCE.P' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'COVARIANCE.S': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        CRITBINOM: {
            category: 'statistical',
            altNames: { ooxml: 'BINOM.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DEVSQ: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code (not #DIV/0!)
                function finalize(numbers, sum) {
                    if (numbers.length === 0) { throw ErrorCodes.NUM; }
                    var mean = sum / numbers.length;
                    return Utils.getSum(numbers, function (number) {
                        var diff = number - mean;
                        return diff * diff;
                    });
                }

                // empty parameters count as zero: DEVSQ(1,) = 0.5
                return FormulaUtils.implementNumericAggregationWithArray(finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        EXPONDIST: {
            category: 'statistical',
            altNames: { ooxml: 'EXPON.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FDIST: {
            category: 'statistical',
            altNames: { ooxml: 'F.DIST.RT' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'F.DIST': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        FINV: {
            category: 'statistical',
            altNames: { ooxml: 'F.INV.RT' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'F.INV': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FISHER: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FISHERINV: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FORECAST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        FREQUENCY: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'mat'
        },

        FTEST: {
            category: 'statistical',
            altNames: { ooxml: 'F.TEST' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        GAMMA: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GAMMADIST: {
            category: 'statistical',
            altNames: { ooxml: 'GAMMA.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        GAMMAINV: {
            category: 'statistical',
            altNames: { ooxml: 'GAMMA.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        GAMMALN: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        'GAMMALN.PRECISE': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GAUSS: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GEOMEAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code
                // a single 0 in the parameters results in the #NUM! error code
                function finalize(result, count) {
                    if ((result === 0) || (count === 0)) { throw ErrorCodes.NUM; }
                    return Math.pow(result, 1 / count);
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return FormulaUtils.implementNumericAggregation(1, FormulaUtils.multiply, finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        GROWTH: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        HARMEAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // adds reciprocal of the passed number to the intermediate result
                function aggregate(result, number) {
                    // a single 0 in the parameters results in the #NUM! error code
                    if (number === 0) { throw ErrorCodes.NUM; }
                    return result + 1 / number;
                }

                // default result (no numbers found in parameters) is the #N/A error code
                function finalize(result, count) {
                    if (count === 0) { throw ErrorCodes.NA; }
                    return count / result;
                }

                // empty parameters count as zero: GEOMEAN(1,) = #NUM!
                return FormulaUtils.implementNumericAggregation(0, aggregate, finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        HYPGEOMDIST: {
            category: 'statistical',
            altNames: { ooxml: 'HYPGEOM.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        INTERCEPT: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        KURT: {
            category: 'statistical',
            minParams: 1,
            type: 'val'
        },

        LARGE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        LINEST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        LOGEST: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        LOGINV: {
            category: 'statistical',
            altNames: { ooxml: 'LOGNORM.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        LOGNORMDIST: {
            category: 'statistical',
            altNames: { ooxml: 'LOGNORM.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        MAX: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MAX(-1,) = 0
            resolve: implementMinMaxAggregation('max', SKIP_MAT_SKIP_REF)
        },

        MAXA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MAXA(-1,) = 0
            resolve: implementMinMaxAggregation('max', SKIP_MAT_ZERO_REF)
        },

        MEDIAN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // default result (no numbers found in parameters) is the #NUM! error code
                function finalize(numbers) {
                    var count = numbers.length;
                    if (count === 0) { throw ErrorCodes.NUM; }
                    numbers = _.sortBy(numbers);
                    // even array length: return arithmetic mean of both numbers in the middle of the array
                    return (count % 2 === 0) ? ((numbers[count / 2 - 1] + numbers[count / 2]) / 2) : numbers[(count - 1) / 2];
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return FormulaUtils.implementNumericAggregationWithArray(finalize, SKIP_MAT_SKIP_REF);
            }())
        },

        MIN: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MIN(1,) = 0
            resolve: implementMinMaxAggregation('min', SKIP_MAT_SKIP_REF)
        },

        MINA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as zero: MINA(1,) = 0
            resolve: implementMinMaxAggregation('min', SKIP_MAT_ZERO_REF)
        },

        MODE: {
            category: 'statistical',
            altNames: { ooxml: 'MODE.SNGL' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            resolve: (function () {

                // finds the most used number in all collected numbers
                function finalize (numbers) {

                    // default result (all parameters skipped) is the #N/A error code
                    if (numbers.length === 0) { throw ErrorCodes.NA; }

                    var // reduce numbers to their counts (maps each distinct number to its count)
                        counts = _.countBy(numbers),
                        // find the highest count
                        maxCount = _.reduce(counts, function (max, num) { return Math.max(max, num); }, 0),
                        // a map that contains all numbers occuring the most as key
                        map = {};

                    // no duplicate numbers: throw the #N/A error code
                    if (maxCount === 1) { throw ErrorCodes.NA; }

                    // insert each number into the map that occurs the most
                    _.each(counts, function (count, number) {
                        if (count === maxCount) { map[number] = true; }
                    });

                    // find the first number in the original array that occurs the most
                    // e.g.: =MODE(3,3,4,4) results in 3; but: =MODE(4,4,3,3) results in 4
                    return _.find(numbers, function (number) { return number in map; });
                }

                // create and return the resolver function to be assigned to the 'resolve' option of a function descriptor
                return FormulaUtils.implementNumericAggregationWithArray(finalize, {
                    valMode: 'exact', // value operands: exact match for numbers (no strings, no Booleans)
                    matMode: 'skip', // matrix operands: skip strings and Booleans
                    refMode: 'exact', // reference operands: exact match for numbers (no strings, no Booleans)
                    emptyParam: true, // empty parameters result in #VALUE! error
                    complexRef: false // do not accept multi-range and multi-sheet references
                });
            }())
        },

        'MODE.MULT': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 1,
            type: 'val'
        },

        NEGBINOMDIST: {
            category: 'statistical',
            altNames: { ooxml: 'NEGBINOM.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        NORMDIST: {
            category: 'statistical',
            altNames: { ooxml: 'NORM.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        NORMINV: {
            category: 'statistical',
            altNames: { ooxml: 'NORM.INV' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        NORMSDIST: {
            category: 'statistical',
            altNames: { ooxml: 'NORM.S.DIST' },
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        NORMSINV: {
            category: 'statistical',
            altNames: { ooxml: 'NORM.S.INV' },
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        PEARSON: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERCENTILE: {
            category: 'statistical',
            altNames: { ooxml: 'PERCENTILE.INC' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'PERCENTILE.EXC': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERCENTRANK: {
            category: 'statistical',
            altNames: { ooxml: 'PERCENTRANK.INC' },
            minParams: 2,
            maxParams: { ooxml: 3, odf: 2 },
            type: 'val'
        },

        'PERCENTRANK.EXC': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        PERMUT: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PERMUTATIONA: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        PHI: {
            category: 'statistical',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        POISSON: {
            category: 'statistical',
            altNames: { ooxml: 'POISSON.DIST' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        PROB: {
            category: 'statistical',
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        QUARTILE: {
            category: 'statistical',
            altNames: { ooxml: 'QUARTILE.INC' },
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'QUARTILE.EXC': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        RANK: {
            category: 'statistical',
            altNames: { ooxml: 'RANK.EQ' },
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        'RANK.AVG': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        RSQ: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SKEW: {
            category: 'statistical',
            minParams: 1,
            type: 'val'
        },

        'SKEW.P': {
            category: 'statistical',
            supported: 'ooxml', // ODF name: SKEWP
            minParams: 1,
            type: 'val'
        },

        SKEWP: {
            category: 'statistical',
            supported: 'odf', // OOXML name: SKEW.P
            minParams: 1,
            type: 'val'
        },

        SLOPE: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SMALL: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        STANDARDIZE: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        STDEV: {
            category: 'statistical',
            altNames: { ooxml: 'STDEV.S' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEV(1,) = 0.707
            resolve: implementStdDevAggregation('dev', false, SKIP_MAT_SKIP_REF)
        },

        STDEVA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVA(1,) = 0.707
            resolve: implementStdDevAggregation('dev', false, SKIP_MAT_ZERO_REF)
        },

        STDEVP: {
            category: 'statistical',
            altNames: { ooxml: 'STDEV.P' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVP(1,) = 0.5
            resolve: implementStdDevAggregation('dev', true, SKIP_MAT_SKIP_REF)
        },

        STDEVPA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: STDEVPA(1,) = 0.5
            resolve: implementStdDevAggregation('dev', true, SKIP_MAT_ZERO_REF)
        },

        STEYX: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TDIST: {
            category: 'statistical',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'T.DIST': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'T.DIST.2T': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.DIST.RT': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TINV: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.INV': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        'T.INV.2T': {
            category: 'statistical',
            supported: 'ooxml',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TREND: {
            category: 'statistical',
            minParams: 1,
            maxParams: 4,
            type: 'mat'
        },

        TRIMMEAN: {
            category: 'statistical',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TTEST: {
            category: 'statistical',
            altNames: { ooxml: 'T.TEST' },
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        VAR: {
            category: 'statistical',
            altNames: { ooxml: 'VAR.S' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VAR(1,) = 0.5
            resolve: implementStdDevAggregation('var', false, SKIP_MAT_SKIP_REF)
        },

        VARA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARA(1,) = 0.5
            resolve: implementStdDevAggregation('var', false, SKIP_MAT_ZERO_REF)
        },

        VARP: {
            category: 'statistical',
            altNames: { ooxml: 'VAR.P' },
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARP(1,) = 0.25
            resolve: implementStdDevAggregation('var', true, SKIP_MAT_SKIP_REF)
        },

        VARPA: {
            category: 'statistical',
            minParams: 1,
            type: 'val',
            signature: 'any',
            // empty parameters count as 0: VARPA(1,) = 0.25
            resolve: implementStdDevAggregation('var', true, SKIP_MAT_ZERO_REF)
        },

        WEIBULL: {
            category: 'statistical',
            altNames: { ooxml: 'WEIBULL.DIST' },
            minParams: 4,
            maxParams: 4,
            type: 'val'
        },

        ZTEST: {
            category: 'statistical',
            altNames: { ooxml: 'Z.TEST' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        }
    };

});
