/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/matrixfuncs', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/matrix'
], function (SheetUtils, FormulaUtils, Matrix) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with matrixes.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, matrixes can be given as literals in formulas,
     * or can be extracted from rectangular cell ranges. The client-side
     * formula engine provides the function signature type 'mat' (and its sub
     * types such as 'mat:num'), and the class Matrix whose instances contain
     * the elements of a matrix in a two-dimensional array.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // exports ================================================================

    return {

        MDETERM: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat'
        },

        MINVERSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat'
        },

        MMULT: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'mat',
            signature: 'mat:num mat:num',
            resolve: function (matrix1, matrix2) {

                // width of first matrix must match height of second matrix
                if (matrix1.cols() !== matrix2.rows()) { throw ErrorCodes.VALUE; }

                // calculate all elements of the result matrix
                return Matrix.generate(matrix1.rows(), matrix2.cols(), function (row, col) {
                    var elem = 0;
                    for (var i = 0, l = matrix1.cols(); i < l; i += 1) {
                        elem += (matrix1.get(row, i) * matrix2.get(i, col));
                    }
                    return elem;
                });
            }
        },

        MUNIT: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'val:int',
            resolve: function (dim) {
                if (dim < 1) { throw ErrorCodes.VALUE; }
                if (dim > Math.min(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT)) { throw FormulaUtils.UNSUPPORTED_ERROR; }
                return Matrix.generate(dim, dim, function (row, col) {
                    return (row === col) ? 1 : 0;
                });
            }
        },

        SUMX2MY2: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SUMX2PY2: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        SUMXMY2: {
            category: 'matrix',
            minParams: 2,
            maxParams: 2,
            type: 'val'
        },

        TRANSPOSE: {
            category: 'matrix',
            minParams: 1,
            maxParams: 1,
            type: 'mat',
            signature: 'mat:any',
            resolve: function (matrix) {
                return Matrix.generate(matrix.cols(), matrix.rows(), function (row, col) {
                    return matrix.get(col, row);
                });
            }
        }
    };

});
