/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

(function () {

    'use strict';

    // constants ==============================================================

    /**
     * Tolerance for 'almost' assertions.
     *
     * @constant
     */
    chai.EPSILON = 1e-14;

    /**
     * Placeholder for the iterator tester. See function chai.runIterator() for
     * details.
     *
     * @constant
     */
    chai.ITERATOR = {};

    // properties =============================================================

    /**
     * A property asserting whether the actual value is zero, or almost zero.
     * Precisely, returns whether the absolute value is less than chai.EPSILON.
     *
     * @example
     *  expect(Math.sin(Math.PI)).to.be.almostZero;
     */
    chai.Assertion.addProperty('almostZero', function () {

        var value = this._obj,
            passed = Math.abs(value) < chai.EPSILON;

        this.assert(passed, 'expected #{act} to be almost zero', 'expected #{act} to not be almost zero', value);
    });

    // methods ================================================================

    /**
     * A method asserting whether the actual number and the specified number
     * are almost equal, with a relative error less than chai.EPSILON.
     *
     * @example
     *  expect(Math.sin(Math.PI/2)).to.almostEqual(1);
     *
     * @param {Number} expected
     *  The expected number that will be compared with the actual number.
     */
    chai.Assertion.addMethod('almostEqual', function (expected) {

        var value = this._obj,
            passed = Math.abs((value - expected) / expected) < chai.EPSILON;

        this.assert(passed, 'expected #{act} to almost equal #{exp}', 'expected #{act} to not almost equal #{exp}', expected, value);
    });

    // static functions =======================================================

    /**
     * A helper function that executes the passed function which expects and
     * invokes an iterator callback function, and collects information about
     * every invocation of the iterator function.
     *
     * @param {Function} callback
     *  The function to be tested by this helper function.
     *
     * @param {Object|Null} context
     *  The context object bound to the tested function.
     *
     * @param {Array} args
     *  The arguments to be passed to the tested function. The placeholder
     *  object chai.ITERATOR must be inserted at the position where the tested
     *  function expects the iterator callback function.
     *
     * @param {Function} [breaker]
     *  If specified, a callback function that will be executed on every
     *  invocation of the iterator callback function, and that allows to return
     *  a custom value. This can be used to break the running iteration loop at
     *  any specific step. Receives all arguments received by the iterator
     *  function. Its return value will be the result of the iterator function.
     *
     * @returns {Object}
     *  A result object with the following properties:
     *  - {Array} runs
     *      An array containing descriptor objects for all invocations of the
     *      iterator function. The length of this array represents the number
     *      of iterator invocations. Each object in the array contains the
     *      following properties:
     *      - {Any} context
     *          The context bound to the current invocation of the iterator
     *          callback function.
     *      - {Array} args
     *          The arguments passed to the current invocation of the iterator
     *          callback function, converted to a plain JS array.
     *      - {Any} ret
     *          The return value of the current invocation of the iterator
     *          callback function.
     *  - {Array} args
     *      For convenience, all arguments of all invocations of the iterator
     *      callback function, as array of arrays (all 'args' properties from
     *      all descriptor objects contained in the 'runs' property).
     *  - {Any} ret
     *      The final return value of the iterating method.
     */
    chai.runIterator = function (callback, context, args, breaker) {

        var // the result object to be returned
            result = { runs: [] };

        // the iterator callback function injected into the method to be tested
        function iterator() {
            result.runs.push({ context: this, args: _.toArray(arguments) });
            if (breaker) { return breaker.apply(this, arguments); }
        }

        // replace the ITERATOR placeholder in the passed arguments with the actual iterator callback function
        args = _(args).map(function (arg) { return (arg === chai.ITERATOR) ? iterator : arg; });

        // execute the method to be tested; the iterator function will collect all invocations
        result.ret = callback.apply(context, args);

        // pluck all invocation arguments into the 'args' property
        result.args = _.pluck(result.runs, 'args');

        return result;
    };

    // ========================================================================
}());
