/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define(['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    describe('OX Documents Base Framework', function () {
        describe('Utils module', function () {

            it('should exist', function () {
                expect(Utils).to.be.an('object');
            });

            // constants ======================================================

            describe('constant "ELLIPSIS_CHAR"', function () {
                it('should exist', function () {
                    expect(Utils.ELLIPSIS_CHAR).to.equal('\u2026');
                });
            });

            describe('constant "BREAK"', function () {
                it('should exist', function () {
                    expect(Utils.BREAK).to.exist;
                });
            });

            describe('constant "NOOP"', function () {
                it('should exist', function () {
                    expect(Utils.NOOP).to.be.a('function');
                });
                it('should return calling context', function () {
                    var ctxt = {};
                    expect(Utils.NOOP.call(ctxt)).to.equal(ctxt);
                });
            });

            describe('constant "SCHEME_COLOR_COUNT"', function () {
                it('should exist', function () {
                    expect(Utils.SCHEME_COLOR_COUNT).to.be.a('number');
                    expect(Utils.SCHEME_COLOR_COUNT).to.be.above(0);
                });
            });

            describe('constant "RETINA"', function () {
                it('should exist', function () {
                    expect(Utils.RETINA).to.be.a('boolean');
                });
            });

            describe('constant "IE9"', function () {
                it('should exist', function () {
                    expect(Utils.IE9).to.be.a('boolean');
                });
            });

            describe('constant "IPAD"', function () {
                it('should exist', function () {
                    expect(Utils.IPAD).to.be.a('boolean');
                });
            });

            describe('constant "SLOW_DEVICE"', function () {
                it('should exist', function () {
                    expect(Utils.SLOW_DEVICE).to.be.a('boolean');
                });
            });

            describe('constant "MAX_NODE_SIZE"', function () {
                it('should exist', function () {
                    expect(Utils.MAX_NODE_SIZE).to.be.a('number');
                });
                it('should be positive', function () {
                    expect(Utils.MAX_NODE_SIZE).to.be.above(0);
                });
            });

            describe('constant "MAX_CONTAINER_WIDTH"', function () {
                it('should exist', function () {
                    expect(Utils.MAX_CONTAINER_WIDTH).to.be.a('number');
                });
                it('should be equal to or greater than Utils.MAX_NODE_SIZE', function () {
                    expect(Utils.MAX_CONTAINER_WIDTH).to.be.at.least(Utils.MAX_NODE_SIZE);
                });
            });

            describe('constant "MAX_CONTAINER_HEIGHT"', function () {
                it('should exist', function () {
                    expect(Utils.MAX_CONTAINER_HEIGHT).to.be.a('number');
                });
                it('should be equal to or greater than Utils.MAX_NODE_SIZE', function () {
                    expect(Utils.MAX_CONTAINER_HEIGHT).to.be.at.least(Utils.MAX_NODE_SIZE);
                });
            });

            // methods ========================================================

            describe('method "makeSimpleObject"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('makeSimpleObject');
                });
                it('should create an object with a single property', function () {
                    expect(Utils.makeSimpleObject('key', 'value')).to.deep.equal({ key: 'value' });
                    expect(Utils.makeSimpleObject('key', 1)).to.deep.equal({ key: 1 });
                    expect(Utils.makeSimpleObject('key', false)).to.deep.equal({ key: false });
                    expect(Utils.makeSimpleObject('key', null)).to.deep.equal({ key: null });
                    expect(Utils.makeSimpleObject('key', [1, 2])).to.deep.equal({ key: [1, 2] });
                    expect(Utils.makeSimpleObject('key', { a: 1 })).to.deep.equal({ key: { a: 1 } });
                    expect(Utils.makeSimpleObject('', 1)).to.deep.equal({ '': 1 });
                });
            });

            // ----------------------------------------------------------------

            describe('method "hasProperty"', function () {

                var object = { abc: 1, bcd: 2 };

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('hasProperty');
                });
                it('should return true for existing properties', function () {
                    expect(Utils.hasProperty(object, 'abc')).to.be['true'];
                    expect(Utils.hasProperty(object, 'bcd')).to.be['true'];
                });
                it('should return false for missing properties', function () {
                    expect(Utils.hasProperty(object, 'abcd')).to.be['false'];
                    expect(Utils.hasProperty(object, 'Abc')).to.be['false'];
                });
                it('should return true for existing properties selected by RE', function () {
                    expect(Utils.hasProperty(object, /^a/)).to.be['true'];
                    expect(Utils.hasProperty(object, /d$/)).to.be['true'];
                    expect(Utils.hasProperty(object, /b|x/)).to.be['true'];
                });
                it('should return false for missing properties selected by RE', function () {
                    expect(Utils.hasProperty(object, /a$/)).to.be['false'];
                    expect(Utils.hasProperty(object, /^d/)).to.be['false'];
                    expect(Utils.hasProperty(object, /abc./)).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "hasAnyProperty"', function () {

                var object = { abc: 1, bcd: 2 };

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('hasAnyProperty');
                });
                it('should return true for existing properties', function () {
                    expect(Utils.hasAnyProperty(object, ['abc', 'bcd', 'cde'])).to.be['true'];
                });
                it('should return false for missing properties', function () {
                    expect(Utils.hasAnyProperty(object, ['abcd', 'Abc'])).to.be['false'];
                });
                it('should accept an empty array', function () {
                    expect(Utils.hasAnyProperty(object, [])).to.be['false'];
                    expect(Utils.hasAnyProperty({}, [])).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "hasAllProperties"', function () {

                var object = { abc: 1, bcd: 2 };

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('hasAllProperties');
                });
                it('should return true for existing properties', function () {
                    expect(Utils.hasAllProperties(object, ['abc'])).to.be['true'];
                    expect(Utils.hasAllProperties(object, ['bcd'])).to.be['true'];
                    expect(Utils.hasAllProperties(object, ['abc', 'bcd'])).to.be['true'];
                });
                it('should return false for missing properties', function () {
                    expect(Utils.hasAllProperties(object, ['abc', 'bcd', 'abcd'])).to.be['false'];
                });
                it('should accept an empty array', function () {
                    expect(Utils.hasAllProperties(object, [])).to.be['true'];
                    expect(Utils.hasAllProperties({}, [])).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "hasEqualProperties"', function () {

                var object1 = { abc: 1, bcd: [2, 3], cde: 4, def: 5 },
                    object2 = { abc: 1, bcd: [2, 3], cde: 5 };

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('hasEqualProperties');
                });
                it('should return true for equal property values', function () {
                    expect(Utils.hasEqualProperties(object1, object2, ['abc'])).to.be['true'];
                    expect(Utils.hasEqualProperties(object1, object2, ['bcd'])).to.be['true'];
                    expect(Utils.hasEqualProperties(object1, object2, ['abc', 'bcd'])).to.be['true'];
                });
                it('should return true for missing properties', function () {
                    expect(Utils.hasEqualProperties(object1, object2, ['xyz'])).to.be['true'];
                    expect(Utils.hasEqualProperties(object1, object2, ['abc', 'xyz'])).to.be['true'];
                });
                it('should return false for different property values', function () {
                    expect(Utils.hasEqualProperties(object1, object2, ['cde'])).to.be['false'];
                    expect(Utils.hasEqualProperties(object1, object2, ['def'])).to.be['false'];
                    expect(Utils.hasEqualProperties(object1, object2, ['abc', 'cde'])).to.be['false'];
                    expect(Utils.hasEqualProperties(object1, object2, ['abc', 'def'])).to.be['false'];
                });
                it('should accept an empty array', function () {
                    expect(Utils.hasEqualProperties(object1, object2, [])).to.be['true'];
                    expect(Utils.hasEqualProperties({}, object2, [])).to.be['true'];
                });
                it('should use the comparator function', function () {
                    function comp(v1, v2) { return v1 === v2; }
                    expect(Utils.hasEqualProperties(object1, object2, ['abc'], comp)).to.be['true'];
                    expect(Utils.hasEqualProperties(object1, object2, ['bcd'], comp)).to.be['false'];
                    expect(Utils.hasEqualProperties(object1, object2, ['cde'], comp)).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "equalArrays"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('equalArrays');
                });
                it('should recognize references to the same array', function () {
                    var array = [1, 2];
                    expect(Utils.equalArrays(array, array)).to.be['true'];
                });
                it('should recognize arrays with equal elements', function () {
                    var obj = {};
                    expect(Utils.equalArrays([1, obj], [1, obj])).to.be['true'];
                });
                it('should recognize arrays with unequal elements', function () {
                    expect(Utils.equalArrays([1, 2], [1])).to.be['false'];
                    expect(Utils.equalArrays([1, 2], [1, 3])).to.be['false'];
                });
                it('should not compare arrays deeply', function () {
                    expect(Utils.equalArrays([1, [2, 3]], [1, [2, 3]])).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "compareNumberArrays"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('compareNumberArrays');
                });
                it('should recognize equal arrays', function () {
                    expect(Utils.compareNumberArrays([1], [1])).to.equal(0);
                    expect(Utils.compareNumberArrays([1, 2, 3, 4], [1, 2, 3, 4])).to.equal(0);
                });
                it('should recognize different arrays', function () {
                    expect(Utils.compareNumberArrays([1], [2])).to.be.below(0);
                    expect(Utils.compareNumberArrays([2], [1])).to.be.above(0);
                    expect(Utils.compareNumberArrays([1, 2, 3, 4], [1, 2, 4, 3])).to.be.below(0);
                    expect(Utils.compareNumberArrays([1, 2, 4, 3], [1, 2, 3, 4])).to.be.above(0);
                });
                it('should compare arrays of different size', function () {
                    expect(Utils.compareNumberArrays([1, 2, 3], [1, 2, 3, 4])).to.be.below(0);
                    expect(Utils.compareNumberArrays([1, 2, 3, 4], [1, 2, 3])).to.be.above(0);
                });
                it('should use specified maximum length', function () {
                    expect(Utils.compareNumberArrays([1, 2, 3], [1, 2, 3, 4], 3)).to.equal(0);
                    expect(Utils.compareNumberArrays([1, 2, 3, 4], [1, 2, 4, 3], 2)).to.equal(0);
                    expect(Utils.compareNumberArrays([1, 2, 3, 4], [2, 1, 3, 4], 0)).to.equal(0);
                });
                it('should accept empty arrays', function () {
                    expect(Utils.compareNumberArrays([], [])).to.equal(0);
                    expect(Utils.compareNumberArrays([], [1])).to.be.below(0);
                    expect(Utils.compareNumberArrays([1], [])).to.be.above(0);
                });
            });

            // ----------------------------------------------------------------

            describe('method "spliceValue"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('spliceValue');
                });
                it('should remove all existing values in-place and return their count', function () {
                    var array = [1, 2, 3, 4, 5, 4, 3, 2, 1];
                    expect(Utils.spliceValue(array, 3)).to.equal(2);
                    expect(array).to.deep.equal([1, 2, 4, 5, 4, 2, 1]);
                });
                it('should accept value not contained in the array', function () {
                    var array = [1, 2, 3, 4, 5, 4, 3, 2, 1];
                    expect(Utils.spliceValue(array, 10)).to.equal(0);
                    expect(array).to.deep.equal([1, 2, 3, 4, 5, 4, 3, 2, 1]);
                });
                it('should compare elements deeply', function () {
                    var array = [{ a: 1 }, { a: 2 }, { a: 3 }];
                    expect(Utils.spliceValue(array, { a: 2 })).to.equal(1);
                    expect(array).to.deep.equal([{ a: 1 }, { a: 3 }]);
                });
                it('should use the comparator function', function () {
                    var array = [1, 2, 3, 4, 5, 4, 3, 2, 1];
                    function comp(v1, v2) { return Math.abs(v1 - v2) === 1; }
                    expect(Utils.spliceValue(array, 3, comp)).to.equal(4);
                    expect(array).to.deep.equal([1, 3, 5, 3, 1]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "iterateArray"', function () {

                var ARRAY = [2, 4, 6];

                function breaker(v, i) {
                    if (i === 1) { return Utils.BREAK; }
                }

                function run(array, options, breaker) {
                    return chai.runIterator(Utils.iterateArray, Utils, [array, chai.ITERATOR, options], breaker);
                }

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('iterateArray');
                });
                it('should visit all array elements', function () {
                    var result = run(ARRAY);
                    expect(result.ret).to.be['undefined'];
                    expect(result.args).to.deep.equal([[2, 0, ARRAY], [4, 1, ARRAY], [6, 2, ARRAY]]);
                });
                it('should visit all array elements in reversed order', function () {
                    var result = run(ARRAY, { reverse: true });
                    expect(result.ret).to.be['undefined'];
                    expect(result.args).to.deep.equal([[6, 2, ARRAY], [4, 1, ARRAY], [2, 0, ARRAY]]);
                });
                it('should accept an empty array', function () {
                    var result = run([]);
                    expect(result.ret).to.be['undefined'];
                    expect(result.args).to.be.empty;
                });
                it('should bind the specified context to the iterator function', function () {
                    var context = {},
                        result = run(ARRAY, { context: context });
                    expect(result.ret).to.be['undefined'];
                    expect(result.runs[0].context).to.equal(context);
                    expect(result.runs[1].context).to.equal(context);
                    expect(result.runs[2].context).to.equal(context);
                });
                it('should break if iterator returns Utils.BREAK', function () {
                    var result = run(ARRAY, undefined, breaker);
                    expect(result.ret).to.equal(Utils.BREAK);
                    expect(result.args).to.deep.equal([[2, 0, ARRAY], [4, 1, ARRAY]]);
                });
                it('should break in reverse mode if iterator returns Utils.BREAK', function () {
                    var result = run(ARRAY, { reverse: true }, breaker);
                    expect(result.ret).to.equal(Utils.BREAK);
                    expect(result.args).to.deep.equal([[6, 2, ARRAY], [4, 1, ARRAY]]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "findFirstIndex"', function () {

                var ARRAY1 = [2, 4, 6, 8, 6, 4, 2],
                    ARRAY2 = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31];

                function greater(value) {
                    return function (element) { return element > value; };
                }

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('findFirstIndex');
                });
                it('should return array index of first matching element in linear search', function () {
                    expect(Utils.findFirstIndex(ARRAY1, greater(0))).to.equal(0);
                    expect(Utils.findFirstIndex(ARRAY1, greater(1))).to.equal(0);
                    expect(Utils.findFirstIndex(ARRAY1, greater(2))).to.equal(1);
                    expect(Utils.findFirstIndex(ARRAY1, greater(3))).to.equal(1);
                    expect(Utils.findFirstIndex(ARRAY1, greater(4))).to.equal(2);
                    expect(Utils.findFirstIndex(ARRAY1, greater(5))).to.equal(2);
                    expect(Utils.findFirstIndex(ARRAY1, greater(6))).to.equal(3);
                    expect(Utils.findFirstIndex(ARRAY1, greater(7))).to.equal(3);
                });
                it('should return -1 if no array element matches in linear search', function () {
                    expect(Utils.findFirstIndex(ARRAY1, greater(8))).to.equal(-1);
                    expect(Utils.findFirstIndex(ARRAY1, greater(9))).to.equal(-1);
                });
                it('should visit all elements preceding the matching element in linear search', function () {
                    var result = chai.runIterator(Utils.findFirstIndex, Utils, [ARRAY2, chai.ITERATOR], greater(28));
                    expect(result.runs).to.have.length(30);
                });
                it('should return array index of first matching element in binary search', function () {
                    expect(Utils.findFirstIndex(ARRAY2, greater(-2), { sorted: true })).to.equal(0);
                    expect(Utils.findFirstIndex(ARRAY2, greater(-1), { sorted: true })).to.equal(0);
                    expect(Utils.findFirstIndex(ARRAY2, greater(0), { sorted: true })).to.equal(1);
                    expect(Utils.findFirstIndex(ARRAY2, greater(1), { sorted: true })).to.equal(2);
                    expect(Utils.findFirstIndex(ARRAY2, greater(2), { sorted: true })).to.equal(3);
                    expect(Utils.findFirstIndex(ARRAY2, greater(3), { sorted: true })).to.equal(4);
                    expect(Utils.findFirstIndex(ARRAY2, greater(4), { sorted: true })).to.equal(5);
                    expect(Utils.findFirstIndex(ARRAY2, greater(7), { sorted: true })).to.equal(8);
                    expect(Utils.findFirstIndex(ARRAY2, greater(10), { sorted: true })).to.equal(11);
                    expect(Utils.findFirstIndex(ARRAY2, greater(13), { sorted: true })).to.equal(14);
                    expect(Utils.findFirstIndex(ARRAY2, greater(16), { sorted: true })).to.equal(17);
                    expect(Utils.findFirstIndex(ARRAY2, greater(19), { sorted: true })).to.equal(20);
                    expect(Utils.findFirstIndex(ARRAY2, greater(22), { sorted: true })).to.equal(23);
                    expect(Utils.findFirstIndex(ARRAY2, greater(25), { sorted: true })).to.equal(26);
                    expect(Utils.findFirstIndex(ARRAY2, greater(28), { sorted: true })).to.equal(29);
                    expect(Utils.findFirstIndex(ARRAY2, greater(29), { sorted: true })).to.equal(30);
                    expect(Utils.findFirstIndex(ARRAY2, greater(30), { sorted: true })).to.equal(31);
                });
                it('should return -1 if no array element matches in binary search', function () {
                    expect(Utils.findFirstIndex(ARRAY2, greater(31), { sorted: true })).to.equal(-1);
                    expect(Utils.findFirstIndex(ARRAY2, greater(32), { sorted: true })).to.equal(-1);
                });
                it('should return array index of first matching element for short arrays in binary search', function () {
                    expect(Utils.findFirstIndex([1], greater(0), { sorted: true })).to.equal(0);
                    expect(Utils.findFirstIndex([1], greater(1), { sorted: true })).to.equal(-1);
                    expect(Utils.findFirstIndex([1, 2], greater(0), { sorted: true })).to.equal(0);
                    expect(Utils.findFirstIndex([1, 2], greater(1), { sorted: true })).to.equal(1);
                    expect(Utils.findFirstIndex([1, 2], greater(2), { sorted: true })).to.equal(-1);
                });
                it('should not visit all elements preceding the matching element in binary search', function () {
                    var result = chai.runIterator(Utils.findFirstIndex, Utils, [ARRAY2, chai.ITERATOR, { sorted: true }], greater(28));
                    expect(result.runs).to.have.length.below(6);
                });
                it('should accept an empty array', function () {
                    expect(Utils.findFirstIndex([], greater(0))).to.equal(-1);
                });
                it('should bind the specified context to the iterator function', function () {
                    var context = {},
                        result = chai.runIterator(Utils.findFirstIndex, Utils, [ARRAY1, chai.ITERATOR, { context: context }], greater(0));
                    expect(result.runs[0].context).to.equal(context);
                });
            });

            // ----------------------------------------------------------------

            describe('method "findLastIndex"', function () {

                var ARRAY1 = [2, 4, 6, 8, 6, 4, 2],
                    ARRAY2 = [31, 30, 29, 28, 27, 26, 25, 24, 23, 22, 21, 20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0];

                function greater(value) {
                    return function (element) { return element > value; };
                }

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('findLastIndex');
                });
                it('should return array index of last matching element in linear search', function () {
                    expect(Utils.findLastIndex(ARRAY1, greater(0))).to.equal(6);
                    expect(Utils.findLastIndex(ARRAY1, greater(1))).to.equal(6);
                    expect(Utils.findLastIndex(ARRAY1, greater(2))).to.equal(5);
                    expect(Utils.findLastIndex(ARRAY1, greater(3))).to.equal(5);
                    expect(Utils.findLastIndex(ARRAY1, greater(4))).to.equal(4);
                    expect(Utils.findLastIndex(ARRAY1, greater(5))).to.equal(4);
                    expect(Utils.findLastIndex(ARRAY1, greater(6))).to.equal(3);
                    expect(Utils.findLastIndex(ARRAY1, greater(7))).to.equal(3);
                });
                it('should return -1 if no array element matches in linear search', function () {
                    expect(Utils.findLastIndex(ARRAY1, greater(8))).to.equal(-1);
                    expect(Utils.findLastIndex(ARRAY1, greater(9))).to.equal(-1);
                });
                it('should visit all elements following the matching element in linear search', function () {
                    var result = chai.runIterator(Utils.findLastIndex, Utils, [ARRAY2, chai.ITERATOR], greater(28));
                    expect(result.runs).to.have.length(30);
                });
                it('should return array index of last matching element in binary search', function () {
                    expect(Utils.findLastIndex(ARRAY2, greater(-2), { sorted: true })).to.equal(31);
                    expect(Utils.findLastIndex(ARRAY2, greater(-1), { sorted: true })).to.equal(31);
                    expect(Utils.findLastIndex(ARRAY2, greater(0), { sorted: true })).to.equal(30);
                    expect(Utils.findLastIndex(ARRAY2, greater(1), { sorted: true })).to.equal(29);
                    expect(Utils.findLastIndex(ARRAY2, greater(2), { sorted: true })).to.equal(28);
                    expect(Utils.findLastIndex(ARRAY2, greater(3), { sorted: true })).to.equal(27);
                    expect(Utils.findLastIndex(ARRAY2, greater(4), { sorted: true })).to.equal(26);
                    expect(Utils.findLastIndex(ARRAY2, greater(7), { sorted: true })).to.equal(23);
                    expect(Utils.findLastIndex(ARRAY2, greater(10), { sorted: true })).to.equal(20);
                    expect(Utils.findLastIndex(ARRAY2, greater(13), { sorted: true })).to.equal(17);
                    expect(Utils.findLastIndex(ARRAY2, greater(16), { sorted: true })).to.equal(14);
                    expect(Utils.findLastIndex(ARRAY2, greater(19), { sorted: true })).to.equal(11);
                    expect(Utils.findLastIndex(ARRAY2, greater(22), { sorted: true })).to.equal(8);
                    expect(Utils.findLastIndex(ARRAY2, greater(25), { sorted: true })).to.equal(5);
                    expect(Utils.findLastIndex(ARRAY2, greater(28), { sorted: true })).to.equal(2);
                    expect(Utils.findLastIndex(ARRAY2, greater(29), { sorted: true })).to.equal(1);
                    expect(Utils.findLastIndex(ARRAY2, greater(30), { sorted: true })).to.equal(0);
                });
                it('should return -1 if no array element matches in linear search', function () {
                    expect(Utils.findLastIndex(ARRAY2, greater(31), { sorted: true })).to.equal(-1);
                    expect(Utils.findLastIndex(ARRAY2, greater(32), { sorted: true })).to.equal(-1);
                });
                it('should return array index of last matching element for short arrays in binary search', function () {
                    expect(Utils.findLastIndex([1], greater(0), { sorted: true })).to.equal(0);
                    expect(Utils.findLastIndex([1], greater(1), { sorted: true })).to.equal(-1);
                    expect(Utils.findLastIndex([2, 1], greater(0), { sorted: true })).to.equal(1);
                    expect(Utils.findLastIndex([2, 1], greater(1), { sorted: true })).to.equal(0);
                    expect(Utils.findLastIndex([2, 1], greater(2), { sorted: true })).to.equal(-1);
                });
                it('should not visit all elements preceding the matching element in binary search', function () {
                    var result = chai.runIterator(Utils.findLastIndex, Utils, [ARRAY2, chai.ITERATOR, { sorted: true }], greater(28));
                    expect(result.runs).to.have.length.below(6);
                });
                it('should accept an empty array', function () {
                    expect(Utils.findLastIndex([], greater(0))).to.equal(-1);
                });
                it('should bind the specified context to the iterator function', function () {
                    var context = {},
                        result = chai.runIterator(Utils.findLastIndex, Utils, [ARRAY1, chai.ITERATOR, { context: context }], greater(0));
                    expect(result.runs[0].context).to.equal(context);
                });
            });

            // ----------------------------------------------------------------

            describe('method "findFirst"', function () {

                var ARRAY1 = [2, 4, 6, 8, 7, 5, 3],
                    ARRAY2 = [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30];

                function greater(value) {
                    return function (element) { return element > value; };
                }

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('findFirst');
                });
                it('should return first matching element in linear search', function () {
                    expect(Utils.findFirst(ARRAY1, greater(0))).to.equal(2);
                    expect(Utils.findFirst(ARRAY1, greater(1))).to.equal(2);
                    expect(Utils.findFirst(ARRAY1, greater(2))).to.equal(4);
                    expect(Utils.findFirst(ARRAY1, greater(3))).to.equal(4);
                    expect(Utils.findFirst(ARRAY1, greater(4))).to.equal(6);
                    expect(Utils.findFirst(ARRAY1, greater(5))).to.equal(6);
                    expect(Utils.findFirst(ARRAY1, greater(6))).to.equal(8);
                    expect(Utils.findFirst(ARRAY1, greater(7))).to.equal(8);
                });
                it('should return undefined if no array element matches in linear search', function () {
                    expect(Utils.findFirst(ARRAY1, greater(8))).to.be['undefined'];
                    expect(Utils.findFirst(ARRAY1, greater(9))).to.be['undefined'];
                });
                it('should visit all elements preceding the matching element in linear search', function () {
                    var result = chai.runIterator(Utils.findFirst, Utils, [ARRAY2, chai.ITERATOR], greater(26));
                    expect(result.runs).to.have.length(15);
                });
                it('should return first matching element in binary search', function () {
                    expect(Utils.findFirst(ARRAY2, greater(-2), { sorted: true })).to.equal(0);
                    expect(Utils.findFirst(ARRAY2, greater(-1), { sorted: true })).to.equal(0);
                    expect(Utils.findFirst(ARRAY2, greater(0), { sorted: true })).to.equal(2);
                    expect(Utils.findFirst(ARRAY2, greater(1), { sorted: true })).to.equal(2);
                    expect(Utils.findFirst(ARRAY2, greater(2), { sorted: true })).to.equal(4);
                    expect(Utils.findFirst(ARRAY2, greater(3), { sorted: true })).to.equal(4);
                    expect(Utils.findFirst(ARRAY2, greater(4), { sorted: true })).to.equal(6);
                    expect(Utils.findFirst(ARRAY2, greater(7), { sorted: true })).to.equal(8);
                    expect(Utils.findFirst(ARRAY2, greater(10), { sorted: true })).to.equal(12);
                    expect(Utils.findFirst(ARRAY2, greater(13), { sorted: true })).to.equal(14);
                    expect(Utils.findFirst(ARRAY2, greater(16), { sorted: true })).to.equal(18);
                    expect(Utils.findFirst(ARRAY2, greater(19), { sorted: true })).to.equal(20);
                    expect(Utils.findFirst(ARRAY2, greater(22), { sorted: true })).to.equal(24);
                    expect(Utils.findFirst(ARRAY2, greater(25), { sorted: true })).to.equal(26);
                    expect(Utils.findFirst(ARRAY2, greater(28), { sorted: true })).to.equal(30);
                    expect(Utils.findFirst(ARRAY2, greater(29), { sorted: true })).to.equal(30);
                });
                it('should return undefined if no array element matches in binary search', function () {
                    expect(Utils.findFirst(ARRAY2, greater(30), { sorted: true })).to.be['undefined'];
                    expect(Utils.findFirst(ARRAY2, greater(31), { sorted: true })).to.be['undefined'];
                });
                it('should return first matching element for short arrays in binary search', function () {
                    expect(Utils.findFirst([1], greater(0), { sorted: true })).to.equal(1);
                    expect(Utils.findFirst([1], greater(1), { sorted: true })).to.be['undefined'];
                    expect(Utils.findFirst([1, 2], greater(0), { sorted: true })).to.equal(1);
                    expect(Utils.findFirst([1, 2], greater(1), { sorted: true })).to.equal(2);
                    expect(Utils.findFirst([1, 2], greater(2), { sorted: true })).to.be['undefined'];
                });
                it('should not visit all elements preceding the matching element in binary search', function () {
                    var result = chai.runIterator(Utils.findFirst, Utils, [ARRAY2, chai.ITERATOR, { sorted: true }], greater(26));
                    expect(result.runs).to.have.length.below(5);
                });
                it('should accept an empty array', function () {
                    expect(Utils.findFirst([], greater(0))).to.be['undefined'];
                });
                it('should bind the specified context to the iterator function', function () {
                    var context = {},
                        result = chai.runIterator(Utils.findFirst, Utils, [ARRAY1, chai.ITERATOR, { context: context }], greater(0));
                    expect(result.runs[0].context).to.equal(context);
                });
            });

            // ----------------------------------------------------------------

            describe('method "findLast"', function () {

                var ARRAY1 = [2, 4, 6, 8, 7, 5, 3],
                    ARRAY2 = [30, 28, 26, 24, 22, 20, 18, 16, 14, 12, 10, 8, 6, 4, 2, 0];

                function greater(value) {
                    return function (element) { return element > value; };
                }

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('findLast');
                });
                it('should return last matching element in linear search', function () {
                    expect(Utils.findLast(ARRAY1, greater(0))).to.equal(3);
                    expect(Utils.findLast(ARRAY1, greater(1))).to.equal(3);
                    expect(Utils.findLast(ARRAY1, greater(2))).to.equal(3);
                    expect(Utils.findLast(ARRAY1, greater(3))).to.equal(5);
                    expect(Utils.findLast(ARRAY1, greater(4))).to.equal(5);
                    expect(Utils.findLast(ARRAY1, greater(5))).to.equal(7);
                    expect(Utils.findLast(ARRAY1, greater(6))).to.equal(7);
                    expect(Utils.findLast(ARRAY1, greater(7))).to.equal(8);
                });
                it('should return undefined if no array element matches in linear search', function () {
                    expect(Utils.findLast(ARRAY1, greater(8))).to.be['undefined'];
                    expect(Utils.findLast(ARRAY1, greater(9))).to.be['undefined'];
                });
                it('should visit all elements following the matching element in linear search', function () {
                    var result = chai.runIterator(Utils.findLast, Utils, [ARRAY2, chai.ITERATOR], greater(26));
                    expect(result.runs).to.have.length(15);
                });
                it('should return last matching element in binary search', function () {
                    expect(Utils.findLast(ARRAY2, greater(-2), { sorted: true })).to.equal(0);
                    expect(Utils.findLast(ARRAY2, greater(-1), { sorted: true })).to.equal(0);
                    expect(Utils.findLast(ARRAY2, greater(0), { sorted: true })).to.equal(2);
                    expect(Utils.findLast(ARRAY2, greater(1), { sorted: true })).to.equal(2);
                    expect(Utils.findLast(ARRAY2, greater(2), { sorted: true })).to.equal(4);
                    expect(Utils.findLast(ARRAY2, greater(3), { sorted: true })).to.equal(4);
                    expect(Utils.findLast(ARRAY2, greater(4), { sorted: true })).to.equal(6);
                    expect(Utils.findLast(ARRAY2, greater(7), { sorted: true })).to.equal(8);
                    expect(Utils.findLast(ARRAY2, greater(10), { sorted: true })).to.equal(12);
                    expect(Utils.findLast(ARRAY2, greater(13), { sorted: true })).to.equal(14);
                    expect(Utils.findLast(ARRAY2, greater(16), { sorted: true })).to.equal(18);
                    expect(Utils.findLast(ARRAY2, greater(19), { sorted: true })).to.equal(20);
                    expect(Utils.findLast(ARRAY2, greater(22), { sorted: true })).to.equal(24);
                    expect(Utils.findLast(ARRAY2, greater(25), { sorted: true })).to.equal(26);
                    expect(Utils.findLast(ARRAY2, greater(28), { sorted: true })).to.equal(30);
                    expect(Utils.findLast(ARRAY2, greater(29), { sorted: true })).to.equal(30);
                });
                it('should return undefined if no array element matches in binary search', function () {
                    expect(Utils.findLast(ARRAY2, greater(30), { sorted: true })).to.be['undefined'];
                    expect(Utils.findLast(ARRAY2, greater(31), { sorted: true })).to.be['undefined'];
                });
                it('should return last matching element for short arrays in binary search', function () {
                    expect(Utils.findLast([1], greater(0), { sorted: true })).to.equal(1);
                    expect(Utils.findLast([1], greater(1), { sorted: true })).to.be['undefined'];
                    expect(Utils.findLast([2, 1], greater(0), { sorted: true })).to.equal(1);
                    expect(Utils.findLast([2, 1], greater(1), { sorted: true })).to.equal(2);
                    expect(Utils.findLast([2, 1], greater(2), { sorted: true })).to.be['undefined'];
                });
                it('should not visit all elements following the matching element in binary search', function () {
                    var result = chai.runIterator(Utils.findLast, Utils, [ARRAY2, chai.ITERATOR, { sorted: true }], greater(26));
                    expect(result.runs).to.have.length.below(5);
                });
                it('should accept an empty array', function () {
                    expect(Utils.findLast([], greater(0))).to.be['undefined'];
                });
                it('should bind the specified context to the iterator function', function () {
                    var context = {},
                        result = chai.runIterator(Utils.findLast, Utils, [ARRAY1, chai.ITERATOR, { context: context }], greater(0));
                    expect(result.runs[0].context).to.equal(context);
                });
            });

            // ----------------------------------------------------------------

            describe('method "mod"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('mod');
                });
                it('should return modulo for zero', function () {
                    expect(Utils.mod(0, 5)).to.equal(0);
                });
                it('should return modulo for positive values', function () {
                    expect(Utils.mod(2, 5)).to.equal(2);
                    expect(Utils.mod(5, 5)).to.equal(0);
                    expect(Utils.mod(7, 5)).to.equal(2);
                    expect(Utils.mod(10, 5)).to.equal(0);
                });
                it('should return modulo for negative values', function () {
                    expect(Utils.mod(-2, 5)).to.equal(3);
                    expect(Utils.mod(-5, 5)).to.equal(0);
                    expect(Utils.mod(-7, 5)).to.equal(3);
                    expect(Utils.mod(-10, 5)).to.equal(0);
                });
            });

            // ----------------------------------------------------------------

            describe('method "boolEq"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('boolEq');
                });
                it('should return false for a truthy and a falsy value', function () {
                    expect(Utils.boolEq(true, false)).to.be['false'];
                    expect(Utils.boolEq(1, false)).to.be['false'];
                    expect(Utils.boolEq('a', false)).to.be['false'];
                    expect(Utils.boolEq(/a/, false)).to.be['false'];
                    expect(Utils.boolEq([], false)).to.be['false'];
                    expect(Utils.boolEq({}, false)).to.be['false'];
                    expect(Utils.boolEq(true, 0)).to.be['false'];
                    expect(Utils.boolEq(true, '')).to.be['false'];
                    expect(Utils.boolEq(true, null)).to.be['false'];
                    expect(Utils.boolEq(true, undefined)).to.be['false'];
                    expect(Utils.boolEq(1, undefined)).to.be['false'];
                });
                it('should return false for a falsy and a truthy value', function () {
                    expect(Utils.boolEq(false, true)).to.be['false'];
                    expect(Utils.boolEq(0, true)).to.be['false'];
                    expect(Utils.boolEq('', true)).to.be['false'];
                    expect(Utils.boolEq(null, true)).to.be['false'];
                    expect(Utils.boolEq(undefined, true)).to.be['false'];
                    expect(Utils.boolEq(false, 1)).to.be['false'];
                    expect(Utils.boolEq(false, 'a')).to.be['false'];
                    expect(Utils.boolEq(false, /a/)).to.be['false'];
                    expect(Utils.boolEq(false, [])).to.be['false'];
                    expect(Utils.boolEq(false, {})).to.be['false'];
                    expect(Utils.boolEq(0, {})).to.be['false'];
                });
                it('should return true for two truthy values', function () {
                    expect(Utils.boolEq(true, true)).to.be['true'];
                    expect(Utils.boolEq(1, true)).to.be['true'];
                    expect(Utils.boolEq('a', true)).to.be['true'];
                    expect(Utils.boolEq(/a/, true)).to.be['true'];
                    expect(Utils.boolEq([], true)).to.be['true'];
                    expect(Utils.boolEq({}, true)).to.be['true'];
                    expect(Utils.boolEq(true, 1)).to.be['true'];
                    expect(Utils.boolEq(true, 'a')).to.be['true'];
                    expect(Utils.boolEq(true, /a/)).to.be['true'];
                    expect(Utils.boolEq(true, [])).to.be['true'];
                    expect(Utils.boolEq(true, {})).to.be['true'];
                    expect(Utils.boolEq(1, {})).to.be['true'];
                });
                it('should return true for two falsy values', function () {
                    expect(Utils.boolEq(false, false)).to.be['true'];
                    expect(Utils.boolEq(0, false)).to.be['true'];
                    expect(Utils.boolEq('', false)).to.be['true'];
                    expect(Utils.boolEq(null, false)).to.be['true'];
                    expect(Utils.boolEq(undefined, false)).to.be['true'];
                    expect(Utils.boolEq(false, 0)).to.be['true'];
                    expect(Utils.boolEq(false, '')).to.be['true'];
                    expect(Utils.boolEq(false, null)).to.be['true'];
                    expect(Utils.boolEq(false, undefined)).to.be['true'];
                    expect(Utils.boolEq(0, undefined)).to.be['true'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "boolXor"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('boolXor');
                });
                it('should return true for a truthy and a falsy value', function () {
                    expect(Utils.boolXor(true, false)).to.be['true'];
                    expect(Utils.boolXor(1, false)).to.be['true'];
                    expect(Utils.boolXor('a', false)).to.be['true'];
                    expect(Utils.boolXor(/a/, false)).to.be['true'];
                    expect(Utils.boolXor([], false)).to.be['true'];
                    expect(Utils.boolXor({}, false)).to.be['true'];
                    expect(Utils.boolXor(true, 0)).to.be['true'];
                    expect(Utils.boolXor(true, '')).to.be['true'];
                    expect(Utils.boolXor(true, null)).to.be['true'];
                    expect(Utils.boolXor(true, undefined)).to.be['true'];
                    expect(Utils.boolXor(1, undefined)).to.be['true'];
                });
                it('should return true for a falsy and a truthy value', function () {
                    expect(Utils.boolXor(false, true)).to.be['true'];
                    expect(Utils.boolXor(0, true)).to.be['true'];
                    expect(Utils.boolXor('', true)).to.be['true'];
                    expect(Utils.boolXor(null, true)).to.be['true'];
                    expect(Utils.boolXor(undefined, true)).to.be['true'];
                    expect(Utils.boolXor(false, 1)).to.be['true'];
                    expect(Utils.boolXor(false, 'a')).to.be['true'];
                    expect(Utils.boolXor(false, /a/)).to.be['true'];
                    expect(Utils.boolXor(false, [])).to.be['true'];
                    expect(Utils.boolXor(false, {})).to.be['true'];
                    expect(Utils.boolXor(0, {})).to.be['true'];
                });
                it('should return false for two truthy values', function () {
                    expect(Utils.boolXor(true, true)).to.be['false'];
                    expect(Utils.boolXor(1, true)).to.be['false'];
                    expect(Utils.boolXor('a', true)).to.be['false'];
                    expect(Utils.boolXor(/a/, true)).to.be['false'];
                    expect(Utils.boolXor([], true)).to.be['false'];
                    expect(Utils.boolXor({}, true)).to.be['false'];
                    expect(Utils.boolXor(true, 1)).to.be['false'];
                    expect(Utils.boolXor(true, 'a')).to.be['false'];
                    expect(Utils.boolXor(true, /a/)).to.be['false'];
                    expect(Utils.boolXor(true, [])).to.be['false'];
                    expect(Utils.boolXor(true, {})).to.be['false'];
                    expect(Utils.boolXor(1, {})).to.be['false'];
                });
                it('should return false for two falsy values', function () {
                    expect(Utils.boolXor(false, false)).to.be['false'];
                    expect(Utils.boolXor(0, false)).to.be['false'];
                    expect(Utils.boolXor('', false)).to.be['false'];
                    expect(Utils.boolXor(null, false)).to.be['false'];
                    expect(Utils.boolXor(undefined, false)).to.be['false'];
                    expect(Utils.boolXor(false, 0)).to.be['false'];
                    expect(Utils.boolXor(false, '')).to.be['false'];
                    expect(Utils.boolXor(false, null)).to.be['false'];
                    expect(Utils.boolXor(false, undefined)).to.be['false'];
                    expect(Utils.boolXor(0, undefined)).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "minMax"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('minMax');
                });
                it('should return the interval minimum for small values', function () {
                    expect(Utils.minMax(1, 3, 5)).to.equal(3);
                    expect(Utils.minMax(2, 3, 5)).to.equal(3);
                });
                it('should return the passed value if it is inside the interval', function () {
                    expect(Utils.minMax(3, 3, 5)).to.equal(3);
                    expect(Utils.minMax(4, 3, 5)).to.equal(4);
                    expect(Utils.minMax(5, 3, 5)).to.equal(5);
                });
                it('should return the interval maximum for large values', function () {
                    expect(Utils.minMax(6, 3, 5)).to.equal(5);
                    expect(Utils.minMax(7, 3, 5)).to.equal(5);
                });
            });

            // ----------------------------------------------------------------

            describe('method "round"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('round');
                });
                it('should round positive numbers to integer precision', function () {
                    expect(Utils.round(2, 1)).to.equal(2);
                    expect(Utils.round(2.1, 1)).to.equal(2);
                    expect(Utils.round(2.2, 1)).to.equal(2);
                    expect(Utils.round(2.3, 1)).to.equal(2);
                    expect(Utils.round(2.4, 1)).to.equal(2);
                    expect(Utils.round(2.5, 1)).to.equal(3);
                    expect(Utils.round(2.6, 1)).to.equal(3);
                    expect(Utils.round(2.7, 1)).to.equal(3);
                    expect(Utils.round(2.8, 1)).to.equal(3);
                    expect(Utils.round(2.9, 1)).to.equal(3);
                    expect(Utils.round(3, 1)).to.equal(3);
                });
                it('should round negative numbers to integer precision', function () {
                    expect(Utils.round(-2, 1)).to.equal(-2);
                    expect(Utils.round(-2.1, 1)).to.equal(-2);
                    expect(Utils.round(-2.2, 1)).to.equal(-2);
                    expect(Utils.round(-2.3, 1)).to.equal(-2);
                    expect(Utils.round(-2.4, 1)).to.equal(-2);
                    expect(Utils.round(-2.5, 1)).to.equal(-2);
                    expect(Utils.round(-2.6, 1)).to.equal(-3);
                    expect(Utils.round(-2.7, 1)).to.equal(-3);
                    expect(Utils.round(-2.8, 1)).to.equal(-3);
                    expect(Utils.round(-2.9, 1)).to.equal(-3);
                    expect(Utils.round(-3, 1)).to.equal(-3);
                });
                it('should round numbers near 0 to integer precision', function () {
                    expect(Utils.round(-0.9, 1)).to.equal(-1);
                    expect(Utils.round(-0.500001, 1)).to.equal(-1);
                    expect(Utils.round(-0.5, 1)).to.equal(0);
                    expect(Utils.round(-0.1, 1)).to.equal(0);
                    expect(Utils.round(0, 1)).to.equal(0);
                    expect(Utils.round(0.1, 1)).to.equal(0);
                    expect(Utils.round(0.499999, 1)).to.equal(0);
                    expect(Utils.round(0.5, 1)).to.equal(1);
                    expect(Utils.round(0.9, 1)).to.equal(1);
                });
                it('should round positive numbers to custom precision', function () {
                    expect(Utils.round(227, 10)).to.equal(230);
                    expect(Utils.round(227, 20)).to.equal(220);
                    expect(Utils.round(227, 50)).to.equal(250);
                    expect(Utils.round(227, 100)).to.equal(200);
                    expect(Utils.round(227, 200)).to.equal(200);
                    expect(Utils.round(227, 500)).to.equal(0);
                    expect(Utils.round(227, 0.1)).to.equal(227);
                    expect(Utils.round(2.27, 0.01)).to.equal(2.27);
                    expect(Utils.round(2.27, 0.02)).to.equal(2.28);
                    expect(Utils.round(2.27, 0.05)).to.equal(2.25);
                    expect(Utils.round(2.27, 0.1)).to.equal(2.3);
                    expect(Utils.round(2.27, 0.2)).to.equal(2.2);
                    expect(Utils.round(2.27, 0.5)).to.equal(2.5);
                });
                it('should round negative numbers to custom precision', function () {
                    expect(Utils.round(-227, 10)).to.equal(-230);
                    expect(Utils.round(-227, 20)).to.equal(-220);
                    expect(Utils.round(-227, 50)).to.equal(-250);
                    expect(Utils.round(-227, 100)).to.equal(-200);
                    expect(Utils.round(-227, 200)).to.equal(-200);
                    expect(Utils.round(-227, 500)).to.equal(0);
                    expect(Utils.round(-227, 0.1)).to.equal(-227);
                    expect(Utils.round(-2.27, 0.01)).to.equal(-2.27);
                    expect(Utils.round(-2.27, 0.02)).to.equal(-2.26);
                    expect(Utils.round(-2.27, 0.05)).to.equal(-2.25);
                    expect(Utils.round(-2.27, 0.1)).to.equal(-2.3);
                    expect(Utils.round(-2.27, 0.2)).to.equal(-2.2);
                    expect(Utils.round(-2.27, 0.5)).to.equal(-2.5);
                });
                it('should round numbers near 0 to custom precision', function () {
                    expect(Utils.round(-5.000001, 10)).to.equal(-10);
                    expect(Utils.round(-5, 10)).to.equal(0);
                    expect(Utils.round(0, 10)).to.equal(0);
                    expect(Utils.round(4.999999, 10)).to.equal(0);
                    expect(Utils.round(5, 10)).to.equal(10);
                    expect(Utils.round(-0.05000001, 0.1)).to.equal(-0.1);
                    expect(Utils.round(-0.05, 0.1)).to.equal(0);
                    expect(Utils.round(0, 0.1)).to.equal(0);
                    expect(Utils.round(0.04999999, 0.1)).to.equal(0);
                    expect(Utils.round(0.05, 0.1)).to.equal(0.1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "roundDown"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('roundDown');
                });
                it('should round positive numbers to integer precision', function () {
                    expect(Utils.roundDown(2, 1)).to.equal(2);
                    expect(Utils.roundDown(2.1, 1)).to.equal(2);
                    expect(Utils.roundDown(2.2, 1)).to.equal(2);
                    expect(Utils.roundDown(2.3, 1)).to.equal(2);
                    expect(Utils.roundDown(2.4, 1)).to.equal(2);
                    expect(Utils.roundDown(2.5, 1)).to.equal(2);
                    expect(Utils.roundDown(2.6, 1)).to.equal(2);
                    expect(Utils.roundDown(2.7, 1)).to.equal(2);
                    expect(Utils.roundDown(2.8, 1)).to.equal(2);
                    expect(Utils.roundDown(2.9, 1)).to.equal(2);
                    expect(Utils.roundDown(3, 1)).to.equal(3);
                });
                it('should round negative numbers to integer precision', function () {
                    expect(Utils.roundDown(-2, 1)).to.equal(-2);
                    expect(Utils.roundDown(-2.1, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.2, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.3, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.4, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.5, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.6, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.7, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.8, 1)).to.equal(-3);
                    expect(Utils.roundDown(-2.9, 1)).to.equal(-3);
                    expect(Utils.roundDown(-3, 1)).to.equal(-3);
                });
                it('should round numbers near 0 to integer precision', function () {
                    expect(Utils.roundDown(-0.9, 1)).to.equal(-1);
                    expect(Utils.roundDown(-0.5, 1)).to.equal(-1);
                    expect(Utils.roundDown(-0.1, 1)).to.equal(-1);
                    expect(Utils.roundDown(0, 1)).to.equal(0);
                    expect(Utils.roundDown(0.1, 1)).to.equal(0);
                    expect(Utils.roundDown(0.5, 1)).to.equal(0);
                    expect(Utils.roundDown(0.9, 1)).to.equal(0);
                });
                it('should round positive numbers to custom precision', function () {
                    expect(Utils.roundDown(227, 10)).to.equal(220);
                    expect(Utils.roundDown(227, 20)).to.equal(220);
                    expect(Utils.roundDown(227, 50)).to.equal(200);
                    expect(Utils.roundDown(227, 100)).to.equal(200);
                    expect(Utils.roundDown(227, 200)).to.equal(200);
                    expect(Utils.roundDown(227, 500)).to.equal(0);
                    expect(Utils.roundDown(227, 0.1)).to.equal(227);
                    expect(Utils.roundDown(2.27, 0.01)).to.equal(2.27);
                    expect(Utils.roundDown(2.27, 0.02)).to.equal(2.26);
                    expect(Utils.roundDown(2.27, 0.05)).to.equal(2.25);
                    expect(Utils.roundDown(2.27, 0.1)).to.equal(2.2);
                    expect(Utils.roundDown(2.27, 0.2)).to.equal(2.2);
                    expect(Utils.roundDown(2.27, 0.5)).to.equal(2);
                });
                it('should round negative numbers to custom precision', function () {
                    expect(Utils.roundDown(-227, 10)).to.equal(-230);
                    expect(Utils.roundDown(-227, 20)).to.equal(-240);
                    expect(Utils.roundDown(-227, 50)).to.equal(-250);
                    expect(Utils.roundDown(-227, 100)).to.equal(-300);
                    expect(Utils.roundDown(-227, 200)).to.equal(-400);
                    expect(Utils.roundDown(-227, 500)).to.equal(-500);
                    expect(Utils.roundDown(-227, 0.1)).to.equal(-227);
                    expect(Utils.roundDown(-2.27, 0.01)).to.equal(-2.27);
                    expect(Utils.roundDown(-2.27, 0.02)).to.equal(-2.28);
                    expect(Utils.roundDown(-2.27, 0.05)).to.equal(-2.3);
                    expect(Utils.roundDown(-2.27, 0.1)).to.equal(-2.3);
                    expect(Utils.roundDown(-2.27, 0.2)).to.equal(-2.4);
                    expect(Utils.roundDown(-2.27, 0.5)).to.equal(-2.5);
                });
                it('should round numbers near 0 to custom precision', function () {
                    expect(Utils.roundDown(-5, 10)).to.equal(-10);
                    expect(Utils.roundDown(0, 10)).to.equal(0);
                    expect(Utils.roundDown(5, 10)).to.equal(0);
                    expect(Utils.roundDown(-0.05, 0.1)).to.equal(-0.1);
                    expect(Utils.roundDown(0, 0.1)).to.equal(0);
                    expect(Utils.roundDown(0.05, 0.1)).to.equal(0);
                });
            });

            // ----------------------------------------------------------------

            describe('method "roundUp"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('roundUp');
                });
                it('should round positive numbers to integer precision', function () {
                    expect(Utils.roundUp(2, 1)).to.equal(2);
                    expect(Utils.roundUp(2.1, 1)).to.equal(3);
                    expect(Utils.roundUp(2.2, 1)).to.equal(3);
                    expect(Utils.roundUp(2.3, 1)).to.equal(3);
                    expect(Utils.roundUp(2.4, 1)).to.equal(3);
                    expect(Utils.roundUp(2.5, 1)).to.equal(3);
                    expect(Utils.roundUp(2.6, 1)).to.equal(3);
                    expect(Utils.roundUp(2.7, 1)).to.equal(3);
                    expect(Utils.roundUp(2.8, 1)).to.equal(3);
                    expect(Utils.roundUp(2.9, 1)).to.equal(3);
                    expect(Utils.roundUp(3, 1)).to.equal(3);
                });
                it('should round negative numbers to integer precision', function () {
                    expect(Utils.roundUp(-2, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.1, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.2, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.3, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.4, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.5, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.6, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.7, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.8, 1)).to.equal(-2);
                    expect(Utils.roundUp(-2.9, 1)).to.equal(-2);
                    expect(Utils.roundUp(-3, 1)).to.equal(-3);
                });
                it('should round numbers near 0 to integer precision', function () {
                    expect(Utils.roundUp(-0.9, 1)).to.equal(0);
                    expect(Utils.roundUp(-0.5, 1)).to.equal(0);
                    expect(Utils.roundUp(-0.1, 1)).to.equal(0);
                    expect(Utils.roundUp(0, 1)).to.equal(0);
                    expect(Utils.roundUp(0.1, 1)).to.equal(1);
                    expect(Utils.roundUp(0.5, 1)).to.equal(1);
                    expect(Utils.roundUp(0.9, 1)).to.equal(1);
                });
                it('should round positive numbers to custom precision', function () {
                    expect(Utils.roundUp(227, 10)).to.equal(230);
                    expect(Utils.roundUp(227, 20)).to.equal(240);
                    expect(Utils.roundUp(227, 50)).to.equal(250);
                    expect(Utils.roundUp(227, 100)).to.equal(300);
                    expect(Utils.roundUp(227, 200)).to.equal(400);
                    expect(Utils.roundUp(227, 500)).to.equal(500);
                    expect(Utils.roundUp(227, 0.1)).to.equal(227);
                    expect(Utils.roundUp(2.27, 0.01)).to.equal(2.27);
                    expect(Utils.roundUp(2.27, 0.02)).to.equal(2.28);
                    expect(Utils.roundUp(2.27, 0.05)).to.equal(2.3);
                    expect(Utils.roundUp(2.27, 0.1)).to.equal(2.3);
                    expect(Utils.roundUp(2.27, 0.2)).to.equal(2.4);
                    expect(Utils.roundUp(2.27, 0.5)).to.equal(2.5);
                });
                it('should round negative numbers to custom precision', function () {
                    expect(Utils.roundUp(-227, 10)).to.equal(-220);
                    expect(Utils.roundUp(-227, 20)).to.equal(-220);
                    expect(Utils.roundUp(-227, 50)).to.equal(-200);
                    expect(Utils.roundUp(-227, 100)).to.equal(-200);
                    expect(Utils.roundUp(-227, 200)).to.equal(-200);
                    expect(Utils.roundUp(-227, 500)).to.equal(0);
                    expect(Utils.roundUp(-227, 0.1)).to.equal(-227);
                    expect(Utils.roundUp(-2.27, 0.01)).to.equal(-2.27);
                    expect(Utils.roundUp(-2.27, 0.02)).to.equal(-2.26);
                    expect(Utils.roundUp(-2.27, 0.05)).to.equal(-2.25);
                    expect(Utils.roundUp(-2.27, 0.1)).to.equal(-2.2);
                    expect(Utils.roundUp(-2.27, 0.2)).to.equal(-2.2);
                    expect(Utils.roundUp(-2.27, 0.5)).to.equal(-2);
                });
                it('should round numbers near 0 to custom precision', function () {
                    expect(Utils.roundUp(-5, 10)).to.equal(0);
                    expect(Utils.roundUp(0, 10)).to.equal(0);
                    expect(Utils.roundUp(5, 10)).to.equal(10);
                    expect(Utils.roundUp(-0.05, 0.1)).to.equal(0);
                    expect(Utils.roundUp(0, 0.1)).to.equal(0);
                    expect(Utils.roundUp(0.05, 0.1)).to.equal(0.1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "roundSignificantDigits"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('roundSignificantDigits');
                });
                it('should round positive numbers to specified precision', function () {
                    expect(Utils.roundSignificantDigits(1.23456789, 1)).to.equal(1);
                    expect(Utils.roundSignificantDigits(1.23456789, 2)).to.equal(1.2);
                    expect(Utils.roundSignificantDigits(1.23456789, 3)).to.equal(1.23);
                    expect(Utils.roundSignificantDigits(1.23456789, 4)).to.equal(1.235);
                    expect(Utils.roundSignificantDigits(1.23456789, 5)).to.equal(1.2346);
                    expect(Utils.roundSignificantDigits(1.23456789, 6)).to.equal(1.23457);
                    expect(Utils.roundSignificantDigits(1.23456789, 7)).to.equal(1.234568);
                    expect(Utils.roundSignificantDigits(1.23456789, 8)).to.equal(1.2345679);
                    expect(Utils.roundSignificantDigits(1.23456789, 9)).to.equal(1.23456789);
                    expect(Utils.roundSignificantDigits(1.23456789, 10)).to.equal(1.23456789);
                    expect(Utils.roundSignificantDigits(0.00123456789, 1)).to.equal(0.001);
                    expect(Utils.roundSignificantDigits(0.00123456789, 3)).to.equal(0.00123);
                    expect(Utils.roundSignificantDigits(0.00123456789, 5)).to.equal(0.0012346);
                    expect(Utils.roundSignificantDigits(123.456789, 1)).to.equal(100);
                    expect(Utils.roundSignificantDigits(123.456789, 3)).to.equal(123);
                    expect(Utils.roundSignificantDigits(123.456789, 5)).to.equal(123.46);
                });
                it('should round negative numbers to specified precision', function () {
                    expect(Utils.roundSignificantDigits(-1.23456789, 1)).to.equal(-1);
                    expect(Utils.roundSignificantDigits(-1.23456789, 2)).to.equal(-1.2);
                    expect(Utils.roundSignificantDigits(-1.23456789, 3)).to.equal(-1.23);
                    expect(Utils.roundSignificantDigits(-1.23456789, 4)).to.equal(-1.235);
                    expect(Utils.roundSignificantDigits(-1.23456789, 5)).to.equal(-1.2346);
                    expect(Utils.roundSignificantDigits(-1.23456789, 6)).to.equal(-1.23457);
                    expect(Utils.roundSignificantDigits(-1.23456789, 7)).to.equal(-1.234568);
                    expect(Utils.roundSignificantDigits(-1.23456789, 8)).to.equal(-1.2345679);
                    expect(Utils.roundSignificantDigits(-1.23456789, 9)).to.equal(-1.23456789);
                    expect(Utils.roundSignificantDigits(-1.23456789, 10)).to.equal(-1.23456789);
                    expect(Utils.roundSignificantDigits(-0.00123456789, 1)).to.equal(-0.001);
                    expect(Utils.roundSignificantDigits(-0.00123456789, 3)).to.equal(-0.00123);
                    expect(Utils.roundSignificantDigits(-0.00123456789, 5)).to.equal(-0.0012346);
                    expect(Utils.roundSignificantDigits(-123.456789, 1)).to.equal(-100);
                    expect(Utils.roundSignificantDigits(-123.456789, 3)).to.equal(-123);
                    expect(Utils.roundSignificantDigits(-123.456789, 5)).to.equal(-123.46);
                });
            });

            // ----------------------------------------------------------------

            describe('method "iterateRange"', function () {

                function breaker(v) {
                    if (v === 3) { return Utils.BREAK; }
                }

                function run(begin, end, options, breaker) {
                    return chai.runIterator(Utils.iterateRange, Utils, [begin, end, chai.ITERATOR, options], breaker);
                }

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('iterateRange');
                });
                it('should visit all integers in a half-open range', function () {
                    var result = run(2, 5);
                    expect(result.ret).to.be['undefined'];
                    expect(result.args).to.deep.equal([[2], [3], [4]]);
                });
                it('should use the specified positive step size', function () {
                    expect(_.flatten(run(2, 5, { step: 2 }).args)).to.deep.equal([2, 4]);
                    expect(_.flatten(run(2, 5, { step: 0.5 }).args)).to.deep.equal([2, 2.5, 3, 3.5, 4, 4.5]);
                });
                it('should use the specified negative step size', function () {
                    expect(_.flatten(run(5, 2, { step: -1 }).args)).to.deep.equal([5, 4, 3]);
                    expect(_.flatten(run(5, 2, { step: -2 }).args)).to.deep.equal([5, 3]);
                    expect(_.flatten(run(5, 2, { step: -0.5 }).args)).to.deep.equal([5, 4.5, 4, 3.5, 3, 2.5]);
                });
                it('should accept an empty range', function () {
                    expect(run(2, 2).runs).to.be.empty;
                });
                it('should bind the specified context to the iterator function', function () {
                    var context = {},
                        result = run(2, 5, { context: context });
                    expect(result.ret).to.be['undefined'];
                    expect(result.runs[0].context).to.equal(context);
                    expect(result.runs[1].context).to.equal(context);
                    expect(result.runs[2].context).to.equal(context);
                });
                it('should break if iterator returns Utils.BREAK', function () {
                    var result = run(2, 5, undefined, breaker);
                    expect(result.ret).to.equal(Utils.BREAK);
                    expect(_.flatten(result.args)).to.deep.equal([2, 3]);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getSum"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getSum');
                });
                it('should return the sum of the properties of an object', function () {
                    expect(Utils.getSum({ a: 1, b: -2, c: 3 }, _.identity)).to.equal(2);
                });
                it('should return the sum of the elements of an array', function () {
                    expect(Utils.getSum([1, -2, 3], _.identity)).to.equal(2);
                });
                it('should use the results of the passed iterator', function () {
                    expect(Utils.getSum({ a: 1, b: -2, c: 3 }, Math.abs)).to.equal(6);
                    expect(Utils.getSum([1, -2, 3], Math.abs)).to.equal(6);
                });
                it('should bind the specified context to the iterator function', function () {
                    var context = {},
                        result = chai.runIterator(Utils.getSum, Utils, [[1, -2, 3], chai.ITERATOR, { context: context }], _.identity);
                    expect(result.ret).to.equal(2);
                    expect(result.runs[0].context).to.equal(context);
                    expect(result.runs[1].context).to.equal(context);
                    expect(result.runs[2].context).to.equal(context);
                });
            });

            // ----------------------------------------------------------------

            describe('method "convertLength"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('convertLength');
                });
                it('should convert pixels to a target unit', function () {
                    expect(Utils.convertLength(4800, 'px', 'px')).to.almostEqual(4800);
                    expect(Utils.convertLength(4800, 'px', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertLength(4800, 'px', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertLength(4800, 'px', 'in')).to.almostEqual(50);
                    expect(Utils.convertLength(4800, 'px', 'cm')).to.almostEqual(127);
                    expect(Utils.convertLength(4800, 'px', 'mm')).to.almostEqual(1270);
                });
                it('should convert picas to a target unit', function () {
                    expect(Utils.convertLength(43200, 'pc', 'px')).to.almostEqual(4800);
                    expect(Utils.convertLength(43200, 'pc', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertLength(43200, 'pc', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertLength(43200, 'pc', 'in')).to.almostEqual(50);
                    expect(Utils.convertLength(43200, 'pc', 'cm')).to.almostEqual(127);
                    expect(Utils.convertLength(43200, 'pc', 'mm')).to.almostEqual(1270);
                });
                it('should convert points to a target unit', function () {
                    expect(Utils.convertLength(2653200, 'pt', 'px')).to.almostEqual(3537600);
                    expect(Utils.convertLength(2653200, 'pt', 'pc')).to.almostEqual(31838400);
                    expect(Utils.convertLength(2653200, 'pt', 'pt')).to.almostEqual(2653200);
                    expect(Utils.convertLength(2653200, 'pt', 'in')).to.almostEqual(36850);
                    expect(Utils.convertLength(2653200, 'pt', 'cm')).to.almostEqual(93599);
                    expect(Utils.convertLength(2653200, 'pt', 'mm')).to.almostEqual(935990);
                });
                it('should convert inches to a target unit', function () {
                    expect(Utils.convertLength(50, 'in', 'px')).to.almostEqual(4800);
                    expect(Utils.convertLength(50, 'in', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertLength(50, 'in', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertLength(50, 'in', 'in')).to.almostEqual(50);
                    expect(Utils.convertLength(50, 'in', 'cm')).to.almostEqual(127);
                    expect(Utils.convertLength(50, 'in', 'mm')).to.almostEqual(1270);
                });
                it('should convert centimeters to a target unit', function () {
                    expect(Utils.convertLength(127, 'cm', 'px')).to.almostEqual(4800);
                    expect(Utils.convertLength(127, 'cm', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertLength(127, 'cm', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertLength(127, 'cm', 'in')).to.almostEqual(50);
                    expect(Utils.convertLength(127, 'cm', 'cm')).to.almostEqual(127);
                    expect(Utils.convertLength(127, 'cm', 'mm')).to.almostEqual(1270);
                });
                it('should convert millimeters to a target unit', function () {
                    expect(Utils.convertLength(1270, 'mm', 'px')).to.almostEqual(4800);
                    expect(Utils.convertLength(1270, 'mm', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertLength(1270, 'mm', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertLength(1270, 'mm', 'in')).to.almostEqual(50);
                    expect(Utils.convertLength(1270, 'mm', 'cm')).to.almostEqual(127);
                    expect(Utils.convertLength(1270, 'mm', 'mm')).to.almostEqual(1270);
                });
                it('should round to the specified precision', function () {
                    expect(Utils.convertLength(1, 'in', 'mm', 0.01)).to.equal(25.4);
                    expect(Utils.convertLength(1, 'in', 'mm', 0.1)).to.equal(25.4);
                    expect(Utils.convertLength(1, 'in', 'mm', 1)).to.equal(25);
                    expect(Utils.convertLength(1, 'in', 'mm', 10)).to.equal(30);
                    expect(Utils.convertLength(1, 'in', 'mm', 100)).to.equal(0);
                });
            });

            // ----------------------------------------------------------------

            describe('method "convertLengthToHmm"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('convertLengthToHmm');
                });
                it('should convert a source unit to 1/100 mm', function () {
                    expect(Utils.convertLengthToHmm(216, 'px')).to.equal(5715);
                    expect(Utils.convertLengthToHmm(216, 'pc')).to.equal(635);
                    expect(Utils.convertLengthToHmm(216, 'pt')).to.equal(7620);
                    expect(Utils.convertLengthToHmm(216, 'in')).to.equal(548640);
                    expect(Utils.convertLengthToHmm(216, 'cm')).to.equal(216000);
                    expect(Utils.convertLengthToHmm(216, 'mm')).to.equal(21600);
                });
                it('should round to integers', function () {
                    expect(Utils.convertLengthToHmm(0.015, 'mm')).to.equal(2);
                });
            });

            // ----------------------------------------------------------------

            describe('method "convertHmmToLength"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('convertHmmToLength');
                });
                it('should convert 1/100 mm to a target unit', function () {
                    expect(Utils.convertHmmToLength(127000, 'px')).to.almostEqual(4800);
                    expect(Utils.convertHmmToLength(127000, 'pc')).to.almostEqual(43200);
                    expect(Utils.convertHmmToLength(127000, 'pt')).to.almostEqual(3600);
                    expect(Utils.convertHmmToLength(127000, 'in')).to.almostEqual(50);
                    expect(Utils.convertHmmToLength(127000, 'cm')).to.almostEqual(127);
                    expect(Utils.convertHmmToLength(127000, 'mm')).to.almostEqual(1270);
                });
                it('should round to the specified precision', function () {
                    expect(Utils.convertHmmToLength(2540, 'mm', 0.01)).to.equal(25.4);
                    expect(Utils.convertHmmToLength(2540, 'mm', 0.1)).to.equal(25.4);
                    expect(Utils.convertHmmToLength(2540, 'mm', 1)).to.equal(25);
                    expect(Utils.convertHmmToLength(2540, 'mm', 10)).to.equal(30);
                    expect(Utils.convertHmmToLength(2540, 'mm', 100)).to.equal(0);
                });
            });

            // ----------------------------------------------------------------

            describe('method "convertCssLength"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('convertCssLength');
                });
                it('should convert pixels to a target unit', function () {
                    expect(Utils.convertCssLength('4800px', 'px')).to.almostEqual(4800);
                    expect(Utils.convertCssLength('4800px', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertCssLength('4800px', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertCssLength('4800px', 'in')).to.almostEqual(50);
                    expect(Utils.convertCssLength('4800px', 'cm')).to.almostEqual(127);
                    expect(Utils.convertCssLength('4800px', 'mm')).to.almostEqual(1270);
                });
                it('should convert picas to a target unit', function () {
                    expect(Utils.convertCssLength('43200pc', 'px')).to.almostEqual(4800);
                    expect(Utils.convertCssLength('43200pc', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertCssLength('43200pc', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertCssLength('43200pc', 'in')).to.almostEqual(50);
                    expect(Utils.convertCssLength('43200pc', 'cm')).to.almostEqual(127);
                    expect(Utils.convertCssLength('43200pc', 'mm')).to.almostEqual(1270);
                });
                it('should convert points to a target unit', function () {
                    expect(Utils.convertCssLength('2653200pt', 'px')).to.almostEqual(3537600);
                    expect(Utils.convertCssLength('2653200pt', 'pc')).to.almostEqual(31838400);
                    expect(Utils.convertCssLength('2653200pt', 'pt')).to.almostEqual(2653200);
                    expect(Utils.convertCssLength('2653200pt', 'in')).to.almostEqual(36850);
                    expect(Utils.convertCssLength('2653200pt', 'cm')).to.almostEqual(93599);
                    expect(Utils.convertCssLength('2653200pt', 'mm')).to.almostEqual(935990);
                });
                it('should convert inches to a target unit', function () {
                    expect(Utils.convertCssLength('50in', 'px')).to.almostEqual(4800);
                    expect(Utils.convertCssLength('50in', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertCssLength('50in', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertCssLength('50in', 'in')).to.almostEqual(50);
                    expect(Utils.convertCssLength('50in', 'cm')).to.almostEqual(127);
                    expect(Utils.convertCssLength('50in', 'mm')).to.almostEqual(1270);
                });
                it('should convert centimeters to a target unit', function () {
                    expect(Utils.convertCssLength('127cm', 'px')).to.almostEqual(4800);
                    expect(Utils.convertCssLength('127cm', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertCssLength('127cm', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertCssLength('127cm', 'in')).to.almostEqual(50);
                    expect(Utils.convertCssLength('127cm', 'cm')).to.almostEqual(127);
                    expect(Utils.convertCssLength('127cm', 'mm')).to.almostEqual(1270);
                });
                it('should convert millimeters to a target unit', function () {
                    expect(Utils.convertCssLength('1270mm', 'px')).to.almostEqual(4800);
                    expect(Utils.convertCssLength('1270mm', 'pc')).to.almostEqual(43200);
                    expect(Utils.convertCssLength('1270mm', 'pt')).to.almostEqual(3600);
                    expect(Utils.convertCssLength('1270mm', 'in')).to.almostEqual(50);
                    expect(Utils.convertCssLength('1270mm', 'cm')).to.almostEqual(127);
                    expect(Utils.convertCssLength('1270mm', 'mm')).to.almostEqual(1270);
                });
                it('should round to the specified precision', function () {
                    expect(Utils.convertCssLength('1in', 'mm', 0.01)).to.equal(25.4);
                    expect(Utils.convertCssLength('1in', 'mm', 0.1)).to.equal(25.4);
                    expect(Utils.convertCssLength('1in', 'mm', 1)).to.equal(25);
                    expect(Utils.convertCssLength('1in', 'mm', 10)).to.equal(30);
                    expect(Utils.convertCssLength('1in', 'mm', 100)).to.equal(0);
                });
            });

            // ----------------------------------------------------------------

            describe('method "convertCssLengthToHmm"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('convertCssLengthToHmm');
                });
                it('should convert a source unit to 1/100 mm', function () {
                    expect(Utils.convertCssLengthToHmm('216px')).to.equal(5715);
                    expect(Utils.convertCssLengthToHmm('216pc')).to.equal(635);
                    expect(Utils.convertCssLengthToHmm('216pt')).to.equal(7620);
                    expect(Utils.convertCssLengthToHmm('216in')).to.equal(548640);
                    expect(Utils.convertCssLengthToHmm('216cm')).to.equal(216000);
                    expect(Utils.convertCssLengthToHmm('216mm')).to.equal(21600);
                });
                it('should round to integers', function () {
                    expect(Utils.convertCssLengthToHmm('0.015mm')).to.equal(2);
                });
            });

            // ----------------------------------------------------------------

            describe('method "convertHmmToCssLength"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('convertHmmToCssLength');
                });
                it('should convert 1/100 mm to a target unit', function () {
                    expect(Utils.convertHmmToCssLength(127000, 'px', 1)).to.equal('4800px');
                    expect(Utils.convertHmmToCssLength(127000, 'pc', 1)).to.equal('43200pc');
                    expect(Utils.convertHmmToCssLength(127000, 'pt', 1)).to.equal('3600pt');
                    expect(Utils.convertHmmToCssLength(127000, 'in', 1)).to.equal('50in');
                    expect(Utils.convertHmmToCssLength(127000, 'cm', 1)).to.equal('127cm');
                    expect(Utils.convertHmmToCssLength(127000, 'mm', 1)).to.equal('1270mm');
                });
                it('should round to the specified precision', function () {
                    expect(Utils.convertHmmToCssLength(2540, 'mm', 0.01)).to.equal('25.4mm');
                    expect(Utils.convertHmmToCssLength(2540, 'mm', 0.1)).to.equal('25.4mm');
                    expect(Utils.convertHmmToCssLength(2540, 'mm', 1)).to.equal('25mm');
                    expect(Utils.convertHmmToCssLength(2540, 'mm', 10)).to.equal('30mm');
                    expect(Utils.convertHmmToCssLength(2540, 'mm', 100)).to.equal('0mm');
                });
            });

            // ----------------------------------------------------------------

            describe('method "containsToken"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('containsToken');
                });
                it('should return true for an existing token', function () {
                    expect(Utils.containsToken('a b   c\td ef ghi', 'a')).to.be['true'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'b')).to.be['true'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'c')).to.be['true'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'd')).to.be['true'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'ef')).to.be['true'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'ghi')).to.be['true'];
                });
                it('should return false for a missing token', function () {
                    expect(Utils.containsToken('a b   c\td ef ghi', 'aa')).to.be['false'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'ab')).to.be['false'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'de')).to.be['false'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'e')).to.be['false'];
                    expect(Utils.containsToken('a b   c\td ef ghi', 'j')).to.be['false'];
                });
            });

            // ----------------------------------------------------------------

            describe('method "addToken"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('addToken');
                });
                it('should add a missing token', function () {
                    expect(Utils.addToken('a b   c\td', 'e')).to.equal('a b c d e');
                    expect(Utils.addToken('', 'e')).to.equal('e');
                });
                it('should skip an existing token', function () {
                    expect(Utils.addToken('a b   c\td', 'a')).to.equal('a b c d');
                    expect(Utils.addToken('a b   c\td', 'b')).to.equal('a b c d');
                    expect(Utils.addToken('a b   c\td', 'c')).to.equal('a b c d');
                    expect(Utils.addToken('a b   c\td', 'd')).to.equal('a b c d');
                });
                it('should handle the special "nothing" token', function () {
                    expect(Utils.addToken('a b', 'a', 'none')).to.equal('a b');
                    expect(Utils.addToken('a b', 'e', 'none')).to.equal('a b e');
                    expect(Utils.addToken('', 'a', 'none')).to.equal('a');
                    expect(Utils.addToken('a', 'a', 'none')).to.equal('a');
                    expect(Utils.addToken('none', 'a', 'none')).to.equal('a');
                    expect(Utils.addToken('a none', 'a', 'none')).to.equal('a');
                    expect(Utils.addToken('a b none', 'a', 'none')).to.equal('a b');
                    expect(Utils.addToken('a b none', 'e', 'none')).to.equal('a b e');
                });
            });

            // ----------------------------------------------------------------

            describe('method "removeToken"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('removeToken');
                });
                it('should skip a missing token', function () {
                    expect(Utils.removeToken('a b   c\td', 'e')).to.equal('a b c d');
                    expect(Utils.removeToken('', 'e')).to.equal('');
                });
                it('should remove an existing token', function () {
                    expect(Utils.removeToken('a b   c\td', 'a')).to.equal('b c d');
                    expect(Utils.removeToken('a b   c\td', 'b')).to.equal('a c d');
                    expect(Utils.removeToken('a b   c\td', 'c')).to.equal('a b d');
                    expect(Utils.removeToken('a b   c\td', 'd')).to.equal('a b c');
                });
                it('should handle the special "nothing" token', function () {
                    expect(Utils.removeToken('a b', 'a', 'none')).to.equal('b');
                    expect(Utils.removeToken('a b', 'e', 'none')).to.equal('a b');
                    expect(Utils.removeToken('', 'a', 'none')).to.equal('none');
                    expect(Utils.removeToken('a', 'a', 'none')).to.equal('none');
                    expect(Utils.removeToken('none', 'a', 'none')).to.equal('none');
                    expect(Utils.removeToken('a none', 'a', 'none')).to.equal('none');
                    expect(Utils.removeToken('a b none', 'a', 'none')).to.equal('b');
                    expect(Utils.removeToken('a b none', 'e', 'none')).to.equal('a b');
                });
            });

            // ----------------------------------------------------------------

            describe('method "toggleToken"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('toggleToken');
                });
                it('should add a missing token', function () {
                    expect(Utils.toggleToken('a b   c\td', 'e', true)).to.equal('a b c d e');
                    expect(Utils.toggleToken('', 'e', true)).to.equal('e');
                });
                it('should skip a missing token', function () {
                    expect(Utils.toggleToken('a b   c\td', 'e', false)).to.equal('a b c d');
                    expect(Utils.toggleToken('', 'e', false)).to.equal('');
                });
                it('should skip an existing token', function () {
                    expect(Utils.toggleToken('a b   c\td', 'a', true)).to.equal('a b c d');
                    expect(Utils.toggleToken('a b   c\td', 'b', true)).to.equal('a b c d');
                    expect(Utils.toggleToken('a b   c\td', 'c', true)).to.equal('a b c d');
                    expect(Utils.toggleToken('a b   c\td', 'd', true)).to.equal('a b c d');
                });
                it('should remove an existing token', function () {
                    expect(Utils.toggleToken('a b   c\td', 'a', false)).to.equal('b c d');
                    expect(Utils.toggleToken('a b   c\td', 'b', false)).to.equal('a c d');
                    expect(Utils.toggleToken('a b   c\td', 'c', false)).to.equal('a b d');
                    expect(Utils.toggleToken('a b   c\td', 'd', false)).to.equal('a b c');
                });
                it('should handle the special "nothing" token when adding a token', function () {
                    expect(Utils.toggleToken('a b', 'a', true, 'none')).to.equal('a b');
                    expect(Utils.toggleToken('a b', 'e', true, 'none')).to.equal('a b e');
                    expect(Utils.toggleToken('', 'a', true, 'none')).to.equal('a');
                    expect(Utils.toggleToken('a', 'a', true, 'none')).to.equal('a');
                    expect(Utils.toggleToken('none', 'a', true, 'none')).to.equal('a');
                    expect(Utils.toggleToken('a none', 'a', true, 'none')).to.equal('a');
                    expect(Utils.toggleToken('a b none', 'a', true, 'none')).to.equal('a b');
                    expect(Utils.toggleToken('a b none', 'e', true, 'none')).to.equal('a b e');
                });
                it('should handle the special "nothing" token when removing a token', function () {
                    expect(Utils.toggleToken('a b', 'a', false, 'none')).to.equal('b');
                    expect(Utils.toggleToken('a b', 'e', false, 'none')).to.equal('a b');
                    expect(Utils.toggleToken('', 'a', false, 'none')).to.equal('none');
                    expect(Utils.toggleToken('a', 'a', false, 'none')).to.equal('none');
                    expect(Utils.toggleToken('none', 'a', false, 'none')).to.equal('none');
                    expect(Utils.toggleToken('a none', 'a', false, 'none')).to.equal('none');
                    expect(Utils.toggleToken('a b none', 'a', false, 'none')).to.equal('b');
                    expect(Utils.toggleToken('a b none', 'e', false, 'none')).to.equal('a b');
                });
            });

            // ----------------------------------------------------------------

            describe('method "replaceSubString"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('replaceSubString');
                });
                it('should replace a substring', function () {
                    expect(Utils.replaceSubString('abcdef', 0, 1, 'ghi')).to.equal('ghibcdef');
                    expect(Utils.replaceSubString('abcdef', 1, 5, 'ghi')).to.equal('aghif');
                    expect(Utils.replaceSubString('abcdef', 5, 6, 'ghi')).to.equal('abcdeghi');
                    expect(Utils.replaceSubString('abcdef', 5, 9, 'ghi')).to.equal('abcdeghi');
                    expect(Utils.replaceSubString('abcdef', 8, 9, 'ghi')).to.equal('abcdefghi');
                    expect(Utils.replaceSubString('abcdef', 3, 3, 'ghi')).to.equal('abcghidef');
                });
            });

            // ----------------------------------------------------------------

            describe('method "repeatString"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('repeatString');
                });
                it('should repeat the passed string', function () {
                    expect(Utils.repeatString('ab', 0)).to.equal('');
                    expect(Utils.repeatString('ab', 1)).to.equal('ab');
                    expect(Utils.repeatString('ab', 2)).to.equal('abab');
                    expect(Utils.repeatString('ab', 3)).to.equal('ababab');
                    expect(Utils.repeatString('ab', 4)).to.equal('abababab');
                    expect(Utils.repeatString('ab', 5)).to.equal('ababababab');
                    expect(Utils.repeatString('ab', 6)).to.equal('abababababab');
                    expect(Utils.repeatString('ab', 7)).to.equal('ababababababab');
                    expect(Utils.repeatString('ab', 8)).to.equal('abababababababab');
                });
                it('should repeat the passed string with a separator', function () {
                    expect(Utils.repeatString('ab', 0, 'c')).to.equal('');
                    expect(Utils.repeatString('ab', 1, 'c')).to.equal('ab');
                    expect(Utils.repeatString('ab', 2, 'c')).to.equal('abcab');
                    expect(Utils.repeatString('ab', 3, 'c')).to.equal('abcabcab');
                    expect(Utils.repeatString('ab', 4, 'c')).to.equal('abcabcabcab');
                    expect(Utils.repeatString('ab', 5, 'c')).to.equal('abcabcabcabcab');
                    expect(Utils.repeatString('ab', 6, 'c')).to.equal('abcabcabcabcabcab');
                    expect(Utils.repeatString('ab', 7, 'c')).to.equal('abcabcabcabcabcabcab');
                    expect(Utils.repeatString('ab', 8, 'c')).to.equal('abcabcabcabcabcabcabcab');
                });
                it('should repeat the empty string', function () {
                    expect(Utils.repeatString('', 0)).to.equal('');
                    expect(Utils.repeatString('', 1)).to.equal('');
                    expect(Utils.repeatString('', 2)).to.equal('');
                    expect(Utils.repeatString('', 3)).to.equal('');
                    expect(Utils.repeatString('', 4)).to.equal('');
                    expect(Utils.repeatString('', 5)).to.equal('');
                    expect(Utils.repeatString('', 6)).to.equal('');
                    expect(Utils.repeatString('', 7)).to.equal('');
                    expect(Utils.repeatString('', 8)).to.equal('');
                });
                it('should repeat the empty string with a separator', function () {
                    expect(Utils.repeatString('', 0, 'c')).to.equal('');
                    expect(Utils.repeatString('', 1, 'c')).to.equal('');
                    expect(Utils.repeatString('', 2, 'c')).to.equal('c');
                    expect(Utils.repeatString('', 3, 'c')).to.equal('cc');
                    expect(Utils.repeatString('', 4, 'c')).to.equal('ccc');
                    expect(Utils.repeatString('', 5, 'c')).to.equal('cccc');
                    expect(Utils.repeatString('', 6, 'c')).to.equal('ccccc');
                    expect(Utils.repeatString('', 7, 'c')).to.equal('cccccc');
                    expect(Utils.repeatString('', 8, 'c')).to.equal('ccccccc');
                });
            });

            // ----------------------------------------------------------------

            describe('method "trimString"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('trimString');
                });
                it('should trim whitespace at the beginning', function () {
                    expect(Utils.trimString('   abcd')).to.equal('abcd');
                    expect(Utils.trimString('\xa0 \xa0abcd')).to.equal('abcd');
                    expect(Utils.trimString('\u2028 \u2029abcd')).to.equal('abcd');
                    expect(Utils.trimString('\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0fabcd')).to.equal('abcd');
                    expect(Utils.trimString('\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1fabcd')).to.equal('abcd');
                    expect(Utils.trimString('\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8fabcd')).to.equal('abcd');
                    expect(Utils.trimString('\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9fabcd')).to.equal('abcd');
                });
                it('should trim whitespace at the end', function () {
                    expect(Utils.trimString('abcd   ')).to.equal('abcd');
                    expect(Utils.trimString('abcd\xa0 \xa0')).to.equal('abcd');
                    expect(Utils.trimString('abcd\u2028 \u2029')).to.equal('abcd');
                    expect(Utils.trimString('abcd\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f')).to.equal('abcd');
                    expect(Utils.trimString('abcd\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f')).to.equal('abcd');
                    expect(Utils.trimString('abcd\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f')).to.equal('abcd');
                    expect(Utils.trimString('abcd\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f')).to.equal('abcd');
                });
                it('should trim whitespace at both sides', function () {
                    expect(Utils.trimString('   abcd   ')).to.equal('abcd');
                    expect(Utils.trimString('\xa0 \xa0abcd\xa0 \xa0')).to.equal('abcd');
                    expect(Utils.trimString('\u2028 \u2029abcd\u2028 \u2029')).to.equal('abcd');
                });
                it('should not trim whitespace in the middle', function () {
                    expect(Utils.trimString('ab  cd')).to.equal('ab  cd');
                });
            });

            // ----------------------------------------------------------------

            describe('method "cleanString"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('cleanString');
                });
                it('should replace all kinds of non-printable characters with whitespace', function () {
                    expect(Utils.cleanString('a\x00b\x01c\x02d\x03e\x04f\x05g\x06h\x07i\x08j\x09k\x0al\x0bm\x0cn\x0do\x0ep\x0fq')).to.equal('a b c d e f g h i j k l m n o p q');
                    expect(Utils.cleanString('a\x10b\x11c\x12d\x13e\x14f\x15g\x16h\x17i\x18j\x19k\x1al\x1bm\x1cn\x1do\x1ep\x1fq')).to.equal('a b c d e f g h i j k l m n o p q');
                    expect(Utils.cleanString('a\x80b\x81c\x82d\x83e\x84f\x85g\x86h\x87i\x88j\x89k\x8al\x8bm\x8cn\x8do\x8ep\x8fq')).to.equal('a b c d e f g h i j k l m n o p q');
                    expect(Utils.cleanString('a\x90b\x91c\x92d\x93e\x94f\x95g\x96h\x97i\x98j\x99k\x9al\x9bm\x9cn\x9do\x9ep\x9fq')).to.equal('a b c d e f g h i j k l m n o p q');
                });
            });

            // ----------------------------------------------------------------

            describe('method "trimAndCleanString"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('trimAndCleanString');
                });
                it('should trim and clean the string', function () {
                    expect(Utils.trimAndCleanString('\xa0  a\x00b\x01c\x02d\x03e\x04f\x05g\x06h\x07i\x08j\x09k\x0al\x0bm\x0cn\x0do\x0ep\x0fq  \xa0')).to.equal('a b c d e f g h i j k l m n o p q');
                });
            });

            // ----------------------------------------------------------------

            describe('method "capitalize"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('capitalize');
                });
                it('should capitalize the first letter of the string', function () {
                    expect(Utils.capitalize('abc def ghi jkl')).to.equal('Abc def ghi jkl');
                    expect(Utils.capitalize('Abc def ghi jkl')).to.equal('Abc def ghi jkl');
                });
            });

            // ----------------------------------------------------------------

            describe('method "capitalizeWords"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('capitalizeWords');
                });
                it('should capitalize the first letter of the string', function () {
                    expect(Utils.capitalizeWords('abc def ghi jkl')).to.equal('Abc Def Ghi Jkl');
                    expect(Utils.capitalizeWords('Abc def ghi jkl')).to.equal('Abc Def Ghi Jkl');
                });
            });

            // ----------------------------------------------------------------

            describe('method "escapeHTML"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('escapeHTML');
                });
                it('should escape all special characters', function () {
                    expect(Utils.escapeHTML('abc&def')).to.equal('abc&amp;def');
                    expect(Utils.escapeHTML('abc<def')).to.equal('abc&lt;def');
                    expect(Utils.escapeHTML('abc>def')).to.equal('abc&gt;def');
                    expect(Utils.escapeHTML('abc/def')).to.equal('abc&#47;def');
                    expect(Utils.escapeHTML('abc"def')).to.equal('abc&quot;def');
                    expect(Utils.escapeHTML('abc\'def')).to.equal('abc&#39;def');
                    expect(Utils.escapeHTML('abc\xa0def')).to.equal('abc&nbsp;def');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIsoDateString"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getIsoDateString');
                });
                it('should create an iso conform date string', function () {
                    expect(Utils.getIsoDateString()).to.be.a('string');
                    expect(Utils.getIsoDateString().length).to.equal(24);
                    expect(Utils.getIsoDateString({ useSeconds: true }).length).to.equal(24);
                    expect(Utils.getIsoDateString({ useSeconds: false })).to.be.a('string');
                    expect(Utils.getIsoDateString({ useSeconds: false }).length).to.equal(20);
                    expect(Utils.getIsoDateString().match(/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{3}Z$/)[0]).to.be.a('string');
                    expect(Utils.getIsoDateString({ useSeconds: false }).match(/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:00Z$/)[0]).to.be.a('string');
                });
            });

            // ----------------------------------------------------------------

            var OPT = {
                    s: 'a',
                    se: '',
                    i: 1,
                    p: 2.5,
                    bt: true,
                    bf: false,
                    n: null,
                    u: undefined,
                    a: [1],
                    ae: [],
                    o: { v: 1 },
                    oe: {},
                    f: $.noop,
                    r: /a/
                };

            describe('method "getOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getOption');
                });
                it('should return the specified property', function () {
                    expect(Utils.getOption(OPT, 's')).to.equal('a');
                    expect(Utils.getOption(OPT, 'se')).to.equal('');
                    expect(Utils.getOption(OPT, 'i')).to.equal(1);
                    expect(Utils.getOption(OPT, 'p')).to.equal(2.5);
                    expect(Utils.getOption(OPT, 'bt')).to.equal(true);
                    expect(Utils.getOption(OPT, 'bf')).to.equal(false);
                    expect(Utils.getOption(OPT, 'n')).to.be['null'];
                    expect(Utils.getOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getOption(OPT, 'a')).to.deep.equal([1]);
                    expect(Utils.getOption(OPT, 'ae')).to.deep.equal([]);
                    expect(Utils.getOption(OPT, 'o')).to.deep.equal({ v: 1 });
                    expect(Utils.getOption(OPT, 'oe')).to.deep.equal({});
                    expect(Utils.getOption(OPT, 'f')).to.equal($.noop);
                    expect(Utils.getOption(OPT, 'r')).to.deep.equal(/a/);
                });
                it('should return undefined for missing properties', function () {
                    expect(Utils.getOption({}, 'x')).to.be['undefined'];
                    expect(Utils.getOption({ X: 1 }, 'x')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getOption(null, 'x')).to.be['undefined'];
                    expect(Utils.getOption(undefined, 'x')).to.be['undefined'];
                    expect(Utils.getOption('a', 'x')).to.be['undefined'];
                    expect(Utils.getOption(1, 'x')).to.be['undefined'];
                    expect(Utils.getOption(true, 'x')).to.be['undefined'];
                    expect(Utils.getOption([1], 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getOption({}, 'x', 'a')).to.equal('a');
                    expect(Utils.getOption({}, 'x', 1)).to.equal(1);
                    expect(Utils.getOption({}, 'x', true)).to.equal(true);
                    expect(Utils.getOption({}, 'x', null)).to.be['null'];
                    expect(Utils.getOption({}, 'x', [1])).to.deep.equal([1]);
                    expect(Utils.getOption({}, 'x', { v: 1 })).to.deep.equal({ v: 1 });
                    expect(Utils.getOption({}, 'x', $.noop)).to.equal($.noop);
                    expect(Utils.getOption({}, 'x', /a/)).to.deep.equal(/a/);
                    expect(Utils.getOption(null, 'x', 'a')).to.equal('a');
                });
            });

            // ----------------------------------------------------------------

            describe('method "getStringOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getStringOption');
                });
                it('should return the specified string property', function () {
                    expect(Utils.getStringOption(OPT, 's')).to.equal('a');
                    expect(Utils.getStringOption(OPT, 'se')).to.equal('');
                });
                it('should return undefined for other properties', function () {
                    expect(Utils.getStringOption(OPT, 'i')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'p')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'bt')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'bf')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'n')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'a')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'ae')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'o')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'oe')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'f')).to.be['undefined'];
                    expect(Utils.getStringOption(OPT, 'r')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getStringOption(null, 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getStringOption(OPT, 'i', 'b')).to.equal('b');
                    expect(Utils.getStringOption(OPT, 'i', 2.5)).to.equal(2.5);
                    expect(Utils.getStringOption({}, 'x', 'b')).to.equal('b');
                    expect(Utils.getStringOption({}, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getStringOption(null, 'x', 'b')).to.equal('b');
                    expect(Utils.getStringOption(null, 'x', 2.5)).to.equal(2.5);
                });
                it('should return the default value for empty strings', function () {
                    expect(Utils.getStringOption(OPT, 's', 'b', true)).to.equal('a');
                    expect(Utils.getStringOption(OPT, 'se', 'b', true)).to.equal('b');
                    expect(Utils.getStringOption(OPT, 'se', 2.5, true)).to.equal(2.5);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getBooleanOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getBooleanOption');
                });
                it('should return the specified Boolean property', function () {
                    expect(Utils.getBooleanOption(OPT, 'bt')).to.equal(true);
                    expect(Utils.getBooleanOption(OPT, 'bf')).to.equal(false);
                });
                it('should return undefined for other properties', function () {
                    expect(Utils.getBooleanOption(OPT, 's')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'se')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'i')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'p')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'n')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'a')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'ae')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'o')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'oe')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'f')).to.be['undefined'];
                    expect(Utils.getBooleanOption(OPT, 'r')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getBooleanOption(null, 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getBooleanOption(OPT, 'i', true)).to.equal(true);
                    expect(Utils.getBooleanOption(OPT, 'i', 2.5)).to.equal(2.5);
                    expect(Utils.getBooleanOption({}, 'x', true)).to.equal(true);
                    expect(Utils.getBooleanOption({}, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getBooleanOption(null, 'x', true)).to.equal(true);
                    expect(Utils.getBooleanOption(null, 'x', 2.5)).to.equal(2.5);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getNumberOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getNumberOption');
                });
                it('should return the specified number property', function () {
                    expect(Utils.getNumberOption(OPT, 'i')).to.equal(1);
                    expect(Utils.getNumberOption(OPT, 'p')).to.equal(2.5);
                });
                it('should return undefined for other properties', function () {
                    expect(Utils.getNumberOption(OPT, 's')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'se')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'bt')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'bf')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'n')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'a')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'ae')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'o')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'oe')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'f')).to.be['undefined'];
                    expect(Utils.getNumberOption(OPT, 'r')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getNumberOption(null, 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getNumberOption(OPT, 's', 2.5)).to.equal(2.5);
                    expect(Utils.getNumberOption(OPT, 's', true)).to.equal(true);
                    expect(Utils.getNumberOption({}, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getNumberOption({}, 'x', true)).to.equal(true);
                    expect(Utils.getNumberOption(null, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getNumberOption(null, 'x', true)).to.equal(true);
                });
                it('should return the minimum value', function () {
                    expect(Utils.getNumberOption(OPT, 'p', 1, 0)).to.equal(2.5);
                    expect(Utils.getNumberOption(OPT, 'p', 1, 0, 5)).to.equal(2.5);
                    expect(Utils.getNumberOption(OPT, 'p', 1, 4)).to.equal(4);
                    expect(Utils.getNumberOption(OPT, 'p', 1, 4, 5)).to.equal(4);
                });
                it('should return the maximum value', function () {
                    expect(Utils.getNumberOption(OPT, 'p', 1, 0, 4)).to.equal(2.5);
                    expect(Utils.getNumberOption(OPT, 'p', 1, 0, 1)).to.equal(1);
                });
                it('should round to the specified precision', function () {
                    expect(Utils.getNumberOption(OPT, 'p', 1, 0, 4, 1)).to.equal(3);
                    expect(Utils.getNumberOption(OPT, 'p', 1, 0, 4, 0.2)).to.equal(2.6);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getIntegerOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getIntegerOption');
                });
                it('should return the specified integer property', function () {
                    expect(Utils.getIntegerOption(OPT, 'i')).to.equal(1);
                    expect(Utils.getIntegerOption(OPT, 'p')).to.equal(3);
                });
                it('should return undefined for other properties', function () {
                    expect(Utils.getIntegerOption(OPT, 's')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'se')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'bt')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'bf')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'n')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'a')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'ae')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'o')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'oe')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'f')).to.be['undefined'];
                    expect(Utils.getIntegerOption(OPT, 'r')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getIntegerOption(null, 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getIntegerOption(OPT, 's', 2.5)).to.equal(3);
                    expect(Utils.getIntegerOption(OPT, 's', true)).to.equal(true);
                    expect(Utils.getIntegerOption({}, 'x', 2.5)).to.equal(3);
                    expect(Utils.getIntegerOption({}, 'x', true)).to.equal(true);
                    expect(Utils.getIntegerOption(null, 'x', 2.5)).to.equal(3);
                    expect(Utils.getIntegerOption(null, 'x', true)).to.equal(true);
                });
                it('should return the minimum value', function () {
                    expect(Utils.getIntegerOption(OPT, 'p', 1, 0)).to.equal(3);
                    expect(Utils.getIntegerOption(OPT, 'p', 1, 0, 5)).to.equal(3);
                    expect(Utils.getIntegerOption(OPT, 'p', 1, 4)).to.equal(4);
                    expect(Utils.getIntegerOption(OPT, 'p', 1, 4, 5)).to.equal(4);
                });
                it('should return the maximum value', function () {
                    expect(Utils.getIntegerOption(OPT, 'p', 1, 0, 4)).to.equal(3);
                    expect(Utils.getIntegerOption(OPT, 'p', 1, 0, 1)).to.equal(1);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getObjectOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getObjectOption');
                });
                it('should return the specified object property', function () {
                    expect(Utils.getObjectOption(OPT, 'o')).to.deep.equal({ v: 1 });
                    expect(Utils.getObjectOption(OPT, 'oe')).to.deep.equal({});
                });
                it('should return undefined for other properties', function () {
                    expect(Utils.getObjectOption(OPT, 's')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'se')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'i')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'p')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'bt')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'bf')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'n')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'a')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'ae')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'f')).to.be['undefined'];
                    expect(Utils.getObjectOption(OPT, 'r')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getObjectOption(null, 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getObjectOption(OPT, 's', 2.5)).to.equal(2.5);
                    expect(Utils.getObjectOption(OPT, 's', true)).to.equal(true);
                    expect(Utils.getObjectOption({}, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getObjectOption({}, 'x', true)).to.equal(true);
                    expect(Utils.getObjectOption(null, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getObjectOption(null, 'x', true)).to.equal(true);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getFunctionOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getFunctionOption');
                });
                it('should return the specified function property', function () {
                    expect(Utils.getFunctionOption(OPT, 'f')).to.equal($.noop);
                });
                it('should return undefined for other properties', function () {
                    expect(Utils.getFunctionOption(OPT, 's')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'se')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'i')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'p')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'bt')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'bf')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'n')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'a')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'ae')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'o')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'oe')).to.be['undefined'];
                    expect(Utils.getFunctionOption(OPT, 'r')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getFunctionOption(null, 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getFunctionOption(OPT, 's', 2.5)).to.equal(2.5);
                    expect(Utils.getFunctionOption(OPT, 's', true)).to.equal(true);
                    expect(Utils.getFunctionOption({}, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getFunctionOption({}, 'x', true)).to.equal(true);
                    expect(Utils.getFunctionOption(null, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getFunctionOption(null, 'x', true)).to.equal(true);
                });
            });

            // ----------------------------------------------------------------

            describe('method "getArrayOption"', function () {
                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('getArrayOption');
                });
                it('should return the specified array property', function () {
                    expect(Utils.getArrayOption(OPT, 'a')).to.deep.equal([1]);
                    expect(Utils.getArrayOption(OPT, 'ae')).to.deep.equal([]);
                });
                it('should return undefined for other properties', function () {
                    expect(Utils.getArrayOption(OPT, 's')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'se')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'i')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'p')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'bt')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'bf')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'n')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'u')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'o')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'oe')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'f')).to.be['undefined'];
                    expect(Utils.getArrayOption(OPT, 'r')).to.be['undefined'];
                });
                it('should return undefined for missing object', function () {
                    expect(Utils.getArrayOption(null, 'x')).to.be['undefined'];
                });
                it('should return the default value', function () {
                    expect(Utils.getArrayOption(OPT, 'i', 'b')).to.equal('b');
                    expect(Utils.getArrayOption(OPT, 'i', 2.5)).to.equal(2.5);
                    expect(Utils.getArrayOption({}, 'x', 'b')).to.equal('b');
                    expect(Utils.getArrayOption({}, 'x', 2.5)).to.equal(2.5);
                    expect(Utils.getArrayOption(null, 'x', 'b')).to.equal('b');
                    expect(Utils.getArrayOption(null, 'x', 2.5)).to.equal(2.5);
                });
                it('should return the default value for empty arrays', function () {
                    expect(Utils.getArrayOption(OPT, 'a', [2], true)).to.deep.equal([1]);
                    expect(Utils.getArrayOption(OPT, 'ae', [2], true)).to.deep.equal([2]);
                    expect(Utils.getArrayOption(OPT, 'ae', 2.5, true)).to.equal(2.5);
                });
            });

            // ----------------------------------------------------------------

            describe('method "extendOptions"', function () {

                var O1 = { a: 1, b: 'b', c: [1, 2] }, O2 = { d: true }, O3 = { a: 2, e: null };

                it('should exist', function () {
                    expect(Utils).itself.to.respondTo('extendOptions');
                });
                it('should extend multiple objects', function () {
                    expect(Utils.extendOptions(O1, O2, O3)).to.deep.equal({ a: 2, b: 'b', c: [1, 2], d: true, e: null });
                    expect(Utils.extendOptions(O3, O2, O1)).to.deep.equal({ a: 1, b: 'b', c: [1, 2], d: true, e: null });
                });
                it('should keep the parameters unmodified', function () {
                    expect(O1).to.deep.equal({ a: 1, b: 'b', c: [1, 2] });
                    expect(O2).to.deep.equal({ d: true });
                    expect(O3).to.deep.equal({ a: 2, e: null });
                });
                it('should deeply extend embedded objects', function () {
                    expect(Utils.extendOptions({ a: { b: { c: 1 } } }, { a: { b: { d: 2 }, e: 3 } })).to.deep.equal({ a: { b: { c: 1, d: 2 }, e: 3 } });
                });
                it('should not deeply extend embedded arrays', function () {
                    expect(Utils.extendOptions({ a: [1] }, { a: { b: 1 } })).to.deep.equal({ a: { b: 1 } });
                });
                it('should accept missing parameters', function () {
                    expect(Utils.extendOptions(null, { a: 1 }, undefined, { b: 2 })).to.deep.equal({ a: 1, b: 2 });
                });
            });

            // ----------------------------------------------------------------

            describe('selector "JQ_TEXTNODE_SELECTOR"', function () {
                var span = $('<span>abc</span>');
                it('should exist', function () {
                    expect(Utils.JQ_TEXTNODE_SELECTOR).to.be.a('function');
                });
                it('should match text nodes', function () {
                    expect(span.contents().is(Utils.JQ_TEXTNODE_SELECTOR)).to.be['true'];
                });
                it('should not match other nodes', function () {
                    expect(span.is(Utils.JQ_TEXTNODE_SELECTOR)).to.be['false'];
                    expect($(document).is(Utils.JQ_TEXTNODE_SELECTOR)).to.be['false'];
                    expect($(window).is(Utils.JQ_TEXTNODE_SELECTOR)).to.be['false'];
                });
            });

            // mix-ins ========================================================

            describe('should support the core/event interface"', function () {
                expect(Utils).itself.to.respondTo('on');
                expect(Utils).itself.to.respondTo('off');
                expect(Utils).itself.to.respondTo('one');
                expect(Utils).itself.to.respondTo('trigger');
            });

            // ----------------------------------------------------------------

            describe('should support the logger interface"', function () {
                expect(Utils).itself.to.respondTo('log');
                expect(Utils).itself.to.respondTo('info');
                expect(Utils).itself.to.respondTo('warn');
                expect(Utils).itself.to.respondTo('error');
                expect(Utils).itself.to.respondTo('exception');
            });

            // ----------------------------------------------------------------

        });
    });
});
