/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/basepopup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin'
], function (Utils, TriggerObject, TimerMixin) {

    'use strict';

    // class BasePopup ========================================================

    /**
     * Wrapper class for a DOM node used as pop-up window, shown on top of the
     * application window, and relative to an arbitrary DOM node.
     *
     * Instances of this class trigger the following events:
     * - 'popup:beforeshow'
     *      Before the pop-up node will be shown. If an event handler calls the
     *      method preventDefault() at the event object, the pop-up node will
     *      not be shown.
     * - 'popup:show'
     *      After the pop-up node has been shown.
     * - 'popup:beforehide'
     *      Before the pop-up node will be hidden. In difference to the event
     *      'popup:beforeshow', it is not possible to veto hiding the pop-up
     *      node by calling the method preventDefault() of the event object.
     * - 'popup:hide'
     *      After the pop-up node has been hidden.
     * - 'popup:beforelayout'
     *      Before the contents of the pop-up node, the absolute position of
     *      the anchor node, or the browser window size has changed, and the
     *      position and size of the pop-up node needs to be adjusted.
     * - 'popup:layout'
     *      After the contents of the pop-up node, the absolute position of the
     *      anchor node or the browser window size has changed, and the
     *      position and size of the pop-up node has been adjusted (in
     *      auto-layout mode).
     * - 'popup:beforeclear'
     *      Before the pop-up node will be cleared.
     * - 'popup:busy'
     *      After the pop-up node has switched to busy mode (see method
     *      BasePopup.busy() for details).
     * - 'popup:idle'
     *      After the pop-up node has left the busy mode (see method
     *      BasePopup.idle() for details).
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the pop-up root node.
     *  @param {Boolean} [initOptions.autoLayout=true]
     *      If set to true or omitted, the pop-up node will be positioned and
     *      sized automatically next to the anchor node. If the available space
     *      is not sufficient for the contents of the pop-up node, scroll bars
     *      will be displayed. If set to false, the pop-up node must be sized
     *      and positioned manually inside event handlers of the events
     *      'popup:beforelayout' and 'popup:layout'.
     *  @param {Null|Object|Function} [initOptions.boundingBox=null]
     *      The specification of a rectangle in the document page used to
     *      restrict the position and size of the pop-up node to. Can be one of
     *      the following values:
     *      - {Null}: There is no custom bounding box available. By default,
     *          the entire visible area of the browser window will be used as
     *          bounding box.
     *      - {Object}: A constant rectangle (an object with the properties
     *          'left', 'top', 'width', and 'height'). The position is
     *          interpreted absolutely in the entire document page.
     *      - {HTMLElement|jQuery}: The reference to a DOM element, or a jQuery
     *          collection wrapping a DOM element, whose position and size will
     *          be used as bounding box (uses the visible client area of
     *          scrollable DOM elements). While displaying the pop-up node, the
     *          position of the bounding box will be updated dynamically, in
     *          case the position and/or size of the DOM node changes, e.g. due
     *          to scrolling or zooming.
     *      - {Function}: A callback function that will be invoked every time
     *          before displaying the pop-up node, and that must return a value
     *          of any of the data types described above.
     *  @param {Number} [initOptions.boundingPadding=6]
     *      The minimum distance between the pop-up node and the borders of the
     *      bounding box (see option 'boundingBox' above), in pixels.
     *  @param {Boolean} [initOptions.suppressScrollBars=false]
     *      If set to true, no scroll bars will be added to the root node when
     *      the contents of the pop-up node become too large to be shown
     *      completely.
     *  @param {Null|Object|Function} [initOptions.anchor=null]
     *      The specification of an anchor rectangle used to calculate the
     *      position and size of the pop-up node. The pop-up node will always
     *      be placed next to the anchor (see following options for specifying
     *      the preferred behavior when calculating the position and size of
     *      the pop-up node). Can be any value that is supported by the option
     *      'boundingBox' (in case of DOM elements, the outer size of the
     *      anchor node is used instead of the inner client area). If no anchor
     *      has been specified, the pop-up node will keep its position and will
     *      be expanded to the size of its contents, but it will always kept
     *      inside the bounding box.
     *  @param {Null|Object|Function} [initOptions.anchorBox=null]
     *      The specification of a rectangle used to decide whether the anchor
     *      rectangle (see option 'anchor' above) is still visible. If the
     *      current anchor leaves the anchor box rectangle, the pop-up node
     *      will be closed automatically. Can be any value that is supported by
     *      the option 'boundingBox'. By default, a constant anchor rectangle
     *      is considered to be always visible. DOM anchor nodes are checked
     *      for their own visibility. This option will be ignored, if no anchor
     *      has been specified.
     *  @param {Number} [initOptions.anchorPadding=1]
     *      The distance between pop-up node and anchor rectangle (see option
     *      'anchor' above), in pixels. This option will be ignored, if no
     *      anchor has been specified.
     *  @param {String} [initOptions.anchorBorder='bottom top']
     *      The preferred position of the pop-up node relative to the anchor.
     *      Must be a space-separated list containing the tokens 'left',
     *      'right', 'top', and 'bottom'. The implementation picks the best
     *      position according to the available space around the anchor. If
     *      more than one position can be used (without needing to shrink the
     *      pop-up node), the first matching position token in this list will
     *      be used. This option will be ignored, if no anchor has been
     *      specified.
     *  @param {String} [initOptions.anchorAlign='leading']
     *      The preferred alignment of the pop-up node relative to the anchor.
     *      Must be a space-separated list containing the tokens 'leading',
     *      'trailing', and 'center'. The value 'leading' tries to align the
     *      left borders (if shown above or below) respectively the top borders
     *      (if shown left or right) of the pop-up node and the anchor
     *      position, the value 'trailing' tries to align the right/bottom
     *      borders, and the value 'center' tries to place the pop-up node
     *      centered to the anchor. If multiple alignments have been passed as
     *      space-separated list, the alignment will be used that corresponds
     *      to the preferred position selected from the option 'anchorBorder'.
     *      If the list in 'anchorAlign' is shorter than the list of the option
     *      'anchorBorder', the last existing alignment will be used. This
     *      option will be ignored, if no anchor has been specified.
     *  @param {Null|Object|Function} [initOptions.anchorAlignBox=null]
     *      The specification of a rectangle used to calculate the position of
     *      the pop-up node according to the alignment mode (see option
     *      'anchorAlign' above). Can be any value that is supported by the
     *      option 'boundingBox'. Example: If the pop-up node will be shown
     *      below the anchor rectangle (option 'anchorBorder' has been set to
     *      'bottom'), the vertical position of the pop-up node will be set
     *      below the anchor rectangle, and the horizontal position will be
     *      calculated according to the option 'anchorAlign' and the rectangle
     *      calculated from this option. By default, the anchor position will
     *      be used as alignment box. This option will be ignored, if no anchor
     *      has been specified.
     *  @param {Boolean} [initOptions.coverAnchor=false]
     *      If set to true, the anchor can be covered by the pop-up node if
     *      there is not enough space around the anchor. By default, the pop-up
     *      node will be shrunken in order to fit it next to the anchor without
     *      covering the anchor rectangle. This option will be ignored, if no
     *      anchor has been specified.
     *  @param {Boolean} [initOptions.expandWidth=false]
     *      If set to true, the minimum width of all direct children of the
     *      content node will be set to the width of the anchor rectangle, so
     *      that the pop-up node is always at least as wide as the anchor. Used
     *      in automatic layout mode only (see option 'autoLayout' above).
     *  @param {String} [options.restrictSize='none']
     *      If set to 'width', the width of the content node will be restricted
     *      to the available horizontal space, and the contents may grow
     *      vertically. A vertical scroll bar will be shown, if the contents do
     *      not fit into the available space. If set to 'height', the height of
     *      the content node will be restricted to the available vertical
     *      space, and the contents may grow horizontally. A horizontal scroll
     *      bar will be shown, if the contents do not fit into the available
     *      space. Otherwise, the size of the content node will not be
     *      restricted.
     *  @param {Number} [initOptions.minSize=1]
     *      If specified, the minimum size to be reserved for the pop-up node,
     *      if either its width or height is restricted (see option
     *      'restrictSize' above). This option has no effect, if the size is
     *      not restricted.
     *  @param {Number} [initOptions.maxSize=0x7FFFFFFF]
     *      If specified, the maximum size to be reserved for the pop-up node,
     *      if either its width or height is restricted (see option
     *      'restrictSize' above). This option has no effect, if the size is
     *      not restricted.
     *  @param {Number} [options.scrollPadding=0]
     *      Minimum distance between the borders of the visible area and the
     *      child node used by the method BasePopup.scrollToChildNode().
     *  @param {String} [initOptions.role]
     *      The ARIA role attribute that will be set at the content DOM node.
     *  @param {HTMLElement|jQuery} [initOptions.rootContainerNode]
     *      The DOM container node for the pop-up node. By default, the pop-up
     *      node will be appended into the <body> element of the page.
     *      ATTENTION: Currently, it is not possible to use automatic layout
     *      (see option 'autoLayout') when using a custom container node.
     *      Manual positioning and sizing of the pop-up node is necessary.
     *  @param {Function} [initOptions.prepareLayoutHandler]
     *      A callback function that will be invoked while updating the
     *      position and size of the pop-up node. Will be invoked before
     *      calculating the total size of the pop-up contents, and the position
     *      of the pop-up node relative to the anchor. Will be called in the
     *      context of this instance. Receives the following parameters:
     *      (1) {Object|Null} anchorPosition
     *          The effective anchor position (see option 'anchor' above), or
     *          null, if no anchor is available.
     *      (2) {Object|Null} availableSizes
     *          A map with multiple rectangle sizes (with 'width' and 'height'
     *          properties) available for the contents of the pop-up node
     *          around the anchor, mapped by the border side names 'left',
     *          'right', 'top', and 'bottom'; , or null, if no anchor is
     *          available.
     *      May return an object with the following (optional) properties (the
     *      result will be ignored, if no anchor is available):
     *      - {String} [anchorBorder]
     *          A temporary override for the constructor option 'anchorBorder'.
     *      - {String} [anchorAlign]
     *          A temporary override for the constructor option 'anchorAlign'.
     *  @param {Function} [initOptions.updateLayoutHandler]
     *      A callback function that will be invoked while updating the
     *      position and size of the pop-up node. Will be invoked after the
     *      total size of the pop-up contents has been calculated, and before
     *      finally setting the position and size of the pop-up node. Will be
     *      called in the context of this instance. Receives the following
     *      parameters:
     *      (1) {Object} contentSize
     *          The total size required for the contents of the pop-up node, in
     *          the properties 'width' and 'height'.
     *      (2) {Object} availableSize
     *          The maximum size available for the pop-up contents, according
     *          to the position of the anchor border, in the properties 'width'
     *          and 'height'.
     *      May return a custom content size object in order to influence the
     *      resulting size of the pop-up node.
     */
    function BasePopup(initOptions) {

        var // self reference (the Group instance)
            self = this,

            // the root pop-up DOM node (attribute unselectable='on' is needed for IE to prevent focusing the node when clicking on a scroll bar)
            rootNode = $('<div class="io-ox-office-main popup-container" unselectable="on">'),

            // the content node inside the root node used to calculate the effective node size
            contentNode = $('<div class="popup-content">').appendTo(rootNode),

            // the container for the busy node shown while the pop-up is in busy state
            busyNode = $('<div class="popup-content">'),

            // automatic position and size of the pop-up node
            autoLayout = Utils.getBooleanOption(initOptions, 'autoLayout', true),

            // the bounding box used as position/size restriction (rectangle, DOM node, or callback function)
            boundingBoxSpec = Utils.getOption(initOptions, 'boundingBox', null),

            // distance between pop-up node and the borders of the bounding box
            boundingPadding = Utils.getIntegerOption(initOptions, 'boundingPadding', 6, 0),

            // whether to prevent adding scroll bars to the pop-up node
            suppressScrollBars = Utils.getBooleanOption(initOptions, 'suppressScrollBars', false),

            // the anchor position specification (rectangle, DOM node, or callback function)
            anchorSpec = Utils.getOption(initOptions, 'anchor', null),

            // the anchor box specification (rectangle, DOM node, or callback function)
            anchorBoxSpec = Utils.getOption(initOptions, 'anchorBox', null),

            // distance between pop-up node and anchor node
            anchorPadding = Utils.getIntegerOption(initOptions, 'anchorPadding', 1),

            // preferred border positions of the pop-up node relative to the anchor, as array
            anchorBorders = Utils.getTokenListOption(initOptions, 'anchorBorder', ['bottom', 'top']),

            // alignment of the pop-up node relative to the anchor node, as array (corresponding to 'anchorBorder')
            anchorAlignments = Utils.getTokenListOption(initOptions, 'anchorAlign', ['leading']),

            // the alignment position specification (rectangle, DOM node, or callback function)
            anchorAlignBoxSpec = Utils.getOption(initOptions, 'anchorAlignBox', null),

            // whether the pop-up node is allowed to cover the anchor rectangle
            coverAnchor = Utils.getBooleanOption(initOptions, 'coverAnchor', false),

            // whether to expand the content node width to the anchor width
            expandWidth = Utils.getBooleanOption(initOptions, 'expandWidth', false),

            // whether to restrict the width or height of the content node
            restrictSize = Utils.getStringOption(initOptions, 'restrictSize', 'none'),

            // minimum size for the content node in restricted size mode
            minSize = Utils.getIntegerOption(initOptions, 'minSize', 1, 1),

            // maximum size for the content node in restricted size mode
            maxSize = Utils.getIntegerOption(initOptions, 'maxSize', 0x7FFFFFFF, minSize),

            // padding when scrolling to a child node of this pop-up node
            scrollPadding = Utils.getIntegerOption(initOptions, 'scrollPadding', 0),

            // ARIA role
            ariaRole = Utils.getStringOption(initOptions, 'role', ''),

            // the container node for this pop-up node
            rootContainerNode = $(Utils.getOption(initOptions, 'rootContainerNode', window.document.body)),

            // callback to prepare layouting (calculation of position and size of this pop-up node)
            prepareLayoutHandler = Utils.getFunctionOption(initOptions, 'prepareLayoutHandler', $.noop),

            // callback to influence position and size of the contents while layouting
            updateLayoutHandler = Utils.getFunctionOption(initOptions, 'updateLayoutHandler', $.noop),

            // whether the pop-up node will be aligned to the current anchor position
            isAnchored = true,

            // an interval timer to refresh the pop-up node while it is visible
            refreshTimer = null,

            // the width and height of the virtual keyboard on touch devices
            keyboardWidth = 0,
            keyboardHeight = 0;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Adds the properties 'right' and 'bottom' to the passed rectangle,
         * containing the distances of that rectangle to the respective borders
         * of the document page.
         *
         * @param {Object} rectangle
         *  (in/out) The rectangle object that will be extended with 'right'
         *  and 'bottom' properties.
         *
         * @returns {Object}
         *  A reference to the passed and extended rectangle, for convenience.
         */
        function addTrailingDistances(rectangle) {

            var // size of the entire document page
                pageSize = Utils.getPageSize();

            rectangle.right = pageSize.width - rectangle.width - rectangle.left;
            rectangle.bottom = pageSize.height - rectangle.height - rectangle.top;
            return rectangle;
        }

        /**
         * If the passed position specification is a function, invokes it and
         * returns its result, otherwise returns the passed value as-is.
         *
         * @param {Null|Object|HTMLElement|jQuery|Function} positionSpec
         *  A position specification (any value expected for example by the
         *  constructor option 'boundingBox').
         *
         * @returns {Null|Object|HTMLElement|jQuery}
         *  The result of a callback function; otherwise the passed value.
         */
        function resolveCallback(positionSpec) {
            return _.isFunction(positionSpec) ? positionSpec.call(self) : positionSpec;
        }

        /**
         * Returns the effective position and size of a position specification.
         *
         * @param {Null|Object|HTMLElement|jQuery|Function} positionSpec
         *  A position specification (any value expected for example by the
         *  constructor option 'boundingBox').
         *
         * @param {Boolean} [clientArea=false]
         *  If set to true, this method will return the position and size of
         *  the client area of a DOM node (inner area without borders and
         *  scroll bars), instead of the outer node position and size.
         *
         * @returns {Object|Null}
         *  The rectangle of the passed position specification, if available.
         */
        function resolvePositionSpec(positionSpec, clientArea) {

            var // the resulting position
                rectangle = null;

            // resolve callback functions
            positionSpec = resolveCallback(positionSpec);

            // resolve position of DOM element or jQuery object
            if ((positionSpec instanceof HTMLElement) || (positionSpec instanceof $)) {
                if ($(positionSpec).length > 0) {
                    rectangle = clientArea ? Utils.getClientPositionInPage(positionSpec) : Utils.getNodePositionInPage(positionSpec);
                }
            } else if (_.isObject(positionSpec)) {
                // add right/bottom distance to the rectangle
                rectangle = addTrailingDistances(_.clone(positionSpec));
            }

            // do not return rectangles without valid size
            return (rectangle && (rectangle.width > 0) && (rectangle.height > 0)) ? rectangle : null;
        }

        /**
         * Returns the position and size of the bounding box, as specified with
         * the constructor option 'boundingBox', further reduced to the visible
         * area of the browser window.
         *
         * @param {Boolean} [reduceByPadding=false]
         *  If set to true, the resulting visible area will be reduced by the
         *  padding that has been specified with the constructor option
         *  'boundingPadding'.
         */
        function resolveBoundingBox(reduceByPadding) {

            var // the current position of the bounding box (client area, may become null)
                boundingBox = resolvePositionSpec(boundingBoxSpec, true),
                // current position/size of the visible area of the browser window
                windowArea = { left: window.pageXOffset, top: window.pageYOffset },
                // the size of the entire document page
                pageSize = Utils.getPageSize(),
                // additional reduction size
                padding = reduceByPadding ? boundingPadding : 0;

            // The properties window.inner(Width|Height) for their own are not
            // reliable when virtual keyboard on touch devices is visible. This
            // 'hack' works for applications with fixed page size.
            windowArea.width = Math.max(Math.min(window.innerWidth, pageSize.width - keyboardWidth), 0);
            windowArea.height = Math.max(Math.min(window.innerHeight, pageSize.height - keyboardHeight), 0);

            // intersect window area with the bounding box, if existing
            // (results in null, if bounding box is not part of the window area)
            boundingBox = boundingBox ? Utils.getIntersectionRectangle(boundingBox, windowArea) : windowArea;
            if (!boundingBox) { return null; }

            // add padding to the bounding box
            boundingBox.left += padding;
            boundingBox.top += padding;
            boundingBox.width = Math.max(0, boundingBox.width - 2 * padding);
            boundingBox.height = Math.max(0, boundingBox.height - 2 * padding);
            addTrailingDistances(boundingBox);

            // do not return bounding boxes without valid size
            return ((boundingBox.width > 0) && (boundingBox.height > 0)) ? boundingBox : null;
        }

        /**
         * Returns the current position of the anchor in the document page,
         * restricted to the current position of the anchor box if specified.
         *
         * @returns {Object|Null}
         *  The current anchor position, if the 'anchor' option has been set in
         *  the constructor options, otherwise null.
         */
        function resolveAnchorPosition() {

            var // the anchor rectangle
                anchorRect = resolvePositionSpec(anchorSpec),
                // the position of the client area of the anchor box
                anchorBoxRect = resolvePositionSpec(anchorBoxSpec, true);

            // no anchor box available: return anchor position
            if (!anchorRect || !anchorBoxRect) { return anchorRect; }

            // reduce anchor position to anchor box (anchor may become null)
            anchorRect = Utils.getIntersectionRectangle(anchorRect, anchorBoxRect);
            return anchorRect ? addTrailingDistances(anchorRect) : null;
        }

        /**
         * Returns whether the current anchor is visible. Checks the actual
         * visibility of DOM nodes. Fixed rectangles are assumed to be visible.
         *
         * @returns {Boolean}
         *  Whether the current anchor is visible.
         */
        function isAnchorVisible() {

            var // the available bounding box
                boundingBox = resolveBoundingBox(),
                // resolve anchor callback function to current value
                anchorValue = resolveCallback(anchorSpec);

            // no bounding box available: pop-up node cannot be visible
            if (!_.isObject(boundingBox)) { return false; }

            // no valid anchor available: pop-up node always visible
            if (!_.isObject(anchorValue)) { return true; }

            // check visibility of DOM element or jQuery object
            if ((anchorValue instanceof HTMLElement) || (anchorValue instanceof $)) {
                if (!$(anchorValue).is(':visible')) { return false; }
            }

            // resolve anchor rectangle, check that anchor overlaps with bounding box
            anchorValue = resolveAnchorPosition(anchorValue);
            return _.isObject(anchorValue) && Utils.rectanglesOverlap(anchorValue, boundingBox);
        }

        /**
         * Restricts the size of the passed content node to the specified size.
         *
         * @returns {Object}
         *  The effective size of the passed content node.
         */
        function updateContentNodeSize(node, size, useMin) {

            // handle restricted content node size accordingly
            switch (restrictSize) {
            case 'width':
                node.css({ minWidth: useMin ? minSize : '', maxWidth: (minSize <= size.width) ? Math.min(maxSize, size.width) : '', minHeight: '', maxHeight: '' });
                break;
            case 'height':
                node.css({ minWidth: '', maxWidth: '', minHeight: useMin ? minSize : '', maxHeight: (minSize <= size.height) ? Math.min(maxSize, size.height) : '' });
                break;
            default:
                node.css({ minWidth: '', maxWidth: '', minHeight: '', maxHeight: '' });
            }

            // Browsers usually report node sizes rounded to integers when using properties like
            // 'offsetWidth', 'scrollWidth', etc. (example: content width of a tooltip according
            // to its text contents is 123.45 pixels, but browsers report 123 pixels). When using
            // this rounded width at the outer node, the text will break into two text lines.
            // Therefore, use the helper method Utils.getCeilNodeSize() to receive the node size
            // rounded up to integers.
            return Utils.getCeilNodeSize(node);
        }

        /**
         * Recalculates the size and position of the pop-up node, according to
         * the current bounding box and node anchor.
         */
        function refreshNodePosition() {

            var // the available bounding box
                boundingBox = resolveBoundingBox(true),
                // total width/height of outer borders of the root node
                borderWidth = rootNode.outerWidth() - rootNode.width(),
                borderHeight = rootNode.outerHeight() - rootNode.height(),
                // available space for pop-up contents without outer borders of the root node
                maxContentWidth = Math.max(0, boundingBox.width - borderWidth),
                maxContentHeight = Math.max(0, boundingBox.height - borderHeight),

                // current position of the pop-up anchor
                anchorPosition = isAnchored ? resolveAnchorPosition() : null,
                // the sizes available for the pop-up contents at every side of the anchor
                availableSizes = anchorPosition ? {
                    left: { width: Utils.minMax(anchorPosition.left - boundingBox.left - anchorPadding - borderWidth, 0, maxContentWidth), height: maxContentHeight },
                    right: { width: Utils.minMax(anchorPosition.right - boundingBox.right - anchorPadding - borderWidth, 0, maxContentWidth), height: maxContentHeight },
                    top: { width: maxContentWidth, height: Utils.minMax(anchorPosition.top - boundingBox.top - anchorPadding - borderHeight, 0, maxContentHeight) },
                    bottom: { width: maxContentWidth, height: Utils.minMax(anchorPosition.bottom - boundingBox.bottom - anchorPadding - borderHeight, 0, maxContentHeight) }
                } : null,

                // the allowed borders of the anchor node used for the pop-up node (returned by callback)
                localBorders = null,
                // the resulting border of the anchor node used for the pop-up node
                preferredBorder = null,
                // the allowed alignments of the pop-up node to the border node (returned by callback)
                localAlignments = null,
                // the resulting alignment of the pop-up node to the border node
                preferredAlignment = null,
                // resulting width and height available for the pop-up node contents
                availableSize = null,

                // resulting size of the pop-up node contents
                contentSize = null,
                // whether scroll bars will be shown
                scrollHor = false, scrollVer = false,
                // whether busy mode is currently active
                isBusy = self.isBusy(),
                // the node to be adjusted (content node or busy node)
                currentNode = rootNode.children().first(),

                // rescue current position before manipulating the root node for automatic positioning
                nodeOffset = rootNode.offset(),
                // new CSS properties of the pop-up node
                cssProps = {};

            // remove previous expansion to anchor width
            if (anchorPosition && expandWidth) {
                currentNode.children().css({ minWidth: '' });
            }

            // call layout preparation callback handler if available
            var result = isBusy ? null : prepareLayoutHandler.call(self, anchorPosition, availableSizes);
            localBorders = Utils.getTokenListOption(result, 'anchorBorder');
            localBorders = _.isArray(localBorders) ? localBorders : anchorBorders;
            localAlignments = Utils.getTokenListOption(result, 'anchorAlign');
            localAlignments = _.isArray(localAlignments) ? localAlignments : anchorAlignments;

            // expand width of all node children to anchor width
            if (anchorPosition && expandWidth) {
                currentNode.children().css({ minWidth: anchorPosition.width });
            }

            // set all sizes to auto to get effective size of the pop-up node contents
            rootNode.css({ left: 0, top: '', right: '', bottom: '100%', width: 'auto', height: 'auto' });
            currentNode.css({ width: 'auto', height: 'auto' });

            // find preferred position of the pop-up node according to position of the anchor
            if (anchorPosition) {

                // calculate the ratio of the pop-up node being visible at every side of the group
                _.each(availableSizes, function (size) {
                    var resultSize = updateContentNodeSize(currentNode, size, true);
                    size.ratio = ((resultSize.width > 0) && (resultSize.height > 0)) ? (Math.min(size.width / resultSize.width, 1) * Math.min(size.height / resultSize.height, 1)) : 0;
                });

                // prefer the best side that can take the pop-up node, in the given order
                var maxRatio = 0;
                _.any(localBorders, function (border, index) {
                    var ratio = (border in availableSizes) ? availableSizes[border].ratio : 0;
                    if (ratio > maxRatio) {
                        preferredBorder = border;
                        preferredAlignment = localAlignments[Math.min(index, localAlignments.length - 1)];
                        maxRatio = ratio;
                        if (ratio === 1) { return true; }
                    }
                });

                // check validity of the preferred anchor border
                if (!_.isString(preferredBorder) || !(preferredBorder in availableSizes)) { return; }
            }

            // the resulting size available for the pop-up node contents
            if (anchorPosition && !coverAnchor) {
                availableSize = availableSizes[preferredBorder];
            } else {
                availableSize = { width: maxContentWidth, height: maxContentHeight };
            }

            // fix size of content node to prevent any reformatting when shrinking the root node
            contentSize = updateContentNodeSize(currentNode, availableSize, false);
            if (!isBusy) { _.extend(contentSize, updateLayoutHandler.call(self, _.clone(contentSize), _.clone(availableSize))); }
            contentNode.css(contentSize);

            // calculate resulting size of the root node; add scroll bars if available width or height is not sufficient
            scrollHor = !suppressScrollBars && (contentSize.width > availableSize.width);
            scrollVer = !suppressScrollBars && (contentSize.height > availableSize.height);
            cssProps.width = Math.min(availableSize.width, contentSize.width + (scrollVer ? Utils.SCROLLBAR_WIDTH : 0));
            cssProps.height = Math.min(availableSize.height, contentSize.height + (scrollHor ? Utils.SCROLLBAR_HEIGHT : 0));
            cssProps.overflowX = scrollHor ? 'scroll' : 'hidden';
            cssProps.overflowY = scrollVer ? 'scroll' : 'hidden';

            // calculate resulting position of the pop-up root node
            if (anchorPosition) {
                (function () {

                    var vertical = Utils.isVerticalPosition(preferredBorder),
                        alignOffsetName = vertical ? 'left' : 'top',
                        alignSizeName = vertical ? 'width' : 'height',
                        alignPosition = resolvePositionSpec(anchorAlignBoxSpec) || anchorPosition,
                        alignOffset = alignPosition[alignOffsetName],
                        alignSize = alignPosition[alignSizeName],
                        anchorOffset = anchorPosition[alignOffsetName],
                        anchorSize = anchorPosition[alignSizeName];

                    // first part of the position (keep in bounding box, outside of the anchor)
                    switch (preferredBorder) {
                    case 'top':
                        cssProps.bottom = Utils.getPageSize().height - anchorPosition.top + anchorPadding;
                        break;
                    case 'bottom':
                        cssProps.top = anchorPosition.top + anchorPosition.height + anchorPadding;
                        break;
                    case 'left':
                        cssProps.right = Utils.getPageSize().width - anchorPosition.left + anchorPadding;
                        break;
                    case 'right':
                        cssProps.left = anchorPosition.left + anchorPosition.width + anchorPadding;
                        break;
                    }

                    // second part of the position (use specified alignment mode, keep in bounding box)
                    switch (preferredAlignment) {
                    case 'trailing':
                        cssProps[alignOffsetName] = Utils.minMax(alignOffset + alignSize, anchorOffset, anchorOffset + anchorSize) - cssProps[alignSizeName];
                        break;
                    case 'center':
                        cssProps[alignOffsetName] = Math.floor(Utils.minMax(alignOffset + alignSize / 2, anchorOffset, anchorOffset + anchorSize) - cssProps[alignSizeName] / 2);
                        break;
                    default:
                        cssProps[alignOffsetName] = Utils.minMax(alignOffset, anchorOffset, anchorOffset + anchorSize);
                    }
                }());
            } else {
                // no anchor available: keep current position of the root node
                _.extend(cssProps, nodeOffset);
            }

            // restrict all existing positioning attributes to the bounding box
            cssProps.left = _.isNumber(cssProps.left) ? Math.max(boundingBox.left, Math.min(cssProps.left, boundingBox.left + boundingBox.width - cssProps.width - borderWidth)) : '';
            cssProps.right = _.isNumber(cssProps.right) ? Math.max(boundingBox.right, Math.min(cssProps.right, boundingBox.right + boundingBox.width - cssProps.width - borderWidth)) : '';
            cssProps.top = _.isNumber(cssProps.top) ? Math.max(boundingBox.top, Math.min(cssProps.top, boundingBox.top + boundingBox.height - cssProps.height - borderHeight)) : '';
            cssProps.bottom = _.isNumber(cssProps.bottom) ? Math.max(boundingBox.bottom, Math.min(cssProps.bottom, boundingBox.bottom + boundingBox.height - cssProps.height - borderHeight)) : '';

            if (((rootContainerNode instanceof $) && !_.isEqual(rootContainerNode.get(0), window.document.body)) || (!(rootContainerNode instanceof $) && !_.isEqual(rootContainerNode, window.document.body))) {
                cssProps.left = (cssProps.left - $(rootContainerNode).offset().left);
                cssProps.top = (cssProps.top - $(rootContainerNode).offset().top);
            }

            // apply final CSS formatting
            rootNode.css(cssProps);

            // debug logging
            if (Utils.LayoutLogger.isLoggingActive()) {
                Utils.LayoutLogger.info('BasePopup.refreshNodePosition()');
                Utils.LayoutLogger.log(' bounding:  ' + JSON.stringify(boundingBox).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' anchor:    ' + JSON.stringify(anchorPosition).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' available: ' + JSON.stringify(availableSizes).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' content:   ' + JSON.stringify(contentSize).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' preferred: ' + preferredBorder);
                Utils.LayoutLogger.log(' result:    ' + JSON.stringify(cssProps).replace(/"/g, '').replace(/,/g, ', '));
            }
        }

        /**
         * Checks whether the position of this pop-up node needs to be updated,
         * according to the current position of the anchor node (a scrollable
         * parent node may have been scrolled). In that case, the events
         * 'popup:beforelayout' and 'popup:layout' will be triggered to all
         * listeners. In automatic layout mode, the position and size of the
         * pop-up node will be updated automatically (the events will be
         * triggered anyway).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceLayout=false]
         *      If set to true, the current position of the anchor will not be
         *      checked, and the position and size of this pop-up node will
         *      always be refreshed.
         *  @param {Boolean} [options.autoHide=false]
         *      If set to true, the pop-up node will be hidden automatically
         *      when the anchor node becomes invisible (may happen either
         *      directly, e.g. by removing or hiding the anchor node; or
         *      indirectly, e.g. when hiding any parent nodes of the anchor, or
         *      when scrolling the anchor outside the visible area).
         */
        var refreshNode = (function () {

            var // the DOM element that was last focused
                lastActiveElement = null,
                // last position and size of the bounding box
                lastBoundingBox = null,
                // last position and size of the anchor node
                lastAnchorPosition = null;

            function refreshNode(options) {

                var // whether to update the position of the pop-up node
                    needsLayout = Utils.getBooleanOption(options, 'forceLayout', false),
                    // the position of the bounding box
                    boundingBox = null,
                    // the position of the anchor node
                    anchorPosition = null;

                // do nothing, if the pop-up node is not visible currently (e.g. debounced refresh)
                if (!self.isVisible()) { return; }

                // reset internal states when layout updated is forced by caller
                if (needsLayout) {
                    lastActiveElement = lastBoundingBox = lastAnchorPosition = null;
                }

                // close pop-up node, if the bounding box or the anchor is not visible anymore
                boundingBox = resolveBoundingBox();
                if (autoLayout && Utils.getBooleanOption(options, 'autoHide', false) && (!boundingBox || !isAnchorVisible())) {
                    self.hide();
                    return;
                }

                // on touch devices, get the size of the virtual keyboard
                if (Utils.IPAD && (lastActiveElement !== document.activeElement)) {
                    lastActiveElement = document.activeElement;
                    needsLayout = true;
                    keyboardWidth = keyboardHeight = 0;
                    if (Utils.isSoftKeyboardOpen() || $(lastActiveElement).is('input,textarea')) {
                        keyboardHeight = Utils.getSoftkeyboardHeight();
                    }
                }

                // check whether the visible area of the browser window has changed,
                // or the position of the anchor node (for example, by scrolling a
                // parent node of the anchor node)
                anchorPosition = resolveAnchorPosition();
                needsLayout = needsLayout || !_.isEqual(lastBoundingBox, boundingBox) || !_.isEqual(lastAnchorPosition, anchorPosition);
                lastBoundingBox = boundingBox;
                lastAnchorPosition = anchorPosition;

                // refresh the position of the pop-up node, notify all listeners
                if (needsLayout) {
                    self.trigger('popup:beforelayout');
                    if (autoLayout) { refreshNodePosition(); }
                    self.trigger('popup:layout');
                }
            }

            return refreshNode;
        }());

        // public methods -----------------------------------------------------

        /**
         * Returns the root DOM node of this pop-up, as jQuery object.
         *
         * @returns {jQuery}
         *  The root DOM node of this pop-up.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Updates the size and position of the pop-up node immediately.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.refreshImmediately = function () {
            refreshNode({ forceLayout: true, autoHide: true });
            return this;
        };

        /**
         * Updates the size and position of the pop-up node debounced.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.refresh = this.createDebouncedMethod(Utils.NOOP, this.refreshImmediately);

        /**
         * Returns whether the pop-up node is currently visible.
         *
         * @returns {Boolean}
         *  Whether the pop-up node is currently visible.
         */
        this.isVisible = function () {
            return rootNode.parent().length > 0;
        };

        /**
         * Shows the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.show = function () {

            // do nothing if the pop-up node is already open
            if (this.isVisible()) { return this; }

            // notify listeners before showing the pop-up node, event handlers may veto
            var beforeShowEvent = new $.Event('popup:beforeshow');
            this.trigger(beforeShowEvent);
            if (beforeShowEvent.isDefaultPrevented()) { return this; }

            // do nothing if the anchor is invisible (check here, event handlers of
            // the 'popup:beforeshow' event may have initialized the current anchor)
            if (autoLayout && !isAnchorVisible()) { return this; }

            // show and position the pop-up node, notify all listeners
            rootContainerNode.append(rootNode);
            isAnchored = true;
            refreshNode({ forceLayout: true });
            this.trigger('popup:show');

            // 'show' event handles must not hide the pop-up node (TODO: veto via preventDefault?)
            if (!this.isVisible()) {
                Utils.error('BasePopup.show(): "show" event handler has hidden the pop-up node');
                return this;
            }

            // start a timer that regularly refreshes the position of the pop-up node, and
            // hides the pop-up node automatically when the anchor node becomes inaccessible
            refreshTimer = this.repeatDelayed(function () { refreshNode({ autoHide: true }); }, Modernizr.touch ? 200 : 50);

            return this;
        };

        /**
         * Hides the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.hide = function () {

            // do nothing if the pop-up node is not visible
            if (!this.isVisible()) { return this; }

            // stop the automatic refresh timer
            refreshTimer.abort();
            refreshTimer = null;

            // hide the pop-up node, notify all listeners
            this.trigger('popup:beforehide');
            rootNode.detach();
            this.trigger('popup:hide');

            return this;
        };

        /**
         * Toggles the visibility of the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.toggle = function () {
            return this.isVisible() ? this.hide() : this.show();
        };

        /**
         * Returns the content node of this pop-up node, as jQuery object. All
         * contents to be shown in the pop-up have to be inserted into this
         * content node.
         *
         * @returns {jQuery}
         *  The content node of this pop-up.
         */
        this.getContentNode = function () {
            return contentNode;
        };

        /**
         * Returns whether this pop-up node contains any nodes.
         *
         * @returns {Boolean}
         *  Whether this pop-up node contains any nodes.
         */
        this.hasContents = function () {
            return contentNode[0].childNodes.length > 0;
        };

        /**
         * Removes all contents from this pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.clearContents = function () {
            if (this.hasContents()) {
                this.trigger('popup:beforeclear');
                contentNode.empty();
                this.refresh();
            }
            return this;
        };

        /**
         * Inserts the passed DOM elements before the existing contents of this
         * pop-up node.
         *
         * @param {HTMLElement|jQuery|String} [nodes ...]
         *  The DOM nodes to be inserted into this pop-up node, as plain HTML
         *  element, as jQuery collection, or as HTML mark-up string.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.prependContentNodes = function () {
            contentNode.prepend.apply(contentNode, arguments);
            this.refresh();
            return this;
        };

        /**
         * Appends the passed DOM elements to the contents of this pop-up node.
         *
         * @param {HTMLElement|jQuery|String} [nodes ...]
         *  The DOM nodes to be inserted into this pop-up node, as plain HTML
         *  element, as jQuery collection, or as HTML mark-up string.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.appendContentNodes = function () {
            contentNode.append.apply(contentNode, arguments);
            this.refresh();
            return this;
        };

        /**
         * Inserts the passed HTML mark-up into the content node of this pop-up
         * (all existing contents will be removed).
         *
         * @param {String} markup
         *  The HTML mark-up to be inserted into this pop-up node, as string.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setContentMarkup = function (markup) {
            this.clearContents(); // clean up and trigger events
            contentNode[0].innerHTML = markup;
            this.refresh();
            return this;
        };

        /**
         * Returns whether the pop-up node is currently in busy mode (the
         * regular contents of the pop-up node are hidden, and a busy spinner
         * element is shown instead).
         *
         * @returns {Boolean}
         *  Whether the pop-up node is currently in busy mode.
         */
        this.isBusy = function () {
            return busyNode.parent().length > 0;
        };


        /**
         * Switches this pop-up node to busy mode, showing a busy spinner only,
         * regardless of the actual contents.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.busy = function () {

            // do nothing if the pop-up node is in busy mode
            if (this.isBusy()) { return this; }

            // hide content node, and show busy node
            busyNode.empty().append($('<div class="popup-busy">').busy());
            rootNode.prepend(busyNode);
            contentNode.addClass('hidden');
            this.refresh();

            // notify listeners
            this.trigger('popup:busy');
            return this;
        };

        /**
         * Leaves the busy mode of this pop-up started with the method
         * BasePopup.busy().
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.idle = function () {

            // do nothing if the pop-up node is not in busy mode
            if (!this.isBusy()) { return this; }

            // hide busy node, and show content node
            busyNode.empty().detach();
            contentNode.removeClass('hidden');
            this.refresh();

            // notify listeners
            this.trigger('popup:idle');
            return this;
        };

        /**
         * Scrolls the passed child node into the visible area of this pop-up
         * node.
         *
         * @param {HTMLElement|jQuery} childNode
         *  The DOM element that will be made visible by scrolling this pop-up
         *  node. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (childNode) {
            if (this.isVisible()) {
                Utils.scrollToChildNode(rootNode, childNode, { padding: scrollPadding });
            }
            return this;
        };

        /**
         * Returns the current position of the bounding box (the value passed
         * with the 'boundingBox' constructor option).
         *
         * @returns {Object|Null}
         *  The current position of the bounding box, if available; or null, if
         *  the bounding box is outside the visible area of the browser window.
         *  The size of the returned rectangle has already been reduced by the
         *  padding as specified with the constructor option 'boundingPadding'.
         */
        this.getBoundingBox = function () {
            return resolveBoundingBox(true);
        };

        /**
         * Changes the bounding box specification of this pop-up node (the
         * value passed with the 'boundingBox' constructor option) dynamically.
         *
         * @param {Null|Object|Function} anchorBox
         *  The new bounding box specification. See the description of the
         *  constructor option 'boundingBox' for details.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setBoundingBox = function (boundingBox) {
            boundingBoxSpec = boundingBox;
            return this.refreshImmediately();
        };

        /**
         * Returns the current position of the anchor (the value passed with
         * the 'anchor' constructor option).
         *
         * @returns {Object|Null}
         *  The current position of the anchor, if available; otherwise null.
         */
        this.getAnchor = function () {
            return resolveAnchorPosition();
        };

        /**
         * Changes the anchor specification of this pop-up node (the value
         * passed with the 'anchor' constructor option) dynamically.
         *
         * @param {Null|Object|HTMLElement|jQuery|Function} anchor
         *  The new anchor specification. See the description of the
         *  constructor option 'anchor' for details.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setAnchor = function (anchor) {
            anchorSpec = anchor;
            isAnchored = true;
            return this.refreshImmediately();
        };

        /**
         * Changes the anchor box specification of this pop-up node (the value
         * passed with the 'anchorBox' constructor option) dynamically.
         *
         * @param {Null|Object|HTMLElement|jQuery|Function} anchorBox
         *  The new anchor box specification. See the description of the
         *  constructor option 'anchorBox' for details.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setAnchorBox = function (anchorBox) {
            anchorBoxSpec = anchorBox;
            return this.refreshImmediately();
        };

        /**
         * Changes the current position of the pop-up node directly. The pop-up
         * node will remain at the specified position, instead of being aligned
         * to the current anchor.
         *
         * @param {Number} left
         *  Left offset of the pop-up node in the document page, in pixels.
         *
         * @param {Number} top
         *  Top offset of the pop-up node in the document page, in pixels.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setNodePosition = function (left, top) {
            isAnchored = false;
            rootNode.css({ left: left, right: '', top: top, bottom: '' });
            return this.refreshImmediately();
        };

        /**
         * Sets the position of the pop-up node according to the passed CSS
         * positioning attributes. The resulting position of the pop-up node
         * will be adjusted according to the current visible area of the
         * browser window.
         *
         * @internal
         *  Intended to be used by sub classes of this class for custom
         *  rendering of the pop-up node.
         *
         * @deprecated
         *
         * @param {Object} cssProps
         *  The CSS positioning properties for the pop-up node, relative to the
         *  document page. Must contain one horizontal offset (either 'left' or
         *  'right'), and one vertical offset (either 'top' or 'bottom').
         *  Optionally, may contain the new size ('width' and/or 'height') of
         *  the pop-up node. May contain any other CSS properties.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this._setNodePosition = function (cssProps) {

            var // available area in the document page
                boundingBox = resolveBoundingBox(true),
                // the outer width and height of the pop-up node (may be missing in passed properties)
                nodeWidth = _.isNumber(cssProps.width) ? (cssProps.width + rootNode.outerWidth() - rootNode.width()) : rootNode.outerWidth(),
                nodeHeight = _.isNumber(cssProps.height) ? (cssProps.height + rootNode.outerHeight() - rootNode.height()) : rootNode.outerHeight();

            // restrict all existing positioning attributes to the visible area
            if (boundingBox) {
                cssProps.left = _.isNumber(cssProps.left) ? Math.max(boundingBox.left, Math.min(cssProps.left, boundingBox.left + boundingBox.width - nodeWidth)) : '';
                cssProps.right = _.isNumber(cssProps.right) ? Math.max(boundingBox.right, Math.min(cssProps.right, boundingBox.right + boundingBox.width - nodeWidth)) : '';
                cssProps.top = _.isNumber(cssProps.top) ? Math.max(boundingBox.top, Math.min(cssProps.top, boundingBox.top + boundingBox.height - nodeHeight)) : '';
                cssProps.bottom = _.isNumber(cssProps.bottom) ? Math.max(boundingBox.bottom, Math.min(cssProps.bottom, boundingBox.bottom + boundingBox.height - nodeHeight)) : '';
                rootNode.css(cssProps);
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // additional CSS classes passed to constructor
        rootNode.addClass(Utils.getStringOption(initOptions, 'classes', ''));

        // disable context menu
        rootNode.on('contextmenu', function (event) {
            if (!$(event.target).is('input,textarea')) { return false; }
        });

        // disable dragging of controls (otherwise, it is possible to drag buttons and other controls around)
        rootNode.on('contextmenu drop dragstart dragenter dragexit dragover dragleave', false);

        // ARIA role
        if (ariaRole.length > 0) {
            contentNode.attr('role', ariaRole);
        }

        // log layout events
        Utils.LayoutLogger.logEvent(this, 'popup:beforeshow popup:show popup:beforehide popup:hide popup:beforeclear');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            rootNode.remove();
            self = rootNode = contentNode = boundingBoxSpec = anchorSpec = anchorBoxSpec = anchorAlignBoxSpec = refreshTimer = null;
        });

    } // class BasePopup

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BasePopup });

});
