/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/canvaswrapper',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/config',
     'io.ox/office/tk/object/baseobject'
    ], function (Utils, Config, BaseObject) {

    'use strict';

    var // convenience constant
        PI2 = 2 * Math.PI,

        // whether to use the native line dashes provided by the rendering context (if available)
        USE_NATIVE_DASH = !Config.DEBUG || Config.getUrlFlag('office:native-dash', true);

    // class Path =============================================================

    /**
     * Represents a rendering path for a canvas rendering context, consisting
     * of several path operations.
     *
     * @constructor
     */
    function Path() {

        /**
         * All recorded operations of this path, as object with the following
         * properties:
         * - {String} op.name
         *      The name of the path operation (the name of the related native
         *      method of the rendering context).
         * - {Array} op.args
         *      All arguments to be passed to the native context method.
         */
        this.ops = [];

    } // class Path

    // public methods ---------------------------------------------------------

    /**
     * Opens a new empty subpath in this path, starting at the specified
     * position.
     *
     * @param {Number} x
     *  The X coordinate of the starting point of the new subpath.
     *
     * @param {Number} y
     *  The Y coordinate of the starting point of the new subpath.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.moveTo = function (x, y) {
        this.ops.push({ name: 'moveTo', args: [x, y] });
        return this;
    };

    /**
     * Adds a new position to the current (last) subpath in this path that will
     * be connected to the previous position with a straight line.
     *
     * @param {Number} x
     *  The X coordinate of the new position in the current subpath.
     *
     * @param {Number} y
     *  The Y coordinate of the new position in the current subpath.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.lineTo = function (x, y) {
        this.ops.push({ name: 'lineTo', args: [x, y] });
        return this;
    };

    /**
     * Adds a circular arc to the current (last) subpath in this path. The
     * start position of the arc will be connected to the end position of the
     * subpath with a straight line.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the arc.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the arc.
     *
     * @param {Number} r
     *  The radius of the arc.
     *
     * @param {Number} a1
     *  The angle (radiant) of the arc's start position. The value 0 represents
     *  a position on the positive part of the X axis.
     *
     * @param {Number} a2
     *  The angle (radiant) of the arc's end position. The value 0 represents a
     *  position on the positive part of the X axis.
     *
     * @param {Boolean} [ccw=false]
     *  Whether to draw the arc counterclockwise from its start position to its
     *  end position. By default, the arc will be drawn clockwise.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.arc = function (x, y, r, a1, a2, ccw) {
        this.ops.push({ name: 'arc', args: [x, y, r, a1, a2, ccw] });
        return this;
    };

    /**
     * Adds an elliptic arc to the current (last) subpath in this path. The
     * start position of the arc will be connected to the end position of the
     * subpath with a straight line.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the arc.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the arc.
     *
     * @param {Number} rx
     *  The radius of the arc on the X axis.
     *
     * @param {Number} ry
     *  The radius of the arc on the Y axis.
     *
     * @param {Number} a0
     *  The rotation angle (radiant) of the entire ellipse, including the
     *  following start and end angles.
     *
     * @param {Number} a1
     *  The angle (radiant) of the arc's start position. The value 0 represents
     *  a position on the positive part of the X axis.
     *
     * @param {Number} a2
     *  The angle (radiant) of the arc's end position. The value 0 represents a
     *  position on the positive part of the X axis.
     *
     * @param {Boolean} [ccw=false]
     *  Whether to draw the arc counterclockwise from its start position to its
     *  end position. By default, the arc will be drawn clockwise.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.ellipse = function (x, y, rx, ry, a0, a1, a2, ccw) {
        this.ops.push({ name: 'ellipse', args: [x, y, rx, ry, a0, a1, a2, ccw] });
        return this;
    };

    /**
     * Closes the current (last) subpath by adding a straight line to its start
     * position, and opens a new subpath at the start position of the current
     * subpath.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.close = function () {
        this.ops.push({ name: 'closePath', args: [] });
        return this;
    };

    /**
     * Adds a new subpath consisting of a straight line. After that, opens
     * another new subpath at the starting point of the line.
     *
     * @param {Number} x1
     *  The X coordinate of the starting point of the line.
     *
     * @param {Number} y1
     *  The Y coordinate of the starting point of the line.
     *
     * @param {Number} x2
     *  The X coordinate of the end point of the line.
     *
     * @param {Number} y2
     *  The Y coordinate of the end point of the line.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushLine = function (x1, y1, x2, y2) {
        return this.moveTo(x1, y1).lineTo(x2, y2).moveTo(x1, y1);
    };

    /**
     * Adds a new subpath consisting of a closed rectangle. After that, opens
     * another new subpath at the starting corner of the rectangle.
     *
     * @param {Number|Object} x
     *  The X coordinate of the starting point of the rectangle, or a complete
     *  rectangle with the numeric properties 'left', 'top', 'width', and
     *  'height'.
     *
     * @param {Number} y
     *  The Y coordinate of the starting point of the rectangle (ignored, if
     *  parameter 'x' is a rectangle object).
     *
     * @param {Number} w
     *  The width of the rectangle (ignored, if parameter 'x' is a rectangle
     *  object).
     *
     * @param {Number} h
     *  The height of the rectangle (ignored, if parameter 'x' is a rectangle
     *  object).
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushRect = function (x, y, w, h) {
        if (_.isObject(x)) { y = x.top; w = x.width; h = x.height, x = x.left; }
        return this.moveTo(x, y).lineTo(x + w, y).lineTo(x + w, y + h).lineTo(x, y + h).close();
    };

    /**
     * Adds a new subpath consisting of a complete circle. After that, opens
     * another new subpath at the center point of the circle.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the circle.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the circle.
     *
     * @param {Number} r
     *  The radius of the circle.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushCircle = function (x, y, r) {
        return this.moveTo(x + r, y).arc(x, y, r, 0, PI2).moveTo(x, y);
    };

    /**
     * Adds a new subpath consisting of a complete ellipse. After that, opens
     * another new subpath at the center point of the ellipse.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the ellipse.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the ellipse.
     *
     * @param {Number} rx
     *  The radius of the ellipse on the X axis.
     *
     * @param {Number} ry
     *  The radius of the ellipse on the Y axis.
     *
     * @param {Number} [a=0]
     *  The rotation angle (radiant) of the ellipse.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushEllipse = function (x, y, rx, ry, a) {
        return this.moveTo(x + rx, y).ellipse(x, y, rx, ry, a || 0, 0, PI2).moveTo(x, y);
    };

    // class BasePathGenerator ================================================

    /**
     * Base class for different rendering path generators, intended to provide
     * custom implementations of path generation algorithms. Sub classes MUST
     * provide all methods that can be stored in an instance of the class Path.
     *
     * @constructor
     *
     * @param {CanvasRenderingContext2D} context
     *  The wrapped rendering context of a canvas element.
     */
    var BasePathGenerator = _.makeExtendable(function (context) {

        // public properties --------------------------------------------------

        // the wrapped rendering context
        this.context = context;

    }); // class BasePathGenerator

    // path API ---------------------------------------------------------------

    BasePathGenerator.prototype.beginPath = function () {
        Utils.error('BasePathGenerator.beginPath(): missing API implementation');
    };

    // public methods ---------------------------------------------------------

    /**
     * Generates the passed path in the wrapped rendering context using its
     * native implementation.
     *
     * @param {Path} path
     *  The description of the rendering path to be generated.
     */
    BasePathGenerator.prototype.generatePath = function (path) {
        this.beginPath();
        _.each(path.ops, function (op) {
            this[op.name].apply(this, op.args);
        }, this);
    };

    // class NativePathGenerator ==============================================

    /**
     * This class implements the rendering path API of a 2D rendering context
     * of a canvas element. It uses the native methods of the wrapped rendering
     * context, but applies a fixed global translation to all operations. This
     * is needed to workaround the limited floating-point precision provided by
     * the canvas (single-precision instead of double-precision, see bug 35653,
     * and http://stackoverflow.com/a/8874802).
     *
     * @constructor
     *
     * @extends BasePathGenerator
     *
     * @param {CanvasRenderingContext2D} context
     *  The wrapped rendering context of a canvas element.
     *
     * @param {Number} x0
     *  The global translation on the X axis that will be applied to all path
     *  operations.
     *
     * @param {Number} y0
     *  The global translation on the Y axis that will be applied to all path
     *  operations.
     */
    var NativePathGenerator = BasePathGenerator.extend({ constructor: function (context, x0, y0) {

        // base constructor ---------------------------------------------------

        BasePathGenerator.call(this, context);

        // public properties --------------------------------------------------

        // the global X translation
        this.x0 = x0;

        // the global Y translation
        this.y0 = y0;

    }}); // class NativePathGenerator

    // path API ---------------------------------------------------------------

    NativePathGenerator.prototype.beginPath = function () {
        this.context.beginPath();
    };

    NativePathGenerator.prototype.moveTo = function (x, y) {
        this.context.moveTo(x - this.x0, y - this.y0);
    };

    NativePathGenerator.prototype.lineTo = function (x, y) {
        this.context.lineTo(x - this.x0, y - this.y0);
    };

    NativePathGenerator.prototype.arcTo = function (x1, y1, x2, y2, r) {
        this.context.arcTo(x1 - this.x0, y1 - this.y0, x2 - this.x0, y2 - this.y0, r);
    };

    NativePathGenerator.prototype.quadraticCurveTo = function (cx, cy, x, y) {
        this.context.quadraticCurveTo(cx - this.x0, cy - this.y0, x - this.x0, y - this.y0);
    };

    NativePathGenerator.prototype.bezierCurveTo = function (cx1, cy1, cx2, cy2, x, y) {
        this.context.bezierCurveTo(cx1 - this.x0, cy1 - this.y0, cx2 - this.x0, cy2 - this.y0, x - this.x0, y - this.y0);
    };

    NativePathGenerator.prototype.arc = function (x, y, r, a1, a2, ccw) {
        this.context.arc(x - this.x0, y - this.y0, r, a1, a2, ccw);
    };

    NativePathGenerator.prototype.ellipse = function (x, y, rx, ry, a0, a1, a2, ccw) {
        this.context.ellipse(x - this.x0, y - this.y0, rx, ry, a0, a1, a2, ccw);
    };

    NativePathGenerator.prototype.closePath = function () {
        this.context.closePath();
    };

    // class DashedPathGenerator ==============================================

    /**
     * This class implements the rendering path API of a 2D rendering context
     * of a canvas element. Provides a manual implementation of dashed lines in
     * case the rendering context does not support dashed lines natively.
     * Applies a fixed global translation to all path operations. This is
     * needed to workaround the limited floating-point precision provided by
     * the canvas (single-precision instead of double-precision, see bug 35653,
     * and http://stackoverflow.com/a/8874802).
     *
     * @constructor
     *
     * @extends BasePathGenerator
     *
     * @param {CanvasRenderingContext2D} context
     *  The wrapped rendering context of a canvas element.
     *
     * @param {Number} x0
     *  The global translation on the X axis that will be applied to all path
     *  operations.
     *
     * @param {Number} y0
     *  The global translation on the Y axis that will be applied to all path
     *  operations.
     */
    var DashedPathGenerator = BasePathGenerator.extend({ constructor: function (context, x0, y0) {

        // base constructor ---------------------------------------------------

        BasePathGenerator.call(this, context);

        // public properties --------------------------------------------------

        // the global X translation
        this.x0 = x0;

        // the global Y translation
        this.y0 = y0;

        // start position of current subpath
        this.start = null;

        // current path position
        this.point = null;

        // current dash segments
        this.segments = null;

        // total length of the current dash pattern
        this.length = 0;

        // current dash pattern offset for the next line segment
        this.offset = 0;

    }}); // class DashedPathGenerator

    // path API ---------------------------------------------------------------

    // 'beginPath' operation: start with a new empty path
    DashedPathGenerator.prototype.beginPath = function () {
        this.context.beginPath();
        this.start = this.point = null;
    };

    // 'moveTo' operation: start a new subpath, and reset dash offset
    DashedPathGenerator.prototype.moveTo = function (x, y) {
        this.start = { x: x, y: y };
        this.point = { x: x, y: y };
        this.offset = this.context.lineDashOffset;
    };

    // 'lineTo' operation: connect current position with straight line (if used
    // without initial 'moveTo', this operation acts as 'moveTo' by itself)
    DashedPathGenerator.prototype.lineTo = function (x, y) {

        // move to target position, if path is empty
        if (!this.point) { return this.moveTo(x, y); }

        var // self reference
            self = this,
            // start coordinates
            sx = this.point.x,
            sy = this.point.y,
            // signed width of the entire line box
            boxWidth = x - sx,
            // signed height of the entire line box
            boxHeight = y - sy,
            // exact length the entire line
            lineLength = Math.sqrt(boxWidth * boxWidth + boxHeight * boxHeight),
            // effective offset for the next rendered pattern (start negative, outside the line segment)
            lineOffset = -this.offset;

        // process a single pattern (callbacks need to be defined outside of while loop)
        function generatePattern() {
            _.each(self.segments, function (segment) {

                var // start offset of the line segment along the rendered line
                    segOffset = lineOffset + segment.a,
                    // end offset of the line segment along the rendered line
                    segEndOffset = lineOffset + segment.b;

                // move to segment offset inside line box (but not for a pending part of a segment
                // remaining from a previous line; in that case segOffset is less than 0)
                if ((0 <= segOffset) && (segOffset <= lineLength)) {
                    self.context.moveTo(sx + boxWidth * segOffset / lineLength, sy + boxHeight * segOffset / lineLength);
                }

                // do not draw the segment, if it is outside the line box
                if ((0 < segEndOffset) && (segOffset < lineLength)) {
                    segEndOffset = Math.min(segEndOffset, lineLength);
                    self.context.lineTo(sx + boxWidth * segEndOffset / lineLength, sy + boxHeight * segEndOffset / lineLength);
                }
            });
        }

        // check for zero-length lines
        if (lineLength === 0) { return; }

        // apply global translation to the start point (passed to native context methods 'moveTo' and 'lineTo')
        sx -= this.x0;
        sy -= this.y0;

        // draw patterns until the end of the line is reached
        while (lineOffset < lineLength) {
            generatePattern();
            lineOffset += this.length;
        }

        // update current path position and dash pattern offset
        this.point.x = x;
        this.point.y = y;
        this.offset = (this.length - (lineOffset - lineLength) % this.length) % this.length;
    };

    // 'arc' operation: add straight line to start position of the arc
    DashedPathGenerator.prototype.arc = function (x, y, r, a1, a2, ccw) {
        this.ellipse(x, y, r, r, 0, a1, a2, ccw);
    };

    // 'ellipse' operation: add straight line to start position of the arc
    DashedPathGenerator.prototype.ellipse = function (x, y, rx, ry, a0, a1, a2, ccw) {

        var // self reference
            self = this,
            // sine and cosine of the global ellipse rotation
            sina0 = Math.sin(a0), cosa0 = Math.cos(a0);

        // draws an arc particle to the passed angle
        function lineToAngle(a) {
            var sina = Math.sin(a), cosa = Math.cos(a),
                offX = rx * cosa * cosa0 - ry * sina * sina0,
                offY = rx * cosa * sina0 + ry * sina * cosa0;
            self.lineTo(x + offX, y + offY);
            return Math.sqrt(offX * offX + offY * offY);
        }

        var // add line to start position, store resulting radius for iteration
            r = lineToAngle(a1);

        // do nothing else, if angles are equal (full circle really requires different angles, e.g. 0 to 2*pi)
        if (a1 === a2) { return; }

        // single full circle, if angles have larger distance than 2*pi (in correct direction)
        if ((ccw ? -1 : 1) * (a2 - a1) > PI2) { a2 = a1; }

        // normalize passed angles, render arc particles according to direction
        a1 = Utils.mod(a1, PI2);
        a2 = Utils.mod(a2, PI2);
        if (ccw) {
            if (a1 <= a2) { a2 -= PI2; }
            while (a1 > a2) {
                a1 = Math.max(a2, a1 - 3 / r);
                r = lineToAngle(a1);
            }
        } else {
            if (a2 <= a1) { a2 += PI2; }
            while (a1 < a2) {
                a1 = Math.min(a2, a1 + 3 / r);
                r = lineToAngle(a1);
            }
        }
    };

    // 'closePath' operation: add straight line to start of current subpath
    // (if no 'moveTo' follows, the subpath will be continued at that start position)
    DashedPathGenerator.prototype.closePath = function () {
        if (this.start) {
            this.lineTo(this.start.x, this.start.y);
        }
    };

    // public methods ---------------------------------------------------------

    /**
     * Sets the current line dash pattern.
     *
     * @param {Array} pattern
     *  The new dash pattern, as expected by the native method of a canvas
     *  rendering context.
     */
    DashedPathGenerator.prototype.setLineDash = function (pattern) {

        // empty dash settings: reset line dash (solid lines)
        this.length = 0;
        if (pattern.length === 0) {
            this.segments = null;
            return;
        }

        // duplicate pattern, if array length is odd
        if (pattern.length % 2) {
            pattern = pattern.concat(pattern);
        }

        // calculate offsets of the dash segments relative to the pattern
        this.segments = [];
        for (var index = 0; index < pattern.length; index += 2) {
            var endOffset = this.length + pattern[index];
            this.segments.push({ a: this.length, b: endOffset });
            this.length = endOffset + pattern[index + 1];
        }
    };

    // class ContextWrapper ===================================================

    /**
     * This wrapper class provides a higher-level API for a 2D rendering
     * context of a canvas element.
     *
     * @constructor
     *
     * @param {CanvasRenderingContext2D} context
     *  The wrapped rendering context of a canvas element.
     *
     * @param {Number} x0
     *  The global translation on the X axis that will be applied to all path
     *  operations.
     *
     * @param {Number} y0
     *  The global translation on the Y axis that will be applied to all path
     *  operations.
     */
    function ContextWrapper(context, x0, y0) {

        var // whether native line dash support is available
            NATIVE_DASH_SUPPORT = USE_NATIVE_DASH && _.isFunction(context.setLineDash),

            // generator for native paths in the rendering context
            nativeGenerator = new NativePathGenerator(context, x0, y0),

            // generator for manually dashed paths in the rendering context
            dashedGenerator = NATIVE_DASH_SUPPORT ? null : new DashedPathGenerator(context, x0, y0),

            // whether filling path area is currently active
            fillActive = false,

            // whether stroking path outlines is currently active
            strokeActive = false,

            // current font settings
            fontStyle = {
                family: 'sans-serif',
                size: 10,
                bold: false,
                italic: false
            };

        // private methods ----------------------------------------------------

        /**
         * Sets the current line dash pattern.
         *
         * @param {Number|Array|Null} pattern
         *  The new dash pattern. See method ContextWrapper.setLineStyle() for
         *  details.
         */
        function initLineDash(pattern) {

            // convert parameter to pattern array
            if (_.isNumber(pattern)) {
                pattern = [pattern];
            } else if (!_.isArray(pattern)) {
                pattern = [];
            }

            // initialize native line dash, or line dash at path generator
            if (dashedGenerator) {
                dashedGenerator.setLineDash(pattern);
            } else {
                context.setLineDash(pattern);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Sets the fill style for rendering the inner area of a path.
         *
         * @param {Object|Null} options
         *  Fill style parameters. Omitting a property will not change the
         *  respective style setting. Passing null as parameter instead of an
         *  object will disable rendering the inner fill area completely.
         *  @param {String} [options.style]
         *      The fill style for rendering the inner fill area (CSS color
         *      descriptor, e.g. as '#RRGGBB' style, or in function style).
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setFillStyle = function (options) {

            // activate/deactivate area fill
            fillActive = _.isObject(options);
            if (fillActive) {

                // new fill color/style
                if ('style' in options) {
                    context.fillStyle = options.style;
                }
            }

            return this;
        };

        /**
         * Sets the stroke style, and other line properties, for rendering the
         * outline of a path.
         *
         * @param {Object|Null} options
         *  Stroke style parameters. Omitting a property will not change the
         *  respective style setting. Passing null as parameter instead of an
         *  object will disable rendering the path outline completely.
         *  @param {String} [options.style]
         *      The line style for rendering line segments of a path (CSS color
         *      descriptor, e.g. as '#RRGGBB' style, or in function style).
         *  @param {Number} [options.width]
         *      The width of the line segments.
         *  @param {Array|Number|Null} [options.pattern]
         *      The line dash pattern, as pairs of numbers in a flat array.
         *      Each number in the array MUST be non-negative, and the sum of
         *      all numbers (the length of the pattern) MUST be greater than
         *      zero. The first entry of each pair in this array represents the
         *      length of a visible line segment, the second entry of each pair
         *      represents the length of the gap to the next line segment. By
         *      passing multiple pairs it is possible to specify more complex
         *      dash patterns like dash-dot-dotted lines etc. If the number of
         *      elements in the array is odd, the resulting dash pattern will
         *      be the concatenation of two copies of the array. If set to a
         *      number, uses a simple pattern with equally sized segments and
         *      gaps. If set to null, the current line dash pattern will be
         *      removed, and solid lines will be rendered.
         *  @param {Number} [options.patternOffset]
         *      The initial pattern offset for every subpath in a path stroked
         *      with a dash pattern.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setLineStyle = function (options) {

            // activate/deactivate outlines
            strokeActive = _.isObject(options);
            if (strokeActive) {

                // new line color/style
                if ('style' in options) {
                    context.strokeStyle = options.style;
                }

                // new line width
                if ('width' in options) {
                    context.lineWidth = options.width;
                }

                // new line dash pattern
                if ('pattern' in options) {
                    initLineDash(options.pattern);
                }

                // dash pattern offset
                if ('patternOffset' in options) {
                    context.lineDashOffset = options.patternOffset;
                }
            }

            return this;
        };

        /**
         * Sets the font style, and other font properties, for rendering of
         * texts in the canvas.
         *
         * @param {Object} options
         *  Font style parameters. Omitting a property will not change the
         *  respective style setting.
         *  @param {String} [options.family]
         *      The complete CSS font family (comma-separated font names).
         *      Initial value is 'sans-serif'.
         *  @param {Number} [options.size]
         *      The font size, in pixels.
         *  @param {Boolean} [options.bold]
         *      Whether to set the font to bold style.
         *  @param {Boolean} [options.italic]
         *      Whether to set the font to italic style.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setFontStyle = function (options) {
            _.extend(fontStyle, options);
            context.font = (fontStyle.bold ? 'bold ' : '') + (fontStyle.italic ? 'italic ' : '') + fontStyle.size + 'pt ' + fontStyle.family;
            return this;
        };

        /**
         * Sets the global opaqueness for all drawing operations (fill, stroke,
         * text, and images).
         *
         * @param {Number} alpha
         *  The opaqueness. The value 1 represents full opacity, the value 0
         *  represents full transparency.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setGlobalAlpha = function (alpha) {
            context.globalAlpha = alpha;
            return this;
        };

        /**
         * Changes the translation of the coordinate system.
         *
         * @param {Number} dx
         *  The horizontal translation.
         *
         * @param {Number} dy
         *  The vertical translation.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.translate = function (dx, dy) {
            context.translate(dx, dy);
            return this;
        };

        /**
         * Changes the scaling factors of the coordinate system.
         *
         * @param {Number} fx
         *  The horizontal scaling factor.
         *
         * @param {Number} fy
         *  The vertical scaling factor.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.scale = function (fx, fy) {
            context.scale(fx, fy);
            return this;
        };

        /**
         * Creates a new empty path object. Path objects are expected by other
         * rendering methods of this canvas wrapper. A caller may create as
         * many path objects as needed at the same time.
         *
         * @returns {Path}
         *  A new empty path object.
         */
        this.createPath = function () {
            return new Path();
        };

        /**
         * Renders the specified path into the wrapped canvas element.
         *
         * @param {Path} path
         *  The path to be rendered.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawPath = function (path) {

            var // whether line dash has to be generated manually
                manualLineDash = _.isObject(dashedGenerator) && _.isArray(dashedGenerator.segments);

            // generate native path for filled area, solid lines, or native dash support
            if (fillActive || (strokeActive && !manualLineDash)) {
                nativeGenerator.generatePath(path);
            }

            // fill the area surrounded by the path
            if (fillActive) {
                context.fill();
            }

            // generate manually dashed path, if native dash support is missing
            if (strokeActive && manualLineDash) {
                dashedGenerator.generatePath(path);
            }

            // draw the path outline
            if (strokeActive) {
                context.stroke();
            }

            return this;
        };

        /**
         * Renders a straight line into the wrapped canvas element.
         *
         * @param {Number} x1
         *  The X coordinate of the starting point of the line.
         *
         * @param {Number} y1
         *  The Y coordinate of the starting point of the line.
         *
         * @param {Number} x2
         *  The X coordinate of the end point of the line.
         *
         * @param {Number} y2
         *  The Y coordinate of the end point of the line.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawLine = function (x1, y1, x2, y2) {
            return this.drawPath(this.createPath().pushLine(x1, y1, x2, y2));
        };

        /**
         * Renders a rectangle into the wrapped canvas element.
         *
         * @param {Number|Object} x
         *  The X coordinate of the starting point of the rectangle, or a
         *  complete rectangle with the numeric properties 'left', 'top',
         *  'width', and 'height'.
         *
         * @param {Number} y
         *  The Y coordinate of the starting point of the rectangle (ignored,
         *  if parameter 'x' is a rectangle object).
         *
         * @param {Number} w
         *  The width of the rectangle (ignored, if parameter 'x' is a
         *  rectangle object).
         *
         * @param {Number} h
         *  The height of the rectangle (ignored, if parameter 'x' is a
         *  rectangle object).
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawRect = function (x, y, w, h) {
            return this.drawPath(this.createPath().pushRect(x, y, w, h));
        };

        /**
         * Renders a circle into the wrapped canvas element.
         *
         * @param {Number} x
         *  The X coordinate of the center point of the circle.
         *
         * @param {Number} y
         *  The Y coordinate of the center point of the circle.
         *
         * @param {Number} r
         *  The radius of the circle.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawCircle = function (x, y, r) {
            return this.drawPath(this.createPath().pushCircle(x, y, r));
        };

        /**
         * Renders an ellipse into the wrapped canvas element.
         *
         * @param {Number} x
         *  The X coordinate of the center point of the ellipse.
         *
         * @param {Number} y
         *  The Y coordinate of the center point of the ellipse.
         *
         * @param {Number} rx
         *  The radius of the ellipse on the X axis.
         *
         * @param {Number} ry
         *  The radius of the ellipse on the Y axis.
         *
         * @param {Number} [a=0]
         *  The rotation angle (radiant) of the ellipse.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawEllipse = function (x, y, rx, ry, a) {
            return this.drawPath(this.createPath().pushEllipse(x, y, rx, ry, a));
        };

        /**
         * Clears a rectangle (sets all pixels to full transparency) in the
         * wrapped canvas element.
         *
         * @param {Number|Object} x
         *  The X coordinate of the starting point of the rectangle, or a
         *  complete rectangle with the numeric properties 'left', 'top',
         *  'width', and 'height'.
         *
         * @param {Number} y
         *  The Y coordinate of the starting point of the rectangle (ignored,
         *  if parameter 'x' is a rectangle object).
         *
         * @param {Number} w
         *  The width of the rectangle (ignored, if parameter 'x' is a
         *  rectangle object).
         *
         * @param {Number} h
         *  The height of the rectangle (ignored, if parameter 'x' is a
         *  rectangle object).
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.clearRect = function (x, y, w, h) {
            if (_.isObject(x)) { y = x.top; w = x.width; h = x.height, x = x.left; }
            context.clearRect(x - x0, y - y0, w, h);
            return this;
        };

        /**
         * Returns the width of the passed string.
         *
         * @param {String} text
         *  The text whose width will be calculated.
         *
         * @returns {Number}
         *  The width of the passed string, as floating-point pixels.
         */
        this.getTextWidth = function (text) {
            return context.measureText(text).width;
        };

        /**
         * Returns the width of the passed single character. To increase
         * precision for different browsers (some browsers return a rounded
         * integral width for single characters), the character will be
         * repeated ten times, and a tenth of the width of the resulting
         * string will be returned.
         *
         * @param {String} char
         *  A single character.
         *
         * @returns {Number}
         *  The width of the passed character, as floating-point pixels.
         */
        this.getCharacterWidth = function (char) {
            return context.measureText(Utils.repeatString(char, 10)).width / 10;
        };

    } // class ContextWrapper

    // class CanvasWrapper ====================================================

    /**
     * A wrapper class for a canvas DOM element. Provides additional
     * convenience methods to render into the canvas.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {HTMLElement|jQuery} [options.canvasElement]
     *      An existing <canvas> element to be wrapped by this instance. By
     *      default, a new <canvas> element will be created automatically.
     *  @param {String} [options.classes]
     *      Additional CSS classes (space-separated list) that will be added to
     *      the passed or created <canvas> element.
     */
    function CanvasWrapper(options) {

        var // the current canvas DOM element, as jQuery object
            canvasNode = $(Utils.getOption(options, 'canvasElement', '<canvas width="1" height="1">')),

            // additional classes for the canvas element
            classes = Utils.getStringOption(options, 'classes'),

            // the rendering context of the canvas
            context = canvasNode[0].getContext('2d'),

            // the current rectangle represented by the canvas element
            x0 = 0,
            y0 = 0,
            width = canvasNode[0].width,
            height = canvasNode[0].height;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // public methods -----------------------------------------------------

        /**
         * Returns the wrapped canvas element.
         *
         * @returns {jQuery}
         *  The wrapped canvas DOM element, as jQuery object.
         */
        this.getNode = function () {
            return canvasNode;
        };

        /**
         * Changes the offset and size of the canvas element, and clears all
         * its contents.
         *
         * @param {Object} rectangle
         *  The new rectangle represented by the canvas DOM element, with the
         *  following properties:
         *  @param {Number} [rectangle.left=0]
         *      Global horizontal translation, in pixels. A pixel drawn at this
         *      position will appear at the left border of the canvas area.
         *  @param {Number} [rectangle.top=0]
         *      Global vertical translation, in pixels. A pixel drawn at this
         *      position will appear at the top border of the canvas area.
         *  @param {Number} rectangle.width
         *      Width of the bitmap in the canvas area, in pixels. Will also be
         *      set as element size of the <canvas> DOM element.
         *  @param {Number} rectangle.height
         *      Height of the bitmap in the canvas area, in pixels. Will also
         *      be set as element size of the <canvas> DOM element.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.initialize = function (rectangle) {

            var // width and height of the canvas, as element attributes and CSS properties
                sizeProps = { width: rectangle.width, height: rectangle.height };

            // store new canvas rectangle
            x0 = Utils.getIntegerOption(rectangle, 'left', 0);
            y0 = Utils.getIntegerOption(rectangle, 'top', 0);
            width = rectangle.width;
            height = rectangle.height;

            // set new bitmap size and element size
            canvasNode.attr(sizeProps).css(sizeProps);

            // clear all contents (canvas may not clear itself, if its size does not change)
            return this.clear();
        };

        /**
         * Clears the entire area covered by the canvas element.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.clear = function () {
            context.clearRect(0, 0, width, height);
            return this;
        };

        /**
         * Saves the current state of the canvas (colors, transformation,
         * clipping, etc.), sets up a rectangular clipping region, invokes the
         * passed callback function, and restores the canvas state afterwards.
         *
         * @param {Object} rectangle
         *  The clipping rectangle, with the numeric properties 'left', 'top',
         *  'width', and 'height'.
         *
         * @param {Function} callback
         *  The callback function. The symbol 'this' inside the callback
         *  function will be bound to this wrapper instance.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.clip = function (rectangle, callback) {
            context.save();
            context.beginPath();
            context.rect(rectangle.left - x0, rectangle.top - y0, rectangle.width, rectangle.height);
            context.clip();
            callback.call(this);
            context.restore();
            return this;
        };

        /**
         * Saves the current state of the canvas (colors, transformation,
         * clipping, etc.), invokes the passed callback function, and restores
         * the canvas state afterwards.
         *
         * @param {Function} callback
         *  The callback function to be invoked. Receives the following
         *  parameters:
         *  (1) {ContextWrapper} context
         *      The context wrapper of the canvas element, which provides a
         *      more convenient rendering API than the native rendering context
         *      of the canvas, and that adds polyfills for missing features of
         *      the current browser.
         *  (2) {Number} width
         *      The current width of the canvas area, in pixels.
         *  (3) {Number} height
         *      The current height of the canvas area, in pixels.
         *  The symbol 'this' inside the callback function will be bound to
         *  this canvas wrapper instance.
         *
         * @returns {CanvasWrapper}
         *  A reference to this instance.
         */
        this.render = function (callback) {
            context.save();
            callback.call(this, new ContextWrapper(context, x0, y0), width, height);
            context.restore();
            return this;
        };

        /**
         * Returns a data URL with the current contents of the canvas.
         *
         * @returns {String}
         *  A data URL with the current contents of the canvas.
         */
        this.getDataURL = function () {
            return canvasNode[0].toDataURL();
        };

        /**
         * return current ImageData object with size of content of the current canvas-state
         * every entry is one info of Red, Greeb, Blue or Alpha
         * range is 0-255
         *
         * @returns {Array}
         */
        this.getImageData = function() {
            return context.getImageData(0, 0, width, height).data;
        };

        // initialization -----------------------------------------------------

        // add CSS class names passed to the constructor
        if (classes) { canvasNode.addClass(classes); }

        // add polyfills for missing native CanvasRenderingContext2D methods
        if (!_.isFunction(context.ellipse)) {
            context.ellipse = function (x, y, rx, ry, a0, a1, a2, ccw) {
                this.translate(x, y);
                this.rotate(a0);
                this.scale(rx, ry);
                this.arc(0, 0, 1, a1, a2, ccw);
                this.scale(1 / rx, 1 / ry);
                this.rotate(-a0);
                this.translate(-x, -y);
            };
        }

        if (!_.isNumber(context.lineDashOffset)) {
            context.lineDashOffset = 0;
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            canvasNode.remove();
            canvasNode = context = null;
        });

    } // class CanvasWrapper

    // exports ================================================================

    // derive this class from class BaseObject
    return BaseObject.extend({ constructor: CanvasWrapper });

});
