/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/view', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/button',
    'io.ox/office/tk/control/checkbox',
    'io.ox/office/tk/control/label',
    'io.ox/office/tk/control/unitfield',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/view/editview',
    'io.ox/office/drawinglayer/view/popup/chartlabelsmenu',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/clipboard',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/view/statuspane',
    'io.ox/office/spreadsheet/view/gridpane',
    'io.ox/office/spreadsheet/view/headerpane',
    'io.ox/office/spreadsheet/view/cornerpane',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'io.ox/office/spreadsheet/view/mixin/selectionmixin',
    'io.ox/office/spreadsheet/view/mixin/celleditmixin',
    'io.ox/office/spreadsheet/view/mixin/dialogsmixin',
    'io.ox/office/spreadsheet/view/mixin/hyperlinkmixin',
    'io.ox/office/spreadsheet/view/mixin/sortmixin',
    'io.ox/office/spreadsheet/view/mixin/viewfuncmixin',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendercache',
    'gettext!io.ox/office/spreadsheet',
    'less!io.ox/office/spreadsheet/view/style'
], function (Utils, KeyCodes, Forms, Button, CheckBox, Label, UnitField, Color, Border, EditView, ChartLabelsMenu, Config, Operations, SheetUtils, PaneUtils, Clipboard, CellCollection, StatusPane, GridPane, HeaderPane, CornerPane, Labels, Controls, SelectionMixin, CellEditMixin, DialogsMixin, HyperlinkMixin, SortMixin, ViewFuncMixin, RenderUtils, RenderCache, gt) {

    'use strict';

    var // warning message texts for model error codes
        RESULT_MESSAGES = {

            'sheetname:empty':
                gt('The sheet name must not be empty. Please enter a name.'),

            'sheetname:invalid':
                gt('This sheet name contains invalid characters. Please ensure that the name does not contain / \\ * ? : [ or ], and does neither start nor end with an apostrophe.'),

            'sheetname:used':
                gt('This sheet name is already used. Please enter another name.'),

            'sheet:locked':
                gt('You cannot use this command on a protected sheet.'),

            'cols:overflow':
                gt('It is not possible to modify more than %1$d columns at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_COLS_COUNT)),

            'rows:overflow':
                gt('It is not possible to modify more than %1$d rows at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_ROWS_COUNT)),

            'cells:locked':
                gt('Protected cells cannot be modified.'),

            'cells:overflow':
                gt('It is not possible to modify more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)),

            'merge:overlap':
                //#. Warning text: trying to merge multiple cell ranges which overlap each other
                gt('Overlapping ranges cannot be merged.'),

            'merge:overflow':
                //#. Warning text: trying to merge too many cell ranges at the same time
                //#. %1$d is the maximum number of cell ranges that can be merged at a time
                //#, c-format
                gt('It is not possible to merge more than %1$d ranges at the same time.', _.noI18n(SheetUtils.MAX_MERGED_RANGES_COUNT)),

            'table:multiselect':
                //#. Warning text: tried to create a table range from a cell multi-selection
                gt('Table cannot be created on multiple cell ranges.'),

            'table:overlap':
                //#. Warning text: tried to create a table range above cells that are already filtered
                gt('Table cannot be created on another filtered range.'),

            'autofilter:multiselect':
                //#. Warning text: tried to create an auto filter from a cell multi-selection
                gt('Filter cannot be applied to multiple cell ranges.'),

            'autofilter:overlap':
                //#. Warning text: tried to create an auto filter above cells that are already filtered
                gt('Filter cannot be applied to another filtered range.'),

            'autofilter:blank':
                //#. Warning text: tried to filter an empty cell range
                gt('Filtering cannot be performed on blank cells.'),

            'sort:multiselect':
                //#. Warning text: tried to sort data of a cell multi-selection
                gt('Sorting cannot be performed on multiple cell ranges.'),

            'sort:blank':
                //#. Warning text: tried to sort an empty cell range
                gt('Sorting cannot be performed on blank cells.'),

            'formula:invalid':
                gt('The formula contains an error.'),

            'formula:matrix':
                gt('Parts of a matrix formula cannot be changed.'),

            'validation:source':
                //#. Warning text: the values for a drop-down list attached to a spreadsheet cell could not be found
                gt('No source data available for the drop-down list.'),

            'paste:overflow':
                gt('It is not possible to paste more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)),

            'paste:locked':
                gt('Pasting into protected cells is not allowed.')
        },

        // the size of the freeze separator nodes, in pixels
        FROZEN_SPLIT_SIZE = 1,

        // the size of the split separator nodes, in pixels
        DYNAMIC_SPLIT_SIZE = 2,

        // the highlighted inner size of the split tracking node, in pixels
        TRACKING_SPLIT_SIZE = 4,

        // the additional margin of the split tracking nodes, in pixels
        TRACKING_MARGIN = 3,

        // the position offset of tracking nodes compared to split lines
        TRACKING_OFFSET = (TRACKING_SPLIT_SIZE - DYNAMIC_SPLIT_SIZE) / 2 + TRACKING_MARGIN,

        // default settings for a pane side
        DEFAULT_PANE_SIDE_SETTINGS = {
            visible: true,
            offset: 0,
            size: 0,
            frozen: false,
            showOppositeScroll: false,
            hiddenSize: 0
        },

        // delay time for destruction of unused rendering caches
        RENDER_CACHE_DESTROY_DELAY = (Utils.COMPACT_DEVICE ? 2 : 10) * 60000,

        // clipboard client and server id to identify the copy data when pasting.
        clipboardId = { client: null, server: null };

    // class SpreadsheetView ==================================================

    /**
     * Represents the entire view of a spreadsheet document. Contains the view
     * panes of the sheet currently shown (the 'active sheet'), which contain
     * the scrollable cell grids (there will be several view panes, if the
     * sheet view is split or frozen); and the selections of all existing
     * sheets.
     *
     * Additionally to the events triggered by the base class EditView, an
     * instance of this class triggers events if global contents of the
     * document, or the contents of the current active sheet have been changed.
     * The following events will be triggered:
     * - 'before:activesheet'
     *      Before another sheet will be activated. Event handlers receive the
     *      zero-based index of the old active sheet in the document.
     * - 'change:activesheet'
     *      After another sheet has been activated. Event handlers receive the
     *      zero-based index of the new active sheet in the document, and the
     *      sheet model instance of the new active sheet.
     * - 'change:sheets'
     *      After the collection of all sheets in the document has been
     *      changed (after inserting, deleting, moving, renaming, hiding a
     *      visible, or showing a hidden sheet).
     * - 'change:layoutdata'
     *      After the selection layout data has been updated, either directly
     *      or from the response data of a server view update.
     * - 'sheet:triggered'
     *      Any event from the active sheet will be forwarded with this event.
     *      Event handlers receive the type name of the event, and all other
     *      parameters sent with the original sheet event.
     * - 'change:sheet:attributes'
     *      After the formatting attributes of the active sheet have been
     *      changed.
     * - 'change:sheet:viewattributes'
     *      After the view attributes of the active sheet have been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes of the active sheet with their new values.
     * - 'change:selection'
     *      Will be triggered directly after the 'change:sheet:viewattributes'
     *      event caused by a changed selection, for convenience. Event
     *      handlers receive the new selection.
     * - 'celledit:enter', 'celledit:leave'
     *      After entering or leaving the cell in-place edit mode.
     * - 'celledit:change'
     *      After changing text while cell in-place edit mode is active.
     * - 'celledit:reject'
     *      After attempting to enter or to leave the cell in-place edit mode
     *      has been rejected, e.g. when trying to edit a protected cell, or
     *      when trying to commit an invalid/incomplete formula expression.
     * - 'insert:columns', 'change:columns', 'delete:columns'
     *      The respective event forwarded from the column collection of the
     *      active sheet. See class ColRowCollection for details.
     * - 'insert:rows', 'change:rows', 'delete:rows'
     *      The respective event forwarded from the row collection of the
     *      active sheet. See class ColRowCollection for details.
     * - 'insert:merged', 'delete:merged'
     *      The respective event forwarded from the merge collection of the
     *      active sheet. See class MergeCollection for details.
     * - 'insert:table', 'change:table', 'delete:table'
     *      The respective event forwarded from the table collection of the
     *      active sheet. See class TableCollection for details.
     * - 'change:cells', 'change:usedarea'
     *      The respective event forwarded from the cell collection of the
     *      active sheet. See class CellCollection for details.
     * - 'insert:drawing', 'delete:drawing', 'change:drawing'
     *      The respective event forwarded from the drawing collection of the
     *      active sheet. See class SheetDrawingCollection for details.
     * - 'render:cellselection'
     *      After the cell selection of the active grid pane has actually been
     *      rendered into the DOM selection layer node (rendering may happen
     *      debounced after several 'change:selection' view events).
     * - 'render:drawingselection'
     *      After the drawing selection of the active grid pane has actually
     *      been rendered into the DOM drawing layer (rendering may happen
     *      debounced after several 'change:selection' view events). Event
     *      handlers receive the DOM drawing frames currently selected in the
     *      active grid pane, as jQuery collection.
     *
     * @constructor
     *
     * @extends EditView
     * @extends SelectionMixin
     * @extends CellEditMixin
     * @extends DialogsMixin
     * @extends HyperlinkMixin
     * @extends ViewFuncMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this spreadsheet view.
     */
    function SpreadsheetView(app) {

        var // self reference
            self = this,

            // the scrollable root DOM node containing the headers and panes
            rootNode = $('<div class="abs pane-root">'),

            // the top-left corner pane
            cornerPane = null,

            // settings for all pane sides, mapped by pane side identifier
            paneSideSettings = {},

            // the row/column header panes, mapped by position keys
            headerPanes = {},

            // the grid panes, mapped by position keys
            gridPanes = {},

            // the status pane with the sheet tabs
            statusPane = null,

            // the floating menu for chart axis and labels formatting
            chartLabelsMenu = null,

            // rendering caches, mapped by UIDs of the sheet models
            renderCaches = {},

            // the split line separating the left and right panes
            colSplitLineNode = $('<div class="split-line columns">'),

            // the split line separating the top and bottom panes
            rowSplitLineNode = $('<div class="split-line rows">'),

            // the split tracking node separating left and right panes
            colSplitTrackingNode = $('<div class="abs split-tracking columns" style="width:' + TRACKING_SPLIT_SIZE + 'px;padding:0 ' + TRACKING_MARGIN + 'px;" tabindex="-1">'),

            // the split tracking node separating top and bottom panes
            rowSplitTrackingNode = $('<div class="abs split-tracking rows" style="height:' + TRACKING_SPLIT_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px 0;" tabindex="-1">'),

            // the split tracking node covering the intersection of the other tracking points
            centerSplitTrackingNode = $('<div class="abs split-tracking columns rows" style="width:' + TRACKING_SPLIT_SIZE + 'px;height:' + TRACKING_SPLIT_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px;" tabindex="-1">'),

            // all split tracking nodes, as jQuery collection
            allSplitTrackingNodes = colSplitTrackingNode.add(rowSplitTrackingNode).add(centerSplitTrackingNode),

            // tracking overlay nodes for column/row resizing
            resizeOverlayNode = $('<div class="abs resize-tracking"><div class="abs leading"></div><div class="abs trailing"></div></div>'),

            // the collections of visible and hidden sheets
            visibleSheets = [],
            hiddenSheets = [],

            // the spreadsheet document model, and the style sheet containers
            docModel = null,
            documentStyles = null,

            // the model and rendering cache of the active sheet
            activeSheetModel = null,
            activeRenderCache = null,

            // collections of the active sheet
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            cellCollection = null,
            tableCollection = null,
            validationCollection = null,
            drawingCollection = null,

            // the current status text label
            statusText = null;

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            initDebugHandler: initDebugHandler,
            initGuiHandler: initGuiHandler,
            initDebugGuiHandler: initDebugGuiHandler,
            grabFocusHandler: grabFocusHandler,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, top: 20, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        SelectionMixin.call(this, app, rootNode);
        CellEditMixin.call(this, app, rootNode);
        DialogsMixin.call(this, app, rootNode);
        HyperlinkMixin.call(this, app);
        SortMixin.call(this, app);
        ViewFuncMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Destroys the rendering cache associated to the specified sheet.
         *
         * @param {String} uid
         *  The unique identifier of a sheet model.
         */
        function destroyRenderCache(uid) {

            // rendering cache may be deleted already
            if (!(uid in renderCaches)) { return; }

            RenderUtils.log('SpreadsheetView.destroyRenderCache(): deleting cache for uid=' + uid);

            // abort auto-destruction timer
            if (renderCaches[uid].timer) {
                renderCaches[uid].timer.abort();
            }

            // destroy the cache instance, remove the descriptor from the map
            renderCaches[uid].cache.destroy();
            delete renderCaches[uid];
        }

        /**
         * Triggers a 'before:activesheet' event in preparation to change the
         * current active sheet.
         */
        function prepareSetActiveSheet() {

            // do nothing if no sheet is currently active (initially after import)
            if (!activeSheetModel) { return; }

            var // the unique identifier of the active sheet
                uid = activeSheetModel.getUid();

            // leave cell edit mode before switching sheets; TODO: do not leave if active sheet just moves
            self.leaveCellEditMode('auto');

            // create a timer for auto-destruction of the rendering cache
            renderCaches[uid].timer = self.executeDelayed(function () {
                destroyRenderCache(uid);
            }, RENDER_CACHE_DESTROY_DELAY);

            // notify listeners with the index of the old active sheet
            self.trigger('before:activesheet', docModel.getActiveSheet());
        }

        /**
         * Activates the specified sheet, and triggers a 'change:activesheet'
         * event with the new sheet index.
         *
         * @param {Number} sheet
         *  The zero-based index of the new active sheet.
         */
        function setActiveSheet(sheet) {

            // register index of the new active sheet at the document model
            docModel.setActiveSheet(sheet);

            // get model instance and collections of the new active sheet
            activeSheetModel = docModel.getSheetModel(sheet);
            colCollection = activeSheetModel.getColCollection();
            rowCollection = activeSheetModel.getRowCollection();
            mergeCollection = activeSheetModel.getMergeCollection();
            cellCollection = activeSheetModel.getCellCollection();
            tableCollection = activeSheetModel.getTableCollection();
            validationCollection = activeSheetModel.getValidationCollection();
            drawingCollection = activeSheetModel.getDrawingCollection();

            var // the unique identifier of the active sheet
                uid = activeSheetModel.getUid(),
                // descriptor for the rendering cache
                cacheData = renderCaches[uid];

            // abort the auto-destruction timer of an existing cache
            if (cacheData && cacheData.timer) {
                cacheData.timer.abort();
                cacheData.timer = null;
            }

            // initialize a new rendering cache
            else if (!cacheData) {
                RenderUtils.log('SpreadsheetView.setActiveSheet(): creating cache for uid=' + uid);

                // create a new cache, initialize the descriptor object for the map
                renderCaches[uid] = cacheData = {
                    cache: new RenderCache(app, activeSheetModel),
                    timer: null
                };

                // forward all events of the rendering cache to the view
                cacheData.cache.on('triggered', function () {
                    if (cacheData.cache === activeRenderCache) {
                        self.trigger.apply(self, _.toArray(arguments).slice(1));
                    }
                });
            }

            // store reference to the active rendering cache
            activeRenderCache = cacheData.cache;

            // notify listeners
            self.trigger('change:activesheet', sheet, activeSheetModel);
            self.trigger('change:sheet:viewattributes', activeSheetModel.getViewAttributes());

            // refresh the layout of the header and grid panes
            initializePanes();
        }

        /**
         * Updates the collection of visible sheets.
         */
        function updateVisibleSheets() {

            visibleSheets = [];
            hiddenSheets = [];

            // collect all supported and visible sheets
            docModel.iterateSheetModels(function (sheetModel, sheet, name) {

                // ignore unsupported sheet types
                if (!self.isSheetTypeSupported(sheetModel.getType())) { return; }

                // collect visible and hidden sheets, count hidden sheets
                if (sheetModel.getMergedAttributes().sheet.visible) {
                    visibleSheets.push({ sheet: sheet, name: name });
                } else {
                    hiddenSheets.push({ sheet: sheet, name: name });
                }
            });
        }

        /**
         * Returns the array index of the specified sheet in the collection of
         * all visible sheets.
         */
        function getVisibleIndex(sheet) {
            var arrayIndex = _.sortedIndex(visibleSheets, { sheet: sheet }, 'sheet');
            return ((0 <= arrayIndex) && (arrayIndex < visibleSheets.length) && (visibleSheets[arrayIndex].sheet === sheet)) ? arrayIndex : -1;
        }

        /**
         * Handles changed view attributes in the active sheet. Shows a changed
         * zoom factor in the status label and redraws the view with the new
         * zoom factor, and triggers change events for the attributes.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // hide cell selection while drawings are selected
            if ('selection' in attributes) {
                rootNode.toggleClass('drawing-selection', attributes.selection.drawings.length > 0);
            }

            // update header and grid panes, after split/freeze settings have changed
            if (Utils.hasProperty(attributes, /^(zoom$|split)/)) {
                initializePanes();
            }
        }

        /**
         * Handles changed column attributes in the active sheet. Updates the
         * size of frozen panes after the effective width of the columns has
         * changed.
         */
        function changeColumnsHandler(event, interval, attributes) {
            if (self.hasFrozenSplit() && ('column' in attributes) && Utils.hasProperty(attributes.column, /^(width|visible)$/)) {
                initializePanes();
            }
        }

        /**
         * Handles changed row attributes in the active sheet. Updates the size
         * of frozen panes after the effective height of the rows has changed.
         */
        function changeRowsHandler(event, interval, attributes) {
            if (self.hasFrozenSplit() && ('row' in attributes) && Utils.hasProperty(attributes.row, /^(height|visible)$/)) {
                initializePanes();
            }
        }

        /**
         * Prepares insertion of a sheet into the document.
         */
        function beforeInsertSheetHandler(event, sheet) {
            // active sheet will change, if a sheet will be inserted before it
            if (sheet <= docModel.getActiveSheet()) {
                prepareSetActiveSheet();
            }
        }

        /**
         * Finalizes insertion of a sheet into the document.
         */
        function afterInsertSheetHandler(event, sheet, sheetModel) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet();

            // registers an event handler that will be called while the sheet is active
            function handleActiveSheetEvent(eventType, options) {

                var // the local event handler called before the event will be triggered
                    handlerFunc = Utils.getFunctionOption(options, 'handler'),
                    // the type of the event to be triggered
                    triggerType = Utils.getStringOption(options, 'trigger');

                // handle specified event of the event source object, do nothing if the sheet is not active
                sheetModel.on(eventType, function () {
                    if (sheetModel === activeSheetModel) {
                        if (_.isFunction(handlerFunc)) {
                            handlerFunc.apply(self, _.toArray(arguments));
                        }
                        if (_.isString(triggerType)) {
                            self.trigger.apply(self, [triggerType].concat(_.toArray(arguments).slice(1)));
                        }
                    }
                });
            }

            // listen to changes in the sheet, notify listeners of the view
            handleActiveSheetEvent('change:attributes', { trigger: 'change:sheet:attributes' });
            handleActiveSheetEvent('change:viewattributes', { handler: changeSheetViewAttributesHandler, trigger: 'change:sheet:viewattributes' });
            handleActiveSheetEvent('insert:columns', { trigger: 'insert:columns' });
            handleActiveSheetEvent('change:columns', { handler: changeColumnsHandler, trigger: 'change:columns' });
            handleActiveSheetEvent('delete:columns', { trigger: 'delete:columns' });
            handleActiveSheetEvent('delete:rows', { trigger: 'delete:rows' });
            handleActiveSheetEvent('change:rows', { handler: changeRowsHandler, trigger: 'change:rows' });
            handleActiveSheetEvent('insert:rows', { trigger: 'insert:rows' });
            handleActiveSheetEvent('insert:merged', { trigger: 'insert:merged' });
            handleActiveSheetEvent('delete:merged', { trigger: 'delete:merged' });
            handleActiveSheetEvent('insert:table', { trigger: 'insert:table' });
            handleActiveSheetEvent('change:table', { trigger: 'change:table' });
            handleActiveSheetEvent('delete:table', { trigger: 'delete:table' });
            handleActiveSheetEvent('change:cells', { trigger: 'change:cells' });
            handleActiveSheetEvent('change:usedarea', { trigger: 'change:usedarea' });
            handleActiveSheetEvent('insert:drawing', { trigger: 'insert:drawing' });
            handleActiveSheetEvent('delete:drawing', { trigger: 'delete:drawing' });
            handleActiveSheetEvent('change:drawing', { trigger: 'change:drawing' });

            // a generic 'triggered' event for any event of the active sheet
            handleActiveSheetEvent('triggered', { trigger: 'sheet:triggered' });

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet <= activeSheet) {
                setActiveSheet(activeSheet + 1);
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Prepares deletion of a sheet from the document.
         */
        function beforeDeleteSheetHandler(event, sheet, sheetModel) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet(),
                // unique identifier of the sheet model to be deleted
                uid = sheetModel.getUid();

            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.cancelCellEditMode();
            }

            // active sheet will change, if a sheet before will be deleted, or itself
            if (sheet <= activeSheet) {
                prepareSetActiveSheet();
            }

            // delete the rendering cache of the sheet
            destroyRenderCache(uid);
        }

        /**
         * Finalizes deletion of a sheet from the document.
         */
        function afterDeleteSheetHandler(event, sheet) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet();

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet < activeSheet) {
                // sheet before active sheet deleted: decrease index of active sheet
                setActiveSheet(activeSheet - 1);
            } else if (sheet === activeSheet) {
                // active sheet deleted: keep index of active sheet, unless it was the last sheet
                setActiveSheet(Math.min(activeSheet, docModel.getSheetCount() - 1));
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Prepares moving a sheet inside the document.
         */
        function beforeMoveSheetHandler() {
            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.cancelCellEditMode();
            }
        }

        /**
         * Finalizes moving a sheet inside the document.
         */
        function afterMoveSheetHandler(event, from, to) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet();

            // update the collection of visible sheets
            updateVisibleSheets();

            if (activeSheet === from) {
                // active sheet has been moved directly
                prepareSetActiveSheet();
                setActiveSheet(to);
            } else if ((from < activeSheet) && (activeSheet <= to)) {
                // active sheet has been moved backwards indirectly
                prepareSetActiveSheet();
                setActiveSheet(activeSheet - 1);
            } else if ((to <= activeSheet) && (activeSheet < from)) {
                // active sheet has been moved forwards indirectly
                prepareSetActiveSheet();
                setActiveSheet(activeSheet + 1);
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Handles a renamed sheet.
         */
        function renameSheetHandler() {

            // update the collection of visible sheets
            updateVisibleSheets();

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Updates the visible sheets collection, if the visibility of any
         * sheet in the document has been changed.
         */
        function changeSheetAttributesHandler(event, sheet, newAttributes, oldAttributes) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet(),
                // index of the active sheet in the (old) set of visible sheets
                activeVisibleSheet = self.getActiveSheet({ visible: true });

            // do nothing, if the visibility of the sheet has not been changed
            if (newAttributes.sheet.visible === oldAttributes.sheet.visible) { return; }

            // update the collection of visible sheets
            updateVisibleSheets();

            // activate another sheet, if the active sheet has been hidden
            if ((sheet === activeSheet) && !newAttributes.sheet.visible && (visibleSheets.length > 0)) {
                activeVisibleSheet = Utils.minMax(activeVisibleSheet, 0, visibleSheets.length - 1);
                self.setActiveSheet(activeVisibleSheet, { visible: true });
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Handles all tracking events for the split separators.
         */
        function splitTrackingHandler(event) {

            var // the event source node
                sourceNode = $(this),
                // whether column split tracking and/or row split tracking is active
                colSplit = sourceNode.hasClass('columns'),
                rowSplit = sourceNode.hasClass('rows'),
                // minimum and maximum position of split lines
                minLeft = self.getHeaderWidth(),
                maxLeft = rootNode.width() - Utils.SCROLLBAR_WIDTH - DYNAMIC_SPLIT_SIZE,
                minTop = self.getHeaderHeight(),
                maxTop = rootNode.height() - Utils.SCROLLBAR_HEIGHT - DYNAMIC_SPLIT_SIZE;

            // returns the X position of a split line according to the current event
            function getTrackingSplitLeft() {
                var offset = paneSideSettings.left.offset + paneSideSettings.left.size + (colSplit ? event.offsetX : 0);
                return (offset - PaneUtils.MIN_PANE_SIZE < minLeft) ? minLeft : (offset + PaneUtils.MIN_PANE_SIZE > maxLeft) ? maxLeft : offset;
            }

            // returns the Y position of a split line according to the current event
            function getTrackingSplitTop() {
                var offset = paneSideSettings.top.offset + paneSideSettings.top.size + (rowSplit ? event.offsetY : 0);
                return (offset - PaneUtils.MIN_PANE_SIZE < minTop) ? minTop : (offset + PaneUtils.MIN_PANE_SIZE > maxTop) ? maxTop : offset;
            }

            function updateNodeOffset(node, propName, offset, min, max) {
                node.toggleClass('collapsed', (offset <= min) || (offset >= max))
                    .add(centerSplitTrackingNode)
                    .css(propName, (offset - TRACKING_OFFSET) + 'px');
            }

            function finalizeTracking() {
                allSplitTrackingNodes.removeClass('tracking-active');
                self.grabFocus();
            }

            switch (event.type) {

            case 'tracking:start':
                colSplitTrackingNode.toggleClass('tracking-active', colSplit);
                rowSplitTrackingNode.toggleClass('tracking-active', rowSplit);
                break;

            case 'tracking:move':
                if (colSplit) { updateNodeOffset(colSplitTrackingNode, 'left', getTrackingSplitLeft(), minLeft, maxLeft); }
                if (rowSplit) { updateNodeOffset(rowSplitTrackingNode, 'top', getTrackingSplitTop(), minTop, maxTop); }
                break;

            case 'tracking:end':
                var splitWidth = null, splitHeight = null;
                if (colSplit) {
                    splitWidth = getTrackingSplitLeft();
                    splitWidth = ((minLeft < splitWidth) && (splitWidth < maxLeft)) ? activeSheetModel.convertPixelToHmm(splitWidth - self.getHeaderWidth()) : 0;
                } else {
                    splitWidth = activeSheetModel.getSplitWidthHmm();
                }
                if (rowSplit) {
                    splitHeight = getTrackingSplitTop();
                    splitHeight = ((minTop < splitHeight) || (splitHeight < maxTop)) ? activeSheetModel.convertPixelToHmm(splitHeight - self.getHeaderHeight()) : 0;
                } else {
                    splitHeight = activeSheetModel.getSplitHeightHmm();
                }
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight);
                finalizeTracking();
                break;

            case 'tracking:cancel':
                initializePanes();
                finalizeTracking();
                break;
            }
        }

        /**
         * Handles 'keydown' events from the view root node.
         */
        function keyDownHandler(event) {

            // do nothing while in-place edit mode is active
            if (self.isCellEditMode()) { return; }

            // special handling for drawing selection
            if (self.hasDrawingSelection()) {

                // delete selected drawings
                if (KeyCodes.matchKeyCode(event, 'DELETE') || KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                    if (self.requireEditMode()) { self.deleteDrawings(); }
                    return false;
                }

                return;
            }

            // enter cell edit mode with current cell contents
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                self.enterCellEditMode();
                return false;
            }

            // enter cell edit mode with empty text area
            if (KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                self.enterCellEditMode({ text: '' });
                return false;
            }

            // clear contents (not formatting) of all sells in the selection
            if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                if (self.requireEditMode()) { self.fillCellRanges(null); }
                return false;
            }

            // selects a drawing, if there is at least one
            if (KeyCodes.matchKeyCode(event, 'F4', { shift: true })) {
                if (drawingCollection.getModelCount() > 0) {
                    self.selectDrawing([0]);
                    self.getActiveGridPane().scrollToDrawingFrame([0]);
                    return false;
                }
            }
        }

        /**
         * Handles 'keypress' events from the view root node. Starts the cell
         * in-place edit mode on-the-fly for valid Unicode characters.
         */
        function keyPressHandler(event) {

            var // the initial text to be inserted into the text area
                initialText = null;

            // do not handle 'keypress' events bubbled up from the active in-place text
            // area, or if drawings are selected
            if (self.isCellEditMode() || self.hasDrawingSelection()) { return; }

            // ignore key events where either CTRL/META or ALT is pressed, ignore
            // SPACE keys with any control keys (used as shortcuts for column/row selection)
            if (((event.charCode > 32) && Utils.boolEq(event.ctrlKey || event.metaKey, event.altKey)) || ((event.charCode === 32) && !KeyCodes.hasModifierKeys(event))) {

                // build string from character code
                initialText = String.fromCharCode(event.charCode);

                // percentage number format: add percent sign to number-like first character
                if ((self.getNumberFormatCategory() === 'percent') && (/^[-0-9]$/.test(initialText) || (initialText === app.getDecimalSeparator()))) {
                    initialText += '%';
                }

                // start cell edit mode, and drop the key event
                self.enterCellEditMode({ quick: true, text: initialText, pos: 1 });
                return false;
            }
        }

        /**
         * configure global search for OX Spreadsheet
         */
        function initSearch() {

            var // helper variables
                win = app.getWindow(),
                search = win.search,
                controller = app.getController(),
                // the ox search panel
                searchDiv = win.nodes.search,
                searchForm = searchDiv.find('form'),
                searchInput = searchForm.find('input[name="query"]'),
                searchButton = searchForm.find('button[data-action="search"]'),
                // additional controls for search & replace
                nextButton = $('<button class="btn btn-default margin-right search-next" style="border-radius:0 4px 4px 0;vertical-align:top;" tabindex="1">' + Forms.createIconMarkup('fa-chevron-right') + '</button>'),
                previousButton = $('<button class="btn btn-default search-prev" style="border-radius:4px 0 0 4px;vertical-align:top;" tabindex="1">' + Forms.createIconMarkup('fa-chevron-left') + '</button>'),
                replaceInput = $('<input type="text" class="form-control" name="replace" tabindex="1">').attr('placeholder', gt('Replace with ...')),
                replaceButton = $('<button class="btn btn-default" style="border-radius:0;" tabindex="1">').text(gt('Replace')),
                replaceAllButton = $('<button class="btn btn-default" tabindex="1">').text(/*#. in search&replace: replace all occurrences */ gt('Replace all')),
                clearReplaceInputIcon = $(Forms.createIconMarkup('fa-times', { classes: 'clear-query' })),
                replaceInputGroup = $('<span class="input-group-btn">'),
                replaceQueryContainer =  $('<div class="search-query-container replace-query-container input-group" style="width:400px;display:inline-table;vertical-align:top;">');

            replaceInputGroup.append(replaceButton, replaceAllButton);
            replaceQueryContainer.append(replaceInput, clearReplaceInputIcon, replaceInputGroup);

            // add next/prev search hit,replace input, replace and replace all buttons
            searchForm.find('.search-query-container').css({ display: 'inline-table'})
                .after(previousButton, nextButton, replaceQueryContainer);

            // init replacement
            search.replacement = '';
            search.replacementInput = replaceInput;

            // Handles the 'search' event from the search panel.
            function searchHandler() {
                controller.executeItem('document/search/nextResult', search.query);
            }

            // when search bar is closed
            function searchCloseHandler() {
                search.clear();
                search.replacementInput.val('');
            }

            // Handles tab and esc key events
            function keyHandler(event) {
                var items, focus, index, down;

                if (event.keyCode === KeyCodes.TAB) {
                    event.preventDefault();

                    items = Forms.findFocusableNodes(this);
                    focus = $(document.activeElement);
                    down = ((event.keyCode === KeyCodes.TAB) && !event.shiftKey);

                    index = (items.index(focus) || 0);
                    if (!down) {
                        index--;
                    } else {
                        index++;
                    }

                    if (index >= items.length) {
                        index = 0;
                    } else if (index < 0) {
                        index = items.length - 1;
                    }
                    items[index].focus();
                    return false;
                }
            }

            // Handles the 'change' event of the search input.
            // Stops the propagation of the 'change' event to all other attached handlers
            // if the focus stays inside the search panel.
            // With that moving the focus out of the search input e.g. via tab
            // doesn't trigger the search and so navigation to other search panel controls
            // is possible.
            function searchInputChangeHandler(event) {
                event.stopImmediatePropagation();
                // if we stop the event propagation we need to update the query here
                search.query = search.getQuery();
                return false;
            }

            function editmodeChangeHandler(event, editMode) {
                replaceButton.toggleClass('disabled', !editMode);
                replaceAllButton.toggleClass('disabled', !editMode);
            }

            // set the tool tips
            Forms.setToolTip(searchInput, gt('Find text'), 'bottom');
            Forms.setToolTip(searchInput.siblings('i'), gt('Clear'), 'bottom');
            Forms.setToolTip(searchForm.find('button[data-action=search]'), gt('Start search'), 'bottom');
            Forms.setToolTip(previousButton, gt('Select previous search result'), 'bottom');
            Forms.setToolTip(nextButton, gt('Select next search result'), 'bottom');
            Forms.setToolTip(replaceInput, gt('Replacement text'), 'bottom');
            Forms.setToolTip(replaceButton, gt('Replace selected search result and select the next result'), 'bottom');
            Forms.setToolTip(replaceAllButton, gt('Replace all search results'), 'bottom');

            // event handling
            self.listenTo(searchInput, 'keydown', function (event) {
                if (KeyCodes.matchKeyCode(event, 'ENTER')) {
                    // prevent that enter key causing lost of focus
                    event.preventDefault();
                    // do the search
                    searchHandler();
                }
            });
            self.listenTo(searchButton, 'click', searchHandler);
            self.listenTo(previousButton, 'click', function () { controller.executeItem('document/search/previousResult'); });
            self.listenTo(nextButton, 'click', function () { controller.executeItem('document/search/nextResult'); });
            self.listenTo(replaceButton, 'click', function () { controller.executeItem('document/search/replaceSelected', { query: search.query, replace: search.replacement }); });
            self.listenTo(replaceAllButton, 'click', function () { controller.executeItem('document/search/replaceAll', { query: search.query, replace: search.replacement }); });
            self.listenTo(searchDiv, 'keydown', keyHandler);
            self.listenTo(replaceInput, 'change', function () { search.replacement = Utils.cleanString(this.value || ''); });
            self.listenTo(clearReplaceInputIcon, 'click', function () { replaceInput.val(''); search.replacement = ''; });

            // TODO: We should make the search able to handle tab between multiple controls in the search bar,
            // registering as first handler to the 'change' event is just an interim solution.
            Utils.bindAsFirstHandler(searchInput, 'change', searchInputChangeHandler, self);

            // subscribe the events handlers above
            self.listenTo(win, 'search:close', searchCloseHandler);

            // handle edit mode change situation
            self.listenTo(docModel, 'change:editmode', editmodeChangeHandler);
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            // insert the root node into the application pane
            self.insertContentNode(rootNode);

            // update view when document edit mode changes
            self.listenTo(docModel, 'change:editmode', function (event, editMode) {
                rootNode.toggleClass('edit-mode', editMode);
            });

            // insert resize overlay node first (will increase its z-index while active)
            rootNode.append(resizeOverlayNode);

            // insert the pane nodes into the DOM
            _.each(gridPanes, function (gridPane) { rootNode.append(gridPane.getNode()); });
            _.each(headerPanes, function (headerPane) { rootNode.append(headerPane.getNode()); });
            rootNode.append(cornerPane.getNode());

            // append the split lines and split tracking nodes
            rootNode.append(colSplitLineNode, rowSplitLineNode, colSplitTrackingNode, rowSplitTrackingNode, centerSplitTrackingNode);

            // enable tracking and register tracking event handlers
            allSplitTrackingNodes.enableTracking();
            allSplitTrackingNodes.on('tracking:start tracking:move tracking:end tracking:cancel', splitTrackingHandler);

            // handle generic key events
            rootNode.on({ keydown: keyDownHandler, keypress: keyPressHandler });

            // handle sheet collection events
            self.listenTo(docModel, 'insert:sheet:before', beforeInsertSheetHandler)
                .listenTo(docModel, 'insert:sheet:after', afterInsertSheetHandler)
                .listenTo(docModel, 'delete:sheet:before', beforeDeleteSheetHandler)
                .listenTo(docModel, 'delete:sheet:after', afterDeleteSheetHandler)
                .listenTo(docModel, 'move:sheet:before', beforeMoveSheetHandler)
                .listenTo(docModel, 'move:sheet:after', afterMoveSheetHandler)
                .listenTo(docModel, 'rename:sheet', renameSheetHandler)
                .listenTo(docModel, 'change:sheet:attributes', changeSheetAttributesHandler);

            // activate a sheet in the view after the document has been imported (also if import has failed)
            app.onImport(function () {

                var // index of the active sheet from document attributes
                    activeSheet = documentStyles.getDocumentAttributes().activeSheet,
                    // array index into the visibleSheets array
                    arrayIndex = Math.max(0, Math.min(visibleSheets.length - 1, _.sortedIndex(visibleSheets, { sheet: activeSheet }, 'sheet')));

                // initialize sheet view settings from the imported sheet attributes
                docModel.iterateSheetModels(function (sheetModel) {
                    sheetModel.initializeViewAttributes();
                });

                // activate a sheet (this causes initial painting)
                self.setActiveSheet(arrayIndex, { visible: true });
            });

            // create the status pane
            self.addPane(statusPane = new StatusPane(app));

            // repaint all grid and header panes on layout changes
            self.on('refresh:layout', initializePanes);

            // update locked state of active sheet in the DOM
            self.on('change:activesheet change:sheet:attributes', function () {
                rootNode.toggleClass('sheet-locked', activeSheetModel.isLocked());
            });

            // update CSS marker class for cell in-place edit mode
            self.on('celledit:enter celledit:leave', function () {
                rootNode.toggleClass('cell-edit', self.isCellEditMode());
            });

            // show error messages if leaviong cell edit mode fails
            self.on('celledit:reject', function (event, cause) {
                if (_.isString(cause)) { self.yellOnResult(cause); }
            });

            // process events triggered by header panes
            _.each(headerPanes, function (headerPane, paneSide) {

                var columns = PaneUtils.isColumnSide(paneSide),
                    offsetAttr = columns ? 'left' : 'top',
                    sizeAttr = columns ? 'width' : 'height',
                    showResizeTimer = null;

                function showResizerOverlayNode(offset, size) {
                    resizeOverlayNode.attr('data-orientation', columns ? 'columns' : 'rows');
                    updateResizerOverlayNode(offset, size);
                    // Delay displaying the overlay nodes, otherwise they cover
                    // the resizer drag nodes of the header pane which will
                    // interfere with double click detection.
                    showResizeTimer = self.executeDelayed(function () {
                        rootNode.addClass('tracking-active');
                    }, 200);
                }

                function updateResizerOverlayNode(offset, size) {
                    var relativeOffset = paneSideSettings[paneSide].offset + offset - headerPane.getVisiblePosition().offset;
                    resizeOverlayNode.toggleClass('collapsed', size === 0);
                    resizeOverlayNode.find('>.leading').css(sizeAttr, (Math.max(0, relativeOffset) + 10) + 'px');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, (relativeOffset + size) + 'px');
                }

                function hideResizerOverlayNode() {
                    if (showResizeTimer) {
                        showResizeTimer.abort();
                        showResizeTimer = null;
                    }
                    rootNode.removeClass('tracking-active');
                    resizeOverlayNode.find('>.leading').css(sizeAttr, '');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, '');
                }

                // visualize resize tracking from header panes
                headerPane.on({
                    'resize:start': function (event, offset, size) {
                        if (self.leaveCellEditMode('auto', { validate: true })) {
                            showResizerOverlayNode(offset, size);
                        } else {
                            event.preventDefault();
                        }
                    },
                    'resize:move': function (event, offset, size) {
                        updateResizerOverlayNode(offset, size);
                    },
                    'resize:end': function () {
                        hideResizerOverlayNode();
                    }
                });
            });

            // init global search for spreadsheet
            initSearch();
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            var cellStyleCollection = documentStyles.getStyleCollection('cell');

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoNode('version', { header: 'application', tooltip: 'Calc Engine version' })
                .addDebugInfoHeader('selection', { tooltip: 'Current sheet selection' })
                .addDebugInfoNode('sheet', { tooltip: 'Active (visible) sheet' })
                .addDebugInfoNode('ranges', { tooltip: 'Selected cell ranges' })
                .addDebugInfoNode('draw', { tooltip: 'Selected drawing objects' })
                .addDebugInfoNode('auto', { tooltip: 'Auto-fill range' })
                .addDebugInfoHeader('formatting', { tooltip: 'Formatting of the active cell' })
                .addDebugInfoNode('cell', { tooltip: 'Explicit cell attributes of active cell' })
                .addDebugInfoNode('char', { tooltip: 'Explicit character attributes of active cell' })
                .addDebugInfoNode('style', { tooltip: 'Cell style sheet of active cell' })
                .addDebugInfoNode('format', { tooltip: 'Number format settings of active cell' });

            // log the version of the Calc Engine
            app.onImport(function () {
                operationsPane.setDebugInfoText('application', 'version', app.getEngineVersion());
            });

            // log information about the active sheet
            self.on('change:activesheet change:sheets change:usedarea', function () {
                var used = (cellCollection && (cellCollection.getUsedCols() > 0) && (cellCollection.getUsedRows() > 0)) ? SheetUtils.getRangeName({ start: [0, 0], end: cellCollection.getLastUsedCell() }) : '<empty>';
                operationsPane.setDebugInfoText('selection', 'sheet', 'index=' + docModel.getActiveSheet() + ', name="' + self.getSheetName() + '", used=' + used);
            });

            // log all selection events
            self.on('change:sheet:viewattributes', function (event, attributes) {
                if ('selection' in attributes) {
                    var selection = attributes.selection;
                    operationsPane
                        .setDebugInfoText('selection', 'ranges', 'count=' + selection.ranges.length + ', ranges=' + SheetUtils.getRangesName(selection.ranges) + ', active=' + selection.activeRange + ', cell=' + SheetUtils.getCellName(selection.activeCell) + (selection.originCell ? (', origin=' + SheetUtils.getCellName(selection.originCell)) : ''))
                        .setDebugInfoText('selection', 'draw', 'count=' + selection.drawings.length + ', frames=' + ((selection.drawings.length > 0) ? ('[' + (selection.drawings).map(JSON.stringify).join() + ']') : '<none>'));
                }
                if ('autoFillData' in attributes) {
                    var autoFill = attributes.autoFillData;
                    operationsPane.setDebugInfoText('selection', 'auto', autoFill ? ('border=' + autoFill.border + ', count=' + autoFill.count) : '<none>');
                }
            });

            // log explicit formatting of the active cell
            self.on('change:layoutdata', function () {

                var cellContents = self.getCellContents(),
                    attributes = cellContents.explicit;

                function stringify(attrs) { return attrs ? Utils.stringifyForDebug(attrs) : ''; }
                operationsPane
                    .setDebugInfoText('formatting', 'cell', stringify(attributes.cell))
                    .setDebugInfoText('formatting', 'char', stringify(attributes.character))
                    .setDebugInfoText('formatting', 'style', _.isString(attributes.styleId) ? ('id="' + attributes.styleId + '", name="' + cellStyleCollection.getName(attributes.styleId) + '"') : '')
                    .setDebugInfoText('formatting', 'format', stringify(cellContents.format));
            });

            // process debug events triggered by grid panes
            _.each(gridPanes, function (gridPane) {
                gridPane.on('debug:clipboard', function (event, content) {
                    clipboardPane.setClipboardContent(content);
                });
            });
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler(viewMenuGroup) {

            var
                fontToolBar         = null,
                fontStyleToolBar    = null,
                painterToolBar      = null,
                colorToolBar        = null,
                cellToolBar         = null,
                numberformatToolBar = null,
                cellborderToolBar   = null,
                cellstyleToolBar    = null,
                alignmentToolBar    = null,
                dataToolBar         = null,
                insertToolBar       = null,
                rowsColsToolBar     = null,
                drawingToolBar      = null,
                chartStyleToolBar   = null,
                chartDataToolBar    = null,
                chartLegendToolBar  = null,
                chartSourceToolBar  = null,

                pickFontFamily      = new Controls.FontFamilyPicker(app),
                pickFontSize        = new Controls.FontSizePicker(),
                pickTextColor       = new Controls.TextColorPicker(app),
                pickFillColor       = new Controls.FillColorPicker(app),
                pickAlignmentHor    = new Controls.AlignmentPicker('horizontal'),
                pickAlignmentVer    = new Controls.AlignmentPicker('vertical'),
                pickMergeCells      = new Controls.MergeCellsPicker(app),
                pickFormatCategory  = new Controls.FormatCategoryPicker(app),
                pickFormatCode      = new Controls.FormatCodePicker(app),
                pickCellBorderMode  = new Controls.CellBorderModePicker(app, { dropDownVersion: { label: Labels.CELL_BORDERS_LABEL } }),
                pickBorderStyle     = new Controls.CellBorderStylePicker(app, { dropDownVersion: { label: /*#. line style (solid, dashed, dotted) of borders in paragraphs and tables cells */ gt.pgettext('borders', 'Border style') } }),
                pickCellBorderColor = new Controls.CellBorderColorPicker(app, { dropDownVersion: { label: Labels.BORDER_COLOR_LABEL } }),
                pickCellStyle       = new Controls.CellStylePicker(app),
                pickChartType       = new Controls.ChartTypePicker(app),
                pickChartColorSet   = new Controls.ChartColorSetPicker(app),
                pickChartStyleSet   = new Controls.ChartStyleSetPicker(app),
                pickChartLegend     = new Controls.ChartLegendPicker(),

                btnInsertChart      = new Controls.ChartTypePicker(app, Labels.INSERT_CHART_BUTTON_OPTIONS),
                btnInsertHyperlink  = new Controls.Button(Labels.HYPERLINK_BUTTON_OPTIONS),
                btnInsertImage      = new Controls.Button(Labels.INSERT_IMAGE_BUTTON_OPTIONS),
                btnInsertRow        = new Controls.Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }),
                btnInsertColumn     = new Controls.Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }),
                btnDeleteRow        = new Controls.Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }),
                btnDeleteColumn     = new Controls.Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }),
                btnDeleteDrawing    = new Controls.Button(Labels.DELETE_DRAWING_BUTTON_OPTIONS),
                btnBold             = new Controls.Button(Labels.BOLD_BUTTON_OPTIONS),
                btnItalic           = new Controls.Button(Labels.ITALIC_BUTTON_OPTIONS),
                btnUnderline        = new Controls.Button(Labels.UNDERLINE_BUTTON_OPTIONS),
                btnStrike           = new Controls.Button(Labels.STRIKEOUT_BUTTON_OPTIONS),
                btnReset            = new Controls.Button(Labels.CLEAR_FORMAT_BUTTON_OPTIONS),
                btnPainter          = new Controls.Button({ icon: 'docs-format-painter', tooltip: Labels.FORMAT_PAINTER_LABEL, toggle: true, dropDownVersion: { label: Labels.FORMAT_PAINTER_LABEL } }),
                btnLinebreak        = new Controls.Button({ icon: 'docs-insert-linebreak', tooltip: /*#. checkbox: automatic word wrapping (multiple text lines) in spreadsheet cells */ gt('Automatic word wrap'), toggle: true, dropDownVersion: { label: gt('Automatic word wrap') } }),
                btnFilter           = new Controls.Button({ icon: 'docs-filter', label: /*#. Button label: filter a cell range */ gt.pgettext('filter', 'Filter'), tooltip: gt('Filter the selected cells'), toggle: true, smallerVersion: {hideLabel: true} }),
                btnFilterRefresh    = new Controls.Button({ icon: 'fa-refresh', label: /*#. Button label: refresh a filtered cell range */ gt.pgettext('filter', 'Reapply'), tooltip: gt('Reapply the filter in the selected cells'), smallerVersion: {hideLabel: true}  }),
                btnSum              = new Controls.Button({ icon: 'docs-auto-sum', label: gt('Sum'), tooltip: /*#. automatically create a SUM function for selected cells */ gt('Sum automatically'), value: 'SUM', smallerVersion: { css: { width: 35 }, hideLabel: true } }),
                btnChartLabels      = new Controls.Button(_.extend(Labels.CHART_LABELS_BUTTON_OPTIONS, { 'aria-owns': chartLabelsMenu.getUid() })),
                btnChartExchange    = new Controls.Button({ icon: 'fa-none', label: /*#. switch orientation of data series in rectangular source data of a chart object */ gt.pgettext('chart-source', 'Switch rows and columns') }),

                labelRowSize        = new Controls.ColRowSizeLabel(false),
                labelColumnSize     = new Controls.ColRowSizeLabel(true),

                fieldRowSize        = new Controls.ColRowSizeField(false),
                fieldColumnSize     = new Controls.ColRowSizeField(true),

                cBoxChartDataLabel  = new Controls.CheckBox(Labels.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS),
                cBoxChartVaryColor  = new Controls.CheckBox(Labels.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS),
                cBoxChartSource     = new Controls.CheckBox({ label: /*#. change source data for a chart object in a spreadsheet */ gt.pgettext('chart-source', 'Edit data references') }),
                cBoxChartFirstRow   = new Controls.CheckBox({ label: /*#. decide if first row of source data is handled as label(headline) or as contentvalues */ gt.pgettext('chart-source', 'First row as label') }),
                cBoxChartFirstCol   = new Controls.CheckBox({ label: /*#. decide if first column of source data is handled as label or as contentvalues */ gt.pgettext('chart-source', 'First column as label') }),

                grpFormatCategory   = new Controls.FormatCategoryGroup(app),
                sortMenuBtn         = new Controls.SortMenuButton(app),

                // only for ODF
                pickBorderWidth     = null;

            // only for ODF
            if (!app.isOOXML()) {
                pickBorderWidth = new Controls.BorderWidthPicker({dropDownVersion: { label: Labels.BORDER_WIDTH_LABEL }});
            }


            // -----------------------------------------------------
            // TABS
            //      prepare all tabs (for normal or combined panes)
            // -----------------------------------------------------
            if (!self.panesCombined()) {
                self.createToolBarTab('format',         { label: Labels.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/cell' });
                self.createToolBarTab('data',           { label: /*#. tool bar title for sorting, filtering, and other data operations */ gt.pgettext('menu-title', 'Data'), visibleKey: 'document/editable/cell' });
                self.createToolBarTab('insert',         { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable/cell' });
                self.createToolBarTab('rowscols',       { label: /* menu title: insert/delete/resize rows and columns in the sheet */ gt.pgettext('menu-title', 'Rows/Columns'), visibleKey: 'document/editable/cell' });
                self.createToolBarTab('drawing',        { labelKey: 'drawing/type/label', visibleKey: 'document/editable/drawing' });

            } else {
                self.createToolBarTab('format',         { label: gt('Font'),                visibleKey: 'document/editable/cell' });
                self.createToolBarTab('alignment',      { label: gt('Alignment'),           visibleKey: 'document/editable/cell' });
                self.createToolBarTab('cell',           { label: gt('Cell'),                visibleKey: 'document/editable/cell' });
                self.createToolBarTab('numberformat',   { label: gt('Number format'),       visibleKey: 'document/editable/cell' });
                self.createToolBarTab('data',           { label: gt('Data'),                visibleKey: 'document/editable/cell' });
                self.createToolBarTab('rowscols',       { label: gt('Rows/Columns'),        visibleKey: 'document/editable/cell' });
                self.createToolBarTab('drawing',        { labelKey: 'drawing/type/label',   visibleKey: 'document/editable/drawing' });
            }

            // -----------------------------------------------------
            // TOOLBARS
            //      prepare all toolbars (for normal or combined panes)
            // -----------------------------------------------------
            fontToolBar                 = self.createToolBar('format', {priority: 1, prepareShrink: true, icon: 'fa-font', tooltip: Labels.FONT_STYLES_LABEL });
            fontStyleToolBar            = self.createToolBar('format', {priority: 2, prepareShrink: true, icon: 'docs-font-bold', caretTooltip: gt('More font styles'), splitBtn: { key: 'character/bold', options: Controls.BOLD_BUTTON_OPTIONS }});
            painterToolBar              = self.createToolBar('format', {priority: 3});
            if (!self.panesCombined()) {
                colorToolBar            = self.createToolBar('format', {priority: 4, prepareShrink: true, icon: 'docs-color', tooltip: gt('Colors') });
                cellToolBar             = self.createToolBar('format', {priority: 5, prepareShrink: true, icon: 'fa-align-center', tooltip: Labels.ALIGNMENT_HEADER_LABEL });
                numberformatToolBar     = self.createToolBar('format', {priority: 6, hideable: true, prepareShrink: true, icon: 'docs-percent', tooltip: Labels.NUMBERFORMAT_HEADER_LABEL });
                cellborderToolBar       = self.createToolBar('format', {priority: 7, hideable: true, prepareShrink: true, icon: 'docs-cell-style', tooltip: Labels.CELL_BORDER_LABEL });
                cellstyleToolBar        = self.createToolBar('format', {priority: 8, hideable: true});
            } else {
                alignmentToolBar        = self.createToolBar('alignment', {prepareShrink: true, icon: 'docs-cell-h-align-auto', tooltip: gt('Alignment')});
                cellToolBar             = self.createToolBar('cell', {prepareShrink: true, icon: 'fa-table', tooltip: gt('Cell')});
                numberformatToolBar     = self.createToolBar('numberformat', {prepareShrink: true, icon: 'fa-angle-double-down', tooltip: gt('Number format')});
            }
            dataToolBar                 = self.createToolBar('data');
            if (!self.panesCombined()) {
                insertToolBar           = self.createToolBar('insert');
            }
            rowsColsToolBar             = self.createToolBar('rowscols');
            drawingToolBar              = self.createToolBar('drawing');
            if (!self.panesCombined()) {
                chartStyleToolBar       = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 1 });
                chartDataToolBar        = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 2, hideable: true });
                chartLegendToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 3, hideable: true });
                chartSourceToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 4, hideable: true });
            }


            // -----------------------------------------------------
            // CONTROLS
            //      add all controls
            // -----------------------------------------------------
            // FORMAT ------------------------------------------
            fontToolBar
                .addGroup('character/fontname', pickFontFamily)
                .addGap()
                .addGroup('character/fontsize', pickFontSize);

            fontStyleToolBar
                .addGroup('character/bold', btnBold)
                .addGroup('character/italic', btnItalic)
                .addGroup('character/underline', btnUnderline);

            if (!self.panesCombined()) {
                fontStyleToolBar
                    .addSeparator({classes: 'noVerticalSeparator'});
            }

            fontStyleToolBar
                .addGroup('character/format', new Controls.CompoundButton(app, { icon: 'docs-font-format', tooltip: gt('More font styles'), dropDownVersion: {visible: false}})
                    .addGroup('character/strike', btnStrike)
                    .addSeparator()
                    .addGroup('cell/reset', btnReset)
                )
                .addGroup('character/strike', btnStrike.clone({dropDownVersion: {visible: true}}))
                .addSeparator({classes: 'hidden'})
                .addGroup('cell/reset', btnReset.clone({dropDownVersion: {visible: true}}));

            if (!self.panesCombined()) {
                painterToolBar
                    .addGroup('cell/painter', btnPainter);

                colorToolBar
                    .addGroup('character/color', pickTextColor)
                    .addGroup('cell/fillcolor', pickFillColor);

                cellToolBar
                    .addGroup('cell/alignhor', pickAlignmentHor)
                    .addGroup('cell/alignvert', pickAlignmentVer)
                    .addGroup('cell/linebreak', btnLinebreak)
                    .addGroup('cell/merge', pickMergeCells);

                numberformatToolBar
                    .addGroup('cell/numberformat/category', grpFormatCategory)
                    .addGroup('cell/numberformat/category', pickFormatCategory)
                    .addGroup('cell/numberformat/code', pickFormatCode);
            }

            if (self.panesCombined()) {
                painterToolBar
                    .addGroup('character/color', pickTextColor);
            }

            // ALIGNMENT ------------------------------------------
            if (self.panesCombined()) {
                alignmentToolBar
                    .addGroup('cell/alignhor', pickAlignmentHor)
                    .addGroup('cell/alignvert', pickAlignmentVer)
                    .addGroup('cell/linebreak', btnLinebreak)
                    .addGroup('cell/merge', pickMergeCells);
            }

            // BORDERS ------------------------------------------
            if (!self.panesCombined()) {
                if (app.isOOXML()) {
                    cellborderToolBar
                        .addGroup('cell/border/mode', pickCellBorderMode)
                        .addGroup('cell/border/style/preset', pickBorderStyle)
                        .addGroup('cell/border/color', pickCellBorderColor);

                } else {
                    cellborderToolBar
                        .addGroup('cell/border/mode', pickCellBorderMode)
                        .addGroup('cell/border/style', pickBorderStyle)
                        .addGroup('cell/border/width', pickBorderWidth)
                        .addGroup('cell/border/color', pickCellBorderColor);
                }
            }

            // CELLS ------------------------------------------
            if (!self.panesCombined()) {
                cellstyleToolBar
                    .addGroup('cell/stylesheet', pickCellStyle);
            }

            if (self.panesCombined()) {
                cellToolBar
                    .addGroup('cell/fillcolor', pickFillColor)
                    .addGroup('cell/border/mode', pickCellBorderMode)
                    .addGroup('cell/border/style', pickBorderStyle)
                    .addGroup('cell/border/color', pickCellBorderColor);

                numberformatToolBar
                    .addGroup('cell/numberformat/category', grpFormatCategory)
                    .addGroup('cell/numberformat/category', pickFormatCategory)
                    .addGroup('cell/numberformat/code', pickFormatCode);
            }

            // DATA ------------------------------------------
            if (self.panesCombined()) {
                dataToolBar
                    .addGroup('cell/autoformula', btnSum)
                    .addSeparator();
            }
            dataToolBar
                .addGroup('cell/sort', sortMenuBtn)
                .addSeparator()
                .addGroup('table/filter', btnFilter)
                .addGroup('table/refresh', btnFilterRefresh);

            // INSERT ------------------------------------------
            if (!self.panesCombined()) {
                insertToolBar
                    .addGroup('cell/autoformula', btnSum)
                    .addSeparator()
                    .addGroup('character/hyperlink/dialog', btnInsertHyperlink)
                    .addSeparator()
                    .addGroup('image/insert/dialog', btnInsertImage)
                    .addGap()
                    .addGroup('chart/insert', btnInsertChart);
            }

            rowsColsToolBar
                .addGroup('row/insert', btnInsertRow)
                .addGroup('row/delete', btnDeleteRow);

            if (!self.panesCombined()) {
                rowsColsToolBar
                    .addGroup('row/height/active', labelRowSize)
                    .addGroup('row/height/active', fieldRowSize);
            }

            rowsColsToolBar
                .addSeparator()
                .addGroup('column/insert', btnInsertColumn)
                .addGroup('column/delete', btnDeleteColumn);

            if (!self.panesCombined()) {
                rowsColsToolBar
                    .addGroup('column/width/active', labelColumnSize)
                    .addGroup('column/width/active', fieldColumnSize);
            }

            // DRAWING ------------------------------------------
            drawingToolBar
                .addGroup('drawing/delete', btnDeleteDrawing);

            if (!self.panesCombined()) {
                chartStyleToolBar
                    .addGroup('drawing/charttype', pickChartType)
                    .addGap()
                    .addGroup('drawing/chartlabels', btnChartLabels)
                    .addSeparator()
                    .addGroup('drawing/chartcolorset', pickChartColorSet)
                    .addGroup('drawing/chartstyleset', pickChartStyleSet);

                chartDataToolBar
                    .addGroup(null, new Controls.CompoundButton(app, Labels.CHART_DATA_POINTS_BUTTON_OPTIONS)
                        .addGroup('drawing/chartdatalabel', cBoxChartDataLabel)
                        .addGroup('drawing/chartvarycolor', cBoxChartVaryColor)
                    );

                chartLegendToolBar
                    .addGroup('drawing/chartlegend/pos', pickChartLegend);

                chartSourceToolBar
                    .addGroup(null, new Controls.CompoundButton(app, { label: /*#. menu title: options to modify the data source of a chart object */ gt.pgettext('chart-source', 'Data source') })
                        .addGroup('drawing/chartsource', cBoxChartSource)
                        .addGroup('drawing/chartexchange', btnChartExchange)
                        .addGroup('drawing/chartfirstrow', cBoxChartFirstRow)
                        .addGroup('drawing/chartfirstcol', cBoxChartFirstCol)
                    );
            }


            // the 'View' drop-down menu
            viewMenuGroup
                .addSectionLabel(Labels.ZOOM_LABEL)
                .addGroup('view/zoom/dec', new Controls.Button(Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom/inc', new Controls.Button(Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom', new Controls.PercentageLabel(), { inline: true })
                .addSectionLabel(/*#. menu title: settings for splits and frozen columns/rows in a spreadsheet */ gt.pgettext('sheet-split', 'Split and freeze'))
                .addGroup('view/split/dynamic', new Controls.DynamicSplitCheckBox())
                .addGroup('view/split/frozen', new Controls.FrozenSplitCheckBox(app))
                .addSectionLabel(Labels.OPTIONS_LABEL)
                // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                .addGroup('view/toolbars/show', new Controls.CheckBox(Labels.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                .addGroup('document/users', new Controls.CheckBox(Labels.SHOW_COLLABORATORS_CHECKBOX_OPTIONS))
                .addGroup('view/grid/show', new Controls.CheckBox({ label: /*#. check box label: show/hide cell grid in the sheet */ gt('Show grid lines'), tooltip: gt('Show or hide the cell grid lines in the current sheet') }))
                .addGroup('view/statuspane/show', new Controls.CheckBox({ label: /*#. check box label: show/hide the sheet tabs at the bottom border */ gt('Show sheet tabs'), tooltip: gt('Show or hide the sheet tabs below the sheet area') }));
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         */
        function initDebugGuiHandler(viewMenuGroup, actionMenuGroup) {

            // add debug actions to the drop-down menu
            actionMenuGroup
                .addSectionLabel(_.noI18n('View update'))
                .addGroup('debug/view/update', new Button({ label: _.noI18n('Request full view update'),              value: 'view' }))
                .addGroup('debug/view/update', new Button({ label: _.noI18n('Request view update of selected cells'), value: 'cells' }))
                .addGroup('debug/view/update', new Button({ label: _.noI18n('Request update of selection data'),      value: 'selection' }))
                .addSectionLabel(_.noI18n('Drawings'))
                .addGroup('debug/insert/drawing', new Button({ label: _.noI18n('Insert a generic drawing object') }));
        }

        /**
         * Moves the browser focus into the active sheet pane.
         */
        function grabFocusHandler() {
            // prevent JS errors when importing the document fails, and the view does not have activated a sheet yet
            if (activeSheetModel) {
                self.getActiveGridPane().grabFocus();
            }
        }

        /**
         * Refreshes the visibility, position, and size of all header panes,
         * grid panes, and tracking nodes.
         */
        var initializePanes = app.createDebouncedActionsMethod(this, function () {

            // view layout updates may be requested in case the import has failed, and no sheet is available
            if (!activeSheetModel) { return; }

            var // the size of the header corner node (and thus of the row/column header nodes)
                headerWidth = 0, headerHeight = 0,
                // whether frozen split mode is active
                frozenSplit = self.hasFrozenSplit(),
                // whether dynamic split mode is really active
                dynamicSplit = !frozenSplit && self.hasSplit(),
                // the size of the split lines
                splitLineSize = frozenSplit ? FROZEN_SPLIT_SIZE : DYNAMIC_SPLIT_SIZE,
                // whether the split lines are visible
                colSplit = false, rowSplit = false,
                // start position of the split lines
                splitLineLeft = activeSheetModel.getSplitWidth(),
                splitLineTop = activeSheetModel.getSplitHeight();

            // expand dynamic splits to minimum split size
            if (dynamicSplit) {
                if (splitLineLeft > 0) { splitLineLeft = Math.max(splitLineLeft, PaneUtils.MIN_PANE_SIZE); }
                if (splitLineTop > 0) { splitLineTop = Math.max(splitLineTop, PaneUtils.MIN_PANE_SIZE); }
            }

            // whether the left and top panes are visible
            paneSideSettings.left.visible = (dynamicSplit || frozenSplit) && (splitLineLeft > 0);
            paneSideSettings.top.visible = (dynamicSplit || frozenSplit) && (splitLineTop > 0);

            // calculate current size of header nodes
            // TODO: use current maximum row index visible in the panes
            cornerPane.initializePaneLayout(docModel.getMaxRow());
            headerWidth = self.getHeaderWidth();
            headerHeight = self.getHeaderHeight();

            // calculate inner width of left panes
            if (paneSideSettings.left.visible) {
                paneSideSettings.left.offset = headerWidth;
                paneSideSettings.left.size = splitLineLeft;
                paneSideSettings.left.hiddenSize = frozenSplit ? colCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorLeft')) : 0;
            } else {
                _.extend(paneSideSettings.left, { offset: 0, size: 0, hiddenSize: 0 });
            }

            // calculate inner height of top panes
            if (paneSideSettings.top.visible) {
                paneSideSettings.top.offset = headerHeight;
                paneSideSettings.top.size = splitLineTop;
                paneSideSettings.top.hiddenSize = frozenSplit ? rowCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorTop')) : 0;
            } else {
                _.extend(paneSideSettings.top, { offset: 0, size: 0, hiddenSize: 0 });
            }

            // calculate effective position of split lines
            splitLineLeft = paneSideSettings.left.offset + paneSideSettings.left.size;
            splitLineTop = paneSideSettings.top.offset + paneSideSettings.top.size;

            // determine whether right and bottom panes are visible (must have enough room in split and frozen mode)
            paneSideSettings.right.visible = !paneSideSettings.left.visible || (splitLineLeft + splitLineSize + PaneUtils.MIN_PANE_SIZE + Utils.SCROLLBAR_WIDTH <= rootNode.width());
            paneSideSettings.bottom.visible = !paneSideSettings.top.visible || (splitLineTop + splitLineSize + PaneUtils.MIN_PANE_SIZE + Utils.SCROLLBAR_HEIGHT <= rootNode.height());

            // visibility of the split lines
            colSplit = paneSideSettings.left.visible && paneSideSettings.right.visible;
            rowSplit = paneSideSettings.top.visible && paneSideSettings.bottom.visible;

            // calculate the resulting grid pane positions and sizes
            if (paneSideSettings.right.visible) {
                paneSideSettings.right.offset = colSplit ? (splitLineLeft + splitLineSize) : headerWidth;
                paneSideSettings.right.size = rootNode.width() - paneSideSettings.right.offset;
                paneSideSettings.right.hiddenSize = frozenSplit ? (paneSideSettings.left.hiddenSize + paneSideSettings.left.size) : 0;
            } else if (paneSideSettings.left.visible) {
                paneSideSettings.left.size = rootNode.width() - headerWidth;
            }
            if (paneSideSettings.bottom.visible) {
                paneSideSettings.bottom.offset = rowSplit ? (splitLineTop + splitLineSize) : headerHeight;
                paneSideSettings.bottom.size = rootNode.height() - paneSideSettings.bottom.offset;
                paneSideSettings.bottom.hiddenSize = frozenSplit ? (paneSideSettings.top.hiddenSize + paneSideSettings.top.size) : 0;
            } else if (paneSideSettings.top.visible) {
                paneSideSettings.top.size = rootNode.height() - headerHeight;
            }

            // set frozen mode (left/top panes are not scrollable in frozen mode in their
            // own direction, e.g. left frozen panes are not scrollable to left/right)
            paneSideSettings.left.frozen = frozenSplit;
            paneSideSettings.right.frozen = false;
            paneSideSettings.top.frozen = frozenSplit;
            paneSideSettings.bottom.frozen = false;

            // set up scroll bar visibility in the opposite direction (e.g. left/right
            // scroll bars of top panes are hidden, if bottom panes are visible)
            paneSideSettings.left.showOppositeScroll = !paneSideSettings.right.visible;
            paneSideSettings.right.showOppositeScroll = true;
            paneSideSettings.top.showOppositeScroll = !paneSideSettings.bottom.visible;
            paneSideSettings.bottom.showOppositeScroll = true;

            // initialize the header panes
            _.each(headerPanes, function (headerPane, paneSide) {
                headerPane.initializePaneLayout(paneSideSettings[paneSide]);
            });

            // initialize the grid panes
            _.each(gridPanes, function (gridPane, panePos) {
                gridPane.initializePaneLayout(paneSideSettings[PaneUtils.getColPaneSide(panePos)], paneSideSettings[PaneUtils.getRowPaneSide(panePos)]);
            });

            // visibility and position of the split lines
            colSplitLineNode.toggle(colSplit).css({ left: splitLineLeft, width: splitLineSize });
            rowSplitLineNode.toggle(rowSplit).css({ top: splitLineTop, height: splitLineSize });

            // visibility and position of the split tracking nodes
            colSplitTrackingNode.toggle(dynamicSplit && colSplit).css({ left: splitLineLeft - TRACKING_OFFSET });
            rowSplitTrackingNode.toggle(dynamicSplit && rowSplit).css({ top: splitLineTop - TRACKING_OFFSET });
            centerSplitTrackingNode.toggle(dynamicSplit && colSplit && rowSplit).css({ left: splitLineLeft - TRACKING_OFFSET, top: splitLineTop - TRACKING_OFFSET });
        });

        // public methods -----------------------------------------------------

        /**
         * Returns the specified grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getGridPane = function (panePos) {
            return gridPanes[panePos];
        };

        /**
         * Returns the specified grid pane, if it is visible. Otherwise,
         * returns the nearest visible grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the preferred grid pane. If only one
         *  grid pane is visible, it will be returned regardless of this
         *  parameter. If one of the pane sides contained in the passed pane
         *  position is not visible, returns the grid pane from the other
         *  visible pane side (for example, if requesting the top-left pane
         *  while only bottom panes are visible, returns the bottom-left pane).
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getVisibleGridPane = function (panePos) {

            // jump to other column pane, if the pane side is hidden
            if (!paneSideSettings[PaneUtils.getColPaneSide(panePos)].visible) {
                panePos = PaneUtils.getNextColPanePos(panePos);
            }

            // jump to other row pane, if the pane side is hidden
            if (!paneSideSettings[PaneUtils.getRowPaneSide(panePos)].visible) {
                panePos = PaneUtils.getNextRowPanePos(panePos);
            }

            // now, panePos points to a visible grid pane (at least one pane is always visible)
            return gridPanes[panePos];
        };

        /**
         * Returns the grid pane that is currently focused.
         *
         * @returns {GridPane}
         *  The focused grid pane instance.
         */
        this.getActiveGridPane = function () {
            return this.getVisibleGridPane(activeSheetModel.getViewAttribute('activePane'));
        };

        /**
         * Invokes the passed iterator function for all grid panes contained in
         * this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all grid pane instances. Receives
         *  the following parameters:
         *  (1) {GridPane} gridPane
         *      The current grid pane instance.
         *  (2) {String} panePos
         *      The position identifier of the grid pane.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateGridPanes = function (iterator, options) {
            return _.any(gridPanes, function (gridPane, panePos) {
                return iterator.call(this, gridPane, panePos) === Utils.BREAK;
            }, Utils.getOption(options, 'context')) ? Utils.BREAK : undefined;
        };

        /**
         * Returns the specified header pane.
         *
         * @param {String} paneSide
         *  The identifier of the pane side ('left', 'right', 'top', 'bottom').
         *
         * @returns {HeaderPane}
         *  The header pane instance at the specified pane side.
         */
        this.getHeaderPane = function (paneSide) {
            return headerPanes[paneSide];
        };

        /**
         * Returns the horizontal header pane (left or right) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The horizontal header pane instance at the corresponding pane side.
         */
        this.getColHeaderPane = function (panePos) {
            return headerPanes[PaneUtils.getColPaneSide(panePos)];
        };

        /**
         * Returns the vertical header pane (top or bottom) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The vertical header pane instance at the corresponding pane side.
         */
        this.getRowHeaderPane = function (panePos) {
            return headerPanes[PaneUtils.getRowPaneSide(panePos)];
        };

        /**
         * Returns the first visible header pane in the specified direction
         * (for example, for columns returns the left header pane, if it is
         * visible, otherwise the right header pane).
         *
         * @param {Boolean} columns
         *  If set to true, returns the first visible column header pane (left
         *  or right); otherwise returns the first visible row header pane (top
         *  or bottom).
         *
         * @returns {HeaderPane}
         *  The first visible header pane instance in the specified direction.
         */
        this.getVisibleHeaderPane = function (columns) {
            return columns ?
                (paneSideSettings.left.visible ? headerPanes.left : headerPanes.right) :
                (paneSideSettings.top.visible ? headerPanes.top : headerPanes.bottom);
        };

        /**
         * Returns a header pane associated to the grid pane that is currently
         * focused.
         *
         * @param {Boolean} columns
         *  If set to true, returns the column header pane (left or right);
         *  otherwise returns the row header pane (top or bottom).
         *
         * @returns {HeaderPane}
         *  A header pane instance associated to the grid pane that is
         *  currently focused.
         */
        this.getActiveHeaderPane = function (columns) {
            var panePos = activeSheetModel.getViewAttribute('activePane');
            return columns ? this.getColHeaderPane(panePos) : this.getRowHeaderPane(panePos);
        };

        /**
         * Invokes the passed iterator function for all header panes contained
         * in this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all header pane instances.
         *  Receives the following parameters:
         *  (1) {HeaderPane} headerPane
         *      The current header pane instance.
         *  (2) {String} paneSide
         *      The identifier of the pane side ('left', 'top', etc.).
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateHeaderPanes = function (iterator, options) {
            return _.any(headerPanes, function (headerPane, paneSide) {
                return iterator.call(this, headerPane, paneSide) === Utils.BREAK;
            }, Utils.getOption(options, 'context')) ? Utils.BREAK : undefined;
        };

        /**
         * Returns the top-left corner pane.
         *
         * @returns {CornerPane}
         *  The corner pane instance.
         */
        this.getCornerPane = function () {
            return cornerPane;
        };

        /**
         * Returns the current width of the row header panes.
         *
         * @returns {Number}
         *  The current width of the row header panes, in pixels.
         */
        this.getHeaderWidth = function () {
            return cornerPane.getNode().outerWidth();
        };

        /**
         * Returns the current height of the column header panes.
         *
         * @returns {Number}
         *  The current height of the column header panes, in pixels.
         */
        this.getHeaderHeight = function () {
            return cornerPane.getNode().outerHeight();
        };

        /**
         * Returns the status pane containing the sheet tabs, and the subtotal
         * label.
         */
        this.getStatusPane = function () {
            return statusPane;
        };

        /**
         * Returns the floating menu for axis and label formatting of chart
         * objects.
         *
         * @returns {ChartLabelsMenu}
         *  The floating menu for axis and label formatting of chart objects.
         */
        this.getChartLabelsMenu = function () {
            return chartLabelsMenu;
        };

        /**
         * Returns the intersection of the passed column/row intervals with the
         * visible intervals of the header panes in the specified direction.
         *
         * @param {Object|Array} intervals
         *  A single column/row interval, or an array of intervals.
         *
         * @param {Boolean} columns
         *  Whether to return the column intervals of the left and right header
         *  panes (true), or the row intervals of the top and bottom header
         *  panes (false).
         *
         * @returns {Array}
         *  The parts of the passed column/row intervals which are covered by
         *  the respective visible header panes.
         */
        this.getVisibleIntervals = function (intervals, columns) {

            var // the resulting intervals covered by the header panes
                resultIntervals = [];

            // pick the intervals of all visible header panes in the correct orientation
            _.each(headerPanes, function (headerPane, paneSide) {
                if ((PaneUtils.isColumnSide(paneSide) === columns) && headerPane.isVisible()) {
                    resultIntervals = resultIntervals.concat(SheetUtils.getIntersectionIntervals(intervals, headerPane.getRenderInterval()));
                }
            });

            // unify intervals in case the header panes cover the same sheet area
            return SheetUtils.getUnifiedIntervals(resultIntervals);
        };

        /**
         * Shows a notification message box, if a corresponding warning message
         * text exists for the passed error code.
         *
         * @param {String} result
         *  The result code received from any document model operation.
         *
         * @param {String} [success]
         *  A special message for the success case. Will cause a message box
         *  with 'success' styling.
         *
         * @returns {Boolean}
         *  Whether the passed result is NOT an error code (success).
         */
        this.yellOnResult = function (result, success) {

            // show warning for existing message text, ignore other result codes
            if ((result === '') && _.isString(success)) {
                this.yell({ type: 'success', message: success });
            } else if (result in RESULT_MESSAGES) {
                this.yell({ type: 'info', message: RESULT_MESSAGES[result] });
            }

            // return whether the result is not an error code (empty string)
            return result === '';
        };

        /**
         * Shows a notification message box, if the passed promise rejects with
         * an error code with a corresponding warning message text.
         *
         * @param {jQuery.Promise} promise
         *  The promise of a Deferred object representing an asynchronous
         *  document model operation. If the promise will be rejected with a
         *  supported error code, a warning box will be shown to the user.
         *
         * @param {String} [success]
         *  A special message for the success case (if the promise has been
         *  resolved). Will cause a message box with 'success' styling.
         *
         * @returns {jQuery.Promise}
         *  The promise passed to this method, for convenience.
         */
        this.yellOnPromise = function (promise, success) {

            // show success message (empty string is used as indicator code for success)
            promise.done(function () {
                self.yellOnResult('', success);
            });

            // show error message (check that the promise rejects with a string)
            promise.fail(function (result) {
                if (_.isString(result)) { self.yellOnResult(result); }
            });

            // return the promise for convenience
            return promise;
        };

        /**
         * Tries to initiate an edit action. if the document is in read-only
         * mode, a notification will be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the document is posssible.
         */
        this.requireEditMode = function () {
            var editMode = docModel.getEditMode();
            if (!editMode) { app.rejectEditAttempt(); }
            return editMode;
        };

        /**
         * Returns the current status text label.
         *
         * @returns {String}
         *  The current status text.
         */
        this.getStatusLabel = function () {
            return statusText;
        };

        /**
         * Sets the passed text as current status text label, shown in the
         * status pane at the bottom border of the application pane.
         *
         * @param {String} text
         *  The new status text.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setStatusLabel = function (text) {
            if (statusText !== text) {
                statusText = text;
                app.getController().update();
            }
            return this;
        };

        /**
         * Returns the effective zoom factor to be used for the active sheet,
         * according to the value of the 'zoom' view attribute, and the type of
         * the current device. On touch devices, the effective zoom factor will
         * be increased for better usability.
         *
         * @returns {Number}
         *  The effective zoom factor for the active sheet.
         */
        this.getEffectiveZoom = function () {
            return activeSheetModel.getEffectiveZoom();
        };

        /**
         * Calculates the effective font size for the passed font size,
         * according to the current zoom factor.
         *
         * @param {Number} fontSize
         *  The original font size, in points.
         *
         * @returns {Number}
         *  The effective font size, in points.
         */
        this.getEffectiveFontSize = function (fontSize) {
            return activeSheetModel.getEffectiveFontSize(fontSize);
        };

        /**
         * Calculates the effective line height to be used in spreadsheet cells
         * for the passed character attributes, according to the current zoom
         * factor of this sheet.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height. See
         *  SheetModel.getEffectiveLineHeight() for details.
         *
         * @returns {Number}
         *  The line height for the passed character attributes, in pixels.
         */
        this.getEffectiveLineHeight = function (charAttributes) {
            return activeSheetModel.getEffectiveLineHeight(charAttributes);
        };

        /**
         * Returns the effective horizontal padding between cell grid lines and
         * the text contents of the cell for the current zoom factor.
         *
         * @returns {Number}
         *  The effective horizontal cell content padding, in points.
         */
        this.getEffectiveCellPadding = function () {
            return activeSheetModel.getEffectiveCellPadding();
        };

        /**
         * Returns the effective grid color of the active sheet. The automatic
         * color will be replaced with black.
         *
         * @param {Object} autoColor
         *  The color to be returned as replacement for the automatic color.
         *
         * @returns {Object}
         *  The effective grid color of the active sheet.
         */
        this.getEffectiveGridColor = function (autoColor) {
            return activeSheetModel ? activeSheetModel.getEffectiveGridColor(autoColor) : autoColor;
        };

        /**
         * Returns the effective font size for the current zoom factor to be
         * used for the column and row labels in all header panes.
         *
         * @returns {Number}
         *  The effective font size for all header panes, in points.
         */
        this.getHeaderFontSize = function () {
            return Utils.minMax(Utils.round(5 * this.getEffectiveZoom() + 6, 0.1), 8, 30);
        };

        // active sheet -------------------------------------------------------

        /**
         * Returns the number of visible sheets in this spreadsheet view.
         *
         * @returns {Number}
         *  The number of visible sheets in the document.
         */
        this.getVisibleSheetCount = function () {
            return visibleSheets.length;
        };

        /**
         * Returns the number of hidden sheets in this spreadsheet view that
         * can be made visible.
         *
         * @returns {Number}
         *  The number of hidden sheets in the document. Does NOT include the
         *  sheets with unsupported types that will always be hidden.
         */
        this.getHiddenSheetCount = function () {
            return hiddenSheets.length;
        };

        /**
         * Invokes the passed iterator function for all visible sheet contained
         * in this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all visible sheets. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The current sheet model instance.
         *  (2) {Number} index
         *      The zero-based sheet index in the collection of all visible
         *      sheets.
         *  (3) {String} sheetName
         *      The name of the sheet.
         *  (4) {Number} sheet
         *      The zero-based sheet index in the document model.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateVisibleSheets = function (iterator, options) {
            return Utils.iterateArray(visibleSheets, function (sheetInfo, index) {
                return iterator.call(this, docModel.getSheetModel(sheetInfo.sheet), index, sheetInfo.name, sheetInfo.sheet);
            }, options);
        };

        /**
         * Returns the zero-based index of the active sheet currently displayed
         * in this spreadsheet view. Optionally, the index of the active sheet
         * inside the collection of all visible sheets can be returned.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the index of the active sheet inside the
         *      collection of all visible sheets will be returned instead of
         *      the physical sheet index.
         *
         * @returns {Number}
         *  The zero-based index of the active sheet, depending on the passed
         *  options. Will return -1, if the index in the collection of visible
         *  sheets is requested, but the active sheet is currently hidden.
         */
        this.getActiveSheet = function (options) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet();

            // look up the index of the active sheet in the collection of visible sheets,
            // or return the plain sheet index of the active sheet
            return Utils.getBooleanOption(options, 'visible', false) ? getVisibleIndex(activeSheet) : activeSheet;
        };

        /**
         * Activates a sheet in this spreadsheet view instance.
         *
         * @param {Number} index
         *  The zero-based index of the new active sheet. If the specified
         *  sheet is currently hidden, it will not be activated.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the passed sheet index will be interpreted as
         *      index into the collection of all visible sheets, instead of the
         *      physical sheet index.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet, options) {

            var // the model of the new active sheet
                sheetModel = null;

            // get sheet index from the collection of visible sheets
            if (Utils.getBooleanOption(options, 'visible', false)) {
                sheet = ((0 <= sheet) && (sheet < visibleSheets.length)) ? visibleSheets[sheet].sheet : -1;
            }

            // the model of the new active sheet
            sheetModel = docModel.getSheetModel(sheet);

            // do nothing, if active sheet does not change, or if the sheet is hidden
            if (sheetModel && (sheetModel !== activeSheetModel) && (getVisibleIndex(sheet) >= 0)) {
                prepareSetActiveSheet();
                setActiveSheet(sheet);
            }

            return this;
        };

        /**
         * Activates the preceding visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activatePreviousSheet = function () {
            return this.setActiveSheet(this.getActiveSheet({ visible: true }) - 1, { visible: true });
        };

        /**
         * Activates the next visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activateNextSheet = function () {
            return this.setActiveSheet(this.getActiveSheet({ visible: true }) + 1, { visible: true });
        };

        /**
         * Activates a sheet for the import preview, while the document import
         * is still running.
         *
         * @internal
         *  Called from the preview handler of the application. MUST NOT be
         *  called directly.
         *
         * @param {Number} index
         *  The zero-based index of the initial active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        this.activatePreviewSheet = function (sheet) {

            var // the sheet model
                sheetModel = docModel.getSheetModel(sheet);

            // no preview, if the sheet does not exist
            if (sheetModel) {

                // initialize view settings (scroll position, selection, zoom, split)
                sheetModel.initializeViewAttributes();

                // try to activate the preview sheet (may fail, e.g. for hidden sheets)
                this.setActiveSheet(sheet);

                // bug 36070: preview sheet contains invalid formulas, need to refresh
                app.onImportSuccess(function () {
                    sheetModel.getCellCollection().fetchAllFormulaCells();
                });
            }

            // return whether the sheet has been activated successfully
            return docModel.getActiveSheet() >= 0;
        };

        /**
         * Returns the model instance of the active sheet.
         *
         * @returns {SheetModel}
         *  The model instance of the active sheet.
         */
        this.getSheetModel = function () {
            return activeSheetModel;
        };

        /**
         * Returns the rendering cache for the active sheet model.
         *
         * @returns {RenderCache}
         *  The rendering cache for the active sheet model.
         */
        this.getRenderCache = function () {
            return activeRenderCache;
        };

        /**
         * Returns the cell collection instance of the active sheet.
         *
         * @returns {CellCollection}
         *  The cell collection instance of the active sheet.
         */
        this.getCellCollection = function () {
            return cellCollection;
        };

        /**
         * Returns the column collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The column collection instance of the active sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the row collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The row collection instance of the active sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the merge collection instance of the active sheet.
         *
         * @returns {MergeCollection}
         *  The merge collection instance of the active sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        /**
         * Returns the table collection instance of the active sheet.
         *
         * @returns {TableCollection}
         *  The table collection instance of the active sheet.
         */
        this.getTableCollection = function () {
            return tableCollection;
        };

        /**
         * Returns the validation collection instance of the active sheet.
         *
         * @returns {ValidationCollection}
         *  The validation collection instance of the active sheet.
         */
        this.getValidationCollection = function () {
            return validationCollection;
        };

        /**
         * Returns the drawing collection instance of the active sheet.
         *
         * @returns {SheetDrawingCollection}
         *  The drawing collection instance of the active sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return activeSheetModel.convertPixelToHmm(length);
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return activeSheetModel.convertHmmToPixel(length);
        };

        /**
         * Returns the absolute position of the passed cell range in the active
         * sheet, in 1/100 of millimeters, as well as in pixels according to
         * the current sheet zoom factor.
         *
         * @param {Object} range
         *  The address of the cell range in the active sheet.
         *
         * @returns {Object}
         *  The absolute position and size of the range in the sheet. See
         *  SheetModel.getRangeRectangle() for details.
         */
        this.getRangeRectangle = function (range) {
            return activeSheetModel.getRangeRectangle(range);
        };

        /**
         * Returns the absolute position of the specified cell in the active
         * sheet, in 1/100 of millimeters, as well as in pixels according to
         * the current sheet zoom factor.
         *
         * @param {Number[]} address
         *  The address of the cell in the active sheet.
         *
         * @param {Object} [options]
         *  Optional parameters. See method SheetModel.getCellRectangle() for
         *  details.
         *
         * @returns {Object}
         *  The absolute position and size of the cell in the active sheet. See
         *  method SheetModel.getCellRectangle() for details.
         */
        this.getCellRectangle = function (address, options) {
            return activeSheetModel.getCellRectangle(address, options);
        };

        /**
         * Creates a localized sheet name that is not yet used in the document.
         *
         * @returns {String}
         *  A sheet name not yet used in this document.
         */
        this.generateUnusedSheetName = function () {

            var // the new sheet name
                sheetName = '',
                // zero-based sheet index for the new sheet name
                nameIndex = docModel.getSheetCount() - 1;

            // generate a valid name
            while ((sheetName.length === 0) || docModel.hasSheet(sheetName)) {
                sheetName = Labels.getSheetName(nameIndex);
                nameIndex += 1;
            }

            return sheetName;
        };

        // scrolling ----------------------------------------------------------

        /**
         * Scrolls the active grid pane to the specified cell. In frozen split
         * view, activates the grid pane that contains the specified cell
         * address, and scrolls it to the cell.
         *
         * @param {Number[]} address
         *  The address of the cell to be scrolled to.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address) {

            var // the frozen column and row interval
                colInterval = null, rowInterval = null,
                // preferred pane side identifiers
                colSide = null, rowSide = null,
                // the resulting grid pane that will be scrolled
                gridPane = null;

            if (this.hasFrozenSplit()) {
                // select appropriate grid pane according to the passed cell address
                colInterval = activeSheetModel.getSplitColInterval();
                colSide = (_.isObject(colInterval) && (address[0] <= colInterval.last)) ? 'left' : 'right';
                rowInterval = activeSheetModel.getSplitRowInterval();
                rowSide = (_.isObject(rowInterval) && (address[1] <= rowInterval.last)) ? 'top' : 'bottom';
                gridPane = this.getVisibleGridPane(PaneUtils.getPanePos(colSide, rowSide)).grabFocus();
            } else {
                gridPane = this.getActiveGridPane();
            }

            gridPane.scrollToCell(address);
            return this;
        };

        /**
         * Scrolls the active grid pane to the specified drawing frame. In
         * frozen split view, scrolls the bottom-right grid pane instead.
         *
         * @param {Number[]} position
         *  The position of the drawing frame to be scrolled to.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position) {

            var // the resulting grid pane that will be scrolled
                gridPane = this.hasFrozenSplit() ? this.getVisibleGridPane('bottomRight') : this.getActiveGridPane();

            gridPane.scrollToDrawingFrame(position);
            return this;
        };

        // clipboard operations -----------------------------------------------

        /**
         * Creates a client clipboard ID and resets the server clipboard ID.
         *
         * @returns {String}
         *  The client clipboard ID.
         */
        this.createClientClipboardId = function () {
            clipboardId = {client: Clipboard.createClientClipboardId(), server: null};
            return clipboardId.client;
        };

        /**
         * Returns the server clipboard ID if the client clipboard ID matches
         * the given one, otherwise null.
         *
         * @param {String} clientClipboardId
         *  The client clipboard ID.
         *
         * @returns {String|Null}
         *  The corresponding server clipboard ID.
         */
        this.getServerClipboardId = function (clientClipboardId) {
            return (clipboardId.client === clientClipboardId) ? clipboardId.server : null;
        };

        /**
         * Initiates the server side clipboard copy action.
         *
         * @param {Object} selection
         *  The current selection.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the copy action request.
         */
        this.clipboardServerCopy = function (selection) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet(),
                // source cell range address for copying into clipboard
                activeRange = selection.ranges[selection.activeRange],
                // the server request
                request = null;

            // send request to create a new clipboard object on the server
            request = app.sendActionRequest('copy', _.extend({ sheet: activeSheet }, activeRange));

            // enable busy blocker screen
            this.enterBusy();
            request.always(function () { self.leaveBusy(); });

            // store identifier of the server clipboard object
            request.done(function (result) {
                clipboardId.server = result.handle;
            });

            // return the promise of the request
            return request.promise().fail(function (response) {
                Utils.error('SpreadsheetView.clipboardServerCopy(): creating server clipboard failed:', response);
            });
        };

        /**
         * Initiates the server side clipboard paste action.
         * To avoid that operations are applied during the clipboard, the following steps are performed:
         * - enter busy state
         * - send paste request
         * - wait for paste request's  response
         * - wait for 'docs:update' event
         * - leave busy state again
         *
         * @param {Object} selection
         *  The current selection.
         *
         * @param {String} serverClipboardId
         *  The server clipboard id that identifies the data on the server side.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the paste action request.
         */
        this.clipboardServerPaste = function (selection, serverClipboardId) {

            var // index of the active sheet
                activeSheet = docModel.getActiveSheet(),
                // target cell range address for pasting clipboard data
                activeRange = selection.ranges[selection.activeRange],
                // the server request
                request = null;

            /*
             * Returns a promise of a Deferred object that will be resolved if
             * a 'docs:update' event that results from a clipboard paste will
             * be recieved within the next 60 seconds.
             *
             * @returns {jQuery.Promise}
             *  The promise of a Deferred object, that will be resolved if a
             *  'docs:update' notification has been received after at most 60
             *  seconds, or rejected otherwise.
             */
            function waitForPasteDocsUpdate() {

                var // the timer promise for the timeout
                    timer = null,
                    // the resulting Deferred object
                    def = $.Deferred();

                // event listener for the 'docs:update' notifications
                function updateHandler(data) {
                    // checks if the event is caused by the paste request
                    if (data.pasted) { def.resolve(); }
                }

                // listen to the next 'docs:update' events until a paste event has been received
                self.listenTo(app, 'docs:update', updateHandler);
                def.always(function () {
                    self.stopListeningTo(app, 'docs:update', updateHandler);
                });

                // set up the 60 seconds timeout to reject the Deferred object
                timer = self.executeDelayed(function () { def.reject(); }, 60000);
                def.always(function () { timer.abort(); });

                // reject edit attempt which will show an error alert to the user
                def.fail(function () { app.rejectEditAttempt('clipboardTimeout'); });

                return def.promise();
            }

            // send request for the clipboard data stored on the server
            request = app.sendActionRequest('paste', _.extend({ handle: serverClipboardId, sheet: activeSheet }, activeRange));

            // wait for the view update notification event
            request = request.then(waitForPasteDocsUpdate);

            // enable busy blocker screen
            this.enterBusy();
            request.always(function () { self.leaveBusy(); });

            // return the promise of the request
            return request.fail(function (response) {
                Utils.error('SpreadsheetView.clipboardServerPaste(): pasting server clipboard failed:', response);
            });
        };

        // initialization -----------------------------------------------------

        // early initialization after MVC construction
        app.onInit(function () {

            // store reference to document model and the style sheet containers
            docModel = app.getModel();
            documentStyles = docModel.getDocumentStyles();

            // create floating menus
            chartLabelsMenu = new ChartLabelsMenu(app, {
                anchorBox: function () { return self.getActiveGridPane().getScrollNode(); }
            });
        });

        // marker for touch devices and browser types
        Utils.addDeviceMarkers(rootNode);

        // create the header pane instances, initialize settings for pane sides
        _.each(['left', 'right', 'top', 'bottom'], function (paneSide) {
            paneSideSettings[paneSide] = _.clone(DEFAULT_PANE_SIDE_SETTINGS);
            headerPanes[paneSide] = new HeaderPane(app, paneSide);
        });

        // create the cell collections and grid pane instances
        _.each(['topLeft', 'topRight', 'bottomLeft', 'bottomRight'], function (panePos) {
            gridPanes[panePos] = new GridPane(app, panePos);
        });

        // create the corner pane instance
        cornerPane = new CornerPane(app);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.each(renderCaches, function (data, uid) { destroyRenderCache(uid); });
            _.invoke(gridPanes, 'destroy');
            _.invoke(headerPanes, 'destroy');
            cornerPane.destroy();
            chartLabelsMenu.destroy();
            gridPanes = headerPanes = cornerPane = statusPane = null;
            chartLabelsMenu = renderCaches = null;
            colSplitLineNode = rowSplitLineNode = resizeOverlayNode = null;
            colSplitTrackingNode = rowSplitTrackingNode = centerSplitTrackingNode = allSplitTrackingNodes = null;
            docModel = documentStyles = null;
            activeSheetModel = activeRenderCache = null;
            colCollection = rowCollection = mergeCollection = cellCollection = null;
            tableCollection = validationCollection = drawingCollection = null;
        });

    } // class SpreadsheetView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: SpreadsheetView });

});
