/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/cellrendermixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/canvaswrapper',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, CanvasWrapper, Color, Border, SheetUtils, PaneUtils, CellCollection, RenderUtils) {

    'use strict';

    var // maximum number of content cells to be rendered at once in the background loop (max. 300)
        MAX_RENDER_CELL_COUNT = 3 * Utils.PERFORMANCE_LEVEL,

        // shortcut for an empty rectangle
        EMPTY_RECTANGLE = { left: 0, top: 0, width: 0, height: 0 },

        // reduction of clipping area size at leading border for canvas rendering,
        // needed to protect the existing outer cell borders
        LEADING_CLIP_SIZE_REDUCTION = Math.floor(SheetUtils.MIN_CELL_SIZE / 2),

        // reduction of clipping area size at trailing border for canvas rendering,
        // needed to protect the existing outer cell borders
        TRAILING_CLIP_SIZE_REDUCTION = SheetUtils.MIN_CELL_SIZE - LEADING_CLIP_SIZE_REDUCTION;

    // mix-in class CellRenderMixin ===========================================

    /**
     * Mix-in class for the class GridPane which renders the cell contents,
     * cell formatting, and the cell grid lines into a single grid pane.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this renderer.
     *
     * @param {jQuery} cellLayerNode
     *  The cell layer DOM node of the grid pane extended with this mix-in.
     */
    function CellRenderMixin(app, cellLayerNode) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            docModel = null,
            docView = null,

            // the style sheet container and other collections of the document
            documentStyles = null,
            fontCollection = null,
            numberFormatter = null,

            // single canvas for all content layers (fill, border, grid) for better performance (especially tablets/phones)
            canvasWrapper = new CanvasWrapper(),

            // the layer containing outlines for table ranges
            tableLayerNode = $('<div class="grid-layer table-layer">'),

            // the text layer (container for the cell text elements)
            textLayerNode = $('<div class="grid-layer text-layer">'),

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null,

            // all existing cell DOM nodes, mapped by address key
            cellNodes = {},

            // all cells with DOM contents to be rendered in a background loop
            pendingCells = {};

        // private methods ----------------------------------------------------

        /**
         * Renders the grid lines between the cells into the passed rendering
         * context.
         */
        var gridLayerRenderer = RenderUtils.profileMethod('rendering grid layer', function (context, range) {

            var // rendering cache of the active sheet
                renderCache = docView.getRenderCache(),
                // the current grid color (use a dark gray instead of black for automatic grid color)
                gridColor = docView.getEffectiveGridColor({ type: 'rgb', value: '666666' }),
                // the path object containing all grid line positions
                path = context.createPath();

            // adds either all vertical or all horizontal grid lines to the path
            function addGridLines(columns) {

                var // the array index into cell addresses
                    addrIndex = columns ? 0 : 1,
                    // the column/row collection in the other direction (for intervals of merged ranges)
                    collection = columns ? docView.getRowCollection() : docView.getColCollection(),
                    // merged ranges spanning over multiple columns/rows (causing multiple line segments)
                    usedMergedRanges = _.filter(renderCache.getShrunkenMergedRanges(), function (mergedRange) {
                        return mergedRange.start[addrIndex] < mergedRange.end[addrIndex];
                    }),
                    // layer start offset in rendering direction
                    layerOffset = columns ? layerRectangle.top : layerRectangle.left,
                    // layer size in rendering direction
                    layerSize = columns ? layerRectangle.height : layerRectangle.width,
                    // the function to draw a line into the canvas
                    drawLineFunc = columns ?
                        function (x, y1, y2) { path.pushLine(x, y1, x, y2); } :
                        function (y, x1, x2) { path.pushLine(x1, y, x2, y); };

                // add the grid lines for all visible columns/rows to the path
                renderCache.iterateInterval(range, columns, function (entry) {

                    var // use the end position of the current column/row
                        offset = entry.offset + entry.size,
                        // all merged ranges that will break the grid line
                        coveredMergedRanges = _.filter(usedMergedRanges, function (mergedRange) {
                            return (mergedRange.start[addrIndex] <= entry.index) && (entry.index < mergedRange.end[addrIndex]);
                        }),
                        // the merged column/row intervals of the merged ranges
                        mergedIntervals = SheetUtils.getIntervals(coveredMergedRanges, !columns),
                        // the coordinates of the line segments, as flat array
                        lineCoords = null;

                    // calculate the line start/end coordinates
                    lineCoords = _.chain(mergedIntervals)
                        // get the pixel positions of the merged column/row intervals
                        .map(_.bind(collection.getIntervalPosition, collection))
                        // filter hidden merged ranges
                        .filter(function (position) { return position.size > 0; })
                        // adjust layer offset in positions and convert to (nested) integer array
                        .map(function (position) { return [position.offset, position.offset + position.size]; })
                        // convert to a flat array, add start and end position of the layer rectangle
                        .flatten().unshift(layerOffset).push(layerOffset + layerSize)
                        // get the resulting array
                        .value();

                    // draw all line segments
                    for (var index = 0, length = lineCoords.length; index < length; index += 2) {
                        if (lineCoords[index] < lineCoords[index + 1]) {
                            drawLineFunc(offset, lineCoords[index], lineCoords[index + 1]);
                        }
                    }
                });
            }

            // translate context by half a pixel to position the lines on whole pixels, paint everything with 20% opacity
            context.translate(-0.5, -0.5);
            context.setGlobalAlpha(0.2);

            // add all grid lines to the path, render all lines at once
            addGridLines(false);
            addGridLines(true);
            context.setLineStyle({ style: documentStyles.getCssColor(gridColor, 'line'), width: 1 });
            context.drawPath(path);
        });

        /**
         * Renders the fill styles of all cells into the passed rendering
         * context.
         */
        var fillLayerRenderer = RenderUtils.profileMethod('rendering fill layer', function (context, range) {

            var // rendering cache of the active sheet
                renderCache = docView.getRenderCache(),
                // all columns with an explicit fill color, mapped by column index
                colFillStyles = {},
                // all rows with an explicit fill color (or transparent), mapped by row index
                rowFillStyles = {},
                // all cells that need to be cleared (in order to delete default column/row formatting)
                clearCells = [],
                // all cells that need to be filled
                fillCells = [];

            // paints or clears a rectangle in the canvas (also fills the trailing pixel of the preceding left/top cells)
            function drawRect(left, top, width, height, fill) {
                if (fill) {
                    context.setFillStyle({ style: fill.css });
                    context.drawRect(left - 1, top - 1, width + 1, height + 1);
                } else {
                    context.clearRect(left - 1, top - 1, width + 1, height + 1);
                }
            }

            // paints the rectangle of the passed cell data
            function drawCellData(cellData) {
                var rect = cellData.rectangle;
                drawRect(rect.left, rect.top, rect.width, rect.height, cellData.fill);
            }

            // render entire filled columns
            renderCache.iterateInterval(range, true, function (colEntry) {
                if (colEntry.fill) {
                    drawRect(colEntry.offset, layerRectangle.top, colEntry.size, layerRectangle.height, colEntry.fill);
                    colFillStyles[colEntry.index] = colEntry.fill.css;
                }
            });

            // render entire filled rows (but skip rows without 'custom' flag)
            renderCache.iterateInterval(range, false, function (rowEntry) {
                if (rowEntry.custom) {
                    // render transparent rows too, need to clean areas with column formatting
                    drawRect(layerRectangle.left, rowEntry.offset, layerRectangle.width, rowEntry.size, rowEntry.fill);
                    rowFillStyles[rowEntry.index] = rowEntry.fill ? rowEntry.fill.css : null;
                }
            });

            // Collect all cell to be filled or cleared (transparent cells may need to clear the default column/row
            // background). Collect cells to be cleared in a separate array to be able to draw them before the filled
            // cells which want to expand the filled area by 1 pixel to the left/top into their predecessors.
            renderCache.iterateCells(range, function (cellData) {

                var // column and row index of the current cell
                    col = cellData.address[0],
                    row = cellData.address[1],
                    // current default fill color from column or row (already rendered above)
                    defaultFillStyle = (row in rowFillStyles) ? rowFillStyles[row] : (col in colFillStyles) ? colFillStyles[col] : null,
                    // CSS fill color of the current cell
                    cellFillStyle = cellData.fill ? cellData.fill.css : null,
                    // target array to push the current cell data object to
                    targetArray = cellData.fill ? fillCells : clearCells;

                // Bug 37041: Improve performance by checking the default column/row fill color (do not render
                // cells that do not change the existing fill color). Note that merged ranges must always be
                // rendered, because they cover other columns and/or rows with potentially different fill colors!
                if (cellData.mergedRange || (defaultFillStyle !== cellFillStyle)) {
                    targetArray.push(cellData);
                }
            });

            // first, clear all transparent cells (done before drawing all filled cells, otherwise
            // the trailing pixel of preceding filled cells would be cleared by these cells)
            _.each(clearCells, drawCellData);
            _.each(fillCells, drawCellData);
        });

        /**
         * Renders the border styles of all cells into the passed rendering
         * context.
         */
        var borderLayerRenderer = RenderUtils.profileMethod('rendering border layer', function (context, range) {

            var // matrix with resolved border formatting styles
                matrix = docView.getRenderCache().getBorderMatrix(range);

            _.each(matrix, function (entry) {
                _.each(entry.borders, function (border, pos) {

                    var vert = (pos === 'l') || (pos === 'r'),
                        offset = -((border.width / 2) % 1),
                        x = (pos === 'r' ? entry.x2 : entry.x1) - (vert ? 0 : 1),
                        y = (pos === 'b' ? entry.y2 : entry.y1) - (vert ? 1 : 0),
                        end = vert ? entry.y2 : entry.x2,
                        lineWidth = border.double ? Math.max(1, Math.round(border.width / 3)) : border.width,
                        pattern = Border.getBorderPattern(border.style, lineWidth),
                        path = context.createPath();

                    // initialize context line style for the border line
                    context.setLineStyle({ style: border.color.css, width: lineWidth, pattern: pattern });

                    // add all border lines to the path
                    if (border.double) {
                        var leadOffset = -Math.round((border.width - lineWidth) / 2 - offset) - offset,
                            trailOffset = leadOffset + Math.round(border.width / 3 * 2);
                        if (vert) {
                            path.pushLine(x + leadOffset, y, x + leadOffset, end).pushLine(x + trailOffset, y, x + trailOffset, end);
                        } else {
                            path.pushLine(x, y + leadOffset, end, y + leadOffset).pushLine(x, y + trailOffset, end, y + trailOffset);
                        }
                    } else if (vert) {
                        path.pushLine(x + offset, y, x + offset, end);
                    } else {
                        path.pushLine(x, y + offset, end, y + offset);
                    }

                    // draw the border
                    context.drawPath(path);
                });
            });
        });

        /**
         * Returns a copy of the passed column/row interval, which will be
         * expanded to include the next available visible entry in both
         * directions.
         *
         * @param {Object} interval
         *  The column/row interval to be expanded, with the zero-based index
         *  properties 'first' and 'last'.
         *
         * @param {Boolean} columns
         *  Whether the passed interval is a column interval (true), or a row
         *  interval (false).
         *
         * @returns {Object}
         *  The expanded interval. If no visible entry has been found before or
         *  after the passed interval, the respective index of the passed
         *  original interval will be returned.
         */
        function expandIntervalToNextVisible(interval, columns) {

            var // the column/row collection
                collection = columns ? docView.getColCollection() : docView.getRowCollection(),
                // preceding visible column/row entry
                prevEntry = collection.getPrevVisibleEntry(interval.first - 1),
                // following visible column/row entry
                nextEntry = collection.getNextVisibleEntry(interval.last + 1);

            return { first: prevEntry ? prevEntry.index : interval.first, last: nextEntry ? nextEntry.index : interval.last };
        }

        /**
         * Returns a copy of the passed range address, which will be expanded
         * to include the next available visible column and row in all four
         * directions.
         *
         * @param {Object} range
         *  The address of the range to be expanded.
         *
         * @returns {Object}
         *  The address of the expanded range. If no visible column or row has
         *  been found before or after the range, the respective original index
         *  will be returned in the address.
         */
        function expandRangeToNextVisible(range) {

            var // the expanded column and row intervals
                colInterval = expandIntervalToNextVisible(SheetUtils.getColInterval(range), true),
                rowInterval = expandIntervalToNextVisible(SheetUtils.getRowInterval(range), false);

            return SheetUtils.makeRangeFromIntervals(colInterval, rowInterval);
        }

        /**
         * Renders all content layers and the grid lines into the canvas
         * element. The grid lines will be rendered above the cell background
         * rectangles, but below the border lines and cell text contents.
         *
         * @param {Object|Array} renderRanges
         *  The cell ranges to be rendered in the canvas.
         */
        var renderCanvasLayers = RenderUtils.profileMethod('CellRenderer.renderCanvasLayers()', function (renderRanges) {

            var // the bounding range containing all rendering ranges
                boundRange = null;

            RenderUtils.log('origRanges=' + SheetUtils.getRangesName(_.getArray(renderRanges)));

            // restrict the rendered ranges to the layer range, and unify the ranges
            // (nothing to render, if the changed cells are outside the layer range)
            renderRanges = SheetUtils.getUnifiedRanges(SheetUtils.getIntersectionRanges(renderRanges, layerRange));
            if (renderRanges.length === 0) { return; }

            // expand all rendered ranges by one column/row in each direction to be able
            // to paint or delete thick borders covering the areas of adjacent cells (the
            // resulting ranges may overlap at their borders which is required for clipping)
            renderRanges = _.map(renderRanges, expandRangeToNextVisible);

            // restrict the final rendered ranges to the layer range again, get their bounding range
            renderRanges = SheetUtils.getIntersectionRanges(renderRanges, layerRange);
            boundRange = SheetUtils.getBoundingRange(renderRanges);

            // if the rendering ranges occupy more than 50% of the bounding range, render the entire bounding range
            if (SheetUtils.getCellCountInRanges(renderRanges) > (SheetUtils.getCellCount(boundRange) / 2)) {
                renderRanges = [boundRange];
            }

            // process all ranges (expand all rendered ranges by one column/row in each direction
            // to be able to paint or delete thick borders covering the areas of adjacent cells)
            RenderUtils.log('layerRange=' + SheetUtils.getRangeName(layerRange) + ' renderRanges=' + SheetUtils.getRangesName(renderRanges));
            _.each(renderRanges, function (renderRange) {

                var // the rectangle of the rendering range, for clipping
                    clipRect = docView.getRangeRectangle(renderRange);

                // reduce the clipping rectangle by half of the minimum cell size (which is the maximum
                // border width) in each direction to not clear the outer borders of the rendered range
                if (layerRange.start[0] < renderRange.start[0]) {
                    clipRect.left += LEADING_CLIP_SIZE_REDUCTION;
                    clipRect.width -= LEADING_CLIP_SIZE_REDUCTION;
                }
                if (renderRange.end[0] < layerRange.end[0]) {
                    clipRect.width -= TRAILING_CLIP_SIZE_REDUCTION;
                }
                if (layerRange.start[1] < renderRange.start[1]) {
                    clipRect.top += LEADING_CLIP_SIZE_REDUCTION;
                    clipRect.height -= LEADING_CLIP_SIZE_REDUCTION;
                }
                if (renderRange.end[1] < layerRange.end[1]) {
                    clipRect.height -= TRAILING_CLIP_SIZE_REDUCTION;
                }

                // clip to the rendering area in the canvas
                canvasWrapper.clip(clipRect, function () {

                    // clean the rendering area
                    canvasWrapper.clear();

                    // render all cell background rectangles
                    canvasWrapper.render(function (context) {
                        fillLayerRenderer(context, renderRange);
                    });

                    // do not render anything if the grid is invisible
                    if (docView.getSheetViewAttribute('showGrid')) {
                        canvasWrapper.render(function (context) {
                            gridLayerRenderer(context, renderRange);
                        });
                    }

                    // render all cell borders
                    canvasWrapper.render(function (context) {
                        borderLayerRenderer(context, renderRange);
                    });
                });
            });
        });

        /**
         * Creates elements for the outlines of all visible table ranges.
         */
        function renderTableOutlines() {

            var // range addresses of all tables
                tableRanges = [],
                // the HTML mark-up for the table layer
                markup = '';

            // collect all table ranges (expand auto-filter to available content range)
            docView.getTableCollection().iterateTables(function (tableModel) {
                var tableRange = tableModel.getRange(),
                    dataRange = tableModel.getDataRange();
                tableRanges.push(dataRange ? SheetUtils.getBoundingRange(tableRange, dataRange) : tableRange);
            });

            // create the HTML mark-up for all visible table ranges
            self.iterateRangesForRendering(tableRanges, function (range, index, rectangle) {
                markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"></div>';
            }, { alignToGrid: true });
            tableLayerNode[0].innerHTML = markup;
        }

        /**
         * Removes the cell node with the specified address key from the cell
         * DOM layer.
         *
         * @param {String} mapKey
         *  The unique address key of the cell.
         */
        function removeCellNode(mapKey) {
            // using a JS map that caches the cell nodes is much faster than querying the cell from the DOM
            if (mapKey in cellNodes) {
                cellNodes[mapKey].remove();
                delete cellNodes[mapKey];
            }
        }

        /**
         * Resets the complete rendering cache.
         */
        function resetPendingCells() {
            pendingCells = {};
        }

        /**
         * Renders the specified number of pending cells into the DOM cell
         * layer, and removes the information of the rendered cells from the
         * rendering cache.
         *
         * @param {Number} maxCount
         *  The maximum number of cells to be rendered during this invocation.
         */
        var renderPendingCells = RenderUtils.profileMethod('CellRenderer.renderPendingCells()', function (maxCount) {

            // Bug 31125: to prevent performance problems when rendering cells,
            // modifying the DOM (insert contents, change node size and position),
            // and accessing measurement properties of DOM elements (size, position)
            // MUST NOT be performed alternating. After any modification of the
            // DOM the browser will update its internal rendering tree before it
            // returns any measurement properties. That update may take a few dozen
            // milliseconds (especially Chrome and Internet Explorer), thus doing
            // that for each cell will result in a massive performance fall-off.

            // This method renders the specified number of cells, and performs each
            // step (create HTML mark-up, insert mark-up, post-process cells) for
            // all rendered cells at once, before continuing with the next step.

            var // the rendering cache used to resolve cell descriptors
                renderCache = docView.getRenderCache(),
                // sorted addresses of all pending cells
                sortedAddresses = _.values(pendingCells),
                // all cells to be rendered in this invocation
                renderCells = [],
                // the inner padding for text content nodes (according to zoom)
                cellPadding = docView.getEffectiveCellPadding(),
                // difference between cell width and maximum content width (cell padding left/right, 1px grid line)
                totalPadding = 2 * cellPadding + 1,
                // the HTML mark-up of all rendered cells
                markup = '';

            // STEP 1: sort the array of all pending cells by cell address (bug 31123)
            sortedAddresses.sort(SheetUtils.compareCells);

            // STEP 2: extract the specified number of cells from the cache of pending cells,
            // calculate DOM position of the cell, prepare a few CSS attribute values
            RenderUtils.takeTime('STEP 2: preparing ' + Math.min(maxCount, sortedAddresses.length) + ' of ' + sortedAddresses.length + ' cells', function () {
                _.all(sortedAddresses, function (address) {

                    var // the cell key
                        mapKey = SheetUtils.getCellKey(address),
                        // the cell descriptor from the rendering cache (may be null)
                        cellData = renderCache.getCellEntry(mapKey);

                    // remove the cell from the DOM and the cache of pending cells
                    removeCellNode(mapKey);
                    delete pendingCells[mapKey];

                    // insert cell data into the array of cells to be rendered now
                    // cell may have been deleted from the cache in the meantime)
                    if (cellData) {
                        renderCells.push(cellData);
                        maxCount -= 1;
                    }

                    // repeat until the specified number of cells has been found
                    return maxCount > 0;
                });
            });

            // early exit if no more cells are left
            if (renderCells.length === 0) { return; }

            // STEP 3: create HTML mark-up of the clip node, content node, and text contents (this part
            // uses fontCollection.getTextWidht(), modifications of the DOM are not allowed here!)
            RenderUtils.takeTime('STEP 3: create content mark-up for ' + renderCells.length + ' cells', function () {
                _.each(renderCells, function (cellData) {

                    var // style attribute value for the cell clip node
                        clipStyle = '',
                        // style attribute value for the cell content node (parent of the text spans)
                        contentStyle = 'padding-left:' + cellPadding + 'px;padding-right:' + cellPadding + 'px;',
                        // style attribute value for the text span (for now, only one span for entire text)
                        spanStyle = '',

                        // attribute map for the 'cell' attribute family
                        cellAttributes = cellData.attributes.cell,
                        // attribute map for the 'character' attribute family
                        charAttributes = cellData.attributes.character,

                        // the available width for the cell text
                        availableWidth = Math.max(2, cellData.rectangle.width - totalPadding),
                        // whether to show negative numbers in red text color
                        redText = cellData.format.red && CellCollection.isNumber(cellData) && (cellData.result < 0),
                        // CSS text color, calculated from text color and fill color attributes
                        textColor = documentStyles.getCssTextColor(redText ? Color.RED : charAttributes.color, [cellAttributes.fillColor]),
                        // CSS text decoration, calculated from underline and strike attributes
                        textDecoration = documentStyles.getCssTextDecoration(charAttributes),
                        // additional width for the left/right border of the clip node
                        clipData = cellData.clipData,
                        // the mark-up for the final text contents
                        textMarkup = null;

                    // generates the HTML mark-up of a single text span
                    function getSpanMarkup(text) {
                        return '<span style="' + spanStyle + '">' + Utils.escapeHTML(text) + '</span>';
                    }

                    // cache character attributes with effective font size and line height for current zoom factor in cell data
                    charAttributes = cellData.charAttributes = _.clone(charAttributes);
                    charAttributes.fontSize = docView.getEffectiveFontSize(charAttributes.fontSize);
                    charAttributes.lineHeight = docView.getEffectiveLineHeight(cellData.attributes.character);

                    // horizontal alignment will be set later (calculate optimal
                    // width of cells with justified alignment before setting it)
                    cellData.contentCss = { textAlign: PaneUtils.getCssTextAlignment(cellData) };

                    // add vertical alignment
                    switch (cellAttributes.alignVert) {
                    case 'top':
                        contentStyle += 'top:0;';
                        break;
                    case 'middle':
                        // alignment will be updated dynamically below (needs content node in DOM)
                        break;
                    case 'justify':
                        // alignment will be updated dynamically below (needs content node in DOM)
                        contentStyle += 'top:0;';
                        break;
                    default:
                        // 'bottom' alignment, and fall-back for unknown alignments
                        contentStyle += 'bottom:-1px;';
                    }

                    // create CSS formatting for character attributes
                    spanStyle += 'font-family:' + documentStyles.getCssFontFamily(charAttributes.fontName) + ';';
                    spanStyle += 'line-height:' + charAttributes.lineHeight + 'px;';
                    if (charAttributes.fontSize !== 11) { spanStyle += 'font-size:' + charAttributes.fontSize + 'pt;'; }
                    if (charAttributes.bold) { spanStyle += 'font-weight:bold;'; }
                    if (charAttributes.italic) { spanStyle += 'font-style:italic;'; }
                    if (textColor !== '#000000') { spanStyle += 'color:' + textColor + ';'; }
                    if (textDecoration !== 'none') { spanStyle += 'text-decoration:' + textDecoration + ';'; }

                    // create mark-up for the text contents
                    cellData.wrappedText = CellCollection.isWrappedText(cellData);
                    if (cellData.wrappedText) {
                        // automatic text wrapping: CSS's 'white-space:pre-wrap' does not work together with
                        // 'text-align:justify', need to process white-space dynamically and use 'white-space:normal' instead
                        contentStyle += 'white-space:normal;word-wrap:break-word;';
                        // create a span element for each text line
                        textMarkup = _.map(cellData.display.split(/\n/), function (textLine) {
                            textLine = Utils.cleanString(textLine).replace(/\s/g, ' ').replace(/^ /, '\xa0').replace(/ {2}/g, ' \xa0').replace(/ $/, '\xa0');
                            return getSpanMarkup(textLine);
                        }).join('<br>');
                    } else if (CellCollection.isText(cellData)) {
                        // text cells without wrapping: calculate available space for the cell text
                        // over preceding/following empty cells (but always crop merged ranges)
                        if (clipData) {
                            clipStyle += 'left:' + (-clipData.clipLeft) + 'px;right:' + (-clipData.clipRight) + 'px;';
                            contentStyle += 'left:' + (-clipData.hiddenLeft) + 'px;right:' + (-clipData.hiddenRight) + 'px;';
                        }
                        // remove line breaks from text, get optimal width
                        textMarkup = cellData.display.replace(/\n/g, '');
                        cellData.contentWidth = fontCollection.getTextWidth(textMarkup, charAttributes);
                        // create a single span element for the final text
                        textMarkup = getSpanMarkup(textMarkup);
                    } else {
                        // get width of display string according to character attributes
                        cellData.contentWidth = fontCollection.getTextWidth(cellData.display, charAttributes);
                        // use display text if it fits into the available space
                        if (cellData.contentWidth <= availableWidth) {
                            textMarkup = cellData.display;
                        }
                        // number cells with 'General' number format: reformat display string dynamically according to available space (TODO: for other dynamic formats too)
                        else if (CellCollection.isNumber(cellData) && _.isFinite(cellData.result) && (cellData.format.cat === 'standard') && (cellData.display !== CellCollection.PENDING_DISPLAY)) {
                            // will be null, if no valid display string can be found for the available width (e.g. cell too narrow)
                            textMarkup = numberFormatter.formatStandardNumber(cellData.result, SheetUtils.MAX_LENGTH_STANDARD_CELL, charAttributes, availableWidth);
                        }
                        // set railroad track error for cells where display string does not fit
                        if (!_.isString(textMarkup)) {
                            var hashCharWidth = fontCollection.getTextWidth('#', charAttributes);
                            textMarkup = Utils.repeatString('#', Math.max(1, Math.floor(availableWidth / hashCharWidth)));
                        }
                        // create a single span element for the final text
                        textMarkup = getSpanMarkup(textMarkup);
                    }

                    // create the clip node and the content node
                    cellData.contentMarkup = '<div class="clip" style="' + clipStyle + '"><div class="content" style="' + contentStyle + '">' + textMarkup + '</div></div>';
                });
            });

            // STEP 4: create HTML mark-up for all cells to be rendered
            RenderUtils.takeTime('STEP 4: create mark-up for ' + renderCells.length + ' cells', function () {
                _.each(renderCells, function (cellData) {

                    var // the address of the cell
                        address = cellData.address,
                        // the merged range the cell is part of
                        mergedRange = cellData.mergedRange,
                        // the location of the cell in the sheet, in pixels
                        rectangle = cellData.rectangle,
                        // style attribute value for the root node of the cell
                        cellStyle = '',
                        // the effective horizontal alignment
                        textAlign = PaneUtils.getCssTextAlignment(cellData);

                    // convert cell position relative to layer node
                    rectangle = _.clone(rectangle);
                    rectangle.left -= layerRectangle.left;
                    rectangle.top -= layerRectangle.top;

                    // restrict width of oversized cells to browser limit, according to horizontal alignment
                    if (rectangle.width > Utils.MAX_NODE_SIZE) {
                        var widthDiff = rectangle.width - Utils.MAX_NODE_SIZE,
                            reducedLeftInside = rectangle.left + widthDiff > 0,
                            reducedRightInside = rectangle.left + Utils.MAX_NODE_SIZE < layerRectangle.width;
                        switch (textAlign) {
                        case 'left':
                            // Reduce node width to keep cell contents left-aligned
                            // correctly. If the new right border will be located inside
                            // the layer rectangle after width reduction, adjust left border
                            // (which is far outside the layer rectangle in that case).
                            rectangle.left += reducedRightInside ? widthDiff : 0;
                            break;
                        case 'right':
                            // Adjust left offset of the node to keep cell contents
                            // right-aligned correctly, unless the new left border of
                            // the cell will be located inside the layer rectangle after
                            // width reduction (right border is far outside the layer
                            // rectangle in that case).
                            rectangle.left += reducedLeftInside ? 0 : widthDiff;
                            break;
                        default:
                            // Adjust left offset by half of the width correction to
                            // keep the centered text at its position, unless the left
                            // or right border of the cell will be located inside the
                            // layer rectangle after adjustment or width reduction.
                            rectangle.left += reducedRightInside ? widthDiff : reducedLeftInside ? 0 : Math.round(widthDiff / 2);
                        }
                        rectangle.width = Utils.MAX_NODE_SIZE;
                    }

                    // restrict height of oversized cells to browser limit, according to vertical alignment
                    if (rectangle.height > Utils.MAX_NODE_SIZE) {
                        var heightDiff = rectangle.height - Utils.MAX_NODE_SIZE,
                            reducedTopInside = rectangle.top + heightDiff > 0,
                            reducedBottomInside = rectangle.top + Utils.MAX_NODE_SIZE < layerRectangle.height;
                        switch (cellData.attributes.cell.alignVert) {
                        case 'top':
                            // Reduce node height to keep cell contents top-aligned
                            // correctly. If the new bottom border will be located inside
                            // the layer rectangle after height reduction, adjust top border
                            // (which is far outside the layer rectangle in that case).
                            rectangle.top += reducedBottomInside ? heightDiff : 0;
                            break;
                        case 'bottom':
                            // Adjust top offset of the node to keep cell contents
                            // bottom-aligned correctly, unless the new top border of
                            // the cell will be located inside the layer rectangle after
                            // height reduction (bottom border is far outside the layer
                            // rectangle in that case).
                            rectangle.top += reducedTopInside ? 0 : heightDiff;
                            break;
                        default:
                            // Adjust top offset by half of the height correction to
                            // keep the centered text at its position, unless the top
                            // or bottom border of the cell will be located inside the
                            // layer rectangle after adjustment or height reduction.
                            rectangle.top += reducedBottomInside ? heightDiff : reducedTopInside ? 0 : Math.round(heightDiff / 2);
                        }
                        rectangle.height = Utils.MAX_NODE_SIZE;
                    }

                    // add effective position and size of the cell to the style attribute
                    cellStyle += PaneUtils.getRectangleStyleMarkup(rectangle);

                    // generate the HTML mark-up for the cell node
                    markup += '<div class="cell" data-col="' + address[0] + '" data-row="' + address[1] + '" data-address="' + SheetUtils.getCellKey(address) + '"';
                    if (mergedRange) {
                        if (mergedRange.start[0] < mergedRange.end[0]) { markup += ' data-col-span="' + SheetUtils.getColCount(mergedRange) + '"'; }
                        if (mergedRange.start[1] < mergedRange.end[1]) { markup += ' data-row-span="' + SheetUtils.getRowCount(mergedRange) + '"'; }
                    }
                    if (cellData.wrappedText) { markup += ' data-wrapped="true"'; }
                    markup += ' style="' + cellStyle + '">';

                    // add text contents created in the previous step
                    markup += cellData.contentMarkup;

                    // close the root cell node
                    markup += '</div>';
                });
            });

            // STEP 5: insert the entire HTML mark-up into the cell layer node, extract DOM nodes
            // (this part modifies the DOM, reading node size properties is not allowed here!)
            textLayerNode.append(markup);
            _.each(renderCells, function (cellData) {
                var mapKey = SheetUtils.getCellKey(cellData.address);
                cellNodes[mapKey] = cellData.cellNode = textLayerNode.find('>[data-address="' + mapKey + '"]');
                cellData.contentNode = cellData.cellNode.find('>.clip>.content');
            });

            // STEP 6: post-processing: update vertical alignment, calculate optimal width of justified cells
            // (this part reads the size of DOM nodes, modifications of the DOM are not allowed here!)
            _.each(renderCells, function (cellData) {

                // update vertical alignment 'middle'
                function updateMiddleAlignment() {
                    cellData.contentCss.top = (cellData.rectangle.height - cellData.contentNode.height()) / 2;
                }

                // update vertical alignment 'justify'
                function updateJustifyAlignment() {

                    var // the total height of the content node (line height has been set to 100% above)
                        contentHeight = cellData.contentNode.height(),
                        // the effective font size of the cell, in pixels
                        fontSize = Utils.convertLength(cellData.charAttributes.fontSize, 'pt', 'px'),
                        // the effective line height, in pixels
                        normalLineHeight = cellData.charAttributes.lineHeight,
                        // the free space between text line border and characters, in pixels
                        normalLinePadding = (normalLineHeight - fontSize) / 2,
                        // the number of text lines
                        lineCount = Math.round(contentHeight / normalLineHeight),
                        // the free space between text line border and characters with target line height, in pixels
                        targetLinePadding = 0,
                        // the effective line height, in pixels
                        targetLineHeight = 0;

                    if (lineCount > 1) {
                        targetLinePadding = (cellData.rectangle.height - 2 * normalLinePadding - lineCount * fontSize) / (2 * lineCount - 2);
                        targetLineHeight = 2 * targetLinePadding + fontSize;
                    }
                    targetLineHeight = Math.max(targetLineHeight, normalLineHeight);
                    (cellData.spanCss || (cellData.spanCss = {})).lineHeight = Math.floor(targetLineHeight) + 'px';
                    cellData.contentCss.top = Math.round(normalLinePadding - targetLinePadding);
                    // store real (unexpanded) content height in 1/100 mm (used to calculate optimal row height)
                    cellData.optimalHeightHmm = documentStyles.getLineHeight(cellData.attributes.character) * lineCount;
                }

                // update vertical alignment dynamically
                switch (cellData.attributes.cell.alignVert) {
                case 'middle':
                    updateMiddleAlignment();
                    break;
                case 'justify':
                    if (cellData.wrappedText) { updateJustifyAlignment(); }
                    break;
                }

                // write the optimal width of wrapped text cells as data attribute into the cell
                if (cellData.wrappedText) {
                    cellData.contentWidth = _.reduce(cellData.contentNode[0].childNodes, function (memo, textSpan) {
                        return Math.max(memo, textSpan.offsetWidth);
                    }, 0);
                }
            });

            // STEP 7: insert CSS formatting created in the post-processing step
            // (this part modifies the DOM, reading node size properties is not allowed here!)
            _.each(renderCells, function (cellData) {

                // add content width (used to calculate optimal width of columns)
                if (_.isNumber(cellData.contentWidth) && (cellData.contentWidth > 0)) {
                    var contentWidth = cellData.contentWidth + totalPadding + 0.5;

                    // workaround for Bug 35162
                    // element width does not support subpixels, but canvas does support it,
                    // so the result is incompatible
                    if (!_.browser.iOS && _.browser.Safari) {
                        contentWidth++;
                    }

                    var optimalWidth = docView.convertPixelToHmm(contentWidth);
                    cellData.cellNode.attr('data-optimal-width', optimalWidth);
                }

                // add content height (used to calculate optimal height of rows)
                if (_.isNumber(cellData.optimalHeightHmm) && (cellData.optimalHeightHmm > 0)) {
                    cellData.cellNode.attr('data-optimal-height', cellData.optimalHeightHmm);
                }

                // additional CSS formatting attributes
                cellData.contentNode.css(cellData.contentCss);
                if (cellData.spanCss) {
                    cellData.contentNode.find('>span').css(cellData.spanCss);
                }
            });
        });

        /**
         * Renders all cells contained in the passed map or cell ranges.
         *
         * @param {Object|Array|Null} renderRanges
         *  The address of a cell range, or an array of cell range addresses of
         *  changed cell ranges, without needing to pass specific cell entries
         *  in the parameter 'changedCells'. May be null, if passing changed
         *  cells only in the parameter 'changedCells'.
         *
         * @param {Object|Null} changedCells
         *  Descriptors for all changed cells (cell entries from the rendering
         *  cache), mapped by cell key. Deleted (blank) cells are represented
         *  by null values in the map. May be null, if cell range addresses are
         *  passed in the parameter 'renderRanges'.

         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.immediate=0]
         *      The number of cells to be rendered immediately. All other
         *      remaining cells will be rendered in a background loop.
         */
        var renderChangedCells = (function () {

            var // the timer representing the background loop
                timer = null;

            // registers a new cell for debounced rendering
            function registerCell(mapKey, address) {
                // do not store the current cell data (may be changed/deleted until the cell will be rendered)
                pendingCells[mapKey] = address;
            }

            // returns whether the passed cell entry covers the own layer range
            function isInLayerRange(cellData) {
                return cellData.mergedRange ? SheetUtils.rangeOverlapsRange(layerRange, cellData.mergedRange) : SheetUtils.rangeContainsCell(layerRange, cellData.address);
            }

            // starts a background loop that renders a few cells at a time
            function renderCells() {
                // check whether the background loop already runs
                if (timer || _.isEmpty(pendingCells)) { return; }
                // create a new background loop
                timer = self.repeatDelayed(function () {
                    renderPendingCells(MAX_RENDER_CELL_COUNT);
                    if (_.isEmpty(pendingCells)) { return Utils.BREAK; }
                }, 10);
                // clean up after all cells have been rendered
                timer.always(function () { timer = null; });
            }

            // debounced method to register and render pending cells
            var registerContentCell = self.createDebouncedMethod(registerCell, renderCells);

            // the actual renderChangedCells() method
            function renderChangedCells(renderRanges, changedCells, options) {

                var // number of content cells to be rendered immediately
                    immediate = Utils.getIntegerOption(options, 'immediate', 0, 0);

                // ensure that 'renderRanges' is an array
                renderRanges = renderRanges ? _.getArray(renderRanges) : [];

                // collect all cell addresses and convert them to ranges (the map contains
                // null values for deleted cells, therefore convert map keys to addresses)
                renderRanges = renderRanges.concat(SheetUtils.joinCellsToRanges(_.chain(changedCells).keys().map(SheetUtils.parseCellKey).value())),

                // add all merged ranges
                _.each(changedCells, function (cellData) {
                    if (cellData && cellData.mergedRange) {
                        renderRanges.push(cellData.mergedRange);
                    }
                });

                // render the cell formatting into the canvas
                if (renderRanges.length > 0) {
                    renderCanvasLayers(renderRanges);
                    renderTableOutlines();
                }

                // register content cells for rendering in a background loop
                RenderUtils.takeTime('register content cells', function () {
                    _.each(changedCells, function (cellData, mapKey) {
                        if (cellData && cellData.display && isInLayerRange(cellData)) {
                            registerContentCell(mapKey, cellData.address);
                        } else {
                            removeCellNode(mapKey);
                        }
                    });
                });

                // render the first cells synchronously (looks better when entering multiple single cells manually one-by-one)
                if (immediate > 0) { renderPendingCells(immediate); }
            }

            return RenderUtils.profileMethod('CellRenderer.renderChangedCells()', renderChangedCells);
        }());

        /**
         * Updates the position, size, and text clipping settings of all
         * existing cell nodes in the DOM cell layer. Cell nodes for cells that
         * are not visible anymore will be removed from the cell layer.
         *
         * @param {Boolean} [refreshClip=false]
         *  If set to true, the text clipping settings will be updated too. By
         *  default, only the cell position and size will be updated.
         */
        var updateCellGeometry = RenderUtils.profileMethod('CellRenderer.updateCellGeometry()', function (refreshClip) {

            var // the rendering cache of the active sheet
                renderCache = docView.getRenderCache();

            // process all existing cell nodes
            _.each(cellNodes, function (cellNode, mapKey) {

                var // the cache entry for the cell (will be null for deleted/invisible cells)
                    cellData = renderCache.getCellEntry(mapKey);

                // update geometry of existing cell nodes
                if (cellData) {
                    cellNode.css({
                        left: cellData.rectangle.left - layerRectangle.left,
                        top: cellData.rectangle.top - layerRectangle.top,
                        width: cellData.rectangle.width,
                        height: cellData.rectangle.height
                    });
                    if (refreshClip && cellData.clipData) {
                        cellNode.find('>.clip').css({
                            left: -cellData.clipData.clipLeft,
                            right: -cellData.clipData.clipRight
                        }).find('>.content').css({
                            left: -cellData.clipData.hiddenLeft,
                            right: -cellData.clipData.hiddenRight
                        });
                    }
                } else {
                    // remove cell nodes from DOM that do not exist in the cache anymore
                    removeCellNode(mapKey);
                }
            });
        });

        /**
         * Updates the layers according to the passed range and operation.
         */
        var changeLayerRangeHandler = RenderUtils.profileMethod('CellRenderer.changeLayerRangeHandler()', function (event, newLayerRange, newLayerRectangle, refresh) {

            var // the rendering cache
                renderCache = docView.getRenderCache(),
                // the first column that needs to be refreshed completely
                refreshCol = Utils.getIntegerOption(refresh, 'col', docModel.getMaxCol() + 1),
                // the first rows that needs to be refreshed completely
                refreshRow = Utils.getIntegerOption(refresh, 'row', docModel.getMaxRow() + 1),
                // whether to refresh positions, sizes, and clipping settings of all cells
                refreshGeometry = Utils.getBooleanOption(refresh, 'geometry', false),
                // the old layer range, used to check whether the column/row interval has changed
                oldLayerRange = layerRange,
                // the cell ranges whose content cells will be refreshed
                contentRanges = null,
                // the map with new content cells to be rendered into the cell DOM layer
                contentCells = null;

            // store new layer range and rectangle for convenience
            layerRange = newLayerRange;
            layerRectangle = newLayerRectangle;
            RenderUtils.log('old=' + (oldLayerRange ? SheetUtils.getRangeName(oldLayerRange) : 'null') + ' new=' + SheetUtils.getRangeName(newLayerRange) + ' refresh=' + JSON.stringify(refresh));

            // update size of the canvas element
            canvasWrapper.initialize(layerRectangle);

            // update position, size, and text clipping of all existing DOM cell nodes
            updateCellGeometry(refreshGeometry);

            // collect all cell ranges whose content cells need to be rendered
            if (!oldLayerRange || refreshGeometry) {
                contentRanges = [layerRange];
            } else {
                contentRanges = SheetUtils.getRemainingRanges(layerRange, oldLayerRange);
                if (refreshCol <= layerRange.end[0]) {
                    contentRanges.push(_.copy(layerRange, true));
                    _.last(contentRanges).start[0] = refreshCol;
                }
                if (refreshRow <= layerRange.end[1]) {
                    contentRanges.push(_.copy(layerRange, true));
                    _.last(contentRanges).start[1] = refreshRow;
                }
                // restrict ranges to the own layer range, and unify the ranges
                contentRanges = SheetUtils.getUnifiedRanges(SheetUtils.getIntersectionRanges(contentRanges, layerRange));
            }
            RenderUtils.log('contentRanges=' + SheetUtils.getRangesName(contentRanges));

            // pick all cell entries in the passed ranges from the rendering cache
            contentCells = {};
            RenderUtils.takeTime('pick cells from content ranges', function () {
                _.each(contentRanges, function (range) {
                    renderCache.iterateCells(range, function (cellData, mapKey) {
                        contentCells[mapKey] = cellData;
                    });
                });
                RenderUtils.log('contentCells=', contentCells);
            });

            // render the changed cell ranges (refresh entire canvas, its size has changed)
            renderChangedCells(layerRange, contentCells);

            // repaint all table range outlines
            renderTableOutlines();
        });

        /**
         * Resets this renderer. Clears all internal caches and the layer DOM
         * elements.
         */
        function hideLayerRangeHandler() {
            layerRange = layerRectangle = null;
            resetPendingCells();
            canvasWrapper.initialize(EMPTY_RECTANGLE);
            tableLayerNode.empty();
            textLayerNode.empty();
        }

        /**
         * Updates the cell grid lines after specific sheet view attributes
         * have been changed.
         */
        function changeViewAttributesHandler(event, attributes) {

            // update cell grid if visibility or grid color has changed
            if (Utils.hasProperty(attributes, /^(showGrid|gridColor)$/)) {
                renderCanvasLayers(layerRange);
            }

            // update visibility of table range outlines according to grid line visibility
            if ('showGrid' in attributes) {
                tableLayerNode.toggle(attributes.showGrid);
            }
        }

        /**
         * Handles updates of the rendering cache, and renders all changed
         * cell contents in the own rendering layers.
         */
        function renderCacheUpdateHandler(event, renderRanges, changedCells) {
            // render the first 5 cells immediately (best results when changing single cells via GUI quickly)
            renderChangedCells(renderRanges, changedCells, { immediate: 5 });
        }

        /**
         * Handles updates of the rendering cache, after a sheet operation
         * (inserted or deleted columns/rows) has been applied, and updates the
         * DOM cell layer accordingly.
         */
        function renderCacheOperationHandler() {
            // TODO: update existing cell nodes
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM cell nodes matching the specified selector.
         *
         * @param {String} selector
         *  The jQuery selector used to filter the existing cell nodes.
         *
         * @returns {jQuery}
         *  All matching cell nodes, as jQuery collection.
         */
        this.findCellNodes = function (selector) {
            return textLayerNode.find(selector);
        };

        /**
         * Returns the DOM cell node at the specified address.
         *
         * @param {Number[]|String} address
         *  The address of the cell, or the unique cell key of the address.
         *
         * @returns {jQuery}
         *  The DOM node of the cell, as jQuery object (empty, if the cell node
         *  does not exist).
         */
        this.getCellNode = function (address) {
            var mapKey = _.isString(address) ? address : SheetUtils.getCellKey(address);
            return (mapKey in cellNodes) ? cellNodes[mapKey] : $();
        };

        /**
         * Returns all DOM cell nodes in the specified column.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @returns {jQuery}
         *  The DOM nodes of all cells in the specified column, as jQuery
         *  collection.
         */
        this.getCellNodesInCol = function (col) {
            return textLayerNode.find('>[data-col="' + col + '"]');
        };

        /**
         * Returns all DOM cell nodes in the specified row.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {jQuery}
         *  The DOM nodes of all cells in the specified row, as jQuery
         *  collection.
         */
        this.getCellNodesInRow = function (row) {
            return textLayerNode.find('>[data-row="' + row + '"]');
        };

        // initialization -----------------------------------------------------

        // insert all layer nodes into the layer root
        cellLayerNode.append(canvasWrapper.getNode(), tableLayerNode, textLayerNode);

        // initialize class members
        app.onInit(function () {

            // the spreadsheet model and view
            docModel = app.getModel();
            docView = app.getView();

            // the style sheet container of the document
            documentStyles = docModel.getDocumentStyles();
            fontCollection = docModel.getFontCollection();
            numberFormatter = docModel.getNumberFormatter();

            // listen to updates of the layer range covered by the grid pane
            self.on('change:layerrange', changeLayerRangeHandler);
            self.on('hide:layerrange', hideLayerRangeHandler);

            // reset cache, if active sheet changes
            self.listenTo(docView, 'before:activesheet', resetPendingCells);

            // update grid lines
            self.listenToWhenVisible(docView, 'change:sheet:viewattributes', changeViewAttributesHandler);

            // start rendering after any cells in the rendering cache have changed
            self.listenToWhenVisible(docView, 'cache:update', renderCacheUpdateHandler);
            self.listenToWhenVisible(docView, 'cache:operation', renderCacheOperationHandler);

            // render table range outlines
            self.listenToWhenVisible(docView, 'insert:table delete:table change:table', self.createDebouncedMethod($.noop, renderTableOutlines));
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            canvasWrapper.destroy();
            app = cellLayerNode = self = null;
            docModel = docView = documentStyles = fontCollection = numberFormatter = null;
            canvasWrapper = tableLayerNode = textLayerNode = cellNodes = pendingCells = null;
        });

    } // class CellRenderMixin

    // exports ================================================================

    return CellRenderMixin;

});
