/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/dialogsmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/editframework/view/editdialogs',
    'io.ox/office/drawinglayer/view/imageutil',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, KeyCodes, Forms, EditDialogs, ImageUtil, gt) {

    'use strict';

    // mix-in class SpreadsheetDialogsMixin ===================================

    /**
     * Mix-in class for the class SpreadsheetView that adds methods to show
     * modal dialogs.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function SpreadsheetDialogsMixin(app) {

        var // self reference (the spreadsheet view instance)
            self = this;

        // private methods ----------------------------------------------------

        /**
         * If the cell in-place edit mode is active, the current edit text will
         * be validated and committed to the document.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that has been resolved, if the
         *  cell edit mode was not active, or has been left successfully; or
         *  that has been rejected, if the cell edit mode is still active (e.g.
         *  if the current formula expression is invalid).
         */
        function leaveCellEditMode() {
            return self.leaveCellEditMode('auto', { validate: true }) ? $.when() : $.Deferred().reject();
        }

        /**
         * Shows a dialog that allows to enter a sheet name, and will invoke an
         * arbitrary callback function for the sheet name. The dialog will be
         * kept open until a valid sheet name has been entered.
         *
         * @param {Function} callback
         *  The callback function invoked when the OK button of the dialog has
         *  been clicked. Receives the current sheet name shown in the dialog.
         *  Must return a Boolean value that states whether the operation using
         *  the sheet name has been finished successfully. In case of an error,
         *  the dialog will be kept open.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.okLabel]
         *      An alternative label text for the OK button.
         *  @param {String} [options.initialName]
         *      The initial sheet name to be shown when the dialog is opened.
         *      If omitted, the name of the active sheet will be used.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been copied
         *  successfully; or will be rejected, if the dialog has been canceled,
         *  or if the document has switched to read-only mode.
         */
        function showSheetNameDialog(title, callback, options) {

            function showDialog(initialName) {

                var // the dialog box
                    dialog = new EditDialogs.ModalInputDialog({
                        title: title,
                        okLabel: Utils.getStringOption(options, 'okLabel'),
                        value: initialName,
                        placeholder: gt('Enter sheet name')
                    }),
                    // the promise of the dialog show() call
                    promise = null;

                // close dialog automatically after losing edit rights
                self.closeDialogOnReadOnlyMode(dialog);

                // show the dialog
                promise = dialog.show();

                // show dialog as long as the callback function returns false
                return promise.then(function () {
                    var sheetName = dialog.getText();
                    return callback.call(self, sheetName) ? sheetName : showDialog(sheetName);
                });
            }

            // show the dialog, return the piped Deferred
            return leaveCellEditMode().then(function () {
                return showDialog(Utils.getStringOption(options, 'initialName', self.getSheetName()));
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Shows a dialog that allows to enter a name of a new sheet copied
         * from the active sheet. The dialog will be kept open until a valid
         * sheet name has been entered.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been copied
         *  successfully; or will be rejected, if the dialog has been canceled,
         *  or if the document has switched to read-only mode.
         */
        this.showCopySheetDialog = function () {
            // the callback function will be called in the context of this view instance
            return showSheetNameDialog(gt('Copy Sheet'), this.copySheet, { okLabel: gt('Copy'), initialName: this.generateUnusedSheetName() });
        };

        /**
         * Shows a dialog that allows to enter a new name for the active sheet.
         * The dialog will be kept open until a valid sheet name has been
         * entered.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been renamed
         *  successfully; or will rejected, if the dialog has been canceled, or
         *  if the document has switched to read-only mode.
         */
        this.showRenameSheetDialog = function () {
            // the callback function will be called in the context of this view instance
            return showSheetNameDialog(gt('Rename Sheet'), this.setSheetName, { okLabel: gt('Rename') });
        };

        /**
         * Shows a dialog that give the user the posibillity to decide wheather the
         * current selection should expand or not.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the expanding of the selection; or rejected without
         *  expanding. After that, both options executes the sort.
         */
        this.showExpandSelectionForSortingDialog = function () {
            var dialog = new EditDialogs.ModalQueryDialog(gt('Expand the selection?'), { width: 300, title: gt('Sorting') });

            return dialog.show();
        };

        /**
         * Shows the 'Insert Image' dialog and inserts the resulting image into
         * the active sheet of the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the dialog has been closed and the image has been inserted
         *  into the document.
         */
        this.showInsertImageDialog = function () {

            // shows the generic 'Insert Image' dialog
            function showInsertImageDialog() {
                return ImageUtil.showInsertImageDialog(app);
            }

            // inserts the image into the document
            function insertImage(imageDescriptor) {
                return self.insertImage(imageDescriptor).fail(function () {
                    app.rejectEditAttempt('image');
                });
            }

            // leave cell edit mode, show the dialog, handle the dialog result
            return leaveCellEditMode().then(showInsertImageDialog).then(insertImage);
        };

        /**
         * Shows the 'Reorder Sheets' dialog and creates and applies the
         * respective 'moveSheet' operations.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after the
         *  dialog has been closed.
         */
        this.showReorderSheetsDialog = function () {

            // leave cell edit mode, show the dialog
            return leaveCellEditMode().then(function () {

                var docModel = app.getModel(),

                    names = docModel.getSheetNames(),
                    currentSheet = self.getActiveSheet(),
                    currentChosen = null,

                    collectOps = [],

                    dHolder = $('<div>').addClass('dialog-ordersheets-holder'),

                    list = $('<div>', { role: 'tablist' }).addClass('dialog-ordersheets-list'),

                    upButton = $.button({ click: upButtonClickHandler })
                            .addClass('btn')
                            .attr('tabindex', 1)
                            .css({ marginTop: '-3px', marginBottom: '5px', display: 'block' })
                            .append(Forms.createIconMarkup('fa-arrow-up')),

                    downButton = $.button({ data: { dir: 1 }, click: downButtonClickHandler })
                            .addClass('btn')
                            .attr('tabindex', 1)
                            .css('display', 'block')
                            .append(Forms.createIconMarkup('fa-arrow-down')),

                    control = $('<div>').addClass('dialog-ordersheets-control').append(upButton, downButton),

                    dialog = new EditDialogs.ModalDialog({ width: 300, title: gt('Reorder sheets') });

                function upButtonClickHandler() {
                    var index = currentChosen.attr('data-index') | 0;
                    var other = currentChosen.prev('*');
                    if (other.length > 0) {
                        var otherIndex = other.attr('data-index') | 0;
                        currentChosen.insertBefore(other);
                        collectOps.push({
                            from: index,
                            to: otherIndex
                        });
                        currentChosen.attr('data-index', otherIndex);
                        other.attr('data-index', index);
                        checkButtonsActive();
                    }
                    checkScrollOutside();
                }

                function downButtonClickHandler() {
                    var index = currentChosen.attr('data-index') | 0;
                    var other = currentChosen.next('*');
                    if (other.length > 0) {
                        var otherIndex = other.attr('data-index') | 0;
                        currentChosen.insertAfter(other);
                        collectOps.push({
                            from: index,
                            to: otherIndex
                        });
                        currentChosen.attr('data-index', otherIndex);
                        other.attr('data-index', index);
                        checkButtonsActive();
                    }
                    checkScrollOutside();
                }

                function checkScrollOutside() {
                    var pHeight = dHolder.height();
                    var cHeight = currentChosen.height();
                    var pos = currentChosen.position();
                    if (pos.top < 0) {
                        Utils.scrollToChildNode(dHolder, currentChosen);
                    } else if (pos.top + cHeight > pHeight) {
                        Utils.scrollToChildNode(dHolder, currentChosen);
                    }
                }

                function checkButtonsActive() {
                    if (currentChosen.prev('*').length > 0) {
                        upButton.removeClass('disabled');
                        upButton.css({ pointerEvents: 'auto' });
                        upButton.attr('tabindex', 1);
                    } else {
                        upButton.addClass('disabled');
                        upButton.css({ pointerEvents: 'none'});
                        upButton.removeAttr('tabindex');
                    }

                    if (currentChosen.next('*').length > 0) {
                        downButton.removeClass('disabled');
                        downButton.css({ pointerEvents: 'auto' });
                        downButton.attr('tabindex', 1);
                    } else {
                        downButton.addClass('disabled');
                        downButton.css({ pointerEvents: 'none'});
                        downButton.removeAttr('tabindex');
                    }
                }

                function selectButton(evt) {
                    currentChosen.removeClass('btn-primary');
                    currentChosen = $(evt.currentTarget);
                    currentChosen.addClass('btn-primary');
                    checkButtonsActive();
                }

                // close dialog automatically after losing edit rights
                self.closeDialogOnReadOnlyMode(dialog);

                _.each(names, function (name, index) {
                    var sheetModel = docModel.getSheetModel(index);

                    if (!sheetModel.getMergedAttributes().sheet.visible) {
                        return;
                    }

                    // check the sheet type
                    if (!self.isSheetTypeSupported(sheetModel.getType())) { return; }

                    name = _.noI18n(Utils.escapeHTML(name));
                    var button = $('<button class="btn" data-index="' + index + '" title="' + name + '" tabindex="1" role="button">' + name + '</button>');

                    button.on('click', selectButton);
                    if (index === currentSheet) {
                        button.addClass('btn-primary');
                        currentChosen = button;
                    }
                    list.append(button);
                });

                var cnt = dialog.getBody();
                cnt.css({
                    paddingRight: '2px',
                    paddingTop: 0,
                    paddingBottom: 0
                });

                function handleUpDown(upDown, ctrlKey) {
                    var button;
                    if (ctrlKey) {
                        button = upDown ? upButton : downButton;
                    } else {
                        button = upDown ? currentChosen.prev('*') : currentChosen.next('*');
                    }
                    button.click();
                    currentChosen.focus();
                }

                /**
                 * Handles 'keydown' events on the dialog node
                 * up/down changes the selected sheet in the dialog
                 * ctrl + up/down 'moves' the selected sheet inside the dialog
                 * enter click on the ok button
                 */
                function keyDownHandler(evt) {
                    switch (evt.keyCode) {
                    case KeyCodes.UP_ARROW:
                        handleUpDown(true, evt.ctrlKey);
                        return true;
                    case KeyCodes.DOWN_ARROW:
                        handleUpDown(false, evt.ctrlKey);
                        return true;
                    }
                }

                cnt.parent().on('keydown', keyDownHandler);

                dHolder.append(list);

                dialog.append(dHolder);
                dialog.append(control);

                checkButtonsActive();

                var promise = dialog.show();

                checkScrollOutside();

                return promise.done(function () {
                    docModel.getUndoManager().enterUndoGroup(function () {
                        _.each(collectOps, function (op) {
                            docModel.moveSheet(op.from, op.to);
                        });
                    });
                });
            });
        };

    } // class SpreadsheetDialogsMixin

    // exports ================================================================

    return SpreadsheetDialogsMixin;

});
