/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/formatcodepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, RadioList, Labels, gt) {

    'use strict';

    // class FormatCodePicker =================================================

    /**
     * A drop-down list control for selection of predefined number formats
     * depending to currently selected category in the FormatCategoryPicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioList.
     */
    function FormatCodePicker(app, initOptions) {

        var // self reference
            self = this,

            // the model and view of the document
            docModel = null,
            docView = null,

            // the category of the format codes currently shown in the list
            listCategory = null;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: Labels.MORE_LABEL,
            tooltip: /*#. special number format codes for different categories of number formats */ gt.pgettext('format-code', 'Format codes')
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the label of the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {

            var // the current number format category
                currentCategory = docView.getNumberFormatCategory(),
                // all codes for the current number format
                categoryCodes = docModel.getNumberFormatter().getCategoryCodes(currentCategory),
                // the active format code
                formatCodeEntry = _.findWhere(categoryCodes, { value: value }),
                //  menu instance of the radio list
                menu = self.getMenu();

            // refill the list, if the category has changed
            if (listCategory !== currentCategory) {
                self.clearMenu();
                _.each(categoryCodes, function (entry) {
                    self.createOptionButton(entry.value, {
                        label: _.noI18n(entry.label),
                        labelStyle: entry.red ? 'color:#f00000;' : null
                    });
                });
                listCategory = currentCategory;
            }

            // handle format codes which are not supported by injecting a 'Custom' button on the fly.
            if (!formatCodeEntry) {
                var customNode = menu.findItemNodes(Labels.CUSTOM_FORMAT_VALUE);
                // if custom button already exist, just select it and quit
                if (customNode.length !== 0) {
                    menu.selectItemNode(customNode);
                    return;
                }
                // create custom button and select it
                self.createOptionButton(Labels.CUSTOM_FORMAT_VALUE, { label: /*#. number format category in spreadsheets: all user-defined number formats */ gt.pgettext('format-code', 'Custom') });
                menu.selectMatchingItemNodes(Labels.CUSTOM_FORMAT_VALUE);
            } else {
                // remove old, invalid custom node
                menu.deleteItemNodes(Labels.CUSTOM_FORMAT_VALUE);
            }
        }

        // initialization -----------------------------------------------------

        // initialize class members when application is ready
        app.onInit(function () {
            docModel = app.getModel();
            docView = app.getView();
        });

        // register custom update handler to update button caption
        this.registerUpdateHandler(updateHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = docModel = docView = null;
        });

    } // class FormatCodePicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: FormatCodePicker });

});
