/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/chartcreator',
    ['io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/operations',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/formula/tokenarray',
     'io.ox/office/drawinglayer/view/chartstyleutil',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Operations, SheetUtils, TokenArray, ChartStyleUtil, gt) {

    'use strict';

    var // default width of new chart objects
        CHART_DEFAULT_WIDTH = Utils.convertHmmToLength(15000, 'px', 1),

        // default height of new chart objects
        CHART_DEFAULT_HEIGHT = Utils.convertHmmToLength(10000, 'px', 1);

    var DEFAULT_TEXTCOLOR = {type: 'auto'};
    var DEFAULT_LINECOLOR = {transformations: [{type: 'lumMod', value: 15000}, {type: 'lumOff',value: 85000}],type: 'scheme', value: 'text1'};
    var STANDARD_SHAPE = {type: 'solid', color: DEFAULT_LINECOLOR, width: 36};
    var NONE_SHAPE = {type: 'none'};
    var STANDARD_CHAR = {color: DEFAULT_TEXTCOLOR};
    var HEAD_CHAR = {color: DEFAULT_TEXTCOLOR, fontSize: 16};

    var INSERT_DEFAULTS = (function () {

        var AXIS_ENABLED =  { axis: {label: true},  line: NONE_SHAPE, character: STANDARD_CHAR};
        var AXIS_DISABLED = { axis: {label: false}, line: NONE_SHAPE};
        var GRID_ENABLED =  { line: STANDARD_SHAPE};
        var GRID_DISABLED = { line: NONE_SHAPE};

        return {
            normal: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED,
            },
            xy: {
                xAxis: AXIS_ENABLED,
                xGrid: GRID_ENABLED,
                yAxis: AXIS_ENABLED,
                yGrid: GRID_ENABLED,
            },
            pie: {
                xAxis: AXIS_DISABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_DISABLED,
                yGrid: GRID_DISABLED,
            }
        };
    }());

    function getFormula(app, sheet, from, to) {
        if (!from) {
            throw 'no from assigned';
        }
        var range = { start: from, end: to || from };
        var tokenArray = new TokenArray(app, null, { trigger: 'never' });
        tokenArray.appendRange(range, { sheet: sheet, abs: true });
        var formula = tokenArray.getFormula();
        tokenArray.destroy();

        return formula;
    }

    function addDataSeries(app, generator, sourceSheet, position, series, keyFrom, keyTo, title, valueFrom, valueTo, bubbleFrom, bubbleTo, bubbleArray) {
        var insert = {
            values: getFormula(app, sourceSheet, valueFrom, valueTo)
        };
        if (title) {
            insert.title = getFormula(app, sourceSheet, title);
        }
        if (keyFrom) {
            insert.names = getFormula(app, sourceSheet, keyFrom, keyTo);
        }
        if (bubbleFrom) {
            insert.bubbles = getFormula(app, sourceSheet, bubbleFrom, bubbleTo);
        } else if (bubbleArray) {
            insert.bubbles = bubbleArray;
        }
        var properties = {
            series: series,
            attrs: {
                series: insert,
                line: {type: 'solid'},
                fill: {type: 'solid'},
            }
        };
        generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties);
    }

    function generateSeriesOperations(app, sourceRange, sourceSheet, generator, position, cells, axis, chart) {

        if (_.isUndefined(chart.varyColors)) {
            chart.varyColors = true;
        }

        generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: { chart: chart } });

        var start = sourceRange.start;
        var end = sourceRange.end;

        var useAxis;
        if (axis !== 0 && axis !== 1) {
            if (cells[0] && !cells[1]) {
                useAxis = 0;
            } else if (!cells[0] && cells[1]) {
                useAxis = 1;
            } else {
                var width = end[0] - start[0];
                var height = end[1] - start[1];
                if (width >= height) {
                    useAxis = 1;
                } else {
                    useAxis = 0;
                }
            }
        } else {
            useAxis = axis;
        }

        var seriesCount;
        if (chart.type && chart.type.indexOf('bubble') === 0) {
            seriesCount = generateBubbleSeriesOperations(app, sourceRange, sourceSheet, generator, position, cells, useAxis);
        } else {
            seriesCount = generateStandardSeriesOperations(app, sourceRange, sourceSheet, generator, position, cells, useAxis);
        }
        if ((seriesCount === 1 && !chart.varyColors) || (chart.type === 'scatter2d' || chart.type === 'line2d')) {
            var send = {};
            if (seriesCount === 1 && !chart.varyColors) {
                send.varyColors = true;
            }
            //CURVED is a workaround, filter has not all the information it needs!
            send.curved = chart.curved;
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: { chart: send } });
        }
        return seriesCount;
    }

    /**
     * makes an address array, needed for the code, which does not know if its row or column oriented
     */
    function makeAddress(activeAxis, otherAxis, activeValue, otherValue) {
        var res = [];
        res[activeAxis] = activeValue;
        res[otherAxis] = otherValue;
        return res;
    }

    /**
     * generate all series from start to end.
     * Depending if first row/column contains a String,
     * it is defined as Names (datapoint) and Titles (dataseries)
     *
     *  @param isNumber {Array} first is a Boolean for the cell under the first cell
     *                         second is a Boolean for the cell right of the first cell
     *  @param activeAxis [1 or 0] column or row for the direction of any series data
     */
    function generateStandardSeriesOperations(app, sourceRange, sourceSheet, generator, position, cells, activeAxis) {

        var start = sourceRange.start;
        var end = sourceRange.end;

        var max = end[activeAxis] - start[activeAxis];
        var otherAxis = (activeAxis - 1) * -1;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        for (var i = 0; i < max; i++) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            addDataSeries(app, generator, sourceSheet, position, i, keyFrom, keyTo, title, valueFrom, valueTo);
        }
        return max;
    }

    /**
     * generate all bubble-series from start to end.
     * Every second row/columns are values, the others are bubbles (sizes).
     * Depending if first row/column contains a String,
     * it is defined as Titles (dataseries).
     * If the count of the series is odd,
     * the first row/column is defined as Names (datapoints)
     *
     *  @param isNumber {Array} first is a Boolean for the cell under the first cell
     *                         second is a Boolean for the cell right of the first cell
     *  @param activeAxis [1 or 0] column or row for the direction of any series data
     */
    function generateBubbleSeriesOperations(app, sourceRange, sourceSheet, generator, position, cells, activeAxis) {

        var start = sourceRange.start;
        var end = sourceRange.end;

        var max = end[activeAxis] - start[activeAxis];
        if (max === 0) {
            //exception only for 1D - Bubbles, behaves like excel
            activeAxis = (activeAxis - 1) * -1;
            max = end[activeAxis] - start[activeAxis];
        }
        var otherAxis = (activeAxis - 1) * -1;

        var keyFrom, keyTo;

        if (!cells[activeAxis]) {
            keyFrom = null;
            keyTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                keyFrom = start;
            } else {
                keyFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            keyTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        var size = 0;

        for (var i = 0; i < max; i += 2) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            var bubbleFrom = null;
            var bubbleTo = null;
            var bubbleArray = null;

            if (i===max-1) {
                bubbleArray = [];
                var arraySize = 1 + (valueTo[otherAxis] - valueFrom[otherAxis]);
                for (var j = 0; j < arraySize; j++) {
                    bubbleArray.push(1);
                }
            } else {
                bubbleFrom = makeAddress(activeAxis, otherAxis, valueFrom[activeAxis] + 1, valueFrom[otherAxis]);
                bubbleTo = makeAddress(activeAxis, otherAxis, valueTo[activeAxis] + 1, valueTo[otherAxis]);
            }

            addDataSeries(app, generator, sourceSheet, position, size, keyFrom, keyTo, title, valueFrom, valueTo, bubbleFrom, bubbleTo, bubbleArray);
            size++;
        }
        return size;
    }

    function getContentForRange(app, sourceRange, sourceSheet, def, follow, forceColLabel, forceRowLabel) {

        var sheetModel = app.getModel().getSheetModel(sourceSheet);

        var start = sourceRange.start;
        var cols = SheetUtils.getColCount(sourceRange);
        var rows = SheetUtils.getRowCount(sourceRange);

        if (cols > 50 || rows > 50) {
            app.getView().yell({ type: 'warning', message: gt('It is not possible to create a chart out of more than 50 input cells.') });
            if (def) {
                def.reject();
            }
        } else {
            var bottEntry = sheetModel.getRowCollection().getNextVisibleEntry(start[1] + 1);
            var rightEntry = sheetModel.getColCollection().getNextVisibleEntry(start[0] + 1);

            var bottAddress = [start[0], bottEntry ? bottEntry.index : start[1]];
            var rightAddress = [rightEntry ? rightEntry.index : start[0], start[1]];

            var rangesArray = [[
                    { sheet: sourceSheet, start: bottAddress, end: bottAddress },
                    { sheet: sourceSheet, start: rightAddress, end: rightAddress },
                    { sheet: sourceSheet, start: start, end: start }
                ]];

            app.getModel().queryCellContents(rangesArray, { attributes: true }).done(function (res) {

                var bottom = false;
                var right = false;
                var axis = null;

                // first array element of the result is an array with the contents of the three cells
                var bottomCell = res[0][0];
                var rightCell = res[0][1];
                var startCell = res[0][2];

                function isDate(cellContents) {
                    return (/^(date|time|datetime)$/).test(cellContents.format.cat);
                }

                function cellIsLabel(cell) {
                    if (!startCell) { return false; }
                    if (!startCell.display.length) { return true; }
                    return isDate(cell) || (cell ? !_.isNumber(cell.result) : false);
                }

                if (cols === 1 || rows === 1) {
                    var firstString = startCell ? !_.isNumber(startCell.result) : false;
                    if (cols === 1) {
                        axis = 0;
                        if (firstString) {
                            right = true;
                        }
                    } else {
                        axis = 1;
                        if (firstString) {
                            bottom = true;
                        }
                    }
                } else {
                    bottom = cellIsLabel(bottomCell);
                    right = cellIsLabel(rightCell);
                }

                if (!_.isUndefined(forceColLabel)) {
                    bottom = forceColLabel;
                }
                if (!_.isUndefined(forceRowLabel)) {
                    right = forceRowLabel;
                }
                follow([bottom, right], axis);
            });
        }
    }

    // class ChartCreator =====================================================

    /**
     * ChartCreator makes an insert-Chart-Op and Insert-Chart-DataSeries chosen by current Selection
     * and changes the Selection to a Multiselection by title, names and values
     */
    function ChartCreator(app, chart) {
        var docView = app.getView();
        var gridPane = docView.getActiveGridPane();

        var def = $.Deferred();

        var drawingCollection = docView.getDrawingCollection();
        var selection = docView.getSelection();

        var range = selection.ranges[selection.activeRange];

        getContentForRange(app, range, docView.getActiveSheet(), def, function (cells, axis) {

            var rect = gridPane.getVisibleRectangle();

            var clip = {
                left: Math.max(0, Math.round(rect.left + (rect.width - CHART_DEFAULT_WIDTH) / 2)),
                top: Math.max(0, Math.round(rect.top + (rect.height - CHART_DEFAULT_HEIGHT) / 2)),
                width: CHART_DEFAULT_WIDTH,
                height: CHART_DEFAULT_HEIGHT
            };

            var attrs = drawingCollection.getAttributesForRectangle(clip);

            docView.insertDrawing('chart', attrs, function (generator, sheet, position) {

                var seriesCount = generateSeriesOperations(app, range, sheet, generator, position, cells, axis, chart);

                var legendPos;
                if (seriesCount === 1) {
                    //TODO: title
                    legendPos = 'off';
                    generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: 'main', attrs: { character: HEAD_CHAR } });
                } else {
                    legendPos = 'bottom';
                }
                generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: { legend: { pos: legendPos}, character: STANDARD_CHAR } });

                var defaults;
                switch (chart.type) {
                case 'bubble2d':
                case 'scatter2d':
                    defaults = INSERT_DEFAULTS.xy;
                    break;
                case 'pie2d':
                case 'donut2d':
                    defaults = INSERT_DEFAULTS.pie;
                    break;
                default:
                    defaults = INSERT_DEFAULTS.normal;
                    break;
                }

                if (!chart.chartStyleId) {
                    chart.chartStyleId = 2;
                    generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: { chart: { chartStyleId: 2 }, fill: { color: { type: 'scheme', value: 'light1', fallbackValue: 'FFFFFF' } } } });
                }

                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: 'x', attrs: defaults.xAxis });
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: 'y', attrs: defaults.yAxis });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: 'x', attrs: defaults.xGrid });
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: 'y', attrs: defaults.yGrid });
            });
            def.resolve();
        });


        /**
         * the deferred Object handles noting at the Moment,
         * its just for Future cases, when the situation could be changed
         * @return {Jquery|Deferred}
         */
        this.deferred = function () {
            return def;
        };
    }


    /**
     *
     * @param app
     * @param range whole start to end area of the new series
     * @param sheet
     * @param chartModel is used for the position and id in the document and as fallback for chart-type-checks
     * @param axis series direction
     * @param chartAttrs is used for changing other attributes of the chart and for the type info (bubble or not)
     * @param forceColLabel is set, it overwrites the internal logic for detecting first column as label
     * @param forceRowLabel is set, it overwrites the internal logic for detecting first column as label (headline)
     */
    ChartCreator.generateOperations = function (app, sourceRange, sourceSheet, chartModel, axis, chartAttrs, forceColLabel, forceRowLabel) {

        var chart;
        if (chartAttrs) {
            chart = chartAttrs;
        } else {
            var attrs = chartModel.getMergedAttributes();
            var type = attrs.chart.type;
            //CURVED is a workaround, filter has not all the information it needs!
            chart = {
                type: type,
                curved: attrs.chart.curved
            };
        }

        getContentForRange(app, sourceRange, sourceSheet, null, function (cells, overwrittenAxis) {

            //only overwritten if source range height is 1 oder width is 1
            if (overwrittenAxis === 0 || overwrittenAxis === 1) {
                axis = overwrittenAxis;
            }
            var docModel = app.getModel();
            var generator = chartModel.getSheetModel().createOperationsGenerator();
            var position = chartModel.getSheetPosition();

            _.times(chartModel.getSeriesCount(), function () {
                generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: 0 });
            });

            generateSeriesOperations(app, sourceRange, sourceSheet, generator, position, cells, axis, chart);
            docModel.applyOperations(generator.getOperations());
        }, forceColLabel, forceRowLabel);

    };

    ChartCreator.getStandardShape = function() {
        return STANDARD_SHAPE;
    };

    ChartCreator.getStandardChar = function() {
        return STANDARD_CHAR;
    };

    ChartCreator.getHeadChar = function() {
        return HEAD_CHAR;
    };

    ChartCreator.getNoneShape = function() {
        return NONE_SHAPE;
    };
    // exports ================================================================

    return ChartCreator;

});
