/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/validationcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, ModelObject, AttributedModel, SheetUtils, TokenArray) {

    'use strict';

    // class ValidationModel ==================================================

    /**
     * Stores data validation settings for a single range in a specific sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this data validation range.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this data validation range.
     *
     * @param {Array} ranges
     *  The addresses of all cell ranges that contain the data validation
     *  settings.
     *
     * @param {Object} [attrs]
     *  The data validation attribute set.
     */
    var ValidationModel = AttributedModel.extend({ constructor: function (app, sheetModel, ranges, attrs) {

        var // self reference
            self = this,

            // the spreadsheet document model
            docModel = app.getModel(),

            // token array representing the validation attribute 'value1'
            tokenArray1 = new TokenArray(app, sheetModel),

            // token array representing the validation attribute 'value2'
            tokenArray2 = new TokenArray(app, sheetModel);

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, attrs, { families: 'validation' });

        // private methods ----------------------------------------------------

        /**
         * Updates internal settings after the validation attributes of this
         * collection entry have been changed.
         */
        function changeAttributesHandler(event, newAttributes, oldAttributes) {

            // extract the validation attributes from the passed attribute sets
            newAttributes = (newAttributes || self.getMergedAttributes()).validation;
            oldAttributes = oldAttributes ? oldAttributes.validation : {};

            // parse a changed formula expression in attribute 'value1'
            if (newAttributes.value1 !== oldAttributes.value1) {
                tokenArray1.invokeSilently('parseFormula', newAttributes.value1);
            }

            // parse a changed formula expression in attribute 'value2'
            if (newAttributes.value2 !== oldAttributes.value2) {
                tokenArray2.invokeSilently('parseFormula', newAttributes.value2);
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this validation model for
         * the specified sheet.
         *
         * @internal
         *  Used by the class ValidationCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {ValidationModel}
         *  A clone of this validation model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new model instance and relocate the sheet references
            var valModel = new ValidationModel(app, targetModel, ranges, this.getExplicitAttributes(true));
            valModel.relocateSheet(sheetModel.getIndex(), targetModel.getIndex());
            return valModel;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the addresses of all cell ranges of this collection entry.
         *
         * @returns {Array}
         *  The addresses of all cell ranges.
         */
        this.getRanges = function () {
            return _.copy(ranges, true);
        };

        /**
         * Returns whether the passed cell address is contained by the cell
         * ranges of this collection entry.
         *
         * @param {Number[]} address
         *  The address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is contained in this collection entry.
         */
        this.containsCell = function (address) {
            return SheetUtils.rangesContainCell(ranges, address);
        };

        /**
         * Returns the reference addresses of this collection entry. The
         * reference address is the top-left cell of the bounding range of all
         * ranges of this collection entry.
         *
         * @returns {Number[]}
         *  The address of the reference cell of this collection entry.
         */
        this.getRefAddress = function () {
            return SheetUtils.getBoundingRange(ranges).start;
        };

        /**
         * Returns the list contents for validations of type 'list' or 'source'
         * according to the formula expression contained in the attribute
         * 'value1'.
         *
         * @param {Number[]} targetAddress
         *  The target address used to resolve relative references in the
         *  formula expression.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with an
         *  array of strings, if this collection entry is a data validation of
         *  type 'list' or 'source', and the formula expression contained in
         *  the attribute 'value1' could be resolved successfully to a list of
         *  strings. Otherwise, the promise will be rejected. Additionally, the
         *  promise will contain the method 'abort()' that is able to abort a
         *  running server request. In this case, the promise will be rejected
         *  immediately.
         */
        this.queryListValues = function (targetAddress) {

            var // the original query for 'source' validation
                sourceQuery = null,
                // the resulting promise
                resultPromise = null;

            // resolve formula to constant strings (validation type 'list')
            function resolveList() {
                var listValues = tokenArray1.resolveStringList();
                return _.isArray(listValues) ? $.when(listValues) : $.Deferred().reject();
            }

            // resolve formula to source ranges (validation type 'source')
            function resolveSource() {
                var refAddress = self.getRefAddress(),
                    sourceRanges = tokenArray1.resolveRangeList({ refAddress: refAddress, targetAddress: targetAddress });
                if (!_.isArray(sourceRanges)) { return $.Deferred().reject(); }
                sourceQuery = docModel.queryCellContents([sourceRanges], { hidden: true, maxCount: 1000 });
                return sourceQuery.then(function (resultContents) {
                    return _.pluck(resultContents[0], 'display');
                });
            }

            // resolve source data by validation type
            switch (this.getMergedAttributes().validation.type) {
            case 'list':
                resultPromise = resolveList();
                break;
            case 'source':
                resultPromise = resolveSource();
                break;
            default:
                // wrong validation type
                resultPromise = $.Deferred().reject();
            }

            // add an 'abort()' method to the promise that aborts the server query
            return _.extend(resultPromise, {
                abort: function () {
                    if (sourceQuery) { sourceQuery.abort(); }
                    return this;
                }
            });
        };

        /**
         * Changes the addresses of all cell ranges of this collection entry.
         *
         * @param {Array} newRanges
         *  The addresses of the new cell ranges. The old ranges of this
         *  collection entry will be replaced completely.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.setRanges = function (newRanges) {
            ranges = _.copy(newRanges, true);
            return this;
        };

        /**
         * Reduces the cell ranges containing the data validation settings to
         * the difference of these ranges and the passed ranges.
         *
         * @returns {Boolean}
         *  Whether the reduced ranges are still valid (not all deleted). If
         *  false is returned, this collection entry must be deleted from the
         *  validation collection.
         */
        this.reduceRanges = function (otherRanges) {
            ranges = SheetUtils.getRemainingRanges(ranges, otherRanges);
            return ranges.length > 0;
        };

        /**
         * Transforms the addresses of the cell ranges containing the data
         * validation settings, after columns or rows have been inserted into
         * or removed from the sheet.
         *
         * @returns {Boolean}
         *  Whether the transformed ranges are still valid (not all deleted).
         *  If false is returned, this collection entry must be deleted from
         *  the validation collection.
         */
        this.transformRanges = function (interval, insert, columns) {
            ranges = docModel.transformRanges(ranges, interval, insert, columns, 'expandEnd');
            return ranges.length > 0;
        };

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         */
        this.transformValues = function (sheet, interval, insert, columns) {
            tokenArray1.transformFormula(sheet, interval, insert, columns);
            tokenArray2.transformFormula(sheet, interval, insert, columns);
            return true;
        };

        /**
         * Changes all formula expressions stored in the validation attributes
         * 'value1' and 'value2' with sheet references containing the specified
         * old sheet index to the new sheet index.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            tokenArray1.relocateSheet(oldSheet, newSheet);
            tokenArray2.relocateSheet(oldSheet, newSheet);
            return this;
        };

        // initialization -----------------------------------------------------

        // unify the range addresses
        ranges = SheetUtils.getUnifiedRanges(ranges);

        // parse the source formulas after changing value attributes
        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // update source values after one of the token array has changed (but prevent updating the token array again)
        tokenArray1.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ validation: { value1: tokenArray1.getFormula() } });
            self.on('change:attributes', changeAttributesHandler);
        });
        tokenArray2.on('triggered', function () {
            self.off('change:attributes', changeAttributesHandler);
            self.setAttributes({ validation: { value2: tokenArray2.getFormula() } });
            self.on('change:attributes', changeAttributesHandler);
        });

        // adjust formulas according to sheet indexes passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[ValidationModel.length];
            if (_.isObject(cloneData)) {
                tokenArray1.relocateSheet(cloneData.oldSheet, cloneData.newSheet);
                tokenArray2.relocateSheet(cloneData.oldSheet, cloneData.newSheet);
            }
        }(arguments));

        // destroy all class members
        this.registerDestructor(function () {
            tokenArray1.destroy();
            tokenArray2.destroy();
            app = sheetModel = null;
            self = docModel = tokenArray1 = tokenArray2 = null;
        });

    }}); // class ValidationModel

    // class ValidationCollection =============================================

    /**
     * Stores data validation settings in a specific sheet. Data validation can
     * be used to restrict the values allowed to be entered in cell ranges, and
     * to show tooltips and error messages.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function ValidationCollection(app, sheetModel) {

        var // the spreadsheet document model
            docModel = app.getModel(),

            // all collection entries with validation settings
            entries = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Returns the collection entry that covers the specified cell.
         *
         * @param {Number[]} address
         *  The address of a cell.
         *
         * @returns {ValidationModel|Null}
         *  The collection entry for the specified cell. If the cell is not
         *  validated at all, returns null.
         */
        function getEntry(address) {
            var entry = _.find(entries, function (entry2) {
                return entry2.containsCell(address);
            });
            return entry ? entry : null;
        }

        /**
         * Invokes the specified method on all collection entries.
         *
         * @param {String} methodName
         *  The name of the method to be invoked on each collection entry. If
         *  the method returns the value false, the respective entry will be
         *  removed from the collection.
         *
         * @param {ValidationModel|Null} excludeEntry
         *  A collection entry that will be skipped (its method will not be
         *  invoked).
         *
         * @param {Any} ...
         *  Other parameters that will be passed to the specified method.
         */
        function invokeEntryMethod(methodName, excludeEntry) {
            var args = _.toArray(arguments).slice(invokeEntryMethod.length);
            Utils.iterateArray(entries, function (entry, index) {
                if ((entry !== excludeEntry) && (entry[methodName].apply(entry, args) === false)) {
                    entry.destroy();
                    entries.splice(index, 1);
                }
            }, { reverse: true }); // reverse to be able to delete in-place
        }

        /**
         * Recalculates the position of all validation ranges, after columns or
         * rows have been inserted into or deleted from the sheet.
         */
        function transformRanges(interval, insert, columns) {
            invokeEntryMethod('transformRanges', null, interval, insert, columns);
        }

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         */
        function transformValues(sheet, interval, insert, columns) {
            invokeEntryMethod('transformValues', null, sheet, interval, insert, columns);
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this collection for the
         * specified sheet.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {ValidationCollection}
         *  A clone of this collection, initialized for ownership by the passed
         *  sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new collection, pass all own entries as hidden parameter
            return new ValidationCollection(app, targetModel, entries);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the validation settings for the specified cell.
         *
         * @param {Number[]} address
         *  The address of a cell.
         *
         * @returns {Object|Null}
         *  The validation settings of the specified cell, in the following
         *  properties:
         *  - {Array} ranges
         *      The addresses of all related cell ranges containing the same
         *      validation settings.
         *  - {Object} attributes
         *      All validation attributes.
         *  If the cell is not validated at all, returns null.
         */
        this.getValidationSettings = function (address) {
            var entry = getEntry(address);
            return entry ? {
                ranges: entry.getRanges(),
                attributes: entry.getMergedAttributes().validation
            } : null;
        };

        /**
         * Returns the list contents for a data validation of type 'list' or
         * 'source' at the specified cell address.
         *
         * @param {Number[]} address
         *  The address of a cell.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with an
         *  array of strings, if this collection entry is a data validation of
         *  type 'list' or 'source', and the formula expression contained in
         *  the attribute 'value1' could be resolved successfully to a list of
         *  strings. Otherwise, the promise will be rejected. Additionally, the
         *  promise will contain the method 'abort()' that is able to abort a
         *  running server request. In this case, the promise will be rejected
         *  immediately.
         */
        this.queryListValues = function (address) {
            var entry = getEntry(address);
            return entry ? entry.queryListValues(address) : $.Deferred().reject();
        };

        /**
         * Restricts the valid values of a range of cells according to the
         * specified settings.
         *
         * @param {Array} ranges
         *  The addresses of all cell ranges that contain the data validation
         *  settings.
         *
         * @param {Object} attrs
         *  The validation attributes.
         *
         * @param {Number} [index]
         *  The zero-based insertion index for the new collection entry. If
         *  omitted, the entry will be appended to the collection.
         *
         * @returns {Boolean}
         *  Whether the validation settings have been created successfully.
         */
        this.insertValidation = function (ranges, attrs, index) {

            var // the new collection entry
                entry = new ValidationModel(app, sheetModel, ranges, { validation: attrs });

            // reduce ranges of existing entries (this may result in removing entries completely)
            invokeEntryMethod('reduceRanges', null, entry.getRanges());

            // insert the new collection entry
            index = _.isNumber(index) ? Utils.minMax(index, 0, entries.length) : entries.length;
            entries.splice(index, 0, entry);

            // TODO: trigger change events
            return true;
        };

        /**
         * Changes the ranges and/or validation attributes of an existing
         * collection entry.
         *
         * @param {Number} index
         *  The zero-based index of the collection entry to be changed.
         *
         * @param {Array|Null} ranges
         *  The new addresses of the cell ranges for the collection entry. If
         *  set to null, the original ranges will not be modified.
         *
         * @param {Object|Null} attrs
         *  The new validation attributes. If set to null, the current
         *  attributes will not be modified.
         *
         * @returns {Boolean}
         *  Whether the validation settings have been changed successfully.
         */
        this.changeValidation = function (index, ranges, attrs) {

            var // the collection entry to be modified
                entry = entries[index];

            // validate passed collection index
            if (!_.isObject(entry)) { return false; }

            // change validation ranges
            if (_.isArray(ranges) && (ranges.length > 0)) {

                // reduce ranges of other existing entries (this may result in removing entries completely)
                invokeEntryMethod('reduceRanges', entry, ranges);

                // set the new ranges at the collection entry
                entry.setRanges(ranges);
            }

            // change all other validation attributes
            if (_.isObject(attrs)) {
                entry.setAttributes({ validation: attrs });
            }

            return true;
        };

        /**
         * Deletes validation settings from this collection.
         *
         * @param {Number} index
         *  The zero-based index of the collection entry to be deleted.
         *
         * @returns {Boolean}
         *  Whether the validation settings have been deleted successfully.
         */
        this.deleteValidation = function (index) {

            var // the collection entry to be deleted
                entry = entries[index];

            // validate passed collection index
            if (!_.isObject(entry)) { return false; }

            // destroy and remove the collection entry
            entry.destroy();
            entries.splice(index, 1);

            return true;
        };

        // initialization -----------------------------------------------------

        // update ranges after inserting/deleting columns or rows in the own sheet
        sheetModel.registerTransformationHandler(transformRanges);

        // update source formulas after inserting/deleting columns or rows in any sheet
        docModel.registerTransformationHandler(transformValues);

        // clone collection entries passed as hidden argument by the clone() method
        if (_.isArray(arguments[ValidationCollection.length])) {
            entries = _.invoke(arguments[ValidationCollection.length], 'clone', sheetModel);
        }

        // destroy all class members
        this.registerDestructor(function () {
            docModel.unregisterTransformationHandler(transformValues);
            _.invoke(entries, 'destroy');
            app = sheetModel = docModel = entries = null;
        });

    } // class ValidationCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ValidationCollection });

});
