/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/reference', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenutils'
], function (Utils, SheetUtils, TokenUtils) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // class Reference ========================================================

    /**
     * Representation of a reference literal: an unresolved list of cell range
     * addresses including their sheet indexes.
     *
     * @constructor
     *
     * @param {Object|Array} [ranges]
     *  The cell range, or an array of cell ranges to be inserted into this
     *  instance. Each cell range must contain the following properties:
     *  - {Number} sheet1
     *      The zero-based index of the first sheet containing the range.
     *  - {Number} sheet2
     *      The zero-based index of the last sheet containing the range.
     *  - {Number[]} start
     *      The address of the top-left cell in the range.
     *  - {Number[]} end
     *      The address of the bottom-right cell in the range.
     */
    function Reference(ranges) {

        // private methods ----------------------------------------------------

        /**
         * Throws a CIRCULAR_ERROR error code, if the passed range contains the
         * specified reference cell.
         */
        function checkCircularReference(range, refSheet, refAddress) {
            if ((range.sheet1 <= refSheet) && (refSheet <= range.sheet2) && SheetUtils.rangeContainsCell(range, refAddress)) {
                throw TokenUtils.CIRCULAR_ERROR;
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the number of ranges contained in this reference.
         *
         * @returns {Number}
         *  The number of ranges contained in this reference.
         */
        this.getLength = function () {
            return ranges.length;
        };

        /**
         * Returns the number of all cells contained in all ranges of this
         * reference.
         *
         * @returns {Number}
         *  The number of all cells contained in all ranges of this reference.
         */
        this.getCellCount = function () {
            return Utils.getSum(ranges, function (range) {
                return (range.sheet2 - range.sheet1 + 1) * SheetUtils.getCellCount(range);
            });
        };

        /**
         * Returns all ranges contained in this reference.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.sameSheet=false]
         *      If set to true, all ranges must refer to the same single sheet.
         *
         * @returns {Array}
         *  All ranges contained in this reference, if available.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The #VALUE! error code, if the option 'sameSheet' is set, and the
         *      ranges in this reference refer to different sheets, or if at
         *      least one range refers to multiple sheets.
         */
        this.getRanges = function (options) {

            // reference must not be empty
            if (ranges.length === 0) { throw ErrorCodes.NULL; }

            // check that all ranges refer to the same single sheet
            if (Utils.getBooleanOption(options, 'sameSheet', false)) {

                // check that first range refers to a single sheet
                if (ranges[0].sheet1 !== ranges[0].sheet2) { throw ErrorCodes.VALUE; }

                // check sheet indexes of the following ranges
                for (var index = 1, length = ranges.length, sheet = ranges[0].sheet1; index < length; index += 1) {
                    if ((ranges[index].sheet1 !== sheet) || (ranges[index].sheet2 !== sheet)) {
                        throw ErrorCodes.VALUE;
                    }
                }
            }

            return _.clone(ranges);
        };

        /**
         * Returns the cell range of this reference, but only if it contains
         * exactly one range pointing to a single sheet (but optionally also to
         * a range of sheets).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.valueError=false]
         *      If set to true, a #VALUE! error code (instead of a #REF! error
         *      code) will be thrown, if the reference contains more than one
         *      range.
         *  @param {Boolean} [options.multiSheet=false]
         *      If set to true, the range may refer to multiple sheets.
         *
         * @returns {Object}
         *  The range object, if available, with the properties 'start', 'end',
         *  'sheet1', and 'sheet2'.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The specified error code, if the reference contains more than one
         *      range.
         *  - The #VALUE! error code, if the range in this reference refers to
         *      multiple sheets, and the option 'multiSheet'. has not been set.
         */
        this.getSingleRange = function (options) {

            // must not be empty
            if (ranges.length === 0) {
                throw ErrorCodes.NULL;
            }

            // must be a single range in a single sheet
            if (ranges.length > 1) {
                throw Utils.getBooleanOption(options, 'valueError', false) ? ErrorCodes.VALUE : ErrorCodes.REF;
            }

            // must point to a single sheet, unless multiple sheets are allowed (always throw the #VALUE! error code)
            if (!Utils.getBooleanOption(options, 'multiSheet', false) && (ranges[0].sheet1 !== ranges[0].sheet2)) {
                throw ErrorCodes.VALUE;
            }

            return ranges[0];
        };

        /**
         * Extracts a single cell range from this reference, intended to be
         * resolved to the cell contents in the range. The reference must
         * contain exactly one range address pointing to a single sheet, and
         * the specified reference cell must not be part of that range.
         *
         * @param {Number} refSheet
         *  The zero-based index of the reference sheet.
         *
         * @param {Number[]} refAddress
         *  The address of the reference cell.
         *
         * @returns {Object}
         *  The range object, if available, with the properties 'start', 'end',
         *  'sheet1', and 'sheet2'. The sheet indexes will be equal.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The #VALUE! error code, if the reference contains more than one
         *      range, or if the range refers to multiple sheets.
         *  - The CIRCULAR_ERROR error code, if a range is available, but the
         *      reference cell is part of that range (circular reference).
         */
        this.getValueRange = function (refSheet, refAddress) {

            var // resolve to a single range (throw #VALUE! on multiple ranges)
                range = this.getSingleRange({ valueError: true });

            // detect circular references
            checkCircularReference(range, refSheet, refAddress);
            return range;
        };

        /**
         * Invokes the passed iterator for all cell ranges in this reference,
         * intended to be resolved to the cell contents in the ranges.
         *
         * @param {Number} refSheet
         *  The zero-based index of the reference sheet.
         *
         * @param {Number[]} refAddress
         *  The address of the reference cell.
         *
         * @param {Function} iterator
         *  The iterator callback function invoked for every range. Receives
         *  the following parameters:
         *  (1) {Object} range
         *      The current range, with the properties 'start', 'end',
         *      'sheet1', and 'sheet2'.
         *  (2) {Number} index
         *      The zero-based array index of the range in this reference.
         *  May return Utils.BREAK to stop iteration immediately.
         *
         * @param {Object} [context]
         *  The calling context for the iterator function.
         *
         * @returns {Utils.BREAK|Undefined}
         *  The result of the iterator.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The CIRCULAR_ERROR error code, if the reference cell is part of
         *      any of the ranges in this reference (circular reference).
         */
        this.iterateValueRanges = function (refSheet, refAddress, iterator, context) {

            // must not be empty
            if (ranges.length === 0) { throw ErrorCodes.NULL; }

            // call passed iterator for all ranges
            return Utils.iterateArray(ranges, function (range, index) {
                checkCircularReference(range, refSheet, refAddress);
                return iterator.call(context, range, index);
            });
        };

        /**
         * Returns the string representation of this reference for debug
         * logging.
         */
        this.toString = function () {
            return _.map(ranges, function (range) {
                return range.sheet1 + ':' + range.sheet2 + '!' + SheetUtils.getRangeName(range);
            }).join(',');
        };

        // initialization -----------------------------------------------------

        // convert ranges to an array
        ranges = _.isArray(ranges) ? _.clone(ranges) : _.isObject(ranges) ? [ranges] : [];

    } // class Reference

    // exports ================================================================

    return Reference;

});
