/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/besselfuncs', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes,

        // maximum number of iterations to prevent endless loops etc.
        MAX_ITERATIONS = 50000,

        // minimum difference between iteration steps
        EPSILON = 1e-15,

        // mathematical constants
        PI = Math.PI,
        PI_2 = PI / 2,
        PI_4 = PI / 4,
        EULER_GAMMA = 0.5772156649015328606,

        // shortcuts to built-in Math functions
        floor = Math.floor,
        abs = Math.abs,
        pow = Math.pow,
        sqrt = Math.sqrt,
        exp = Math.exp,
        log = Math.log,
        sin = Math.sin,
        cos = Math.cos;

    // private global functions ===============================================

    function besselI(value, order) {

        if (order < 0) { throw ErrorCodes.NUM; }

        var result = 0,
            maxIteration = 2000,
            xHalf = value / 2,
            k = 0,
            term = 1;

        for (k = 1; k <= order; ++k) {
           term = term / k * xHalf;
        }
        result = term;
        if (term !== 0) {
            k = 1;
            do {
                term = term * xHalf / k * xHalf / (k + order);
                result += term;
                k += 1;
            } while ((abs(term) > abs(result) * EPSILON) && (k < maxIteration));
        }
        return result;
    }

    function besselJ(value, order) {

        if (order < 0) { throw ErrorCodes.NUM; }

        var sign,
            estimateIteration,
            asymptoticPossible,
            epsilon = 1e-15,
            hasfound = false,
            k= 0,
            u,
            mBar, gBar, gBarDeltaU,
            g = 0,
            deltaU = 0,
            bar = -1,
            twoDivPI = 2 / PI;

        function fmod(v1, v2) {
            return v1 - floor(v1 / v2) * v2;
        }

        if (value === 0) {
            return (order === 0) ? 1 : 0;
        }
        sign = (value % 2 === 1 && value < 0) ? -1 : 1;
        value = abs(value);

        estimateIteration = value * 1.5 + order;
        asymptoticPossible = pow(value, 0.4) > order;
        if (estimateIteration > MAX_ITERATIONS) {
            if (asymptoticPossible) {
                return sign * sqrt(twoDivPI / value) * cos(value - order * PI_2 - PI_4);
            }
            throw ErrorCodes.NUM;
        }

        if (value === 0) {
            u = 1;
            gBarDeltaU = 0;
            gBar = -2 / value;
            deltaU = gBarDeltaU / gBar;
            u = u + deltaU;
            g = -1 / gBar;
            bar = bar * g;
            k = 2;
        } else  {
            u=0;
            for (k =1; k<= order - 1; k = k + 1) {
                mBar = 2 * fmod(k - 1, 2) * bar;
                gBarDeltaU = - g * deltaU - mBar * u; // alpha_k = 0
                gBar = mBar - 2 * k / value + g;
                deltaU = gBarDeltaU / gBar;
                u = u + deltaU;
                g = -1 / gBar;
                bar = bar * g;
            }
            // Step alpha_N = 1
            mBar = 2 * fmod(k - 1, 2) * bar;
            gBarDeltaU = bar - g * deltaU - mBar * u; // alpha_k = 1
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u = u + deltaU;
            g = -1 / gBar;
            bar = bar * g;
            k = k + 1;
        }
        // Loop until desired accuracy, always alpha_k = 0
        do {
            mBar = 2 * fmod(k - 1, 2) * bar;
            gBarDeltaU = - g * deltaU - mBar * u;
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u = u + deltaU;
            g = -1 / gBar;
            bar = bar * g;
            hasfound = abs(deltaU) <= abs(u) * epsilon;
            k = k + 1;
        } while (!hasfound && k <= MAX_ITERATIONS);

        if (hasfound) {
            return u * sign;
        }

        throw ErrorCodes.VALUE;
    }

    function besselK0(value) {

        var result, num2, y;

         if (value <= 2) {
             num2 = value * 0.5;
             y = num2 * num2;

             result = -log(num2) * besselI(value, 0) +
                    (-EULER_GAMMA + y * (0.4227842 + y * (0.23069756 + y * (0.0348859 +
                        y * (0.00262698 + y * (0.0001075 + y * 0.74e-5))))));
         } else {
             y = 2 / value;

             result = exp(-value) / sqrt(value) * (1.25331414 + y * (-0.07832358 +
                    y * (0.02189568 + y * (-0.01062446 + y * (0.00587872 +
                    y * (-0.00251540 + y * 0.00053208))))));
         }
        return result;
    }

    function besselK1(value) {

        var result, num2, y;

        if (value <= 2) {
            num2 = value / 2;
            y = num2 * num2;

            result = log(num2) * besselI(value, 1) +
                    (1 + y * (0.15443144 + y * (-0.67278579 + y * (-0.18156897 + y * (-0.1919402e-1 +
                        y * (-0.00110404 + y * -0.4686e-4)))))) / value;
        } else {
            y = 2 / value;

            result = exp(-value) / sqrt(value) * (1.25331414 + y * (0.23498619 +
                    y * (-0.0365562 + y * (0.01504268 + y * (-0.00780353 +
                    y * (0.00325614 + y * -0.00068245))))));
        }

        return result;
    }

    function besselK(value, order) {

        if (order < 0) { throw ErrorCodes.NUM; }

        var result = 0,
            tox, bkm, bkp, bk;

        switch (order) {
        case 0:
            result = besselK0(value);
            break;
        case 1:
            result = besselK1(value);
            break;
        default:
            tox = 2 / value;
            bkm = besselK0(value);
            bk = besselK1(value);

            for (var n = 1; n < order; n++) {
                bkp = bkm + n * tox * bk;
                bkm = bk;
                bk = bkp;
            }

            result = bk;
        }
        return result;
    }

    function besselY0(value) {

        var alpha = log(value / 2) + EULER_GAMMA,
            u,
            k = 1,
            mBar = 0,
            gBarDeltaU = 0,
            gBar = -2 / value,
            deltaU = gBarDeltaU / gBar,
            g = -1 / gBar,
            fBar = -1 * g,
            signAlpha = 1,
            km1mod2,
            hasFound = false;

        if (value <= 0) {
            throw ErrorCodes.VALUE;
        }
        if (value > 5e+6) {
            return sqrt(1 / PI / value) * (sin(value) - cos(value));
        }
        u = alpha;
        k += 1;
        do {
            km1mod2 = (k - 1) % 2;
            mBar = 2 * km1mod2 * fBar;
            if  (km1mod2 === 0) {
                alpha = 0;
            } else {
               alpha = signAlpha * (4 / k);
               signAlpha = -signAlpha;
            }
            gBarDeltaU = fBar * alpha - g * deltaU - mBar * u;
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u += deltaU;
            g = -1 / gBar;
            fBar *= g;
            hasFound = abs(deltaU) <= abs(u) * EPSILON;
            k += 1;
        }
        while (!hasFound && k < MAX_ITERATIONS);
        if (hasFound) {
            return u / PI_2;
        }
        throw ErrorCodes.VALUE;
    }

    function besselY1(value) {

        var alpha = 1 / value,
            fBar = -1,
            g = 0,
            u = alpha,
            k = 1,
            mBar = 0,
            gBarDeltaU,
            gBar = -2 / value,
            deltaU,
            signAlpha = -1,
            km1mod2,
            q,
            hasFound = false;

        if (value <= 0) {
            throw ErrorCodes.VALUE;
        }
        if (value > 5e6) {
            return - sqrt(1 / PI / value) * (sin(value) + cos(value));
        }
        alpha = 1 - EULER_GAMMA - log(value / 2);
        gBarDeltaU = -alpha;
        deltaU = gBarDeltaU / gBar;
        u += deltaU;
        g = -1 / gBar;
        fBar = fBar * g;

        k += 1;
        do {
            km1mod2 = (k - 1) % 2;
            mBar = 2 * km1mod2 * fBar;
            q = (k - 1) / 2;
            if (km1mod2 === 0) {
               alpha = signAlpha * (1 / q + 1 / (q + 1));
               signAlpha = -signAlpha;
            } else {
                alpha = 0;
            }
            gBarDeltaU = fBar * alpha - g * deltaU - mBar * u;
            gBar = mBar - 2 * k / value + g;
            deltaU = gBarDeltaU / gBar;
            u += deltaU;
            g = -1 / gBar;
            fBar *= g;
            hasFound = abs(deltaU) <= abs(u) * EPSILON;
            k += 1;
        }
        while (!hasFound && k < MAX_ITERATIONS);
        if (hasFound) {
            return -u / PI_2;
        }
        throw ErrorCodes.VALUE;
    }

    function besselY(value, order) {

        if (order < 0) { throw ErrorCodes.NUM; }

        var result,
            byp,
            tox = 2 / value,
            bym,
            by;

        switch (order) {
        case 0:
            result = besselY0(value);
            break;
        case 1:
            result = besselY1(value);
            break;
        default:
            bym = besselY0(value);
            by = besselY1(value);

            for (var n = 1; n < order; n += 1) {
                byp = n * tox * by - bym;
                bym = by;
                by = byp;
            }
            result = by;
        }

        return result;
    }

    // exports ================================================================

    return { I: besselI, J: besselJ, K: besselK, Y: besselY };

});
