/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellattributesmodel',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, TriggerObject, TimerMixin, Border, Config, SheetUtils, CellAttributesModel, TokenArray) {

    'use strict';

    var // maximum number of cells contained in a server response
        MAX_CELLS_IN_RESPONSE = Utils.minMax(Math.floor(10 * Utils.PERFORMANCE_LEVEL), 100, 1000),

        // whether to calculate results of all changed formula cells locally
        LOCAL_FORMULAS = Config.DEBUG && Config.getUrlFlag('spreadsheet:local-formulas');

    // static private functions ===============================================

    /**
     * Convert error code text to error code objects, and removes leading
     * apostrophes from strings.
     */
    function convertToCellValue(value) {
        return !_.isString(value) ? value : (value[0] === '#') ? SheetUtils.makeErrorCode(value) : value.replace(/^'/, '');
    }

    // class CellCollection ===================================================

    /**
     * Collects cell contents and formatting attributes for a single sheet.
     * To save memory and to improve performance, instances of this class store
     * specific parts of the sheet only. More cell data will be fetched from
     * the server on demand.
     *
     * Triggers the following events:
     * - 'change:cells'
     *      After the contents or formatting of some cells in this collection
     *      have been changed. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Array} ranges
     *          The cell range addresses of all changed cells.
     *      (3) {Object} [options]
     *          Additional options passed to the event listeners. The following
     *          options are supported:
     *          - {Boolean} [options.merge]
     *              If set to true, the event originates from merging or
     *              unmerging the notified cell ranges.
     * - 'change:usedarea'
     *      After the size of the used area in the sheet has been changed.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} usedCols
     *          The number of used columns in the sheet.
     *      (3) {Number} usedRows
     *          The number of used rows in the sheet.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this collection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function CellCollection(app, sheetModel) {

        var // self reference
            self = this,

            // the document model, and the cell style sheet container
            docModel = app.getModel(),
            styleCollection = docModel.getStyleCollection('cell'),
            numberFormatter = docModel.getNumberFormatter(),

            // the collections of the active sheet
            colCollection = sheetModel.getColCollection(),
            rowCollection = sheetModel.getRowCollection(),
            mergeCollection = sheetModel.getMergeCollection(),

            // token array used to calculate formula results of any cell entries locally
            tokenArray = new TokenArray(app, sheetModel, { trigger: 'never', grammar: 'ui' }),

            // all cell entries mapped by column/row index
            cellMap = {},

            // the number of cell models in this collection (counted separately for performance)
            cellCount = 0,

            // the current bounding ranges (all ranges covered by this collection)
            boundRanges = [],

            // number of used columns and rows in the sheet
            usedCols = 0,
            usedRows = 0;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private class EntryModel -------------------------------------------

        /**
         * Represents the collection entries of a CellCollection instance.
         *
         * @constructor
         *
         * @extends CellAttributesModel
         */
        var EntryModel = CellAttributesModel.extend({ constructor: function (data, address) {

            // base constructor - - - - - - - - - - - - - - - - - - - - - - - -

            CellAttributesModel.call(this, app, Utils.getObjectOption(data, 'attrs'));

            // initialization - - - - - - - - - - - - - - - - - - - - - - - - -

            // add public properties
            this.display = Utils.getStringOption(data, 'display', '');
            this.result = convertToCellValue(Utils.getOption(data, 'result', null));
            this.formula = Utils.getStringOption(data, 'formula', null);
            this.format = _.extend({}, CellCollection.DEFAULT_CELL_DATA.format, Utils.getObjectOption(data, 'format'));

            // update current settings, e.g. display string according to formula or number format
            this.updateValue(address);

        }}); // class EntryModel

        // methods (prototype, collection entries are mass objects) - - - - - -

        /**
         * Returns a descriptor object for the cell entry, suitable to be
         * passed to iterator functions.
         *
         * @returns {Object}
         *  A descriptor object for this collection entry, with the properties
         *  'address', 'display', 'result', 'formula', 'format', 'attributes',
         *  'explicit', and 'exists'.
         */
        EntryModel.prototype.getData = function (address) {
            return {
                address: address,
                display: this.display,
                result: this.result,
                formula: this.formula,
                format: _.clone(this.format),
                attributes: this.getMergedAttributes(),
                explicit: this.getExplicitAttributes(),
                exists: true
            };
        };

        /**
         * Updates the properties of this cell entry, according to the current
         * formula and number format.
         *
         * @param {Number[]} [address]
         *  Current address of the cell entry, used for local calculation of
         *  formula results.
         *
         * @param {Object} [parsedValue]
         *  Optional content properties ('display', 'result', or 'formula') to
         *  be changed.
         */
        EntryModel.prototype.updateValue = function (address, parsedValue) {

            // add passed content properties
            _.extend(this, parsedValue);

            // update formula result locally if specified via debug URL option
            if (LOCAL_FORMULAS && _.isString(this.formula) && address && (this.formula[0] === '=')) {
                tokenArray.parseFormula(this.formula.substr(1));
                var result = tokenArray.interpretFormula('val', sheetModel.getIndex(), address);
                switch (result.type) {
                case 'result':
                    this.result = result.value;
                    this.display = getDisplayText(this.result, '');
                    break;
                case 'warn':
                case 'error':
                    switch (result.value) {
                    case 'circular':
                        this.result = SheetUtils.ErrorCodes.VALUE;
                        this.display = app.convertErrorCodeToString(SheetUtils.ErrorCodes.VALUE);
                        break;
                    case 'unsupported':
                    case 'internal':
                    case 'missing':
                    case 'unexpected':
                        this.result = SheetUtils.ErrorCodes.NA;
                        this.display = app.convertErrorCodeToString(SheetUtils.ErrorCodes.NA);
                    }
                }
            }

            // create appropriate display text for 'General' number format (TODO: for other dynamic formats too, or even all formats?)
            if (_.isNumber(this.result) && _.isFinite(this.result) && (this.format.cat === 'standard')) {
                this.display = numberFormatter.formatStandardNumber(this.result, SheetUtils.MAX_LENGTH_STANDARD_CELL);
            }
        };

        var // default model for entries missing in the collection
            defaultEntry = new EntryModel();

        // override the getData() method of the default entry, change the 'exists' property of the result to false
        defaultEntry.getData = function (address) {
            var cellData = EntryModel.prototype.getData.call(this, address);
            cellData.exists = false;
            return cellData;
        };

        // private methods ----------------------------------------------------

        var logUsedArea = SheetUtils.isLoggingActive() ? this.createDebouncedMethod($.noop, function () {
            SheetUtils.log('CellCollection: usedRange=' + SheetUtils.getRangeName(self.getUsedRange()));
        }) : $.noop;

        var logBoundRanges = SheetUtils.isLoggingActive() ? this.createDebouncedMethod($.noop, function () {
            SheetUtils.log('CellCollection: boundRanges=' + SheetUtils.getRangesName(boundRanges));
        }) : $.noop;

        /**
         * Joins the passed cell addresses to an array of range addresses, and
         * triggers a 'change:cells' event, if at least one cell address has
         * been passed to this method.
         *
         * @param {Array|Object} [ranges]
         *  The address of a changed cell range, or an array of multiple range
         *  addresses. Can be set to null to pass cell addresses only via the
         *  parameter 'cells' (see below).
         *
         * @param {Array} [cells]
         *  An array with cell addresses of changed single cells. Can be set to
         *  null or omitted to pass cell ranges only via the parameter 'ranges'
         *  (see above).
         *
         * @param {Object} [options]
         *  Additional options passed to the event listeners. See the
         *  description of the 'change:cells' event in the documentation of the
         *  class constructor for details.
         */
        function triggerChangeCellsEvent(ranges, cells, options) {

            var // the changed cell ranges
                changedRanges = _.isObject(ranges) ? _.getArray(ranges) : [],
                // the changed cells, joined to ranges
                changedCells = _.isArray(cells) ? SheetUtils.joinCellsToRanges(cells) : [],
                // the resulting unified ranges
                allRanges = (changedRanges.length > 0) ? SheetUtils.getUnifiedRanges(changedRanges.concat(changedCells)) : changedCells,
                // the bounding range of all changed cells
                boundRange = SheetUtils.getBoundingRange(allRanges),
                // ratio of bounding range covered by the changed range
                coveredRatio = boundRange ? (SheetUtils.getCellCountInRanges(allRanges) / SheetUtils.getCellCount(boundRange)) : 0;

            // do not trigger, if the passed parameters were missing/empty
            if (allRanges.length > 0) {
                // notify the bounding range, if it is covered by at least 80%
                self.trigger('change:cells', (coveredRatio >= 0.8) ? [boundRange] : allRanges, options);
            }
        }

        /**
         * Sets the number of used columns and rows in the sheet, and triggers
         * a 'change:usedarea' event, if at least one of the counts has been
         * changed.
         *
         * @param {Number} newUsedCols
         *  The number of columns used in the sheet.
         *
         * @param {Number} newUsedRows
         *  The number of rows used in the sheet.
         */
        function setUsedArea(newUsedCols, newUsedRows) {

            // if one count is zero, the other count will be zero too
            if ((newUsedCols === 0) || (newUsedRows === 0)) {
                newUsedCols = newUsedRows = 0;
            }

            // update members and trigger event
            if ((usedCols !== newUsedCols) || (usedRows !== newUsedRows)) {
                usedCols = newUsedCols;
                usedRows = newUsedRows;
                self.trigger('change:usedarea', usedCols, usedRows);
                logUsedArea();
            }
        }

        /**
         * Extends the number of used columns and rows in the sheet, so that
         * the specified cell is included in the used area, and triggers a
         * 'change:usedarea' event, if at least one of the counts has been
         * changed.
         *
         * @param {Number[]} address
         *  The address of the cell to be includes in the used area.
         */
        function extendUsedArea(address) {
            setUsedArea(Math.max(address[0] + 1, usedCols), Math.max(address[1] + 1, usedRows));
        }

        /**
         * Extends the bounding ranges of this collection with the specified
         * cell ranges.
         *
         * @param {Object|Array} ranges
         *  A single cell range address, or an array of cell range addresses to
         *  be included into the bounding ranges of this collection.
         */
        function extendBoundRanges(ranges) {
            boundRanges = SheetUtils.getUnifiedRanges(boundRanges.concat(_.getArray(ranges)));
            logBoundRanges();
        }

        /**
         * Converts the passed result value to an intermediate display string.
         *
         * @param {Any} result
         *  The typed result value.
         *
         * @param {String} defText
         *  Default text for missing/unsupported result values.
         *
         * @returns {String}
         *  A best-matching display string for the result value.
         */
        function getDisplayText(result, defText) {
            return _.isString(result) ? result :
                _.isNumber(result) ? numberFormatter.formatStandardNumber(result, SheetUtils.MAX_LENGTH_STANDARD_CELL) :
                _.isBoolean(result) ? app.getBooleanLiteral(result) :
                SheetUtils.isErrorCode(result) ? app.convertErrorCodeToString(result) :
                defText;
        }

        /**
         * Tries to detect the type of the passed cell text with a few simple
         * tests, and returns a partial cell entry object containing the typed
         * value and formula properties.
         *
         * @param {Object} contents
         *  A cell content descriptor, with optional 'value' and 'result'
         *  properties.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the passed cell value must be a string and will
         *      be parsed to determine the data type.
         *
         * @returns {Object|Null}
         *  A cell entry object containing the properties 'display' with the
         *  passed cell text, 'result' with the converted value (null, number,
         *  boolean, error code, or string), and 'formula' if the passed value
         *  may be a formula. If no value was passed, returns null.
         */
        function parseCellValue(contents, parse) {

            var // the value passed in the contents object
                value = contents.value,
                // the formula result passed in the contents object
                result = Utils.getOption(contents, 'result'),
                // the resulting data object
                cellData = null;

            // parse value if passed
            if (_.isString(value)) {
                cellData = { display: value, result: null, formula: null };
                if (value.length === 0) {
                    cellData.result = null;
                } else if (!_.isUndefined(result)) {
                    // formula with precalculated result
                    cellData.result = convertToCellValue(result);
                    cellData.display = getDisplayText(cellData.result, CellCollection.PENDING_DISPLAY);
                    cellData.formula = value;
                } else if (/^=./.test(value)) {
                    // formula without precalculated result
                    cellData.result = null;
                    cellData.display = CellCollection.PENDING_DISPLAY;
                    cellData.formula = value;
                } else if (parse) {
                    cellData.result = numberFormatter.parseString(value);
                    if (_.isBoolean(cellData.result) || SheetUtils.isErrorCode(cellData.result)) {
                        cellData.display = cellData.display.toUpperCase();
                    }
                } else {
                    cellData.result = value;
                }
            } else if (!_.isUndefined(value)) {
                cellData = { display: getDisplayText(value, ''), result: value, formula: null };
            }

            return cellData;
        }

        /**
         * Returns the cell entry with the specified cell address from the
         * collection.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @returns {EntryModel|Null}
         *  The cell entry; or null, if the entry does not exist.
         */
        function getCellEntry(address) {
            var cellEntry = cellMap[SheetUtils.getCellKey(address)];
            return cellEntry ? cellEntry : null;
        }

        /**
         * Inserts a new entry into this collection. An existing entry will be
         * deleted.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @param {EntryModel} cellEntry
         *  The new cell entry to be inserted into this collection.
         */
        function putCellEntry(address, cellEntry) {
            var key = SheetUtils.getCellKey(address);
            if (key in cellMap) { cellMap[key].destroy(); } else { cellCount += 1; }
            cellMap[key] = cellEntry;
        }

        /**
         * Removes an entry from this collection and destroys it.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @return {Boolean}
         *  Whether an existing cell entry at the position has been deleted.
         */
        function deleteCellEntry(address) {
            var key = SheetUtils.getCellKey(address);
            if (key in cellMap) {
                cellMap[key].destroy();
                delete cellMap[key];
                cellCount -= 1;
                return true;
            }
            return false;
        }

        /**
         * Moves an entry in this collection to a new position. An existing
         * entry at the target position will be deleted.
         *
         * @param {Number[]} fromAddress
         *  The address of the cell entry to be moved.
         *
         * @param {Number[]} toAddress
         *  The address of the target position.
         *
         * @return {Boolean}
         *  Whether the passed addresses are different, and an existing cell
         *  entry has been moved to the new location, or an existing cell entry
         *  at the old position has been deleted.
         */
        function moveCellEntry(fromAddress, toAddress) {

            var fromKey = SheetUtils.getCellKey(fromAddress),
                toKey = SheetUtils.getCellKey(toAddress),
                moved = false;

            if (fromKey !== toKey) {
                if (toKey in cellMap) {
                    cellMap[toKey].destroy();
                    delete cellMap[toKey];
                    cellCount -= 1;
                    moved = true;
                }
                if (fromKey in cellMap) {
                    cellMap[toKey] = cellMap[fromKey];
                    delete cellMap[fromKey];
                    moved = true;
                }
            }

            return moved;
        }

        /**
         * Creates a new entry for this collection. An existing entry will be
         * deleted.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @param {Object} [data]
         *  A data object from the server response of a view update request. If
         *  missing, creates a default entry representing an empty unformatted
         *  cell. May contain the properties 'display', 'result', 'formula',
         *  'format', and 'attrs'.
         *
         * @returns {EntryModel}
         *  A new collection entry. Always contains the properties 'display',
         *  'result', 'formula', and 'format'.
         */
        function createCellEntry(address, data) {
            var cellEntry = new EntryModel(data, address);
            putCellEntry(address, cellEntry);
            return cellEntry;
        }

        /**
         * Returns the cell entry with the specified cell address from the
         * collection for read/write access. If the cell entry is missing, it
         * will be created, and will be initialized with the default formatting
         * of the column or row.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @returns {EntryModel}
         *  The collection entry corresponding to the passed cell address.
         */
        function getOrCreateCellEntry(address) {

            var // the existing cell entry
                cellEntry = getCellEntry(address),
                // the row descriptor with default formatting for empty cells
                rowEntry = null,
                // the data used to initialize a new cell entry
                data = null;

            // return existing cell entry
            if (cellEntry) { return cellEntry; }

            // no entry found: create a new entry with either default row or column formatting
            rowEntry = rowCollection.getEntry(address[1]);
            if (rowEntry.attributes.row.customFormat) {
                data = { attrs: rowEntry.explicit };
            } else {
                data = { attrs: colCollection.getEntry(address[0]).explicit };
            }
            return createCellEntry(address, data);
        }

        /**
         * Visits all cell positions in the passed range, regardless of their
         * contents.
         *
         * @param {Object} range
         *  The address of a single cell range.
         *
         * @param {Function} iterator
         *  The callback function that will be invoked for each cell in the
         *  passed range. Receives the following parameters:
         *  (1) {Object} colEntry
         *      The column collection entry of the cell.
         *  (2) {Object} rowEntry
         *      The row collection entry of the cell.
         *  (3) {EntryModel|Null} cellEntry
         *      The collection entry of the cell, if existing; otherwise null.
         *  (4) {Number[]} address
         *      The address of the visited cell.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {String} [options.hidden='none']
         *      Specifies how to handle cells contained in hidden rows or
         *      columns. Must be one of the following values:
         *      - 'none' (or omitted): Only visible cells will be visited.
         *      - 'all': All visible and hidden cells will be visited.
         *      - 'last': All visible cells, and all cells contained in hidden
         *          columns and/or rows that precede a visible column/row will
         *          be visited.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the cells in the range will be visited in
         *      reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateCellsInRange(range, iterator, options) {

            var // the column and row interval covered by the passed range
                colInterval = SheetUtils.getColInterval(range),
                rowInterval = SheetUtils.getRowInterval(range),
                // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to visit hidden cell entries
                hiddenMode = Utils.getStringOption(options, 'hidden', 'none'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false);

            return rowCollection.iterateEntries(rowInterval, function (rowEntry) {
                return colCollection.iterateEntries(colInterval, function (colEntry) {
                    var address = [colEntry.index, rowEntry.index];
                    return iterator.call(context, colEntry, rowEntry, getCellEntry(address), address);
                }, { hidden: hiddenMode, reverse: reverse });
            }, { hidden: hiddenMode, reverse: reverse });
        }

        /**
         * Visits all collection entries that exist in the passed ranges, in no
         * specific order.
         *
         * @param {Object|Array}
         *  The address of a single range, or an array of range addresses.
         *
         * @param {Function} iterator
         *  The callback function that will be invoked for each existing
         *  collection entry in the passed cell ranges. Receives the following
         *  parameters:
         *  (1) {EntryModel} cellEntry
         *      The current collection entry.
         *  (2) {String} key
         *      The unique key of the collection entry in the cell map.
         *  (3) {Number[]} address
         *      The cell address of the collection entry.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateExistingEntries(ranges, iterator) {

            var // convert to disjunct ranges
                uniqueRanges = SheetUtils.getUnifiedRanges(ranges),
                // result of the _.any() loops
                result = false;

            // performance: decide whether to iterate over cell addresses in the ranges, or over the cell map
            if (cellCount < SheetUtils.getCellCountInRanges(uniqueRanges)) {

                // more cells in the ranges than collection entries: process all collection entries
                result = _.any(cellMap, function (cellEntry, key) {
                    var address = SheetUtils.parseCellKey(key);
                    return SheetUtils.rangesContainCell(uniqueRanges, address) && (iterator.call(self, cellEntry, key, address) === Utils.BREAK);
                });

            } else {

                // more collection entries than cells in the ranges: iterate all cells in the ranges
                result = _.any(uniqueRanges, function (range) {
                    for (var row = range.start[1]; row <= range.end[1]; row += 1) {
                        for (var col = range.start[0]; col <= range.end[0]; col += 1) {
                            var address = [col, row], key = SheetUtils.getCellKey(address);
                            if ((key in cellMap) && (iterator.call(self, cellMap[key], key, address) === Utils.BREAK)) { return true; }
                        }
                    }
                });
            }

            // if result is true, the iterator callback function has returned Utils.BREAK
            return result ? Utils.BREAK : undefined;
        }

        /**
         * Deletes all cell entries that exist in the passed ranges.
         *
         * @param {Object|Array}
         *  The address of a single range, or an array of range addresses.
         *
         * @returns {Array}
         *  An array with the addresses of all existing cells that have been
         *  deleted.
         */
        function deleteEntriesInRanges(ranges) {

            var // the addresses of all deleted cells
                deletedCells = [];

            // delete all existing collection entries
            iterateExistingEntries(ranges, function (cellEntry, key, address) {
                cellEntry.destroy();
                delete cellMap[key];
                cellCount -= 1;
                deletedCells.push(address);
            });

            return deletedCells;
        }

        /**
         * Moves all cell entries that exist in the passed range, by the
         * specified amount of columns and rows.
         *
         * @returns {Array}
         *  An array with the addresses of the old and new positions of all
         *  existing cells that have been moved, and the existing cells in the
         *  target range that have been deleted.
         */
        function moveEntriesInRange(range, moveCols, moveRows) {

            var // the target range to move the cells to
                targetRange = docModel.getCroppedMovedRange(range, moveCols, moveRows),
                // a temporary map used to collect moved entries (prevents overwriting existing entries)
                tempMap = {},
                // the addresses of all moved and deleted cells
                changedCells = null;

            // delete the parts of the target range not covered by the original range
            changedCells = deleteEntriesInRanges(SheetUtils.getRemainingRanges(targetRange, range));

            // move all cell entries into the temporary map
            iterateExistingEntries(range, function (cellEntry, key, address) {

                // store the old cell address
                changedCells.push(address);

                // move the cell to the temporary map
                address[0] += moveCols;
                address[1] += moveRows;
                tempMap[SheetUtils.getCellKey(address)] = cellEntry;
                changedCells.push(address);
                delete cellMap[key];
            });

            // insert all cell entries back into the map
            _.each(tempMap, function (cellEntry, key) {
                cellMap[key] = cellEntry;
            });

            return changedCells;
        }

        /**
         * Inserts new cells into the passed cell ranges, and moves the old
         * cells down or to the right.
         *
         * @param {Object} range
         *  The address of the cell range to be inserted.
         *
         * @param {String} direction
         *  Either 'columns' to move the existing cells horizontally (to the
         *  right or left according to the sheet orientation), or 'rows' to
         *  move the existing cells down.
         *
         * @returns {Array}
         *  An array with the cell addresses of all changed cells (the
         *  addresses of inserted cells with formatting copied from the
         *  preceding column/row, and the old and new addresses of existing
         *  moved cells).
         */
        function insertCells(range, columns) {

            var // the insertion interval
                interval = SheetUtils.getInterval(range, columns),
                // the address array index used to move the cells
                addrIndex = columns ? 0 : 1,
                // the last available column/row index in the sheet
                maxIndex = docModel.getMaxIndex(columns),
                // number of columns the entries will be moved
                moveCols = columns ? SheetUtils.getColCount(range) : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : SheetUtils.getRowCount(range),
                // the cell range with all cells to be moved
                moveRange = _.copy(range, true),
                // the entire modified range (inserted and moved)
                modifiedRange = _.copy(range, true),
                // the bounding ranges inside the moved range
                movedBoundRanges = null,
                // the addresses of all changed cells
                changedCells = [];

            // calculate moved and modified range
            moveRange.end[addrIndex] = maxIndex - (columns ? moveCols : moveRows);
            modifiedRange.end[addrIndex] = maxIndex;
            if (moveRange.start[addrIndex] > moveRange.end[addrIndex]) { moveRange = null; }

            // calculate the moved parts of the bounding ranges
            if (moveRange) {
                movedBoundRanges = SheetUtils.getIntersectionRanges(boundRanges, moveRange);
                _.each(movedBoundRanges, function (boundRange, index) {
                    movedBoundRanges[index] = docModel.getCroppedMovedRange(boundRange, moveCols, moveRows);
                });
            }

            // cut the entire modified area (inserted and moved) from the bounding ranges, and
            // add the target position of the existing moved parts of the old bounding ranges
            boundRanges = SheetUtils.getRemainingRanges(boundRanges, modifiedRange);
            logBoundRanges();
            if (movedBoundRanges) { extendBoundRanges(movedBoundRanges); }

            // move the existing cell entries
            if (moveRange) {
                changedCells = changedCells.concat(moveEntriesInRange(moveRange, moveCols, moveRows));
            }

            // insert new cell entries according to preceding cells
            if (interval.first > 0) {

                var // the column/row collection in the specified move direction
                    collection = columns ? colCollection : rowCollection,
                    // the range address of the preceding column/row
                    prevRange = _.copy(range, true),
                    // the names of the border attributes in move direction
                    innerBorderName1 = columns ? 'borderLeft' : 'borderTop',
                    innerBorderName2 = columns ? 'borderRight' : 'borderBottom',
                    // the names of the border attributes in the opposite direction
                    outerBorderName1 = columns ? 'borderTop' : 'borderLeft',
                    outerBorderName2 = columns ? 'borderBottom' : 'borderRight',
                    // limit number of cell entries inserted into the collection
                    count = 0;

                // adjust column/row indexes for the range address preceding the inserted range
                prevRange.start[addrIndex] = prevRange.end[addrIndex] = interval.first - 1;

                // process all cell entries existing in the previous column/row
                self.iterateCellsInRanges(SheetUtils.getIntersectionRanges(boundRanges, prevRange), function (prevCellData) {

                    var // the explicit cell attributes of the preceding cell entry
                        prevCellAttrs = prevCellData.explicit.cell,
                        // the cell entry after the inserted range
                        nextCellData = null,
                        // the explicit cell attributes of the following cell entry
                        nextCellAttrs = null,
                        // the explicit attributes to be set at the new cells
                        newAttributes = _.copy(prevCellData.explicit),
                        // a helper cell address to obtain cell entries from the collection
                        tempAddress = _.clone(prevCellData.address);

                    // returns whether the passed cell attribute map contains two equal border attributes
                    function hasEqualBorders(cellAttributes, borderName1, borderName2) {
                        var border1 = cellAttributes[borderName1], border2 = cellAttributes[borderName2];
                        return _.isObject(border1) && _.isObject(border2) && Border.isEqual(border1, border2);
                    }

                    // returns the border attribute, if it is equal in both passed cell attribute maps
                    function getEqualBorder(cellAttributes1, borderName1, cellAttributes2, borderName2) {
                        var border1 = cellAttributes1[borderName1], border2 = cellAttributes2[borderName2];
                        return (_.isObject(border1) && _.isObject(border2) && Border.isEqual(border1, border2)) ? border1 : null;
                    }

                    // get following cell entry
                    tempAddress[addrIndex] = range.end[addrIndex] + 1;
                    if (tempAddress[addrIndex] <= maxIndex) {
                        nextCellData = self.getCellEntry(tempAddress);
                        nextCellAttrs = nextCellData.explicit.cell;
                    }

                    // prepare explicit border attributes for the new cells
                    if (_.isObject(prevCellAttrs)) {

                        // remove old borders
                        delete newAttributes.cell[innerBorderName1];
                        delete newAttributes.cell[innerBorderName2];
                        delete newAttributes.cell[outerBorderName1];
                        delete newAttributes.cell[outerBorderName2];

                        // copy borders, that are equal in preceding and following cell entries
                        if (_.isObject(nextCellAttrs)) {

                            var border = null;

                            // insert columns: clone top border if equal in preceding and following column
                            // insert rows: clone left border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, outerBorderName1, nextCellAttrs, outerBorderName1))) {
                                newAttributes.cell[outerBorderName1] = _.copy(border, true);
                            }

                            // insert columns: clone bottom border if equal in preceding and following column
                            // insert rows: clone right border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, outerBorderName2, nextCellAttrs, outerBorderName2))) {
                                newAttributes.cell[outerBorderName2] = _.copy(border, true);
                            }

                            // insert columns: clone left/right border if equal in preceding and following column
                            // insert rows: clone top/bottom border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, innerBorderName2, nextCellAttrs, innerBorderName1))) {
                                // ... but only if *both* borders in either preceding or following column/row are equal
                                if (hasEqualBorders(prevCellAttrs, innerBorderName1, innerBorderName2) || hasEqualBorders(nextCellAttrs, innerBorderName1, innerBorderName2)) {
                                    newAttributes.cell[innerBorderName1] = _.copy(border, true);
                                    newAttributes.cell[innerBorderName2] = _.copy(border, true);
                                }
                            }
                        }

                        // remove empty cell attribute map
                        if (_.isEmpty(newAttributes.cell)) {
                            delete newAttributes.cell;
                        }
                    }

                    // no cell entries, if no explicit attributes left
                    if (_.isEmpty(newAttributes)) { return; }

                    // prevent memory overflow or performance problems
                    count += SheetUtils.getIntervalSize(interval);
                    if (count > SheetUtils.MAX_FILL_CELL_COUNT) { return Utils.BREAK; }

                    // create equal cell entries inside the insertion interval
                    collection.iterateEntries(interval, function (moveEntry) {
                        tempAddress[addrIndex] = moveEntry.index;
                        changedCells.push(_.clone(tempAddress));
                        createCellEntry(tempAddress, { attrs: newAttributes });
                    }, { hidden: 'all' });

                }, { type: 'existing', hidden: 'all' });
            }

            return changedCells;
        }

        /**
         * Removes the passed cell ranges from the collection, and moves the
         * remaining cells up or to the left.
         *
         * @param {Object} range
         *  The address of the cell range to be deleted.
         *
         * @param {Boolean} columns
         *  Either true to move the remaining cells horizontally (to the left
         *  or right according to the sheet orientation), or false to move the
         *  remaining cells up.
         *
         * @returns {Array}
         *  An array with the cell addresses of all changed cells (the
         *  addresses of the deleted cells, and the old and new addresses of
         *  existing moved cells).
         */
        function deleteCells(range, columns) {

            var // the address array index used to move the cells
                addrIndex = columns ? 0 : 1,
                // the last available column/row index in the sheet
                maxIndex = docModel.getMaxIndex(columns),
                // number of columns the entries will be moved
                moveCols = columns ? -SheetUtils.getColCount(range) : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : -SheetUtils.getRowCount(range),
                // the cell range with all cells to be moved
                moveRange = _.copy(range, true),
                // the entire modified range (deleted and moved)
                modifiedRange = _.copy(range, true),
                // the bounding ranges inside the moved range
                movedBoundRanges = null,
                // the addresses of all changed cells
                changedCells = null;

            // calculate moved and modified range
            moveRange.start[addrIndex] = range.end[addrIndex] + 1;
            moveRange.end[addrIndex] = modifiedRange.end[addrIndex] = maxIndex;
            if (moveRange.start[addrIndex] > moveRange.end[addrIndex]) { moveRange = null; }

            // calculate the target position of the moved parts of the bounding ranges
            if (moveRange) {
                movedBoundRanges = SheetUtils.getIntersectionRanges(boundRanges, moveRange);
                _.each(movedBoundRanges, function (boundRange, index) {
                    movedBoundRanges[index] = docModel.getCroppedMovedRange(boundRange, moveCols, moveRows);
                });
            }

            // cut the entire modified area (deleted and moved) from the bounding ranges, and
            // add the target position of the existing moved parts of the old bounding ranges
            boundRanges = SheetUtils.getRemainingRanges(boundRanges, modifiedRange);
            logBoundRanges();
            if (movedBoundRanges) { extendBoundRanges(movedBoundRanges); }

            // clear the entire deleted range, then move the following cell entries
            changedCells = deleteEntriesInRanges(range);
            if (moveRange) {
                changedCells = changedCells.concat(moveEntriesInRange(moveRange, moveCols, moveRows));
            }

            return changedCells;
        }

        /**
         * Updates the cell entries after changing the formatting attributes of
         * entire columns or rows.
         */
        function changeCellAttributesInInterval(interval, columns, attributes, options) {

            var // the column/row range covering the passed interval
                range = docModel.makeFullRange(interval, columns),
                // all merged ranges covering the range partly or completely
                mergedRanges = null,
                // special treatment for border attributes applied to entire ranges
                rangeBorders = Utils.getBooleanOption(options, 'rangeBorders', false),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = Utils.getBooleanOption(options, 'visibleBorders', false),
                // additional options for border attributes
                attributeOptions = { visibleBorders: visibleBorders },
                // the row intervals of the bounding ranges covering the modified columns
                rowIntervals = null,
                // the number of modified cells
                count = 0;

            // do nothing if the passed attributes do not modify the cell formatting
            if (!_.isObject(attributes) || (_.isEmpty(attributes.cell) && _.isEmpty(attributes.character) && !('styleId' in attributes))) {
                return;
            }

            // all merged ranges covering the range partly or completely
            mergedRanges = mergeCollection.getMergedRanges(range);

            // update all existing cell entries
            self.iterateCellsInRanges(range, function (cellData) {

                // do not update any merged ranges here
                if (SheetUtils.rangesContainCell(mergedRanges, cellData.address)) { return; }

                // add options for inner/outer border treatment
                attributeOptions.innerLeft = rangeBorders && (cellData.address[0] > range.start[0]);
                attributeOptions.innerRight = rangeBorders && (cellData.address[0] < range.end[0]);
                attributeOptions.innerTop = rangeBorders && (cellData.address[1] > range.start[1]);
                attributeOptions.innerBottom = rangeBorders && (cellData.address[1] < range.end[1]);

                // update the cell entry
                self.updateCellEntry(cellData.address, undefined, attributes, attributeOptions);

            }, { type: 'existing', hidden: 'all' });

            // update reference cell of all merged ranges located completely inside the passed range
            _.each(mergedRanges, function (mergedRange) {

                // do not handle merged ranges not contained completely
                if (!SheetUtils.rangeContainsRange(range, mergedRange)) { return; }

                // add options for inner/outer border treatment
                attributeOptions.innerLeft = rangeBorders && (mergedRange.start[0] > range.start[0]);
                attributeOptions.innerRight = rangeBorders && (mergedRange.end[0] < range.end[0]);
                attributeOptions.innerTop = rangeBorders && (mergedRange.start[1] > range.start[1]);
                attributeOptions.innerBottom = rangeBorders && (mergedRange.end[1] < range.end[1]);

                // update the cell entry
                self.updateCellEntry(mergedRange.start, undefined, attributes, attributeOptions);
            });

            // column formatting: create cell entries for all rows with explicit default formats
            if (!columns) { return; }

            // the row intervals of the bounding ranges covering the modified columns
            rowIntervals = SheetUtils.getRowIntervals(SheetUtils.getIntersectionRanges(boundRanges, range));

            // visit all rows with custom formatting attributes
            return rowCollection.iterateEntries(rowIntervals, function (rowEntry) {

                // add options for inner/outer border treatment
                attributeOptions.innerTop = rangeBorders && (rowEntry.index > range.start[1]);
                attributeOptions.innerBottom = rangeBorders && (rowEntry.index < range.end[1]);

                // prevent memory overflow or performance problems
                count += SheetUtils.getIntervalSize(interval);
                if (count > SheetUtils.MAX_FILL_CELL_COUNT) { return Utils.BREAK; }

                // update cells in all columns of the row
                colCollection.iterateEntries(interval, function (colEntry) {

                    var // the cell address
                        address = [colEntry.index, rowEntry.index];

                    // do not modify existing cell entries (already updated above)
                    if (getCellEntry(address)) { return; }

                    // add options for inner/outer border treatment
                    attributeOptions.innerLeft = rangeBorders && (colEntry.index > interval.first);
                    attributeOptions.innerRight = rangeBorders && (colEntry.index < interval.last);

                    // update the cell entry
                    self.updateCellEntry(address, undefined, attributes, attributeOptions);

                }, { hidden: 'all' });
            }, { hidden: 'all', customFormat: true });
        }

        /**
         * Imports the passed cell contents array as received from a server
         * notification.
         *
         * @param {Array} rangeContents
         *  The contents array to be imported into this cell collection.
         *
         * @returns {Array}
         *  The addresses of all imported cell ranges.
         */
        function importRangeContents(rangeContents) {

            var // the imported cell ranges
                changedRanges = [];

            // process all entries in the passed array
            _.each(rangeContents, function (rangeData) {

                // check validity of range address, add missing 'end' property
                if (!_.isObject(rangeData)) { return; }
                if (!('end' in rangeData)) { rangeData.end = rangeData.start; }
                if (!docModel.isValidRange(rangeData)) { return; }

                // create entries for all cells in the range
                for (var row = rangeData.start[1]; row <= rangeData.end[1]; row += 1) {
                    for (var col = rangeData.start[0]; col <= rangeData.end[0]; col += 1) {
                        createCellEntry([col, row], rangeData);
                    }
                }

                // add the range to the result array
                changedRanges.push({ start: rangeData.start, end: rangeData.end });
            });

            return changedRanges;
        }

        /**
         * Fetches the contents and formatting of the passed cell ranges, and
         * updates the entries of this collection. If the passed ranges are too
         * large, the server may respond with less cells than requested. The
         * response of the promise returned by this method will contain the
         * addresses of the ranges that have actually been imported.
         *
         * @param {Array} ranges
         *  The addresses of the cell ranges to be imported.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with the
         *  array of range addresses of all cells that have been returned by
         *  the server request.
         */
        function requestRangeContents(ranges) {

            var // the server request
                promise = null;

            // do nothing without any ranges, or if application is in quit/error state
            if (ranges.length === 0) { return $.when(ranges); }

            // do nothing if application is in quit/error state
            if (app.isInQuit() || (app.getState() === 'error')) { return $.Deferred().reject(); }

            // join and unify the cell ranges
            ranges = SheetUtils.getUnifiedRanges(ranges);
            SheetUtils.info('CellCollection.requestRangeContents(): requesting cells: ranges=' + SheetUtils.getRangesName(ranges) + ' max=' + MAX_CELLS_IN_RESPONSE);

            // wait for pending actions before sending the server request (fix for bug 30044,
            // must wait for the 'insertSheet' operation before requesting data for that sheet)
            promise = app.saveChanges().then(function () {
                return app.sendActionRequest('updateview', {
                    sheet: sheetModel.getIndex(),
                    locale: Config.LOCALE,
                    cells: ranges,
                    maxCells: MAX_CELLS_IN_RESPONSE
                });
            });

            // import all cells contained in the response
            promise = promise.then(function (responseData) {

                // bug 36913: sheet may have been deleted when the response arrives
                // TODO: need a way to bind promise chains to the lifetime of an arbitrary object
                if (!self || self.destroyed) { return; }

                SheetUtils.info('CellCollection.requestRangeContents(): response=', responseData);

                var // the ranges covered by the server response (default: all requested ranges)
                    responseRanges = ranges,
                    // the deleted cells in the imported ranges
                    changedCells = null,
                    // the changed ranges in the imported ranges
                    changedRanges = null;

                // update the number of used columns and rows
                setUsedArea(
                    Utils.getIntegerOption(responseData.sheet, 'usedCols', usedCols),
                    Utils.getIntegerOption(responseData.sheet, 'usedRows', usedRows)
                );

                // extract the range addresses contained in the response
                if (_.isArray(responseData.cellRanges) && (responseData.cellRanges.length > 0)) {
                    responseRanges = responseData.cellRanges;
                }

                // merge the new ranges into the bounding ranges of this collection
                extendBoundRanges(responseRanges);

                // delete the cell range
                changedCells = deleteEntriesInRanges(responseRanges);

                // import cell contents
                if (_.isArray(responseData.cells)) {
                    changedRanges = importRangeContents(responseData.cells);
                }

                // notify all change listeners
                triggerChangeCellsEvent(changedRanges, changedCells);

                // resolve returned promise with the response ranges
                return responseRanges;
            });

            return promise.fail(function (response) {
                if (response !== 'abort') {
                    SheetUtils.error('CellCollection.requestRangeContents(): request failed, response:', response);
                }
            });
        }

        /**
         * Fetches the contents and formatting of all passed cell ranges, and
         * updates the entries of this collection. If the passed ranges are too
         * large, the server may respond with less cells than requested. This
         * method will send further server requests, until all passed ranges
         * have been imported.
         *
         * @param {Array} ranges
         *  The addresses of the cell ranges to be updated.
         */
        var fetchAllRangeContents = (function () {

            var // the cell ranges to be queried
                pendingRanges = [];

            // direct callback: collect the passed cell range addresses
            function registerRequest(ranges) {
                // put new requests to the front to give newer requests a high priority
                pendingRanges = ranges.concat(pendingRanges);
            }

            // deferred callback: send single server request for all collected ranges
            function executeRequest() {

                // nothing to do, if pending ranges are still empty
                if (pendingRanges.length === 0) { return; }

                var // local reference to the pending ranges array
                    requestRanges = pendingRanges;

                // immediately clear the pending ranges array (new ranges may be fetched while the server request is running)
                pendingRanges = [];

                // request pending cell ranges from server
                requestRangeContents(requestRanges).done(function (responseRanges) {

                    var // missing cell ranges (requested, but not contained in the server response)
                        missingRanges = SheetUtils.getRemainingRanges(requestRanges, responseRanges);

                    // add missing ranges to the pending ranges array, and start a new server query
                    if (missingRanges.length > 0) {
                        fetchAllRangeContents(missingRanges);
                    }
                });
            }

            // create and return the debounced method fetchAllRangeContents()
            return self.createDebouncedMethod(registerRequest, executeRequest, { delay: 50, maxDelay: 250 });
        }());

        /**
         * Handles 'docs:update:cells' notifications. If the bound ranges of
         * this collection covers any changed cells, they will be queried from
         * the server.
         *
         * @param {Object} changedData
         *  A map with cell descriptors and range addresses of changed ranges
         *  per sheet.
         */
        function updateCellsHandler(changedData) {

            var // extract the changed data of the own sheet
                sheetData = changedData[sheetModel.getIndex()],
                // changed ranges in the own sheet
                changedRanges = null;

            // nothing to do, if the own sheet did not change at all
            if (!_.isObject(sheetData)) { return; }

            // update the number of used columns and rows
            setUsedArea(
                Utils.getIntegerOption(sheetData, 'usedCols', usedCols),
                Utils.getIntegerOption(sheetData, 'usedRows', usedRows));

            // import all cell contents, and notify all change listeners
            if (_.isArray(sheetData.rangeContents)) {
                triggerChangeCellsEvent(importRangeContents(sheetData.rangeContents));
            }

            // request cell contents for all other changed ranges
            if (_.isArray(sheetData.dirtyRanges)) {
                // reduce to the bounding ranges of this collection
                changedRanges = _.map(boundRanges, _.partial(SheetUtils.getIntersectionRanges, sheetData.dirtyRanges));
                // flatten the nested array of ranges
                fetchAllRangeContents(_.flatten(changedRanges, true));
            }
        }

        /**
         * Updates this collection, after columns or rows have been inserted
         * into or deleted from the sheet.
         */
        function transformationHandler(interval, insert, columns) {

            var // the range address of the inserted/deleted cells
                range = docModel.makeFullRange(interval, columns);

            if (insert) {
                insertCells(range, columns);
            } else {
                deleteCells(range, columns);
            }
        }

        /**
         * Updates all affected cell entries after merged ranges have been
         * inserted into the sheet.
         */
        function insertMergedHandler(event, ranges, options) {

            // moves the formatting attributes of the reference cell to a column or row
            function moveAttributesToColRow(interval, columns) {

                var // the column/row collection to be modified
                    collection = columns ? colCollection : rowCollection,
                    // the address of the reference cell
                    address = columns ? [interval.first, 0] : [0, interval.first],
                    // the cell entry for the reference cell
                    cellEntry = getCellEntry(address);

                function setCollectionAttributes(attributes) {
                    if (!columns) { attributes = _.extend({ row: { customFormat: true } }, attributes); }
                    collection.setAttributes(interval, attributes, { silent: true });
                }

                // delete all old explicit attributes of the column/row
                setCollectionAttributes(styleCollection.buildNullAttributes());

                // copy attributes of the cell to the column/row
                if (cellEntry) { setCollectionAttributes(cellEntry.getExplicitAttributes()); }
            }

            // add to bounding ranges (all cells have the same format)
            extendBoundRanges(ranges);

            // move value and formatting of first non-empty cell in each range to reference cell
            _.each(ranges, function (range) {

                var // the first non-empty cell will be moved to the reference cell
                    moved = false;

                // delete all covered cells, move first non-empty cell to reference cell
                self.iterateCellsInRanges(range, function (cellData) {
                    if (!moved && !CellCollection.isBlank(cellData)) {
                        // move first non-empty cell entry to the reference position
                        moved = true;
                        moveCellEntry(cellData.address, range.start);
                    } else if (!_.isEqual(cellData.address, range.start)) {
                        // remove all other cell entries (but the first) from the collection
                        deleteCellEntry(cellData.address);
                    }
                }, { type: 'existing', hidden: 'all' });

                // special handling for column and row ranges
                if (docModel.isColRange(range)) {
                    moveAttributesToColRow(SheetUtils.getColInterval(range), true);
                } else if (docModel.isRowRange(range)) {
                    moveAttributesToColRow(SheetUtils.getRowInterval(range), false);
                }
            });

            // notify listeners (unless this is an implicit change from a column/row operation)
            if (!Utils.getBooleanOption(options, 'implicit', false)) {
                triggerChangeCellsEvent(ranges, null, { merge: true });
            }
        }

        /**
         * Updates all affected cell entries after merged ranges have been
         * deleted from the sheet.
         */
        function deleteMergedHandler(event, ranges, options) {

            var // the parts of the passed ranges covered by the collection
                changedRanges = [];

            // do nothing for merged ranges that have been deleted implicitly while deleting
            // columns or rows (will be handled by the method CellCollection.transformationHandler())
            if (Utils.getBooleanOption(options, 'implicit', false)) { return; }

            // copy formatting of reference cell to all other cells formerly hidden by the
            // merged range (but not for entire column/row ranges)
            _.each(ranges, function (range) {

                // do not create cell entries for entire columns/rows
                // TODO: unmerging columns needs cell entries for explicit formatted rows
                if (docModel.isColRange(range) || docModel.isRowRange(range)) { return; }

                var // the cell entry of the reference cell
                    cellEntry = getCellEntry(range.start),
                    // the explicit attributes of the reference cell
                    attributes = cellEntry ? cellEntry.getExplicitAttributes() : null;

                // nothing to do, if the cell has no formatting attributes
                if (_.isEmpty(attributes)) {
                    extendBoundRanges(range);
                    changedRanges.push(range);
                    return;
                }

                var // performance: do not fill more than a specific number of cells
                    fillRanges = SheetUtils.shortenRangesByCellCount(range, SheetUtils.MAX_UNMERGE_CELL_COUNT);

                // add to bounding ranges (all cells have the same format)
                extendBoundRanges(fillRanges);

                // register in changed ranges, add to bounding ranges (all cells have the same format)
                changedRanges = changedRanges.concat(fillRanges);

                // create the cell entries
                self.iterateCellsInRanges(fillRanges, function (cellData) {
                    self.updateCellEntry(cellData.address, undefined, attributes);
                }, { hidden: 'all' });
            });

            // notify listeners
            triggerChangeCellsEvent(changedRanges, null, { merge: true });
        }

        /**
         * preformat should be called when cell is changed,
         * if value is assigned or in the attribues is a numberformat-entry
         * the numberformatter is called and assignedValue gets an display-entry
         * if assignedValue is null a new object is created
         *
         * @return assignedValue or a new object
         */
        function preformat(assignValue, value, attributes, cellData) {
            var
                // numberformatter for pre-format the values
                formatter = docModel.getNumberFormatter(),
                // temporary cell-attributes for the formating
                assignAttrs,
                // temporary cell-result for the formating
                assignResult,
                // the numberformat-entry taken from the operation
                numberFormat = null;

            assignValue = assignValue || {};

            //if the operation has a numberformat-entry then the numberformat was changed, so we want to pre-format
            if (attributes && attributes.cell && attributes.cell.numberFormat && attributes.cell.numberFormat) {
                numberFormat = attributes.cell.numberFormat;
            }

            //if the operation has a value-entry then the value was changed, so we want to pre-format
            if (_.isString(value) || _.isNumber(value) || numberFormat) {
                if (numberFormat) {
                    assignAttrs = attributes;
                    assignResult = cellData.result;
                } else {
                    assignAttrs = cellData.attributes;
                    assignResult = value;

                    if (cellData.format) {
                        // unformat/parse only needed if values are changed
                        var unform = formatter.unformatEditString(assignResult, cellData.format.cat);
                        if (_.isString(unform) || _.isNumber(unform)) {
                            assignResult = unform;
                            assignValue.result = unform;
                        }
                    }
                }
                if (_.isString(assignResult) || _.isNumber(assignResult)) {
                    var result = formatter.formatCellContent(assignResult, assignAttrs);
                    if (_.isString(result) || _.isNumber(result)) {
                        assignValue.display = result;
                    }
                }
            }

            return assignValue;
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this cell collection for
         * the specified sheet.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {CellCollection}
         *  A clone of this cell collection, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new collection, pass all own entries as hidden parameter
            return new CellCollection(app, targetModel, {
                cellMap: cellMap,
                cellCount: cellCount,
                boundRanges: boundRanges,
                usedCols: usedCols,
                usedRows: usedRows
            });
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the number of defined cells in the sheet.
         *
         * @returns {Number}
         *  The number of defined cells in the sheet.
         */
        this.getCellCount = function () {
            return cellCount;
        };

        /**
         * Returns the number of columns in the used area of the sheet.
         *
         * @returns {Number}
         *  The number of columns in the used area of the sheet.
         */
        this.getUsedCols = function () {
            return usedCols;
        };

        /**
         * Returns the number of rows in the used area of the sheet.
         *
         * @returns {Number}
         *  The number of rows in the used area of the sheet.
         */
        this.getUsedRows = function () {
            return usedRows;
        };

        /**
         * Returns the number of columns or rows in the used area of the sheet.
         *
         * @param {Boolean} columns
         *  Whether to return the number of used columns (true), or the number
         *  of used rows (false).
         *
         * @returns {Number}
         *  The number of columns/rows in the used area of the sheet.
         */
        this.getUsedCount = function (columns) {
            return columns ? usedCols : usedRows;
        };

        /**
         * Returns the address of the last used cell in the sheet. If the sheet
         * is completely empty, returns the address [0, 0].
         *
         * @returns {Number[]}
         *  The address of the last used cell in the sheet.
         */
        this.getLastUsedCell = function () {
            return ((usedCols > 0) && (usedRows > 0)) ? [usedCols - 1, usedRows - 1] : [0, 0];
        };

        /**
         * Returns the range address of the used area in the sheet. If the
         * sheet is completely empty, returns a range address covering the cell
         * A1 only.
         *
         * @returns {Object}
         *  The range address of the used area in the sheet.
         */
        this.getUsedRange = function () {
            return { start: [0, 0], end: this.getLastUsedCell() };
        };

        /**
         * Returns whether the passed cell address is contained in the bounding
         * ranges of this collection.
         *
         * @param {Array[2]} address
         *  The cell address to be checked.
         *
         * @returns {Boolean}
         *  Whether the bounding ranges of this collection contain the passed
         *  cell address.
         */
        this.containsCell = function (address) {
            return SheetUtils.rangesContainCell(boundRanges, address);
        };

        /**
         * Returns whether the passed ranges are completely contained in the
         * bounding ranges of this collection.
         *
         * @param {Object|Array} ranges
         *  A cell range address, or an array of range addresses.
         *
         * @returns {Boolean}
         *  Whether the bounding ranges of this collection contain the passed
         *  cell ranges.
         */
        this.containsRanges = function (ranges) {
            return SheetUtils.getRemainingRanges(ranges, boundRanges).length === 0;
        };

        /**
         * Returns the merged default attribute set used for undefined cells.
         *
         * @returns {Object}
         *  The merged default attribute set for undefined cells, containing
         *  the default column and row attributes.
         */
        this.getDefaultAttributes = function () {
            return defaultEntry.getMergedAttributes();
        };

        /**
         * Returns whether the specified cell is blank (no result value). A
         * blank cell may contain formatting attributes.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether the specified cell is blank.
         */
        this.isBlankCell = function (address) {
            return CellCollection.isBlank(getCellEntry(address));
        };

        /**
         * Returns the result value of the cell at the specified address.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @returns {Number|String|Boolean|Null}
         *  The typed result value of the cell, or the formula result.
         */
        this.getCellResult = function (address) {
            var cellEntry = getCellEntry(address);
            return cellEntry ? cellEntry.result : null;
        };

        /**
         * Returns the formula string of the cell at the specified address.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @returns {String|Null}
         *  The formula string of the specified cell; or null, if the cell does
         *  not contain a formula.
         */
        this.getCellFormula = function (address) {
            var cellEntry = getCellEntry(address);
            return cellEntry ? cellEntry.formula : null;
        };

        /**
         * Returns a descriptor object for the cell at the specified address.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @returns {Object}
         *  The cell descriptor object, with the following properties:
         *  - {Number[]} address
         *      The address of the cell.
         *  - {String} display
         *      The formatted display string of the cell.
         *  - {Number|String|Boolean|Null} result
         *      The typed result value of the cell, or the formula result.
         *  - {String|Null} formula
         *      The formula definition, if the cell contains a formula.
         *  - {Object} attributes
         *      The resulting merged attribute set for the cell. If the cell is
         *      undefined, this object will be set to the appropriate default
         *      cell formatting attributes of the row or column.
         *  - {Object} explicit
         *      The explicit attributes of the cell. If the cell is undefined,
         *      this object will be set to the appropriate default cell
         *      formatting attributes of the row or column.
         *  - {Object} format
         *      Additional number format settings for the cell.
         *  - {Boolean} exists
         *      Whether the cell entry exists in the collection. Will be set to
         *      false, if the cell descriptor has been built for an undefined
         *      cell position in the collection. The properties 'explicit' and
         *      'attributes' of this result have been built from default column
         *      and row attributes in this case.
         */
        this.getCellEntry = function (address) {

            var // existing collection entry
                cellEntry = getCellEntry(address),
                // the cell descriptor object
                cellData = null,
                // descriptor from row or column collection
                colRowEntry = null;

            // return descriptor of existing collection entries
            if (cellEntry) { return cellEntry.getData(address); }

            // build descriptor for undefined cells (use row or column formatting)
            cellData = defaultEntry.getData(address);

            // if the row entry contains attributes, use them (ignore column default attributes)
            colRowEntry = rowCollection.getEntry(address[1]);
            if (!colRowEntry.attributes.row.customFormat) {
                colRowEntry = colCollection.getEntry(address[0]);
            }
            cellData.attributes = colRowEntry.attributes;
            cellData.explicit = colRowEntry.explicit;

            return cellData;
        };

        /**
         * Invokes the passed iterator function for specific cells in the
         * passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  The address of a single range, or an array of range addresses whose
         *  cells will be iterated. All specified cells will be visited by this
         *  method, regardless of the size of the ranges.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all matching cells in the passed
         *  ranges. The ranges will be processed independently, cells covered
         *  by several ranges will be visited multiple times. Receives the
         *  following parameters:
         *  (1) {Object} cellData
         *      The cell descriptor object, as returned by the method
         *      CellCollection.getCellEntry(). Will be an existing object with
         *      complete default settings for all undefined cells not contained
         *      in this collection.
         *  (2) {Object} originalRange
         *      The address of the range containing the current cell (one of
         *      the ranges contained in the 'ranges' parameter passed to this
         *      method).
         *  (3) {Object} [mergedRange]
         *      If the option 'merged' has been set to true (see below), the
         *      address of the merged range containing the current cell will be
         *      passed. If no merged range exists for the visited cell, this
         *      parameter will be undefined.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Object} [options.boundRange]
         *      If specified, the range address of the bounding range this
         *      method invocation is restricted to. If omitted, the full ranges
         *      will be visited.
         *  @param {String} [options.type='all']
         *      Specifies which type of cells will be visited. Must be one of
         *      the following values:
         *      - 'all': All defined and undefined cells will be visited.
         *      - 'existing': Only cells that exist in this collection will be
         *          visited, regardless of their content value (including blank
         *          but formatted cells).
         *      - 'content': Only non-blank cells (with any content value) will
         *          be visited.
         *  @param {String} [options.hidden='none']
         *      Specifies how to handle cells contained in hidden rows or
         *      columns. Must be one of the following values:
         *      - 'none' (or omitted): Only visible cells will be visited.
         *      - 'all': All visible and hidden cells will be visited.
         *      - 'last': All visible cells, and all cells contained in hidden
         *          columns and/or rows that precede a visible column/row will
         *          be visited.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows; but in reversed order, if the option
         *      'reverse' is set to true). Otherwise, the order in which the
         *      cells will be visited is undetermined.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the ranges AND the cells in each range will be
         *      visited in reversed order. This option will be ignored, if the
         *      option 'ordered' has not been set to true.
         *  @param {Boolean} [options.merged=false]
         *      If set to true, the 'mergedRange' parameter of the iterator
         *      callback function will be initialized with the address of the
         *      merged range containing the visited cell. To improve iteration
         *      performance, this will not be done by default.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInRanges = function (ranges, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // the bounding range to restrict the visited area
                customBoundRange = Utils.getObjectOption(options, 'boundRange', docModel.getSheetRange()),
                // which cell entries (by type) will be visited
                cellType = Utils.getStringOption(options, 'type', 'all'),
                // whether to visit hidden cell entries
                hiddenMode = Utils.getStringOption(options, 'hidden', 'none'),
                // whether to visit cells in order
                ordered = Utils.getBooleanOption(options, 'ordered', false),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to pass the addresses of merged ranges to the iterator
                findMerged = Utils.getBooleanOption(options, 'merged', false),
                // the merged ranges covering the passed ranges
                mergedRanges = findMerged ? mergeCollection.getMergedRanges(ranges) : null;

            // process all ranges (inside the specified bounding range)
            return SheetUtils.iterateIntersectionRanges(ranges, customBoundRange, function (range, origRange) {

                function invokeIterator(cellData) {
                    var mergedRange = mergedRanges ? SheetUtils.findFirstRange(mergedRanges, cellData.address) : undefined;
                    return iterator.call(context, cellData, origRange, mergedRange);
                }

                // visit all cells (defined and undefined); simply loop over rows and columns (always in-order)
                if ((cellType !== 'existing') && (cellType !== 'content')) {
                    return iterateCellsInRange(range, function (colEntry, rowEntry, cellEntry, address) {

                        var // the descriptor object to be passed to the iterator
                            cellData = null;

                        // build the cell descriptor (create cell data with default formatting for empty cells)
                        if (cellEntry) {
                            cellData = cellEntry.getData(address);
                        } else {
                            cellData = defaultEntry.getData(address);
                            // if the row entry contains attributes, use them (ignore column default attributes)
                            cellData.attributes = rowEntry.attributes.row.customFormat ? rowEntry.attributes : colEntry.attributes;
                            cellData.explicit = rowEntry.attributes.row.customFormat ? rowEntry.explicit : colEntry.explicit;
                        }

                        // invoke the iterator function
                        return invokeIterator(cellData);

                    }, options);
                }

                // visit only existing cells: collect existing cell entries (needed for sorting)
                return (function () {

                    var // visible column and row intervals in the current range
                        colIntervals = getVisitedIntervals(colCollection, SheetUtils.getColInterval(range)),
                        rowIntervals = getVisitedIntervals(rowCollection, SheetUtils.getRowInterval(range)),
                        // number of visible columns, rows, and cells in the current range
                        colCount = Utils.getSum(colIntervals, SheetUtils.getIntervalSize),
                        rowCount = Utils.getSum(rowIntervals, SheetUtils.getIntervalSize),
                        totalCount = colCount * rowCount;

                    // returns the intervals to be actually visited for the specified interval
                    function getVisitedIntervals(collection, interval) {
                        return (hiddenMode === 'all') ? [interval] : collection.getVisibleIntervals(interval, { lastHidden: (hiddenMode === 'last') ? 1 : 0 });
                    }

                    // returns whether the passed intervals contain a specific index
                    function intervalsContainIndex(intervals, index) {
                        var arrayIndex = _.sortedIndex(intervals, { last: index }, 'last');
                        return (arrayIndex < intervals.length) && (intervals[arrayIndex].first <= index);
                    }

                    // returns whether the existing cell will be visited according to passed cell type
                    function isVisitedCell(cellEntry) {
                        return cellEntry && ((cellType === 'existing') || !CellCollection.isBlank(cellEntry));
                    }

                    // little helper to visit all cell entries in the current range, by iterating the cell map directly
                    function iterateCellMap(callback) {
                        return _.any(cellMap, function (cellEntry, key) {
                            if (isVisitedCell(cellEntry)) {
                                var address = SheetUtils.parseCellKey(key);
                                if (intervalsContainIndex(colIntervals, address[0]) && intervalsContainIndex(rowIntervals, address[1])) {
                                    return callback.call(self, cellEntry, address, key) === Utils.BREAK;
                                }
                            }
                        }) ? Utils.BREAK : undefined;
                    }

                    // no visible column and row contained in the range
                    if (totalCount === 0) { return; }

                    // performance: iterate over the entire cell map, if more cells are in the range than collection entries
                    if (cellCount < totalCount) {

                        // if 'ordered' option has not been passed, visit the cells unordered
                        if (!ordered) {
                            return iterateCellMap(function (cellEntry, address) {
                                return invokeIterator(cellEntry.getData(address));
                            });
                        }

                        // collect and sort all cells to be visited
                        var visitedEntries = [];
                        iterateCellMap(function (cellEntry, address) {
                            visitedEntries.push({ address: address, cellEntry: cellEntry });
                        });
                        visitedEntries.sort(function (entry1, entry2) { return SheetUtils.compareCells(entry1.address, entry2.address); });

                        // invoke iterator for all sorted entries
                        return Utils.iterateArray(visitedEntries, function (entry) {
                            return invokeIterator(entry.cellEntry.getData(entry.address));
                        }, { reverse: reverse });
                    }

                    // more collection entries than cells in the ranges: iterate all cells in the range (always ordered)
                    return iterateCellsInRange(range, function (colEntry, rowEntry, cellEntry, address) {
                        if (isVisitedCell(cellEntry)) {
                            return invokeIterator(cellEntry.getData(address));
                        }
                    }, options);
                }());

            }, { reverse: reverse });
        };

        /**
         * Invokes the passed iterator function for specific cells in a single
         * column or row, while moving away from the start cell into the
         * specified directions.
         *
         * @param {Number[]} address
         *  The address of the cell that will be visited first (the option
         *  'skipStartCell' can be used, if iteration shall start with the
         *  nearest available neighbor of the cell, and not with the cell
         *  itself).
         *
         * @param {String} directions
         *  How to move to the next cells while iterating. Supported values are
         *  'up', 'down', 'left', or 'right', or any white-space separated list
         *  of these values. Multiple directions will be processed in the same
         *  order as specified in the parameter.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all matching cells in the
         *  specified direction(s). Receives the following parameters:
         *  (1) {Object} cellData
         *      The cell descriptor object, as returned by the method
         *      CellCollection.getCellEntry(). Will be an existing object with
         *      complete default settings for all undefined cells not contained
         *      in this collection.
         *  (2) {Object} [mergedRange]
         *      If the option 'merged' has been set to true (see below), the
         *      address of the merged range containing the current cell will be
         *      passed. If no merged range exists for the visited cell, this
         *      parameter will be undefined.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Object} [options.boundRange]
         *      If specified, the range address of the bounding range this
         *      method invocation is restricted to. If omitted, the entire area
         *      of the sheet will be used.
         *  @param {String} [options.type='all']
         *      Specifies which type of cells will be visited. Must be one of
         *      the following values:
         *      - 'all': All defined and undefined cells will be visited.
         *      - 'existing': Only cells that exist in this collection will be
         *          visited, regardless of their content value (including empty
         *          but formatted cells).
         *      - 'content': Only non-empty cells (with any content value) will
         *          be visited.
         *  @param {String} [options.hidden='none']
         *      Specifies how to handle cells contained in hidden rows or
         *      columns. Must be one of the following values:
         *      - 'none' (or omitted): Only visible cells will be visited.
         *      - 'all': All visible and hidden cells will be visited.
         *      - 'last': All visible cells, and all cells contained in hidden
         *          columns and/or rows that precede a visible column/row will
         *          be visited.
         *  @param {Boolean} [options.skipStartCell=false]
         *      If set to true, iteration will start at the nearest visible
         *      neighbor of the cell specified in the 'address' parameter
         *      instead of that cell.
         *  @param {Boolean} [options.merged=false]
         *      If set to true, the 'mergedRange' parameter of the iterator
         *      callback function will be initialized with the address of the
         *      merged range containing the visited cell.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInLine = function (address, directions, iterator, options) {

            var // number of cells to skip before iteration starts
                skipCount = Utils.getBooleanOption(options, 'skipStartCell', false) ? 1 : 0;

            return Utils.iterateArray(directions.split(/\s+/), function (direction) {

                var // whether to iterate with variable column index
                    columns = /^(left|right)$/.test(direction),
                    // whether to iterate in reversed order (towards first column/row)
                    reverse = /^(up|left)$/.test(direction),
                    // the array index in cell addresses
                    addrIndex = columns ? 0 : 1,
                    // the entire column/row range containing the passed address
                    range = columns ? docModel.makeRowRange(address[1]) : docModel.makeColRange(address[0]);

                // check passed direction identifier
                if (!columns && !reverse && (direction !== 'down')) {
                    Utils.error('CellCollection.iterateCellsInLine(): invalid direction "' + direction + '"');
                    return Utils.BREAK;
                }

                // reduce range to leading or trailing part (do nothing, if the range becomes invalid)
                if (reverse) {
                    range.end[addrIndex] = address[addrIndex] - skipCount;
                } else {
                    range.start[addrIndex] = address[addrIndex] + skipCount;
                }
                if (range.start[addrIndex] > range.end[addrIndex]) { return; }

                // visit all cells in the resulting range, according to the passed settings
                return self.iterateCellsInRanges(range, function (cellData, origRange, mergedRange) {
                    return iterator.call(this, cellData, mergedRange);
                }, Utils.extendOptions(options, { reverse: reverse, ordered: true }));
            });
        };

        /**
         * Returns the next available non-blank cell inside the layer range.
         *
         * @param {Number[]} address
         *  The address of the cell whose nearest adjacent content cell will be
         *  searched.
         *
         * @param {String} direction
         *  The direction to look for the content cell. Must be one of the
         *  values 'left', 'right', 'up', or 'down'.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method CellCollection.iterateCellsInLine(), except for 'type' and
         *  'skipStartCell'.
         *
         * @returns {Object|Null}
         *  The descriptor of an existing content cell; or null, if no content
         *  cell has been found.
         */
        this.findNearestContentCell = function (address, direction, options) {

            var // the resulting cell entry
                resultCellData = null;

            // iterate to the first available content cell
            this.iterateCellsInLine(address, direction, function (cellData) {
                resultCellData = cellData;
                return Utils.BREAK;
            }, Utils.extendOptions(options, { type: 'content', skipStartCell: true }));

            return resultCellData;
        };

        /**
         * Returns the content range (the entire range containing consecutive
         * content cells) surrounding the specified cell range.
         *
         * @param {Object} range
         *  The address of the cell range whose content range will be searched.
         *
         * @param {Object} [options}
         *  Optional parameters:
         *  @param {String} [options.directions='up down left right']
         *      All directions the passed range will be expanded to, as
         *      space-separated token list with the tokens 'up', 'down',
         *      'left', and 'right'. If omitted, the range will be expanded to
         *      all four directions.
         *
         * @returns {Object}
         *  The address of the content range of the specified cell.
         */
        this.findContentRange = SheetUtils.profileMethod('CellCollection.findContentRange()', function (range, options) {

            var // collection of merged ranges in the active sheet
                mergeCollection = sheetModel.getMergeCollection(),
                // the resulting content range (start by expanding to merged ranges)
                currRange = mergeCollection.expandRangeToMergedRanges(range),
                // the result of the last iteration, to detect exiting the loop
                lastRange = null,
                // all directions the range will be expanded to
                directions = Utils.getTokenListOption(options, 'directions', ['up', 'down', 'left', 'right']);

            // array iterator function, must be defined outside the do-while loop
            function expandRangeBorder(direction, index) {

                var // whether to move towards the end of the sheet
                    forward = /^(down|right)$/.test(direction),
                    // whether to move through columns in the same row
                    columns = /^(left|right)$/.test(direction),
                    // array index into cell addresses, in expansion direction
                    addrIndex = columns ? 0 : 1,
                    // maximum column/row index in the sheet, in expansion direction
                    maxIndex = docModel.getMaxIndex(columns),
                    // array index into cell addresses, in sidewise direction
                    addrIndex2 = columns ? 1 : 0,
                    // maximum column/row index in the sheet, in sidewise direction
                    maxIndex2 = docModel.getMaxIndex(!columns),
                    // adjacent range used to find a content cell next to the current range
                    boundRange = _.copy(currRange, true);

                // do nothing, if the range already reached the sheet borders
                if (forward ? (currRange.end[addrIndex] === maxIndex) : (currRange.start[addrIndex] === 0)) {
                    directions.splice(index, 1);
                    return false;
                }

                // build a range next to the current range (expand sideways, unless these directions are not active anymore)
                boundRange.start[addrIndex] = boundRange.end[addrIndex] = forward ? (currRange.end[addrIndex] + 1) : (currRange.start[addrIndex] - 1);
                if ((boundRange.start[addrIndex2] > 0) && _.contains(directions, columns ? 'up' : 'left')) { boundRange.start[addrIndex2] -= 1; }
                if ((boundRange.start[addrIndex2] < maxIndex2) && _.contains(directions, columns ? 'down' : 'right')) { boundRange.end[addrIndex2] += 1; }

                // find a content cell next to the current range (including hidden columns/rows)
                self.iterateCellsInRanges(boundRange, function (cellData1) {

                    var // reference to start or end address in the current range, according to direction
                        currAddress = forward ? currRange.end : currRange.start;

                    // content cell found: expand the current range
                    currAddress[addrIndex] = cellData1.address[addrIndex];

                    // performance: iterate into the expanded direction as long as there are adjacent content cells
                    self.iterateCellsInLine(cellData1.address, direction, function (cellData2) {
                        if (CellCollection.isBlank(cellData2)) { return Utils.BREAK; }
                        currAddress[addrIndex] = cellData2.address[addrIndex];
                    }, { hidden: 'all', skipStartCell: true });

                    // content cell found and range expanded: exit loop
                    return Utils.BREAK;
                }, { hidden: 'all', type: 'content' });
            }

            // expand in all four directions until the range does not change anymore
            do {

                // store result of last iteration for comparison below
                lastRange = _.copy(currRange, true);

                // expand current range into all remaining directions
                Utils.iterateArray(directions, expandRangeBorder, { reverse: true });

                // expand the range to include merged ranges
                currRange = mergeCollection.expandRangeToMergedRanges(currRange);

            } while ((directions.length > 0) && !_.isEqual(currRange, lastRange));

            // return the resulting range
            return currRange;
        });

        /**
         * Returns whether all passed ranges are blank (no values). Ignores
         * cell formatting attributes.
         *
         * @param {Object|Array} ranges
         *  The address of a single range, or an array of range addresses whose
         *  cells will be checked.
         *
         * @returns {Boolean}
         *  Whether all cells in the passed ranges are blank.
         */
        this.areRangesBlank = function (ranges) {

            var // whether any cell has a value
                hasContentCell = false;

            // try to find the first content cell
            this.iterateCellsInRanges(ranges, function () {
                hasContentCell = true;
                return Utils.BREAK;
            }, { type: 'content', hidden: 'all' });

            return !hasContentCell;
        };

        /**
         * Checks whether the passed cell range is covered by the bounding
         * ranges of this cell collection. Missing cells will be queried from
         * the server, and listeners of this collection will be notified with a
         * 'change:cells' event.
         *
         * @param {Object} range
         *  The address of the cell range to be included in this collection.
         *
         * @param {Object} [options]
         *  Additional options for this method.
         *  @param {Boolean} [options.visible=false]
         *      If set to true, only the visible parts of the cell range will
         *      be requested from the server. Cells contained in hidden columns
         *      or rows will be skipped, except for cells located in the last
         *      hidden columns or rows preceding the next visible column/row.
         *  @param {Boolean} [options.merged=false]
         *      If set to true, the reference cells of merged ranges that
         *      overlap the passed cell range will be updated too.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.fetchCellRange = function (range, options) {

            var // the resulting cell range addresses of the visible parts of the passed cell range
                visibleRanges = null,
                // the parts of these ranges that are not yet contained in the bounding ranges
                missingRanges = null;

            // request only the visible parts of the range
            if (Utils.getBooleanOption(options, 'visible', false)) {

                var // the visible column/row intervals in the passed range
                    colIntervals = colCollection.getVisibleIntervals(SheetUtils.getColInterval(range), { lastHidden: 5 }),
                    rowIntervals = rowCollection.getVisibleIntervals(SheetUtils.getRowInterval(range), { lastHidden: 5 });

                // combine the column/row intervals to a range list
                visibleRanges = SheetUtils.makeRangesFromIntervals(colIntervals, rowIntervals);
            } else {
                visibleRanges = [range];
            }

            // add the reference cells of the merged ranges overlapping with the resulting visible ranges
            if (Utils.getBooleanOption(options, 'merged', false)) {

                var // the merged ranges overlapping with the visible ranges
                    mergedRanges = mergeCollection.getMergedRanges(visibleRanges);

                if (mergedRanges.length > 0) {
                    _.each(mergedRanges, function (mergedRange) {
                        visibleRanges.push({ start: mergedRange.start, end: _.clone(mergedRange.start) });
                    });
                    visibleRanges = SheetUtils.getUnifiedRanges(visibleRanges);
                }
            }

            // get the missing parts of the visible ranges that have to be requested from the server
            missingRanges = SheetUtils.getRemainingRanges(visibleRanges, boundRanges);
            if (missingRanges.length > 0) {
                fetchAllRangeContents(missingRanges);
            }

            return this;
        };

        /**
         * Fetches all formula cells existing in this cell collection from the
         * server. Used to validate preliminary contents shown while loading
         * the document.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.fetchAllFormulaCells = SheetUtils.profileMethod('CellCollection.fetchAllFormulaCells()', function () {

            var // the cell addresses of all existing formula cells
                addresses = [];

            // collect addresses of all formula cells
            _.each(cellMap, function (cellEntry, key) {
                if (_.isString(cellEntry.formula)) {
                    addresses.push(SheetUtils.parseCellKey(key));
                }
            });

            // fetch all formula cells from the server
            if (addresses.length > 0) {
                fetchAllRangeContents(SheetUtils.joinCellsToRanges(addresses));
            }

            return this;
        });

        /**
         * Recalculates the results of all formula cells existing in this cell
         * collection locally.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after all
         *  formula cells have been recalculated.
         */
        this.refreshAllFormulaCells = LOCAL_FORMULAS ? SheetUtils.profileAsyncMethod('CellCollection.refreshAllFormulaCells()', function () {

            var // the cell addresses of all existing formula cells
                addresses = [];

            return this.iterateObjectSliced(cellMap, function (cellEntry, key) {
                if (_.isString(cellEntry.formula)) {
                    var address = SheetUtils.parseCellKey(key);
                    addresses.push(address);
                    cellEntry.updateValue(address);
                }
            }).done(function () {
                triggerChangeCellsEvent(null, addresses);
            });
        }) : $.when;

        /**
         * Returns the results and display strings of all cells contained in
         * the passed cell ranges. If the cell collection does not cover all
         * passed ranges, the missing data will be requested from the server.
         *
         * @param {Array} rangesArray
         *  An array of arrays (!) of cell range addresses. Each array element
         *  is treated as an independent range list.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, all cells contained in hidden columns or rows
         *      will be included into the result. By default, only visible
         *      cells will be returned.
         *  @param {Boolean} [options.attributes=false]
         *      If set to true, the cell content objects will contain the
         *      merged formatting attributes of the cells.
         *  @param {Boolean} [options.compressed=false]
         *      If set to true, the returned cell arrays will be optimized:
         *      Consecutive cells with equal contents and formatting will be
         *      represented by a single array element with an additional
         *      property 'count'. Especially useful, if large ranges located in
         *      the unused area of the sheet are queried.
         *  @param {Number} [options.maxCount]
         *      If specified, the maximum number of cells per range list that
         *      will be returned in the result, regardless how large the passed
         *      ranges are (the passed ranges will be shortened before the cell
         *      contents will be requested). Example: The first array element
         *      in the parameter 'rangesArray' covers 2000 cells (in several
         *      ranges), the second array element covers 500 cells. With this
         *      option set to the value 1000, the first range list will be
         *      shortened to 1000 cells, the second range list will be resolved
         *      completely. If omitted, and the ranges are too large (as
         *      defined by the constant SheetUtils.MAX_QUERY_CELL_COUNT), the
         *      entire request will be rejected (in compressed mode, the size
         *      of the compressed result array will be checked against that
         *      constant, not the number of cells in the original ranges).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with the
         *  cell contents referred by the passed ranges. The result will be an
         *  array of arrays of cell content objects (the length of the outer
         *  result array will be equal to the length of the 'rangesArray'
         *  parameter). Each cell object in the inner arrays will contain the
         *  following properties:
         *  - {String} cellData.display
         *      The display string of the cell result (will be the empty string
         *      for blank cells).
         *  - {Number|String|Boolean|ErrorCode|Null} cellData.result
         *      The typed result value (will be null for blank cells).
         *  - {Object} [cellData.attrs]
         *      The merged formatting attributes of the cell. Will only be set,
         *      if the option 'attributes' has been set to true (see above).
         *  - {Object} [cellData.format]
         *      Additional number format information for the cell format code.
         *      Will only be set, if the option 'attributes' has been set (see
         *      above).
         *  - {Number} [cellData.count]
         *      Always set in compressed mode (see option 'compressed' above).
         *      Contains the number of consecutive cells with equal contents
         *      and formatting represented by this array element.
         *  The cells in the inner arrays will be in order of the range lists.
         *  Cells from a single range will be ordered row-by-row.
         */
        this.queryCellContents = function (rangesArray, options) {

            var // the Deferred object that will be resolved with the cell contents
                def = $.Deferred(),
                // the abortable promise returned by this method
                promise = this.createAbortablePromise(def),
                // whether to include hidden cells
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // whether to insert the merged formatting attributes
                attributes = Utils.getBooleanOption(options, 'attributes', false),
                // compressed mode
                compressed = Utils.getBooleanOption(options, 'compressed', false),
                // the maximum number of cells in the result
                maxCount = Utils.getIntegerOption(options, 'maxCount', null, 1);

            // inserts the passed result entry into the array (handles compressed mode)
            function pushResult(results, entry, count) {

                var // the preceding result, for compressed mode
                    prevEntry = compressed ? _.last(results) : null;

                // increase count of previous entry, if the new entry is equal
                if (prevEntry) {
                    entry.count = prevEntry.count;
                    if (_.isEqual(prevEntry, entry)) {
                        prevEntry.count += count;
                        return;
                    }
                }

                // insert the new entry into the array (prepare count for compressed mode)
                if (compressed) {
                    entry.count = count;
                    results.push(entry);
                } else {
                    for(; count > 0; count -= 1) { results.push(entry); }
                }
            }

            // pushes result entries for blank cells without formatting attributes
            function pushBlanks(results, count) {
                if (count > 0) {
                    pushResult(results, { result: null, display: '' }, count);
                }
            }

            // collects all cells in the passed range without formatting attributes
            function collectWithoutAttributes(results, range) {

                var // all existing entries, needed for ordering
                    existingEntries = [],
                    // number of columns in the range
                    colCount = SheetUtils.getColCount(range),
                    // linear index of next expected cell in the range (used to fill blank cells)
                    nextIndex = 0;

                // collect all existing cells in the range
                iterateExistingEntries(range, function (cellEntry, key, address) {
                    var index = (address[1] - range.start[1]) * colCount + (address[0] - range.start[0]);
                    existingEntries.push({ index: index, cellEntry: cellEntry });
                });

                // sort by cell address (row-by-row)
                existingEntries.sort(function (entry1, entry2) { return entry1.index - entry2.index; });

                // fill the result array with existing cells and preceding blank cells
                _.each(existingEntries, function (entry) {
                    pushBlanks(results, entry.index - nextIndex);
                    pushResult(results, { result: entry.cellEntry.result, display: entry.cellEntry.display }, 1);
                    nextIndex = entry.index + 1;
                });

                // fill up following blank cells
                pushBlanks(results, SheetUtils.getCellCount(range) - nextIndex);
            }

            // collects all cells in the passed range with formatting attributes
            function collectWithAttributes(results, range) {
                iterateCellsInRange(range, function (colEntry, rowEntry, cellEntry, address) {
                    // resolve formatting attributes (including column/row formatting)
                    cellEntry = self.getCellEntry(address);
                    // push a single result entry
                    pushResult(results, {
                        result: cellEntry.result,
                        display: cellEntry.display,
                        attrs: cellEntry.attributes,
                        format: cellEntry.format
                    }, 1);
                });
            }

            // collects all cells in the passed ranges, and returns a flat result array
            function collectCellData(ranges) {
                var results = [];
                _.each(ranges, function (range) {
                    if (attributes) {
                        collectWithAttributes(results, range);
                    } else {
                        collectWithoutAttributes(results, range);
                    }
                });
                return results;
            }

            // reduce the ranges to their visible parts
            if (!hidden) {
                rangesArray = _.map(rangesArray, function (ranges) {
                    return sheetModel.getVisibleRanges(ranges);
                });
            }

            // shorten the ranges if specified
            if (_.isNumber(maxCount)) {
                rangesArray = _.map(rangesArray, function (ranges) {
                    return SheetUtils.shortenRangesByCellCount(ranges, maxCount);
                });
            }

            // reject the entire request, if the ranges are too large (in compressed
            // mode, the size of the result array will be checked later)
            if (!compressed && _.any(rangesArray, function (ranges) {
                return SheetUtils.getCellCountInRanges(ranges) > SheetUtils.MAX_QUERY_CELL_COUNT;
            })) {
                def.reject('overflow');
                return promise;
            }

            // request missing cell data from server, resolve the ranges to the result data
            (function resolveRanges() {

                var // the missing ranges not yet covered by the cell collection
                    missingRanges = SheetUtils.getRemainingRanges(_.flatten(rangesArray, true), boundRanges),
                    // the array of all result arrays
                    resultsArray = null;

                // creates and pushes a new result array to the overall result, checks array length
                function pushResultsForRanges(ranges) {
                    var results = collectCellData(ranges);
                    if (compressed && (results.length > SheetUtils.MAX_QUERY_CELL_COUNT)) {
                        def.reject('overflow');
                        return false;
                    }
                    resultsArray.push(results);
                    return true;
                }

                // request missing ranges from server, repeat resolving ranges recursively
                if (missingRanges.length > 0) {
                    requestRangeContents(missingRanges).done(resolveRanges).fail(_.bind(def.reject, def));
                    return;
                }

                // resolve the Deferred object with the resulting array of arrays
                // of cell entries, if no ranges are missing anymore
                resultsArray = [];
                if (_.all(rangesArray, pushResultsForRanges)) {
                    def.resolve(resultsArray);
                }
            }());

            // return the abortable promise
            return promise;
        };

        /**
         * Updates the value and/or formatting attributes of a cell entry.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @param {Object} [parsedValue]
         *  The parsed value for the cell entry, in the attributes 'display',
         *  'result', 'formula', and 'format'.
         *
         * @param {Object} [attributes]
         *  If specified, additional explicit attributes for the cell.
         *
         * @param {Object} [options]
         *  Optional parameters that will be passed to the method
         *  CellAttributesModel.setCellAttributes(). Additionally, the
         *  following options are supported:
         *  @param {String} [options.cat]
         *      The number format category the format code contained in the
         *      passed formatting attributes belongs to.
         */
        this.updateCellEntry = function (address, parsedValue, attributes, options) {

            var // the cell entry to be updated
                cellEntry = null,
                // the new number format category
                category = Utils.getStringOption(options, 'cat', '');

            // update the cell value
            if (_.isObject(parsedValue)) {
                // do not create new entry, if cell will be cleared
                cellEntry = _.isNull(parsedValue.result) ? getCellEntry(address) : getOrCreateCellEntry(address);
                if (cellEntry) { cellEntry.updateValue(address, parsedValue); }
            }

            // update formatting attributes
            if (_.isObject(attributes)) {
                if (!cellEntry) { cellEntry = getOrCreateCellEntry(address); }
                cellEntry.setCellAttributes(attributes, options);
            }

            // update number format category
            if (category.length > 0) {
                if (!cellEntry) { cellEntry = getOrCreateCellEntry(address); }
                cellEntry.format.cat = category;
            }
        };

        /**
         * Handler for the 'setCellContents' operation. Changes the contents
         * and formatting attributes of the cells in this collection, and
         * triggers a 'change:cells' event for all changed cells.
         *
         * @param {Number[]} start
         *  The address of the first cell to be changed.
         *
         * @param {Object} operation
         *  The complete 'setCellContents' operations object.
         *
         * @returns {Boolean}
         *  Whether the passed operation is valid, and all cell contents have
         *  been inserted successfully.
         */
        this.setCellContents = function (start, operation) {

            var // the cell contents (2D array)
                contents = Utils.getArrayOption(operation, 'contents'),
                // whether to parse the cell contents
                parse = _.isString(operation.parse),
                // the maximum column index in the sheet
                maxCol = docModel.getMaxCol(),
                // the maximum number of columns in a changed row
                colCount = 0,
                // the result of the row iterator
                result = null,
                // the addresses of all changed cells
                changedCells = [];

            // check that the 'contents' operation property is a 2D array
            if (!contents || !_.all(contents, _.isArray)) { return false; }

            // check validity of the covered row interval
            if (start[1] + contents.length - 1 > docModel.getMaxRow()) { return false; }

            // process all rows in the contents array
            result = Utils.iterateArray(contents, function (rowContents, rowIndex) {

                var // the address of the current cell entry
                    address = [start[0], start[1] + rowIndex];

                // process all cell entries in the row array
                result = Utils.iterateArray(rowContents, function (cellContents) {

                    var // number of repetitions for this cell entry
                        repeat = Utils.getIntegerOption(cellContents, 'repeat', 1);

                    // check validity of the covered column interval
                    if (address[0] + repeat - 1 > maxCol) { return Utils.BREAK; }

                    // skip gaps (null value is allowed in operation and leaves cell unmodified)
                    if (_.isNull(cellContents)) {
                        address[0] += repeat;
                        return;
                    }

                    // otherwise, array element must be an object
                    if (!_.isObject(cellContents)) { return Utils.BREAK; }

                    // update the affected cell entries with the parsed value
                    _.times(repeat, function () {

                        var parsedValue = parseCellValue(cellContents, parse),
                            entry = self.getCellEntry(address);

                        if (_.isObject(parsedValue)) {
                            parsedValue = preformat(parsedValue, parsedValue.result, null, entry);
                        }

                        self.updateCellEntry(address, parsedValue, cellContents.attrs);
                        changedCells.push(_.clone(address));
                        address[0] += 1;
                    });
                });

                // update the maximum column count
                colCount = Math.max(colCount, address[0] - start[0]);

                return result;
            });

            // update the used area of the sheet
            if ((colCount > 0) && (contents.length > 0)) {
                extendUsedArea([start[0] + colCount - 1, start[1] + contents.length - 1]);
            }

            // notify all change listeners
            triggerChangeCellsEvent(null, changedCells);

            // if result is Utils.BREAK, the 'contents' property of the operation is invalid
            return result !== Utils.BREAK;
        };

        /**
         * Handler for the 'fillCellRange' operation. Fills all cells in the
         * specified cell range with the same value and formatting attributes.
         *
         * @param {Object} range
         *  The address of a single cell range.
         *
         * @param {Object} operation
         *  The complete 'fillCellRange' operations object.
         *
         * @returns {Boolean}
         *  Whether the passed operation is valid, and all cells have been
         *  filled successfully.
         */
        this.fillCellRange = function (range, operation) {

            var // the parsed value for all cell in all ranges
                parsedValue = parseCellValue(operation, _.isString(operation.parse)),
                // the new attributes passed in the operation
                attributes = operation.attrs,
                // special treatment for border attributes applied to entire ranges
                rangeBorders = Utils.getBooleanOption(operation, 'rangeBorders', false),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = Utils.getBooleanOption(operation, 'visibleBorders', false),
                // delete value: visit existing cells with a value only (bug 32610)
                iteratorType = (CellCollection.isBlank(parsedValue) && !_.isObject(attributes)) ? 'content' : 'all';

            // nothing to do, if neither value nor formatting will be changed
            if (!_.isObject(parsedValue) && !_.isObject(attributes)) { return true; }

            // update all affected cell entries
            this.iterateCellsInRanges(range, function (cellData, origRange, mergedRange) {

                var // additional options for border attributes
                    attributeOptions = { visibleBorders: visibleBorders },
                    // temporary object create from parsedValue, if parsedValue is null and required, an object will be created
                    assignValue = parsedValue;

                // do not update cells hidden by merged ranges
                if (!mergedRange || _.isEqual(mergedRange.start, cellData.address)) {

                    // add options for inner/outer border treatment
                    mergedRange = mergedRange || { start: cellData.address, end: cellData.address };
                    attributeOptions.innerLeft = rangeBorders && (mergedRange.start[0] > origRange.start[0]);
                    attributeOptions.innerRight = rangeBorders && (mergedRange.end[0] < origRange.end[0]);
                    attributeOptions.innerTop = rangeBorders && (mergedRange.start[1] > origRange.start[1]);
                    attributeOptions.innerBottom = rangeBorders && (mergedRange.end[1] < origRange.end[1]);

                    assignValue = preformat(assignValue, operation.value, attributes, cellData);

                    // update the cell entry
                    self.updateCellEntry(cellData.address, assignValue, attributes, attributeOptions);
                }
            }, { type: iteratorType, hidden: 'all', merged: true });

            // update the bounding ranges and the used area of the sheet
            extendBoundRanges(range);
            extendUsedArea(range.end);

            // notify all change listeners
            triggerChangeCellsEvent(range);
            return true;
        };

        /**
         * Handler for the 'clearCellRange' operation. Clears value and
         * formatting of all cells in the specified cell range.
         *
         * @param {Object} range
         *  The address of a single cell range.
         *
         * @param {Object} operation
         *  The complete 'clearCellRange' operations object.
         *
         * @returns {Boolean}
         *  Whether the passed operation is valid, and all cells have been
         *  cleared successfully.
         */
        this.clearCellRange = function (range/*, operation*/) {

            var // all existing entries that have been deleted
                changedCells = deleteEntriesInRanges(range);

            // update the bounding ranges of the sheet (used area cannot be shrunken locally)
            extendBoundRanges(range);

            // notify all change listeners
            triggerChangeCellsEvent(null, changedCells);
            return true;
        };

        /**
         * Handler for the 'autoFill' operation.
         *
         * @param {Object} range
         *  The address of a single cell range.
         *
         * @param {Object} operation
         *  The complete 'autoFill' operations object.
         *
         * @returns {Boolean}
         *  Whether the passed operation is valid, and the range has been
         *  filled successfully.
         */
        this.autoFill = function (range, operation) {

            var // the target range to be filled with this operation
                targetRange = _.copy(range, true),
                // whether to fill cells horizontally
                columns = null,
                // whether to fill cells before the source range
                reverse = null,
                // the cell contents to be inserted into the target cells
                contents = { display: CellCollection.PENDING_DISPLAY, result: null },
                // the cell formatting to be inserted into the target cells
                attributes = null,
                // whether to increase the numbers in the target cells
                incNumbers = false;

            // calculate the target range to be filled
            if (!docModel.isValidAddress(operation.target)) { return false; }
            if (range.end[0] === operation.target[0]) {
                columns = false;
                if (operation.target[1] < range.start[1]) {
                    // fill cells above source range
                    targetRange.start[1] = operation.target[1];
                    targetRange.end[1] = range.start[1] - 1;
                    reverse = true;
                } else if (range.end[1] < operation.target[1]) {
                    // fill cells below source range
                    targetRange.start[1] = range.end[1] + 1;
                    targetRange.end[1] = operation.target[1];
                    reverse = false;
                }
            } else if (range.end[1] === operation.target[1]) {
                columns = true;
                if (operation.target[0] < range.start[0]) {
                    // fill cells left of source range
                    targetRange.start[0] = operation.target[0];
                    targetRange.end[0] = range.start[0] - 1;
                    reverse = true;
                } else if (range.end[0] < operation.target[0]) {
                    // fill cells right of source range
                    targetRange.start[0] = range.end[0] + 1;
                    targetRange.end[0] = operation.target[0];
                    reverse = false;
                }
            }

            // bail out on invalid target address
            if (!_.isBoolean(columns) && !_.isBoolean(reverse)) { return false; }

            // prepare auto-fill for a single cell as source range
            if (_.isEqual(range.start, range.end)) {

                // copy all cell contents and formatting to the target range
                contents = this.getCellEntry(range.start);
                attributes = contents.explicit;
                delete contents.attributes;
                delete contents.explicit;

                // bug 33439: do not auto fill formula cells
                if (_.isString(contents.formula)) {
                    contents.display = CellCollection.PENDING_DISPLAY;
                }

                // number cell: increase the cell value (bug 31121: not for formatted numbers)
                else if (CellCollection.isNumber(contents) && (contents.format.cat === 'standard')) {
                    incNumbers = true;
                    if (reverse) {
                        contents.result -= (SheetUtils.getCellCount(targetRange) + 1);
                    }
                }
            }

            // update all cells in the target range
            if (SheetUtils.getCellCount(targetRange) <= SheetUtils.MAX_FILL_CELL_COUNT) {
                this.iterateCellsInRanges(targetRange, function (cellData, origRange, mergedRange) {
                    // do not update cells hidden by merged ranges
                    if (!mergedRange || _.isEqual(cellData.address, mergedRange.start)) {
                        if (incNumbers) {
                            contents.result += 1;
                            contents.display = String(contents.result);
                        }
                        self.updateCellEntry(cellData.address, contents, attributes);
                    }
                }, { hidden: 'all', merged: true });
            }

            // update the bounding ranges and the used area of the sheet
            extendBoundRanges(targetRange);
            extendUsedArea(targetRange.end);

            // notify all change listeners
            triggerChangeCellsEvent(targetRange);
            return true;
        };

        /**
         * Sets the specified number format category to all cells in the passed
         * cell ranges.
         *
         * @param {Object|Array} ranges
         *  The address of a single cell range, or an array of cell range
         *  addresses.
         *
         * @param {String} category
         *  The number format category.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.setFormatCategory = function (ranges, category) {

            // update all affected cell entries
            ranges = SheetUtils.getUnifiedRanges(ranges);
            this.iterateCellsInRanges(ranges, function (cellData, origRange, mergedRange) {
                // do not update cells hidden by merged ranges
                if (!mergedRange || _.isEqual(mergedRange.start, cellData.address)) {
                    self.updateCellEntry(cellData.address, undefined, undefined, { cat: category });
                }
            }, { hidden: 'all', merged: true });

            return this;
        };

        /**
         * Inserts new cells into the passed cell ranges, and moves the old
         * cells down or to the right.
         *
         * @param {Object} range
         *  The address of the cell range to be inserted.
         *
         * @param {Boolean} columns
         *  True to move the existing cells horizontally (to the right or left
         *  according to the sheet orientation), or false to move the existing
         *  cells down.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.insertCells = function (range, columns) {

            var // the addresses of all changed (inserted and moved) cells
                changedCells = insertCells(range, columns);

            // notify all change listeners
            triggerChangeCellsEvent(null, changedCells);
            return this;
        };

        /**
         * Removes the passed cell ranges from the collection, and moves the
         * remaining cells up or to the left.
         *
         * @param {Object} range
         *  The address of the cell range to be deleted.
         *
         * @param {Boolean} columns
         *  True to move the existing cells horizontally (to the left or right
         *  according to the sheet orientation), or false to move the existing
         *  cells up.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.deleteCells = function (range, columns) {

            var // the addresses of all changed (deleted and moved) cells
                changedCells = deleteCells(range, columns);

            // notify all change listeners
            triggerChangeCellsEvent(null, changedCells);
            return this;
        };

        // initialization -----------------------------------------------------

        // application notifies changed contents/results of cells
        this.listenTo(app, 'docs:update:cells', updateCellsHandler);

        // update cell entries after inserting/deleting columns or rows in the sheet
        sheetModel.registerTransformationHandler(transformationHandler);

        // update cell entries after formatting entire columns in the sheet
        colCollection.on('change:entries', function (event, interval, attributes, options) {
            changeCellAttributesInInterval(interval, true, attributes, options);
        });

        // update cell entries after formatting entire rows in the sheet
        rowCollection.on('change:entries', function (event, interval, attributes, options) {
            changeCellAttributesInInterval(interval, false, attributes, options);
        });

        // update cell entries after merging or unmerging ranges
        mergeCollection.on({
            'insert:merged': insertMergedHandler,
            'delete:merged': deleteMergedHandler
        });

        // clone collection entries passed as hidden argument by the clone() method
        (function (args) {
            var cloneData = args[CellCollection.length];
            if (_.isObject(cloneData)) {
                _.each(cloneData.cellMap, function (cellEntry, key) {
                    cellMap[key] = new EntryModel({
                        display: cellEntry.display,
                        result: cellEntry.result,
                        formula: cellEntry.formula,
                        format: cellEntry.format,
                        attrs: cellEntry.getExplicitAttributes()
                    });
                });
                cellCount = cloneData.cellCount;
                boundRanges = _.copy(cloneData.boundRanges, true);
                usedCols = cloneData.usedCols;
                usedRows = cloneData.usedRows;
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(cellMap, 'destroy');
            defaultEntry.destroy();
            tokenArray.destroy();
            app = docModel = styleCollection = numberFormatter = null;
            sheetModel = colCollection = rowCollection = mergeCollection = null;
            defaultEntry = cellMap = boundRanges = null;
        });

    } // class CellCollection

    // static constants and methods -------------------------------------------

    /**
     * Standard cell data object for empty cells.
     */
    CellCollection.DEFAULT_CELL_DATA = {
        display: '',
        result: null,
        formula: null,
        format: { cat: 'standard', places: 0, group: false, red: false }
    };

    /**
     * Display text for pending cells (e.g. for new formula cells waiting for
     * the server response with the formula result).
     */
    CellCollection.PENDING_DISPLAY = Utils.ELLIPSIS_CHAR;

    /**
     * Returns whether the cell with the passed descriptor does not contain any
     * value.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  true.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor is a blank cell.
     */
    CellCollection.isBlank = function (cellData) {
        return !cellData || _.isNull(cellData.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is a numeric cell,
     * including formula cells resulting in a number, including positive and
     * negative infinity.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a numeric result.
     */
    CellCollection.isNumber = function (cellData) {
        return !!cellData && _.isNumber(cellData.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is a Boolean cell,
     * including formula cells resulting in a Boolean value.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a Boolean result.
     */
    CellCollection.isBoolean = function (cellData) {
        return !!cellData && _.isBoolean(cellData.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is an error code
     * cell, including formula cells resulting in an error code.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has an error code (instance
     *  of the class ErrorCode) as result.
     */
    CellCollection.isError = function (cellData) {
        return !!cellData && SheetUtils.isErrorCode(cellData.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is a text cell,
     * including formula cells resulting in a text value.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a text result.
     */
    CellCollection.isText = function (cellData) {
        return !!cellData && _.isString(cellData.result);
    };

    /**
     * Returns whether the passed cell descriptor will wrap the text contents
     * of a cell.
     */
    function hasWrappingAttributes(cellData) {
        return cellData.attributes.cell.wrapText || (cellData.attributes.cell.alignHor === 'justify');
    }

    /**
     * Returns whether the cell with the passed descriptor wraps its text
     * contents at the left and right cell borders. This happens for all cells
     * with text result value, which either have the cell attribute 'wrapText'
     * set to true, or contain a justifying horizontal alignment.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell wraps its text contents automatically.
     */
    CellCollection.isWrappedText = function (cellData) {
        return CellCollection.isText(cellData) && hasWrappingAttributes(cellData);
    };

    /**
     * Returns whether the cell with the passed descriptor overflows its text
     * contents over the left and right cell borders. This happens for all
     * cells with text result value, which do not have the cell attribute
     * 'wrapText' set to true, and do not contain a justifying horizontal
     * alignment.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell contains text that can overflow the cell borders.
     */
    CellCollection.isOverflowText = function (cellData) {
        return CellCollection.isText(cellData) && !hasWrappingAttributes(cellData);
    };

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CellCollection });

});
