/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/view/pageloader', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/pdf/pdfview',
    'io.ox/office/baseframework/lib/pdftextlayerbuilder',
], function (Utils, Forms, BaseObject, TimerMixin, PDFView) {

    'use strict';

    var // the maximum number of simultaneous pending AJAX page requests
        MAX_REQUESTS = Modernizr.touch ? 2 : 5,

        // the number of AJAX requests currently running
        runningRequests = 0,

        // the temporary node used to calculate the original size of a page image
        tempNode = $('<div>').css('position', 'absolute');

    // private global functions ===============================================

    /**
     * Calculates the original size of the passed page image node.
     *
     * @param {jQuery} imageNode
     *  The node whose size will be calculated. May be an <img> element linking
     *  to an image, or an <svg> element containing parsed SVG mark-up.
     *
     * @param {Number} [zoom]
     *  If specified, the zoom factor passed in the server request. Needed to
     *  calculate the original page size from the physical size of the image.
     *
     * @returns {Object|Null}
     *  The size of the image node (in pixels), in the properties 'width' and
     *  'height'; or null if the size is not valid (zero).
     */
    function calculateImageSize(imageNode, zoom) {

        var // the original parent node
            parentNode = imageNode.parent(),
            // the resulting size of the passed image node
            width = 0, height = 0;

        // try to use the naturalWidth/naturalHeight attributes of <img> elements
        if (imageNode.is('img')) {
            // remove the 'max-width' attribute added by Bootstrap CSS
            imageNode.css('max-width', 'none');
            width = imageNode[0].naturalWidth || 0;
            height = imageNode[0].naturalHeight || 0;
        }

        // naturalWidth/naturalHeight may not work for <img> elements linking to SVG
        if ((width === 0) || (height === 0)) {

            // insert the image node into a temporary unrestricted node and get its size
            tempNode.append(imageNode).appendTo('body');
            width = imageNode.width();
            height = imageNode.height();

            // move image node back to its parent
            parentNode.append(imageNode);
            tempNode.remove();
        }

        // zoom correction
        if (_.isNumber(zoom) && (zoom > 0)) {
            width /= zoom;
            height /= zoom;
        }

        return ((width > 0) && (height > 0)) ? { width: width, height: height } : null;
    }

    // class PageLoader =======================================================

    /**
     * The queued page loader sending server requests for pages to be rendered.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends TimerMixin
     */
    function PageLoader(app) {

        var // self reference
            self = this,
            // list of pages waiting to load their page contents, keyed by priority
            queue = { high: [], medium: [], low: [] },

            // waiting and running requests, keyed by page number
            map = {},

            // the background loop processing the pending pages
            timer = null,

            pdfView = null;

        // base constructors --------------------------------------------------

        BaseObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Loads the specified page into the passed DOM node, after clearing
         * all its old contents.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object waiting for the image data. Will
         *  be resolved with the original size of the page (as object with the
         *  properties 'width' and 'height', in pixels).
         */
        function loadPageIntoNode(pageNode, pageNumber, options) {

            var // the Deferred object waiting for the image
                def = null,
                // the jquery node
                jqPageNode = $(pageNode),
                // the target image format
                format = Utils.getStringOption(options, 'format', 'png');
                // default options contain a page zoom of 1.0
                options = $.extend({pageNumber: pageNumber, pageZoom: 1.0, }, options || {});

            function resolveSize() {
                var pageSize = calculateImageSize(jqPageNode.children().first(), Utils.getNumberOption(options, 'zoom'));

                if (pageSize) {
                    jqPageNode.data('page-size', pageSize);
                }

                return pageSize || $.Deferred().reject();
            }

            // -----------------------------------------------------------------

            if (format === 'pdf') {
                jqPageNode.data('data-rendertype', 'pdf');
                jqPageNode.data('data-pagezoom', options.pageZoom);

                var pageSize = pdfView.createPDFPageNode(jqPageNode, options);

                if (pageSize) {
                    (def = $.Deferred()).resolve(pageSize);
                } else {
                    (def = $.Deferred()).reject();
                }
            } else if (_.browser.Chrome && (format === 'svg')) {
                jqPageNode.data('data-rendertype', 'svg');

                // as SVG mark-up (Chrome does not show embedded images in <img> elements linked to an SVG file)
                def = app.getModel().loadPageAsSvgMarkup(pageNumber, Utils.getStringOption(options, 'priority', 'medium')).then(function (svgMarkup) {
                    // do NOT use the jQuery.html() method for SVG mark-up!
                    jqPageNode[0].innerHTML = svgMarkup;
                    // resolve with original image size
                    return resolveSize();
                });
            } else {
                jqPageNode.data('data-rendertype', 'img');

                // Bug 25765: Safari cannot parse SVG mark-up, and cannot show images embedded in <img> elements linking to SVG
                if (_.browser.Safari && (format === 'svg')) {
                    options = Utils.extendOptions(options, { format: 'jpg', zoom: Utils.RETINA ? 2 : 1 });
                }

                // preferred: as an image element linking to the image file
                def = app.getModel().loadPageAsImage(pageNumber, options).then(function (imgNode) {
                    jqPageNode.empty().append(imgNode);
                    // resolve with original image size (naturalWidth/naturalHeight with SVG does not work in IE10)
                    return resolveSize();
                });
            }

            return def.promise();
        }

        /**
         * Registers a page node for deferred loading.
         */
        function registerPageNode(pageNode, pageNumber, options) {

            var // the pending data to be inserted into the array
                pageData = null,
                // the jquery node
                jqPageNode = $(pageNode),
                // the request priority
                priority = Utils.getStringOption(options, 'priority', 'medium');

            // check if the page has been registered already
            if (pageNumber in map) {
                return map[pageNumber].def.promise();
            }

            // insert a busy node into the page node
            jqPageNode.empty().append($('<div>').addClass('abs').busy());

            // insert the page information into the pending array
            pageData = { node: jqPageNode, page: pageNumber, options: options, def: $.Deferred() };
            queue[priority].push(pageData);
            map[pageNumber] = pageData;

            return pageData.def.promise();
        }

        /**
         * Loads all queued pages in a background loop.
         */
        function loadQueuedPages() {

            if (!pdfView) {
                pdfView = new PDFView(app.getModel().getPDFDocument());
            }

            // check if the background loop is already running
            if (timer) {
                return;
            }

            // create a new background loop that processes all waiting pages
            timer = self.repeatDelayed(function () {

                var // data of next page to be loaded
                    pageData = null;

                // restrict number of requests running simultaneously (but do not break the loop);
                // load high-priority page even if limit has been reached but not exceeded
                if ((runningRequests > MAX_REQUESTS) || ((runningRequests === MAX_REQUESTS) && (queue.high.length === 0))) {
                    return;
                }

                // abort the background loop, if no more pages have to be loaded
                if (!(pageData = queue.high.shift() || queue.medium.shift() || queue.low.shift())) {
                    return Utils.BREAK;
                }

                // load the page contents
                runningRequests += 1;
                loadPageIntoNode(pageData.node, pageData.page, pageData.options)
                .always(function () {
                    runningRequests -= 1;
                    delete map[pageData.page];
                })
                .done(function (pageSize) {
                    pageData.node.removeClass('page-error');
                    pageData.def.resolve(pageSize);
                })
                .fail(function (result) {
                    pageData.node.addClass('page-error').css({ visibility: 'visible' }).html('<div class="error-icon">' + Forms.createIconMarkup('fa-times') + '</div>');
                    pageData.def.reject(result);
                });

            }, 20);

            // forget reference to the timer, when all page requests are running
            timer.always(function () {
                timer = null;
            });
        }

        /**
         * Renders the PDF page with given page number and zoom
         * into the given canvas node, that already has the
         * appropriate size and style size attributes set
         * @param {jQuery} canvasNode
         *  The target canvas node, as jQuery object.
         *
         * @param {Number} page
         *  The one-based index of the page to be rendered.
         *
         * @param {Number} pageZoom
         *  The zoom of the page for rendering.
         *
         * @returns {jquery promise}
         *  The jQuery promise that will be resolved when the
         *  page and the page text have been rendered successfully
         *  or rejected on error.
         */

        // methods ------------------------------------------------------------

        /**
         * Aborts all queued but not yet running AJAX page requests. The
         * associated Deferred objects of the page requests (returned by the
         * method PageLoader.loadPage()) will neither be resolved nor rejected.
         * Page requests already running will be finished though.
         *
         * @returns {PageLoader}
         *  A reference to this instance.
         */
        this.abortQueuedRequests = function () {
            _.each(queue, function (array, priority) {
                _.each(array, function (pageData) {
                    pageData.node.empty();
                    delete map[pageData.page];
                });
                queue[priority] = [];
            });
            return this;
        };

        /**
         * Loads the page contents into the passed page node in a background
         * task. The number of AJAX page requests running simultaneously is
         * globally restricted.
         *
         * @param {jQuery} pageNode
         *  The target page node, as jQuery object.
         *
         * @param {Number} pageNumber
         *  The one-based index of the page to be loaded.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.format='png']
         *      The image format. Supported values are 'jpg', 'png', and 'svg'.
         *  @param {Number} [options.width]
         *      If specified, the requested width of the image, in pixels. If
         *      the option 'options.height' is specified too, the resulting
         *      width may be less than this value.
         *  @param {Number} [options.height]
         *      If specified, the requested height of the image, in pixels. If
         *      the option 'options.width' is specified too, the resulting
         *      height may be less than this value.
         *  @param {Boolean} [options.printing=false]
         *      Specifies if the page is to be rendered for printing or just viewing
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  page has been loaded successfully; or rejected on error.
         */
        this.loadPage = this.createDebouncedMethod(registerPageNode, loadQueuedPages);

        /**
         * Returns the current zoom factor of the passed page node. The zoom
         * factor will be stored in the page node after the page has been
         * loaded and rendered. If the passed page has not been loaded yet,
         * returns the default zoom factor of 1.
         *
         * @returns {Number}
         *  The current zoom factor.
         */
        this.getPageZoom = function (pageNode) {
            return $(pageNode).data('page-zoom') || 1;
        };

        /**
         * Recalculates the size of the passed page node, according to the
         * original page size and zoom factor.
         *
         * @param {HTMLElement|jQuery} pageNode
         *  The page node containing the SVG contents.
         *
         * @param {Number} pageZoom
         *  The new zoom factor, as floating point number (the value 1
         *  represents the original page size).
         *
         * @returns {PageLoader}
         *  A reference to this instance.
         */
        this.setPageZoom = function (pageNode, pageZoom) {
            var // the page number
                pageNumber = Utils.getElementAttributeAsInteger(pageNode, 'data-page', 1),
                // the jquery node
                jqPageNode = $(pageNode),
                // the original size of the page
                pageSize = app.getModel().getOriginalPageSize(pageNumber),
                // the type od node to be rendered ('pdf', 'svg' or 'img)
                renderType = jqPageNode.data('data-rendertype'),
                // the resulting width/height
                width = Math.floor(pageSize.width * pageZoom),
                height = Math.ceil(pageSize.height * pageZoom);

            // make node visible, it may be invisible after initial loading
            jqPageNode.css({visibility: 'visible'});

            if (renderType === 'pdf') {
                // <canvas> element: render page into canvas and create textoverlay
                pdfView.renderPDFPage(jqPageNode, Utils.getElementAttributeAsInteger(jqPageNode, 'data-page', 1), pageZoom);
            } else if (renderType === 'img') {
                // <img> element: resize with CSS width/height
                jqPageNode.children().first().width(width).height(height);
            } else {
                // <svg> element (Chrome): scale with CSS zoom (supported in WebKit)
                jqPageNode.children().first().css('zoom', pageZoom);
            }

            // Chrome bug/problem: sometimes, the page node has width 0 (e.g., if browser zoom is
            // not 100%) regardless of existing SVG, must set its size explicitly to see anything...
            jqPageNode.width(width).height(height).data('page-zoom', pageZoom);

            return this;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = queue = map = timer = null;
        });

    } // class PageLoader

    // exports ================================================================

    // derive this class from class BaseObject
    return BaseObject.extend({ constructor: PageLoader });

});
