/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/editframework/view/hyperlinkutil', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/dialogs',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/editframework/view/editdialogs',
    'io.ox/office/tk/keycodes',
    'gettext!io.ox/office/text'
], function (Utils, Forms, Dialogs, HyperlinkUtils, EditDialogs, KeyCodes, gt) {

    'use strict';

    // static class HyperlinkUtil =============================================

    var HyperlinkUtil = {};

    // public methods ---------------------------------------------------------

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {String} text
     *  The optional text which represents the URL
     *
     * @param {String} url
     *  An optional URL which is set for the supplied text
     *
     * @param {BaseApplication} app
     *  The calling app
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button has been activated, or rejected if the dialog has been canceled.
     *  The done handlers registered at the promise object will receive the
     *  entered text.
     */
    HyperlinkUtil.showDialog = function (text, url, app) {
        return HyperlinkUtil.showHyperlinkDialog(app, { valueURL: url, valueText: text });
    };

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {BaseApplication} app
     *  The calling application.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.valueText='']
     *      The initial value of the text field.
     *  @param {String} [options.valueURL='']
     *      The initial value of the URL field.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button or the remove button have been activated, or rejected if the
     *  dialog has been canceled. The done handlers registered at the promise
     *  object will receive a object containing the text and URL entered by the
     *  user. The object contains null for text and the URL if remove has been
     *  clicked.
     */
    HyperlinkUtil.showHyperlinkDialog = function (app, options) {

        var // the document model
            model = app.getModel(),

            // the input field for the display text of the URL
            textValue = Utils.getStringOption(options, 'valueText', ''),
            textLabelId = _.uniqueId('text'),
            textInputId = _.uniqueId('text'),
            textInput = $('<input>', {
                value: textValue,
                tabindex: 0,
                id: textInputId,
                role: 'textbox',
                'aria-labelledby': textLabelId
            }).addClass('form-control'),

             // the input field for the URL
            urlValue = Utils.getStringOption(options, 'valueURL', ''),
            urlLabelId = _.uniqueId('url'),
            urlInputId = _.uniqueId('url'),
            urlInput = $('<input>', {
                value: urlValue,
                tabindex: 0,
                id: urlInputId,
                role: 'textbox',
                'aria-labelledby': urlLabelId
            }).addClass('form-control'),

            // the dialog instance
            title = (urlValue.length === 0) ? gt('Insert Hyperlink') : gt('Edit Hyperlink'),
            dialog = new Dialogs.ModalDialog({ title: title, okLabel: gt('Insert') }),

            // the resulting promise
            promise = null;

        // close dialog automatically after losing edit rights
        app.getView().closeDialogOnReadOnlyMode(dialog);

        // add the input controls to the dialog body
        dialog.append(
            $('<div>').addClass('form-group').append(
                $('<label>', { 'for': textInputId, id: textLabelId }).text(/*#. "Insert URL" dialog: The display text of the URL */ gt('Text')),
                textInput
            ),
            $('<div>').addClass('form-group').append(
                $('<label>', { 'for': urlInputId, id: urlLabelId }).text(/*#. "Insert URL" dialog: The URL itself */ gt('URL')),
                urlInput
            )
        );

        // only add remove button if we already have a set URL
        if (Utils.getStringOption(options, 'valueURL', '').length > 0) {
            dialog.addAlternativeButton('remove', gt('Remove'), 'remove', { tabIndex: 1 });
            dialog.getButton('remove').addClass('btn-warning');
        }

        function checkTextFields() {
            var textUrl = urlInput.val(),
                text = textInput.val();

            if (!model.getEditMode()) {
                // disable ok and delete button if we don't have edit rights
                dialog.enableOkButton(false).enableButton('remove', false);
            } else if ((text && text.length > 0) && (textUrl && textUrl.length > 0)) {
                dialog.enableOkButton(true);
            } else {
                dialog.enableOkButton(false);
            }
        }

        // register a change handlers to toggle the insert button
        urlInput.on('input', checkTextFields);
        textInput.on('input', checkTextFields);

        // show the dialog
        promise = dialog.show(function () {
            var hasText = textInput.val().length > 0,
                inputNode = hasText ? urlInput : textInput;

            // IE needs to wait until the 'Dialog' is fade in
            if (_.browser.IE) {
            	app.executeDelayed(function () { inputNode.focus(); }, 100);
            } else {
                inputNode.focus();
            }
            checkTextFields();
        });

        // process the result of the dialog
        promise = promise.then(function (action) {
            switch (action) {
            case 'ok':
                var text = textInput.val();
                var url = Utils.trimString(urlInput.val());
                // add http: as default if protocol is missing
                var protIndex = url.indexOf(':');
                if (protIndex === -1) {
                    url = 'http://' + url;
                } else if (!HyperlinkUtils.hasSupportedProtocol(url)) {
                    Utils.warn('you tried to add illegal code ' + url);
                    app.getView().yell({ type: 'warning', message: gt('This hyperlink is invalid.') });
                    url = null;
                }
                return { text: text, url: url };
            case 'remove':
                return { text: null, url: null };
            }
            return $.Deferred().reject();
        });

        return promise.always(function () {
            model.off('change:editmode', checkTextFields);
        });
    };

    /**
     * create a jQuery Popup node with link, edit and remove function
     *
     * @param {BaseApplication} app
     *  The application is necessary for the listeners of edit and remove
     *  (character/hyperlink/dialog and character/hyperlink/remove)
     *
     *
     * @returns {jQuery Element}
     */

    HyperlinkUtil.createPopupNode = function (app) {
        var hyperlinkPopup = $('<div>', { contenteditable: false })
            .addClass('inline-popup hyperlink f6-target').css('display', 'none')
            .append(
                $('<a>').addClass('textwrap').attr({ href: '', rel: 'noreferrer', target: '_blank', tabindex: 0 }),
                $('<br/>'),
                $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Edit')).click(function () { app.getController().executeItem('character/hyperlink/dialog'); return false; }),
                $('<span>').text(' | '),
                $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Remove')).click(function () { app.getController().executeItem('character/hyperlink/remove'); return false; }));

        if (Modernizr.touch) {
            // some CSS changes for touch devices
            hyperlinkPopup.addClass('touch');
            hyperlinkPopup.find('a.popupbutton').addClass('touch');
        }

        hyperlinkPopup.on('keydown', { }, HyperlinkUtil.popupProcessKeydown);

        return hyperlinkPopup;

    };

    HyperlinkUtil.updatePopup = function (url, popup) {
        var link = popup.find('a.textwrap');
        //var link = $('a.textwrap', hyperlinkPopup[0]);
        link.text(url ? limitHyperlinkText(url) : '');
        if (!url || url.charAt(0) === '#') {
            link.attr({
                title: ''
            });
            link.css({
                color: '',
                cursor: 'default',
                textDecoration: 'none'
            });
            link.removeAttr('href');
        } else if (HyperlinkUtils.hasSupportedProtocol(url)) {
            link.css({
                color: '',
                cursor: '',
                textDecoration: ''
            });
            link.attr({
                href: url,
                title: url
            });
        } else {
            link.attr({
                title: gt('Script code is not allowed')
            });
            link.css({
                color: 'red',
                cursor: 'default',
                textDecoration: 'none'
            });
            link.removeAttr('href');
        }
    };

    /**
     * Process keyboard events used to travel in a container and sets the
     * focus to the next/previous element.
     *
     * The function expects that the element which gets the focus on ESCAPE to
     * be in the [data.nextFocus] part of the event. Therefore this function
     * must be set using .on('keydown', {nextFocus: focus}, popupProcessKeydown)
     *
     * @param {Object} event
     *  A keydown event sent by the browser.
     *
     * @returns {Boolean}
     *  Returns false if the browser shouldn't process the keyboard
     *  event.
     */
    HyperlinkUtil.popupProcessKeydown = function (event) {
        var items, focus, index, down;

        if (event.keyCode === KeyCodes.TAB || event.keyCode === KeyCodes.DOWN_ARROW || event.keyCode === KeyCodes.RIGHT_ARROW || event.keyCode === KeyCodes.LEFT_ARROW || event.keyCode === KeyCodes.UP_ARROW) {
            event.preventDefault();

            items = Forms.findFocusableNodes(this);
            focus = $(document.activeElement);
            down = (((event.keyCode === KeyCodes.TAB) && !event.shiftKey) ||
                    (event.keyCode === KeyCodes.DOWN_ARROW) ||
                    (event.keyCode === KeyCodes.RIGHT_ARROW));

            index = (items.index(focus) || 0);
            if (!down) {
                index--;
            } else {
                index++;
            }

            if (index >= items.length) {
                index = 0;
            } else if (index < 0) {
                index = items.length - 1;
            }
            items[index].focus();
            event.preventDefault();
            return false;
        } else if (event.keyCode === KeyCodes.ESCAPE) {
            $(this).hide();
            if (event.data && event.data.nextFocus && event.data.nextFocus.length) {
                $(event.data.nextFocus).focus();
                event.preventDefault();
                return false;
            }
        } else if(event.keyCode === KeyCodes.SPACE || event.keyCode === KeyCodes.ENTER) {
        	if (event.target) {
        		event.target.click();
        		return false;
        	}
        }
    };



    /**
     * adds listener for the popupNode for changing the Editmode
     *
     * @param {SpreadsheetModel} model
     *  The application is necessary for the listeners of edit and remove
     *  (character/hyperlink/dialog and character/hyperlink/remove)
     *
     * @param {jQuery Element} popupNode
     *  @see HyperlinkUtil.createPopupNode
     *
     * @param {Function} listener
     *  an extra listener when Editmode is changed
     */

    HyperlinkUtil.addEditModeListener = function (model, popupNode, listener) {
        var updateEditMode = function () {
            var readOnly = model.getEditMode() !== true;
            popupNode.find('br').toggle(!readOnly);
            popupNode.find('span').toggle(!readOnly);
            popupNode.find('a.popupbutton').toggle(!readOnly);
            listener();
        };
        model.on('change:editmode', updateEditMode);


        // must be called to set the initial state correctly
        updateEditMode();
    };

    /**
     * Shortens a hyperlink to a maximal number of
     * characters.
     *
     * @param text {String} Hyperlink to shorten
     */
    function limitHyperlinkText(text) {
        return (text.length < 256) ? text : (text.slice(0, 128) + Utils.ELLIPSIS_CHAR + text.slice(-128));
    }


    /**
     * checks if popupnode is already created in given parent
     *
     * @param {jQuery Element} parent
     *
     */

    HyperlinkUtil.noPopup = function (parent) {
        if (parent.children('.inline-popup.hyperlink').length === 0) {
            return true;
        }
    };

    // exports ================================================================

    return HyperlinkUtil;

});
