/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/filenamefield', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/textfield',
    'gettext!io.ox/office/editframework'
], function (Utils, Forms, TextField, gt) {

    'use strict';

    // class FileNameField ====================================================

    /**
     * A text field used to edit the file name.
     *
     * @constructor
     *
     * @extends TextField
     */
    function FileNameField(app) {

        var // self reference
            self = this,

            // a special busy spinner visible while committing the file name
            busyOverlayNode = $('<div class="busy-overlay">').hide().busy();

        // base constructor ---------------------------------------------------

        TextField.call(this, {
            classes: 'file-name',
            tooltip: gt('Rename document'),
            placeholder: gt('Document name'),
            width: 230,
            select: true,
            smallerVersion: { css: { width: 100 } }
        });

        // private methods ----------------------------------------------------

        /**
         * Sets a predefined tool tip in edit mode, or the document name in
         * read-only mode.
         */
        function updateToolTip() {
            Forms.setToolTip(self.getNode(), app.getModel().getEditMode() ? gt('Rename document') : _.noI18n(app.getFullFileName()));
        }

        // initialization -----------------------------------------------------

        // show a busy spinner as long as the control commits the value
        // (the time between change event and losing the focus)
        this.getInputNode().after(busyOverlayNode);
        this.on('group:change', function () { busyOverlayNode.show(); });
        this.on('group:focus group:blur', function () { busyOverlayNode.hide(); });

        // set document name as tool tip in read-only mode
        app.onInit(function () {
            app.getModel().on('change:editmode', updateToolTip);
            updateToolTip();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = busyOverlayNode = null;
        });

    } // class FileNameField

    // exports ================================================================

    // derive this class from class TextField
    return TextField.extend({ constructor: FileNameField });

});
