/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/border',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/utils/color'
    ], function (Utils, Color) {

    'use strict';

    var // map operation line styles to CSS line styles (anything else maps to 'solid')
        CSS_LINE_STYLES = {
            none: 'none',
            double: 'double',
            triple: 'double',
            dashed: 'dashed',
            dashSmallGap: 'dashed',
            dotted: 'dotted',
            dotDash: 'dotted',
            dotDotDash: 'dotted',
            dashDotStroked: 'dotted'
        },

        // maximum line width for borders in GUI preview elements, in pixels
        MAX_PREVIEW_LINEWIDTH = 2,

        // minimum line width for borders with style 'double', in pixels
        MIN_DOUBLE_LINEWIDTH = 3;

    // static class Border ====================================================

    /**
     * Predefined border objects.
     */
    var Border = { NONE: { style: 'none' } };

    /**
     * Returns the best CSS border style that can represent the passed line
     * style used in border attributes.
     *
     * @param {String} style
     *  The border line style, as used in formatting attributes of document
     *  operations.
     *
     * @returns {String}
     *  The best matching CSS border style.
     */
    Border.getCssBorderStyle = function (style) {
        return (style in CSS_LINE_STYLES) ? CSS_LINE_STYLES[style] : 'solid';
    };

    /**
     * Returns the border dash pattern for the passed line style and width,
     * intended to be used in a canvas renderer (see class CanvasWrapper).
     *
     * @param {String} style
     *  The border line style, as used in formatting attributes of document
     *  operations.
     *
     * @param {Number} width
     *  The line width, in pixels.
     *
     * @returns {Array|Null}
     *  The dash pattern for the passed line style, or null for solid lines.
     */
    Border.getBorderPattern = (function () {

        var // maps border attribute styles to dash patterns
            BORDER_PATTERNS = {
                dashed: [6, 2],
                dotted: [2, 2],
                dashDot: [6, 2, 2, 2],
                dashDotDot: [6, 2, 2, 2, 2, 2]
            },

            // cache for scaled border patterns, mapped by line width
            borderPatternCache = { 1: BORDER_PATTERNS };

        // the actual getBorderPattern() method to be returned from the local scope
        function getBorderPattern(style, width) {

            var // the pattern map for the passed line style
                patternMap = null;

            // get pattern map for integral line width
            width = Math.floor(width);
            patternMap = borderPatternCache[width];

            // build missing pattern map for new line width
            if (!patternMap) {
                borderPatternCache[width] = patternMap = {};
                _.each(BORDER_PATTERNS, function (pattern, key) {
                    patternMap[key] = _.map(pattern, function (length) {
                        return Math.floor(length * (width / 2 + 0.5));
                    });
                });
            }

            // return pattern for passed line style
            return patternMap[style] || null;
        }

        return getBorderPattern;
    }());

    /**
     * Returns the effective CSS attributes for the passed border value.
     *
     * @param {Object} border
     *  The border object as used in operations.
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.preview=false]
     *      If set to true, the border will be rendered for a preview element
     *      in the GUI. The border width will be restricted to two pixels in
     *      that case.
     *  @param {Object} [options.autoColor]
     *      If specified, a replacement color to be used in case the border
     *      object contains the automatic color.
     *  @param {Boolean} [options.transparentHair=false]
     *      If set to true, border lines with a resulting width less than one
     *      pixel (hair lines) will be drawn half-transparent.
     *
     * @returns {Object}
     *  A map with CSS border attributes. Contains the property 'style' with
     *  the effective CSS border style as string; the property 'width' with the
     *  effective border width in pixels (as number, rounded to entire pixels);
     *  and the property 'color' with the effective CSS color (as string with
     *  leading hash sign).
     */
    Border.getCssBorderAttributes = function (border, theme, options) {

        var style = Utils.getStringOption(border, 'style', 'none'),
            width = Utils.getIntegerOption(border, 'width', 0),
            color = Utils.getObjectOption(border, 'color', Color.AUTO),
            preview = Utils.getBooleanOption(options, 'preview', false),
            autoColor = Utils.getObjectOption(options, 'autoColor'),
            colorOptions = null;

        // convert operation line styles to CSS styles
        style = Border.getCssBorderStyle(style);

        // convert 1/100mm to pixels (round to one digit)
        width = (style === 'none') ? 0 : Utils.convertHmmToLength(width, 'px', 0.1);

        // minimum width for borders with style 'double', maximum width for preview borders
        if ((style === 'double') && (width > 0)) {
            width = preview ? MIN_DOUBLE_LINEWIDTH : Math.max(width, MIN_DOUBLE_LINEWIDTH);
        } else if (preview) {
            width = Math.min(width, MAX_PREVIEW_LINEWIDTH);
        }

        // further processing of the line width
        if (width > 0) {

            // add color transparency for hair lines
            if ((width < 1) && Utils.getBooleanOption(options, 'transparentHair', false)) {
                colorOptions = { alphaMod: 0.5 };
            }

            // round to entire pixels
            width = Math.max(Math.round(width), 1);
        }

        // convert color object to CSS color
        color = Color.getCssColor(color, autoColor || 'line', theme, colorOptions);

        return { style: style, width: width, color: color };
    };

    /**
     * Converts the passed border attribute object to a CSS border value.
     *
     * @param {Object} border
     *  The border object as used in operations.
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Border.getCssBorderAttributes(). Additionally, the following options
     *  are supported:
     *  @param {Boolean} [options.clearNone=false]
     *      If set to true, the return value for invisible borders will be the
     *      empty string instead of the keyword 'none'.
     *
     * @returns {String}
     *  The CSS border value converted from the passed border object.
     */
    Border.getCssBorder = function (border, theme, options) {
        var visible = Border.isVisibleBorder(border),
            attributes = visible ? Border.getCssBorderAttributes(border, theme, options) : null;
        return visible ? (attributes.style + ' ' + attributes.width + 'px ' + attributes.color) :
            Utils.getBooleanOption(options, 'clearNone', false) ? '' : 'none';
    };

    /**
     * Returns whether the passed border object is a valid and visible border
     * style (line style different to 'none', and line width greater than 0).
     *
     * @param {Object} border
     *  The border object as used in operations.
     */
    Border.isVisibleBorder = function (border) {
        return _.isObject(border) && _.isString(border.style) && (border.style !== 'none') && _.isNumber(border.width) && (border.width > 0);
    };

    /**
     * Comparing two borders. These are equal if both borders have the same style, width and color
     *
     * @param {Border} border1
     * The first border to compare
     *
     * @param {Border} border2
     * The second border to compare
     *
     * @returns {Boolean}
     * Whether the two borders are equal.
     */
    Border.isEqual = function (border1, border2) {
        return (border1 &&
                border2 &&
                ((!('style' in border1) && (!('style' in border2))) || (('style' in border1) && ('style' in border2) && (border1.style === border2.style))) &&
                ((!('width' in border1) && (!('width' in border2))) || (('width' in border1) && ('width' in border2) && (border1.width === border2.width))) &&
                ((!('color' in border1) && (!('color' in border2))) || (('color' in border1) && ('color' in border2) && (Color.isEqual(border1.color, border2.color)))));
    };

    // exports ================================================================

    return Border;

});
