/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/popup/drawingframemenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/popup/layermenu',
    'io.ox/office/drawinglayer/view/drawingframe'
], function (Utils, LayerMenu, DrawingFrame) {

    'use strict';

    // class DrawingFrameMenu =================================================

    /**
     * A floating menu attached to the frame node of a specific drawing object.
     *
     * @constructor
     *
     * @extends LayerMenu
     *
     * @param {EditApplication} app
     *  The application containing this menu instance.
     *
     * @param {String} drawingType
     *  The types of drawing objects supported by this menu, as space-separated
     *  list.
     */
    function DrawingFrameMenu(app, drawingType, titleLabel, initOptions) {

        var // self reference
            self = this,

            // the model and view instance of the document
            docModel = app.getModel(),
            docView = app.getView(),

            // the current drawing frame used as menu anchor
            drawingFrame = null;

        // base constructor ---------------------------------------------------

        LayerMenu.call(this, app, titleLabel, Utils.extendOptions({
            anchorBorder: 'right left'
        }, initOptions, {
            anchor: function () { return drawingFrame; },
            anchorPadding: 6,
            coverAnchor: true
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the position of the menu, after the drawing selection has
         * changed. If no matching drawing object is selected, the menu will be
         * hidden.
         */
        function renderDrawingSelectionHandler(event, drawingFrames) {

            // do nothing in read-only mode
            if (!docModel.getEditMode()) { return; }

            // hide menu if selection is anything but a single drawing object of the specified type (TODO: support multi-selections?)
            if ((drawingFrames.length === 1) && _.contains(drawingType, DrawingFrame.getDrawingType(drawingFrames[0]))) {
                drawingFrame = drawingFrames[0];
                self.refreshImmediately();
            } else {
                drawingFrame = null;
                self.hide();
            }
        }

        // initialization -----------------------------------------------------

        // convert space-separated list of drawing object types to array
        drawingType = drawingType.split(/\s+/);

        // initialize class members when application is ready
        app.onInit(function () {
            docModel = app.getModel(),
            docView = app.getView();

            // hide the menu automatically in read-only mode
            self.listenTo(docModel, 'change:editmode', function (event, editMode) {
                if (!editMode) { self.hide(); }
            });

            // adjust position and visibility of the menu according to the current selection
            self.listenTo(docView, 'render:drawingselection', renderDrawingSelectionHandler);
        });

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = self = docModel = docView = null;
        });

    } // class DrawingFrameMenu

    // exports ================================================================

    // derive this class from class LayerMenu
    return LayerMenu.extend({ constructor: DrawingFrameMenu });

});
