/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawinglabels', [
    'gettext!io.ox/office/drawing'
], function (gt) {

    'use strict';

    var // maps drawing object types as defined/used in operations to GUI names and icons
        DRAWING_TYPE_INFOS = {
            chart:     { icon: 'fa-bar-chart-o',        label: /*#. bar charts, line charts, pie charts, etc. */ gt.pgettext('drawing', 'Chart') },
            image:     { icon: 'fa-picture-o',          label: /*#. bitmaps, vector graphics, etc. */ gt.pgettext('drawing', 'Image') },
            shape:     { icon: 'fa-picture-o',          label: /*#. rectangles, circles, lines, stars, etc. */ gt.pgettext('drawing', 'Shape') },
            textframe: { icon: 'docs-insert-textframe', label: /*#. rectangles, circles, etc. with arbitrary text contents */ gt.pgettext('drawing', 'Text frame') },
            diagram:   { icon: 'fa-sitemap',            label: /*#. complex diagrams, organigrams, etc. */ gt.pgettext('drawing', 'Diagram') }
        },

        // fall-back settings for unknown/unsupported drawing objects
        DRAWING_TYPE_DEFAULT_INFO = { icon: 'fa-picture-o', label: gt.pgettext('drawing', 'Drawing') },

        // chart type descriptors for all supported chart type
        CHART_TYPE_STYLES = {
            'column standard':         { cat: 1, title: /*#. Chart type: vertical bars (clustered) */                                                                     gt.pgettext('chart-type', 'Column'),           type: 'column2d',  stacking: 'standard',       cjs: 'column' },
            'column stacked':          { cat: 1, title: /*#. Chart type: vertical bars (stacked from bottom to top) */                                                    gt.pgettext('chart-type', 'Column (stacked)'), type: 'column2d',  stacking: 'stacked',        cjs: 'stackedColumn' },
            'column percentStacked':   { cat: 1, title: /*#. Chart type: vertical bars (stacked from bottom to top with percent scaling) */                               gt.pgettext('chart-type', 'Column (percent)'), type: 'column2d',  stacking: 'percentStacked', cjs: 'stackedColumn100' },
            'bar standard':            { cat: 1, title: /*#. Chart type: horizontal bars (clustered) */                                                                   gt.pgettext('chart-type', 'Bar'),              type: 'bar2d',     stacking: 'standard',       cjs: 'bar' },
            'bar stacked':             { cat: 1, title: /*#. Chart type: horizontal bars (stacked from left to right) */                                                  gt.pgettext('chart-type', 'Bar (stacked)'),    type: 'bar2d',     stacking: 'stacked',        cjs: 'stackedBar' },
            'bar percentStacked':      { cat: 1, title: /*#. Chart type: horizontal bars (stacked from left to right with percent scaling) */                             gt.pgettext('chart-type', 'Bar (percent)'),    type: 'bar2d',     stacking: 'percentStacked', cjs: 'stackedBar100' },
            'line standard':           { cat: 2, title: /*#. Chart type: data points connected with lines */                                                              gt.pgettext('chart-type', 'Line'),             type: 'line2d',    stacking: 'standard',       cjs: 'line' },
            'line standard curved':    { cat: 2, title: /*#. Chart type: data points connected with curved lines */                                                       gt.pgettext('chart-type', 'Line (curved)'),    type: 'line2d',    stacking: 'standard',       cjs: 'spline', curved: true },
            'scatter standard':        { cat: 2, title: /*#. Chart type: data points with free X/Y coordinates connected with lines */                                    gt.pgettext('chart-type', 'Scatter'),          type: 'scatter2d', stacking: 'standard',       cjs: 'line' },
            'scatter standard curved': { cat: 2, title: /*#. Chart type: data points with free X/Y coordinates connected with curved lines */                             gt.pgettext('chart-type', 'Scatter (curved)'), type: 'scatter2d', stacking: 'standard',       cjs: 'spline', curved: true },
            'bubble standard':         { cat: 3, title: /*#. Chart type: data points with free X/Y coordinates drawn as circles/bubbles */                                gt.pgettext('chart-type', 'Bubble'),           type: 'bubble2d',  stacking: 'standard',       cjs: 'bubble' },
            'pie standard':            { cat: 3, title: /*#. Chart type: single pie */                                                                                    gt.pgettext('chart-type', 'Pie'),              type: 'pie2d',     stacking: 'standard',       cjs: 'pie', varyColors: true },
            'donut standard':          { cat: 3, title: /*#. Chart type: one or multiple concentric circles */                                                            gt.pgettext('chart-type', 'Donut'),            type: 'donut2d',   stacking: 'standard',       cjs: 'doughnut', varyColors: true },
            'area standard':           { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (areas overlay each other) */                        gt.pgettext('chart-type', 'Area'),             type: 'area2d',    stacking: 'standard',       cjs: 'area' },
            'area stacked':            { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top) */                      gt.pgettext('chart-type', 'Area (stacked)'),   type: 'area2d',    stacking: 'stacked',        cjs: 'stackedArea' },
            'area percentStacked':     { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top with percent scaling) */ gt.pgettext('chart-type', 'Area (percent)'),   type: 'area2d',    stacking: 'percentStacked', cjs: 'stackedArea100' }
        };

    // set missing Boolean properties explicitly to false
    _.each(CHART_TYPE_STYLES, function(CHART_TYPE) {
        CHART_TYPE.curved = CHART_TYPE.curved || false;
        CHART_TYPE.varyColors = CHART_TYPE.varyColors || false;
    });

    // static class DrawingLabels =============================================

    /**
     * Provides generic caption texts for control groups, menu headers, and
     * other GUI elements in a single map for convenience.
     */
    var DrawingLabels = {};

    // drawing types ----------------------------------------------------------

    /**
     * Returns an appropriate CSS icon class for the passed drawing type.
     *
     * @param {String} type
     *  The type identifier of a drawing object.
     *
     * @returns {String}
     *  A CSS icon class for the passed drawing type.
     */
    DrawingLabels.getDrawingTypeIcon = function (type) {
        return (DRAWING_TYPE_INFOS[type] || DRAWING_TYPE_DEFAULT_INFO).icon;
    };

    /**
     * Returns an appropriate text label for the passed drawing type.
     *
     * @param {String} type
     *  The type identifier of a drawing object.
     *
     * @returns {String}
     *  A text label for the passed drawing type.
     */
    DrawingLabels.getDrawingTypeLabel = function (type) {
        return (DRAWING_TYPE_INFOS[type] || DRAWING_TYPE_DEFAULT_INFO).label;
    };

    // constants for controls -------------------------------------------------

    /**
     * Standard options for the 'Delete drawing' button.
     *
     * @constant
     */
    DrawingLabels.DELETE_DRAWING_BUTTON_OPTIONS = {
        icon: 'fa-trash-o',
        // label: /*#. delete a drawing object from the document */ gt.pgettext('drawing', 'Delete'),
        tooltip: gt.pgettext('drawing', 'Delete the drawing object')
    };

    /**
     * Standard options for the 'Insert image' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_IMAGE_BUTTON_OPTIONS = {
        icon: 'docs-image-insert',
        label: DrawingLabels.getDrawingTypeLabel('image'),
        tooltip: gt.pgettext('drawing', 'Insert an image'),
        smallerVersion: {
            css: { width: 35 },
            hideLabel: true
        }
    };

    /**
     * Standard options for the 'Insert image' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_TEXTFRAME_BUTTON_OPTIONS = {
        icon: DrawingLabels.getDrawingTypeIcon('textframe'),
        label: DrawingLabels.getDrawingTypeLabel('textframe'),
        tooltip: gt.pgettext('drawing', 'Insert a text frame'),
        smallerVersion: {
            css: { width: 35 },
            hideLabel: true
        }
    };

    /**
     * Standard options for the 'Insert chart' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_CHART_BUTTON_OPTIONS = {
        width: null,
        icon: DrawingLabels.getDrawingTypeIcon('chart'),
        label: DrawingLabels.getDrawingTypeLabel('chart'),
        tooltip: gt.pgettext('drawing', 'Insert a chart'),
        updateCaptionMode: 'none'
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_DATA_POINTS_BUTTON_OPTIONS = {
        label: //#. menu title: options to format data points (bars, lines, pie segments) in chart objects
            gt.pgettext('chart-format', 'Data points')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS = {
        label: //#. check box label: show/hide small text labels next to the single data points in charts objects
            gt.pgettext('chart-format', 'Show data point labels')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS = {
        label: //#. check box label: if active, a bar/pie chart will have different colors for each data point (default is the same color for all points)
            gt.pgettext('chart-format', 'Vary colors by point')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_LABELS_BUTTON_OPTIONS = {
        label: //#. button label to open a menu to modify chart objects
            gt.pgettext('chart-format', 'Labels'),
        tooltip: //#. tooltip: menu to modify chart objects
            gt.pgettext('chart-format', 'Settings for labels and axes'),
        toggle: true,
        haspopup: true
    };

    /**
     * A map with predefined border styles (line width, line style) supported by OOXML.
     *
     * @constant
     */
    DrawingLabels.BORDER_OOXML_PRESET_STYLES = [
        { value: 'none:none',         label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'No line'),                           icon: { style: 'none',       width: -1 } },
        { value: 'solid:hair',        label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'Hair line'),                         icon: { style: 'solid',      width: 0 } },
        { value: 'solid:thin',        label: /*#. border line style (width 1 pixel, solid) */ gt.pgettext('borders', 'Thin line'),                            icon: { style: 'solid',      width: 1 } },
        { value: 'dashed:thin',       label: /*#. border line style (width 1 pixel, dashed) */ gt.pgettext('borders', 'Dashed thin line'),                    icon: { style: 'dashed',     width: 1 } },
        { value: 'dotted:thin',       label: /*#. border line style (width 1 pixel, dotted) */ gt.pgettext('borders', 'Dotted thin line'),                    icon: { style: 'dotted',     width: 1 } },
        { value: 'dashDot:thin',      label: /*#. border line style (width 1 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash thin line'),            icon: { style: 'dashDot',    width: 1 } },
        { value: 'dashDotDot:thin',   label: /*#. border line style (width 1 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash thin line'),   icon: { style: 'dashDotDot', width: 1 } },
        { value: 'solid:medium',      label: /*#. border line style (width 2 pixels, solid) */ gt.pgettext('borders', 'Medium line'),                         icon: { style: 'solid',      width: 2 } },
        { value: 'dashed:medium',     label: /*#. border line style (width 2 pixels, dashed) */ gt.pgettext('borders', 'Dashed medium line'),                 icon: { style: 'dashed',     width: 2 } },
        { value: 'dotted:medium',     label: /*#. border line style (width 2 pixel, dotted) */ gt.pgettext('borders', 'Dotted medium line'),                  icon: { style: 'dotted',     width: 2 } },
        { value: 'dashDot:medium',    label: /*#. border line style (width 2 pixel, dash-dot) */ gt.pgettext('borders', 'Dot-and-dash medium line'),          icon: { style: 'dashDot',    width: 2 } },
        { value: 'dashDotDot:medium', label: /*#. border line style (width 2 pixel, dash-dot-dot) */ gt.pgettext('borders', 'Two-dots-and-dash medium line'), icon: { style: 'dashDotDot', width: 2 } },
        { value: 'solid:thick',       label: /*#. border line style (width 3 pixels, solid) */ gt.pgettext('borders', 'Thick line'),                          icon: { style: 'solid',      width: 3 } }
    ];

    /**
     * A map with predefined border styles (line width, line style) supported by ODF.
     *
     * @constant
     */
    DrawingLabels.BORDER_ODF_PRESET_STYLES = [
        { value: 'none:none',         label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'No line'),                           icon: { style: 'none',       width: -1 } },
        { value: 'solid:hair',        label: /*#. border line style (width 0.5 pixels, solid) */ gt.pgettext('borders', 'Hair line'),                         icon: { style: 'solid',      width: 0 } },
        { value: 'solid:thin',        label: /*#. border line style (width 1 pixel, solid) */ gt.pgettext('borders', 'Thin line'),                            icon: { style: 'solid',      width: 1 } },
        { value: 'dashed:thin',       label: /*#. border line style (width 1 pixel, dashed) */ gt.pgettext('borders', 'Dashed thin line'),                    icon: { style: 'dashed',     width: 1 } },
        { value: 'dotted:thin',       label: /*#. border line style (width 1 pixel, dotted) */ gt.pgettext('borders', 'Dotted thin line'),                    icon: { style: 'dotted',     width: 1 } },
        { value: 'solid:medium',      label: /*#. border line style (width 2 pixels, solid) */ gt.pgettext('borders', 'Medium line'),                         icon: { style: 'solid',      width: 2 } },
        { value: 'dashed:medium',     label: /*#. border line style (width 2 pixels, dashed) */ gt.pgettext('borders', 'Dashed medium line'),                 icon: { style: 'dashed',     width: 2 } },
        { value: 'dotted:medium',     label: /*#. border line style (width 2 pixel, dotted) */ gt.pgettext('borders', 'Dotted medium line'),                  icon: { style: 'dotted',     width: 2 } },
        { value: 'solid:thick',       label: /*#. border line style (width 3 pixels, solid) */ gt.pgettext('borders', 'Thick line'),                          icon: { style: 'solid',      width: 3 } }
    ];

    /**
     * A map with chart type descriptors for all supported chart type. Each
     * map element contains the following properties:
     * - {Number} cat
     *      The type category, used for grouping in the type picker control.
     * - {String} title
     *      The localized name of the chart type.
     * - {String} type
     *      The base chart type, as used in document operations.
     * - {String} stacking
     *      The stacking mode: 'standard', 'stacked', or 'percentStacked'.
     * - {String} cjs
     *      The chart type identifier for the CanvasJS library.
     * - {Boolean} curved
     *      Whether the lines in the chart area are smoothed.
     * - {Boolean} varyColors
     *      Whether the data points of a single series show different colors.
     */
    DrawingLabels.CHART_TYPE_STYLES = CHART_TYPE_STYLES;

    // exports ================================================================

    return DrawingLabels;

});
