/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/chartformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'gettext!io.ox/office/drawing'
], function (Utils, ChartStyleUtil, gt) {

    'use strict';

    //'io.ox/office/spreadsheet/model/numberformatter'
    // null date (corresponding to cell value zero) TODO: use null date from document settings

    // class ChartFormatter =====================================================


    function ChartFormatter(chart) {

        //look at 'io.ox/office/spreadsheet/model/numberformatter'
        function getExponent(nr) {
            var exp = Math.floor(Math.log(Math.abs(nr)) / Math.LN10);
            var min = 6;
            if (exp>-min && exp<min) {
                exp = 0;
            }
            return exp;
        }

        //look at 'io.ox/office/spreadsheet/model/numberformatter'
        function getMantissa(nr, exp) {
            return nr / Math.pow(10, exp);
        }

        function newInfoHolder(type, label) {
            return {
                max : Number.NEGATIVE_INFINITY,
                date: false,
                type: type,
                label: label,
                unclean: false,
                exp: 0,
                labelLength: 0
            };
        }

        function handleData(info, source, dataPoint) {
            var realNr = source.result;
            var display = source.display;
            var format = ChartStyleUtil.getNumberFormat(chart.getApp(), source);

            if (format) {
                if (info.format) {
                    if (info.format !== format && info.cat !== source.format.cat) {
                        info.format = 'General';
                        info.date = null;
                    }
                } else {
                    if (/(D+|M+|Y+|h+|m+|s+)/.test(format)) {
                        info.date = format;
                    }
                    info.format = format;
                    info.cat = source.format.cat;
                }
            }

            var label = null;
            var res = null;
            if (info.date) {
                res = realNr;
                info.max = Math.max(info.max, res);
                label = display;

            } else if (realNr === undefined || realNr === null || isNaN(realNr)) {
                info.unclean = true;
                res = 0;
                label = display;
            } else if(info.format){
                info.max = Math.max(info.max, realNr);
                res = realNr;
                label = chart.getApp().getModel().getNumberFormatter().formatValue(realNr, info.format);
            } else {
                info.max = Math.max(info.max, realNr);
                res = realNr;
                label = chart.getApp().getModel().getNumberFormatter().formatStandardNumber(realNr, 12);
            }
            dataPoint[info.label] = Utils.escapeHTML(label);
            dataPoint[info.type] = res;
        }


        function calcMaxExp(info) {
            if(isFinite(info.max)) {
                info.exp = getExponent(info.max);
            } else {
                info.exp = 0;
            }
        }

        function handleMantissa(info, dataPoint) {
            if(info.exp) {
                dataPoint[info.type] = getMantissa(dataPoint[info.type], info.exp);
            }
        }

        function handleLabelLength(info, dataPoint) {
            var label = dataPoint[info.label];
            if (label) {
                info.labelLength = Math.max(info.labelLength, label.length);
            }
        }

        ////////////////////////////////////////////////////////////////////////////

        this.format = function(){
            var data = chart.getModelData();
            var xyChart = chart.isXYType() || chart.getMergedAttributes().chart.type.indexOf('area') === 0;

            var x = newInfoHolder('x', 'label');
            var y = newInfoHolder('y', 'name');

            data.linearX = false;

            if (data.reordered) {
                data.data.reverse();
                data.reordered = false;
            }

            data.legend.forceSmall = false;

            _.each(data.data, function (dataSeries, seriesIndex) {
                dataSeries.seriesIndex = seriesIndex;
                if(dataSeries.nameHolder) {
                    dataSeries.name = dataSeries.nameHolder.display;

                    if (data.legend.verticalAlign === 'center' && dataSeries.name.length > 14) {
                        data.legend.forceSmall = true;
                    }
                } else {
                    dataSeries.name =
                    //#. A data series in a chart object (a sequence of multiple data points with a title)
                    //#. %1$d is the numeric index of the series (e.g. "Series 1", "Series 2", etc.)
                    //#. This label should match the default title for data series in other spreadsheet
                    //#. applications (OpenOffice/LibreOffice Calc, Microsoft Excel).
                    //#, c-format
                    gt('Series %1$d', _.noI18n(seriesIndex + 1));
                }
                _.each(dataSeries.dataPoints, function (dataPoint, n) {
                    dataPoint.x = null;
                    dataPoint.y = null;
                    dataPoint.z = null;

                    handleData(y, dataPoint.valueSource, dataPoint);

                    if (xyChart && dataPoint.nameSource) {
                        handleData(x, dataPoint.nameSource, dataPoint);
                    } else {
                        dataPoint[x.type] = n + 1;
                        data.linearX = true;
                    }

                    //ugly part against double formatting!!!
                    if (dataPoint.nameSource) {
                        dataPoint[x.label] = Utils.escapeHTML(dataPoint.nameSource.display);
                    } else{
                        dataPoint[x.label] = Utils.escapeHTML(String(dataPoint[x.type]));
                    }

                    handleLabelLength(x, dataPoint);
                    handleLabelLength(y, dataPoint);

                    if (dataPoint.sizeSource) {
                        dataPoint.z = dataPoint.sizeSource.result;
                    }else{
                        dataPoint.z = null;
                    }
                    dataPoint.legendText = dataPoint.label;
                });

                dataSeries.toolTipContent = '{label}: {name}';

                if (xyChart && x.unclean) {
                    _.each(dataSeries.dataPoints, function (dataPoint, n) {
                        dataPoint.x = n + 1;
                    });
                }

            });

            if (chart.getMergedAttributes().chart.type.indexOf('bar') === 0 && !data.reordered) {
                data.data.reverse();
                data.reordered = true;
            }


            if(xyChart){
                calcMaxExp(x);
            }
            calcMaxExp(y);

            chart.getAxisModel('x').setFormat(x);
            chart.getAxisModel('y').setFormat(y);

            var thresholdY = 0.005;

            _.each(data.data, function (dataSeries) {
                _.each(dataSeries.dataPoints, function (dataPoint) {
                    if (!xyChart) {
                        //workaround that too small values stay visible
                        var rel = dataPoint.y / y.max;
                        if (Math.abs(rel)<thresholdY) {
                            if (rel<0) {
                                dataPoint.y = -thresholdY * y.max;
                            } else {
                                dataPoint.y = thresholdY * y.max;
                            }
                        }
                    }

                    handleMantissa(y, dataPoint);
                    handleMantissa(x, dataPoint);
                });
            });
        };

        this.update = function(seriesIndex, values, titles, names, bubbles){
            var data = chart.getModelData();
            var dataSeries = data.data[seriesIndex];
            // build series title
            dataSeries.nameHolder = titles[0];
            if(dataSeries.nameHolder) {
                if (!seriesIndex) {
                    chart.setMainName(dataSeries.nameHolder.display);
                }
            } else {
                if (!seriesIndex) {
                    chart.setMainName(null);
                }
            }

            dataSeries.dataPoints = [];
            _.each(values, function (valueCell, n) {
                var dataPoint = {};
                dataSeries.dataPoints.push(dataPoint);
                dataPoint.nameSource = names[n];    //x
                dataPoint.valueSource = valueCell;  //y
                dataPoint.sizeSource = bubbles[n];  //z
                dataPoint.color = 'transparent';
                dataPoint.markerColor = 'transparent';
            });
        };

    } // class ChartFormatter

    // exports ================================================================
    return ChartFormatter;

});
