/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/toolpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/pane'
    ], function (Utils, Pane) {

    'use strict';

    // class ToolPane =========================================================

    /**
     * A view pane attached to a border of the application window that adds
     * specific styling to the form controls contained in its view components.
     * Provides a leading, a centered, and a trailing area (according to the
     * position of the pane) as target nodes for the view components.
     *
     * The method ToolPane.addViewComponent() accepts a new option 'targetArea'
     * that can be set to the values 'leading', 'trailing', and 'center' (the
     * default value).
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {BaseApplication} app
     *  The application containing this tool pane.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  Pane.
     */
    function ToolPane(app, initOptions) {

        var // self reference
            self = this,

            // area nodes, mapped by type identifier
            areaNodes = {};

        // base constructor ---------------------------------------------------

        Pane.call(this, app, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed component into this tool pane.
         */
        function componentInserter(component, options) {

            var // the target area passed in the options
                targetArea = Utils.getStringOption(options, 'targetArea', 'center'),
                // special position of component
                specialPosition = Utils.getOption(options, 'insertBefore', false);

            // insert the component root node (fall-back to center area)
            if (!(targetArea in areaNodes)) { targetArea = 'center'; }

            if (!specialPosition) {
            	areaNodes[targetArea].append(component.getNode());

            } else {
            	specialPosition.getNode().before(component.getNode());
            }

        }

        // methods ------------------------------------------------------------

        /**
         * Returns the fixed-sized root node of the specified tool pane area.
         *
         * @param {String} targetArea
         *  The identifier of the tool pane area: one of 'leading', 'trailing',
         *  and 'center'.
         *
         * @returns {jQuery}
         *  The fixed-sized root node of the specified tool pane area.
         */
        this.getAreaWrapperNode = function (targetArea) {
            return (areaNodes[targetArea] || $()).parent();
        };

        /**
         * Returns the inner and outer sizes of all tool pane areas.
         *
         * @returns {Object}
         *  An object with the following properties:
         *  - {Number} total
         *      Total available size of the tool pane.
         *  - {Object} leading
         *      Content and area size of the leading area, in the following
         *      properties:
         *      - {Number} leading.content
         *          The effective size of the area contents, in pixels.
         *      - {Number} leading.area
         *          The fixed size of the area node itself, in pixels.
         *  - {Object} center
         *      Content and area size of the center area. See the property
         *      'leading' for details.
         *  - {Object} trailing
         *      Content and area size of the trailing area. See the property
         *      'leading' for details.
         */
        this.getAreaSizes = function () {

            var // the name of the size property for the area nodes
                sizeProp = this.isVerticalPosition() ? 'width' : 'height',
                // the resulting sizes
                areaSizes = { total: Utils.getFloorNodeSize(self.getNode())[sizeProp] };

            _.each(areaNodes, function (areaNode, areaType) {
                areaSizes[areaType] = {
                    content: Utils.getCeilNodeSize(areaNode)[sizeProp],
                    area: areaNode.parent()[sizeProp]()
                };
            });

            return areaSizes;
        };

        /**
         * Updates the sizes of the area nodes according to the sizes of their
         * content nodes.
         *
         * @returns {ToolPane}
         *  A reference to this instance.
         */
        this.updateAreaSizes = function () {

            var // the name of the size property for the area nodes
                sizeProp = this.isVerticalPosition() ? 'width' : 'height',
                // the current sizes of the area nodes
                areaSizes = this.getAreaSizes();

            // do not let leading and trailing areas overlap (trailing pane wins over leading over center)
            areaSizes.trailing.area = Math.min(areaSizes.trailing.content, areaSizes.total);
            areaSizes.leading.area = Math.min(areaSizes.leading.content, areaSizes.total - areaSizes.trailing.area);
            areaSizes.center.area = areaSizes.total - areaSizes.leading.area - areaSizes.trailing.area;

            // set the resulting sizes at the wrapper nodes (important for top/bottom panes)
            _.each(areaNodes, function (areaNode, areaType) {
                areaNode.parent()[sizeProp](areaSizes[areaType].area);
            });

            return this;
        };

        // initialization -----------------------------------------------------

        // add CSS marker class for special formatting
        this.getNode().addClass('tool-pane');

        this.getNode().on('touchmove', function(e) {
            e.preventDefault();
        });

        // create component area nodes
        _.each(['leading', 'center', 'trailing'], function (areaType) {
            areaNodes[areaType] = $('<div class="area-container">');
            self.getNode().append($('<div class="float-wrapper ' + areaType + '">').append(areaNodes[areaType]));
        });

        // insert new view components into the area nodes
        this.registerComponentInserter(componentInserter);

        // update size of the areas after the view pane has been resized
        this.on('pane:layout', function () { self.updateAreaSizes(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(areaNodes, 'remove');
            app = initOptions = self = areaNodes = null;
        });

    } // class ToolPane

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: ToolPane });

});
