/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/sidepane',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/toolbox'
    ], function (Utils, Pane, ToolBox) {

    'use strict';

    // class SidePane =========================================================

    /**
     * A view pane attached to the left or right border of the application
     * window. Contains tool boxes stacked vertically. Provides fixed areas at
     * the top and at the bottom of the side pane with tool boxes that will
     * always be visible. The inner area between the fixed areas will be
     * scrollable.
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {BaseApplication} app
     *  The application containing this side pane.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  Pane. The option 'position' will be restricted to the values 'left' and
     *  'right' (the default).
     */
    function SidePane(app, initOptions) {

        var // self reference
            self = this,

            // container node for the upper fixed tool boxes in the side pane
            fixedTopNode = $('<div class="fixed-toolboxes top">'),

            // scrollable container node for the tool boxes in the side panes
            scrollableNode = $('<div class="scrollable-toolboxes">'),

            // container node for the upper fixed tool boxes in the side pane
            fixedBottomNode = $('<div class="fixed-toolboxes bottom">');

        // base constructor ---------------------------------------------------

        Pane.call(this, app, Utils.extendOptions({
            // default options, can be overridden by passed options
            size: SidePane.DEFAULT_WIDTH
        }, initOptions, {
            // fixed options, will override passed options
            position: (Utils.getStringOption(initOptions, 'position') === 'left') ? 'left' : 'right',
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the side pane according to the current browser window size.
         */
        function refreshLayout() {

            if (!self.isVisible()) { return; }

            // update top/bottom positions of scrollable container in case the fixed tool boxes have changed
            scrollableNode.css({ top: fixedTopNode.height() - 1, bottom: fixedBottomNode.height() - 1 });
        }

        /**
         * Inserts the passed tool box into the side pane.
         */
        function toolBoxInserter(toolBox) {
            switch (Utils.getStringOption(toolBox.getOptions(), 'fixed')) {
            case 'top':
                fixedTopNode.append(toolBox.getNode());
                break;
            case 'bottom':
                fixedBottomNode.append(toolBox.getNode());
                break;
            default:
                scrollableNode.append(toolBox.getNode());
            }
        }

        // methods ------------------------------------------------------------

        this.getFixedTopNode = function () {
            return fixedTopNode;
        };

        this.getScrollableNode = function () {
            return scrollableNode;
        };

        this.getFixedBottomNode = function () {
            return fixedBottomNode;
        };

        /**
         * Creates a tool box in this side pane, and registers it for automatic
         * visibility handling.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the
         *  constructor of the class ToolBox. Additionally, the following
         *  options are supported:
         *  @param {String} [options.fixed]
         *      If set to 'top' or 'bottom', the new tool box will be appended
         *      to the upper respectively lower fixed area of the side pane,
         *      and cannot be scrolled outside the visible area. Otherwise, the
         *      tool box will be appended to the scrollable area of the side
         *      pane.
         *  @param {String} [options.visible]
         *      The key of the controller item that controls the visibility of
         *      the tool box. The visibility will be bound to the 'enabled'
         *      state of the respective controller item. If omitted, the tool
         *      box will always be visible.
         *
         * @returns {ToolBox}
         *  The new tool box instance.
         */
        this.createToolBox = function (options) {

            var // the new tool box instance
                toolBox = new ToolBox(app, options),
                // the controller item controlling the visibility of the tool box
                visibleKey = Utils.getStringOption(options, 'visible');

            // insert tool box into the side pane, refresh layout after expanding/collapsing
            this.addViewComponent(toolBox);
            self.listenTo(toolBox, 'expand', refreshLayout);

            // update visibility according to enabled state of controller item
            if (_.isString(visibleKey)) {
                // initially hide the tool box
                toolBox.hide();
                this.listenTo(app.getController(), 'change:items', function (event, changedItems) {
                    var enabled = (visibleKey in changedItems) ? changedItems[visibleKey].enabled : null;
                    if (_.isBoolean(enabled) && (toolBox.isVisible() !== enabled)) {
                        toolBox.toggle(enabled);
                        refreshLayout();
                    }
                });
            }

            return toolBox;
        };

        // initialization -----------------------------------------------------

        // insert the container nodes for fixed and scrollable tool boxes
        this.getNode().addClass('side-pane').append(fixedTopNode, scrollableNode, fixedBottomNode);

        // insert new tool boxes into the own container nodes
        this.registerComponentInserter(toolBoxInserter);

        // update side pane after controller updates (tool box visibility
        // may have changed), and after the view has refreshed the panes
        this.listenTo(app.getController(), 'change:items', refreshLayout);
        this.listenTo(app.getView(), 'refresh:layout', refreshLayout);

        // destroy all class members on destruction
        this.registerDestructor(function () {

            fixedTopNode.remove();
            scrollableNode.remove();
            fixedBottomNode.remove();

            self = scrollableNode = fixedTopNode = fixedBottomNode = null;
        });

    } // class SidePane

    // constants ==============================================================

    /**
     * Default width of side panes, in pixels.
     */
    SidePane.DEFAULT_WIDTH = 249;

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: SidePane });

});
