/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/baseview', [
    'io.ox/core/notifications',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/view/pane',
    'gettext!io.ox/office/main',
    'io.ox/office/tk/nodetracking', // import only
    'less!io.ox/office/baseframework/view/basestyle',
    'less!io.ox/office/baseframework/view/docs-icons'
], function (Notifications, Utils, KeyCodes, Forms, TriggerObject, TimerMixin, Pane, gt) {

    'use strict';

    var // a special value for an element's 'tabindex' attribute to remove it from the
        // focus tab chain, but to be able to find it later with an CSS selector
        HIDDEN_TAB_INDEX = -42;

    // global functions =======================================================

    /**
     * Converts the passed number or object to a complete margin descriptor
     * with the properties 'left', 'right', 'top', and 'bottom'.
     *
     * @param {Number|Object} margin
     *  The margins, as number (for all margins) or as object with the optional
     *  properties 'left', 'right', 'top', and 'bottom'. Missing properties
     *  default to the value zero.
     *
     * @returns {Object}
     *  The margins (in pixels), in the properties 'left', 'right', 'top', and
     *  'bottom'.
     */
    function getMarginFromValue(margin) {

        if (_.isObject(margin)) {
            return _.extend({ left: 0, right: 0, top: 0, bottom: 0 }, margin);
        }

        if (!_.isNumber(margin)) {
            margin = 0;
        }
        return { left: margin, right: margin, top: margin, bottom: margin };
    }

    // class BaseView =========================================================

    /**
     * Base class for the view instance of an office application. Creates the
     * application window, and provides functionality to create and control the
     * top, bottom, and side pane elements.
     *
     * Triggers the following events:
     * - 'refresh:layout': After this view instance has refreshed the layout of
     *      all registered view panes. This event will be triggered after
     *      inserting new view panes into this view, or content nodes into the
     *      application pane, after showing/hiding view panes, while and after
     *      the browser window is resized.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {BaseApplication} app
     *  The application containing this view instance.
     *
     * @param {Object} [initOptions]
     *  Additional options to control the appearance of the view. The following
     *  options are supported:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes to be added to the root node of the
     *      application window.
     *  @param {Function} [initOptions.initHandler]
     *      A callback handler function called to initialize this view instance
     *      after construction. May return a promise to be able to run
     *      asynchronous code during initialization.
     *  @param {Function} [initOptions.initGuiHandler]
     *      A callback handler function called to initialize the graphical
     *      elements of this view instance, after the document has been
     *      imported. May return a promise to be able to run asynchronous code
     *      during initialization.
     *  @param {Function} [initOptions.grabFocusHandler]
     *      A function that has to implement moving the browser focus somewhere
     *      into the application pane. Used in the method BaseView.grabFocus()
     *      of this class. If omitted, the application pane itself will be
     *      focused.
     *  @param {Boolean} [initOptions.contentFocusable=false]
     *      If set to true, the container node for the document contents will
     *      be focusable and will be registered for global focus traversal with
     *      the F6 key.
     *  @param {Boolean} [initOptions.contentScrollable=false]
     *      If set to true, the container node for the document contents will
     *      be scrollable. By default, the size of the container node is locked
     *      and synchronized with the size of the application pane (with regard
     *      to content margins, see the option 'contentMargin' for details).
     *  @param {Number|Object} [initOptions.contentMargin=0]
     *      The margins between the fixed application pane and the embedded
     *      application container node, in pixels. If set to a number, all
     *      margins will be set to the specified value. Otherwise, an object
     *      with the optional properties 'left', 'right', 'top', and 'bottom'
     *      for specific margins for each border. Missing properties default to
     *      the value zero. The content margin can also be modified at runtime
     *      with the method BaseView.setContentMargin().
     *  @param {Number|Object} [initOptions.overlayMargin=0]
     *      The margins between the overlay panes and the inner borders of the
     *      application pane, in pixels. If set to a number, all margins will
     *      be set to the specified value. Otherwise, an object with the
     *      optional properties 'left', 'right', 'top', and 'bottom' for
     *      specific margins for each border. Missing properties default to the
     *      value zero.
     */
    function BaseView(app, initOptions) {

        var // self reference
            self = this,

            // moves the browser focus into a node of the application pane
            grabFocusHandler = Utils.getFunctionOption(initOptions, 'grabFocusHandler'),

            // the application window instance (ox.ui.Window)
            win = null,

            // the DOM node of the application window body
            winBodyNode = null,

            // root node of the application pane (remaining space for document contents)
            appPaneNode = null,

            // root container node for invisible document contents
            hiddenRootNode = $('<div class="abs app-hidden-root">'),

            // root container node for visible document contents (may be scrollable)
            contentRootNode = $('<div class="abs app-content-root">'),

            // container node for application contents
            appContentNode = $('<div class="app-content">').appendTo(contentRootNode),

            // busy node for the application pane
            appBusyNode = $('<div class="abs app-busy">'),

            // the temporary container for all nodes while application is hidden
            tempNode = $('<div>'),

            // all fixed view panes, in insertion order
            fixedPanes = [],

            // all overlay view panes, in insertion order
            overlayPanes = [],

            // whether refreshing the pane layout is currently locked
            layoutLocks = 0,

            // whether refreshing the pane layout is currently running
            layoutRunning = false,

            // whether refreshing the pane layout was requested while it was locked
            layoutPending = false,

            // margins of overlay panes to the borders of the application pane
            overlayMargin = getMarginFromValue(Utils.getOption(initOptions, 'overlayMargin', 0)),

            // whether the content root node is focusable by itself
            contentFocusable = Utils.getBooleanOption(initOptions, 'contentFocusable', false),

            // whether the application is hidden explicitly
            viewHidden = false,

            // cached yell options, shown when application becomes visible
            pendingYellOptions = null,

            // the timer waiting to fade in the blocker element in busy mode
            blockerFadeTimer = null,

            // the timer waiting to show a warning message during busy mode
            blockerWarningTimer = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        function silentRefreshPaneLayout() {

            var // current offsets representing available space in the application window
                offsets = { top: 0, bottom: 0, left: 0, right: 0 };

            function updatePane(pane) {

                var paneNode = pane.getNode(),
                    visible = pane.isVisible(),
                    transparent = pane.isTransparent(),
                    position = pane.getPosition(),
                    vertical = pane.isVerticalPosition(),
                    leading = pane.isLeadingPosition(),
                    sizeFunc = _.bind(vertical ? paneNode.outerHeight : paneNode.outerWidth, paneNode),
                    sizeAttr = vertical ? 'height' : 'width',
                    paneOffsets = _.clone(offsets);

                // remove the position attribute at the opposite border of the pane position
                paneOffsets[vertical ? (leading ? 'bottom' : 'top') : (leading ? 'right' : 'left')] = '';

                // transparent overlay panes: temporarily set to auto size, adjust position of trailing panes
                if (visible && transparent) {
                    paneNode.css(sizeAttr, 'auto');
                    if (!leading) {
                        paneOffsets[position] += sizeFunc();
                    }
                }

                // set pane position, and let pane update its internal size
                // (this may even change the pane size again due to internal refresh)
                paneNode.css(paneOffsets);
                if (visible) {
                    pane.layout();
                    offsets[position] += sizeFunc();
                }

                // transparent overlay panes: set zero size
                if (visible && transparent) {
                    paneNode.css(sizeAttr, 0);
                }
            }

            // prevent recursive processing
            if (layoutRunning) { return; }
            layoutRunning = true;

            // update fixed view panes
            _.each(fixedPanes, updatePane);

            // update the application pane node
            appPaneNode.css(offsets);

            // skip margins for overlay panes
            _.each(offsets, function (offset, pos) { offsets[pos] += overlayMargin[pos]; });

            // update overlay view panes
            _.each(overlayPanes, updatePane);

            // leave recursion guard
            layoutRunning = false;
        }

        /**
         * Adjusts the positions of all view pane nodes.
         */
        function refreshPaneLayout() {

            // do nothing if the window is hidden (but refresh if import runs (fast loading))
            if (!self.isVisible()) {
                return;
            }

            // do nothing if refreshing is currently locked, but set the pending flag
            if (layoutLocks > 0) {
                layoutPending = true;
                return;
            }

            // refresh all panes, and notify listeners
            silentRefreshPaneLayout();
            self.trigger('refresh:layout');
        }

        /**
         * Refreshes the view layout after a view pane has been modified.
         */
        var paneResizeHandler = (function () {

            var // whether to really refresh the layout in the deferred callback
                refreshLayout = false;

            function registerRefresh() {
                // do nothing when view panes are changing during layout updates
                if (!layoutRunning) { refreshLayout = true; }
            }

            function executeRefresh() {
                if (refreshLayout) {
                    refreshLayout = false;
                    refreshPaneLayout();
                }
            }

            return self.createDebouncedMethod(registerRefresh, executeRefresh);
        }());

        /**
         * Shows the specified notification message. If the message is of type
         * 'error', the message will be stored internally and automatically
         * shown again, after the application has been hidden and shown.
         *
         * @param {Object} yellOptions
         *  The notification message data. See method BaseView.yell() for
         *  details.
         */
        function showNotification(yellOptions) {

            var // the notification DOM element
                yellNode = null;

            if (yellOptions.type === 'error') {
                // Bug 28554: no auto-close for error messages
                yellOptions.duration = -1;
                // remember error message, show again after switching applications
                pendingYellOptions = yellOptions;
            } else {
                pendingYellOptions = null;
            }

            // create and show the notification DOM node
            yellNode = Notifications.yell(yellOptions);

            // wait for deferred initialization of the DOM node
            self.listenTo(yellNode, 'notification:appear', function () {

                var // the message node as target for additional contents
                    messageNode = yellNode.find('.message'),
                    // options for a static hyperlink shown below the message text
                    linkOptions = Utils.getObjectOption(yellOptions, 'hyperlink'),
                    // the URL for the static hyperlink
                    linkUrl = Utils.getStringOption(linkOptions, 'url'),
                    // the display text for the static hyperlink
                    linkLabel = Utils.getStringOption(linkOptions, 'label'),
                    // options for an action link shown below the message text
                    actionOptions = Utils.getObjectOption(yellOptions, 'action'),
                    // the key of the controller item for an action link
                    actionKey = Utils.getStringOption(actionOptions, 'itemKey');

                // append a static URL to the alert
                if (linkUrl) {
                    messageNode.append('<div><a href="' + linkUrl + '" target="_blank" tabindex="0">' + (linkLabel || _.noI18n(linkUrl)) + '</a></div>');
                }

                // do not insert an action link, if the controller item is missing or disabled
                if (!actionKey || !app.getController().isItemEnabled(actionKey)) { return; }

                var // the icon class for the link
                    actionIcon = Utils.getStringOption(actionOptions, 'icon'),
                    // the label text for the link
                    actionLabel = Utils.getStringOption(actionOptions, 'label'),
                    // the anchor element to be inserted into the alert node
                    anchorNode = $('<a href="#">');

                // insert the icon and label to the anchor node
                if (actionIcon) { anchorNode.append(Forms.createIconMarkup(actionIcon)); }
                if (actionLabel) { anchorNode.append(Forms.createSpanMarkup(actionLabel)); }

                // execute the controller item on click (unbind the listener when quitting the application)
                self.listenTo(anchorNode, 'click tap', function (event) {
                    event.preventDefault();
                    Notifications.yell.close();
                    app.getController().executeItem(actionKey);
                });

                // insert the link into the alert node
                messageNode.append($('<div>').append(anchorNode));
            });

            // return focus to application after the notification closes
            self.listenTo(yellNode, 'notification:removed', function (event) {
                // bug 33735: only if focus is inside the alert box (e.g. after mouse click)
                if (Forms.containsFocus(event.currentTarget)) {
                    self.executeDelayed(function () { self.grabFocus(); });
                }
            });
        }

        /**
         * Updates the view after the application becomes active/visible.
         */
        function windowShowHandler() {

            // do not show the window contents if view is still hidden explicitly
            if (!self.isVisible()) { return; }

            // bug 29348: show pending notification also, if tempNode is empty
            if (pendingYellOptions && app.isImportFinished() && self.isVisible()) {
                showNotification(pendingYellOptions);
            }

            // move all application nodes from temporary storage into view
            if (tempNode.children().length > 0) {
                winBodyNode.append(tempNode.children());
                // restore invalidated tabindex attribute for browser focus tab chain
                tempNode.find('[tabindex="' + HIDDEN_TAB_INDEX + '"]').attr('tabindex', 1);
            }

            // do not update GUI and grab focus while document is still being imported
            if (app.isImportFinished()) {
                self.executeDelayed(function () {
                    refreshPaneLayout();
                    app.getController().update();
                    self.grabFocus();
                });
            }
        }

        /**
         * Updates the view after the application becomes inactive/hidden.
         */
        function windowHideHandler() {
            // move all application nodes from view to temporary storage
            if (tempNode.children().length === 0) {
                tempNode.append(winBodyNode.children());
                // remove all nodes from browser focus tab chain as long as they are hidden
                tempNode.find('[tabindex="1"]').attr('tabindex', HIDDEN_TAB_INDEX);
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Prepares the view for importing the document. Sets the application
         * to busy state, and invokes the initialization handler passed to the
         * constructor.
         *
         * @internal
         *  Invoked from the import process of the application. DO NOT call
         *  from external code.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  initialization handler has finished.
         */
        this.hideBeforeImport = _.once(function () {

            // set window to busy state
            app.getWindow().busy();
            self.hide();

            // invoke the initialization handler passed to the constructor
            var result = Utils.getFunctionOption(initOptions, 'initHandler', $.noop).call(self);

            // return a promise
            return $.when(result);
        });

        /**
         * Initializes the view after importing the document. Invokes the GUI
         * initialization handler passed to the constructor, and sets the
         * application to idle state.
         *
         * @internal
         *  Invoked from the import process of the application. DO NOT call
         *  from external code.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the GUI
         *  initialization handler has finished.
         */
        this.showAfterImport = _.once(function () {

            // show the application window
            self.show();

            // invoke the GUI initialization handler passed to the constructor
            var result = Utils.getFunctionOption(initOptions, 'initGuiHandler', $.noop).call(self);

            // wait for the callback, immediately refresh all view panes to reduce flicker effects
            return $.when(result).always(function () {
                silentRefreshPaneLayout();
                self.leaveBusy();
            });
        });

        // public methods -----------------------------------------------------

        /**
         * Returns whether the browser is located inside the application pane.
         *
         * @returns {Boolean}
         *  Whether the browser is located inside the application pane.
         */
        this.hasAppFocus = function () {
            return Forms.hasOrContainsFocus(this.getAppPaneNode());
        };

        /**
         * Moves the browser focus into the application pane. Calls the
         * handler function passed to the constructor of this instance.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            if (this.isVisible()) {
                if (_.isFunction(grabFocusHandler)) {
                    grabFocusHandler.call(this);
                } else if (contentFocusable) {
                    contentRootNode.focus();
                } else {
                    Forms.grabFocus(this.getAppPaneNode());
                }
            }
            return this;
        };

        /**
         * Returns whether the contents of the view are currently visible.
         *
         * @returns {Boolean}
         *  Whether the application is active, and the view has not been hidden
         *  manually.
         */
        this.isVisible = function () {
            return !viewHidden && app.isActive();
        };

        /**
         * Hides all contents of the application window and moves them to the
         * internal temporary DOM storage node.
         *
         * @return {BaseView}
         *  A reference to this instance.
         */
        this.hide = function () {
            viewHidden = true;
            windowHideHandler();
            return this;
        };

        /**
         * Shows all contents of the application window, if the application
         * itself is currently active. Otherwise, the contents will be shown
         * when the application becomes visible.
         *
         * @return {BaseView}
         *  A reference to this instance.
         */
        this.show = function () {
            viewHidden = false;
            windowShowHandler();
            return this;
        };

        /**
         * Returns the DOM node of the application pane (the complete inner
         * area between all existing view panes). Note that this is NOT the
         * container node where applications insert their own contents. The
         * method BaseView.insertContentNode() is intended to be used to insert
         * own contents into the application pane.
         *
         * @returns {jQuery}
         *  The DOM node of the application pane.
         */
        this.getAppPaneNode = function () {
            return appPaneNode;
        };

        /**
         * Sets the application into the busy state by displaying a window
         * blocker element covering the application pane. All other view panes
         * (also the overlay panes covering the blocked application pane)
         * remain available.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.appPaneBusy = function () {
            appBusyNode.show().busy();
            return this;
        };

        /**
         * Leaves the busy state from the application pane that has been
         * entered by the method BaseView.appPaneBusy().
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.appPaneIdle = function () {
            appBusyNode.idle().hide();
            return this;
        };

        /**
         * Detaches the document contents in the application pane from the DOM.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.detachAppPane = function () {
            appContentNode.detach();
            return this;
        };

        /**
         * Attaches the document contents in the application pane to the DOM.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.attachAppPane = function () {
            contentRootNode.append(appContentNode);
            return this;
        };

        /**
         * Returns the root container node for the application contents. Note
         * that this is NOT the direct parent node where applications insert
         * their own contents, but the (optionally scrollable) root node
         * containing the target container node for document contents. The
         * method BaseView.insertContentNode() is intended to be used to insert
         * own contents into the application pane.
         *
         * @returns {jQuery}
         *  The DOM root node for visible document contents.
         */
        this.getContentRootNode = function () {
            return contentRootNode;
        };

        /**
         * Returns the current margins between the fixed application pane and
         * the embedded application container node.
         *
         * @returns {Object}
         *  The margins between the fixed application pane and the embedded
         *  application container node (in pixels), in the properties 'left',
         *  'right', 'top', and 'bottom'.
         */
        this.getContentMargin = function () {
            return {
                left: Utils.getElementCssLength(appContentNode, 'margin-left'),
                right: Utils.getElementCssLength(appContentNode, 'margin-right'),
                top: Utils.getElementCssLength(appContentNode, 'margin-top'),
                bottom: Utils.getElementCssLength(appContentNode, 'margin-bottom')
            };
        };

        /**
         * Changes the margin between the fixed application pane and the
         * embedded application container node.
         *
         * @param {Number|Object} margin
         *  The margins between the fixed application pane and the embedded
         *  application container node, in pixels. If set to a number, all
         *  margins will be set to the specified value. Otherwise, an object
         *  with the optional properties 'left', 'right', 'top', and 'bottom'
         *  for specific margins for each border. Missing properties default to
         *  the value zero.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.setContentMargin = function (margin) {
            margin = getMarginFromValue(margin);
            appContentNode.css('margin', margin.top + 'px ' + margin.right + 'px ' + margin.bottom + 'px ' + margin.left + 'px');
            return this;
        };

        /**
         * Adjusts the positions of all view pane nodes.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.refreshPaneLayout = function () {
            refreshPaneLayout();
            return this;
        };

        /**
         * Inserts new DOM nodes into the container node of the application
         * pane.
         *
         * @param {HTMLElement|jQuery} contentNode
         *  The DOM node(s) to be inserted into the application pane. If this
         *  object is a jQuery collection, inserts all contained DOM nodes into
         *  the application pane.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.insertContentNode = function (contentNode) {
            appContentNode.append(contentNode);
            refreshPaneLayout();
            return this;
        };

        /**
         * Removes all DOM nodes from the container node of the application
         * pane.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.removeAllContentNodes = function () {
            appContentNode.empty();
            refreshPaneLayout();
            return this;
        };

        /**
         * Returns the root container node for hidden application contents.
         *
         * @returns {jQuery}
         *  The DOM root node for hidden document contents.
         */
        this.getHiddenRootNode = function () {
            return hiddenRootNode;
        };

        /**
         * Inserts new DOM nodes into the hidden container node of the
         * application pane.
         *
         * @param {HTMLElement|jQuery} nodes
         *  The DOM node(s) to be inserted into the hidden container of the
         *  application pane. If this object is a jQuery collection, inserts
         *  all contained DOM nodes into the application pane.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.insertHiddenNodes = function (nodes) {
            hiddenRootNode.append(nodes);
            return this;
        };

        /**
         * Adds the passed view pane instance into this view.
         *
         * @param {Pane} pane
         *  The view pane instance to be inserted into this view.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.addPane = function (pane) {

            // insert the pane
            (pane.isOverlay() ? overlayPanes : fixedPanes).push(pane);
            (viewHidden ? tempNode : winBodyNode).append(pane.getNode());

            // refresh pane layout when the pane or its contents are changed
            pane.on('pane:show pane:hide pane:resize', paneResizeHandler);
            paneResizeHandler();

            return this;
        };

        /**
         * Prevents refreshing the pane layout while the specified callback
         * function is running. After the callback function has finished, the
         * pane layout will be adjusted once by calling the method
         * BaseView.refreshPaneLayout().
         *
         * @param {Function} callback
         *  The callback function that will be executed synchronously while
         *  refreshing the pane layout is locked.
         *
         * @param {Object} [context]
         *  The context bound to the callback function.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.lockPaneLayout = function (callback, context) {
            layoutLocks += 1;
            callback.call(context);
            layoutLocks -= 1;
            if (layoutLocks === 0) {
                if (layoutPending) { refreshPaneLayout(); }
                layoutPending = false;
            }
            return this;
        };

        /**
         * Sets the application into the busy state by displaying a window
         * blocker element covering the entire GUI of the application. The
         * contents of the header and footer in the blocker element are
         * cleared, and the passed initialization callback function may insert
         * new contents into these elements.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function} [options.initHandler]
         *      A function that can fill custom contents into the header and
         *      footer of the window blocker element. Receives the following
         *      parameters:
         *      (1) {jQuery} headerNode
         *          The header element above the centered progress bar.
         *      (2) {jQuery} footerNode
         *          The footer element below the centered progress bar.
         *      (3) {jQuery} blockerNode
         *          The entire window blocker element (containing the header,
         *          footer, and progress bar elements).
         *      Will be called in the context of this view instance.
         *  @param {Function} [options.cancelHandler]
         *      If specified, a 'Cancel' button will be shown after a short
         *      delay. Pressing that button, or pressing the ESCAPE key, will
         *      execute this callback function, and will leave the busy mode
         *      afterwards. Will be called in the context of this view
         *      instance.
         *  @param {Number} [options.delay=500]
         *      The delay time before the busy blocker element becomes visible
         *      (by fading to a half-transparent fill color). Note that the
         *      busy blocker element exists and blocks everything right after
         *      calling this method, the delay time just specifies when it
         *      becomes visible.
         *  @param {Boolean} [options.immediate=false]
         *      If set to true, the half-transparent busy blocker element will
         *      be shown immediately. Otherwise, the element will be set to
         *      transparent initially, and will fade in after the specified
         *      delay (see option 'delay').
         *  @param {Boolean} [options.showFileName=false]
         *      If set to true, the file name will be shown in the top-left
         *      corner of the blocker element.
         *  @param {String} [options.warningLabel]
         *      If specified, an alert box containing the passed text will be
         *      shown after a specific delay (see option 'warningDelay').
         *      Intended to be used to show a message for long-running actions.
         *  @param {Number} [options.warningDelay=3000]
         *      The delay time before the warning alert becomes visible. If the
         *      blocker element will be faded in delayed by itself, the warning
         *      delay will start after that delay time.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.enterBusy = function (options) {

            // for safety against repeated calls: stops pending animations etc.
            this.leaveBusy();

            // enter busy state, and extend the blocker element
            win.busy(null, null, function () {

                var // the initialization handler
                    initHandler = Utils.getFunctionOption(options, 'initHandler'),
                    // the cancel handler
                    cancelHandler = Utils.getFunctionOption(options, 'cancelHandler'),
                    // the delay time before the blocker element becomes visible
                    delay = Utils.getIntegerOption(options, 'delay', 500, 0),
                    // the label text for the warning alert for long-running actions
                    warningLabel = Utils.getStringOption(options, 'warningLabel', ''),
                    // the delay time before the warning alert box becomes visible
                    warningDelay = Utils.getIntegerOption(options, 'warningDelay', 3000, 0),
                    // the DOM element containing the warning alert box, or the single cancel button
                    warningNode = $('<div>').hide(),
                    // the window blocker element (bound to 'this')
                    blockerNode = this,
                    // the header container node
                    headerNode = blockerNode.find('.header').empty(),
                    // the footer container node
                    footerNode = blockerNode.find('.footer').empty(),
                    // the cancel button element
                    cancelButton = null;

                // keyboard event handler for busy mode (ESCAPE key)
                function busyKeydownHandler(event) {
                    if (event.keyCode === KeyCodes.ESCAPE) {
                        cancelHandler.call(self);
                        return false;
                    }
                }

                // special marker for custom CSS formatting, clear header/footer
                blockerNode.addClass('io-ox-office-blocker');

                // add file name to header area
                if (Utils.getBooleanOption(options, 'showFileName', false)) {
                    headerNode.append($('<div>').addClass('filename clear-title').text(_.noI18n(app.getFullFileName())));
                }

                // on IE, remove stripes (they are displayed too nervous and flickering)
                if (_.browser.IE) {
                    blockerNode.find('.progress-striped').removeClass('progress-striped');
                }

                // execute initialization handler
                if (_.isFunction(initHandler)) {
                    initHandler.call(self, headerNode, footerNode, blockerNode);
                }

                // append the container node for the warning alert box and/or cancel button
                footerNode.append(warningNode);

                // show blocker immediately, or fade it in after a delay
                if (Utils.getBooleanOption(options, 'immediate', false)) {
                    blockerNode.css('opacity', '');
                } else {
                    blockerNode.css('opacity', 0);
                    blockerFadeTimer = self.repeatDelayed(function (index) {
                        blockerNode.css('opacity', (index + 1) / 10);
                    }, { delay: delay, repeatDelay: 50, cycles: 10 });
                    // defer showing the warning box until blocker is visible
                    warningDelay += delay + 500;
                }

                // initialize a cancel button, if a callback handler has been passed
                if (_.isFunction(cancelHandler)) {

                    // create the Cancel button
                    cancelButton = $.button({ label: gt('Cancel') })
                        .addClass('btn-warning')
                        .on('click', function () { cancelHandler.call(self); });

                    // register a keyboard handler for the ESCAPE key
                    winBodyNode.on('keydown', busyKeydownHandler);
                    win.one('idle', function () {
                        winBodyNode.off('keydown', busyKeydownHandler);
                    });

                    // make the blocker focusable for keyboard input
                    blockerNode.attr('tabindex', 1).focus();
                }

                // create a warning alert box if specified, insert cancel button into the alert box
                if (warningLabel.length > 0) {
                    var alertNode = $('<div class="alert alert-warning"><div>' + Utils.escapeHTML(warningLabel) + '</div></div>');
                    if (cancelButton) { alertNode.append($('<div>').append(cancelButton)); }
                    warningNode.append(alertNode);
                } else if (cancelButton) {
                    warningNode.append(cancelButton);
                }

                // show warning alert box after a delay
                if (warningNode.children().length > 0) {
                    blockerWarningTimer = self.executeDelayed(function () {
                        warningNode.show();
                        if (cancelButton) { cancelButton.focus(); }
                    }, warningDelay);
                }
            });

            return this;
        };

        /**
         * Updates the progress bar of the busy blocker element.
         *
         * @param {Number} progress
         *  The progress value (floating-point number between 0 and 1).
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.updateBusyProgress = (function () {

            var // the current progress value to be shown at the progress bar
                currProgress = 0;

            function registerProgress(progress) {
                currProgress = progress;
                return self;
            }

            function showProgress() {
                if (win.nodes.blocker.is(':visible')) {
                    win.busy(currProgress);
                }
            }

            return self.createDebouncedMethod(registerProgress, showProgress, { delay: 500, maxDelay: 500 });
        }());

        /**
         * Leaves the busy state of the application. Hides the window blocker
         * element covering the entire GUI of the application.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.leaveBusy = function () {

            // stop timer that fades in the blocker element delayed
            if (blockerFadeTimer) {
                blockerFadeTimer.abort();
                blockerFadeTimer = null;
            }

            // stop timer that wants to show a warning alert in the blocker element
            if (blockerWarningTimer) {
                blockerWarningTimer.abort();
                blockerWarningTimer = null;
            }

            win.idle();
            return this;
        };

        /**
         * Shows an alert banner, if the application is currently visible.
         * Otherwise, the last alert banner will be cached until the
         * application becomes visible again.
         *
         * @param {Object} yellOptions
         *  The settings for the alert banner:
         *  @param {String} [yellOptions.type='info']
         *      The type of the alert banner. Supported types are 'success',
         *      'info', 'warning', and 'error'.
         *  @param {String} [yellOptions.headline]
         *      An optional headline shown above the message text.
         *  @param {String} yellOptions.message
         *      The message text shown in the alert banner.
         *  @param {Number} []yellOptions.duration]
         *      The time to show the alert banner, in milliseconds; or -1 to
         *      show a permanent alert. Default duration is dependent on the
         *      alert type: permanent for error alerts, and specific durations
         *      defined by the core toolkit for all other types.
         *  @param {Object} [yellOptions.hyperlink]
         *      An arbitrary static hyperlink that will be shown below the
         *      message text, with the following properties:
         *      @param {String} yellOptions.hyperlink.url
         *          The URL for the hyperlink.
         *      @param {String} [yellOptions.hyperlink.label]
         *          The display text shown as hyperlink. If omitted, the URL
         *          will be displayed.
         *  @param {Object} [yellOptions.action]
         *      A descriptor for an action link shown below the message text.
         *      The link will be bound to an arbitrary item of the application
         *      controller. If specified, must be an object with the following
         *      properties:
         *      @param {String} yellOptions.action.itemKey
         *          The key of a controller item that will be bound to the
         *          link. The link will be shown if the controller item is
         *          enabled, and upon clicking the link, the controller item
         *          will be executed.
         *      @param {String} [yellOptions.action.icon]
         *          The class name of an icon shown before the link label. MUST
         *          be an icon from FontAwesome, internal OX Documents bitmap
         *          icons are NOT supported.
         *      @param {String} yellOptions.action.label
         *          A label text for the link.
         *
         * @returns {BaseView}
         *  A reference to this instance.
         */
        this.yell = function (yellOptions) {

            // add default options
            yellOptions = _.extend({ type: 'info' }, yellOptions);

            // only show notification if the application is active, otherwise cache for later use
            if (this.isVisible()) {
                showNotification(yellOptions);
            } else {
                pendingYellOptions = yellOptions;
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        win = app.getWindow();
        winBodyNode = app.getWindowNode();

        // add CSS classes to the window root node, and the temporary container node
        (function () {

            var // all CSS classes (always add the application name)
                windowNodeClasses = 'io-ox-office-main ' + app.getName().replace(/[.\/]/g, '-') + '-main',
                // additional CSS classes from options
                classes = Utils.getStringOption(initOptions, 'classes');

            // add the classes passed through options
            if (classes) { windowNodeClasses += ' ' + classes; }

            // add application-specific classes to the window body node, and the temporary container node
            winBodyNode.addClass(windowNodeClasses);
            tempNode.addClass(windowNodeClasses);
        }());

        // add the temporary container for hidden state into the DOM
        Utils.insertHiddenNodes(tempNode);

        // log layout events
        Utils.LayoutLogger.logEvent(this, 'refresh:layout');

        // create the application pane, and insert the container nodes
        appPaneNode = $('<div class="app-pane">');
        appPaneNode.append(hiddenRootNode, contentRootNode, appBusyNode.hide());
        winBodyNode.append(appPaneNode);

        // keep application in DOM while application is hidden, applications
        // may want to access element geometry in background tasks
        this.listenTo(win, 'show', windowShowHandler);
        this.listenTo(win, 'hide', windowHideHandler);

        // listen to browser window resize events when the application window is visible
        app.registerWindowResizeHandler(refreshPaneLayout);

        // initialize content node from passed options
        contentRootNode.toggleClass('scrolling', Utils.getBooleanOption(initOptions, 'contentScrollable', false));
        this.setContentMargin(Utils.getOption(initOptions, 'contentMargin', 0));

        // make the content root node focusable for global navigation with F6 key
        if (contentFocusable) {
            contentRootNode.addClass('f6-target').attr('tabindex', 1);
        }

        // update view and pane layout after import
        app.onImport(windowShowHandler);

        // show notification alerts when import fails
        app.onImportFailure(function (result) {

            var cause = Utils.getStringOption(result, 'cause', 'unknown'),
                headline = Utils.getStringOption(result, 'headline', gt('Load Error')),
                message = Utils.getStringOption(result, 'message', gt('An error occurred while loading the document.'));

            // special case if a server component is not working correctly
            if (cause === 'bad server component') {
                headline = gt('Server Error');
                message = gt('A Documents server component is not working. Please contact the server administrator.');
            }

            // be silent in server timeout situations
            if (cause !== 'timeout') {
                self.yell({ type: 'error', headline: headline, message: message });
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            // destroy all image nodes (release the browser-internal bitmap data)
            app.destroyImageNodes(winBodyNode);
            app.destroyImageNodes(tempNode);

            _.invoke(fixedPanes, 'destroy');
            _.invoke(overlayPanes, 'destroy');

            tempNode.remove();
            appPaneNode.remove();

            app = initOptions = self = grabFocusHandler = win = winBodyNode = appPaneNode = null;
            hiddenRootNode = contentRootNode = appContentNode = appBusyNode = tempNode = null;
            fixedPanes = overlayPanes = blockerFadeTimer = blockerWarningTimer = null;
        });

    } // class BaseView

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BaseView });

});
