/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/tk/filedescriptor', [
    'io.ox/office/tk/utils',
], function (Utils) {

    'use strict';

    // class FileDescriptor ===================================================

    /**
     * Provides methods specific for file descriptors
     *
     * @constructor
     *
     * @param {Object} [fileDescriptor]
     *  The result object directly provided by the server which contains error
     *  information that are used to initialize ErrorCode instance.
     */
    var FileDescriptor = {};

    // constants --------------------------------------------------------------

    FileDescriptor.ENCODE_DOT = '%2E';

    // static methods ---------------------------------------------------------

    /**
     * Provides a file descriptor as an encoded resource id, which can be used
     * to initialize a real-time connection.
     *
     * @param {Object}
     *  A file descriptor object which contains file specific properties, like
     *  id and folder_id.
     *
     * @returns {String}
     *  A string which references a file and complies to the resource id encoding
     *  scheme. Can be null, if the file descriptor doesn't describe a valid
     *  file.
     */
    FileDescriptor.getAsResourceId = function(file) {
        var // encoded
        result = null;

        if (_.isObject(file)) {
            // create an encoded rt-compliant resource id
            result = encodeURIComponent(Utils.getStringOption(file, 'folder_id', '')).replace(/\./g, FileDescriptor.ENCODE_DOT);
            result += '.';
            result += encodeURIComponent(Utils.getStringOption(file, 'id', '')).replace(/\./g, FileDescriptor.ENCODE_DOT);
        }
        return result;
    };

    // exports ================================================================

    return FileDescriptor;
});
