/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/spinfield', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/textfield'
], function (Utils, KeyCodes, Forms, TextField) {

    'use strict';

    // class SpinField ========================================================

    /**
     * Creates a numeric field control with additional controls used to spin
     * (decrease and increase) the value.
     *
     * @constructor
     *
     * @extends TextField
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the TextField base class.
     *  If the option 'validator' exists, it MUST be an instance of
     *  TextField.NumberValidator (or a sub class). Alternatively, the options
     *  supported by the constructor of the class TextField.NumberValidator can
     *  be passed. In this case, a new instance of TextField.NumberValidator
     *  will be created automatically based on these options. Additionally, the
     *  following options are supported:
     *  @param {Number} [initOptions.smallStep=1]
     *      The distance of small steps that will be used to increase or
     *      decrease the current value of the spin field when using the spin
     *      buttons or the cursor keys.
     *  @param {Number} [initOptions.largeStep=10]
     *      The distance of large steps that will be used to increase or
     *      decrease the current value of the spin field when using the PAGEUP
     *      or PAGEPOWN key.
     *  @param {Boolean} [initOptions.roundStep=false]
     *      If set to true, and the current value of the spin field will be
     *      changed, the value will be rounded to entire multiples of the step
     *      distance. Otherwise, the fractional part will remain in the value.
     */
    function SpinField(initOptions) {

        var // self reference
            self = this,

            // the number validator of this spin field
            validator = Utils.getObjectOption(initOptions, 'validator') || new TextField.NumberValidator(initOptions),

            // the distance for small steps
            smallStep = Utils.getNumberOption(initOptions, 'smallStep', 1, 0),

            // the distance for large steps
            largeStep = Utils.getNumberOption(initOptions, 'largeStep', 10, 0),

            // whether to round while applying steps
            roundStep = Utils.getBooleanOption(initOptions, 'roundStep', false);

        // base constructors --------------------------------------------------

        TextField.call(this, Utils.extendOptions({ keyboard: 'number' }, initOptions, { validator: validator }));

        // private methods ----------------------------------------------------

        /**
         * Changes the current value of the spin field, according to the passed
         * step and the rounding mode passed to the constructor.
         */
        function changeValue(step) {

            var // the current value of the spin field
                oldValue = self.getFieldValue(),
                // the current value, rounded to the previous/next border
                roundedValue = (step < 0) ? Utils.roundDown(oldValue, -step) : Utils.roundUp(oldValue, step),
                // the new value, depending on rounding mode
                newValue = (!roundStep || (oldValue === roundedValue)) ? (oldValue + step) : roundedValue;

            // set new value, after restricting to minimum/maximum
            newValue = validator.restrictValue(newValue);
            self.setValue(newValue);
        }

        /**
         * Creates and returns a spin button. Initializes tracking used to
         * change the current value.
         */
        function createSpinButton(increase) {

            var // create the spin button element
                spinButton = $(Forms.createButtonMarkup({ focusable: false })).addClass(increase ? 'spin-up' : 'spin-down');

            // enables or disables node tracking according to own enabled state
            function initializeTracking() {
                if (self.isEnabled()) {
                    spinButton.enableTracking({ autoRepeat: true });
                } else {
                    spinButton.disableTracking();
                }
            }

            // add the caret symbol
            spinButton.append(Forms.createCaretMarkup(increase ? 'up' : 'down'));

            // enable/disable node tracking according to own enabled state
            self.on('group:enable', initializeTracking);
            initializeTracking();

            // register tracking event listeners to change the current value
            spinButton.on({
                'tracking:start tracking:repeat': function () { changeValue(increase ? smallStep : -smallStep); },
                'tracking:end tracking:cancel': function () { self.executeDelayed(function () { self.getInputNode().focus(); }); }
            });

            // enable/disable the button according to the current value
            self.registerUpdateHandler(function (value) {
                var enabled = increase ? (value < validator.getMax()) : (value > validator.getMin());
                Forms.enableNodes(spinButton, enabled);
            });

            return spinButton;
        }

        /**
         * Handles keyboard events in the text field.
         */
        function inputKeyDownHandler(event) {

            if (KeyCodes.hasModifierKeys(event)) { return; }

            switch (event.keyCode) {
            case KeyCodes.UP_ARROW:
                changeValue(smallStep);
                return false;
            case KeyCodes.DOWN_ARROW:
                changeValue(-smallStep);
                return false;
            case KeyCodes.PAGE_UP:
                changeValue(largeStep);
                return false;
            case KeyCodes.PAGE_DOWN:
                changeValue(-largeStep);
                return false;
            }
        }

        /**
         * Handles updates of the input text field
         */
        function updateHandler(value) {
            this.getInputNode().attr({ 'aria-valuenow': value, 'aria-valuetext': this.getInputNode().val() });
        }

        // initialization -----------------------------------------------------

        // add special marker class used to adjust formatting, add spin buttons
        this.getNode().addClass('spin-field').append(
            $('<div>').addClass('spin-wrapper').append(createSpinButton(true), createSpinButton(false))
        );

        // add ARIA attributes (note: role 'spinbutton' would be correct but doesn't work with VoiceOver)
        this.getInputNode().attr({ role: 'slider', 'aria-valuemin': validator.getMin(), 'aria-valuemax': validator.getMax() });

        // register event handlers
        this.getInputNode().on('keydown', inputKeyDownHandler);

        // register input field update handler
        this.registerUpdateHandler(updateHandler);

    } // class SpinField

    // exports ================================================================

    // derive this class from class TextField
    return TextField.extend({ constructor: SpinField });

});
