/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/tk/control/picture', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/group',
    'io.ox/office/tk/control/captionmixin',
    'io.ox/office/tk/control/widthmixin'
], function (Utils, Forms, Group, CaptionMixin, WidthMixin) {

    'use strict';

    // class Picture ============================================================

    /**
     * Creates a form control that displays an image.
     *
     * @constructor
     *
     * @extends Group
     * @extends CaptionMixin
     * @extends WidthMixin
     *
     * @param {String} source
     *  The source URL of the image to be displayed.
     *  TODO URL validator, fallback image?
     *  TODO add role = image for background case
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Group base class
     *  constructor, all image options supported by the method
     *  Forms.createImageMarkup(), and all options of the mix-in class
     *  WidthMixin. Additionally:
     *  @param {Object} [initOptions.backgroundMode=false]
     *      Whether the image should be displayed as a CSS background.
     */
    function Picture(source, initOptions) {

        var // whether the image should be displayed as a CSS background.
            backgroundMode = Utils.getBooleanOption(initOptions, 'backgroundMode', false),

            pictureHeight = Utils.getIntegerOption(initOptions, 'pictureHeight', 48),

            pictureWidth = Utils.getIntegerOption(initOptions, 'pictureWidth', 48),

            backgroundModeAttributes = {
                style: 'background-image:url(' + source + ');' + 'min-width:' + pictureWidth + 'px;' + 'height:' + pictureHeight + 'px;',
                role: 'img'
            },

            // the <img> element mark-up
            imageNode = backgroundMode ? $(Forms.createElementMarkup('div',
                Utils.extendOptions(initOptions, { attributes: backgroundModeAttributes }
            ))) : $(Forms.createImageMarkup(source, initOptions));

        // base constructors --------------------------------------------------

        Group.call(this, initOptions);
        CaptionMixin.call(this, imageNode);
        WidthMixin.call(this, imageNode, initOptions);

        // public methods -----------------------------------------------------

        /**
         * Returns the image element wrapped by this control group.
         *
         * @returns {jQuery}
         *  The image element wrapped by this control group, as jQuery object.
         */
        this.getImageNode = function () {
            return imageNode;
        };

        // sets new image source
        this.setSource = function (newSource) {
            source = newSource;
            if (backgroundMode) {
                imageNode.css('backgroundImage', 'url(' + source + ')');
            } else {
                imageNode.attr('src', source);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        this.getNode().addClass('image-group');
        imageNode.addClass('image-container');

        // apply the background image or insert the <img> mark-up
        this.addChildNodes(imageNode);

    } // class Label

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: Picture });

});

