/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/control/tablestylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/view/control/stylesheetpicker',
    'io.ox/office/text/dom',
    'gettext!io.ox/office/text'
], function (Utils, StyleSheetPicker, DOM, gt) {

    'use strict';

    // class TableStylePicker =================================================

    /**
     * A drop-down menu control for table style sheets.
     *
     * @constructor
     *
     * @extends StyleSheetPicker
     */
    function TableStylePicker(app) {

        var // self reference
            self = this,

            // table style sheets
            tableStyles = null;

        // base constructors --------------------------------------------------

        StyleSheetPicker.call(this, app, 'table', {
            width: 200,
            icon: 'docs-table-style',
            tooltip: /*#. tool tip: predefined styles for text tables */ gt('Table style'),
            gridColumns: 7,
            i18nModulePath: 'io.ox/office/text/resource/tablestylenames',
            smallerVersion: { css: { width: 50 }, hideLabel: true }
        });

        // private methods ----------------------------------------------------

        /**
         * Registers a new table that will be formatted delayed with a specific
         * table style sheet.
         *
         * @param {jQuery} table
         *  The new table to be formatted.
         *
         * @param {String} styleId
         *  The identifier of the table style sheet.
         */
        var updateTableFormatting = (function () {

            var // which tables in list items need formatting
                pendingTables = [],
                // background loop processing all pending tables
                formatTimer = null;

            // direct callback: called every time when updateTableFormatting() has been called
            function registerTable(tableNode, styleId) {
                pendingTables.push({ tableNode: tableNode, styleId: styleId });
            }

            // deferred callback: called once after the specified timeout
            function formatTables() {

                // check if the background loop is already running
                if (formatTimer) { return; }

                // create a new background loop that processes all pending tables
                formatTimer = self.repeatSliced(function () {

                    var // table info with table and style identifier of the next table to be formatted
                        tableInfo = null;

                    // find a pending table that is still in the drop-down menu (menu may have been cleared
                    // in the meantime), abort the background loop, if no more tables are available
                    while ((tableInfo = pendingTables.shift()) && !Utils.containsNode(self.getMenuNode(), tableInfo.tableNode)) {}
                    if (!tableInfo) { return Utils.BREAK; }

                    // update the formatting of the table
                    tableStyles.setElementAttributes(tableInfo.tableNode, {
                        styleId: tableInfo.styleId,
                        table: { exclude: ['lastRow', 'lastCol', 'bandsVert'] }
                    }, { preview: true });
                    tableInfo.tableNode.find(DOM.PARAGRAPH_NODE_SELECTOR).css({ margin: 0 });  // no margins in micro tables allowed
                });

                // forget reference to the timer, when all tables have been formatted
                formatTimer.always(function () { formatTimer = null; });
            }

            // create and return the debounced updateTableFormatting() method (with a delay of 10ms per table)
            return self.createDebouncedMethod(registerTable, formatTables, { delay: 10 });
        }()); // updateTableFormatting()

        /**
         * Initializes a new list item according to the table style sheet.
         */
        function createItemHandler(event, buttonNode, styleId) {

            var paragraphNode = DOM.createParagraphNode().css({ minWidth: 8, minHeight: 4 }),
                cellNode = DOM.createTableCellNode(paragraphNode, { plain: true }),
                cellMarkup = cellNode[0].outerHTML,
                rowMarkup = '<tr>' + Utils.repeatString(cellMarkup, 5) + '</tr>',
                tableNode = $('<table><tbody>' + Utils.repeatString(rowMarkup, 5) + '</tbody></table>');

            buttonNode.addClass('mini-caption').prepend(tableNode);
            updateTableFormatting(tableNode, styleId);
        }

        // initialization -----------------------------------------------------

        // initialize class members
        app.onInit(function () {
            tableStyles = app.getModel().getStyleCollection('table');
        });

        // register a handler that inserts a table element into each list item
        this.getMenu().on('create:item', createItemHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = tableStyles = null;
        });

    } // class TableStylePicker

    // exports ================================================================

    // derive this class from class StyleSheetPicker
    return StyleSheetPicker.extend({ constructor: TableStylePicker });

});
