/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/text/remoteselection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/text/utils/config',
    'io.ox/office/text/position',
    'io.ox/office/text/dom',
    'io.ox/office/text/format/characterstyles'
], function (Utils, BaseObject, Config, Position, DOM, CharacterStyles) {

    'use strict';

    // class RemoteSelection ==================================================

    /**
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     */
    function RemoteSelection(app, rootNode) {

        var // the local selection instance
            selection = null,

            // the display timeout id of the collaborative overlay
            overlayTimeoutId = null;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Calculate physical positions from logical selection.
         *
         * @param {DOM.Point} anchorPoint
         *  the selection start point
         *
         * @param {DOM.Point} focusPoint
         *  the selection end point.
         *
         * @returns {Object | null}
         *  An selection object containing physical start and end coordinates.
         *  - {Number} Object.anchor - selection start absolute coordinate in pixels
         *  - {Number} Object.focus -  selection end absolute coordinate in pixels
         */
        function calculatePhysicalPosition(anchorPoint, focusPoint) {

            // quit if no logical selection is available
            if (!anchorPoint || !focusPoint) { return null; }

            var // get logical start and end positions for restoring text selection later
                selectionStart = selection.getStartPosition(),
                selectionEnd = selection.getEndPosition(),
                backwards = selection.isBackwards(),
                zoom = app.getView().getZoomFactor() / 100;

            // use decoy span technique and get absolute physical positions of a selection point
            function getPhysicalPosition(point) {

                // ignore this current user's own selection points of course
                if (!point) { return null; }

                var caretSpan = $('<span>').addClass('collaborative-caret').text('|'),
                    cursorElement = $(point.node.parentNode),
                    cursorElementLineHeight = parseFloat(cursorElement.css('line-height'));

                // break cursor element on the text offset
                if (point.offset === 0) {
                    cursorElement.before(caretSpan);
                } else {
                    DOM.splitTextSpan(cursorElement, point.offset, {append: true});
                    cursorElement.after(caretSpan);
                }

                // create caret overlay and calculate its position
                var caretTop = (caretSpan.offset().top - rootNode.offset().top) / zoom  - cursorElementLineHeight  + caretSpan.outerHeight(),
                    caretLeft = (caretSpan.offset().left - rootNode.offset().left) / zoom;

                // restore original state of document
                caretSpan.remove();

                if (point.offset > 0) { CharacterStyles.mergeSiblingTextSpans(cursorElement, true); }

                return { top: caretTop, left: caretLeft};
            }

            var position = { anchor: getPhysicalPosition(anchorPoint), focus: getPhysicalPosition(focusPoint) };

            // only reset text selection if the page node is the active element.
            // Otherwise, there is a overlay which gets the focus (BUG #36469)
            if (DOM.isPageNode(document.activeElement)) {
                // return original position of browser cursor after merge
                selection.setTextSelection(selectionStart, selectionEnd, { userDataUpdateHandler: true, backwards: backwards });
            }

            return position;
        }

        /**
         * Draw overlays for a specific remote user's selection.
         *
         * @param {Object} client
         *  Descriptor of a client editing this document.
         */
        function renderCollaborativeSelection(client) {

            var // the selection of the client
                userSelection = Utils.getObjectOption(client.userData, 'selection');

            // draw nothing and quit if:
            // - user is the current user himself
            // - no user data/selection is available.
            // - selection is a drawing selection
            if (!client.remote || !userSelection || (userSelection.type === 'drawing')) {
                return;
            }

            var selectionAnchorPoint = Position.getDOMPosition(rootNode, userSelection.start),
                selectionFocusPoint = Position.getDOMPosition(rootNode, userSelection.end);

            if (!selectionAnchorPoint || !selectionFocusPoint) { return; }

            var username = client.userName,
                overlay = rootNode.find('.collaborative-overlay'),
                usernameOverlay = (_.browser.IE === 10) ? document.createElementNS('http://www.w3.org/2000/svg', 'svg') : document.createElement('div'),
                caretOverlay = $('<div>').addClass('collaborative-cursor'),
                caretHandle = $('<div>').addClass('collaborative-cursor-handle'),
                className = 'user-' + client.colorIndex,
                startNode = $(selectionAnchorPoint.node.parentNode),
                endNode = $(selectionFocusPoint.node.parentNode),
                physicalPosition = calculatePhysicalPosition(selectionAnchorPoint, selectionFocusPoint),
                isRangeSelection = !_.isEqual(physicalPosition.anchor, physicalPosition.focus);

            // draw user cursors and names
            usernameOverlay.setAttribute('class', 'collaborative-username ' + className);
            $(usernameOverlay).css('top', endNode.css('line-height'));
            if (_.browser.IE === 10) { // create SVG text node and calculate its width (no text wrap support in SVG)
                var txtElem = document.createElementNS('http://www.w3.org/2000/svg', 'text');
                $(txtElem).attr({ x: 5, y: 13, style: 'font-size:12px; fill:white'});
                txtElem.appendChild(document.createTextNode(username));
                usernameOverlay.appendChild(txtElem);
                $(usernameOverlay).attr({ width: (username.length * 8) + 'px', style: 'height:2em' });
            } else {
                $(usernameOverlay).text(username);
            }
            caretOverlay.css({ top: physicalPosition.focus.top, left: physicalPosition.focus.left, height: parseFloat(endNode.css('line-height'))}).addClass(className);
            caretOverlay.append(usernameOverlay, caretHandle);
            if (!_.isBoolean(client.selectionChanged) || client.selectionChanged) { $(usernameOverlay).show(); }  // 36800
            caretHandle.hover(function () { $(usernameOverlay).show(); }, function () { $(usernameOverlay).hide(); });
            overlay.append(caretOverlay);

            // draw collaborative selection area highlighting if we have an 'range' selection
            if (isRangeSelection) {

                var zoom = app.getView().getZoomFactor() / 100,
                    startParentNode = startNode.parent(),
                    endParentNode = endNode.parent(),
                    selectionOverlayGroup = $('<div>').addClass('collaborative-selection-group'),
                    highlightWidth = Math.max(startParentNode.width(), endParentNode.width()),
                    rootNodePos = rootNode.offset(),
                    pageContentNodePos = DOM.getPageContentNode(rootNode).offset(),
                    startTop = physicalPosition.anchor.top,
                    startLeft = physicalPosition.anchor.left,
                    endTop = physicalPosition.focus.top,
                    endLeft = physicalPosition.focus.left,
                    startNodeCell = startNode.closest('td'),
                    endNodeCell = endNode.closest('td'),
                    isTableSelection = startNodeCell.length && endNodeCell.length && (!startNodeCell.is(endNodeCell)),
                    svgNamespace = 'http://www.w3.org/2000/svg';

                // special handling of pure table selection
                if (isTableSelection) {

                    // handle special mega cool firefox cell selection
                    if (userSelection.type === 'cell') {

                        var ovStartPos = startNodeCell.offset(),
                            ovEndPos = endNodeCell.offset(),
                            ovWidth = (ovEndPos.left  - ovStartPos.left) / zoom + endNodeCell.outerWidth(),
                            ovHeight = (ovEndPos.top  - ovStartPos.top) / zoom + endNodeCell.outerHeight(),
                            ov = (_.browser.IE === 10) ? document.createElementNS(svgNamespace, 'svg'): document.createElement('div');

                        $(ov).css({
                            top: (ovStartPos.top - rootNodePos.top) / zoom,
                            left: (ovStartPos.left - rootNodePos.left) / zoom,
                            width: ovWidth,
                            height: ovHeight
                        });

                        selectionOverlayGroup.append(ov);

                    } else { // normal table selection (Chrome, IE): iterate cells between start and end

                        var cells = $(Utils.findFarthest(rootNode, endNodeCell, 'table')).find('td'),
                            cellsToHighlight = cells.slice(cells.index(startNodeCell), cells.index(endNodeCell) + 1);

                        cellsToHighlight.each(function () {

                            var cellOv = (_.browser.IE === 10) ? document.createElementNS(svgNamespace, 'svg'): document.createElement('div'),
                                offset = $(this).offset();

                            $(cellOv).css({
                                top: (offset.top - rootNodePos.top) / zoom,
                                left: (offset.left - rootNodePos.left) / zoom,
                                width: $(this).outerWidth(),
                                height: $(this).outerHeight()
                            });

                            selectionOverlayGroup.append(cellOv);

                        });
                    }
                } else { // paragraph / mixed selection

                    var startHeight = parseFloat(startNode.css('line-height')),
                        endHeight = parseFloat(endNode.css('line-height')),
                        startBottom = startTop + startHeight,
                        endBottom = endTop + endHeight,
                        isSingleLineSelection = startBottom === endBottom;

                    // selection area is in a line
                    if (isSingleLineSelection) {

                        var selectionOverlay = (_.browser.IE === 10) ? document.createElementNS(svgNamespace, 'svg'): document.createElement('div');
                        $(selectionOverlay).css({ top: Math.min(startTop, endTop), left: startLeft, width: endLeft - startLeft, height: Math.max(startHeight, endHeight)});
                        selectionOverlayGroup.append(selectionOverlay);

                    } else { // multi line selection area

                        var startPrevNode = $(Utils.findPreviousSiblingNode(startNode)),
                            endPrevNode = $(Utils.findPreviousSiblingNode(endNode)),
                            isListSelection = DOM.isListLabelNode(startPrevNode) || DOM.isListLabelNode(endPrevNode);

                        // start and end node are empty lines (paragraphs)
                        if (!highlightWidth) { highlightWidth = rootNode.width(); }

                        var headOverlay = document.createElement('div'),
                            bodyOverlay = document.createElement('div'),
                            tailOverlay = document.createElement('div');

                        if (_.browser.IE === 10) {
                            headOverlay = document.createElementNS(svgNamespace, 'svg');
                            bodyOverlay = document.createElementNS(svgNamespace, 'svg');
                            tailOverlay = document.createElementNS(svgNamespace, 'svg');
                        }

                        $(headOverlay).css({
                            top: startTop,
                            left: startLeft,
                            width: isListSelection ? (startParentNode.offset().left - rootNodePos.left) / zoom  + startParentNode.width() - startLeft + startPrevNode.width() :
                                (startParentNode.offset().left - rootNodePos.left) / zoom  + startParentNode.width() - startLeft,
                            height: startHeight
                        });

                        $(bodyOverlay).css({
                            top: startTop + startHeight,
                            left: (isListSelection ? pageContentNodePos.left - rootNodePos.left : Math.min(startParentNode.offset().left, endParentNode.offset().left) - rootNodePos.left) / zoom,
                            width: isListSelection ? rootNode.width() : highlightWidth,
                            height: endTop - startTop - startHeight
                        });

                        $(tailOverlay).css({
                            top: endTop,
                            left: isListSelection ? (endNode.offset().left - rootNodePos.left) / zoom - endPrevNode.width() :
                                (endParentNode.offset().left - rootNodePos.left) / zoom,
                            width: endLeft - ((endParentNode.offset().left - rootNodePos.left) / zoom),
                            height: endHeight
                        });

                        selectionOverlayGroup.append(headOverlay, bodyOverlay, tailOverlay);
                    }
                }

                _.each(selectionOverlayGroup.children(), function (overlayElement) {
                    overlayElement.setAttribute('class', 'selection-overlay ' + className);
                });

                overlay.append(selectionOverlayGroup);
            }
            // show user name for a second
            clearTimeout(overlayTimeoutId);

            overlayTimeoutId = setTimeout(function () { overlay.find('.collaborative-username').fadeOut('fast'); }, 3000);
        }

        // public methods -----------------------------------------------------

        /**
         * Draw overlays for the selections of all remote clients.
         *
         * @returns {RemoteSelection}
         *  A reference to this instance.
         */
        this.renderCollaborativeSelections = function () {
            if (Config.SHOW_REMOTE_SELECTIONS) {
                // first, clear all old selection nodes from the DOM
                rootNode.find('.collaborative-overlay').children().remove();
                // render all remote selections
                _.each(app.getActiveClients(), renderCollaborativeSelection);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members when application is alive
        app.onInit(function () {
            selection = app.getModel().getSelection();
        });

        // render selection if the list of active clients has changed
        this.listenTo(app, 'docs:users', _.bind(this.renderCollaborativeSelections, this));

        // destroy all class members
        this.registerDestructor(function () {
            app = rootNode = selection = null;
        });

    } // class RemoteSelection

    // exports ================================================================

    return BaseObject.extend({ constructor: RemoteSelection });

});
