/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/pageLayout',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/tk/object/timermixin',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/editframework/utils/attributeutils',
     'io.ox/office/text/dom',
     'io.ox/office/text/position',
     'io.ox/office/text/format/characterstyles',
     'io.ox/office/text/table'
    ], function (Utils, Dialogs, TriggerObject, TimerMixin, DrawingFrame, AttributeUtils, DOM, Position, CharacterStyles, Table) {

    'use strict';

    // private static functions ===============================================


    // class PageLayout ========================================================

    /**
     * An instance of this class represents a page layout handler in the edited document.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     *
     * @param {TextModel} model
     *  The text model instance.
     */
    function PageLayout(app, rootNode, model) {

        var // self reference
            self = this,
            // container for all style sheets of all attribute families
            documentStyles,
            // node reference containing content nodes
            $pageContentNode = DOM.getPageContentNode(rootNode),
            // style values for page metrics
            pageStyles,
            drawingStyles,
            pageAttributes,
            pageHeight,
            pageMaxHeight,
            pagePaddingLeft,
            pagePaddingRight,
            pagePaddingTop,
            pagePaddingBottom,
            pageWidth,
            footerMargin = 0,
            headerMargin = 0,
            firstPage = false,
            evenOddPages = false,
            // overlay layer element for field tooltip
            $tooltipOverlay = $(),
            // node reference for holding template headers and footers ready for cloning
            $headerFooterPlaceHolder = $('<div>').addClass('header-footer-placeholder').attr('contenteditable', _.browser.WebKit ? 'false' : '').css('display', Utils.SMALL_DEVICE ? 'none' : '' ),
            // collection of headers&footers instances
            headerFooterCollection = [],
            // page break nodes collection
            pageBreaksCollection = [],
            // wrapper object of first header in document
            $firstHeaderWrapper = $(),
            // wrapper object of last footer in document
            $lastFooterWrapper = $(),
            // stored height values of different header&footer types
            heightOfDefHeader = 0,
            heightOfDefFooter = 0,
            heightOfFirstHeader = 0,
            heightOfFirstFooter = 0,
            heightOfEvenHeader = 0,
            heightOfEvenFooter = 0,
            // pageHeights with different header&footer heights
            firstPageHeight = 0,
            evenPageHeight = 0,
            defPageHeight = 0,
            // store the value of current type of page: first, even, default
            currentPageTypeHeight = 0,
            contentEditableTrueNodes = [],
            // global variable containing info for fields update
            pageCount = 1;

        rootNode.append($headerFooterPlaceHolder);

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Initial looping through current 'page content DOM' and doing addition of heights of elements to calculate page breaks positions.
         * This function is called only on document loading, and that's why it loops trough all DOM nodes
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.triggerEvent=false]
         *      If set to true, an event will be triggered after the page breaks are inserted completely.
         */
        function initialPageBreaks(options) {

            var // whether an event shall be triggered after all page breaks are included
                triggerEvent = Utils.getBooleanOption(options, 'triggerEvent', false);

            if (Utils.SMALL_DEVICE || model.isDraftMode()) { return; }
            return Utils.takeTime('pageLayout.initialPageBreaks(): ', function () {
                var
                    processingNodes,
                    contentHeight = 0,
                    elementHeightWithoutMB = 0,
                    elementHeight = 0,
                    nodeHelper,
                    zoomFactor,
                    floatingDrawings,
                    fDrawingHeight,
                    fDrawingWidth = 0,
                    accHeightFromPrevDrawing = 0,
                    heightOfRectPartHelper = 0,
                    elementSpan,
                    offsetPosition = [],
                    newlySplittedSpan,
                    markup = '',
                    pageBreak,
                    elMarginBottom,
                    elMarginTop,
                    diffMargin = 0,
                    prevElMarginBottom = 0,
                    prevPageBreaks,
                    lastPageBreak,
                    $drawing,
                    drawingHeightAndOffset,
                    $element,
                    iterationPromise = null,
                    currentPageNumber = 0;

                if (!pageAttributes) {
                    initializePageAttributes();
                }

                // hack for tablets with re-written reduced css padding after document is loaded
                if (app.getView().getContentRootNode().hasClass('small-device')) {
                    pagePaddingLeft = parseFloat($pageContentNode.parent().css('padding-left'));// 18.898;
                    pagePaddingRight = pagePaddingLeft;
                    pageWidth = $pageContentNode.parent().outerWidth();
                } else {
                    pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001);
                    pagePaddingRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001);
                    pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001);
                }

                if ($firstHeaderWrapper.length === 0) {
                    toggleContentEditableTrueNodes();
                    $firstHeaderWrapper = self.generateHeaderElementHtml();
                    updateStartHeaderFooterStyle();
                    disableChangeTrackInInactiveNodes();
                }

                // check if there are already inserted page breaks for fast display of first pages, and continue after them
                prevPageBreaks = $pageContentNode.find('.page-break');
                if (prevPageBreaks.length > 0) {
                    lastPageBreak = prevPageBreaks.last();
                    if (lastPageBreak.parent().is('td')) {
                        contentHeight = lastPageBreak.parentsUntil('.pagecontent', '.tb-split-nb').height() - (lastPageBreak.parentsUntil('.pagecontent', '.pb-row').next().position().top - lastPageBreak.parentsUntil('.pagecontent', '.tb-split-nb').position().top);
                        processingNodes = lastPageBreak.parentsUntil('.pagecontent', '.tb-split-nb').nextAll('.p, table');
                    } else if (lastPageBreak.parent().hasClass('p')) {
                        contentHeight = lastPageBreak.parent().height() - lastPageBreak.next().position().top;
                        processingNodes = lastPageBreak.parent().nextAll('.p, table');
                    } else if (lastPageBreak.next().hasClass('manual-page-break')) {
                        contentHeight = lastPageBreak.next().height();
                        processingNodes = lastPageBreak.next().nextAll('.p, table');
                    } else {
                        processingNodes = lastPageBreak.nextAll('.p, table');
                    }
                } else {
                    processingNodes = $pageContentNode.children('.p, table');
                }
                // if first call of initialPageBreaks is aborted or interrupted, update page breaks collection
                if (!pageBreaksCollection || pageBreaksCollection.length === 0) {
                    pageBreaksCollection = prevPageBreaks;
                    pageCount = pageBreaksCollection.length + 1;
                }
                currentPageNumber = self.getPageNumber(processingNodes.first());
                if (currentPageNumber === 1 && firstPageHeight !== 0) {
                    pageMaxHeight = firstPageHeight;
                    currentPageTypeHeight = 'first';
                } else if (currentPageNumber % 2 === 0 && evenPageHeight !== 0) {
                    pageMaxHeight = evenPageHeight;
                    currentPageTypeHeight = 'even';
                } else if (defPageHeight !== 0) {
                    pageMaxHeight = defPageHeight;
                    currentPageTypeHeight = 'def';
                }

                // triggering event, so that current scroll position can be saved
                if (triggerEvent) { model.trigger('pageBreak:before'); }

                iterationPromise = self.iterateArraySliced(processingNodes, function (element, iterator) {
//                    console.warn('pageMaxHeight: ', pageMaxHeight);
//                    console.warn('  -> contentHeight: ', contentHeight);

                    $element = $(element);
                    // processing node with floated drawing(s) inside of it
                    fDrawingHeight = 0;
                    floatingDrawings = $element.find('.drawing.float');
                    if (floatingDrawings.length > 0) {
                        _.each(floatingDrawings, function (drawing) {
                            $drawing = $(drawing);
                            drawingHeightAndOffset = $drawing.outerHeight(true) + ($drawing.position().top / zoomFactor);
                            if ($drawing.outerWidth(true) !== $pageContentNode.width()) {
                                if (drawingHeightAndOffset > fDrawingHeight) {
                                    fDrawingHeight = drawingHeightAndOffset;
                                    fDrawingWidth = $drawing.outerWidth(true);
                                }
                            }
                        });
                    }
                    if (accHeightFromPrevDrawing > fDrawingHeight) {
                        fDrawingHeight = 0;
                    }
                    elementHeightWithoutMB = element.offsetHeight;
                    elMarginBottom = parseInt($element.css('margin-bottom'), 10);
                    elMarginTop = parseInt($element.css('margin-top'), 10);
                    if (elMarginTop > prevElMarginBottom) { // el has margin top that is greater than margin bottom of prev el, include diff in page height
                        diffMargin =  elMarginTop - prevElMarginBottom;
                        elementHeightWithoutMB += diffMargin;
                    }
                    elementHeight = elementHeightWithoutMB + elMarginBottom;
                    prevElMarginBottom = elMarginBottom;
                    if (DOM.isManualPageBreakNode(element)) {
                        if ($element.prev().length > 0 || $element.find('.ms-hardbreak-page').length > 0) {
                            contentHeight = manualInsertionOfPageBreak(element, contentHeight, elementHeight);
                        } else { // first element in document, can't have page break before - can happen on delete all where last is manual pb paragraph
                            $element.removeClass('manual-page-break');
                        }
                    }
                    // if current content height is bellow max page height;
                    if (contentHeight + elementHeight <= pageMaxHeight  && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                            contentHeight += elementHeight;
                        }
                        if (accHeightFromPrevDrawing > elementHeight) {
                            accHeightFromPrevDrawing -= elementHeight;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                    // for last element that can fit on page we need to omit margin bottom
                    } else if (contentHeight + elementHeightWithoutMB <= pageMaxHeight  && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                            contentHeight += elementHeightWithoutMB;
                        }
                        if (accHeightFromPrevDrawing > elementHeightWithoutMB) {
                            accHeightFromPrevDrawing -= elementHeightWithoutMB;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                    } else {
                        prevElMarginBottom = 0;
                        if ($element.hasClass('p') && ($element.text().length > 10 || $element.find('.drawing.inline').length > 0) ){ //&& floatingDrawings.length < 1) {
                            // paragraph split preparation
                            if (accHeightFromPrevDrawing > 0 && fDrawingWidth > 0) {
                                offsetPosition = getOffsetPositionFromElement(element, contentHeight, false, accHeightFromPrevDrawing, fDrawingWidth);
                            } else {
                                offsetPosition = getOffsetPositionFromElement(element, contentHeight, false);
                            }
                            if (offsetPosition && offsetPosition.length > 0) { // if offset position is returned
                                if (offsetPosition[0] === 0) { // if offset is 0, no need for p split, insert pb before element
                                    offsetPosition.shift();
                                    contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                                    if (offsetPosition.length > 0) { contentHeight = 0; }
                                }
                                if (offsetPosition.length > 0) {
                                    elementSpan = $element.children('span').first();
                                    while (elementSpan.text().length < 1 || !(elementSpan.is('span'))) {
                                        elementSpan = elementSpan.next();
                                    }
                                    if (elementSpan && elementSpan.length > 0) { //fix for #32563, if div.p contains only empty spans
                                        _.each(offsetPosition, function (offset) {
                                            var elTextLength = $(elementSpan).text().length;
                                            while (elTextLength > 0 && elTextLength <= offset) {
                                                offset -= elTextLength;
                                                elementSpan = elementSpan.next();
                                                while (!elementSpan.is('span')) { // we might fetch some inline div nodes, like div.tab or div.linebreak
                                                    elementSpan = elementSpan.next();
                                                }
                                                elTextLength = $(elementSpan).text().length;
                                            }
                                            newlySplittedSpan = DOM.splitTextSpan(elementSpan, offset, {append: true});
                                            newlySplittedSpan.addClass('break-above-span');
                                            $element.addClass('contains-pagebreak');
                                            elementSpan = newlySplittedSpan;
                                        });
                                    }
                                    zoomFactor = app.getView().getZoomFactor() / 100;
                                    pageBreak = $();
                                    var arrBreakAboveSpans = $element.find('.break-above-span'),
                                        upperPartDiv = 0,
                                        totalUpperParts = 0,
                                        diffPosTopAndPrev = 0,
                                        elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs

                                    _.each(arrBreakAboveSpans, function (breakAboveSpan) {
                                        diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                                        upperPartDiv = $(breakAboveSpan).position().top / zoomFactor - diffPosTopAndPrev;
                                        totalUpperParts += upperPartDiv;
                                        contentHeight += upperPartDiv;
                                        nodeHelper = $element.prev();
                                        while (nodeHelper.hasClass('page-break')) {
                                            nodeHelper = nodeHelper.prev();
                                        }
                                        nodeHelper.addClass('last-on-page');
                                        markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit, elementLeftIndent);
                                        $(breakAboveSpan).before(markup);
                                        pageBreak = $(breakAboveSpan).prev();
                                        contentHeight = 0;
                                    });
                                    contentHeight = elementHeight - totalUpperParts;
                                }
                            } else {
                                contentHeight = insertPageBreaksBetweenDrawings(element, elementHeight, contentHeight, offsetPosition, zoomFactor);
                                //contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                            }

                        } else if (DOM.isTableNode(element) && element.rows.length > 1) { // table split
                            contentHeight = insertPageBreaksInTable(element, elementHeight, contentHeight, { 'initial' : true, 'diffMargin' : diffMargin });
                        } else { // default insertion of page break between elements
                            contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                        }
                    }
                    if (DOM.isNodeWithPageBreakAfterAttribute(element)) {
                        contentHeight = insertManualPageBreakAfter(element, contentHeight);
                    }

                    if (fDrawingHeight > 0) { // store the diff between floated image height and height of containing p, for next iteration
                        accHeightFromPrevDrawing = fDrawingHeight - elementHeight;
                    }
                    //at the last page, we add extra padding to fill in to the size of max page height
                    if (processingNodes.length === iterator + 1) {
                        $pageContentNode.css({
                            'padding-bottom': pageMaxHeight - contentHeight
                        });
                        $element.addClass('last-on-page');
                    }

                }, { delay: 'immediate' })
                .done(function () {
                    if ($lastFooterWrapper.length === 0) {
                        $lastFooterWrapper = self.generateFooterElementHtml();
                    }

                    pageBreaksCollection = $pageContentNode.find('.page-break');
                    // store indexes of page breaks as values for page numbers
                    _.each(pageBreaksCollection, function(pageBreakInCollection, index) {
                        $(pageBreakInCollection).data('page-num', index + 1);
                    });
                    pageCount = pageBreaksCollection.length + 1;

                    populateHeaderFooterPaginationClasses(pageBreaksCollection);
                    replaceAllTypesOfHeaderFooters();
                    updatePageCountFields();
                    updatePageNumberFields();

                    if (triggerEvent) {
                        model.trigger('pageBreak:after');
                    }
                });
                return iterationPromise.promise();
            });
        }

        /**
         * After inserting of page breaks is done, this method will insert correct types of headers&footers
         * inside page breaks. There are four types of headers&footers, listed by priority from highest to lowest:
         *  first   - for the first page only, if defined
         *  even    - for even pages, if defined
         *  default - fallback for all types - lowest prio. Odd pages uses this type
         * Highest priority header/footer, if is defined in document, will overwrite all bellow him.
         */
        function replaceAllTypesOfHeaderFooters() {
            var groupOfHeaderFooterToUpdate = rootNode.find('.page-break, .header-wrapper, .footer-wrapper'),
                placeholderHeader = self.getHeaderFromPlaceholder(),
                placeholderFooter = self.getFooterFromPlaceholder(),
                typesOfHeader = [],
                typesOfHeaderInternal = [],
                typesOfFooter = [],
                typesOfFooterInternal = [],
                selector = '',
                // if there are images inside head/foot, might happen that they are replaced before image resource is loaded in browser
                placeholdersOfImages;

            if (firstPage || app.isODF()) { // for ODF docs this document attribute isn't set
                typesOfHeader.push('header_first');
                typesOfHeaderInternal.push('.header.first');
                typesOfFooter.push('footer_first');
                typesOfFooterInternal.push('.footer.first');
                selector += '.first';
            }
            if (evenOddPages || app.isODF()) { // for ODF docs this document attribute isn't set
                typesOfHeader.push('header_even');
                typesOfHeaderInternal.push('.header.even');
                typesOfFooter.push('footer_even');
                typesOfFooterInternal.push('.footer.even');
                if (selector.length > 0) {
                    selector += ', .even';
                } else {
                    selector += '.even';
                }
            }

            if (app.isODF()) {
                selector = '';
            }

            // default
            if (placeholderHeader.length > 0) {
                groupOfHeaderFooterToUpdate.find('.header').not(selector).replaceWith(placeholderHeader);
            }
            if (placeholderFooter.length > 0) {
                groupOfHeaderFooterToUpdate.find('.footer').not(selector).replaceWith(placeholderFooter);
            }
            // all other types
            for (var i = 0, typesLength = typesOfHeader.length; i < typesLength; i += 1) {
                placeholderHeader = self.getHeaderFromPlaceholder(typesOfHeader[i]);
                if (placeholderHeader.length > 0) {
                    groupOfHeaderFooterToUpdate.find(typesOfHeaderInternal[i]).replaceWith(placeholderHeader);
                }
            }

            for (var i = 0, typesLength = typesOfFooter.length; i < typesLength; i += 1) {
                placeholderFooter = self.getFooterFromPlaceholder(typesOfFooter[i]);
                if (placeholderFooter.length > 0) {
                    groupOfHeaderFooterToUpdate.find(typesOfFooterInternal[i]).replaceWith(placeholderFooter);
                }
            }


            // update absolute positioned drawings inside header/footer
            //model.getDrawingLayer().handleHeaderFooterDrawings(groupOfHeaderFooterToUpdate.find(DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR));

            // empty each placeholder inside image in header/footer, to trigger update formatting (and, replacing with real image resource)
            placeholdersOfImages = groupOfHeaderFooterToUpdate.find('.header, .footer').find('.drawing').children('.placeholder');
            _.each(placeholdersOfImages, function(placeholder) {
                $(placeholder).empty();
                drawingStyles.updateElementFormatting($(placeholder).parent('.drawing'));
            });

            // find text frames with canvas border inside header/footer and repaint them (after cloning they have to be manually repainted) #36576
            model.drawCanvasBorders(groupOfHeaderFooterToUpdate.find('.header, .footer').find(DrawingFrame.BORDER_NODE_SELECTOR));
        }

        /**
         * Initializing page attributes needed for calculation of page breaks.
         * In some scenarios, like i.e. loading from local storage,
         * initialPageBreaks method is not called, and variables are not initialized for later usage.
         *
         */
        function initializePageAttributes() {
            documentStyles = model.getDocumentStyles();
            pageStyles = documentStyles.getStyleCollection('page');
            drawingStyles = documentStyles.getStyleCollection('drawing');
            pageAttributes = pageStyles.getElementAttributes(rootNode);
            pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001);
            pagePaddingRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001);
            pagePaddingTop = Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 0.001);
            pagePaddingBottom = Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 0.001);
            pageHeight = Utils.convertHmmToLength(pageAttributes.page.height, 'px', 0.001);
            pageMaxHeight = pageHeight - pagePaddingTop - pagePaddingBottom;
            pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001);
            headerMargin = Utils.convertHmmToLength(pageAttributes.page.marginHeader, 'px', 0.001);
            footerMargin = Utils.convertHmmToLength(pageAttributes.page.marginFooter, 'px', 0.001);
            firstPage = pageAttributes.page.firstPage;
            evenOddPages = pageAttributes.page.evenOddPages;
        }

        /**
         * In a element (paragraph) where there are drawing objects, but no text spans to split,
         * we insert page breaks directly before desired drawing.
         * For now, only inline type drawings are supported.
         *
         * @param {Node|jQuery} element
         *  Element which is being processed
         *
         * @param {Number} elementHeight
         *  Current height value of all processed nodes
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Array} offsetPosition
         *  possible position(s) where to split text in paragraph
         *
         * @param {Number} zoomFactor
         *  Current zoom factor in document, can be number value, or undefined on doc loading
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after processing, in px
         *
         */
        function insertPageBreaksBetweenDrawings(element, elementHeight, contentHeight, offsetPosition, zoomFactor) {
            var
                $element = $(element),
                $elDrawings = $element.find('.drawing'),
                numOfPageBreaks = ~~((elementHeight + contentHeight) / pageMaxHeight), //get int value how many times is element bigger than page
                multiplicator,
                upperPartDiv = 0,
                totalUpperParts = 0,
                diffPosTopAndPrev = 0,
                numOfInserts = 0,
                pageBreak = $(),
                elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs

            if ($elDrawings.length > 0 && offsetPosition.length < numOfPageBreaks) {
                multiplicator = numOfPageBreaks;
                multiplicator -= 1; // decrement multiplier for 0 based values

                Utils.iterateArray($elDrawings, function (drawing) {
                    if (numOfInserts === numOfPageBreaks) {
                        return Utils.BREAK;
                    }
                    if ($(drawing).hasClass('inline')) {
                        diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                        upperPartDiv = $(drawing).position().top / (zoomFactor || 1) - diffPosTopAndPrev;

                        if (upperPartDiv + $(drawing).outerHeight() > (pageMaxHeight - contentHeight) && multiplicator > -1) {
                            totalUpperParts += upperPartDiv;
                            contentHeight += upperPartDiv;
                            $(drawing).before(generatePagebreakElementHtml(contentHeight, _.browser.WebKit, elementLeftIndent));
                            multiplicator -= 1;
                            numOfInserts += 1;
                            pageBreak = $(drawing).prev();
                            contentHeight = 0;
                            $element.addClass('contains-pagebreak');
                            $(drawing).addClass('break-above-drawing');
                        }
                    }
                });
                contentHeight = elementHeight - totalUpperParts;
            } else {
                contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
            }

            return contentHeight;
        }

        /**
         *  Internal function for page layout rendering!
         *  Creates invisible helper node to calculate position(s) in text where paragraph should be split.
         *  Also stores position of all rest new line beginnings into node with jQuery data()
         *  Words and spaces are occurring in pairs, sequentially. Space can be one character, or more grouped, then nbsp is used
         *
         * @param {Node|jQuery} element
         *  Element whose offset is calculated
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Boolean} modifyingElement
         *  If the current processing node is modified or untouched
         *
         * @param {Number} partHeight - optional
         *  If there is floating drawing from previous paragraphs, use height to create dummy div
         *
         * @param {Number} partWidth - optional
         *  If there is floating drawing from previous paragraphs, use width to create dummy div
         *
         * @returns {Array} offsetPosition
         *  Calculated position(s) in text where paragraph should be split
         *
         */
        function getOffsetPositionFromElement(element, contentHeight, modifyingElement, partHeight, partWidth) {

            var
                offsetPosition = [],
                elementSpan,
                contentHeightCached = contentHeight,
                $newElement,
                $element = $(element),
                elementHeight = $element.height(),
                elementWidth = $element.width(),
                elementLineHeightStyle = $element.css('line-height'),
                oxoPosInPar,
                $elSpan,
                numOfPageBreaks = 0,
                cachedDataForLineBreaks = [],
                cachedDataForLineBreaksLength,
                helperOffsetArray = [],
                elSpanPositionTop,
                charLength = 0,
                //elSpans,
                $elDrawings, //paragraph contains image(s)
                //drawingTopPos,
                //drawingBottomPos,
                cachedPosTop,
                markup,
                splitPosForbidenRanges = [],
                allowedValueInRange;

            function calculateOffsetPosition() {
                var diff = 0,
                    factorIncr = 1,
                    prevPositionTop,
                    prevOxoPos,
                    tempContHeight = contentHeightCached;

                helperOffsetArray = [];

                if (elementHeight > pageMaxHeight) { // more than one insertion of pb
                    numOfPageBreaks = ~~((elementHeight + contentHeightCached) / pageMaxHeight); // how many times we insert page breaks in table
                    if (offsetPosition.length === 0 && cachedDataForLineBreaksLength > 0) {
                        Utils.iterateArray(cachedDataForLineBreaks, function (el, iterator) {
                            if (el.distanceTop + diff > (pageMaxHeight * factorIncr) - tempContHeight && (prevPositionTop && prevPositionTop <= (pageMaxHeight * factorIncr) - tempContHeight)) {
                                diff += ((pageMaxHeight * factorIncr) - tempContHeight) - prevPositionTop; // difference that will be shifted down (image for ex.)
                                factorIncr += 1;
                                offsetPosition.push(prevOxoPos);
                            } else if (iterator === 0 && (el.distanceTop + diff > (pageMaxHeight * factorIncr) - tempContHeight)) {// first row cannot fit into free space, it is also shifted down (pb on begining, pos 0)
                                offsetPosition.push(0);
                                tempContHeight = 0;
                            }
                            prevPositionTop = el.distanceTop + diff;
                            prevOxoPos = el.oxoPosInPar;
                        });
                        if (offsetPosition.length < numOfPageBreaks) {
                            // now we know space left is not enough for last line, so we put page break before it
                            offsetPosition.push(cachedDataForLineBreaks[cachedDataForLineBreaksLength - 1].oxoPosInPar);
                        }
                    }

                    // normalize values: ex. [150, 400] to [150, 250]
                    _.each(offsetPosition, function (offPos, iterator) {
                        if (iterator > 0) {
                            offPos -=  offsetPosition[iterator - 1];
                        }
                        helperOffsetArray.push(offPos);
                    });
                    offsetPosition = helperOffsetArray.slice();
                } else {
                    //if its not multi paragraph split
                    if (offsetPosition.length === 0 && cachedDataForLineBreaksLength > 0) {
                        Utils.iterateArray(cachedDataForLineBreaks, function (el, iterator) {
                            if (el.distanceTop <= (pageMaxHeight - contentHeightCached)) {
                                offsetPosition.push(el.oxoPosInPar);
                                return Utils.BREAK;
                            } else if (iterator === 0 && (el.distanceTop > (pageMaxHeight - contentHeightCached))) {// first row (backwards) cannot fit into free space, it is also shifted down (pb on begining, pos 0)
                                offsetPosition.push(0);
                                return Utils.BREAK;
                            }
                        }, {reverse: true});
                    }
                    if (offsetPosition.length === 0) {
                        // now we know space left is not enough for last line, so we put page break before it
                        offsetPosition.push(cachedDataForLineBreaks[cachedDataForLineBreaksLength - 1].oxoPosInPar);
                    }
                }
            }

            function wrapWordsWithSpans(elementSpan) {
                var elText,
                    words,
                    markup,
                    spaces,
                    charAtBegining;

                _.each(elementSpan, function (el, index) {
                    elText = $(el).text();
                    words = elText.match(/\S+/g);
                    markup = '';
                    spaces = elText.match(/(\s+)/gi);

                    charAtBegining = (elText[0]) ? elText[0].match(/\S+/g) : null; // check if text begins with characters or whitespace

                    if (words && words.length > 0) {
                        _.each(words, function (word, i) {
                            if (word.length > 0) {
                                var generatedSpace;
                                if (spaces && spaces[i]) {
                                    if (spaces[i].length > 1) {
                                        generatedSpace = '';
                                        for (var j = 0; j < spaces[i].length; j++) { //  translates for ex. '     ' into -> ' &nbsp; &nbsp; ' (5 whitespaces)
                                            if (j % 2 !== 0) {
                                                generatedSpace += ' ';
                                            } else {
                                                generatedSpace += '&nbsp;';
                                            }
                                        }
                                    } else {
                                        if (index === 0 && i === 0 && !charAtBegining) { // fix for first span with whitespace (it will render as 0px span, so we fake content with dot)
                                            generatedSpace = '.';
                                        } else {
                                            generatedSpace = ' ';
                                        }
                                    }
                                } else {
                                    generatedSpace = '';
                                }
                                if (charAtBegining) { // word, space direction
                                    if (generatedSpace.length > 0) {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>' + '<span class="whitespaceData" data-charLength="' +  (charLength += spaces[i].length) + '">' + generatedSpace + '</span>';
                                    } else {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    }
                                } else { // space, word direction
                                    if (generatedSpace.length > 0) {
                                        markup += '<span class="whitespaceData" data-charLength="' +  (charLength += spaces[i].length) + '">' + generatedSpace + '</span>' + '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    } else {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    }
                                }
                            }
                        });
                        $(el).empty().append(markup);
                    }
                });
            }

            if (_.isEmpty($element.data('lineBreaksData')) || modifyingElement) { // if no cached data, or element is modified, calculate all positions
                $newElement = $element.clone(true);
                if (partHeight > 0 && partWidth > 0) { // if there is drawing leftover from previous paragraphs, it will change p layout
                    markup = '<div style="width: ' + partWidth + 'px; height: ' + partHeight + 'px; float: left"></div>';
                    $newElement.prepend(markup);
                }
                $newElement.prependTo('#io-ox-office-temp').css({'width': elementWidth, 'line-height': elementLineHeightStyle});

                // Handling of drawings - 2 groups: inline and floated.
                // inline are treated normally. can split before and after them
                // floating needs to fetch div offset also
                // there has to be range of values inside range of [0, paragraph height] for which splitting text is not valid
                // range itself can be made of ranges, ex. [[0, 25], [166, 333], [400, 550]]
                // Exception: floated drawing at the beginning of doc, in first p, before text
                $elDrawings = $newElement.find('.drawing');

                _.each($elDrawings, function(drawing) {
                    var $drawing = $(drawing),
                        range = [];
                    range[0] = $drawing.position().top;
                    range[1] = range[0] + ($drawing.height() + Utils.getElementCssLength($drawing, 'margin-bottom'));
                    if ($drawing.hasClass('float')) {
                        var diff = $drawing.prev('.offset').outerHeight(true);
                        range[0] = Math.max(0, range[0] - diff);
                    }

                    splitPosForbidenRanges.push(range);
                });

                elementSpan = $newElement.children('span');
                wrapWordsWithSpans(elementSpan);

                cachedPosTop = 0;
                _.each(elementSpan.find('.textData'), function (elSpan) {
                    $elSpan = $(elSpan);
                    if ($elSpan.text().length > 0) { //if (Utils.getDomNode(elSpan).firstChild)
                        elSpanPositionTop = $elSpan.position().top;
                        if ((elSpanPositionTop > 2) && (elSpanPositionTop !== cachedPosTop)) {
                            cachedPosTop = elSpanPositionTop;
                            if (splitPosForbidenRanges.length > 0) {
                                allowedValueInRange = true;
                                _.each(splitPosForbidenRanges, function(range) {
                                    if (elSpanPositionTop > range[0] && elSpanPositionTop < range[1]) {
                                        allowedValueInRange = false;
                                    }
                                });
                                if (allowedValueInRange) {
                                    oxoPosInPar = parseInt($elSpan.attr('data-charLength'), 10) - $elSpan.text().length;
                                    cachedDataForLineBreaks.push({'oxoPosInPar': oxoPosInPar, 'distanceTop': elSpanPositionTop });
                                }
                            } else {
                                oxoPosInPar = parseInt($elSpan.attr('data-charLength'), 10) - $elSpan.text().length;
                                cachedDataForLineBreaks.push({'oxoPosInPar': oxoPosInPar, 'distanceTop': elSpanPositionTop });
                            }
                        }
                    }
                });
                $element.data('lineBreaksData', cachedDataForLineBreaks);
                $newElement.remove(); //clear dom after getting position
            } else {
                cachedDataForLineBreaks = $element.data('lineBreaksData');
            }

            cachedDataForLineBreaksLength = cachedDataForLineBreaks.length;
            if (cachedDataForLineBreaksLength > 0) {
                calculateOffsetPosition();
            }

            return offsetPosition;
        }

        /**
         * Internal function for page layout rendering!
         * Default way for inserting page break node in DOM,
         * in between first level nodes, paragraphs and tables
         *
         * @param {Node|jQuery} element
         *  Element which is being proccessed
         *
         * @param {Number} elementHeight
         *  Current height value of all proccessed nodes
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing, in px
         *
         */
        function defaultInsertionOfPageBreak(element, elementHeight, contentHeight) {
            var
                $element = $(element),
                nodeHelper = $element.prev();

            while (nodeHelper.hasClass('page-break')) {
                nodeHelper = nodeHelper.prev();
            }
            if (nodeHelper.length > 0) {
                nodeHelper.addClass('last-on-page');
                $element.before(generatePagebreakElementHtml(contentHeight, _.browser.WebKit));
            }
            $element.addClass('break-above'); // this is marker for first node bellow pagebreak
            contentHeight = elementHeight; // addition if content is shifted down

            return contentHeight;
        }

        /**
         * Internal function for page layout rendering!
         * Proccesses passed table element node, and appends generated page break in a new row.
         *
         * @param {Node|jQuery} element
         *  Element which is being proccessed
         *
         * @param {Number} elementHeight
         *  Current height value of all proccessed nodes
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.initial=false]
         *      Determines if function is called initialy after document loading.
         *  @param {Number} [options.diffMargin=0]
         *      Difference in margin-top from current element, and margin-bottom from previous,
         *      needs to be stored in accumulator, for more precise calculation.
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing
         *
         */
        function insertPageBreaksInTable(element, elementHeight, contentHeight, options) {
            var
                $element = $(element),
                // if there is difference in margin top of current element, and margin bottom of previous, add this diff as start value to accumulator
                accumulator = (options && options.diffMargin) ? options.diffMargin : 0,
                totalAcc = 0,
                arrAccumulator = [],
                // helper var for big tables
                tempContentHeight = contentHeight,
                tableRows = element.rows,
                breakAboveRows,
                // helper height value, for tables bigger than one page
                tableHeight = elementHeight,
                // height of current row inside table (used for iteration)
                trOffsetHeight,
                // option flag to check is it called during initial page break rendering
                initial = Utils.getBooleanOption(options, 'initial', false),
                //summed value of all rows above last page break
                summedAttrPBHeight = 0,
                // generated string with page break node
                markup,
                // IE needs 2px more to left for split tables, Firefox 1px - due to css border-collapse
                negativeMarginTablesFix = _.browser.IE ? 2 : (_.browser.Firefox ? 1 : 0);

            _.each(tableRows, function (tr) {
                trOffsetHeight = tr.offsetHeight;
                if (accumulator + trOffsetHeight < pageMaxHeight - tempContentHeight) {
                    accumulator += trOffsetHeight;
                } else {
                    $(tr).addClass('break-above-tr');
                    $element.addClass('tb-split-nb');
                    totalAcc += accumulator;
                    arrAccumulator.push(totalAcc);
                    tempContentHeight = trOffsetHeight;
                    totalAcc = tempContentHeight;
                    accumulator = 0;
                }
            });

            if ($(tableRows[0]).hasClass('break-above-tr')) {
                $(tableRows[0]).removeClass('break-above-tr');
                if ($element.prev().length > 0) {
                    markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit);
                    $element.before(markup);
                }
                $element.addClass('break-above'); // this is marker for first node bellow pagebreak
                contentHeight = 0;
                arrAccumulator.shift();
            }

            breakAboveRows = $element.find('.break-above-tr');
            if (breakAboveRows.length > 0) {
                _.each(breakAboveRows, function (breakAboveRow, i) {
                    contentHeight += arrAccumulator[i];
                    markup = generateTableRowPagebreakHTML(contentHeight, negativeMarginTablesFix);
                    $(breakAboveRow).before(markup);
                    if (tableHeight > pageMaxHeight) {
                        contentHeight = 0;
                    } else {
                        contentHeight = tableHeight - arrAccumulator[i];
                    }
                    tableHeight = tableHeight - arrAccumulator[i];
                    summedAttrPBHeight += arrAccumulator[i];
                });
                contentHeight = elementHeight - summedAttrPBHeight; // height of part of table transferred to new page
            } else {
                contentHeight = elementHeight;
                $element.removeClass('tb-split-nb');
            }
            if (!initial && _.browser.Firefox) { Table.forceTableRendering(element); } // forcing Firefox to re-render table (32461)

            return contentHeight;
        }


        /**
         * In ineditable mode, tables have to have all elements with contenteditable set to false.
         * Collection saves all nodes for later restoration, in editmode.
         *
         */
        function toggleContentEditableTrueNodes() {
            contentEditableTrueNodes = $headerFooterPlaceHolder.find('[contenteditable="true"]').attr('contenteditable', false);
            $headerFooterPlaceHolder.find('.p').removeClass('p').addClass('par'); // this is necessary for not placing cursor inside div.p, in some browsers, like IE or FF
        }

        /**
         * Internal function  for page layout rendering!
         * Generating html markup string for Header to insert into page break html markup.
         *
         * @returns {String}
         *  Div header html markup
         */
        function getHeaderHtmlContent(type) {
            var
                type = type || '';

            return '<div class="header inactive-selection ' + type + '" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + ' style="min-height: ' + pagePaddingTop +
                'px; padding: ' + headerMargin + 'px 0 0 ' + (pagePaddingLeft) + 'px;"><div class="cover-overlay" contenteditable="false"></div></div>';
        }

        /**
         * Internal function  for page layout rendering!
         * Generating html markup string for Footer to insert into page break html markup.
         *
         * @returns {String}
         *  Div header html markup
         */
        function getFooterHtmlContent(type) {
            var
                type = type || '';

            return '<div class="footer inactive-selection ' + type + '" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + ' style="min-height: ' + pagePaddingBottom +
            'px; padding: 20px 0 ' + footerMargin + 'px ' + (pagePaddingLeft) + 'px;"><div class="cover-overlay" contenteditable="false"></div></div>';
        }

        function createHeaderContainer(operationId, operationType) {
            return $('<div>').addClass('header inactive-selection ' + operationId + ' ' + operationType)
                .attr('contenteditable', false).attr('data-container-id', operationId)
                .data({ 'id': operationId, 'type': operationType })
                .prepend($('<div>').addClass('cover-overlay').attr('contenteditable', false))
                .append($('<div>').addClass('p').data('implicit', true));
        }

        function createFooterContainer(operationId, operationType) {
            return $('<div>').addClass('footer inactive-selection ' + operationId + ' ' + operationType)
                .attr('contenteditable', false).attr('data-container-id', operationId)
                .data({ 'id': operationId, 'type': operationType })
                .prepend($('<div>').addClass('cover-overlay').attr('contenteditable', false))
                .append($('<div>').addClass('p').data('implicit', true));
        }


        /**
        *  Update styling of header&footer inside placeholder, and fetch their heights
        */
        function updateStartHeaderFooterStyle() {
            var
                headers = $headerFooterPlaceHolder.find('.header'),
                footers = $headerFooterPlaceHolder.find('.footer');
            $headerFooterPlaceHolder.css({
                'width': pageWidth,
                'margin': '0px 0px 150px -' + (pagePaddingLeft) + 'px'
            });
            headers.css({
                'min-height': pagePaddingTop,
                'max-width': pageWidth,
                'padding': headerMargin + 'px ' + pagePaddingRight + 'px 0 ' + (pagePaddingLeft) + 'px'
            });
            footers.css({
                'min-height': pagePaddingBottom,
                'max-width': pageWidth,
                'padding': '5px ' + pagePaddingRight + 'px ' + footerMargin + 'px ' + (pagePaddingLeft) + 'px'
            });

            rootNode.css({
                'padding-bottom': 0,
                'padding-top': 0
            });

            // update height vaules of headers&footers
            _.each(headers, function(header) {
                if (header.classList.contains('header_first')) {
                    heightOfFirstHeader = $(header).outerHeight(true);
                } else if (header.classList.contains('header_even')) {
                    heightOfEvenHeader = $(header).outerHeight(true);
                } else {
                    heightOfDefHeader = $(header).outerHeight(true);
                }
            });
            _.each(footers, function(footer) {
                if (footer.classList.contains('footer_first')) {
                    heightOfFirstFooter = $(footer).outerHeight(true);
                } else if (footer.classList.contains('footer_even')) {
                    heightOfEvenFooter = $(footer).outerHeight(true);
                } else {
                    heightOfDefFooter = $(footer).outerHeight(true);
                }
            });

            if (heightOfFirstHeader !== 0 || heightOfFirstFooter !== 0) {
                firstPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfFirstHeader) - Math.max(pagePaddingBottom, heightOfFirstFooter);
            }
            if (heightOfEvenHeader !== 0 || heightOfEvenFooter !== 0) {
                evenPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfEvenHeader) - Math.max(pagePaddingBottom, heightOfEvenFooter);
            }
            if (heightOfDefHeader !== 0 || heightOfDefFooter !== 0) {
                defPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfDefHeader) - Math.max(pagePaddingBottom, heightOfDefFooter);
            } else {
                defPageHeight = self.getDefPageActiveHeight({ convertToPixel : true });
            }

//            console.warn('firstPageHeight: ', firstPageHeight);
//            console.warn('evenPageHeight: ', evenPageHeight);
//            console.warn('defPageHeight: ', defPageHeight);
        }

        /**
         * On switching to new page, change global properties for main page content height, and type of page
         *
         */
        function switchPageTypeAndHeight() {
            switch (currentPageTypeHeight) {
                case 'first':
                case 'def':
                    if (evenPageHeight !== 0) {
                        pageMaxHeight = evenPageHeight;
                        currentPageTypeHeight = 'even';
                    } else if (currentPageTypeHeight === 'first') {
                        currentPageTypeHeight = 'def';
                        if (defPageHeight !== 0) {
                            pageMaxHeight = defPageHeight;
                        }
                    }
                    break;
                case 'even':
                    pageMaxHeight = defPageHeight;
                    currentPageTypeHeight = 'def';
                    break;
            }
        }

        /**
         * Internal function for page layout rendering!
         * Generating html markup string to insert as page break div element inside page.
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Boolean} contentEditableProperty
         *  Webkit browsers needs to set contenteditable to false, others to empty
         *
         * @param {Number} elementLeftIndent
         *  Left indent of element, in px
         *
         * @returns {String}
         *  Div page break html markup
         */
        function generatePagebreakElementHtml(contentHeight, contentEditableProperty, elementLeftIndent) {
            var leftIndent = elementLeftIndent || 0,
                markup = '<div class="page-break" contenteditable=' + (contentEditableProperty ? '"false"' : '""') + 'style="width: ' + pageWidth +
                'px; margin: ' + (Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px 0px -' + (pagePaddingLeft + leftIndent) + 'px">' + getFooterHtmlContent() + '<div class="inner-pb-line"></div>' + getHeaderHtmlContent() + '</div>';

            switchPageTypeAndHeight();

            return markup;
        }

        /**
         * Internal function for page layout rendering!
         * Generating html markup string to insert inside table as row containing page break div.
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Number} negativeMarginTablesFix
         *  IE needs 2px, FF 1px, due to border collapse
         *
         * @returns {String}
         *  Table row containing div page break html markup
         */
        function generateTableRowPagebreakHTML(contentHeight, negativeMarginTablesFix) {
            var markup = '<tr class="pb-row"><td style="padding: 0px; border: none" colspan="1"><div class="page-break" contenteditable=' +
                (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth + 'px; margin: ' + (Math.max(pageMaxHeight - contentHeight, 0)) +
                'px 0px 0px -' + (pagePaddingLeft + negativeMarginTablesFix) + 'px">' + getFooterHtmlContent() + '<div class="inner-pb-line"></div>' + getHeaderHtmlContent() + '</div></td></tr>';

            switchPageTypeAndHeight();

            return markup;
        }

        /**
         * Internal function for page layout rendering!
         * Default insertion of manual page breaks,
         * it adds page break before passed element if it is paragraph style attribute,
         * or after if it is Microsoft's hardbreak type page
         *
         * @param {Node} element
         *  DOM element currently beeing proccessed
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.cleanup=false]
         *      If its neccessary to do pagebreaks cleanup before insertion.
         *      Happens if it's not called from initial page break render function.
         *
         * @param {Number} contentHeight
         *  Height of content in current page before proccessing
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing
         *
         */
        function manualInsertionOfPageBreak(element, contentHeight, elementHeight, options) {
            var
                $element = $(element),
                cleanup = Utils.getBooleanOption(options, 'cleanup', false),
                msHardbreakPage = $element.find('.ms-hardbreak-page'),
                textSpanAfterHardbreak = msHardbreakPage.last().parent('.hardbreak').next(),
                markup,
                $nodeHelper,
                tempContentHeight = contentHeight,
                elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs;

            if (cleanup) {
                $nodeHelper = $element.prev();
                // remove old page break
                while ($nodeHelper.hasClass('page-break')) {
                    $nodeHelper.remove();
                    $nodeHelper = $element.prev();
                }
                $element.removeClass('manual-page-break');

                prepareNodeAndRemovePageBreak(element);
            }
            if ($element.is('table') && $element.find(DOM.MANUAL_PAGE_BREAK_SELECTOR).length < 1) {
                return contentHeight;
            }

            $element.addClass('last-on-page');
            $element.next().addClass('break-above'); // this is marker for first node bellow pagebreak
            if (msHardbreakPage.length > 0) { // MS Word hardbreak page
                tempContentHeight += msHardbreakPage.last().parent('.hardbreak').position().top;
                _.each(msHardbreakPage, function(msBreak) {
                    if ($(msBreak).parent('.hardbreak').prevAll(DOM.LIST_LABEL_NODE_SELECTOR).length > 0) { // #35649 hardbreak between text and list label
                        markup = generatePagebreakElementHtml(tempContentHeight, _.browser.WebKit);
                        $element.before(markup);
                    } else {
                        markup = generatePagebreakElementHtml(tempContentHeight, _.browser.WebKit, elementLeftIndent);
                        $(msBreak).parent('.hardbreak').next().addClass('break-above-span').before(markup);
                        $element.addClass('contains-pagebreak');
                    }
                });
                if (elementHeight <= pageMaxHeight) {
                    contentHeight = textSpanAfterHardbreak.length > 0 ? $element.height() - textSpanAfterHardbreak.position().top : 0;
                } else {
                    contentHeight = 0;
                }
            } else {
                markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit);
                $element.before(markup);
                contentHeight = 0;
            }

            $element.addClass('manual-page-break');

            return contentHeight;
        }

        /**
         *
         * Internal function for page layout rendering!
         * Support function for Open Document Format, adds manual page break after passed node element
         *
         * @param {Node} element
         *  DOM element currently beeing proccessed
         *
         * @param {Number} contentHeight
         *  Height of content in current page before proccessing; in px
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing; in px
         *
         */

        function insertManualPageBreakAfter(element, contentHeight) {
            $(element).addClass('last-on-page');
            $(element).after(generatePagebreakElementHtml(contentHeight, _.browser.WebKit));
            // new element will always start at the beginning of new page
            contentHeight = 0;
            return contentHeight;
        }

        /**
         * Internal function for page layout rendering!
         * Makes preparation for node proccessing,
         * like cleanup of previous page breaks, and class naming
         *
         * @param {Node} element
         *  DOM element currently beeing proccessed
         *
         */
        function prepareNodeAndRemovePageBreak(element) {
            var nodeHelper,
                $element = $(element);

            if ($element.hasClass('last-on-page')) {
                $element.removeClass('last-on-page');
                // refreshNextPage = true; //content is shifted up, next page has to be recalculated also
            }
            nodeHelper = $element.next();
            //if element is last in document
            if (nodeHelper.length === 0) {
                $element.addClass('last-on-page');
            } else {
                // remove old page break
                while (nodeHelper.hasClass('page-break')) {
                    nodeHelper.remove();
                    nodeHelper = $element.next();
                }
            }
        }


        /**
         * Loops through collection of page breaks,
         * and determines correct class name for each header and footer,
         * regarding of page number.
         *
         * @param {Number[]|jQuery} pageBreaksCollection
         */
        function populateHeaderFooterPaginationClasses(pageBreaksCollection) {

            _.each(pageBreaksCollection, function(pageBreak, index) {
                if (index === 0) {
                    $(pageBreak).children('.footer').addClass('first');
                    $(pageBreak).children('.header').addClass('even');
                } else if (index % 2 === 0) {
                    $(pageBreak).children('.header').addClass('even');
                    $(pageBreak).children('.footer').addClass('default');
                } else {
                    $(pageBreak).children('.footer').addClass('even');
                    $(pageBreak).children('.header').addClass('default');
                }
            });
            $firstHeaderWrapper.children('.header').addClass('first');
            if (pageBreaksCollection.length === 0) {
                $lastFooterWrapper.children('.footer').addClass('first');
            } else if (pageBreaksCollection.length % 2 === 0) {
                $lastFooterWrapper.children('.footer').addClass('default');
            } else {
                $lastFooterWrapper.children('.footer').addClass('even');
            }
        }

        /**
         * Temporary function (needs to be removed after enabling editing of headers&footers),
         * to disable change tracking inside non-editable headers and footers.
         *
         * Important: Function doesn't send operation to the server,
         * it just marks all nodes inside header-footer, so they are skipped in changetrack evaluation.
         *
         * Notice: On edit mode enabled for header/footer, this function has to be removed!
         *
         */
        function disableChangeTrackInInactiveNodes() {
            if (model.getChangeTrack().isActiveChangeTracking()) {
                $headerFooterPlaceHolder.find('*').addClass('marginal');
            }
        }

        /**
         * Internal function which makes checks if fetched node is empty.
         * If not, it is cloned to be used for insertion in header/footer.
         *
         * @param {jQuery} node - if this node in headerFooterCollection is not empty,
         *                           it's cloned to be used as header/footer
         *
         * @returns {jQuery} - node which passes checks
         */
        function cloneNonEmptyHeadFootNode(node) {
            if (node.text().length > 0 || node.find('.drawing, table, ' + DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR).length > 0) {
                return node.clone(true);
            } else {
                return $();
            }
        }

        /**
         * Internal function to update data inside special fields with type: page-count.
         * Frontend needs to overwrite dummy values backend sends,
         * because backend has no knowledge of page count after rendering doc in browser.
         *
         */
        function updatePageCountFields() {
            _.each(rootNode.find(DOM.PAGECOUNT_FIELD_SELECTOR), function (field) {
                $(field.childNodes).empty().html(pageCount);
            });
        }

        /**
         * Internal function to update data inside special fields with type: page-number.
         * Frontend needs to overwrite dummy values backend sends,
         * because backend has no knowledge of page count after rendering doc in browser.
         *
         */
        function updatePageNumberFields() {
            // update in headers & footers
            _.each($pageContentNode.find('.header').find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num') + 1);
            });

            _.each($pageContentNode.find('.footer').find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num'));
            });
            // first header in doc - page number always 1
            $firstHeaderWrapper.children('.header').find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(1);
            // last footer in doc gets total page count as page number
            $lastFooterWrapper.children('.footer').find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(pageCount);

            // update in paragraphs as first level children of pagecontent
            _.each($pageContentNode.children(DOM.PARAGRAPH_NODE_SELECTOR).children(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html(self.getPageNumber(field));
            });
            // update in paragraphs inside tables as first level children of pagecontent
            _.each($pageContentNode.children(DOM.TABLE_NODE_SELECTOR).find('tr').not('.pb-row').find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html(self.getPageNumber(field));
            });
        }

        /**
         * Creates overlay layer for tooltip over user-fields.
         *
         */
        function createTooltipOverlay() {
            $tooltipOverlay = $('<div>').addClass('field-tooltip-layer').attr('contenteditable', 'false');
            rootNode.append($tooltipOverlay);
        }


        // methods ------------------------------------------------------------

        // updates page by inserting page layout
        this.callInitialPageBreaks = function (options) {
            return initialPageBreaks(options);
        };

        this.getallHeadersAndFooters = function () {
            return $pageContentNode.find('.header, .footer').add($firstHeaderWrapper.children('.header')).add($lastFooterWrapper.children('.footer'));
        };

        this.getHeaderFooterPlaceHolder = function () {
            return $headerFooterPlaceHolder;
        };

        this.hasContentHeaderFooterPlaceHolder = function () {
            return $headerFooterPlaceHolder.children().length > 0;
        };

        this.isExistingFirstHeaderWrapper = function () {
            return $firstHeaderWrapper.length > 0;
        };

        this.isExistingLastFooterWrapper = function () {
            return $lastFooterWrapper.length > 0;
        };



        /**
         * Public method to check if first header in document exist, and if does, return it's wrapper's height (in px).
         * If element exists, returns height of wrapper element, of first header in document (which is prepended to pagecontent node)
         * If element doesn't exist, returs 0.
         *
         */
        this.getHeightOfFirstHeaderWrapper = function () {
            return $firstHeaderWrapper ? $firstHeaderWrapper.height() : 0;
        };



        /**
         * Function triggered when INSERT_HEADER_FOOTER operation is called, with type containing: header
         *
         * @returns {jQuery|Node} instance of created first header in document
         */
        this.generateHeaderElementHtml = function () {
            var
                string = '<div class="header-wrapper" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth +
                'px; margin: 0px 0px 0px -' + (pagePaddingLeft) + 'px">' + getHeaderHtmlContent('first') + '</div>';

            $pageContentNode.before(string);

            return $pageContentNode.prev();
        };

        /**
         * Function triggered when INSERT_HEADER_FOOTER operation is called, with type containing: footer
         *
         * @returns {jQuery|Node} instance of created first footer in document
         */
        this.generateFooterElementHtml = function () {
            var
                string = '<div class="footer-wrapper" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth +
                'px; margin: 0px 0px 0px -' + (pagePaddingLeft) + 'px">' + getFooterHtmlContent() + '</div>';

            $pageContentNode.after(string);

            return $pageContentNode.next();
        };


        /**
         * Creates generic placeholder for headers and footers, that are going to use as pattern for all headers/footers in document.
         *
         */
        this.createHeaderFooter = function (operationId, operationType) {
            var isHeader = (/header/i).exec(operationType),
                localCollection;

            localCollection = {
                headFootType : operationType,
                id : operationId,
                node : (isHeader ? createHeaderContainer(operationId, operationType) : createFooterContainer(operationId, operationType))
            };
            if ($headerFooterPlaceHolder.length > 0) {
                $headerFooterPlaceHolder.append(localCollection.node);
            } else {
                $headerFooterPlaceHolder = $('<div>').addClass('header-footer-placeholder').attr('contenteditable',_.browser.WebKit ? 'false' : '').css('display', Utils.SMALL_DEVICE ? 'none' : '' ).append(localCollection.node);
                rootNode.append($headerFooterPlaceHolder);
            }
            headerFooterCollection.push(localCollection);

        };

        /**
         * Gets the header node from placeholder. If footer of passed type is not found, or empty,
         * it fetches default footer type.
         *
         * @param {String} headerType
         *  Type of header - can be default, first, or even. If omitted, header_default is used.
         *
         * @returns {jQuery|Node}
         *  Returns found node, or empty jQuery object
         *
         */
        this.getHeaderFromPlaceholder = function (headerType) {
            var obj;
            if (headerType) {
                obj = _.findWhere(headerFooterCollection, {headFootType: headerType});

                if (obj && obj.node) {
                    return cloneNonEmptyHeadFootNode(obj.node);
                }
            }

            else {
                obj = _.findWhere(headerFooterCollection, {headFootType: 'header_default'});
                if (obj && obj.node) {
                    return cloneNonEmptyHeadFootNode(obj.node);
                }
            }
            return $();
        };

        /**
         * Gets the footer node from placeholder. If footer of passed type is not found, or empty,
         * it fetches default footer type.
         *
         * @param {String} footerType
         *  Type of footer - can be default, first, or even. If omitted, footer_default is used.
         *
         * @returns {jQuery|Node}
         *  Returns found node, or empty jQuery object
         *
         */
        this.getFooterFromPlaceholder = function (footerType) {
            var obj;
            if (footerType) {
                obj = _.findWhere(headerFooterCollection, {headFootType: footerType});

                if (obj && obj.node) {
                    return cloneNonEmptyHeadFootNode(obj.node);
                }
            }

            else {
                obj = _.findWhere(headerFooterCollection, {headFootType: 'footer_default'});
                if (obj && obj.node) {
                    return cloneNonEmptyHeadFootNode(obj.node);
                }
            }

            return $();
        };

        /**
         * On document loading, using fast-load, references to created nodes in placeholder have to be updated
         *
         */
        this.headerFooterCollectionUpdate = function () {
            var
                placeholderChildren = $headerFooterPlaceHolder.children(),
                elementData,
                localCollection;

            _.each(placeholderChildren, function (element) {
                elementData = $(element).data();
                $(element).prepend($('<div>').addClass('cover-overlay').attr('contenteditable', false));

                localCollection = {
                        headFootType : elementData.type,
                        id : elementData.id,
                        node : $(element).attr('data-container-id', elementData.id)
                    };
                headerFooterCollection.push(localCollection);
            });

        };

        /**
         * Gets number of the page where passed node, logical position, or selection start belongs to.
         *
         * @param {jQuery|Node|Number[]|undefined} position
         *  Can accept node, logical position as array, or no argument (then fetches element's selection where cursor is positioned)
         *
         * @returns {Number}
         *  Returns number of page where node is located. If no valid node is fetched, returns 0.
         *
         */
        this.getPageNumber = function (position) {
            var firstPageBreakAbove,
                parentSplit,
                pb,
                traverseUp = false,
                pageNumber = 1;

            if (!position) {
                position = model.getSelection().getStartPosition();
            }

            if (_.isArray(position)) {
                position = Position.getContentNodeElement(rootNode, position.slice(0, 1));
            }

            if ($(position).length === 0) {
                Utils.warn('Unknown position of element!');
                pageNumber = 0;
                return pageNumber;
            }
            if (!pageBreaksCollection || pageBreaksCollection.length === 0) {
                Utils.warn('pageLayout.getPageNumber: No page breaks found at this point.');
                pageNumber = 1;
                return pageNumber;
            }


            firstPageBreakAbove = $(position);



            while (firstPageBreakAbove.length > 0 && !firstPageBreakAbove.hasClass('page-break')) { // fetching first page break element above passed element to fetch its index
                if (firstPageBreakAbove.hasClass('contains-pagebreak') || firstPageBreakAbove.hasClass('tb-split-nb')) { // first level node contains page break inside itself
                    firstPageBreakAbove = firstPageBreakAbove.find('.page-break').first();
                    if (firstPageBreakAbove.length < 1) {
                        pageNumber = null;
                        return pageNumber;
                    }
                    if (traverseUp) {
                        pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
                    } else {
                        pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 1) : 1;
                    }

                    return pageNumber;
                } else {
                    if (firstPageBreakAbove.parent().hasClass('pagecontent')) { // first level node
                        firstPageBreakAbove = firstPageBreakAbove.prev();
                        traverseUp = true;
                    } else {
                        parentSplit = firstPageBreakAbove.parentsUntil('.pagecontent').last();
                        pb = parentSplit.find('.page-break').first();

                        if (parentSplit.hasClass('contains-pagebreak')) { // parent node is paragraph split by page break
                            while (firstPageBreakAbove.length > 0 && !firstPageBreakAbove.hasClass('page-break')) { // search above
                                firstPageBreakAbove = firstPageBreakAbove.prev();
                            }
                            if (firstPageBreakAbove.length < 1) { // if none found, page break is bellow element
                                if (pb.length < 1) {
                                    pageNumber = null;
                                    return pageNumber;
                                }
                                firstPageBreakAbove = pb;
                                pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 1) : 1;
                                return pageNumber;
                            }

                            pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
                            return pageNumber;
                        } else if (parentSplit.hasClass('tb-split-nb')) {  // parent node is table split by page break
                            if (!firstPageBreakAbove.is('tr')) {
                                firstPageBreakAbove = firstPageBreakAbove.parentsUntil('.pagecontent', 'tr').last();
                            }
                            while (firstPageBreakAbove.length > 0 && !firstPageBreakAbove.hasClass('pb-row')) { // search above
                                firstPageBreakAbove = firstPageBreakAbove.prev();
                            }
                            if (firstPageBreakAbove.length < 1) { // if none found, page break is bellow element
                                firstPageBreakAbove = pb;
                                if (pb.length < 1) {
                                    pageNumber = null;
                                    return pageNumber;
                                }

                                pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 1) : 1;
                                return pageNumber;
                            }
                            firstPageBreakAbove = firstPageBreakAbove.find('.page-break');

                            pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
                            return pageNumber;
                        } else if (parentSplit.hasClass('page-break') && firstPageBreakAbove.length > 0) {
                            if (firstPageBreakAbove.parentsUntil(parentSplit, '.header')) {
                                pageNumber = pageBreaksCollection.index(parentSplit) + 2;
                            } else if (firstPageBreakAbove.parentsUntil(parentSplit, '.footer')) {
                                pageNumber =  pageBreaksCollection.index(parentSplit) + 1;
                            }

                            return pageNumber;
                        } else if (parentSplit.length === 0 && firstPageBreakAbove.length > 0) {
                            if (firstPageBreakAbove.parentsUntil('.page', '.header-wrapper').length > 0 ) { // first page header
                                pageNumber = 1;
                                return pageNumber;
                            } else if (firstPageBreakAbove.parentsUntil('.page', '.footer-wrapper').length > 0 ) { // last page footer
                                pageNumber = pageBreaksCollection.length + 1;
                                return pageNumber;
                            }
                        }

                        firstPageBreakAbove = parentSplit.prev();
                        traverseUp = true;
                    }
                }
            }

            pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
            return pageNumber;
        };

        /**
         * Method to get first element (line inside page break) on demanded page number.
         *
         * @param {Number} pageNumber
         *  Number of page for which we want begining of page element.
         *
         * @returns {jQuery|Node|Boolean}
         *  If found, it returns line node, if not, returns boolean false or null (if first page is demanded).
         *
         */
        this.getBeginingElementOnSpecificPage = function (pageNumber) {
            if (!_.isNumber(pageNumber) || pageNumber < 1 || pageNumber % 1 !== 0) {
                return false;
            }
            if (pageBreaksCollection.length === 0 || pageNumber === 1) {
                return null;
            }
            var $element = $(pageBreaksCollection[pageNumber - 2]).find(DOM.PAGE_BREAK_LINE_SELECTOR);

            if (!$element || $element.length < 1) {
                Utils.warn('Page doesn\'t exist!');
                return false;
            }

            return $element;

        };

        /**
         * If parent container doesn't have style "position:relative",
         * safest way is to calculate difference of position.top,
         * between element and his parent.
         *
         * @returns {Number}
         *  Difference in px between top edge of element and top edge of it's parent.
         *
         */
        this.getDistanceTopRelativeToParent = function (element) {
            var $element = $(element),
                $parent = $element.parent();

            return ($parent.length > 0)  ? ($element.position().top - $parent.position().top) : 0;
        };

        /**
         * Getter method for active height in 1/100 mm, of default page (without margins) defined in page attributes
         *
         *  @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.convertToPixel=true]
         *      If set to true, the page height is calculated in pixel. Otherwise
         *      the returned value is in 1/100 mm.
         *
         * @returns {Number}
         *  The height of the page.
         */
        this.getDefPageActiveHeight = function (options) {
            var // whether the page height shall be returned in pixel
                convertToPixel =  Utils.getBooleanOption(options, 'convertToPixel', false);

            if (!pageAttributes) { // #35499 - if updateDocumentFormatting is not executed (start new empty doc), initialization of pageAttributes is required
                initializePageAttributes();
            }

            return convertToPixel ? (pageHeight - pagePaddingTop - pagePaddingBottom) : pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom;
        };

        /**
         * Getter method for page attributes
         *
         * @param {String} property
         *  The property for that the value shall be returned
         *
         * @returns {Number}
         *  The value for the specified page property.
         */
        this.getPageAttribute = function (property) {

            if (!pageAttributes) {
                initializePageAttributes();
            }

            return pageAttributes && pageAttributes.page && pageAttributes.page[property];
        };

        /**
         * Public function that updates value of field, with type page count,
         * based on frontend pageCount calculus.
         *
         * @param {jQuery|Node} field
         *  Field whose value is going to be updated.
         *
         */
        this.updatePageCountField = function(field) {
            $(field).children().empty().html(pageCount);
        };

        /**
         * Public function that updates value of field, with type page number,
         * based on frontend function getPageNumber.
         *
         * @param {jQuery|Node} field
         *  Field whose value is going to be updated.
         *
         */
        this.updatePageNumberField = function(field) {
            $(field).children().empty().html(self.getPageNumber(field));
        };


        /**
         * Callback function for mouseenter event on user type fields.
         * Creates popup with name of user defined field.
         *
         * @param event
         */
        this.createTooltipForField = function (event) {
            var
                $fieldNode = $(event.currentTarget),
                name = $fieldNode.data('name') || '',
                zoomFactor = app.getView().getZoomFactor() / 100,
                tooltipTopPos = ( $fieldNode.parentsUntil('.pagecontent').last().position().top + $fieldNode.position().top + $fieldNode.children().position().top) / zoomFactor,
                tooltipLeftPos = (event.clientX - rootNode.offset().left) / zoomFactor,
                $tooltip = $('<div>').addClass('field-tooltip-box').attr('contenteditable', 'false').text(Utils.escapeHTML(name)) ;

            // fastload stores different data structure of attributes
            if (name.length === 0 && _.isObject($fieldNode.data('attributes')) && _.isObject($fieldNode.data('attributes').field) && $fieldNode.data('attributes').field.name) {
                $tooltip.text($fieldNode.data('attributes').field.name);
            }

            if ($tooltipOverlay.length === 0) {
                createTooltipOverlay();
            }
            $tooltipOverlay.append($tooltip);

            if (document.documentElement.clientHeight - event.clientY - $fieldNode.children().height() * zoomFactor < 30 * zoomFactor) {
                tooltipTopPos = tooltipTopPos - $tooltip.outerHeight() - 15 + 'px';
            } else {
                tooltipTopPos = tooltipTopPos + $fieldNode.children().height() + 'px';
            }

            $tooltip.css({'top': tooltipTopPos, 'left': tooltipLeftPos + 'px' });

        };

        /**
         * Callback function for mouseleave event on user type fields.
         * Removes popup with name of user defined field.
         *
         */
        this.destroyTooltipForField = function () {
            $tooltipOverlay.empty();
        };


        /**
         * Looping through current 'page content DOM' and adding heights of elements to calculate page breaks positions.
         * Starts from element which is modified and continues downwards trough DOM. Stops if no more elements are shifted to new page
         * last-on-page marker: used for checking if content is shifted to another page, if not, stop iteration
         * break-above marker: used to mark element above which page break node should be inserted
         */
        this.insertPageBreaks = function() {
            if (!model.isPageBreakMode() || Utils.SMALL_DEVICE || model.isDraftMode()) { return; } // if page layout is disabled, dont run the function
            Utils.takeTime('pageLayout.insertPageBreaks(): ', function () {
                var
                    contentHeight = 0,
                    floatingDrawings,
                    elementHeight = 0,
                    elementHeightWithoutMB = 0,
                    processingNodes = $(),
                    elementSpan,
                    newlySplittedSpan,
                    nodeHelper,
                    zoomFactor = app.getView().getZoomFactor() / 100,
                    offsetPosition = [],
                    modifyingElement,
                    pageBreak,
                    markup = '',
                    elMarginBottom,
                    elMarginTop,
                    diffMargin = 0,
                    prevElMarginBottom = 0,
                    fDrawingHeight,
                    fDrawingWidth = 0,
                    accHeightFromPrevDrawing = 0,
                    heightOfRectPartHelper = 0,
                    // helper for fetching first node from where to start calculating
                    procNodesFirst,
                    $element,
                    //collectionOfDrawingData = {},
                    //flags for controlling which pages to process, and where to stop with processing
                    controlBreak = false,
                    refreshNextPage = false,
                    refreshPrevPage = false,
                    currentProcessingNode = model.getCurrentProcessingNode(),
                    currentPageNumber = 0;

                if (!pageAttributes) {
                    initializePageAttributes();
                }

                // hack for tablets with rewriten reduced css padding after document is loaded
                if (app.getView().getContentRootNode().hasClass('small-device')) {
                    pagePaddingLeft = parseFloat($pageContentNode.parent().css('padding-left')); //18.898;
                    pagePaddingRight = pagePaddingLeft;
                    pageWidth = $pageContentNode.parent().outerWidth();
                } else {
                    pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001);
                    pagePaddingRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001);
                    pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001);
                }

                function paragraphSplitWithPageBreak(element) {
                    $element = $(element);
                    offsetPosition = [];
                    modifyingElement = currentProcessingNode[0] === element; // flag if we are modifying the element, then cached line positions are invalid
                    if (modifyingElement) {
                        model.getSelection().setInsertTextPoint(null); // #32603 invalidate cached text point - reference to text node that is changed, so it returns wrong value for getFocusPositionBoundRect when scrollToSelection
                    }
                    if (accHeightFromPrevDrawing > 0 && fDrawingWidth > 0) {
                        offsetPosition = getOffsetPositionFromElement(element, contentHeight, modifyingElement, accHeightFromPrevDrawing, fDrawingWidth);
                    } else {
                        offsetPosition = getOffsetPositionFromElement(element, contentHeight, modifyingElement);
                    }
                    if (offsetPosition && offsetPosition.length > 0) { // if offset position is returned
                        if (offsetPosition[0] === 0) { // if offset is 0, no need for p split, insert pb before element
                            offsetPosition.shift();
                            contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                            if (offsetPosition.length > 0) { contentHeight = 0; }
                        }
                        if (offsetPosition.length > 0) {
                            elementSpan = $element.children('span').first();
                            while (elementSpan.text().length < 1 || !(elementSpan.is('span'))) {
                                elementSpan = elementSpan.next();
                            }
                            if (elementSpan && elementSpan.length > 0) { //fix for #32563, if div.p contains only empty spans
                                _.each(offsetPosition, function (offset) {
                                    var elTextLength = $(elementSpan).text().length;
                                    while (elTextLength > 0 && elTextLength <= offset) {
                                        offset -= elTextLength;
                                        elementSpan = elementSpan.next();
                                        while (!elementSpan.is('span')) { // we might fetch some inline div nodes, like div.tab or div.linebreak
                                            elementSpan = elementSpan.next();
                                        }
                                        elTextLength = $(elementSpan).text().length;
                                    }
                                    newlySplittedSpan = DOM.splitTextSpan(elementSpan, offset, {append: true});
                                    newlySplittedSpan.addClass('break-above-span');
                                    $element.addClass('contains-pagebreak');
                                    elementSpan = newlySplittedSpan;
                                });
                            }
                            refreshNextPage = true;

                            zoomFactor = app.getView().getZoomFactor() / 100;
                            pageBreak = $();
                            var arrBreakAboveSpans = $element.find('.break-above-span'),
                                upperPartDiv = 0,
                                totalUpperParts = 0,
                                diffPosTopAndPrev = 0,
                                elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs

                            _.each(arrBreakAboveSpans, function (breakAboveSpan) {
                                diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                                upperPartDiv = $(breakAboveSpan).position().top / zoomFactor - diffPosTopAndPrev;
                                totalUpperParts += upperPartDiv;
                                contentHeight += upperPartDiv;
                                nodeHelper = $element.prev();
                                while (nodeHelper.hasClass('page-break')) {
                                    nodeHelper = nodeHelper.prev();
                                }
                                nodeHelper.addClass('last-on-page');
                                markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit, elementLeftIndent);
                                $(breakAboveSpan).before(markup);
                                pageBreak = $(breakAboveSpan).prev();
                                contentHeight = 0;
                            });
                            contentHeight = elementHeight - totalUpperParts;
                        }
                    } else {
                        contentHeight = insertPageBreaksBetweenDrawings(element, elementHeight, contentHeight, offsetPosition, zoomFactor);
                        //contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                    }
                }

                function processNodeForPageBreaksInsert(element, index) {
                    if (model.checkQuitFromPageBreaks()) { // if there is signal to interup page break processing, break from loop (ex. fast typing in big docs, succesive char deletion)
                        model.setQuitFromPageBreaks(false);
                        //quitFromPageBreak = false;
                        self.executeDelayed(function () {
                            model.insertPageBreaks(currentProcessingNode ? currentProcessingNode : {});
                        }, 50);
                        return Utils.BREAK;
                    }
                    $element = $(element);
                    // first merge back any previous paragraph splits
                    pageBreak = $element.find('.page-break');
                    if ($element.hasClass('contains-pagebreak') || pageBreak.length > 0) {
                        pageBreak.remove();
                        $element.children('.break-above-drawing').removeClass('break-above-drawing');
                        _.each($element.children('span'), function (span) {
                            if (span.classList.contains('break-above-span')) {
                                $(span).removeClass('break-above-span');
                                CharacterStyles.mergeSiblingTextSpans(span);
                            }
                        });
                        if ($element.hasClass('p')) {
                            model.validateParagraphElement(element);
                            $element.removeClass('contains-pagebreak');
                            if ($element.find('.ms-hardbreak-page').length > 0) {
                                $element.addClass('manual-page-break');
                            }
                        }
                    }
                    // if table is split, merge back
                    if ($element.find('tr.pb-row').length > 0) {
                        $element.find('tr.pb-row').remove();
                        $element.find('tr.break-above-tr').removeClass('break-above-tr');
                        if ((_.browser.Firefox) && (DOM.isTableNode(element))) { Table.forceTableRendering(element); } // forcing Firefox to re-render table (32461)
                    }
                    $element.removeClass('tb-split-nb'); // return box shadow property to unsplit tables

                    fDrawingHeight = 0;
                    floatingDrawings = $element.find('.drawing.float');
                    if (floatingDrawings.length > 0) {
                        _.each(floatingDrawings, function (drawing) {
                            if ($(drawing).outerWidth(true) !== $pageContentNode.width()) {
                                if ($(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor) > fDrawingHeight) {
                                    fDrawingHeight = $(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor);
                                    fDrawingWidth = $(drawing).outerWidth(true);
                                }
                            }
                        });
                    }
                    if (accHeightFromPrevDrawing > fDrawingHeight) {
                        fDrawingHeight = 0;
                    }

                    elementHeightWithoutMB = element.offsetHeight;
                    elMarginBottom = Utils.getElementCssLength($element, 'margin-bottom');
                    elMarginTop = Utils.getElementCssLength($element, 'margin-top');

                    if (elMarginTop > prevElMarginBottom) { // el has margin top that is greater than margin bottom of prev el, include diff in page height
                        diffMargin = elMarginTop - prevElMarginBottom;
                        elementHeightWithoutMB += diffMargin;
                    }
                    elementHeight = elementHeightWithoutMB + elMarginBottom;
                    prevElMarginBottom = elMarginBottom;

                    if (DOM.isManualPageBreakNode(element)) {
                        if (index !== 0 || $element.find('.ms-hardbreak-page').length > 0) {
                            contentHeight = manualInsertionOfPageBreak(element, contentHeight, elementHeight, { 'cleanup' : true });
                        } else {
                            $element.removeClass('manual-page-break');
                        }
                    }

                    // if current content height is bellow max page height
                    // and if there is no image floated inside paragraph, or if so, p fits inside page with it also
                    // and if there is no floated image from one of previous paragraphs that intersects current p, or if exist, element can fit on page with it also;
                    if (contentHeight + elementHeight <= pageMaxHeight && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                            contentHeight += elementHeight;
                        }
                        if (accHeightFromPrevDrawing > elementHeight) {
                            accHeightFromPrevDrawing -= elementHeight;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                        if (!DOM.isManualPageBreakNode(element)) {
                            prepareNodeAndRemovePageBreak(element);
                        }
                    // for last element that can fit on page we need to omit margin bottom
                    } else if (contentHeight + elementHeightWithoutMB <= pageMaxHeight && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                            contentHeight += elementHeightWithoutMB;
                        }
                        if (accHeightFromPrevDrawing > elementHeightWithoutMB) {
                            accHeightFromPrevDrawing -= elementHeightWithoutMB;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                        if (!DOM.isManualPageBreakNode(element)) {
                            prepareNodeAndRemovePageBreak(element);
                        }
                    } else { //shift to new page or split
                        prevElMarginBottom = 0;
                        if ($element.hasClass('last-on-page')) { //|| refreshPrevPage || refreshNextPage) {
                            $element.removeClass('last-on-page');
                            //refreshPrevPage = false;
                            //refreshNextPage = false;
                        } else {
                            //controlBreak = true;
                        }
                        nodeHelper = $element.next();
                        $element.prev().addClass('last-on-page');
                        //if element is last in document
                        if (nodeHelper.length === 0) {
                            $element.addClass('last-on-page');
                        } else { // remove old page break
                            while (nodeHelper.hasClass('page-break')) {
                                nodeHelper.remove();
                                nodeHelper = $element.next();
                            }
                        }

                        if ($element.hasClass('p') && ($element.text().length > 10 || $element.find('.drawing.inline').length > 0) ){ // paragraph has to be split
                            paragraphSplitWithPageBreak(element);
                            // end of paragraph split
                        } else if (DOM.isTableNode(element) && element.rows.length > 1) { // table split
                            contentHeight = insertPageBreaksInTable(element, elementHeight, contentHeight, { 'diffMargin': diffMargin});
                        } else { // default insertion of pb between elements
                            contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                        }
                    }
                    if (DOM.isNodeWithPageBreakAfterAttribute(element)) {
                        if ($element.next().length > 0) {
                            contentHeight = insertManualPageBreakAfter(element, contentHeight);
                        } else { // first element in document, can't have page break before - can happen on delete all where last is manual pb paragraph
                            $element.removeClass('manual-pb-after');
                        }
                    }

                    if (fDrawingHeight > 0) { // store the diff between floated image height and height of containing p, for next iteration
                        accHeightFromPrevDrawing = fDrawingHeight - elementHeight;
                    }
                    // at the last page, we add extra padding, to fill it out to the size of max page height
                    if (processingNodes.length === index + 1) {
                        $pageContentNode.css({
                            'padding-bottom': (pageMaxHeight - contentHeight > 0) ? pageMaxHeight - contentHeight : '36px'
                        });
                    }
                    if (controlBreak) {
                        return Utils.BREAK;
                    }
                } //end of processNodeForPageBreaksInsert

                if (model.isProcessingOperations()) {
                    // to prevent triggering method when not all DOM nodes are rendered (still operations in the stack, #32067)
                    self.executeDelayed(function () {
                        model.insertPageBreaks(currentProcessingNode ? currentProcessingNode : {});
                    });
                } else if (model.checkQuitFromPageBreaks()) {
                    model.setQuitFromPageBreaks(false);
                    //quitFromPageBreak = false;
                    self.executeDelayed(function () {
                        model.insertPageBreaks(currentProcessingNode ? currentProcessingNode : {});
                    }, 50);
                } else {
                    // cleanup after deleting big blocs of data at once
                    while ($pageContentNode.children().first().hasClass('page-break')) {
                        $pageContentNode.children().first().remove();
                    }
                    if (currentProcessingNode) {
                        currentProcessingNode = $(currentProcessingNode);
                        if (currentProcessingNode.hasClass('break-above')) { // first element bellow pagebreak, we want to refresh previous page also
                            nodeHelper = currentProcessingNode.prev();
                            while (nodeHelper.hasClass('page-break')) {
                                nodeHelper = nodeHelper.prev();
                            }
                            if (nodeHelper.length > 0) {
                                currentProcessingNode = nodeHelper;
                            } else {
                                currentProcessingNode.prev('.page-break').remove();
                            }
                            refreshPrevPage = true;
                        }
                        if (currentProcessingNode.hasClass('last-on-page')) {
                            nodeHelper = currentProcessingNode.prev();
                            while (nodeHelper.hasClass('page-break')) {
                                nodeHelper = nodeHelper.prev();
                            }
                            if (nodeHelper.length === 0) {
                                currentProcessingNode.prevAll('.page-break').remove();
                            }
                            refreshNextPage = true;
                        }
                        if (currentProcessingNode.parentsUntil('.pagecontent', 'table').length > 0) { //if we fetch valid div.p, but inside of table
                            currentProcessingNode = currentProcessingNode.parentsUntil('.pagecontent', 'table').last();
                        }
                        nodeHelper = currentProcessingNode.prev();
                        if (nodeHelper.hasClass('page-break')) {
                            while (nodeHelper.hasClass('page-break')) { // might happen when deleting multiple elements at once, some pb are left - clean-up
                                nodeHelper = nodeHelper.prev();
                            }
                            if (nodeHelper.length === 0) {
                                currentProcessingNode.prevAll('.page-break').remove();
                                processingNodes = currentProcessingNode.nextAll('.p, table').addBack();
                            } else {
                                nodeHelper = nodeHelper.prevUntil('.page-break, .contains-pagebreak').last();
                                if (nodeHelper.prev().hasClass('contains-pagebreak')) {
                                    contentHeight = (nodeHelper.prev().height() + parseInt(nodeHelper.prev().css('margin-bottom'), 10)) - nodeHelper.prev().children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                                }
                                processingNodes = nodeHelper.nextAll('.p, table').addBack();
                            }
                        //} else if (DOM.isManualPageBreakNode(currentProcessingNode)) {
                          //  processingNodes = nodeHelper.nextAll('div.p, table').addBack();
                        } else if (nodeHelper.hasClass('contains-pagebreak')) { //if node is second bellow pagebreak - fallback
                            if (nodeHelper.find('.break-above-span').length > 0) { // special case: split p with breakabove span, above page break
                                if (DOM.isManualPageBreakNode(currentProcessingNode)) { // fix for inserting manual page break in paragraph bigger than page height
                                    processingNodes = nodeHelper.prevUntil('.page-break').last().nextAll('.p, table').addBack();
                                } else {
                                    processingNodes = currentProcessingNode.nextAll('.p, table').addBack();
                                    contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                                }
                            } else {
                                procNodesFirst = nodeHelper.prevUntil('.page-break, .contains-pagebreak').last();
                                nodeHelper = procNodesFirst.prev();
                                if (nodeHelper.hasClass('contains-pagebreak')) {
                                    processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                                    contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                                } else {
                                    processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                                }
                            }
                        } else {
                            procNodesFirst = currentProcessingNode.prevUntil('.page-break, .contains-pagebreak').last();
                            nodeHelper = procNodesFirst.prev();
                            if (nodeHelper.hasClass('contains-pagebreak')) {
                                processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                                contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                            } else {
                                processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                            }
                        }
                        if (processingNodes.length === 0) { // if we are somewhere on first page, no pageBreak class at the top!
                            processingNodes = $pageContentNode.children('.p, table');
                        }
                    } else if (_.isEmpty(currentProcessingNode)) { // fallback option if we don't fetch any valid node
                        processingNodes = $pageContentNode.children('.p, table');
                        currentProcessingNode = processingNodes.first();
                    }

                    if (processingNodes) {
                        // remove all break-above classes except the top one
                        processingNodes.removeClass('break-above').first().addClass('break-above');

                        currentPageNumber = self.getPageNumber(processingNodes.first());
                        if (currentPageNumber === 1 && firstPageHeight !== 0) {
                            pageMaxHeight = firstPageHeight;
                            currentPageTypeHeight = 'first';
                        } else if (currentPageNumber % 2 === 0 && evenPageHeight !== 0) {
                            pageMaxHeight = evenPageHeight;
                            currentPageTypeHeight = 'even';
                        } else if (defPageHeight !== 0) {
                            pageMaxHeight = defPageHeight;
                            currentPageTypeHeight = 'def';
                        }

                        // process all fetched nodes
                        self.iterateArraySliced(processingNodes, processNodeForPageBreaksInsert, { delay: 50 })
                        .done(function () {
                            pageBreaksCollection = $pageContentNode.find('.page-break');
                            // store indexes of pagebreaks as values for page numbers
                            _.each(pageBreaksCollection, function(pageBreakInCollection, index) {
                                $(pageBreakInCollection).data('page-num', index + 1);
                            });
                            pageCount = pageBreaksCollection.length + 1;

                            populateHeaderFooterPaginationClasses(pageBreaksCollection);
                            replaceAllTypesOfHeaderFooters();
                            updatePageCountFields();
                            updatePageNumberFields();

                            app.getView().recalculateDocumentMargin();
                        });
                    }
                }
            });
        };

    } // end of class PageLayout

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: PageLayout });


});
