/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/drawingstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/utils/attributeutils',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/drawinglayer/model/drawingstylecollection',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/text/drawingResize',
     'io.ox/office/text/dom',
     'io.ox/office/text/position'
    ], function (Utils, AttributeUtils, Border, DrawingStyleCollection, DrawingFrame, DrawingResize, DOM, Position) {

    'use strict';

    var // definitions for text-specific drawing attributes
        DEFINITIONS = {

            /**
             * Margin from top border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginTop: { def: 0 },

            /**
             * Margin from bottom border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginBottom: { def: 0 },

            /**
             * Margin from left border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginLeft: { def: 0 },

            /**
             * Margin from right border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginRight: { def: 0 },

            /**
             * Style, width and color of the left drawing border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the right drawing border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the top drawing border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the bottom drawing border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border));
                }
            },

            // object anchor --------------------------------------------------

            /**
             * Width of the drawing, as number in 1/100 of millimeters. The
             * value 0 will be interpreted as 100% width of the parent element.
             */
            width: {
                def: 0,
                format: function (element, width) {
                    if (!DrawingFrame.isGroupedDrawingFrame(element)) {
                        if (width === 0) {
                            element.width('100%');
                        } else {
                            element.width(Utils.convertHmmToLength(width, 'px', 1));
                        }
                    }
                }
            },

            /**
             * Height of the drawing, as number in 1/100 of millimeters.
             */
            height: {
                def: 0
            },

            /**
             * If set to true, the drawing is rendered as inline element ('as
             * character'), otherwise it is anchored relative to another
             * element (page, paragraph, table cell, ...).
             */
            inline: { def: true },

            anchorHorBase: { def: 'margin' },

            anchorHorAlign: { def: 'left' },

            anchorHorOffset: { def: 0 },

            anchorVertBase: { def: 'margin' },

            anchorVertAlign: { def: 'top' },

            anchorVertOffset: { def: 0 },

            /**
             * Specifies how text floats around the drawing.
             * - 'none': Text does not float around the drawing.
             * - 'square': Text floats around the bounding box of the drawing.
             * - 'tight': Text aligns to the left/right outline of the drawing.
             * - 'through': Text aligns to the entire outline of the drawing.
             * - 'topAndBottom': Text floats above and below the drawing only.
             */
            textWrapMode: { def: 'topAndBottom' },

            /**
             * Specifies on which side text floats around the drawing. Effective
             * only if the attribute 'textWrapMode' is either 'square',
             * 'tight', or 'through'.
             * - 'both': Text floats at the left and right side.
             * - 'left': Text floats at the left side of the drawing only.
             * - 'right': Text floats at the right side of the drawing only.
             * - 'largest': Text floats at the larger side of the drawing only.
             */
            textWrapSide: { def: 'both' }

        },

        // values for the 'textWrapMode' attribute allowing to wrap the text around the drawing
        WRAPPED_TEXT_VALUES = /^(square|tight|through)$/;

    // private global functions ===============================================

    /**
     * Returns whether the passed 'textWrapMode' attribute allows to wrap the
     * text around the drawing.
     */
    function isTextWrapped(textWrapMode) {
        return WRAPPED_TEXT_VALUES.test(textWrapMode);
    }

    // class TextDrawingStyles ================================================

    /**
     * Contains the style sheets for drawing formatting attributes. The CSS
     * formatting will be read from and written to drawing elements of any type.
     *
     * @constructor
     *
     * @extends DrawingStyleCollection
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {DocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function TextDrawingStyles(app, documentStyles) {

        // base constructor ---------------------------------------------------

        DrawingStyleCollection.call(this, app, documentStyles, {
            families: 'line fill image shape changes',
            formatHandler: updateDrawingFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Calculating the largest line height of a span inside a paragraph.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose spans will be investigated.
         *
         * @returns {Number}
         *  The largest line height of the spans in the paragraph in 1/100 mm.
         */
        function getMaximumLineHeightInParagraph(paragraph) {

            var // the largest value for the line height of a span in 1/100 mm
                maxCssLineHeight = 0;

            paragraph.children('span').each(function (index, span) {

                var lineHeight = Utils.convertCssLengthToHmm($(span).css('line-height'));

                if (lineHeight > maxCssLineHeight) {
                    maxCssLineHeight = lineHeight;
                }
            });

            return maxCssLineHeight;
        }

        /**
         * Modifying the offset for all floated drawings, that are positioned behind
         * the moved drawing.
         *
         * @param {jQuery} drawingNode
         *  The moved drawing node.
         *
         * @param {Number} addOffset
         *  The change of the offset node for the drawings, that are located behind the
         *  moved drawing node (in px).
         */
        function modifyFollowingOffsetNodes(drawingNode, addOffset) {

            var node = Utils.getDomNode(drawingNode),
                localDrawing = null;

            while (node) {
                if (DOM.isOffsetNode(node)) {
                    localDrawing = node.nextSibling;
                    $(node).height($(node).height() + addOffset);
                    $(node).remove();
                    $(node).insertBefore(localDrawing);
                }

                node = node.nextSibling;
            }
        }

        /**
         * Will be called for every drawing node whose attributes have been
         * changed. Repositions and reformats the drawing according to the
         * passed attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateDrawingFormatting(drawing, mergedAttributes) {

            var // the drawing attributes of the passed attribute map
                drawingAttributes = mergedAttributes.drawing,
                // the image attributes of the passed attribute map
                imageAttributes = mergedAttributes.image,
                // the paragraph element containing the drawing node
                // -> this can also be a drawing group without paragraphs
                paragraph = drawing.parent(),
                // total width of the paragraph, in 1/100 mm
                paraWidth = Utils.convertLengthToHmm(paragraph.width(), 'px'),
                // preceding node used for vertical offset
                verticalOffsetNode = drawing.prev(DOM.OFFSET_NODE_SELECTOR),
                // offset from top/left/right margin of paragraph element, in 1/100 mm
                topOffset = 0, leftOffset = 0, rightOffset = 0, verticalTopOffset = 0,
                // margins to be applied at the drawing
                topMargin = 0, bottomMargin = 0, leftMargin = 0, rightMargin = 0,
                // text wrapping side (left/right/none)
                wrapMode = 'none',
                // maximum line height in paragraph
                maxLineHeightInParagraph,
                // the page node
                pageNode = app.getModel().getNode(),
                // the current zoom factor in percent
                zoomFactor = app.getView().getZoomFactor() / 100,
                // the height of the top margin in 1/100 mm
                topMarginHeight = 0,
                // the width of the page in 1/100 mm
                pageWidth = Math.round(Utils.convertLengthToHmm(pageNode.outerWidth(true), 'px') / zoomFactor),
                // whether the drawing is positioned absolutely
                drawingPositionedToPage = false,
                // the horizontal and vertical offsets in pixel relative to the page
                topValue = 0, leftValue = 0,
                // an optional text indent of the paragraph
                textIndent = 0,
                // the drawing layer for absolutely positioned drawings
                drawingLayer = app.getModel().getDrawingLayer(),
                // whether the drawing is located in header or footer
                inHeaderFooter = app.isImportFinished() ? drawing.hasClass(DOM.TARGET_NODE_CLASSNAME) : (DOM.getMarginalTargetNode(pageNode, drawing).length > 0),
                // whether the top offset shall be used as distance to the bottom (used in footers)
                useBottomOffset = false,
                // the header or footer root node for the drawing
                targetRootNode = null,
                // the width of the border of the drawing
                borderWidth = mergedAttributes.line.width,
                // the half width of the border of the drawing
                // half border width goes only for text frames, full width for images; cannot be less than 1px (or 26.4 hmm), to be compatible with canvas border, #36576
                halfBorderWidth = DrawingFrame.isTextFrameShapeDrawingFrame(drawing) ? ((0.5 * borderWidth) > 26.5 ? (0.5 * borderWidth) : 26.5 ) : borderWidth,
                // current drawing width, in 1/100 mm
                // -> since the border is drawn with a canvas, the drawing width (including border) needs to be adjusted
                // -> the canvas draws half of the border width outside and half of it inside the drawing.
                drawingWidth = Utils.convertLengthToHmm(drawing.width(), 'px') + borderWidth,
                // the minimum height of a vertical offset node (in hmm). Lower values need to be handled by margin.
                verticalOffsetMinHeight = 600;

            // helper function for horizontal offsets
            // -> not used for in line drawings
            function calculateHorizontalOffset() {

                var // whether the drawing is horizontally positioned at the page
                    isHorizontalAtPage = false,
                    // the width of the left margin in 1/100 mm
                    leftMarginWidth = 0;

                // drawing horizontally aligned to column or margin
                if ((drawingAttributes.anchorHorBase === 'column') || (drawingAttributes.anchorHorBase === 'margin')) {

                    if (drawingPositionedToPage) {  // drawing is located in the drawing layer

                        isHorizontalAtPage = true;

                        switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            leftOffset = (pageWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftOffset = pageWidth - drawingWidth - 100; // '-100' (1mm) to avoid leaving the page
                            break;
                        case 'offset':
                            leftOffset = drawingAttributes.anchorHorOffset;
                            // increasing the left offset by the distance between page and paragraph
                            // Info: The paragraph can be located anywhere, for example in a text frame positioned at the page (35146)
                            leftMarginWidth = Math.round(Utils.convertLengthToHmm(Position.getPixelPositionToRootNodeOffset(pageNode, paragraph, zoomFactor * 100).x, 'px') / zoomFactor);
                            // if the paragraph has an additional left margin set as css attribute, this need to be handled, too (might come from a style ID (35146)
                            if (paragraph) { leftMarginWidth = leftMarginWidth - Utils.convertCssLengthToHmm(paragraph.css('margin-left')); }

                            leftOffset = leftOffset + leftMarginWidth;
                            if (leftOffset + drawingWidth > pageWidth) { leftOffset = pageWidth - drawingWidth; }
                            break;
                        default:
                            leftOffset = 0;
                        }
                    } else {
                        switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            leftOffset = (paraWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftOffset = paraWidth - drawingWidth;
                            break;
                        case 'offset':
                            leftOffset = drawingAttributes.anchorHorOffset;
                            if (drawingAttributes.anchorHorOffset + drawingWidth > paraWidth) {
                                leftOffset = paraWidth - drawingWidth;
                            }
                            break;
                        default:
                            leftOffset = 0;
                        }
                    }

                // drawing horizontally aligned to page
                } else if (drawingAttributes.anchorHorBase === 'page') {

                    if (drawingPositionedToPage) {  // drawing is located in the drawing layer (and not in header or footer)

                        isHorizontalAtPage = true;

                        switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            leftOffset = (pageWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftOffset = pageWidth - drawingWidth - 100; // '-100' (1mm) to avoid leaving the page
                            break;
                        case 'offset':
                            leftOffset = drawingAttributes.anchorHorOffset;
                            if (drawingAttributes.anchorHorOffset + drawingWidth > pageWidth) {
                                leftOffset = pageWidth - drawingWidth;
                            }
                            break;
                        default:
                            leftOffset = 0;
                        }

                    } else {  // the drawing will be positioned inside its paragraph

                        switch (drawingAttributes.anchorHorAlign) {
                        case 'center':
                            leftOffset = (paraWidth - drawingWidth) / 2;
                            break;
                        case 'right':
                            leftOffset = paraWidth - drawingWidth - 100;
                            break;
                        case 'offset':
                            // reducing the left offset by the distance between page and paragraph
                            leftOffset = drawingAttributes.anchorHorOffset;
                            if (drawingAttributes.anchorHorOffset + drawingWidth > pageWidth) { leftOffset = pageWidth - drawingWidth; }
                            // decreasing the left offset by the distance between page and paragraph (TODO: zoomFactor?)
                            leftMarginWidth = Utils.round((Utils.convertCssLengthToHmm(pageNode.css('padding-left')) * zoomFactor), 1);
                            leftOffset = leftOffset - leftMarginWidth;  // reducing offset by the left page margin
                            break;
                        default:
                            leftOffset = 0;
                        }

                    }

                // drawing horizontally aligned to character
                } else if (drawingAttributes.anchorHorBase === 'character') {
                    // Also 'character' looks better, if it is handled like 'column' (33276). The value for 'anchorHorOffset'
                    // is not relative to the character, but to the paragraph.
                    switch (drawingAttributes.anchorHorAlign) {
                    case 'center':
                        leftOffset = (paraWidth - drawingWidth) / 2;
                        break;
                    case 'right':
                        leftOffset = paraWidth - drawingWidth;
                        break;
                    case 'offset':
                        leftOffset = drawingAttributes.anchorHorOffset;
                        break;
                    default:
                        leftOffset = 0;
                    }
                } else {
                    // Other yet unsupported anchor bases
                    leftOffset = 0;
                }

                // finally setting the right offset
                rightOffset = (isHorizontalAtPage ? pageWidth : paraWidth) - leftOffset - drawingWidth;
            }

            // helper function to set vertical offset nodes
            // -> required only for floated drawings
            function handleVerticalOffsetNodes() {

                var // the height of the offset node before moving the drawing
                    oldVerticalOffsetNodeHeight = 0,
                    // the height of the offset node after moving the drawing
                    newVerticalOffsetNodeHeight = 0;

                // saving the old height of the vertical offset node (in pixel)
                if (verticalOffsetNode.length > 0) { oldVerticalOffsetNodeHeight = verticalOffsetNode.height(); }

                if (verticalTopOffset < verticalOffsetMinHeight) {
                    // offset less than defined value of verticalOffsetMinHeight: expand top margin to top of paragraph,
                    // otherwise the first text line overwrites the drawing (taking care of 36167)
                    // Comparing with constant value of 'verticalTopOffset' and not with calculated 'topOffset' to avoid
                    // 'jumping' drawings (36247)
                    topMargin += verticalTopOffset;
                    // remove offset node
                    verticalOffsetNode.remove();
                } else {
                    // create offset node if not existing yet
                    if (verticalOffsetNode.length === 0) {
                        verticalOffsetNode = $('<div>', { contenteditable: false }).addClass('float offset').width(1).insertBefore(drawing);
                    }
                    // set height of the offset node
                    verticalOffsetNode.height(Utils.convertHmmToLength(topOffset, 'px', 1));
                    // Sometimes the browser ignores the change of height of the vertical offset node. This happens often, if there are only
                    // small movements of the drawing upwards. The browser does not ignore the vertical offset node, if it is removed from
                    // the dom and immediately inserted again before the drawing (28312).
                    verticalOffsetNode.remove();
                    verticalOffsetNode.insertBefore(drawing);
                    newVerticalOffsetNodeHeight = verticalOffsetNode.height();
                }

                // modifying the vertical offset for all following floated drawings in the same paragraph
                if (newVerticalOffsetNodeHeight !== oldVerticalOffsetNodeHeight) {
                    // the drawing was moved upwards -> all following floated drawings in the same paragraph need an increased offset node
                    modifyFollowingOffsetNodes(drawing, oldVerticalOffsetNodeHeight - newVerticalOffsetNodeHeight);
                }

            }

            // helper function to set the text wrap mode
            // -> required only for floated drawings
            function setWrapMode() {

                if (isTextWrapped(drawingAttributes.textWrapMode)) {
                    switch (drawingAttributes.textWrapSide) {
                    case 'left':
                        wrapMode = 'left';
                        break;
                    case 'right':
                        wrapMode = 'right';
                        break;
                    case 'both':
                    case 'largest':
                        // no support for 'wrap both sides' in CSS, default to 'largest'
                        wrapMode = (leftOffset >= rightOffset) ? 'left' : 'right';
                        break;
                    default:
                        Utils.warn('updateDrawingFormatting(): invalid text wrap side: ' + drawingAttributes.textWrapSide);
                        wrapMode = 'none';
                    }
                } else {
                    // text does not float beside drawing
                    wrapMode = 'none';
                }

            }

            // setting the height of the drawing. This needs to be done here, because it is dependent from other
            // attribute values.
            function setDrawingHeight(drawing, attributes) {

                var // whether the height of the drawing shall be set
                    // -> no setting of height for grouped drawings and for shapes with property autoResizeHeight
                    setHeight = !(DrawingFrame.isGroupedDrawingFrame(drawing) || attributes.shape.autoResizeHeight);

                if (setHeight) {
                    drawing.height(Utils.convertHmmToLength(attributes.drawing.height, 'px', 1));
                } else if (attributes.shape.autoResizeHeight) {
                    drawing.height(''); // if 'autoResizeHeight' is set, an explicit height must be removed
                }

            }

            // reducing the  width of drawing to width of paragraph or page
            function setMaximumDrawingWidth(options) {

                var // whether the paragraph or the page width is used
                    useParagraphWidth = Utils.getBooleanOption(options, 'useParagraphWidth', true),
                    // the maximum allowed width
                    maxWidth = useParagraphWidth ? paraWidth : pageWidth;

                // reducing width of drawing to width of paragraph or page, if no left or right cropping is enabled (Task 30982)
                if ((drawingWidth > maxWidth) && (imageAttributes.cropLeft === 0) && (imageAttributes.cropRight === 0)) {
                    drawingWidth = maxWidth;
                    drawing.width(Utils.convertHmmToCssLength(drawingWidth - borderWidth, 'px', 1));
                }

            }

            // helper function to update attributes of children of drawings (only text frames yet)
            function handleChildAttributes(drawing) {

                var // the explicit drawing attributes
                    explicitDrawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
                    // the paragraph style sheet container
                    paragraphStyles = null,
                    // the container for the paragraphs and tables inside the drawing
                    textFrameNode = null;

                // checking the fill color (task 36385)
                if (explicitDrawingAttrs && explicitDrawingAttrs.fill) {

                    paragraphStyles = documentStyles.getStyleCollection('paragraph'),
                    textFrameNode = DrawingFrame.getTextFrameNode(drawing);

                    Utils.iterateSelectedDescendantNodes(textFrameNode, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                        paragraphStyles.updateElementFormatting(paragraph);
                    }, undefined, { children: true });
                }

            }

            // Setting the height of the drawing
            setDrawingHeight(drawing, mergedAttributes);

            // nothing to do, if paragraph is a groupcontent object
            // -> in this case the group is responsible for the correct attributes
            if (DrawingFrame.isGroupContentNode(paragraph)) {

                drawing.removeClass('inline float left right').addClass('absolute grouped');  // absolute positioned drawing inside its group!

                // set generic formatting to the drawing frame
                DrawingFrame.updateFormatting(app, drawing, mergedAttributes);

                // simply do nothing, let the group make further attribute settings
                return;
            }

            // setting marker for drawings in header or footer
            if (inHeaderFooter && !app.isImportFinished()) { drawing.addClass(DOM.TARGET_NODE_CLASSNAME); }

            // position
            if (drawingAttributes.inline) {

                // switch from floating to inline mode
                if (!drawing.hasClass('inline')) {

                    // reducing drawing width if required
                    setMaximumDrawingWidth({ useParagraphWidth: true });

                    // remove leading node used for positioning
                    verticalOffsetNode.remove();

                    // Word uses fixed predefined margins in in line mode, we too?
                    drawing.removeClass('float left right absolute').addClass('inline').css('margin', '0 1mm');
                    // ignore other attributes in in line mode

                    // whether the drawing needs to be removed from the drawing layer
                    if (DOM.isDrawingLayerNode(drawing.parent())) {
                        drawingLayer.setAbsoluteDrawingPosition(drawing, { doClean: true });
                        drawingLayer.shiftDrawingAwayFromDrawingLayer(drawing);

                        // the selection needs to be repainted. Otherwise it is possible
                        // to move the drawing to the page borders.
                        DrawingFrame.clearSelection(drawing);
                        // BUG 36372: Text frame gets blue resize/selection-border wrongly (via undo)
                        //  previously check if the drawing was selected before pressing undo
                        if (app.getModel().getSelection().isDrawingSelected(drawing)) {
                            DrawingResize.drawDrawingSelection(app, drawing);
                        }
                    }
                }

                // setting margin also to in-line drawings. This is necessary, because a thick border drawing with
                // canvas might overlap the surrounding text.
                drawing.css({ margin: Utils.convertHmmToCssLength(halfBorderWidth, 'px', 1) });

            } else {

                // calculate top offset (only if drawing is anchored to paragraph or to line)
                // anchor to line means the base line of the character position. so it is not correctly calculated
                // using the paragraph, but it is better to use it, than to ignore the value. the calculation is
                // nearly correct for characters in the first row inside the paragraph.
                if ((drawingAttributes.anchorVertBase === 'paragraph') || (drawingAttributes.anchorVertBase === 'line')) {

                    drawing.removeClass('inline absolute').addClass('float');

                    // whether the drawing needs to be removed from the drawing layer
                    // -> for the following calculations, the drawing must be located inside the paragraph
                    if (DOM.isDrawingLayerNode(drawing.parent())) {
                        drawingLayer.setAbsoluteDrawingPosition(drawing, { doClean: true });
                        drawingLayer.shiftDrawingAwayFromDrawingLayer(drawing);
                    }

                    if (drawingAttributes.anchorVertAlign === 'offset') {
                        topOffset = Math.max(drawingAttributes.anchorVertOffset, 0);
                        verticalTopOffset = topOffset;  // saving the value for later evaluation
                        // For an offset node it is necessary to subtract the height of the text position in
                        // the paragraph in which the drawing is located. The drawing cannot be located higher
                        // than its position allows it.
                        topOffset -= Math.round(Utils.convertLengthToHmm(drawing.offset().top - drawing.parent().offset().top, 'px') / zoomFactor);
                        // adding the height of an already existing offset node
                        if (($(drawing).prev().length) && (DOM.isOffsetNode($(drawing).prev()))) {
                            topOffset += Utils.convertLengthToHmm($(drawing).prev().height(), 'px');
                        }

                        if (topOffset < 0) { topOffset = 0; }
                        // But this is also no final solution, because the drawing is shifted
                        // downwards, if text is included before the drawing position in the paragraph.
                    } else {
                        // TODO: automatic alignment (top/bottom/center/...)
                        topOffset = 0;
                    }

                } else if ((drawingLayer.isActive()) && (drawingAttributes.anchorVertBase === 'page' || drawingAttributes.anchorVertBase === 'margin')) {

                    // setting position 'absolute' to all drawings that have a vertical offset corresponding to the page
                    drawingPositionedToPage = true;

                    // setting the header or footer target node for the drawing
                    if (inHeaderFooter) { targetRootNode = DOM.getMarginalTargetNode(pageNode, drawing); }

                    // if inHeaderFooter is true and targetRootNode is found, it might be better to shift only
                    // those drawings into the drawing layer, that are children of the 'div.header-footer-placeholder'.
                    // All other drawing layers will be removed soon and replaced by these header and footer
                    // templates, that are located inside 'div.header-footer-placeholder'.

                    // whether the drawing needs to be moved to the drawing layer
                    if (DOM.isParagraphNodeComplete(drawing.parent())) { drawingLayer.shiftDrawingIntoDrawingLayer(drawing, targetRootNode); }

                    drawing.removeClass('inline float left right').addClass('absolute');  // absolute positioned!

                    if (verticalOffsetNode.length > 0) {
                        verticalOffsetNode.remove();
                        verticalOffsetNode = null;
                    }

                    if (drawingAttributes.anchorVertAlign === 'offset') {

                        // checking height of top margin, not relying on 'padding-top'
                        topMarginHeight = (drawingAttributes.anchorVertBase === 'margin') ? (app.getModel().getPageLayout().getPageAttribute('marginTop') * zoomFactor) : 0;
                        // offset to 'margin' can be negative, but not to 'page'
                        topOffset = (drawingAttributes.anchorVertBase === 'margin') ? drawingAttributes.anchorVertOffset : Math.max(drawingAttributes.anchorVertOffset, 0);

                        if (inHeaderFooter) {
                            // is this a header or a footer drawing
                            if (DOM.getMarginalTargetNode(pageNode, drawing, { footer: true }).length > 0) {
                                useBottomOffset = true;
                                // the offset from the bottom of the page is the height of the page minus
                                // the specified vertical offset minus the height of the drawing.
                                topOffset = app.getModel().getPageLayout().getPageAttribute('height') - topOffset - topMarginHeight - Utils.convertLengthToHmm(drawing.height(), 'px');
                            } else {
                                // in the header, the page offset can be used without modifications
                                topOffset += topMarginHeight;
                            }
                        } else {
                            topOffset += topMarginHeight;
                        }
                        if (topOffset < 0) { topOffset = 0; }
                    } else {
                        // TODO: automatic alignment (top/bottom/center/...)
                        topOffset = 0;
                    }

                }

                // reducing drawing width if required
                setMaximumDrawingWidth({ useParagraphWidth: !drawingPositionedToPage });

                // calculate top/bottom drawing margins
                topMargin = Utils.minMax(drawingAttributes.marginTop, 0, topOffset);
                bottomMargin = Math.max(drawingAttributes.marginBottom, 0);

                // calculate left/right offset (required before setting wrap mode)
                calculateHorizontalOffset();

                // special handling for floated drawings
                if (!drawingPositionedToPage) {
                    // reducing top offset by top margin
                    if (verticalOffsetNode.length === 0) { topOffset -= topMargin; }
                    // handling of vertical offset nodes
                    handleVerticalOffsetNodes();
                    // determine the text wrapping side and setting the wrap mode
                    setWrapMode();
                }

                // setting calculated values at the drawing element
                if (drawingPositionedToPage) {

                    // reducing the value for left and top offset by the outer margin of the drawing
                    leftOffset -= drawingAttributes.marginLeft;
                    topOffset -= drawingAttributes.marginTop;

                    leftValue = Utils.convertHmmToLength(leftOffset, 'px', 1);
                    topValue = Utils.convertHmmToLength(topOffset, 'px', 1);

                    // the topValue is the distance in pixel to the top page border
                    // -> but the drawing is absolutely positioned to div.page in the drawing layer
                    drawingLayer.setAbsoluteDrawingPosition(drawing, { leftValue: leftValue, topValue: topValue, newValues: true, useBottomOffset: useBottomOffset, targetNode: targetRootNode });

                    // handling left and right margin for absolute drawings
                    leftMargin = Math.max(drawingAttributes.marginLeft, 0);
                    rightMargin = Math.max(drawingAttributes.marginRight, 0);

                } else {

                    // calculate left/right drawing margins, because the drawings are floated
                    switch (wrapMode) {

                    case 'left':
                        // drawing floats at right paragraph margin
                        rightMargin = rightOffset;
                        leftMargin = Math.max(drawingAttributes.marginLeft, 0);
                        // if there is less than 6mm space available for text, occupy all space (no wrapping)
                        if (leftOffset - leftMargin < 600) { leftMargin = Math.max(leftOffset, 0); }
                        break;

                    case 'right':
                        // drawing floats at left paragraph margin
                        leftMargin = leftOffset;
                        rightMargin = Math.max(drawingAttributes.marginRight, 0);
                        // if there is less than 6mm space available for text, occupy all space (no wrapping)
                        if (rightOffset - rightMargin < 600) { rightMargin = Math.max(rightOffset, 0); }
                        break;

                    default:
                        // no wrapping: will be modeled by left-floated with large CSS margins
                        wrapMode = 'right';
                        leftMargin = leftOffset;
                        rightMargin = Math.max(rightOffset, 0);
                    }

                    // set text wrap mode to drawing and offset node
                    drawing.add(verticalOffsetNode).removeClass('left right').addClass((wrapMode === 'left') ? 'right' : 'left');

                    // in Webkit browser there is a bug, that a floated drawing can be moved above the surrounding text at the top
                    // of the drawing. So if the drawing is not located in the first line of the paragraph (then it has an offset
                    // node), the drawing should get a top margin, so that it does not overlap the text (fix for 28011).
                    // Caused by task 36247, it is necessary, that the top margin is set independently from the calculated topOffset.
                    // Otherwise there is a 'jumping' text frame, when text is inserted -> comparing with the defined value
                    // 'verticalTopOffset' is more stable.
                    if (_.browser.WebKit && verticalTopOffset >= verticalOffsetMinHeight) {
                        // receiving the largest 'line-height' of the paragraph.
                        // TODO: Unfortunately this leads to an upwards movement, if the drawing is only moved horizontally
                        maxLineHeightInParagraph = getMaximumLineHeightInParagraph(paragraph);
                        topMargin += maxLineHeightInParagraph;
                        if (drawing.prev().hasClass('offset')) { // if margin is increased, decrease height of offset node for that value, so that page height doesn't get over max size
                            drawing.prev().height(drawing.prev().height() - Utils.convertHmmToLength(maxLineHeightInParagraph, 'px', 1 ));
                        }
                    }

                    // in Firefox the drawing overlaps text, if the paragraph has a negative indent. In this case, the drawing needs
                    // to get a right margin of the same size, so that the text is no longer 'catched' below the drawing (35042).
                    if (_.browser.Firefox) {
                        textIndent = Utils.convertCssLengthToHmm(paragraph.css('text-indent'));
                        if (textIndent < 0 && rightMargin < -textIndent) { rightMargin = -textIndent; }  // margin must be at least text indent
                    }
                }

                // apply CSS formatting to drawing node
                // -> adding the half border width to the margin (border is drawn with a canvas)
                drawing.css({
                    marginTop: Utils.convertHmmToCssLength(topMargin + halfBorderWidth, 'px', 1),
                    marginBottom: Utils.convertHmmToCssLength(bottomMargin + halfBorderWidth, 'px', 1),
                    marginLeft: Utils.convertHmmToCssLength(leftMargin + halfBorderWidth, 'px', 1),
                    marginRight: Utils.convertHmmToCssLength(rightMargin + halfBorderWidth, 'px', 1)
                });
            }

            // change track attribute handling
            app.getModel().getChangeTrack().updateChangeTrackAttributes(drawing, mergedAttributes);

            // set generic formatting to the drawing frame
            DrawingFrame.updateFormatting(app, drawing, mergedAttributes);

            // update formatting of all paragraphs inside the text frame content node (36385)
            if (DrawingFrame.isTextFrameShapeDrawingFrame(drawing)) { handleChildAttributes(drawing); }

            // set visibility of anchor
            if (app.getModel().getSelection().isDrawingSelected(drawing)) {
                app.getView().setVisibleDrawingAnchor(drawing, mergedAttributes);
            }
        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        documentStyles.registerAttributeDefinitions('drawing', DEFINITIONS);

    } // class TextDrawingStyles

    // static methods ---------------------------------------------------------

    /**
     * Returns the drawing position identifier representing the passed drawing
     * attributes.
     *
     * @param {Object} attributes
     *  A map with drawing attributes, as name/value pairs.
     *
     * @returns {String|Null}
     *  The GUI drawing position identifier; or null, if any of the passed
     *  attributes is ambiguous.
     */
    TextDrawingStyles.getPositionFromAttributes = function (attributes) {

        // inline mode overrules floating attributes
        if (_.isNull(attributes.inline)) { return null; }
        if (attributes.inline === true) { return 'inline'; }

        // filter unsupported position anchor modes
        if (!/^(page|column)$/.test(attributes.anchorHorBase) || !/^(page|paragraph|margin|line)$/.test(attributes.anchorVertBase)) { return null; }

        // filter ambiguous or unsupported horizontal alignment types
        if (!/^(left|center|right)$/.test(attributes.anchorHorAlign)) { return null; }

        // filter ambiguous text wrapping modes
        if (_.isNull(attributes.textWrapMode) || _.isNull(attributes.textWrapSide)) { return null; }

        // build the resulting drawing position
        return attributes.anchorHorAlign + ':' + (isTextWrapped(attributes.textWrapMode) ? attributes.textWrapSide : 'none');
    };

    /**
     * Returns the drawing attributes that are needed to represent the passed
     * GUI drawing position.
     *
     * @param {String} position
     *  The GUI drawing position.
     *
     * @param {Object} [origAttributes]
     *  The original attributes of the target drawing object.
     *
     * @param {Object}
     *  A map with drawing attributes, as name/value pairs.
     */
    TextDrawingStyles.getAttributesFromPosition = function (position, origAttributes) {

        var // extract alignment and text wrap mode
            matches = position.match(/^([a-z]+):([a-z]+)$/),
            // resulting drawing attributes
            drawingAttributes = {},
            // the horizontal anchor for the operation
            newAnchorHorBase = null;

        // inline flag overrules all other attributes, no need to return any other (unfortunately it is still necessary for undo reasons)
        if (position === 'inline') {
            return { inline: true,
                     anchorHorBase: null,
                     anchorHorAlign: null,
                     anchorHorOffset: null,
                     anchorVertBase: null,
                     anchorVertAlign: null,
                     anchorVertOffset: null,
                     textWrapMode: null,
                     textWrapSide: null };
        }

        // check that passed position contains alignment and text wrapping mode
        if (!_.isArray(matches) || (matches.length < 3)) {
            Utils.warn('TextDrawingStyles.getAttributesFromPosition(): invalid drawing position: "' + position + '"');
            return {};
        }

        // set horizontal alignment
        newAnchorHorBase = origAttributes.anchorHorBase === 'page' ? 'page' : 'column';
        _(drawingAttributes).extend({ inline: false, anchorHorBase: newAnchorHorBase, anchorHorAlign: matches[1], anchorHorOffset: 0 });

        // set text wrapping mode
        if (matches[2] === 'none') {
            _(drawingAttributes).extend({ textWrapMode: 'topAndBottom', textWrapSide: null });
        } else {
            _(drawingAttributes).extend({ textWrapMode: 'square', textWrapSide: matches[2] });
        }

        // reset vertical alignments for unsupported positions ('margin')
        if (_.isObject(origAttributes) && /^(margin)$/.test(origAttributes.anchorVertBase)) {
            _(drawingAttributes).extend({ anchorVertBase: 'paragraph', anchorVertOffset: 0 });
        }

        return drawingAttributes;
    };

    // exports ================================================================

    // derive this class from class DrawingStyleCollection
    return DrawingStyleCollection.extend({ constructor: TextDrawingStyles });

});
