/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/drawingLayer', [
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/text/utils/textutils',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/format/characterstyles',
    'io.ox/office/text/position'
], function (TriggerObject, TimerMixin, DrawingFrame, AttributeUtils, Utils, Operations, DOM, CharacterStyles, Position) {

    'use strict';

    // class DrawingLayer =====================================================

    /**
     * An instance of this class represents the model for all drawings in the
     * drawing layer of the edited document.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     *
     * @param {TextModel} model
     *  The text model instance.
     */
    function DrawingLayer(app, rootNode) {

        var // self reference
            self = this,
            // a list of all drawings (jQuery) in the drawing layer
            drawings = [],
            // the page layout
            pageLayout = null,
            // the page styles of the document
            pageAttributes = null,
            // a place holder ID for absolute positioned drawings
            placeHolderDrawingID = 1000,
            // the text model object
            model = null,
            // whether the OX Text drawing layer is active (disable this for testing
            // or performance reasons)
            isActive = true;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Initializing page attributes needed for calculation of page breaks.
         * In some scenarios, like ex. loading from local storage,
         * initialPageBreaks method is not called, and variables are not initialized for later usage.
         *
         */
        function initializePageAttributes() {
            pageAttributes = model.getStyleCollection('page').getElementAttributes(rootNode);
        }

        /**
         * Returns whether a drawing overlaps with the page content node. In this case
         * it is necessary to create a space maker element below the drawing. If the drawing
         * is completely located in the border region, no space maker element is required.
         *
         * @param {Object} drawingAttributes
         *  The explicit attributes at the drawing node.
         */
        function drawingOverlapsPageContent(drawingAttrs) {

            // drawingAttrs: {"left":0,"top":0,"width":3970,"height":2980,
            // "name":"Grafik 1","description":"","flipH":true,"flipV":false,"replacementData":"","imageUrl":"word/media/image1.jpeg",
            // "imageData":"","cropLeft":0,"cropRight":0,"cropTop":0,"cropBottom":0,
            // "marginTop":0,"marginBottom":0,"marginLeft":317,"marginRight":317,
            // "borderLeft":{"style":"none"},"borderRight":{"style":"none"},"borderTop":{"style":"none"},"borderBottom":{"style":"none"},
            // "inline":false,"anchorHorBase":"page","anchorHorAlign":"offset","anchorHorOffset":4000,"anchorVertBase":"page",
            // "anchorVertAlign":"offset","anchorVertOffset":6000,"textWrapMode":"topAndBottom","textWrapSide":"both"}

            // pageAttributes: {"width":21000,"height":29700,"marginLeft":2499,"marginRight":2499,"marginTop":2499,"marginBottom":2000}
            if (!pageAttributes) {
                initializePageAttributes();
            }

            // drawing in top page frame
            if ((drawingAttrs.anchorVertBase === 'page') && (drawingAttrs.anchorVertOffset + drawingAttrs.height <= pageAttributes.page.marginTop)) { return false; }
            // drawing in left page frame
            if ((drawingAttrs.anchorHorBase === 'page') && (drawingAttrs.anchorHorOffset + drawingAttrs.width <= pageAttributes.page.marginLeft)) { return false; }
            // drawing in right page frame
            if ((drawingAttrs.anchorHorBase === 'page') && (drawingAttrs.anchorHorOffset >= pageAttributes.page.width - pageAttributes.page.marginRight)) { return false; }

            // TODO: Drawing overlaps with header and footer -> also no space maker element required
            // TODO: anchorVertBase can also be 'column'
            // TODO: anchorHorBase can be any other type

            return true;
        }


        /**
         * Returns whether the passed 'textWrapMode' attribute allows to wrap the
         * text around the drawing.
         *
         * @param {String} textWrapMode
         *  The text wrap mode of a drawing element
         */
        function isTextWrapped(textWrapMode) {

            var // values for the 'textWrapMode' attribute allowing to wrap the text around the drawing
                WRAPPED_TEXT_VALUES = /^(square|tight|through)$/;

            return WRAPPED_TEXT_VALUES.test(textWrapMode);
        }

        /**
         * Inserting the space maker node below the absolutely positioned drawing.
         *
         * @param {Number[]} position
         *  The logical position, at which the space maker element needs to be inserted.
         *
         * @param {jQuery} drawing
         *  The drawing node, already jQuerified.
         *
         * @param {Object} drawingAttributes
         *  The explicit drawing attributes at the drawing node.
         *
         * @param {Object} lineAttributes
         *  The explicit line (border) attributes at the drawing node.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         */
        function insertDrawingSpaceMaker(position, drawing, drawingAttributes, lineAttributes, zoomFactor) {

            var // the space maker node below the drawing
                spaceMakerNode = null,
                // the width of the space maker
                width = 0,
                // the height of the space maker
                height = 0,
                // the paragraph element, parent of the drawing
                paragraph = $(Position.getParagraphElement(rootNode, _.initial(position))), // not drawing.parent(), because this is the drawing layer node
                // the width of the paragraph in px
                paraWidth = paragraph.outerWidth(true),
                // the zoomFactor used for calculations
                zoomValue = zoomFactor / 100,
                // the left distance of the paragraph to the root node (taking care of tables)
                paragraphOffset = Position.getPixelPositionToRootNodeOffset(rootNode, paragraph, zoomFactor),
                // the distance from left border of paragraph to the left border of the drawing in pixel
                leftDistance = Math.round((drawing.position().left - paragraphOffset.x) / zoomValue),
                // the distance from right border of the drawing to the right border of the paragraph in pixel
                rightDistance = Math.round((paragraphOffset.x / zoomValue) + paraWidth - ((drawing.position().left / zoomValue) + drawing.outerWidth(true))),
                // the distance from top border of paragraph to the top border of the drawing in pixel
                topDistance = Math.round((drawing.position().top - paragraphOffset.y) / zoomValue),
                // text wrapping side (left/right/none)
                wrapMode = 'none',
                // the id of the drawing (required for restoring from local storage)
                drawingID = drawing.attr('data-drawingID');

            if (isTextWrapped(drawingAttributes.textWrapMode)) {
                switch (drawingAttributes.textWrapSide) {
                case 'left':
                    wrapMode = 'left';
                    break;
                case 'right':
                    wrapMode = 'right';
                    break;
                case 'both':
                case 'largest':
                    // no support for 'wrap both sides' in CSS, default to 'largest'
                    wrapMode = (leftDistance >= rightDistance) ? 'left' : 'right';
                    break;
                default:
                    Utils.warn('insertDrawingSpaceMaker(): invalid text wrap side: ' + drawingAttributes.textWrapSide);
                    wrapMode = 'none';
                }
            } else {
                // text does not float beside drawing
                wrapMode = 'none';
            }

            // calculating the width of the space maker
            // -> outerWidth already contains the border width, although it is drawing with canvas. But an additional
            //    margin was set to the drawing before, representing the border.
            if (wrapMode === 'none') { width = paraWidth; }
            else if (wrapMode === 'left') { width = drawing.outerWidth(true) + rightDistance; }
            else if (wrapMode === 'right') { width = drawing.outerWidth(true) + leftDistance; }

            // setting the height of the space maker node
            height = drawing.outerHeight(true);

            // it might be necessary to reduce the height, if the drawing is above the paragraph
            if (topDistance < 0) { height += topDistance; }

            // inserting div element to create space for the drawing
            spaceMakerNode = $('<div>').addClass('float ' + ((wrapMode === 'left') ? 'right ' : 'left ') + DOM.DRAWING_SPACEMAKER_CLASS)
                                   .css('position', 'relative')
                                   .height(height + 10)  // TODO: Adding 10 px is not precise enough
                                   .width(width + 10)
                                   .attr('data-drawingID', drawingID);  // Adding the drawing id (required for local storage )

            if (_.last(position) === 0) {
                // inserting at the beginning of the paragraph, no empty span before
                paragraph.prepend(spaceMakerNode);
            } else {
                // splitting text span
                spaceMakerNode.insertAfter(model.prepareTextSpanForInsertion(position));
            }

            // registering the space maker at the absolute positioned drawing
            drawing.data(DOM.DRAWING_SPACEMAKER_LINK, Utils.getDomNode(spaceMakerNode));
        }

        /**
         * Sorting the drawings in the order from top to bottom in the document. This is not
         * necessarily the order inside the DOM.
         *
         * @param {jQuery[]} drawings
         *  The list with all drawings in the drawing layer.
         */
        function sortFromTopToBottomOnOnePage(drawings) {
            return _.sortBy(drawings, function (drawing) {
                return AttributeUtils.getExplicitAttributes(drawing).drawing.anchorVertOffset;
            });
        }

        /**
         * Removing the space maker node that is assigned to a specified drawing node.
         * Additionally it might be necessary to merge the neighboring text nodes
         *
         * @param {Node} drawingSpaceMaker
         *  The space maker element of a drawing.
         */
        function removeDrawingSpaceMaker(drawingSpaceMaker) {

            if (!drawingSpaceMaker) { return; }

            var prev = drawingSpaceMaker.previousSibling,
                tryToMerge = prev && DOM.isTextSpan(prev);
            $(drawingSpaceMaker).remove();
            if (tryToMerge) { CharacterStyles.mergeSiblingTextSpans(prev, true); }
        }

        /**
         * Refreshing the links between a drawing from the drawing layer and its
         * place holder node and its space maker node.
         *
         * @param {Node|jQuery} absoluteDrawing
         *  The absolute positioned drawing in the drawing layer node.
         *
         * @param {Node|jQuery} contentNode
         *  The node, that contains the place holder nodes. This can be the page
         *  content node or any footer or header node.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isMainDrawingLayer=false]
         *      If set to true, this refresh affects the main drawing layer located
         *      next to the page content node. Otherwise the drawing layers in the
         *      header or footer are affected.
         */
        function refreshLinkForAbsoluteDrawing(absoluteDrawing, contentNode, options) {

            var // the jQuery version of the drawing
                $absoluteDrawing = $(absoluteDrawing),
                // the drawing id of the drawing in the drawing layer
                drawingID = $absoluteDrawing.attr('data-drawingID'),
                // the selector string for the search for place holder nodes and space maker nodes
                selectorString = '[data-drawingID=' + drawingID + ']',
                // finding the place holder and optionally the space maker node, that are located inside the page content node
                foundNodes = $(contentNode).find(selectorString),
                // whether the corresponding place holder node was found (this is required)
                placeHolderFound = false,
                // whether the main drawing layer is refreshed, no header or footer
                isMainDrawingLayer = Utils.getBooleanOption(options, 'isMainDrawingLayer', false);

            if (_.isString(drawingID)) { drawingID = parseInt(drawingID, 10); }

            // updating the value for the global drawing ID, so that new drawings in drawing layer
            // get an increased number.
            if (isMainDrawingLayer && _.isNumber(drawingID) && (drawingID >= placeHolderDrawingID)) { placeHolderDrawingID = drawingID + 1; }

            if (foundNodes.length > 0) {

                _.each(foundNodes, function (oneNode) {

                    if (DOM.isDrawingPlaceHolderNode(oneNode)) {
                        // creating links in both directions
                        $absoluteDrawing.data(DOM.DRAWINGPLACEHOLDER_LINK, oneNode);
                        $(oneNode).data(DOM.DRAWINGPLACEHOLDER_LINK, Utils.getDomNode(absoluteDrawing));
                        placeHolderFound = true;
                    } else if (DOM.isDrawingSpaceMakerNode(oneNode)) {
                        // there might be a space maker node with this drawingID, too
                        // -> but not every drawing has a space maker node
                        // creating links in both directions
                        $absoluteDrawing.data(DOM.DRAWING_SPACEMAKER_LINK, oneNode);
                    }
                });

                if (!placeHolderFound) {
                    Utils.error('DrawingLayer.refreshDrawingLayer(): failed to find place holder node with drawing ID: ' + drawingID);
                }

            } else {
                Utils.error('DrawingLayer.refreshDrawingLayer(): failed to find place holder node with drawingID: ' + drawingID);
            }

        }

        // methods ------------------------------------------------------------

        /**
         * Whether the drawing layer is active. This can be disabled for testing
         * or performance reasons.
         *
         * @returns {Boolean}
         *  Whether the drawing layer is active.
         */
        this.isActive = function () {
            return isActive;
        };

        /**
         * Whether the document contains absolutely positioned drawings
         *
         * @returns {Boolean}
         *  Whether the document contains at least one absolutely positioned
         *  drawing.
         */
        this.isEmpty = function () {
            return drawings.length === 0;
        };

        /**
         * Whether the document contains absolutely positioned drawings
         * inside the page content node. This excludes all drawings, that
         * are located in the header or footer.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one absolutely positioned
         *  drawing in the page content node.
         */
        this.containsDrawingsInPageContentNode = function () {

            var // the drawing layer node inside the page div.page
                drawingLayerNode = self.returnDrawingLayerNode();

            return drawingLayerNode && drawingLayerNode.length > 0 && drawingLayerNode.children().length > 0;
        };

        /**
         * Provides the array with all drawings in the drawing layer.
         *
         * @returns {jQuery[]}
         *  The list with all drawings in the drawing layer.
         */
        this.getDrawings = function () {
            return drawings;
        };

        /**
         * Registering one drawing for the drawing layer
         * outside the page content node.
         *
         * @param {Node|jQuery} drawing
         *  One drawing node that is positioned absolutely.
         */
        this.addIntoDrawingLayer = function (drawing) {
            drawings.push($(drawing));
        };

        /**
         * Provides a unique ID for the drawings in the drawing layer and
         * its placeholder elements.
         *
         * @returns {Number}
         *  A unique id.
         */
        this.getNextDrawingID = function () {
            return placeHolderDrawingID++;
        };

        /**
         * Updating the space maker element below the drawings in the drawing layer.
         * This function is called very often and is therefore performance critical.
         * Also needs to update the vertical pixel position of the absolute drawings,
         * because this changes, if the drawing changes the page.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceUpdateAllDrawings=false]
         *      If set to true, all drawing positions are updated, even if the page
         *      did not change. This is especially necessary after loading the
         *      document.
         */
        this.updateAbsoluteDrawings = function (options) {

            Utils.takeTime('DrawingLayer.updateAbsoluteDrawings(): ', function () {

                var // setting zoom factor, keeping percentage
                    zoomFactor = app.getView().getZoomFactor(),
                    // a collector for all page numbers
                    allPageNumbers = [],
                    // a helper object for all drawing nodes
                    allSortedDrawings = {},
                    // an ordered collection of the absolute positioned drawings
                    sortedDrawings = [],
                    // the key name for saving the current page number at the drawing node
                    dataPageKey = 'currentPageNumber',
                    // check if a processing node is registered, so that only following drawings
                    currentNode = model.getCurrentProcessingNode(),
                    // setting a page number until which the drawings need to be updated
                    startPageNumber = 1,
                    // whether the update of all drawings is forced
                    forceUpdateAllDrawings =  Utils.getBooleanOption(options, 'forceUpdateAllDrawings', false);

                if (!pageLayout) { pageLayout = model.getPageLayout(); }

                // Performance: Update only following pages
                if (currentNode) {
                    startPageNumber = pageLayout.getPageNumber(currentNode);
                    // was is successful to determine the start page number?
                    startPageNumber = startPageNumber || 1;

                    if (startPageNumber > 1) { startPageNumber--; }
                }

                // no updates and space maker for drawings in header or footer
                // TODO: This needs to be modified, if header and footer become writable
                sortedDrawings = _.filter(drawings, function (drawing) {
                    return !drawing.hasClass(DOM.TARGET_NODE_CLASSNAME);
                });

                // sorting the drawings corresponding to their appearance in the document
                _.each(sortedDrawings, function(drawing) {
                    // on which page is the place holder node?
                    var page = pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawing));
                    // was is successful to determine the start page number?
                    if (!page) { page = 0; }
//                    if (!page) {
//                        // Try to reuse the number from the cache
//                        if (drawing.data('pageNumber')) { page = drawing.data('pageNumber'); }
//                    }

                    // updating only following drawings
                    if (page >= startPageNumber) {
                        if (!_.contains(allPageNumbers, page)) { allPageNumbers.push(page); }
                        if (!allSortedDrawings[page]) { allSortedDrawings[page] = []; }
                        allSortedDrawings[page].push(drawing);
                        // saving the page number at the drawing
                        $(drawing).data(dataPageKey, page);
                    }
                });

                // sorting the pages
                allPageNumbers.sort(function (a, b) { return a - b; });

                // reset the sorted drawings helper array
                sortedDrawings = [];

                // sorting all drawings on one page corresponding to their vertical offset
                _.each(allPageNumbers, function (page) {
                    if (allSortedDrawings[page].length > 1) {
                        allSortedDrawings[page] = sortFromTopToBottomOnOnePage(allSortedDrawings[page]);  // order is now from top to down in the visibility
                    }
                    // concatenating all drawings into one array
                    sortedDrawings = sortedDrawings.concat(allSortedDrawings[page]);
                });

                // check, if a drawing needs a new vertical offset, caused by page change
                _.each(sortedDrawings, function(drawing) {
                    self.setAbsoluteDrawingPosition(drawing, { forceUpdateAllDrawings: forceUpdateAllDrawings });
                });

                // removing the existing space maker nodes -> maybe they can be reused in the future?
                _.each(sortedDrawings, function(drawing) {
                    var // the space maker node below the drawing
                        spaceMakerNode = DOM.getDrawingSpaceMakerNode(drawing);

                    if (spaceMakerNode) {
                        removeDrawingSpaceMaker(spaceMakerNode);
                        drawing.removeData(DOM.DRAWING_SPACEMAKER_LINK);
                    }
                });

                // iterate over all drawings -> is a space maker node element required below the drawing?
                _.each(sortedDrawings, function(drawing) {

                    var // the attributes explicitely set at the drawing
                        allAttrs = AttributeUtils.getExplicitAttributes(drawing),
                        // the drawing attributes set at the drawing
                        drawingAttrs = allAttrs.drawing,
                        // the line attributes set at the drawing
                        lineAttrs = allAttrs.line || {},
                        // the horizontal and vertical offset of the drawing relative to the page
                        pos = null,
                        // the page info object for a specified pixel position
                        pageInfo = null;
                        // the page number at which the drawing is positioned
                        // page = drawing.data(dataPageKey);

                    // immediately remove the current page number
                    drawing.removeData(dataPageKey);

                    if (drawingOverlapsPageContent(drawingAttrs)) {

                        // -> update or create the space maker element
                        // calculating pixel position relative to the page
                        pos = Position.getPixelPositionToRootNodeOffset(model.getNode(), drawing, zoomFactor);

                        try {
                            // calculating the logical position at the specified position (first position in line)
                            pageInfo = Position.getPositionFromPagePixelPosition(model.getNode(), pos.x, pos.y, zoomFactor, { getFirstTextPositionInLine: true });
                        } catch (ex) {}

                        if (pageInfo && pageInfo.pos) {
                            insertDrawingSpaceMaker(pageInfo.pos, drawing, drawingAttrs, lineAttrs, zoomFactor);
                        }
                    }
                });

                Utils.log('Number of drawings: ' + sortedDrawings.length);
            });
        };

        /**
         * Setting the 'left' and the 'top' css values for absolutely
         * positioned drawings. This function is called, if drawings
         * are inserted into the drawing layer ('newValues' is true) or
         * if they are removed from the drawing layer('doClean' is true)
         * or during update of the positions (called from
         * self.updateAbsoluteDrawings). An update of the vertical
         * position can be necessary, if a paragraph containing a drawing
         * place holder element moves to another page. In this case the
         * absolute positioned drawing has to change its position, too.
         *
         * @param {jQuery} drawingNode
         *  One drawing node that is positioned absolutely. For convenience
         *  reasons this can also be the place holder node.
         *
         * @param {Number} leftValue
         *  The left page offset in pixel.
         *
         * @param {Number} topValue
         *  The top page offset in pixel. This is only the offset to
         *  the current page, not to the div.page element.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.newValues=false]
         *      If set to true, the drawing was inserted, set 'toPage' or modified
         *      in size or position. In this case the data values stored at the
         *      element need to be updated.
         *  @param {Boolean} [options.doClean=false]
         *      If set to true, the drawing was removed from the drawing layer. In
         *      this case the data values stored at the element need to be deleted.
         *  @param {Boolean} [options.useBottomOffset=false]
         *      If set to true, the top value for the vertical offset is interpreted
         *      as bottom offset. This is used for positions in footers.
         *  @param {Boolean} [options.forceUpdateAllDrawings=false]
         *      If set to true, all drawing positions are updated, even if the page
         *      did not change. This is especially necessary after loading the
         *      document.
         *  @param {leftValue} [options.leftValue=0]
         *      The left page offset in pixel.
         *  @param {topValue} [options.topValue=0]
         *      The top page offset in pixel.
         *  @param {jQuery} [options.targetNode]
         *      An optional target node as an alternative to the page div.page. This
         *      is used for header or footer nodes.
         */
        this.setAbsoluteDrawingPosition = function (drawingNode, options) {

            var // whether the drawing offsets shall be cleaned
                doClean = Utils.getBooleanOption(options, 'doClean', false),
                // whether the values have changed (called from drawingStyles.updateDrawingFormatting)
                newValues = Utils.getBooleanOption(options, 'newValues', false),
                // whether the update of all drawings is forced
                forceUpdateAllDrawings =  Utils.getBooleanOption(options, 'forceUpdateAllDrawings', false),
                // the left page offset in pixel
                leftValue = Utils.getNumberOption(options, 'leftValue', 0),
                // the top page offset in pixel
                topValue = Utils.getNumberOption(options, 'topValue', 0),
                // the page number, at which the drawing place holder is located
                pageNumber = 1,
                // the vertical and horizontal offsets relative to the page
                horizontalPageOffset = 0, verticalPageOffset = 0,
                // the vertical offset in the document
                documentOffset = 0,
                // the drawing in the drawing layer
                drawing = DOM.isDrawingPlaceHolderNode(drawingNode) ? $(DOM.getDrawingPlaceHolderNode(drawingNode)) : drawingNode,
                // an optional target node in header or footer
                inHeaderOrFooter = options && options.targetNode && options.targetNode.length > 0,
                // whether the topValue needs to be set as bottom offset (used in footers)
                useBottomOffset = Utils.getBooleanOption(options, 'useBottomOffset', false);

            if (newValues) {

                // called from drawingStyles.updateDrawingFormatting
                horizontalPageOffset = leftValue;
                verticalPageOffset = topValue;

                if (!pageLayout) { pageLayout = model.getPageLayout(); }

                // determining the page number of the drawing, after the document is loaded completely.
                // The page number is always 1 inside headers or footers.
                if (app.isImportFinished() && !inHeaderOrFooter) {
                    pageNumber = pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawing));
                    drawing.data('pageNumber', pageNumber);
                }

                // saving the values at the drawing node
                drawing.data('verticalPageOffset', verticalPageOffset);
                drawing.data('horizontalPageOffset', horizontalPageOffset);

                documentOffset = (pageNumber === 1) ? verticalPageOffset : Position.getVerticalPagePixelPosition(rootNode, pageLayout, pageNumber, app.getView().getZoomFactor(), verticalPageOffset);

            } else if (doClean) {

                // removing the values at the drawing node
                drawing.removeData('verticalPageOffset');
                drawing.removeData('horizontalPageOffset');
                drawing.removeData('pageNumber');

                documentOffset = 0;
                horizontalPageOffset = 0;

            } else {
                // this is the standard update case, only drawing is defined
                // -> modifications are only necessary, if the page has changed
                if (!pageLayout) { pageLayout = model.getPageLayout(); }

                if (drawing.data('currentPageNumber')) {
                    pageNumber = drawing.data('currentPageNumber');
                } else {
                    pageNumber = pageLayout.getPageNumber(DOM.getDrawingPlaceHolderNode(drawing));
                }

                // if it failed to determine the page, simply do nothing (leave drawing where it is)
                if (!pageNumber) { return; }

                // page number not changed, nothing to do
                // -> but updating after first load is required, to be more precise!
                if (!forceUpdateAllDrawings && (pageNumber === drawing.data('pageNumber'))) { return; }

                drawing.data('pageNumber', pageNumber); // setting updated value

                horizontalPageOffset = drawing.data('horizontalPageOffset');
                verticalPageOffset = drawing.data('verticalPageOffset');

                documentOffset = (pageNumber === 1) ? verticalPageOffset : Position.getVerticalPagePixelPosition(rootNode, pageLayout, pageNumber, app.getView().getZoomFactor(), verticalPageOffset);
            }

            if (useBottomOffset) {
                drawing.css({bottom: documentOffset, left: horizontalPageOffset});
            } else {
                drawing.css({top: documentOffset, left: horizontalPageOffset});
            }
        };

        /**
         * Shifting a drawing from the page content node into the drawing
         * layer node. This happens during loading the document or if a
         * drawing mode is switched from 'asCharacter' or 'toParagraph' to
         * 'toPage'.
         *
         * @param {jQuery} drawing
         *  One drawing node that is positioned absolutely.
         *
         * @param {jQuery} [target]
         *  An optional alternative for the destination of the drawing layer. If not
         *  specified, div.page needs to be the parent of the drawing layer. For
         *  headers and footers, it is also possible, that div.header or div.footer
         *  are the parent for the drawing layer.
         */
        this.shiftDrawingIntoDrawingLayer = function (drawing, target) {

            var // the drawing layer node for absolute positioned drawings (will be created, if required)
                drawingLayerNode = self.getDrawingLayerNode(target),
                // the place holder element for the drawing in the page content
                drawingPlaceHolder = $('<div>', { contenteditable: false }).addClass('inline ' + DOM.DRAWINGPLACEHOLDER_CLASS),
                // a drawing ID to connect drawing and drawingPlaceHoder node
                drawingID = self.getNextDrawingID(),
                // whether the drawing needs to be registered in the drawing collection
                doRegister = true;

            // creating and inserting the drawing place holder behind the drawing
            drawing.after(drawingPlaceHolder);

            // removing an offset element before the drawing, if there is one
            if (DOM.isOffsetNode(drawing.prev())) { drawing.prev().remove(); }

            // shifting the drawing itself into the drawing layer node
            drawingLayerNode.append(drawing);

            drawing.attr('data-drawingID', drawingID);
            drawingPlaceHolder.attr('data-drawingID', drawingID);

            // direct link between the drawing and drawing place holder
            drawing.data(DOM.DRAWINGPLACEHOLDER_LINK, Utils.getDomNode(drawingPlaceHolder));
            drawingPlaceHolder.data(DOM.DRAWINGPLACEHOLDER_LINK, Utils.getDomNode(drawing));

            // and finally registering it at the drawing layer handler
            // -> do not register drawings in header or footer, that are NOT located
            // inside 'div.header-footer-placeholder'. All other drawings are
            // 'throw-away' drawings.
            if (DOM.isHeaderOrFooter(target) && !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), target)) { doRegister = false; }

            if (doRegister) { self.addIntoDrawingLayer(drawing); }
        };

        /**
         * Shifting a drawing from the drawing layer node into the page content
         * node. This happens if a drawing mode is switched from 'toPage' to
         * 'asCharacter' or 'toParagraph'.
         *
         * @param {jQuery} drawing
         *  One drawing node that is no longer positioned absolutely.
         *  This can be the place holder node or the drawing in the
         *  drawing layer node for convenience reasons.
         */
        this.shiftDrawingAwayFromDrawingLayer = function (drawing) {

            var // the drawing layer node for absolute positioned drawings
                drawingLayerNode = self.getDrawingLayerNode(),
                // the drawing node in the drawing layer
                drawingNode = null,
                // the drawing place holder in the page content
                placeHolderNode = null,
                // the space maker node for the absolute positioned drawing
                spaceMakerNode = null;

            // supporting space holder node and drawing node in drawing layer
            if (DOM.isDrawingPlaceHolderNode(drawing)) {
                drawingNode = DOM.getDrawingPlaceHolderNode(drawing);
                placeHolderNode = drawing;
            } else if (DOM.isDrawingLayerNode($(drawing).parent())) {
                drawingNode = drawing;
                placeHolderNode = DOM.getDrawingPlaceHolderNode(drawing);
            } else {
                return null;
            }

            // ... removing the space maker node
            spaceMakerNode = DOM.getDrawingSpaceMakerNode(drawingNode);
            if (spaceMakerNode) {
                removeDrawingSpaceMaker(spaceMakerNode);
                $(drawingNode).removeData(DOM.DRAWING_SPACEMAKER_LINK);
            }

            // shifting the drawing behind the place holder node
            $(placeHolderNode).after(drawingNode);

            // ... removing the place holder node
            $(placeHolderNode).remove();
            $(drawingNode).removeData(DOM.DRAWINGPLACEHOLDER_LINK);
            $(drawingNode).removeAttr('data-drawingID');

            // delete drawing layer, if it is no longer required
            if (drawingLayerNode.length === 0) {
                // creating the drawing layer
                $(drawingLayerNode).remove();
                // no more drawings in collector
                drawings = [];
            }

            // remove drawing from the drawings list
            if (!_.isEmpty(drawings)) {
                // ... and finally removing the drawing from the collector 'drawings'
                drawings = _.filter(drawings, function (node) {
                    return Utils.getDomNode(node) !== Utils.getDomNode(drawing);
                });
            }

        };

        /**
         * Removing all drawings from the drawing layer, that have place holder nodes
         * located inside the specified node. The node can be a paragraph that will be
         * removed. In this case it is necessary, that the drawings in the drawing layer
         * are removed, too.
         * This is a simplified function, that simply removes the drawing nodes
         * located inside the drawing layer. It does not take care of space maker
         * nodes or place holder nodes. This is not necessary, because it is assumed,
         * that the specified node will be removed, too.
         * Drawing nodes inside headers and footers are not handled inside this function.
         * These drawing place holder nodes and the complete drawing layer are removed,
         * when the paragraph is removed.
         *
         * @param {HTMLElement|jQuery} node
         *  The node, for which all drawings in the drawing layer will be removed. This
         *  assumes that the node contains place holder nodes. Only the drawings in the
         *  drawing layer will be removed. This function does not take care of place
         *  holder nodes and space maker nodes.
         */
        this.removeAllInsertedDrawingsFromDrawingLayer = function (node) {

            var // the collection of all place holder nodes
                allPlaceHolder = $(node).find(DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR);

            // starting to remove the drawings, if there are place holder nodes
            if (allPlaceHolder.length > 0) {
                _.each(allPlaceHolder, function (placeHolderNode) {

                    var // the drawing node corresponding to the place holder node
                        drawingNode = null;

                    // is the drawing place holder node inside the header or footer?
                    // -> then the parent is a div.par element
                    if (!DOM.isMarginParagraphNode(placeHolderNode.parentNode)) {

                        drawingNode = DOM.getDrawingPlaceHolderNode(placeHolderNode);

                        if (drawingNode) {
                            drawingNode.remove();
                        } else {
                            Utils.warn('removeAllInsertedDrawingsFromDrawingLayer(): failed to find drawing node for place holder node!');
                        }
                    }
                });
            }
        };

        /**
         * Removing one drawing element from the drawing layer node. The parameter
         * is the place holder node in the page content.
         *
         * @param {jQuery} placeHolderNode
         *  The place holder node in the page content
         *
         *  Optional parameters:
         *  @param {Boolean} [options.keepDrawingLayer=false]
         *      If set to true, the drawing in the drawing layer corresponding to the
         *      drawing place holder is NOT removed. This is for example necessary after
         *      a split of paragraph. The default is 'false' so that place holder node,
         *      space maker node and drawing in drawing layer are removed together.
         */
        this.removeFromDrawingLayer = function (placeHolderNode, options) {

            var // the drawing node in the drawing layer
                drawing = DOM.getDrawingPlaceHolderNode(placeHolderNode),
                // whether the drawings in the drawing layer shall not be removed
                // -> this is necessary after splitting a paragraph
                keepDrawingLayer = Utils.getBooleanOption(options, 'keepDrawingLayer', false),
                // the space maker node for the absolute positioned drawing
                spaceMakerNode = null;

            if (keepDrawingLayer) {

                // removing only the place holder node
                $(placeHolderNode).removeData(DOM.DRAWINGPLACEHOLDER_LINK);
                $(placeHolderNode).remove();

            } else {

                // removing links to other nodes
                $(placeHolderNode).removeData(DOM.DRAWINGPLACEHOLDER_LINK);
                $(drawing).removeData(DOM.DRAWINGPLACEHOLDER_LINK);

                // safely release image data (see comments in BaseApplication.destroyImageNodes())
                app.destroyImageNodes(drawing);

                // ... removing the space maker node
                spaceMakerNode = DOM.getDrawingSpaceMakerNode(drawing);

                if (spaceMakerNode) {
                    removeDrawingSpaceMaker(DOM.getDrawingSpaceMakerNode(drawing));
                    $(drawing).removeData(DOM.DRAWING_SPACEMAKER_LINK);
                }

                // ... removing the drawing in the drawing layer
                $(drawing).remove();

                // ... removing the place holder node
                $(placeHolderNode).remove();

                // ... and finally removing the drawing from the collector 'drawings'
                drawings = _.filter(drawings, function (node) {
                    return Utils.getDomNode(node) !== drawing;
                });
            }
        };

        /**
         * After splitting a paragraph with the place holder in the new paragraph (that
         * was cloned before), it is necessary that the drawing in the drawing layer
         * updates its link to the place holder drawing in the new paragraph.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.repairLinksToPageContent = function (paragraph) {

            _.each($(paragraph).children(DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR), function (placeHolder) {
                var drawing = DOM.getDrawingPlaceHolderNode(placeHolder);
                $(drawing).data(DOM.DRAWINGPLACEHOLDER_LINK, placeHolder);
            });

        };

        /**
         * Receiving the drawing layer node for absolute positioned drawings. It
         * is not created within this function. If it does not exist, null is
         * returned.
         *
         * @param {jQuery} [target]
         *  An optional alternative for the destination of the drawing layer. If not
         *  specified, div.page needs to be the parent of the drawing layer. For
         *  headers and footers, it is also possible, that div.header or div.footer
         *  are the parent for the drawing layer.
         *
         * @returns {jQuery|null} drawingLayerNode
         *  The drawing layer node, if is exists. Otherwise null.
         */
        this.returnDrawingLayerNode = function (target) {

            var // the document page node or the target in header or footer
                pageNode = (target && target.length > 0) ? target : model.getNode(),
                // the drawing layer node in the dom
                drawingLayerNode = pageNode.children(DOM.DRAWINGLAYER_NODE_SELECTOR);

            return (drawingLayerNode && drawingLayerNode.length > 0) ? drawingLayerNode : null;
        };

        /**
         * Receiving the drawing layer node for absolute positioned drawings.
         * It is created, if it does not exist yet.
         *
         * @param {jQuery} [target]
         *  An optional alternative for the destination of the drawing layer. If not
         *  specified, div.page needs to be the parent of the drawing layer. For
         *  headers and footers, it is also possible, that div.header or div.footer
         *  are the parent for the drawing layer.
         *
         * @returns {jQuery} drawingLayerNode
         *  The drawing layer node
         */
        this.getDrawingLayerNode = function (target) {

            var // the document page node or the target in header or footer
                pageNode = (target && target.length > 0) ? target : model.getNode(),
                // the drawing layer node in the dom
                drawingLayerNode = self.returnDrawingLayerNode(pageNode),
                // the page content node
                pageContentNode = null;

            // create drawing layer node, if necessary
            if (!drawingLayerNode || (drawingLayerNode && drawingLayerNode.length === 0)) {
                // the drawing layer needs to be created next to the page content node. In the case of
                // a header or footer node, it will be created directly below the element div.header
                // or div.footer. In the latter case, the drawing layer is created directly behind
                // the last paragraph.
                pageContentNode = (target && target.length > 0) ? pageNode.children(DOM.PARAGRAPH_NODE_SELECTOR_COMPLETE).last() : DOM.getPageContentNode(pageNode),
                drawingLayerNode = $('<div>').addClass(DOM.DRAWINGLAYER_CLASS);
                pageContentNode.after(drawingLayerNode);
            }

            return drawingLayerNode;
        };

        /**
         * After load from local storage the links between drawings and drawing place holders
         * in the header and footer nodes need to restored. In this scenario, the drawings in the
         * drawing layer are loaded from local storage. Then in a following step all links
         * saved in the data objects of absolute drawing, place holder node and maybe a space
         * maker node need to be updated after load.
         */
        this.refreshMarginDrawingLayer = function () {

            if (!pageLayout) { pageLayout = model.getPageLayout(); }

            var // a list of all header and footer nodes
                allMargins = pageLayout.getallHeadersAndFooters();

            _.each(allMargins, function (margin) {

                // is there a drawing layer located inside this margin?

                var // the drawings in one drawing layer in one header or footer
                    allDrawings = null,
                    // the drawing layer in one header or footer
                    drawingLayer = $(margin).children(DOM.DRAWINGLAYER_NODE_SELECTOR);

                if (drawingLayer.length > 0) {

                    allDrawings = drawingLayer.children('div.drawing');

                    // the placeholder nodes (and space maker nodes) need to be assigned

                    if (allDrawings.length > 0) {
                        _.each(allDrawings, function (absoluteDrawing) {
                            refreshLinkForAbsoluteDrawing(absoluteDrawing, margin, { isMainDrawingLayer: false });
                        });
                    }
                }
            });
        };



        /**
         * After load from local storage the links between drawings and drawing place holders
         * need to restored.
         * Additionally the global 'drawings' collector is filled with the drawings.
         */
        this.refreshDrawingLayer = function () {

            // creating all links between drawing in drawing layer and the corresponding
            // drawing place holder
            // -> for this the data-drawingid is required

            var // the page content node
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // collecting all drawings in the drawing layer
                allDrawings = self.getDrawingLayerNode().children('div.drawing');

            drawings = [];

            // filling the list of absolute positioned drawings
            _.each(allDrawings, function (oneDrawing) {
                drawings.push($(oneDrawing));
            });

            if (drawings.length > 0) {
                _.each(drawings, function (absoluteDrawing) {
                    refreshLinkForAbsoluteDrawing(absoluteDrawing, pageContentNode, { isMainDrawingLayer: true });
                });
            }
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {

            function updateDrawings() {
                if (!self.isEmpty()) { self.updateAbsoluteDrawings(); }
            }

            model = app.getModel();

            model.one('pageBreak:after', updateDrawings);

            app.getView().on('change:pageWidth', updateDrawings);
        });

    } // class DrawingLayer

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: DrawingLayer });

});
